//YAMP regression and inversion tester

// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.



#include "matv.h"

Matrix ExactHilbInv( int n )
{
  // construct exact hilbert matrix inverse
  Matrix Hi = Fill(n,n,0);

  long double diag=0;
  long double dn = (long double) n;
  for( int i=1; i<=n; i++){
    long double im1 = (double) (i-1.0);
    diag = ( i == 1 ) ? dn : ((dn-im1)*diag*(dn+im1))/(im1*im1);

    long double rest = diag*diag;
    Hi(i,i) = rest/(2.0*im1+1.0);
    for( int j=i+1; j<=n; j++ ){
     long double jm1 = (long double) (j-1);
     rest = -((dn-jm1)*rest*(dn+jm1))/(jm1*jm1);
     Hi(j,i) = Hi(i,j) = rest/(im1+jm1+1.0);
    }
  }
  Hi.show("exact hilbert inverse");
  Hi.release();
  return Hi;
}


void TestHilb( int hilb_ind )
// test svd on hilbert matrix
{
  Matrix hilb = Kron( Fill(1,hilb_ind,1), Index(0, hilb_ind-1));
  // h(i+1,j+1) = 1.0/(i+j+1) ; j=0..n-1, i=0..n-1
  hilb = 1.0 / ( (hilb+Tran(hilb))+1.0);
  hilb.show();
  // use formula. Looses accuracy of 0.001 at n>=11
  Matrix xx = ExactHilbInv( hilb_ind )*hilb;
  xx.show();

  // use sweep
  Matrix Gauss = Inv(hilb)*hilb;
  Gauss.show();

  // use singular valued decomposition
  Matrix S, V, D;
  Svd( hilb, S, V, D);
  Matrix t = Fill(V.r, V.r, 0.0);
  for (int i = 1; i <= V.r; i++) {
  double tt = V(i, 1);
  t(i, i) = 1.0 / tt;
  }

  Matrix Ggauss = (D*t*Tran(S))*hilb;
  Ggauss.show("Generalized inversion");

}

///////////////////// now test regressions on sample data

Matrix getx(int N)
// create an x matrix
{

  Matrix x = Index( 1, N ) - N/2;
  Matrix c1 = Fill(N, 1, 1.0);
  Matrix x2 = x % x;
  x = Ch(c1, Ch(x, x2));
  return x;
}
#ifndef random
#define random( max ) ((rand() % (int)(((max)+1) - (0))) + (0))
#endif

Matrix gety(Matrix &x, Matrix &beta)
// create a y matrix
{
  Matrix y = x*beta;
  srand(123);
  for (int i = 1; i <= y.r; i++) {
    // use sum of 3 uniforms for an approximate
    // normal random variable
    int u = random(100) + random(100) + random(100) + 3;
    y(i, 1) = y(i, 1) + ((double) (u - 150)) / 300.0;
  }
  return y;
}

Matrix regression(Matrix& x, Matrix& y)
// do a multiple linear regression
{

  int N = x.r, p = x.c;
  // solve for regression parameters using sweep
  Matrix yx = Ch(y, x);
  Matrix reg = Sweep(2, p + 1, Tran(yx) *yx);
  // calculate mean square error
  reg(1, 1) = reg(1, 1) / ((double) (N - p));
  reg.show();


  // solve regression using normal equations
  Matrix betahat = Inv(Tran(x) *x) *Tran(x) *y;

  return betahat;
}
Matrix QRregression(Matrix &x, Matrix &y)
{
  // use QR decomposition to solve regression

  Matrix Q, R;

  QR(x, Q, R);
  Matrix betahat = Inv(Tran(R) *R) *Tran(R) *Tran(Q) *y;
  return betahat;
}

Matrix GinvRegression(Matrix &x, Matrix &y)
{
  // use Ginv to solve normal equations
  Matrix betahat = Ginv(Tran(x) *x) *Tran(x) *y;
  return betahat;
}


Matrix SVDregression(Matrix &x, Matrix &y)
{
  // use SVD to solve regression x = S Diag(V) Tran(D)
  Matrix S, V, D;

  Svd(x, S, V, D);
  Matrix t = Fill(V.r, V.r, 0.0);
  for (int i = 1; i <= V.r; i++) {
    double tt = V(i, 1);
    t(i, i) = (fabs(tt) <= 0.0) ? 0.0 : 1.0 / tt;
  }
  Matrix betahat = D*t*Tran(S)*y;
  return betahat;
}

Matrix GetSerialCovar( Matrix &R )
{
  // Parameters to CORR in Timeslab
  // correlogram = CORR(x=R,n=R.r,M=R.r-1,Q=2*R.r,
  //                    iopt=1,r0=r0, per = spectdens)
  // covar = correlogram*r0
  Matrix centered, z, covar, spectdens;
  double n = (double) R.r;
  // center a column vector
  centered = R - (Sum(R))(1, 1) / n;
  // zero pad to length 2n: 2n periodic for full
  // sample spectral density
  centered = Cv(centered, Fill(R.r, R.c, 0));
  // take fft
  z = Fft(centered);
  // take convolution : gives sample spectral density
  spectdens = Sum(z % z / n, COLUMNS);
  // inverse fft for serial correlation (autocorrelation)
  covar = Fft(spectdens, FFALSE);
  // throw away last half.
  covar = Submat(covar, 1,1,R.r, 2);
  return covar;
}

main()
{
  Matrix x, beta(3, 1), y, betahat, resids, serial;

  TestHilb( 11 );

  Setwid(15);
  Setdec(10);

  beta(1, 1) = 1;
  beta(2, 1) = 0.5;
  beta(3, 1) = 0.25;

  x = getx(50);
  y = gety(x, beta);

  betahat = regression(x, y);
  (Tran(beta)).show();
  (Tran(betahat)).show();

  betahat = QRregression(x, y);
  (Tran(beta)).show("QR betahat");
  (Tran(betahat)).show();

  betahat = GinvRegression(x, y);
  (Tran(beta)).show("Ginv betahat");
  (Tran(betahat)).show();

  betahat = SVDregression(x, y);
  (Tran(beta)).show("SVD regression");
  (Tran(betahat)).show();

  resids = y - x*betahat;
  serial = GetSerialCovar(resids);
  (Tran(Submat(serial, 1,1, 5, 1))).show();

  Setwid(6);
  Setdec(3);

  }
