#include "matv.h"

// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.


////////////////////////////// complex number work arounds
#ifdef USE_COMPLEX
#define zeroc complex(0,0)
bboolean GT( complex a, complex b)
{
   return ( abs(a) > abs(b) ) ? TTRUE : FFALSE;
}

bboolean GE( complex a, complex b)
{
   return ( abs(a) >= abs(b) ) ? TTRUE : FFALSE;
}

#else
#define zeroc 0.0
bboolean GT( FLOAT a, FLOAT b)
{
   return ( a > b ) ? TTRUE : FFALSE;
}

bboolean GE( FLOAT a, FLOAT b)
{
   return ( a >= b ) ? TTRUE : FFALSE;
}
#endif

////////////////////////////////////

#ifndef USE_COMPLEX
void heapsort(Matrix &a)
{
  //not available for complex matrices
  int n = a.r;
  int s0, s1, s2, s3, s4, s5, s4m1, s5m1;
  double t1, ts;
  s0 = s1 = n;   /* Shell-Metzger sort, PC-World, May 1986 */
  s1 /= 2;
  while (s1 > 0) {
    s2 = s0 - s1;
    for (s3 = 1; s3 <= s2; s3++) {
      s4 = s3;
      while (s4 > 0) {
        s5 = s4 + s1;
        s4m1 = s4;
        s5m1 = s5;
        if (a(s4m1, 1) > a(s5m1, 1)) {
          t1 = a(s4m1, 1);
          a(s4m1, 1) = a(s5m1, 1);
          a(s5m1, 1) = t1;
          ts = a(s4m1, 2);
          a(s4m1, 2) = a(s5m1, 2);
          a(s5m1, 2) = ts;
          s4 = s4 - s1;
        }
        else {
          s4 = 0;
        }
      }
    }
    s1 /= 2;
  }
}

Matrix MSort(Matrix &a, int col, int order)
{
  //not available for complex matrices
  a.Garbage("MSort");
  Matrix sorted( a.r, a.c);
  Matrix temp;
  if (!order) {
    // sort column col, then reorder other rows
    temp = Fill(a.r, 2, zeroc );
    for (int i = 1; i <= a.r; i++) {
      temp(i, 1) = a(i, col);
      temp(i, 2) = (double) i;
    }
    heapsort(temp);
    for (i = 1; i <= a.r; i++) {
      for (int j = 1; j <= a.c; j++)
        sorted(i, j) = a(((int) temp(i, 2)), j);
    }

  }
  else {
    // sort row col, then reorder other columns
    temp = Fill(a.c, 2, zeroc);
    for (int i = 1; i <= a.c; i++) {
      temp(i, 1) = a(col, i);
      temp(i, 2) = (double) i;
    }
    heapsort(temp);
    for (i = 1; i <= a.c; i++) {
      for (int j = 1; j <= a.r; j++)
        sorted(j, i) = a(j, ((int) temp(i, 2)));
    }

  }
  sorted.release();
  return sorted;
}
#endif


//////////////////  math funtions

Matrix Mexp(Matrix &ROp)
{
  // take exponent of all elements
  ROp.Garbage("Mexp");
  Matrix temp( ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) temp(i, j) = exp(ROp(i, j));
  }
  temp.release();
  return temp;
}
Matrix Mabs(Matrix &ROp)
{
  // take absolute values of all elements
  ROp.Garbage("Mabs");
  Matrix temp( ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) temp(i, j) = fabs(ROp(i, j));
  }
  temp.release();
  return temp;
}
Matrix Mlog(Matrix &ROp)
{
  // take logarithm of all elements
  ROp.Garbage("Mlog");
  Matrix temp(ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) {
      FLOAT R = ROp(i, j);
      temp(i, j) = ( GT(R, zeroc) ) ? log(R) :
      Matrix::Nrerror("Mlog: zero or negative argument to log");
    }
  }
  temp.release();
  return temp;
}
Matrix Msqrt(Matrix &ROp)
{
  // take sqrt of all elements
  ROp.Garbage("Msqrt");
  Matrix temp( ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) {
      FLOAT R = ROp(i, j);
      temp(i, j) = ( GE(R, zeroc) ) ? sqrt(R) :
      Matrix::Nrerror("Msqrt: zero or negative argument to sqrt");
    }
  }
  temp.release();
  return temp;
}

Matrix Msin(Matrix &ROp)
{
  // take exponent of all elements
  ROp.Garbage("Msin");
  Matrix temp( ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) temp(i, j) = sin(ROp(i, j));
  }
  temp.release();
  return temp;
}
Matrix Mcos(Matrix &ROp)
{
  // take exponent of all elements
  ROp.Garbage("Mcos");
  Matrix temp( ROp.r, ROp.c);
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) temp(i, j) = cos(ROp(i, j));
  }
  temp.release();
  return temp;
}

FLOAT Trace(Matrix &ROp)
{
  ROp.Garbage("Trace");
  FLOAT trace;
  trace = zeroc;
  if (ROp.r != ROp.c)
    ROp.Nrerror("Trace: matrix not square in trace");
  for (int i = 1; i <= ROp.r; i++) trace += ROp(i, i);
  return trace;
}
Matrix Helm(int n)
{
  Matrix temp( n, n);
  for (int i = 1; i <= n; i++) {
    for (int j = 1; j <= n; j++) {
      double d = 1.0 / sqrt((double) n);
      double den = (j > 1) ? 1.0 / sqrt((double) j*(j - 1)) : d;
      if (j > 1 && j < i) d = 0;
      if (j > 1 && j == i) d = -den * ((double) (j - 1));
      if (j > 1 && j > i) d = den;
      temp(i, j) = d;
    }
  }
  temp.release();
  return temp;
}

Matrix Index(int lower, int upper, int step)
{
  if (step == 0) Matrix::Nrerror("Index: step is zero");
  int cnter = 0;
  if (lower < upper) {
    for (int i = lower; i <= upper; cnter++, i += step);
  }
  else {
    for (int i = lower; i >= upper; cnter++, i += step);
  }
  if (cnter == 0)
    Matrix::Nrerror(
        "Index: trying to allocate a matrix with zero rows");
  Matrix temp( cnter, 1);

  cnter = 1;
  double di;
  if (lower < upper) {
    for (int i = lower; i <= upper; i += step){
      di = i;
      temp(cnter++, 1) =  di;
    }
  }
  else {
    for (int i = lower; i >= upper; i += step){
      di = i;
      temp(cnter++, 1) =  di;
    }
  }

  temp.release();
  return temp;
}

Matrix Kron(Matrix &a, Matrix &b)
{
  a.Garbage("Kron");
  b.Garbage("Kron");

  int cr = a.r*b.r;
  int cc = a.c*b.c;
  Matrix c( cr, cc);

  for (int i = 1; i <= a.r; i++) {
    for (int j = 1; j <= a.c; j++) {
      for (int k = 1; k <= b.r; k++) {
        for (int l = 1; l <= b.c; l++) {
          c((i - 1)*b.r + k, (j - 1) *b.c + l)  = a(i, j) *b(k,l);
        }
      }
    }
  }
  c.release();
  return c;
}   /* kron */

#define ZERO 1.0E-8
void Pivot(Matrix &Data, int RefRow,
  FLOAT &Determ, enum bboolean &DetEqualsZero)
{
  DetEqualsZero = TTRUE;
  int NewRow = RefRow;
  while (DetEqualsZero && (NewRow < Data.r)) {
    NewRow++;
    if (fabs(Data(NewRow, RefRow)) > ZERO) {
      for (int i = 1; i <= Data.r; i++) {
        FLOAT temp = Data(NewRow, i);
        Data(NewRow, i) = Data(RefRow, i);
        Data(RefRow, i) = temp;
      }
      DetEqualsZero = FFALSE;
      Determ = -Determ;   /* row swap adjustment to det */
    }
  }
}
FLOAT Det(Matrix &Datain)
{
  Datain.Garbage("Det");

  FLOAT Determ;
  Determ = 1.0;
  if (Datain.r == Datain.c) {

    int Dimen = Datain.r;
    Matrix Data( Dimen, Dimen);

    Data  = Datain;
    LFLOAT Coef;
    int Row, RefRow = 0;
    enum bboolean DetEqualsZero = FFALSE;

    while (!(DetEqualsZero) && (RefRow < Dimen - 1)) {
      RefRow++;
      if (fabs(Data(RefRow, RefRow)) < ZERO)
        Pivot(Data, RefRow, Determ, DetEqualsZero);
      if (!(DetEqualsZero))
        for (Row = RefRow + 1; Row <= Dimen; Row++)
          if (fabs(Data(Row, RefRow)) > ZERO) {
            Coef = -((LFLOAT) Data(Row, RefRow)) /
            ((LFLOAT) Data(RefRow, RefRow));
            for (int i = 1; i <= Dimen; i++)
              Data(Row, i) = Data(Row, i) +
                (LFLOAT) (Coef*((LFLOAT) Data(RefRow,i)));
          }
          Determ *= Data(RefRow, RefRow);
    }
    Determ = (DetEqualsZero) ? zeroc : Determ * Data(Dimen, Dimen);
  }
  else Datain.Nrerror("Det: Matrix is not square\n");
  return Determ;
}
#undef ZERO

//--------------- cholesky decomposition ----------------------
// ROp is symmetric, returns upper triangular S where ROp = S'S
//-------------------------------------------------------------
#define EPS 1.0e-7
Matrix Cholesky(Matrix& ROp)
{
  int nr = ROp.r;

  ROp.Garbage("Cholesky");
  for (int i = 1; i <= nr; i++) {
    for (int j = 1; j < i; j++)
      if (fabs(ROp(i, j) - ROp(j, i)) > EPS)
        Matrix::Nrerror("Cholesky: Input matrix is not symmetric");
  }
  Matrix temp;
  temp  = Fill(nr, nr, zeroc);
  for (i = 1; i <= nr; i++) {
    for (int j = i; j <= nr; j++) {
      FLOAT sum;
      sum = 0.0;
      for (int k = 1; k < i; k++) {
        sum += temp(k, i) * conj(temp(k, j));
      }
      if (i == j) temp(i, j) = ( GT(sum,ROp(i, j)) ) ?
        Matrix::Nrerror("Cholesky: negative pivot") :
        sqrt(ROp(i, j) - sum);
      else temp(i, j) = (fabs(temp(i, i)) < EPS) ?
        Matrix::Nrerror("Cholesky: zero or negative pivot") :
        (ROp(i, j) - sum) / temp(i, i);
    }
  }
  temp.release();
  return temp;
}

#ifndef USE_COMPLEX
void QR(Matrix& ROp, Matrix& Q, Matrix& R, bboolean makeq)
{
  //not available for complex matrices
  ROp.Garbage("QR");
  Q.Garbage("QR");
  R.Garbage("QR");

  int r = ROp.r;
  int c = ROp.c;

  Q = ROp;
  R = Fill(c, c, zeroc);
  FLOAT s;

  for (int j = 1; j <= c; j++) {
    FLOAT sigma = zeroc;
    for (int i = j; i <= r; i++) sigma += Q(i, j) * conj(Q(i, j));
    if (fabs(sigma) <= 1.0e-10) Matrix::Nrerror("QR: singular X");
    R(j, j) = s = ( GT(zeroc,Q(j, j)) ) ? sqrt(sigma) : - sqrt(sigma);
    FLOAT beta;
    beta = 1.0 /(s*Q(j, j) - sigma);
    Q(j, j) = Q(j, j) - s;
    for (int k = j + 1; k <= c; k++) {
      FLOAT sum = zeroc;
      for (int i = j; i <= r; i++) sum += Q(i, j) * Q(i, k);
      sum *= beta;
      for (i = j; i <= r; i++) Q(i, k) = Q(i, k) + Q(i, j) *sum;
      R(j, k) = Q(j, k);
    }
  }

  if (makeq) Q = ROp*Ginv(R);

}
#undef EPS
//----------------------------------------------
// Beging Singular value decomposition
//----------------------------------------------

//-----------------------------------------------
// safehypot - Computes sqrt(a*a + b*b) without
// destructive underflow or overflow.
//-----------------------------------------------
LFLOAT safehypot(LFLOAT a, LFLOAT b)
{
//not available for complex matrices
  LFLOAT aba, abb, xx, yy;
  aba = fabs(a);
  abb = fabs(b);
  xx = abb / aba;
  yy = aba / abb;
  if (aba > abb)
    return aba*sqrt(1.0+xx*xx);
  else
    return (abb == 0.0 ? 0.0 : abb*sqrt(1.0+ yy*yy));
}
int minim( int a, int b){
  return (a < b ) ? a : b;
}  

LFLOAT SIGN(LFLOAT a, LFLOAT b)
{
//not available for complex matrices
// an old FORTRAN functions
  return ( b >= 0.0 ) ? fabs(a) : -fabs(a);
}
//-----------------------------------------------------------------
// svdcmp - Computes singular value decomposition of
// the matrix a(1..m,1..n) with m >= n
// a = u Diag(w) Tran(v)  note this replaces a with u in place.
// so make sure to save it before starting.
// Also a, w, and v must be initialized
// if makev == FALSE, v is Fill(1,1,0.0)
// Numerical Recipes in C with a twist
//-----------------------------------------------------------------
int svdcmp(Matrix &a, Matrix &w, Matrix &v, bboolean makev)
{
//not available for complex matrices
#ifndef NO_CHECKING
  a.Garbage("SVD: a is garbage");
  w.Garbage("SVD: w is garbage");
  v.Garbage("SVD: d is garbage");
#endif

  int m = a.r;
  int n = a.c;
  int flag, i, its, j, jj, k, l = 0, nm = 0;
  LFLOAT anorm, c, f, g, h, s, scale, x, y, z;

  if (makev) v = Fill(n, n, 0.0);
    else v = Fill(1, 1, 0.0);
  w = Fill(n, 1, 0.0);

  Matrix rv1 = Fill(n, 1, 0.0);
  g = scale = anorm = 0.0;
  for (i = 1; i <= n; i++) {
    l = i + 1;
    rv1(i) = scale*g;
    g = s = scale = 0.0;
    if (i <= m) {
      for (k = i; k <= m; k++) scale += fabs(a(k, i));
      if (scale) {
        for (k = i; k <= m; k++) {
          a(k, i) /= scale;
          s += a(k, i) *a(k, i);
        }
        f = a(i, i);
        g = -SIGN(sqrt(s), f);
        h = f*g - s;
        a(i, i) = f - g;
        for (j = l; j <= n; j++) {
          for (s = 0.0, k = i; k <= m; k++) s += a(k, i) *a(k, j);
          f = s / h;
          for (k = i; k <= m; k++) a(k, j) += f*a(k, i);
        }
        for (k = i; k <= m; k++) a(k, i) *= scale;
      }
    }
    w(i) = scale*g;
    g = s = scale = 0.0;
    if (i <= m && i != n) {
      for (k = l; k <= n; k++) scale += fabs(a(i, k));
      if (scale) {
        for (k = l; k <= n; k++) {
          a(i, k) /= scale;
          s += a(i, k) *a(i, k);
        }
        f = a(i, l);
        g = -SIGN(sqrt(s), f);
        h = f*g - s;
        a(i, l) = f - g;
        for (k = l; k <= n; k++) rv1(k) = a(i, k) / h;
        for (j = l; j <= m; j++) {
          for (s = 0.0, k = l; k <= n; k++) s += a(j, k) *a(i, k);
          for (k = l; k <= n; k++) a(j, k) += s*rv1(k);
        }
        for (k = l; k <= n; k++) a(i, k) *= scale;
      }
    }
    x = fabs(w(i)) + fabs(rv1(i));
    if (x > anorm) anorm = x;
  }
  if (makev) {
    for (i = n; i >= 1; i--) {
      if (i < n) {
        if (g) {
          for (j = l; j <= n; j++)
            v(j, i) = (a(i, j) / a(i, l)) / g;
          for (j = l; j <= n; j++) {
            for (s = 0.0, k = l; k <= n; k++) s += a(i, k) *v(k, j);
            for (k = l; k <= n; k++) v(k, j) += s*v(k, i);
          }
        }
        for (j = l; j <= n; j++) v(i, j) = v(j, i) = 0.0;
      }
      v(i, i) = 1.0;
      g = rv1(i);
      l = i;
    }
  }
  for (i = minim(m, n); i >= 1; i--) {
    l = i + 1;
    g = w(i);
    for (j = l; j <= n; j++) a(i, j) = 0.0;
    if (g) {
      g = 1.0 / g;
      for (j = l; j <= n; j++) {
        for (s = 0.0, k = l; k <= m; k++) s += a(k, i) *a(k, j);
        f = (s / a(i, i)) *g;
        for (k = i; k <= m; k++) a(k, j) += f*a(k, i);
      }
      for (j = i; j <= m; j++) a(j, i) *= g;
    }
    else for (j = i; j <= m; j++) a(j, i) = 0.0;
    a(i, i) += 1.0;
  }
  for (k = n; k >= 1; k--) {
    for (its = 1; its <= 30; its++) {
      flag = 1;
      for (l = k; l >= 1; l--) {
        nm = l - 1;
        if ((fabs(rv1(l)) + anorm) == anorm) { flag = 0; break; }
        if ((fabs(w(nm)) + anorm) == anorm) break;
      }
      if (flag) {
        c = 0.0;
        s = 1.0;
        for (i = l; i <= k; i++) {
          f = s*rv1(i);
          rv1(i) = c*rv1(i);
          if ((fabs(f) + anorm) == anorm) break;
          g = w(i);
          h = safehypot(f, g);
          w(i) = h;
          h = 1.0 / h;
          c = g*h;
          s = -f*h;
          for (j = 1; i <= m; j++) {
            y = a(j, nm);
            z = a(j, i);
            a(j, nm) = y*c + z*s;
            a(j, i) = z*c - y*s;
          }
        }
      }
      z = w(k);
      if (l == k) {
        if (z < 0.0) {
          w(k) = -z;
          if (makev) for (j = 1; j <= n; j++) v(j, k) = -v(j, k);
        }
        break;
      }
      if (its == 30) return 1;
      x = w(l);
      nm = k - 1;
      y = w(nm);
      g = rv1(nm);
      h = rv1(k);
      f = ((y - z) *(y + z) + (g - h) *(g + h)) / (2.0*h*y);
      g = safehypot(f, 1.0);
      f = ((x - z) *(x + z) + h*((y / (f + SIGN(g, f))) - h)) / x;
      c = s = 1.0;
      for (j = l; j <= nm; j++) {
        i = j + 1;
        g = rv1(i);
        y = w(i);
        h = s*g;
        g = c*g;
        z = safehypot(f, h);
        rv1(j) = z;
        c = f / z;
        s = h / z;
        f = x*c + g*s;
        g = g*c - x*s;
        h = y*s;
        y *= c;
        if (makev) {
          for (jj = 1; jj <= n; jj++) {
            x = v(jj, j);
            z = v(jj, i);
            v(jj, j) = x*c + z*s;
            v(jj, i) = z*c - x*s;
          }
        }
        z = safehypot(f, h);
        w(j) = z;
        if (z) {
          z = 1.0 / z;
          c = f*z;
          s = h*z;
        }
        f = c*g + s*y;
        x = c*y - s*g;
        for (jj = 1; jj <= m; jj++) {
          y = a(jj, j);
          z = a(jj, i);
          a(jj, j) = y*c + z*s;
          a(jj, i) = z*c - y*s;
        }
      }
      rv1(l) = 0.0;
      rv1(k) = f;
      w(k) = x;
    }
  }
  return 0;
}

int Svd(Matrix &A, Matrix &U, Matrix &S, Matrix &V,
      bboolean makeu, bboolean makev)
{
//not available for complex matrices
#ifndef NO_CHECKING
    A.Garbage("SVD");
    U.Garbage("SVD");
    S.Garbage("SVD");
    V.Garbage("SVD");
#endif

  Matrix uu, ss, vv;

  if (A.r < A.c) uu = Tran(A);
    else uu = A;

  int ierr = svdcmp( uu, ss, vv, makev);

  if (A.r < A.c) {
    if (makeu) U = vv;
    if (makev) V = uu;
  } else {
    if (makeu) U = uu;
    if (makev) V = vv;
  }
  S = ss;

  return ierr;
}

//---------------- end svd ------------------------------------

Matrix Ginv(Matrix& ROp)
{
  //not available for complex matrices
  ROp.Garbage("Ginv");
  Matrix u,s,v,g;

  Svd(ROp, u, s, v, TTRUE, TTRUE);
  for (int i = 1; i <= s.r; i++) {
    double t = s(i, 1);
    s(i, 1) = (fabs(t) <= 0.0) ? 0.0 : 1.0 / t;
  }
  g  = v*Diag(s)*Tran(u);
  g.release();
  return g;
}
#endif


//-------------------------- fft ------------------------------
// de Boor (1980) SIAM J SCI. STAT. COMPUT. V1 no 1, pp 173-178
// and NewMat by R. B. Davies a C++ matrix Package
//-------------------------------------------------------------

static void cossin(int n, int d, double& c, double& s)
//not available for complex matrices
// calculate cos(twopi*n/d) and sin(twopi*n/d)
// minimise roundoff error
    {
      long n4 = n * 4;
      int sector =(int) floor((double) n4 / (double) d + 0.5);
      n4 -= sector * d;
      if (sector < 0) sector = 3 - (3 - sector) % 4; else sector %= 4;
      double ratio = 1.5707963267948966192 * (double) n4 / (double) d;

      switch (sector) {
        case 0 : c = cos(ratio);
        s = sin(ratio);
        break;
        case 1 : c = -sin(ratio);
        s = cos(ratio);
        break;
        case 2 : c = -cos(ratio);
        s = -sin(ratio);
        break;
        case 3 : c = sin(ratio);
        s = -cos(ratio);
        break;
      }
}

#ifndef USE_COMPLEX
static void fftstep(Matrix &AB, Matrix &XY,
  int after, int now, int before)
{
//not available for complex matrices
  int gamma = after * before;
  int delta = now * after;
  double r_value, i_value, ridx_value, iidx_value, temp;
  int j, x1, ia, a, a1, x2, ib, a2, idx, in;

  double r_arg = 1.0, i_arg = 0.0;

  int x = 1;
  int m = AB.r - gamma;

  for (j = 0; j < now; j++) {
    a = 1;
    x1 = x;
    x += after;
    for ( ia = 0; ia < after; ia++) {
      // generate sins & cosines explicitly rather than iteratively
      // for more accuracy; but slower
      cossin(- (j*after + ia), delta, r_arg, i_arg);
      a1 = a++;
      x2 = x1++;
      if (now == 2) {
        ib = before;
        while (ib--) {
          a2 = m + a1;
          a1 += after;
          idx = a2 - gamma;
          r_value = AB(a2, 1);
          i_value = AB(a2, 2);
          ridx_value = AB(idx,1);
          iidx_value = AB(idx,2);
          XY(x2, 1) = r_value*r_arg - i_value*i_arg + ridx_value;
          XY(x2, 2) = r_value*i_arg + i_value*r_arg + iidx_value;
          x2 += delta;
        }
      }
      else {
        ib = before;
        while (ib--) {
          a2 = m + a1;
          a1 += after;
          r_value = AB(a2, 1);
          i_value = AB(a2, 2);
          in = now - 1;
          while (in--) {
            a2 -= gamma;
            ridx_value = AB(a2, 1);
            iidx_value = AB(a2, 2);
            temp = r_value;
            r_value = r_value*r_arg - i_value*i_arg + ridx_value;
            i_value = temp*i_arg + i_value*r_arg + iidx_value;
          }
          XY(x2, 1) = r_value;
          XY(x2, 2) = i_value;
          x2 += delta;
        }
      }
    }
  }
}


Matrix Fft(Matrix &ROp, bboolean INZEE)
// if INZEE == TTRUE  then calculate fft
// if INZEE == FFALSE then calculate inverse fft
//not available for complex matrices
{
  ROp.Garbage("FFT");
  if (ROp.c > 2) Matrix::Nrerror("FFT: Input has more than two columns");

  int n = ROp.r;   // length of arrays
  const int nextmx = 26;
  int prime[26] = { 2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37,
    41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83,
    89, 97, 101 };
  int after = 1;
  int before = n;
  int next = 0;
  bboolean inzee = TTRUE;

  Matrix AB( n, 2);
  Matrix XY( n, 2);

  if (ROp.c == 1) {
    for (int i = 1; i <= n; i++) {
      AB(i, 1) = ROp(i, 1);
      AB(i, 2) = 0.0;
    }
  } else AB  = ROp;
  XY  = Fill(n, 2, 0.0);

  if (INZEE == FFALSE) {
    // take complex conjugate for ifft
    for (int i = 1; i <= n; i++) AB(i, 2) = -AB(i, 2);
  }

  do {
    int now, b1;
    for (;;) {
      if (next < nextmx) now = prime[next];
      b1 = before / now;
      if (b1 * now == before) break;
      next++;
      now += 2;
    }
    before = b1;

    if (inzee) fftstep( AB, XY, after, now, before);
    else fftstep(XY, AB, after, now, before);

    inzee =(inzee == TTRUE) ? FFALSE : TTRUE;
    after *= now;
  }
  while (before != 1);

  if (inzee == TTRUE) XY  = AB;
  // divide by n for ifft
  if (INZEE == FFALSE) XY  /= ((double) n);

  XY.release();
  return XY;
}
#endif

#ifdef USE_COMPLEX
static void fftstep(Matrix &AB, Matrix &XY,
  int after, int now, int before)
{
//not available for complex matrices
  int gamma = after * before;
  int delta = now * after;
  double r_value, i_value, ridx_value, iidx_value, temp;
  int j, x1, ia, a, a1, x2, ib, a2, idx, in;

  double r_arg = 1.0, i_arg = 0.0;
  complex argval, value;

  int x = 1;
  int m = AB.r - gamma;

  for (j = 0; j < now; j++) {
    a = 1;
    x1 = x;
    x += after;
    for ( ia = 0; ia < after; ia++) {
      // generate sins & cosines explicitly rather than iteratively
      // for more accuracy; but slower
      cossin(-(j*after + ia), delta, r_arg, i_arg);
      argval = complex(r_arg,i_arg);
      a1 = a++;
      x2 = x1++;
      if (now == 2) {
        ib = before;
        while (ib--) {
          a2 = m + a1;
          a1 += after;
          idx = a2 - gamma;
          XY(x2) = AB(a2)*argval + AB(idx);
          x2 += delta;
        }
      }
      else {
        ib = before;
        while (ib--) {
          a2 = m + a1;
          a1 += after;
          value = AB(a2);
          in = now - 1;
          while (in--) {
            a2 -= gamma;
            value = value*argval + AB(a2);
          }
          XY(x2) = value;
          x2 += delta;
        }
      }
    }
  }
}


Matrix Fft(Matrix &ROp, bboolean INZEE)
// if INZEE == TTRUE  then calculate fft
// if INZEE == FFALSE then calculate inverse fft
// This is the version of the Fft when USE_COMPLEX is defined
{
#ifndef NO_CHECKING
  ROp.Garbage("FFT");
#endif
  if (ROp.c > 1)
     Matrix::Nrerror("Complex FFT: Input has more than one column");

  int n = ROp.r;   // length of arrays
  const int nextmx = 26;
  int prime[26] = { 2, 3, 5, 7, 11, 13, 17, 19, 23, 29, 31, 37,
    41, 43, 47, 53, 59, 61, 67, 71, 73, 79, 83,
    89, 97, 101 };
  int after = 1;
  int before = n;
  int next = 0;
  bboolean inzee = TTRUE;

  Matrix AB = ROp;
  Matrix XY = Fill(n,1,complex(0.0,0.0));

  if (INZEE == FFALSE) {
    // take complex conjugate for ifft
    for (int i = 1; i <= n; i++) AB(i) = conj(AB(i));
  }

  do {
    int now, b1;
    for (;;) {
      if (next < nextmx) now = prime[next];
      b1 = before / now;
      if (b1 * now == before) break;
      next++;
      now += 2;
    }
    before = b1;

    if (inzee) fftstep( AB, XY, after, now, before);
    else fftstep(XY, AB, after, now, before);

    inzee =(inzee == TTRUE) ? FFALSE : TTRUE;
    after *= now;
  }
  while (before != 1);

  if (inzee == TTRUE) XY  = AB;
  // divide by n for ifft
  if (INZEE == FFALSE)
     XY = Conj(XY)/((double)n);

  XY.release();
  return XY;
}
#endif



////////////////////  reshaping functions

Matrix Vec(Matrix& ROp)
{   // send columns of ROp to a vector
  ROp.Garbage("Vec");

  long lrc = ((long) ROp.r) *((long) ROp.c);
  if (lrc >= 32767 || lrc < 1)
    Matrix::Nrerror("Vec: vec produces invalid indices");
  int r = ROp.r*ROp.c;
  int c = 1;
  Matrix temp( r, c);
  int cnter = 1;
  for (int j = 1; j <= ROp.c; j++)
    for (int i = 1; i <= ROp.r; i++) temp(cnter++, 1) = ROp(i, j);

  temp.release();
  return temp;
}

Matrix Vecdiag(Matrix& ROp)
{   // Extract the diagonal from a square matrix and put in a vector
  ROp.Garbage("Vecdiag");
  if (ROp.r != ROp.c)
    Matrix::Nrerror("Vecdiag: ROp is not square");

  Matrix temp( ROp.r, 1);
  for (int i = 1; i <= ROp.r; i++) temp(i, 1) = ROp(i, i);

  temp.release();
  return temp;
}

Matrix Diag(Matrix& ROp)
{   // make a diagonal matrix from a vector or the principal diagonal of
  // a square matrix, off diagonal elements are zero
  ROp.Garbage("Diag");

  if (ROp.r != ROp.c && ROp.c != 1)
    Matrix::Nrerror(
        "Diag: ROp is not square or is not a column vector"  );
  Matrix temp;
  temp  = Fill(ROp.r, ROp.r, zeroc);

  int rr = ROp.r;
  int cc = ROp.c;
  for (int i = 1; i <= rr; i++)
    temp(i, i) = (rr == cc) ? ROp(i, i) : ROp(i, 1);
  temp.release();
  return temp;
}

Matrix Shape(Matrix& ROp, int nrows)
{   // reshape a matrix into n rows, nrows must divide r*c without a
  // remainder
  ROp.Garbage("Shape");

  long nr = (long) nrows;
  long lr = (long) ROp.r;
  long lc = (long) ROp.c;

  if (lr*lc % nr)
    Matrix::Nrerror("Shape: nrows divides r*c with a remainder");


  Matrix temp;
  temp  = ROp;

  long lrc = (lr*lc) / nr;
  if (nr >= 32767 || nr < 1 || lrc >= 32767 || lrc < 1)
    Matrix::Nrerror("Shape: reshape produces invalid indices");
  temp.r = nrows;
  temp.c = (int) lrc;

  temp.release();
  return temp;
}


///////////////////////////// summation functions // maxs and mins

///typedef enum margins { ALL, ROWS, COLUMNS } margins;

Matrix Sum(Matrix& ROp, margins marg)
{   // sum(ROp, ALL) returns 1x1
  // sum(ROp, ROWS) returns 1xc
  // sum(ROp, COLUMNS) returns cx1
  ROp.Garbage("Sum");

  int r, c;
  FLOAT sum;
  switch (marg) {
    case ALL : r = 1; c = 1; break;
    case ROWS : r = 1; c = ROp.c; break;
    case COLUMNS : r = ROp.r; c = 1; break;
    default : Matrix::Nrerror("Sum: invalid margin specification");
  }
  Matrix temp( r, c);

  int i, j;
  switch (marg) {
    case ALL : sum = 0.0;
    for (i = 1; i <= ROp.r; i++)
      for (j = 1; j <= ROp.c; j++) sum += ROp(i, j);
    temp(1, 1) = sum;
    break;
    case ROWS : for (j = 1; j <= c; j++) {
      sum = 0.0;
      for (i = 1; i <= ROp.r; i++) sum += ROp(i, j);
      temp(1, j) = sum;
    }
    break;
    case COLUMNS : for (i = 1; i <= r; i++) {
      sum = 0.0;
      for (j = 1; j <= ROp.c; j++) sum += ROp(i, j);
      temp(i, 1) = sum;
    }
    break;
  }

  temp.release();
  return temp;
}

Matrix Sumsq(Matrix& ROp, margins marg)
{   // sumsq(ROp, ALL) returns 1x1
  // sumsq(ROp, ROWS) returns 1xc
  // sumsq(ROp, COLUMNS) returns cx1
  ROp.Garbage("Sum");

  int r, c;
  FLOAT sum;
  switch (marg) {
    case ALL : r = 1; c = 1; break;
    case ROWS : r = 1; c = ROp.c; break;
    case COLUMNS : r = ROp.r; c = 1; break;
    default : Matrix::Nrerror("Sumsq: invalid margin specification");
  }
  Matrix temp( r, c);

  int i, j;
  switch (marg) {
    case ALL : sum = 0.0;
    for (i = 1; i <= ROp.r; i++)
      for (j = 1; j <= ROp.c; j++) sum += ROp(i, j) * conj(ROp(i, j));
    temp(1, 1) = sum;
    break;
    case ROWS : for (j = 1; j <= c; j++) {
      sum = 0.0;
      for (i = 1; i <= ROp.r; i++) sum += ROp(i, j) * conj(ROp(i, j));
      temp(1, j) = sum;
    }
    break;
    case COLUMNS : for (i = 1; i <= r; i++) {
      sum = 0.0;
      for (j = 1; j <= ROp.c; j++) sum += ROp(i, j) * conj(ROp(i, j));
      temp(i, 1) = sum;
    }
    break;
  }

  temp.release();
  return temp;
}

Matrix Cusum(Matrix& ROp)
{   // cumulative sum along rows
  ROp.Garbage("Cusum");
  Matrix temp( ROp.r, ROp.c);

  FLOAT sum = zeroc;
  for (int i = 1; i <= ROp.r; i++) {
    for (int j = 1; j <= ROp.c; j++) {
      sum += ROp(i, j);
      temp(i, j) = sum;
    }
  }
  temp.release();
  return temp;
}

#ifndef USE_COMPLEX
Matrix Mmax(Matrix& ROp, margins marg)
//not available for complex matrices
{   // Mmax(ROp, ALL) returns 3x1
  // Mmax(ROp, ROWS) returns 3xc
  // Mmax(ROp, COLUMNS) returns cx3
  ROp.Garbage("Sum");

  int r, c;
  switch (marg) {
    case ALL : r = 3; c = 1; break;
    case ROWS : r = 3; c = ROp.c; break;
    case COLUMNS : r = ROp.r; c = 3; break;
    default : Matrix::Nrerror("Mmax: invalid margin specification");
  }
  Matrix temp( r, c);

  int maxr, maxc, i, j;
  double mmax;
  switch (marg) {
    case ALL : mmax = ROp(1, 1);
    maxr = 1; maxc = 1;
    for (i = 1; i <= ROp.r; i++)
      for (j = 1; j <= ROp.c; j++)
        mmax = (mmax < ROp(i, j)) ?
          maxr = i, maxc = j, ROp(i, j) : mmax;
          temp(1, 1) = (double) maxr;
          temp(2, 1) = (double) maxc;
          temp(3, 1) = mmax;
          break;
          case ROWS : for (j = 1; j <= c; j++) {
            mmax = ROp(1, j);
            maxr = 1; maxc = j;
            for (i = 1; i <= ROp.r; i++)
              mmax = (mmax < ROp(i, j)) ? maxr = i, ROp(i,j): mmax;
              temp(1, j) = (double) maxr;
              temp(2, j) = (double) maxc;
              temp(3, j) = mmax;
          }
          break;
          case COLUMNS : for (i = 1; i <= r; i++) {
            mmax = ROp(i, 1);
            maxr = i; maxc = 1;
            for (j = 1; j <= ROp.c; j++)
              mmax = (mmax < ROp(i, j)) ? maxc = j, ROp(i,j):  mmax;
              temp(i, 1) = (double) maxr;
              temp(i, 2) = (double) maxc;
              temp(i, 3) = mmax;
          }
          break;
  }

  temp.release();
  return temp;
}

Matrix Mmin(Matrix& ROp, margins marg)
//not available for complex matrices
{   // Mmin(ROp, ALL) returns 3x1
  // Mmin(ROp, ROWS) returns 3xc
  // Mmin(ROp, COLUMNS) returns cx3
  ROp.Garbage("Sum");

  int r, c;
  switch (marg) {
    case ALL : r = 3; c = 1; break;
    case ROWS : r = 3; c = ROp.c; break;
    case COLUMNS : r = ROp.r; c = 3; break;
    default : Matrix::Nrerror("Mmin: invalid margin specification");
  }
  Matrix temp( r, c);

  int maxr, maxc, i, j;
  double mmin;
  switch (marg) {
    case ALL : mmin = ROp(1, 1);
    maxr = 1; maxc = 1;
    for (i = 1; i <= ROp.r; i++)
      for (j = 1; j <= ROp.c; j++)
        mmin = (mmin > ROp(i, j)) ?
          maxr = i, maxc = j, ROp(i, j) : mmin;
          temp(1, 1) = (double) maxr;
          temp(2, 1) = (double) maxc;
          temp(3, 1) = mmin;
          break;
          case ROWS : for (j = 1; j <= c; j++) {
            mmin = ROp(1, j);
            maxr = 1; maxc = j;
            for (i = 1; i <= ROp.r; i++)
              mmin = (mmin > ROp(i, j)) ? maxr = i, ROp(i,j): mmin;
              temp(1, j) = (double) maxr;
              temp(2, j) = (double) maxc;
              temp(3, j) = mmin;
          }
          break;
          case COLUMNS : for (i = 1; i <= r; i++) {
            mmin = ROp(i, 1);
            maxr = i; maxc = 1;
            for (j = 1; j <= ROp.c; j++)
              mmin = (mmin > ROp(i, j)) ? maxc = j, ROp(i,j):  mmin;
              temp(i, 1) = (double) maxr;
              temp(i, 2) = (double) maxc;
              temp(i, 3) = mmin;
          }
          break;
  }

  temp.release();
  return temp;
}
#endif
//////////////////////////////// elemenatary row and column operations
////////////////// note these functions operate directly on ROp

void Crow(Matrix &ROp, int row, double c)
{   // multiply a row by a constant
  ROp.Garbage("Crow");

  if (row < 1 || row > ROp.r)
    Matrix::Nrerror("Crow: row index out of range");

  for (int i = 1; i <= ROp.c; i++) ROp(row, i) = c*ROp(row, i);
  return;
}

void Srow(Matrix &ROp, int row1, int row2)
{   // swap rows
  ROp.Garbage("Srow");
  if (row1 < 1 || row1 > ROp.r || row2 < 1 || row2 > ROp.r)
    Matrix::Nrerror("Srow: row index out of range");

  FLOAT tmp;
  for (int i = 1; i <= ROp.c; i++) {
    tmp = ROp(row1, i);
    ROp(row1, i) = ROp(row2, i);
    ROp(row2, i) = tmp;
  }
  return;
}

void Lrow(Matrix &ROp, int row1, int row2, double c)
{   // add c*r1 to r2

  ROp.Garbage("Srow");
  if (row1 < 1 || row1 > ROp.r || row2 < 1 || row2 > ROp.r)
    Matrix::Nrerror("Lrow: row index out of range");

  for (int i = 1; i <= ROp.c; i++)
    ROp(row2, i) = ROp(row2, i) + c*ROp(row1, i);

  return;
}

void Ccol(Matrix &ROp, int col, double c)
{   // multiply a col by a constant
  ROp.Garbage("Ccol");

  if (col < 1 || col > ROp.c)
    Matrix::Nrerror("Ccol: col index out of range");

  for (int i = 1; i <= ROp.r; i++) ROp(i, col) = c*ROp(i, col);
  return;
}

void Scol(Matrix &ROp, int col1, int col2)
{   // swap cols
  ROp.Garbage("Scol");
  if (col1 < 1 || col1 > ROp.c || col2 < 1 || col2 > ROp.c)
    Matrix::Nrerror("Scol: col index out of range");

  FLOAT tmp;
  for (int i = 1; i <= ROp.r; i++) {
    tmp = ROp(i, col1);
    ROp(i, col1) = ROp(i, col2);
    ROp(i, col2) = tmp;
  }
  return;
}

void Lcol(Matrix &ROp, int col1, int col2, double c)
{   // add c*c1 to c2

  ROp.Garbage("Scol");
  if (col1 < 1 || col1 > ROp.c || col2 < 1 || col2 > ROp.c)
    Matrix::Nrerror("Lcol: col index out of range");

  for (int i = 1; i <= ROp.r; i++)
    ROp(i, col2) = ROp(i, col2) + c*ROp(i, col1);
  return;
}
