
// Matv - A Simple Matrix Class

// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.


#include "matv.h"

////////////////////////////// complex function substitutions

#ifdef USE_COMPLEX
double fabs( complex x ){ return abs(x); }
#else
FLOAT conj( FLOAT x ){ return x; }
#endif

//////////////////////////////// vector
///// error handler
FLOAT Vector::Nrerror(const char *errormsg)
{
        FLOAT x;
        cout << "\nMATRIX ERROR: " << errormsg << "\nexiting to system\n";
        cerr << "\nMATRIX ERROR: " << errormsg << "\nexiting to system\n";
        x = 2.0;
        exit(1);
        return x;
}

///// Matrix integrity
void Vector::Garbage(const char *errormsg) const
{
#ifndef NO_CHECKING
        int errorint = 0;
        if (signature != SIGNATURE) errorint++;
        if (n <= 0) errorint++;
        if ( !errorint )
     if ( signature != c[-1] ) errorint++;
        if (errorint) {
          cout << "Operating on a Corrupted matrix: " << "\n";
          Nrerror(errormsg);
        }
#endif
}

///// index
FLOAT &Vector::operator () (BIGINT i) const
{
        return c[((BIGINT) i)];
}

///// output
ostream &operator << (ostream &s, Vector &v)
{
#ifndef NO_CHECKING
        v.Garbage("Vector <<: v is garbage");
#endif

        s << "CanDelete: " << (int) v.CanDelete << " CanAlias: "
          << (int) v.CanAlias << " c:\t" << v.c << "\n";

        int kk = Getwid();
        int kkp = Getdec();
        // save stream settings that are modified
        char c = s.fill(' ');
        int kw = s.width( kk );
        int kp = s.precision( kkp );
        long sf = s.setf(ios::right, ios::adjustfield);
        s.setf( ios::fixed );

        s << endl;
        for (BIGINT i = 0; i < v.n; i++) {
                s << setw(kk) << v(i) << " ";
                if (!(i % 7) && i > 0) s << endl;
        }
        s << endl;
        // restore stream settings
        s.fill( c );
        s.width(kw);
        s.precision(kp);
        s.setf( sf );
        return s;
}


///// memory allocator
FLOAT HUGE_PTR *Vector::SetupVectors(BIGINT nn)
{

#ifdef USE_EXTENDER
        static unsigned long myconst =  0x3FFFFFFDL; //32768*32768 - 2;
#else
        static unsigned long myconst =
                  (65536L / ((unsigned long) sizeof (FLOAT))) - 2;
#endif

        if (nn <= 0) Nrerror("SetupVectors: n < 0");
        if (nn > myconst){
                 cerr << "n tried, max possible: " << nn << " " << myconst << endl;
                Nrerror("SetupVectors: too many elements in vector");
         }

        n = nn;
        c = new HUGE_PTR FLOAT [n + 1];
        if ( !c ) Nrerror("SetupVectors: allocation error");
        c[0] = signature = SIGNATURE;
        c++;
        c[0] = 1;
        return c;
}

///// memory deletion
void Vector::PurgeVectors(void)
{
        signature++;
        if (*-- c == SIGNATURE) {
                c[0] = 0;
                delete[] c;
        }
        else
                Vector::Nrerror("PurgeVectors: attempting to delete a deleted vector");
}

////////// constructors
///// default constructor
Vector::Vector(void) : n(1), CanDelete(TTRUE), CanAlias(TTRUE),
                  signature(SIGNATURE)
{
        c = SetupVectors(1);
}

///// allocate an n vector
Vector::Vector(BIGINT nn) : n(nn), CanDelete(TTRUE), CanAlias(TTRUE),
                  signature(SIGNATURE)
{
        c = SetupVectors(nn);
}
Vector::Vector(int r, int cc) : CanDelete(TTRUE), CanAlias(TTRUE), signature(SIGNATURE)
{
        if( r <= 0 || cc <= 0 )
          Vector::Nrerror("Negative index to Matrix constructor");
        n = ((BIGINT)r)*((BIGINT)cc);
        c = SetupVectors(n);
}

///// copy an array into a vector
Vector::Vector(BIGINT nn, FLOAT *x) : n(nn), CanDelete(TTRUE),
                  CanAlias(TTRUE), signature(SIGNATURE)
{
        c = SetupVectors(nn);
        FLOAT HUGE_PTR *cc = c;
        FLOAT HUGE_PTR *acc = x;
        BIGINT i = n;
        while (i--) *cc++ = *acc++;
}
Vector::Vector(BIGINT nn, FLOAT x) : n(nn), CanDelete(TTRUE),
                  CanAlias(TTRUE), signature(SIGNATURE)
{
        c = SetupVectors(nn);
        FLOAT HUGE_PTR *cc = c;
        BIGINT i = n;
        while (i--) *cc++ = x;
}
Vector::Vector(int r, int ccc, FLOAT *x) :  CanDelete(TTRUE),
                  CanAlias(TTRUE), signature(SIGNATURE)
{
        if( r <= 0 || ccc <= 0 )
          Vector::Nrerror("Negative index to Matrix constructor");
        n = ((BIGINT)r)*((BIGINT)ccc);
        c = SetupVectors(n);
        FLOAT HUGE_PTR *cc = c;
        FLOAT HUGE_PTR *acc = x;
        BIGINT i = n;
        while (i--) *cc++ = *acc++;
}
Vector::Vector(int r, int ccc, FLOAT x) :  CanDelete(TTRUE),
                  CanAlias(TTRUE), signature(SIGNATURE)
{
        if( r <= 0 || ccc <= 0 )
          Vector::Nrerror("Negative index to Matrix constructor");
        n = ((BIGINT)r)*((BIGINT)ccc);
        c = SetupVectors(n);
        FLOAT HUGE_PTR *cc = c;
        BIGINT i = n;
        while (i--) *cc++ = x;
}

///// copy constructor
Vector::Vector(const Vector &a) : n(a.n), CanDelete(TTRUE), CanAlias(TTRUE),
                  signature(SIGNATURE)
{
        if (a.CanDelete == FFALSE && a.CanAlias ) {
                // a is not responsible for deleting c and a can have an alias
                c = a.c;
        }
        else {
          c = SetupVectors(a.n);
          FLOAT HUGE_PTR *cc = c;
          FLOAT HUGE_PTR *acc = a.c;
          BIGINT i = n;
                while (i--) *cc++ = *acc++;
        }
}

///// destructor
Vector::~Vector()
{
        n = 0;
        if (CanDelete == TTRUE )
                PurgeVectors();
}

////// assignment
Vector &Vector::operator = (const Vector &a)
{
#ifndef NO_CHECKING
        a.Garbage("Vector Assignment: a is garbage");
        Garbage("Vector Assignment: *this is garbage");
#endif
        if (this == &a) return *this;
        // deal with aliased storage
        if (c == a.c && a.CanDelete == FFALSE) {
                // *this is aliased with a, and a is not
                // responsible for deleting c.
                CanDelete = TTRUE;
                n = a.n;
                return *this;
        }
        if (c == a.c && a.CanDelete == TTRUE) {
                // *this is aliased with a, and a is
                // responsible for deleting c.
                c = SetupVectors(a.n);
                CanDelete = TTRUE;
        }
        // *this is no longer aliased with a
        if (n != a.n) {
                PurgeVectors();
                c = SetupVectors(a.n);
        }
        FLOAT HUGE_PTR *cc = c;
        FLOAT HUGE_PTR *acc = a.c;
        BIGINT i = n;
        while (i--) *cc++ = *acc++;
        return *this;
}

//////////////////////// matrix class

///// indexing
FLOAT &Matrix::operator () (BIGINT i) const
{
//allow indexing of a matrix as if it were a vector
#ifndef NO_CHECKING
        if ( i < 1 || i > rows() )
                Nrerror("operator(): index out of range");
#endif
        return Vector::operator () ( i-1 );
}


FLOAT &Matrix::operator () (int i, int j) const
 {
#ifndef NO_CHECKING
        if (i < 1 || i > r || j < 1 || j > c)
                Nrerror("operator(): index out of range");
#endif
        BIGINT li = i-1;
        BIGINT lj = j-1;
        BIGINT lc = c;
        return Vector::operator () ( li * lc + lj);
}

///// assignment
Matrix &Matrix::operator = (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix Assignment: a is garbage");
        Garbage("Matrix Assignment: *this is garbage");
#endif
        r = a.r;
        c = a.c;
        this->Vector::operator = (a);
        return *this;
}

///// output
//////////////////////////// display
int Setwid(int w)
{
        static int wid = 6;
        wid = (w < 0) ? wid : w;
        return wid;
}
int Setdec(int d)
{
        static int dec = 3;
        dec = (d < 0) ? dec : d;
        if (d >= Getwid()) dec = Getwid() - 1;
        dec = (dec <= 0) ? 0 : dec;
        return dec;
}
int Getwid(void)
{
        static int wid = 6;
        wid = Setwid(- 1);
        return wid;
}
int Getdec(void)
{
        static int dec = 3;
        dec = Setdec(-1);
        return dec;
}


ostream &operator << (ostream &s, const Matrix &m)
{
#ifndef NO_CHECKING
        m.Garbage("<<: m is garbage");
#endif
        int kk = Getwid();
        int kkp = Getdec();
        // save stream settings that are modified
        char c = s.fill(' ');
        int kw = s.width( kk );
        int kp = s.precision( kkp );
        long sf = s.setf( ios::right, ios::adjustfield);
        s.setf( ios::fixed );

        s << endl;
        for (int i = 1; i <= m.r; i++) {
                for (int j = 1; j <= m.c; j++){
                        s << setw( kk ) << m(i, j) << " ";
                }
                s << endl;
        }
        s << endl;
        // restore stream settings on exit.
        s.fill( c );
        s.width(kw);
        s.precision(kp);
        s.setf( sf );
        return s;
}

void Matrix::show(const char *str)
{
#ifndef NO_CHECKING
        Garbage("show: *this is garbage");
#endif
        cout << str << endl;
        cout << *this;
}

//////////// matrix operators

///// addition
Matrix operator + (const Matrix &a,const Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix +: a is garbage");
        b.Garbage("Matrix +: b is garbage");
#endif
        if (a.r != b.r || a.c != b.c)
                a.Nrerror("matrices do not conform in addition");

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) + b(i, j);
        c.release();
        return c;
}

Matrix operator + (const Matrix &a, FLOAT b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix +: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) + b;
        c.release();
        return c;
}

Matrix operator + (FLOAT b, const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix +: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) + b;
        c.release();
        return c;
}

Matrix &Matrix::operator += (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix +=: a is garbage");
        Garbage("Matrix +=: *this is garbage");
#endif
        if (a.r != r || a.c != c)
          a.Nrerror("matrices do not conform in addition");

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) +=  a(i,j);

        return *this;
}

Matrix &Matrix::operator += (FLOAT b)
{
#ifndef NO_CHECKING
        Garbage("Matrix +=: *this is garbage");
#endif

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) +=  b;

        return *this;
}

///// subtraction
Matrix operator - (const Matrix &a, const Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix -: a is garbage");
        b.Garbage("Matrix -: b is garbage");
#endif
        if (a.r != b.r || a.c != b.c)
                a.Nrerror("matrices do not conform in subtraction");

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) - b(i, j);
        c.release();
        return c;
}

Matrix operator - (const Matrix &a, FLOAT b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix -: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) - b;
        c.release();
        return c;
}

Matrix operator - (FLOAT b, const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix -: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = b - a(i, j);
        c.release();
        return c;
}

Matrix operator - (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix Unary -: a is garbage");
#endif
        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = -a(i, j);
        c.release();
        return c;
}

Matrix &Matrix::operator -= ( const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix -=: a is garbage");
        Garbage("Matrix -=: *this is garbage");
#endif
        if (a.r != r || a.c != c)
                a.Nrerror("matrices do not conform in subtraction");

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) -=  a(i,j);

        return *this;
}

Matrix &Matrix::operator -= (FLOAT b)
{
#ifndef NO_CHECKING
        Garbage("Matrix -=: *this is garbage");
#endif

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) -=  b;

        return *this;
}

///// multiplication
Matrix operator *(const Matrix &a, const Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix *: a is garbage");
        b.Garbage("Matrix *: b is garbage");
#endif
        if (a.c != b.r)
                a.Nrerror("matrices do not conform in multiplication");

        Matrix c(a.r, b.c);
        LFLOAT sum;
        for (int i = 1; i <= a.r; i++)
                for (int j = 1; j <= b.c; j++) {
                        sum = 0;
                        for (int k = 1; k <= a.c; k++)
                        sum += (LFLOAT) ((LFLOAT) a(i, k)) *((LFLOAT) b(k, j));
                 c(i, j) =(FLOAT) sum;
        }
        c.release();
        return c;
}

Matrix operator *(const Matrix &a, FLOAT b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix *: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) *b;
        c.release();
        return c;
}

Matrix operator *(FLOAT b, const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix *: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) *b;
        c.release();
        return c;
}

 Matrix &Matrix::operator *= (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix *=: a is garbage");
        Garbage("Matrix *=: *this is garbage");
#endif
        if (a.r != c)
                a.Nrerror("matrices do not conform in multiplication");

        LFLOAT sum;
        Matrix cc( r, a.c );
        for (int i = 1; i <= cc.r; i++)
                for (int j = 1; j <= cc.c; j++){
                         sum = 0;
                         for( int k=1; k<=c; k++)
                          sum += (LFLOAT) ((LFLOAT) (*this)(i, k)) *((LFLOAT) a(k, j));
                  cc(i, j) = (FLOAT) sum;
        }
        *this = cc;
        return *this;
}

Matrix &Matrix::operator *= (FLOAT b)
{
#ifndef NO_CHECKING
        Garbage("Matrix *=: *this is garbage");
#endif

        for (int i = 1; i <= r; i++)
          for (int j = 1; j <= c; j++) (*this)(i, j) *=  b;

        return *this;
}

//////////////// elementwise multiplication
Matrix operator % (const Matrix &a, const Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix %: a is garbage");
        b.Garbage("Matrix %: b is garbage");
#endif

        if (a.r != b.r || a.c != b.c)
                a.Nrerror("matrices do not conform in elementary multiplication");

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) *b(i, j);
        c.release();
        return c;
}

Matrix operator % (const Matrix &a, FLOAT b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix %: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) *b;
        c.release();
        return c;
}

Matrix operator % (FLOAT b, const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix %: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++) c(i, j) = a(i, j) *b;
        c.release();
        return c;
}

Matrix &Matrix::operator %= (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix %=: a is garbage");
        Garbage("Matrix %=: *this is garbage");
#endif
        if (a.r != r || a.c != c)
                a.Nrerror("matrices do not conform in elementwise multiplication");

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) *=  a(i,j);

        return *this;
}

Matrix &Matrix::operator %= (FLOAT b)
{
#ifndef NO_CHECKING
        Garbage("Matrix %=: *this is garbage");
#endif

        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= c; j++) (*this)(i, j) *=  b;

        return *this;
}

///// elementwise division
Matrix operator / (const Matrix &a, const Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix /: a is garbage");
        b.Garbage("Matrix /: b is garbage");
#endif

        if (a.r != b.r || a.c != b.c)
                a.Nrerror("matrices do not conform in division");

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++)
                        if (b(i, j) != 0.0 ) c(i, j) = a(i, j) / b(i, j);
                        else b.Nrerror("zero divisor in elementwise division");
        c.release();
        return c;
}

Matrix operator / (const Matrix &a, FLOAT b)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix /: a is garbage");
#endif

        if (b == 0.0) a.Nrerror("Matrix /: b is zero");

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++)
                        c(i, j) = a(i, j) / b;
        c.release();
        return c;
}

Matrix operator / (FLOAT b, const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix /: a is garbage");
#endif

        Matrix c(a.r, a.c);
        for (int i = 1; i <= c.r; i++)
          for (int j = 1; j <= c.c; j++)
                        if (a(i, j) != 0.0 ) c(i, j) = b / a(i, j);
                        else
                                a.Nrerror("zero divisor in elementwise division");
        c.release();
        return c;
}

Matrix &Matrix::operator /= (const Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Matrix /=: a is garbage");
        Garbage("Matrix /=: *this is garbage");
#endif
        if (a.r != r || a.c != c)
          a.Nrerror("matrices do not conform in elementwise division");

        for (int i = 1; i <= r; i++)
          for (int j = 1; j <= c; j++)
                 (*this)(i, j) /= ( a(i,j) != 0.0 ) ? a(i,j) :
                                        Nrerror("zero divisor in elementwise division");

        return *this;
}

Matrix &Matrix::operator /= (FLOAT b)
{
#ifndef NO_CHECKING
        Garbage("Matrix /=: *this is garbage");
#endif
        if (b == 0.0) Nrerror("zero divisor in elementwise division");

        for (int i = 1; i <= r; i++)
          for (int j = 1; j <= c; j++) (*this)(i, j) /=  b;

        return *this;
}

/////////////////// matrix functions

//// sweep out rows k1 through k2,  operates on "a"
Matrix Sweep(int k1, int k2, Matrix& a)
{
#ifndef NO_CHECKING
        a.Garbage("Sweep: a is garbage");
#endif

        double eps = 1.0e-8;
        LFLOAT d;
        int i, j, k, n, it;

        if (a.c != a.r)
          a.Nrerror("sweep: matrix a not square");

        n = a.r;
        if (k2 < k1) { k = k1; k1 = k2; k2 = k; }

        for (k = k1; k <= k2; k++) {
          if ( fabs( a(k, k) ) < eps)
                 for (it = 1; it <= n; it++)
                         a(it, k) = a(k, it) = 0.0;
          else {
                 d = 1.0 / a(k, k);
                 a(k, k) = d;
                 for (i = 1; i <= n; i++) 
                     if (i != k) 
                        a(i, k) *= (LFLOAT) - d;
                 for (j = 1; j <= n; j++) 
                     if (j != k)
                        a(k, j) *= (LFLOAT) d; 
                 for (i = 1; i <= n; i++) {
                        if (i != k) {
                                for (j = 1; j <= n; j++) {
                                  if (j != k)
                                         a(i, j) += a(i, k) *a(k, j) / d;
                                } // end for j
                        } // end for i != k
                 } // end for i
          } // end else
        } // end for k
        return a;
}   /*sweep*/

///// inversion
Matrix Inv(Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Inv: a is garbage");
#endif

        if (a.c != a.r)
          a.Nrerror("INVERSE: matrix a not square");

        Matrix b = a;
        Sweep(1, b.r, b);
        b.release();
        return b;

}   /* inverse */


///////////////////////// functions

/////// transpose
// Note for complex numbers, Tran produces
// the Hermitian transpose

Matrix Tran(Matrix &a)
{
#ifndef NO_CHECKING
        a.Garbage("Tran: a is garbage");
#endif

        Matrix c(a.c, a.r);
        for (int i = 1; i <= c.r; i++)
                for (int j = 1; j <= c.c; j++)
                         c(i, j) = conj( a(j, i) );
        c.release();
        return c;
}

#ifdef USE_COMPLEX
Matrix Conj( Matrix &a)
{
#ifndef NO_CHECKING
  a.Garbage("Conj");
#endif
  Matrix t(a.r, a.c);
  for( BIGINT i=1; i<=t.rows(); i++)
      t(i) = conj(a(i));
  t.release();
  return t;
}

Matrix Norm( Matrix &a)
{
#ifndef NO_CHECKING
  a.Garbage("Norm");
#endif
  Matrix t(a.r, a.c);
  for( BIGINT i=1; i<=t.rows(); i++)
      t(i) = complex(norm(a(i)),0.0);
  t.release();
  return t;
}

Matrix Real( Matrix &a)
{
#ifndef NO_CHECKING
  a.Garbage("Real");
#endif
  Matrix t(a.r, a.c);
  for( BIGINT i=1; i<=t.rows(); i++)
      t(i) = complex(real(a(i)),0.0);
  t.release();
  return t;
}

Matrix Imag( Matrix &a)
{
#ifndef NO_CHECKING
  a.Garbage("Imag");
#endif
  Matrix t(a.r, a.c);
  for( BIGINT i=1; i<=t.rows(); i++)
      t(i) = complex(0.0,imag(a(i)));
  t.release();
  return t;
}

Matrix Arg( Matrix &a)
{
#ifndef NO_CHECKING
  a.Garbage("Arg");
#endif
  Matrix t(a.r, a.c);
  for( BIGINT i=1; i<=t.rows(); i++)
      t(i) = complex(arg(a(i)),0.0);
  t.release();
  return t;
}
//end use_complex
#endif


////// fill with a constant
Matrix Fill(int r, int cc, FLOAT d)
{
        if (r < 1 || cc < 1) Matrix::Nrerror("invalid indices in Fill");

        Matrix c(r, cc);
        for (int i = 1; i <= r; i++)
                for (int j = 1; j <= cc; j++) c(i, j) = d;
        c.release();
        return c;
}

//////// identity
Matrix Ident(int n)
{
        if (n < 1)
          Matrix::Nrerror("n < 1 in identity matrix");

        Matrix c(n, n);
        for (int i = 1; i <= c.r; i++)
          for (int j = 1; j <= c.c; j++) c(i, j) = (i == j) ? 1 : 0;
        c.release();
        return c;
}

/////// extract a submatrix
Matrix Submat(Matrix &a, int lr, int lc, int r, int c)
{
#ifndef NO_CHECKING
        a.Garbage("Submat: a is garbage");
#endif
        if ((r > a.r) || (c > a.c))
          Matrix::Nrerror("SUBMAT: index out of range");

        int r2 = r - lr + 1;
        int c2 = c - lc + 1;
        Matrix b( r2, c2);
        int lrm1 = lr - 1;
        int lcm1 = lc - 1;
        for (int i = 1; i <= r2; i++) {
                for (int j = 1; j <= c2; j++) b(i, j) = a(lrm1 + i, lcm1 + j);
        }
        b.release();
        return b;
}   /* submat */

/////  horizontal concatination
Matrix Ch(Matrix &a, Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Ch");
        b.Garbage("Ch");
#endif

        if (a.r != b.r)
                a.Nrerror("CH: matrices have different number of rows");

        Matrix c(a.r, a.c+b.c);

        for (int i = 1; i <= a.r; i++) {
                for (int j = 1; j <= a.c; j++)
                        c(i, j) = a(i, j);
        }
        int ii;
        for (i = 1, ii = 1; i <= b.r; i++, ii++) {
                for (int j = 1, jj = 1; j <= b.c; j++, jj++)
                        c(ii, jj + a.c) = b(i, j);
        }
        c.release();
        return c;
} /* ch */

///// vertical concatination
Matrix Cv(Matrix &a, Matrix &b)
{
#ifndef NO_CHECKING
        a.Garbage("Cv");
        b.Garbage("Cv");
#endif

        if (a.c != b.c)
                a.Nrerror("CV: matrices have different number of columns");

        Matrix c( a.r+b.r, a.c);

        for (int i = 1; i <= a.r; i++) {
                for (int j = 1; j <= a.c; j++)
                        c(i, j) = a(i, j);
        }
        int ii;
        for (i = 1, ii = 1; i <= b.r; i++, ii++) {
                for (int j = 1, jj = 1; j <= b.c; j++, jj++)
                        c(ii + a.r, jj) = b(i, j);
        }
        c.release();
        return c;
} /* cv */

/////////////// io

//////// read a matrix
Matrix Reada(char *filename)
{
        ifstream fp(filename);
    char s[241];
        int i, j, r, c;
        FLOAT f;

        if (!fp) {
          cerr << "Cannot open " << filename << ".\n";
          Matrix::Nrerror("ReadA: file open failure");
        }
        fp.getline(s, 240, '\n');
        fp >> r >> c;
        Matrix a(r, c);
        for (i = 1; i <= r; i++) {
          for (j = 1; j <= c; j++)
                 if (fp >> f) a(i, j) = f;
                 else a.Nrerror("Reada: error reading input");
        }
        fp.close();
        a.release();
        return a;
}


Matrix Readb(char *filename)
{
        ifstream fp(filename, ios::binary | ios::in );
        int i, j, r, c;
        FLOAT f;

        if (!fp) {
          cerr << "Cannot open " << filename << ".\n";
          Matrix::Nrerror("Readb: file open failure");
        }
        fp.read( (char *) &i, sizeof(i));
        char *buffer = new char[i+1];
        if( !buffer || i < 0) Matrix::Nrerror("Readb: cannot read filename buffer");
        fp.read( buffer, i*sizeof(char));
        delete[] buffer; //toss it
        
        fp.read( (char *) &r, sizeof(r));
        fp.read( (char *) &c, sizeof(c));
        Matrix a(r, c);
        for (i = 1; i <= r; i++) {
          for (j = 1; j <= c; j++) {
                 if (fp.read( (char *) &f, sizeof(f)) )
                    a(i, j) = f;
                 else 
                 {  fp.close(); 
                    a.Nrerror("Reada: error reading input");
                 }
          } 
        }
        fp.close();
        a.release();
        return a;
}


/////////// write a matrix
void Writea(char *filename, const Matrix &a, const char *filetitle)
{
#ifndef NO_CHECKING
        a.Garbage("Writea: a is garbage");
#endif
        ofstream fp(filename);
        int i, j;

        if (!fp) {
          cerr << "Cannot open " << filename << ".\n";
          a.Nrerror("Writea: file open failure");
        }
        fp << filetitle << "\n";
        fp << a.r << " " << a.c << "\n";
        for (i = 1; i <= a.r; i++) {
          for (j = 1; j <= a.c; j++)
                 if (!(fp << a(i, j) << " ")){
                    fp.close();
                        a.Nrerror("error writing output");
                 }      
          fp << "\n";
        }
        fp.close();
}

void Writeb(char *filename, const Matrix &a, const char *filetitle)
{
#ifndef NO_CHECKING
        a.Garbage("Writea: a is garbage");
#endif
        ofstream fp(filename, ios::binary | ios::out );
        int i, j;

        if (!fp) {
          cerr << "Cannot open " << filename << ".\n";
          a.Nrerror("Writeb: file open failure");
        }
        i = strlen(filetitle);
        
        fp.write( (char *) &i, sizeof(i));
        fp.write( filetitle, i*sizeof(char));
        
        fp.write( (char *) &a.r, sizeof(a.r));
        fp.write( (char *) &a.c, sizeof(a.c));
        for (i = 1; i <= a.r; i++) {
          for (j = 1; j <= a.c; j++)
                 if ( !(fp.write( (char *) &a(i, j), sizeof(FLOAT)) ) ){
                    fp.close();
                        a.Nrerror("error writing output");
                 }      
        }
        fp.close();
}

