// Version: 2.1
// Author: Mark Von Tress, Ph.D.
// Date: 01/07/96

// Copyright(c) Mark Von Tress 1996


// DISCLAIMER: THIS PROGRAM IS PROVIDED AS IS, WITHOUT ANY
// WARRANTY, EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO FITNESS FOR A PARTICULAR PURPOSE. THE AUTHOR DISCLAIMS
// ALL LIABILITY FOR DIRECT OR CONSEQUENTIAL DAMAGES RESULTING
// FROM USE OF THIS PROGRAM.

//
// here is a c++ matrix program for fitting a circle
// using linear least squares.
// The linear parametrization form of the circle is found in
// Protter-Morrey, Analytical Geometry, Addison Wesley,
// Chapter 5.
//
//    0 = (x^2+y^2)+Dx+Ey+F
//
// This equations determines a circle when D, E, and F
// are known. The circle is centered on (-D/2, -E/2), and
// has radius r = 0.5*(D^2 + E^2 - 4F)
//
// Fit using least squares
//
//     z = -(x^2+y^2) = F + Dx + Ey
//
// This is an easy way to do it. You just minimise the sum
// of squared residuals [ (z-F-Dx-Ey) ]. Note that this may
// be accomplished with zero error if there are 3 points not
// on a line. Suppose I have 3 points (-3,4), (4,5), and (1,4)
// Then I have a system of equations to solve:
//
//  -3D + 4E + F = -25
//   4D + 5E + F = -41
//    D - 4E + F = -17
//
// or in matrix terms
//
//   | -3  4  1 | |D| = |-25|
//   |  4  5  1 | |E| = |-41|
//   |  1 -4  1 | |F| = |-17|
//
// where the symbol | represents a matrix bar, rather than
// a determinant symbol. Note this is just a linear equation
// Ax = B, so the solution vector x is x = Inverse(A)*B.
// For this problem D=-2, E=-2 and F = 23, so the circle is
// centered on (1,1) with a radius of 5.
//
// Now suppose I have more than 3 points. Is there a way
// that I can find the vector x so that it minimizes the
// residual sum of squared elements of the vector B-Ax.
// The Moore Penrose inverse of A does this, and it is
// in fact the normal least squares equations
// that give the best solution:
//
// xhat = Inverse(Tran(A)*A)*Tran(A)*B
//
// where Tran is matrix transpose, and Inverse is matrix
// inverse.
//
// I use sweep to do this. It is a modification of
// gaussian elimation with some interesting twists when
// used in linear regression.
//
//
#include <matv.h>
#include <matvgraf.h>
#include <strtype.h>
#include <urandvec.h>
#include <conio.h>
#include <stdlib.h>
#include <time.h>

#define twopi 6.283185307

void getxy(Matrix beta, Matrix &x, Matrix &y, Matrix &Z)
{
   // create a circle centered at (-D/2,E/2) with radius
   // 0.5*sqrt(D*D+E*E-4*F) and toss in some noise.
   NormalRV z;
   double D = beta( 1 );
   double E = beta( 2 );
   double F = beta( 3 );
   double xcenter = -0.5*D;
   double ycenter = -0.5*E;
   double radius  = 0.5*sqrt(D*D+E*E-4*F);
   double xx = 0;
   double increment = twopi/((double)x.r);
   for( int cnter=1; cnter <= x.r; cnter++ ){
      x( cnter ) =  xcenter + radius*cos( xx );
      y( cnter ) =  ycenter + radius*sin( xx );
      Z( cnter ) = 1*z();
      xx += increment;
   }
   //uncomment this for an exact fit.
   //Z = Fill(x.r,1,0);
}
void graphit(int n, Matrix &beta, Matrix &betahat, Matrix &Z)
{
   Matrix x(n,1),y(n,1),z(n,1);
   Matrix xh(n,1),yh(n,1);
   Matrix one = Fill(n,1,1);

   getxy(beta,x,y,z);
   GMatrix g( Ch(x,y));
   getxy(betahat,xh,yh,z);
   *g.PathToDriver = "c:\\bc45\\bgi";
   g.AddVec( Ch(xh,yh), '+' );
   g.AddVec( Ch(x, Z), -'*' );
   g.Show();
}

main()
{
  int n=40;
  Matrix x(n,1), y(n,1), z(n,1);
  srand( (unsigned) time(NULL) );
  Matrix beta(3,1);
  beta(1) = -2;
  beta(2) =  4;
  beta(3) = -20;

  getxy(beta, x, y, z);
  Matrix xyz = Ch( x, y );
  xyz = Ch( xyz, Fill(n,1,1));
  Matrix xy = xyz;
  // combine the linear portion with the noise
  z += xy*beta;
  xyz = Ch( xyz, z );
  cout << xyz;
  Matrix b = Tran(xyz)*xyz;
  // sweep out intercept to variance of z's
  Sweep( 3,3, b);
  double totalssq = b(4,4);
  // now finish the inversion by sweeping out D and E
  Sweep( 1,2, b);

  Matrix betahat = Submat(b, 1,b.c, b.r-1, b.c);
  // residuals
  cout << z - xy*betahat;
  double D = betahat( 1 );
  double E = betahat( 2 );
  double F = betahat( 3 );
  double xcenter = -0.5*D;
  double ycenter = -0.5*E;
  double radius  = 0.5*sqrt(D*D+E*E-4*F);
  double sse = b(b.r,b.r);
  double mse = sse/((double)( xy.r - xy.c ));
  double rsquare = 1 - sse/totalssq;
  cout << b;
  cout << xcenter << " " << ycenter << " " << radius << " "
       << sqrt(mse) << " " << rsquare << endl;
  getch();
  graphit(n, beta, betahat, z );
  return 0;
}