/*                                                                    */
/* SELECT                                                             */
/*                                                                    */
/* Reads all lines from the standard input and puts selected lines    */
/* on the standard output.                                            */
/*                                                                    */
/* Usage:                                                             */
/*                                                                    */
/*     select [options]                                               */
/*                                                                    */
/* Options:                                                           */
/*                                                                    */
/*     -f****   **** is the first column to be printed                */
/*     -t****   **** is the last column to be printed (0 .. all)      */
/*     -m?      ? = i: include lines that match the pattern           */
/*              ? = e: exclude lines that match the pattern           */
/*     -cf****  **** is the first column to be checked                */
/*     -ct****  **** is the last column to be checked                 */
/*     -b?      ? is a non-blank character to be used as blank in     */
/*              the search-pattern string (must precede -p flag)      */
/*     -p&      & is a (char)-pattern to be checked                   */
/*     -?       prints help-text                                      */
/*                                                                    */
/* Defaults:                                                          */
/*                                                                    */
/*     -f1                                                            */
/*     -t79                                                           */
/*     -me                                                            */
/*     -cf10                                                          */
/*     -ct20                                                          */
/*     -b                                                            */
/*     -p                                                  */
/*                                                                    */
/* Example:                                                           */
/*                                                                    */
/*     pstat /c | select 2>nul | more  ... a quick-n-dirty way to     */
/*                                         show all running processes */
/*                                                                    */
/* Remarks:                                                           */
/*                                                                    */
/*     select writes a couple of messages to stderr, that's why you   */
/*     should redirect the stderr to a file - or to the nul-device    */
/*     as in the above example!                                       */
/*                                                                    */
/* (c) Franz Steiger, 1995                                            */
/* e8525101@student.tuwien.ac.at                                      */
/*                                                                    */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define MAXCOLS  300

/* functions */

void usage(void);
void substr(char[],char[],int,int);
                    /* substr(target,source,from_col,to_col);    */
                    /* cols are counted from 1 to strlen(source) */

/* usage */
/* usage */
/* usage */

void usage(void)
{
printf("This program reads all lines from the standard input and   \n");
printf("puts selected lines on the standard output.                \n");
printf("                                                           \n");
printf("Usage:       select [options]                              \n");
printf("Options:                                                   \n");
printf("    -f***    *** is the first column to be printed         \n");
printf("    -t***    *** is the last column to be printed (0 = all)\n");
printf("    -m?      ? = i: include lines that match the pattern   \n");
printf("             ? = e: exclude lines that match the pattern   \n");
printf("    -cf***   *** is the first column to be checked         \n");
printf("    -ct***   *** is the last column to be checked          \n");
printf("    -b?      ? is a non-blank character to be used as blank\n");
printf("             in the search-pattern string (must precede -p "
                                                             "flag)\n");
printf("    -p&      & is a (char)-pattern to be checked           \n");
printf("    -?       prints help-text                              \n");
printf("Defaults:    -f1 -t79 -me -cf10 -ct20 -b -p    \n");
printf("                                                           \n");
printf("select writes a couple of messages to stderr, that's why you\n");
printf("should redirect the stderr to a file - or to the nul-device!\n");
printf("Example:     pstat /c | select 2>nul | more                \n");
printf("                                                           \n");
printf("(c) Franz Steiger, 1995, e8525101@student.tuwien.ac.at      \n");
}/* end usage */


/* main */
/* main */
/* main */

int main(int argc, char **argv)
{
int  from, to;                /* cols to print              */
int  ie;                      /* 1 .. include, 2 .. exclude */
int  pat_from, pat_to;        /* cols to check for pattern  */
int  pat_len;                 /* actual length of pattern   */
char blank;                   /* character to use as blank  */
char pattern [MAXCOLS];

char line_in [MAXCOLS];
char line_out[MAXCOLS];
char test    [MAXCOLS];
char option;
char option_c;
char number[5];
long i,j,k,l;

/* Defaults */

from        = 1;
to          = 79;
pat_from    = 10;
pat_to      = 20;
ie          = 2; /* exclude */
blank       = '';
strcpy(pattern,"           ");
pat_len     = 11;

/* Get parameters */

if (argc > 1)
{
   for (i=1;i<argc;i++)
   {
      if (argv[i][0] != '-')
      {
         printf("Illegal option: %s\n\n", argv[i]);
         printf("Use 'select -?' to show options!\n");
         return -5;
      }
      option = argv[i][1];
      switch(option)
      {
         case 'c':/* set cols for pattern */
         {
            k = strlen(argv[i]);
            if (k < 3)
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("-c must be followed by f or t and "
                      "a column number!\n");
               return -5;
            }
            option_c = argv[i][2];
            switch(option_c)
            {
               case 'f':/* from col */
               {
                  strcpy(number, "0000");
                  if ((k>7) || (k<4))
                  {
                     printf("Illegal option: %s\n", argv[i]);
                     printf("Illegal col-number (from)!\n");
                     return -4;
                  }
                  l = 3;
                  for (j=k-1;j>2;j--)
                  {
                     if (!isdigit(argv[i][j]))
                     {
                        printf("Illegal option: %s\n", argv[i]);
                        printf("-cf must be followed by an integer!\n");
                        return -5;
                     }
                     else
                        number[l--] = argv[i][j];
                  }
                  sscanf(number, "%d", &pat_from);
                  break;
               }
               case 't':/* to col  */
               {
                  strcpy(number, "0000");
                  if ((k>7) || (k<4))
                  {
                     printf("Illegal option: %s\n", argv[i]);
                     printf("Illegal col-number (to)!\n");
                     return -4;
                  }
                  l = 3;
                  for (j=k-1;j>2;j--)
                  {
                     if (!isdigit(argv[i][j]))
                     {
                        printf("Illegal option: %s\n", argv[i]);
                        printf("-ct must be followed by an integer!\n");
                        return -5;
                     }
                     else
                        number[l--] = argv[i][j];
                  }
                  sscanf(number, "%d", &pat_to);
                  break;
               }
               default:
               {
                  printf("Illegal option: %s\n", argv[i]);
                  printf("-c must be followed by f or t and "
                         "a column number!\n");
                  return -5;
               }
            }
            break;
         }
         case 'f':/* from col */
         {
            strcpy(number, "0000");
            k = strlen(argv[i]);
            if ((k>6) || (k<3))
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("Illegal col-number (from)!\n");
               return -4;
            }
            l = 3;
            for (j=k-1;j>1;j--)
            {
               if (!isdigit(argv[i][j]))
               {
                  printf("Illegal option: %s\n", argv[i]);
                  printf("-f must be followed by an integer!\n");
                  return -5;
               }
               else
                  number[l--] = argv[i][j];
            }
            sscanf(number, "%d", &from);
            break;
         }
         case 't':/* to col  */
         {
            strcpy(number, "0000");
            k = strlen(argv[i]);
            if ((k>6) || (k<3))
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("Illegal col-number (to)!\n");
               return -4;
            }
            l = 3;
            for (j=k-1;j>1;j--)
            {
               if (!isdigit(argv[i][j]))
               {
                  printf("Illegal option: %s\n", argv[i]);
                  printf("-t must be followed by an integer!\n");
                  return -5;
               }
               else
                  number[l--] = argv[i][j];
            }
            sscanf(number, "%d", &to);
            break;
         }
         case 'm':/* mode i or e */
         {
            k = strlen(argv[i]);
            if (k != 3)
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("-m must be followed by i or e!\n");
               return -5;
            }
            if (argv[i][2] == 'i') ie = 1;
            else if (argv[i][2] == 'e') ie = 2;
            else
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("-m must be followed by i or e!\n");
               return -5;
            }
            break;
         }
         case 'b':/* blank */
         {
            k = strlen(argv[i]);
            if (k != 3)
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("-b must be followed by exactly "
                      "one character!\n");
               return -5;
            }
            blank = argv[i][2];
            break;
         }
         case 'p':/* pattern */
         {
            k = strlen(argv[i]);
            if (k < 3)
            {
               printf("Illegal option: %s\n", argv[i]);
               printf("No pattern given!\n");
               return -5;
            }
            pat_len = k-2;
            pattern[0] = '\0';
            substr(pattern, argv[i], 3, k);
            for (j=0;j<pat_len;j++)
            {
               if (argv[i][j+2] == blank)
                  pattern[j] = ' ';
            }
            break;
         }
         case '?':/* help */
         {
            usage();
            return 0;
         }
         default:
         {
            printf("Illegal option: %s\n\n", argv[i]);
            printf("Use 'select -?' to show options!\n");
            return -5;
         }
      }
   }
}

if ((to != 0) && (from > to))
{
   printf("Check the -f and -t options ");
   printf("(No cols to print)!\n");
   return -4;
}
if (pat_from > pat_to)
{
   printf("Check the -cf and -ct options ");
   printf("!\n");
   return -4;
}

/* lets go */

while (gets(line_in) != NULL)
{
   line_out [0] = '\0';
   test     [0] = '\0';
   k = strlen(line_in);
   if (k < pat_from)
   {
      fprintf(stderr,"WARNING !! Length of line smaller than the\n"
             "column specified in -cf option !\n");
      fprintf(stderr,"---------> No line printed !\n");
   }
   else
   {
      if (k < pat_to)
      {
         fprintf(stderr,"WARNING !! Length of line smaller than the\n"
                "column specified in -ct option !\n");
         substr(test,line_in,pat_from,k);
      }
      else
         substr(test,line_in,pat_from,pat_to);
   }
   if (k < from)
   {
      fprintf(stderr,"WARNING !! Length of line smaller than starting\n"
             "column in -f option !\n");
      fprintf(stderr,"---------> No line printed !\n");
   }
   else
   {
      if (to != 0)
         if (to < k) k = to;
      substr(line_out,line_in,from,k);
   }
   if (ie == 1)
      if (strstr(test,pattern) != NULL)
         printf("%s\n",line_out);
      else;
   else
      if (strstr(test,pattern) == NULL)
         printf("%s\n",line_out);
      else;
}


return 0;
}/* end main */

/* substr */
/* substr */
/* substr */

void substr(char t[MAXCOLS], char s[MAXCOLS], int cf, int ct)
/*
* Calculates the substring of s, starting at column cf going to
* column ct and puts it into t
*/
{
int i;
for (i = cf-1; i < ct; i++)
{
   t[i-cf+1] = s[i];
}
t[ct-cf+1] = '\0';
}/* end substr */
