Option Explicit
Option Base 1

'
' The default keywords and tokens
'
Global Const TK_FIRST = 100
Global Const TK_DIM = 100       ' variable definition
Global Const TK_MSGBOX = 101    ' message box dialog
Global Const TK_INPUT = 102     ' input variable
Global Const TK_INTEGER = 103   ' integer variable type
Global Const TK_STRING = 104    ' string variable type
Global Const TK_FLOAT = 105     ' float variable type
Global Const TK_AS = 106        ' var. declaration
Global Const TK_CHR = 107       ' chr() function
Global Const TK_IF = 108        ' if
Global Const TK_THEN = 109      ' then
Global Const TK_GOTO = 110      ' goto
Global Const TK_GOSUB = 111     ' gosub
Global Const TK_RETURN = 112    ' return
Global Const TK_DO = 113
Global Const TK_UNTIL = 114
Global Const TK_END = 999       ' end of program

Global CRLF      As String      ' Cr/Lf
Global token     As Integer
Global KeyWord   As String
Global FirstPass As Integer     ' label location pass

Global Const err_title = "Interpreter Error"
Global Const err_stop = MB_OK + MB_ICONSTOP

'
' First pass label declaration scan variables
'
Type LabelStruct

    Label   As String * 16  ' the label identifier
    Cursor  As Integer      ' the code location

End Type
Global Labels() As LabelStruct  ' array of labels
Global ixLabel  As Integer      ' label index

Global Const MAX_STACK = 32
'
' Gosub..Return Stack and Stack Pointer
'
Global ReturnStack(0 To 32) As Integer
Global UntilStack(0 To 32)  As Integer
Global NextStack(0 To 32)   As Integer

'
' This code implements the DIM command
'
'   ctl is the text box control
'   ii is the current character index into the program
'   buf is the program text
'
Sub Dimension (ctl As TextBox, ii As Integer, ByVal buf As String)

Dim varname As String   ' variable name
Dim errmsg  As String   ' error message


    ' look for variable name

    KeyWord = NextToken(ii, buf, token)
    Monitor token, KeyWord

    ' assume ...

    varname = KeyWord

    '
    ' if this token is exists, then it can't be a new identidier
    '
    If Not Expected(ctl, ii, NT_TOKEN_NOTFOUND) Then Exit Sub
    
    '
    ' now, look for the "AS" keyword
    '
    KeyWord = NextToken(ii, buf, token)
    Monitor token, KeyWord

    If Not token = TK_AS Then

	MsgBox "Expected '" & GetTokenKeyword(TK_AS) & "'", err_stop, err_title
	SetError ctl, ii, token, KeyWord

    Else

	KeyWord = NextToken(ii, buf, token)
	Monitor token, KeyWord

	Select Case token

	    Case TK_INTEGER

		errmsg = NewVariable(varname, VTINTEGER, "0")
		If Len(errmsg) > 0 Then GoTo DimError

	    Case TK_STRING

		errmsg = NewVariable(varname, VTSTRING, "")
		If Len(errmsg) > 0 Then GoTo DimError

	    Case TK_FLOAT

		errmsg = NewVariable(varname, VTFLOAT, "0.0")
		If Len(errmsg) > 0 Then GoTo DimError

	Case Else

	    MsgBox "Expected variable data type", err_stop, err_title
	    SetError ctl, ii, token, KeyWord

	    Exit Sub

	End Select
    
    End If

    Monitor -999, "Variable Declared"

GoTo DimExit

DimError:

    MsgBox errmsg, err_stop, err_title
    SetError ctl, ii, token, KeyWord

DimExit:

End Sub

Sub DoDo (ctl As TextBox, ii As Integer, ByVal buf As String)

    If Not Push(ii, UntilStack()) Then

	MsgBox "Do: Out of Stack Space", err_stop, err_title
	SetError ctl, ii - (Len(KeyWord) + 1), token, KeyWord

    End If

End Sub

'
' Implements the Gosub .. Return construct
'
Sub DoGosub (ctl As TextBox, ii As Integer, ByVal buf As String)

  Dim DType      As Integer

    ' collect the label

    KeyWord = NextToken(ii, buf, token)

    If FirstPass Then Exit Sub
	
    ' error if not a defined label

    If Not Expected(ctl, ii, NT_LABEL_FOUND) Then Exit Sub

    ' push the return location onto the stack
    ' if successful, then 'jump' to subroutine location
    
    If Push(ii, ReturnStack()) Then

	ii = Val(GetVariable(KeyWord, DType))

    Else

	MsgBox "Gosub: Out of Stack Space", err_stop, err_title
	SetError ctl, ii - (Len(KeyWord) + 1), token, KeyWord
	    
    End If

End Sub

Sub DoLoop (ctl As TextBox, ii As Integer, ByVal buf As String)

  Dim Expr      As String
  Dim LExpr     As String
  Dim Op        As String
  Dim RExpr     As String
  Dim rc        As Integer
  Dim Success   As Integer

    Expr = GetNumEvalText(ii, buf)

    ' SplitExpression returns false on error (it's located in this module)

    If FirstPass Then Exit Sub

    If SplitExpression(Expr, LExpr, Op, RExpr) Then

	' test the expression

	rc = TestNumExpr(LExpr, Op, RExpr, Success)

	If Not Success Then ' an error in one of the expressions

	    MsgBox EvalErrorString(rc), err_stop, err_title

	    ' this attempts an approximation of where
	    ' the error likely is.

	    SetError ctl, ii - (Len(KeyWord) + 1), token, Expr

	    Exit Sub

	End If

	If Not rc Then ' expression is true

	    Success = Pop(UntilStack())
	    If Success = -1 Then
	    
		MsgBox "Until: Stack Underflow.", err_stop, err_title
		SetError ctl, ii - (Len(KeyWord) + 1), token, Expr

	    Else

		rc = Push(Success, UntilStack())
		ii = Success
	    
	    End If

	End If

    Else ' bad relational expression
	
	MsgBox "Invalid relational expression", err_stop, err_title
	SetError ctl, ii - (Len(KeyWord) + 1), token, Expr

    End If

End Sub

'
' EquateVariable
'
' This code demonstrates a method for setting variables to a value
'
Sub EquateVariable (ctl As TextBox, ii As Integer, ByVal buf As String)

Dim rc      As Integer  ' return code
Dim varname As String   ' variable name
Dim vdata   As String   ' variable data
Dim vtype   As Integer  ' variable type
Dim evalbuf As String   ' evaluator buffer
Dim errmsg  As String   ' evaluator error message

    ' first assume that the variable name
    ' was just encountered, so use the current keyword

    varname = KeyWord

    '
    ' now get the "=" operator
    '
    GetToken ii, buf
    
    If Not Expected(ctl, ii, NT_EQUAL) Then ' the "=" is not there

	MsgBox "Expected '='", err_stop, err_title
	SetError ctl, ii, token, KeyWord

    Else
    
	' gather the current data and type of the variable

	vdata = GetVariable(varname, vtype)

	Select Case vtype

	    ' handle numbers

	    Case VTINTEGER, VTFLOAT

		' parse the text for a number

		vdata = GetNumericStr(ctl, ii, buf)

		If Not token = NT_USER_ERROR Then ' good number

		    ' call VBossAPI and assign the variable it's data

		    rc = SetVariable(varname, vdata)

		    If rc < 0 Then ' oops

			MsgBox "Numeric variable data error", err_stop, err_title
			SetError ctl, ii, token, KeyWord

		    End If
		
		End If
		

	    ' handle strings
	    
	    Case VTSTRING
		
		' parse the text for a string

		vdata = GetString(ctl, ii, buf)
		
		If Not token = NT_USER_ERROR Then ' good string

		    ' assign the variable it's data

		    rc = SetVariable(varname, vdata)

		    If rc < 0 Then ' oops

			MsgBox "String variable data error", err_stop, err_title
			SetError ctl, ii, token, KeyWord

		    End If

		End If

	End Select

    End If

End Sub

'
' Expected()
'
'   This function checks the expected token with the current token.
'   If they are not equal, then a standard error dialog is displayed,
'   and the SetError() procedure is called to signal program failure.
'
Function Expected (ctl As TextBox, ii As Integer, expect As Integer) As Integer

Dim txt As String   ' for building the error message

    ' return TRUE if a match

    If token = expect Then

	Expected = True

    Else

	' build the error message text

	If expect > 0 And expect < NT_MAX_OPERATORS Then

	    '
	    '  An operator. Get a copy of the operator text.
	    '
	    txt = "'" & Mid$(NT_Operators(), expect, 1) & "'"

	ElseIf expect < 0 Then

	    '
	    '  A specific token error type
	    '
	    Select Case expect

		Case NT_PAST_EOL

		    txt = "End of Line"

		Case NT_TOKEN_NOTFOUND

		    txt = "Unique Identifier"

		Case NT_LABEL_FOUND

		    txt = "Label"

		Case NT_VARIABLE_FOUND

		    txt = "Variable"

		Case Else

		    txt = GetTokenKeyword(expect)

		    If txt = "" Then txt = "Identifier"

	    End Select

	Else
	    
	    '
	    ' A programmer-defined keyword
	    '
	    txt = "'" & GetTokenKeyword(expect) & "'"

	End If
	
	' display message

	MsgBox "Expected " & txt, err_stop, err_title
	
	' define error and set cursor in <ctl> textbox

	SetError ctl, ii, token, KeyWord

	' error return

	Expected = False
    
    End If

End Function

'
' GetChar()
'
'   Returns the character representation of the numeric expression
'   located next in the program stream.
'
Function GetChar (ctl As TextBox, ii As Integer, ByVal buf As String) As String

Dim vdata   As String   ' variable data
Dim rc      As Integer  ' return code

    GetChar = ""

    ' evaluate a numeric expression and return
    ' as a string (ie: "123")

    vdata = GetNumericStr(ctl, ii, buf)

    If Not token = NT_USER_ERROR Then ' successful evaluation

	' now validate a character value of 0..255

	If Val(vdata) > 255 Or Val(vdata) < 0 Then

	    MsgBox "Invalid character value", err_stop, err_title
	    SetError ctl, ii, token, KeyWord

	Else

	    ' return the character

	    GetChar = Chr(Val(vdata))

	End If

    End If

End Function

'
' Follow the input stream until the end of an expression
'
Function GetNextExpression (ii As Integer, buf As String, retval As String) As Integer
Dim Looping As Integer
Dim jj      As Integer
Dim ix      As Integer
Dim tkn     As Integer
Dim ch      As String * 1
Dim work$
Dim temp$

    jj = ii

    ' no more than to end of line
    work$ = ParseUntil(ii, buf, Chr(13) & Chr(10))
    ii = jj

    ix = 1

    Looping = (ix < Len(work$))

    While Looping

	jj = ix
	temp$ = NextToken(ix, work$, tkn)

	If tkn = NT_DBL_QUOTE Then

	    ix = ix + 1
	    If ParseUntil(ix, work$, """") = "" Then ix = ix


	End If

	Looping = (ix < Len(work$)) And Not (tkn > TK_FIRST)

    Wend

    

End Function

'
' GetNumericStr()
'
'   Evaluates the program stream for a valid numeric expression
'   and returns the result as a string (ie: "123")
'
Function GetNumericStr (ctl As TextBox, ii As Integer, ByVal buf As String) As String

Dim errmsg  As String   ' returned error message
Dim vdata   As String   ' variable data
Dim rc      As Integer  ' return code

    ' assume error

    GetNumericStr = ""


    '
    ' First, test for built-in functions (ie: MsgBox)
    '
    KeyWord = PeekNextToken(ii, buf, token)

    Select Case token

	Case TK_MSGBOX
	    
	    GetToken ii, buf    ' get the MsgBox
	    GetToken ii, buf    ' now get the "("
	    
	    If Not Expected(ctl, ii, NT_LEFTPAREN) Then
		
		Exit Function
	    
	    Else

		MessageBox ctl, ii, buf, rc
		GetNumericStr = Str(rc)

	    End If

	    GetToken ii, buf

	    If Not Expected(ctl, ii, NT_RIGHTPAREN) Then
		
		Exit Function

	    End If
	    
	    Exit Function

    End Select


    ' collect and evaluate the numeric expression

    vdata = EvalExpression(GetNumEvalText(ii, buf), rc)

    ' check for errors

    If Not rc Then
		    
	errmsg = EvalErrorString(rc)
	MsgBox errmsg, err_stop, err_title
	SetError ctl, ii, token, KeyWord
		
    Else

	' groovy

	GetNumericStr = vdata
	Monitor -999, "NUMERIC EXPRESSION"

    End If

End Function

'
' Wrapper to handle the parsing of numeric evaluation text from
' the program stream.
'
'   This is necessary because of the way ParseUntil() works.  Since
'   it is a word-related function, it returns -1 in ii if there are
'   no more word-blocks in the stream.  We want to retain the
'   value of ii so we can mark syntax errors in the source.
'
Function GetNumEvalText (ii As Integer, ByVal buf As String) As String
Dim jj As Integer
Dim evalbuf As String

    jj = ii
    evalbuf = ParseUntil(ii, buf, Chr(10) & Chr(13) & "&,:;'")
    ii = jj + Len(evalbuf)

    GetNumEvalText = evalbuf

End Function

'
' GetString()
'
'   Parses the program string for constants and variables.
'
'   Numeric constants are not allowed, but numeric variables are.  This
'   means that no conversion functions are required in this implementation
'   since all numeric variables are retrieved as text.
'
Function GetString (ctl As TextBox, ii As Integer, ByVal buf As String) As String

Dim st      As String   ' the string
Dim vtype   As Integer  ' the variable type

    st = ""

    Do

	GetToken ii, buf
	

	'
	'  Test for concantenation characters
	'
	If token = NT_PLUS Or token = NT_AMPERSAND Then
	
	    ' skip the operator
	    
	    GetToken ii, buf
	
	End If
	
	'
	' handle four character types:
	'
	'   'text'
	'   "text"
	'   variable
	'   chr(nn)
	'
	Select Case token

	    ' 'text'

	    Case NT_SNG_QUOTE

		st = st & GetStringConst(ctl, ii, buf, NT_SNG_QUOTE)

		If token = NT_USER_ERROR Then Exit Function

	    ' "text"
	    Case NT_DBL_QUOTE

		st = st & GetStringConst(ctl, ii, buf, NT_DBL_QUOTE)

		If token = NT_USER_ERROR Then Exit Function
	    
	    ' variable
	    Case NT_VARIABLE_FOUND

		st = st & GetVariable(KeyWord, vtype)

		Monitor -999, VTypeName(vtype) & " VARIABLE"

	    ' chr(nn)
	    Case TK_CHR

		st = st & GetChar(ctl, ii, buf)

		If token = NT_USER_ERROR Then Exit Function

		Monitor -999, "CHARACTER"


	Case Else

	    MsgBox "Expected Constant or Variable", err_stop, err_title
	
	    SetError ctl, ii, token, KeyWord

	    Exit Function

	End Select

	' check next token. Is it a + or &?
	KeyWord = PeekNextToken(ii, buf, token)
    
    Loop While token = NT_PLUS Or token = NT_AMPERSAND

    ' return the string

    GetString = st
    

End Function

'
' GetStringConstant()
'
'   Expect and collect a string constant
'
Function GetStringConst (ctl As TextBox, ii As Integer, ByVal buf As String, target As Integer) As String

Dim jj      As Integer
Dim quote   As String
Dim msg     As String
	    
    GetStringConst = ""

    Select Case target
	
	Case NT_SNG_QUOTE

	    quote = "'"
	
	Case NT_DBL_QUOTE

	    quote = """"
    
    End Select

    ' save the current pointer
	    
    jj = ii

    ' use ParseUntil() to grab up to the end of line

    msg = ParseUntil(jj, buf, Chr(10) & Chr(13) & quote)

    ' now reposition out pointer to the end of the string
	    
    ii = ii + Len(msg)

    ' Test the next character.
    ' Is it the quote or end of line?

    If Len(msg) = 0 Then

	GoTo SkipQuote

    ElseIf Not Mid$(buf, jj + 1, 1) = quote Then
    '
    ' Oops. Not the right quote!
    '
	MsgBox "Expected terminating quote (" & quote & ")", err_stop, err_title
	SetError ctl, ii + 1, token, KeyWord  ' token will be set to NT_USER_ERROR
		
    Else

SkipQuote:
	
	ii = ii + 1 ' skip the quote
	GetStringConst = msg
	Monitor -999, "STRING CONSTANT"

    End If

End Function

'
' This wrapper procedure is used to collect the
' next token and do monitoring processes.
'
' By placing them here, a monitor can be maintained while developing the
' language, then removed with a comment character
'
' Remember, token and keyword are GLOBAL varaiables.
'
Sub GetToken (ii As Integer, ByVal buf As String)
    
    KeyWord = NextToken(ii, buf, token)
    Monitor token, KeyWord

End Sub

'
' If <true_expression> then goto <label>
'
'
' This simplistic example demonstrates how you might use
' relational expressions and labels to implement flow control.
'
' It also demonstrates the new VBossAPI function TextNumExpr()
'
Sub IfThenElse (ctl As TextBox, ii As Integer, ByVal buf As String)

Dim LExpr   As String   ' left side of equation
Dim RExpr   As String   ' right side of equation
Dim Op      As String   ' operator
Dim rc      As Integer  ' test return value (true/false)
Dim Success As Integer  ' test execution success flag
Dim Expr    As String   ' parsed expression
Dim vtype   As Integer  ' (unused) variable type holder

    
    'Expr = GetNumEvalText(ii, buf)
    Expr = ""


    ' uses NextToken() to scan the text for expression candidates

    KeyWord = NextToken(ii, buf, token)

    '
    ' while a possible operator, variable etc ... scan for expression
    '
    While token < NT_MAX_OPERATORS And ii < Len(buf)

	Expr = Expr & KeyWord & " "
	KeyWord = NextToken(ii, buf, token)

    Wend


    ' now split the expression into left, operator, right
    ' SplitExpression returns false on error (it's located in this module)

    If SplitExpression(Expr, LExpr, Op, RExpr) Then

	' test the expression

	rc = TestNumExpr(LExpr, Op, RExpr, Success)

	If Not Success Then ' an error in one of the expressions

	    MsgBox EvalErrorString(rc), err_stop, err_title

	    ' this attempts an approximation of where
	    ' the error likely is.

	    SetError ctl, ii - (Len(KeyWord) + 1), token, Expr

	    Exit Sub

	Else

	    If Not Expected(ctl, ii, TK_THEN) Then Exit Sub ' MUST be followed by THEN
	    Monitor token, KeyWord

	    ' collect the GOTO keyword
	    
	    GetToken ii, buf
	    If Not Expected(ctl, ii, TK_GOTO) Then Exit Sub ' MUST be followed by GOTO

	    ' collect the jump label

	    GetToken ii, buf
	    
	    If FirstPass Then Exit Sub
	    
	    If Not Expected(ctl, ii, NT_LABEL_FOUND) Then Exit Sub ' MUST be followed by label

	    ' remember rc above?
	    '
	    ' if the result of the relation test is true, then
	    ' lets jump to the labeled line

	    If rc Then
	    
		' Here we set the "program counter" to the value stored in the label
		' This has the effect of "jumping" to that new location

		ii = Val(GetVariable(KeyWord, rc))

	    End If

	End If

    Else ' bad relational expression
	
	MsgBox "Invalid relational expression", err_stop, err_title
	SetError ctl, ii - (Len(KeyWord) + 1), token, Expr

    End If
	

End Sub

'
' This is the main procedure of the program.
'
' Pass the TextBox control that contains the program:
'    ie:   Interpret Editor
'
Function Interpret (program As TextBox) As Integer

Dim script  As String   ' buffers the program
Dim ii      As Integer  ' index/iterator to script
Dim rc      As Integer  ' generic return code
Dim comment As String   ' comment string
Dim jj      As Integer  ' temp index holder
Dim Label   As String
Dim DType   As Integer

    
    Interpret = False
    
    script = program.Text   ' copy the program
    OSSMain.Monitor.Clear   ' clear the execution monitor list

    program.SelStart = 0
    program.SelLength = 0


    ' first sanity check: any thing to do?
    If Len(script) = 0 Then

	MsgBox "Nothing to execute.", MB_OK + MB_ICONSTOP, "Interpreter Error"
	Exit Function

    Else

	' parse the first command

	ii = 0      ' initialize the character pointer to start interpreting
	token = 0   ' null token

	Do

	    GetToken ii, script  ' parse next keyword

	    Select Case token
		
		Case NT_COLON

		'
		' comment line
		'
		Case NT_SNG_QUOTE

		    ' parse until end of the line

		    comment = ParseUntil(ii, script, Chr(13) & Chr(10))

		'
		' MsgBox <msg> [, <options>[,<title>]]
		'
		Case TK_MSGBOX

		    MessageBox program, ii, script, rc
		    GoTo NextLoop
		
		
		'
		' DIM <varname> AS INTEGER|STRING|FLOAT
		'
		Case TK_DIM
		
		    Dimension program, ii, script
		    GoTo NextLoop
		
		
		'
		' INPUT [<prompt.,] <varname>
		'
		Case TK_INPUT
		
		    QueryVariable program, ii, script
		    GoTo NextLoop

		'
		' <varname> = <expression>
		'
		Case NT_VARIABLE_FOUND

		    comment = GetVariable(KeyWord, DType)
		    If DType = VTLABEL Then GoTo NextLoop
		    
		    EquateVariable program, ii, script
		
		    GoTo NextLoop

		Case TK_DO

		    DoDo program, ii, script

		Case TK_UNTIL

		    DoLoop program, ii, script

		'
		' if ... then ...
		'
		Case TK_IF

		    IfThenElse program, ii, script

		'
		' gosub
		'
		Case TK_GOSUB

		    DoGosub program, ii, script

		Case TK_RETURN

		    If Not FirstPass Then
			
			rc = Pop(ReturnStack())
			
			If rc < 0 Then
			
			    MsgBox "Return: Stack Underflow.", err_stop, err_title
			    SetError program, ii, token, KeyWord
			    Exit Function

			End If

			ii = rc
		    
		    End If
		'
		' goto ...
		'
		Case TK_GOTO

		    GetToken ii, script

		    ' ignore label is this is the first pass

		    If Not FirstPass Then

			comment = GetVariable(KeyWord, DType)
			
			If DType = VTLABEL Then

			    ii = CInt(comment)

			Else

			    MsgBox "Syntax Error: Label expected.", err_stop, err_title
			    SetError program, ii, token, KeyWord
			    Exit Function
			
			End If

		    End If

		Case NT_LABEL_FOUND
		'
		' <label>:
		'
		' skip labels when encountered while running
		'
		    GetToken ii, script ' just skip the label:

		'
		' <label>:
		'
		Case NT_TOKEN_NOTFOUND
		    
		  If FirstPass Then
		    
		    If PeekNextToken(ii, script, token) = ":" Then

			Label = KeyWord
			GetToken ii, script
			rc = AddVariable(Label, VTLABEL, Str(ii))

			ixLabel = ixLabel + 1
			Labels(ixLabel).Label = Label
			Labels(ixLabel).Cursor = ii

			If rc > -1 Then

			    Monitor -999, "LABEL DEFINED @ " & ii

			End If

		    Else

			GoTo SyntaxError

		    End If
		
		  
		  End If

		
		' END or execution error

		Case TK_END, NT_PAST_EOL, NT_USER_ERROR

		    If FirstPass Then
		    
			If token = TK_END Then GoTo NextLoop
			
			If token = NT_USER_ERROR Then
			    
			    Interpret = False
			    Exit Function

			End If

		    End If

		    Monitor -999, "End of Program."
		    Interpret = True
		    Exit Function

	    '
	    ' Un-recognized command
	    '
	    Case Else

SyntaxError:
		MsgBox "Syntax Error: Unknown command '" & KeyWord & "'", err_stop, err_title
		SetError program, ii, token, KeyWord
		Exit Function

	    End Select

NextLoop:

	Loop While Not (token = NT_USER_ERROR)

    End If

    Interpret = True

End Function

Function IsExprToken (tkn As Integer) As Integer

    IsExprToken = True

    Select Case token

	Case NT_PAST_EOL, NT_LABEL_FOUND, NT_PROCEDURE

	    IsExprToken = False

	Case Is > NT_MAX_OPERATORS

	    IsExprToken = False
    
    End Select

End Function

'
' MessageBox()
'
'   This code implements the MsgBox command
'
Sub MessageBox (ctl As TextBox, ii As Integer, ByVal buf As String, key As Integer)

Dim msg     As String   ' the message
Dim opts    As Integer  ' the button options
Dim title   As String   ' the title
		     

    ' some assumtions, since title and opts are optional
    title = "Message Box"
    opts = MB_ICONEXCLAMATION
    
    ' collect the message

    msg = GetString(ctl, ii, buf)
    If token = NT_USER_ERROR Then Exit Sub

    
    ' is there an opts parameter?

    KeyWord = PeekNextToken(ii, buf, token)

    ' if a comma is found, then there is

    If KeyWord = "," Then
    
	GetToken ii, buf

	opts = Val(GetNumericStr(ctl, ii, buf))
	If token = NT_USER_ERROR Then Exit Sub

	'
	' Is there a title parameter?
	'
	KeyWord = PeekNextToken(ii, buf, token)
	
	'
	'  if so, collect it
	'
	If KeyWord = "," Then

	    GetToken ii, buf

	    title = GetString(ctl, ii, buf)
	    If token = NT_USER_ERROR Then Exit Sub
	
	End If

    End If
    
    ' execute the message box

    If FirstPass Then Exit Sub

    key = MsgBox(msg, opts, title)

End Sub

'
' Monitor()
'
'   Maintains the monitor list on the right of the display
'
Sub Monitor (ByVal token As Integer, ByVal KeyWord As String)

    If FirstPass Then Exit Sub

    '
    '  -999 signifies a text message
    '
    If Not token = -999 Then
	'
	'  display the keyword and its class
	'
	OSSMain.Monitor.AddItem KeyWord & Chr(9) & NT_CodeString(token)

    Else
	'
	' keyword, in this case, contains a message
	'
	OSSMain.Monitor.AddItem Chr(9) & KeyWord

    End If

    ' set the highlight to the bottom line

    OSSMain.Monitor.ListIndex = OSSMain.Monitor.ListCount - 1

End Sub

'
' NewVariable()
'
'   This wrapper function attempts to create a new variable.  If unsuccessful,
'   it returns a text error message, otherwise it returns a blank string
'
Function NewVariable (ByVal varname As String, vtype As Integer, ByVal vdata As String) As String

Dim rc  As Integer  ' return code
Dim msg As String   ' error message

    rc = AddVariable(varname, vtype, vdata)

    Select Case rc
	
	Case Is > -1
	' success

	    NewVariable = ""
	    Exit Function
	
	'
	' errors
	'
	Case AKW_NO_MORE_ROOM
	    msg = "No more variable space"
	Case AKW_INVALID_CHAR
	    msg = "Variable contains illegal characters"
	Case AKW_DUPLICATE_KEYWORD
	    msg = "Duplicate variable name"
	Case AKW_KEYWORD_TOO_LONG
	    msg = "Identifier too long"
	Case Else
	    msg = "Variable declaration failure"
    
    End Select

    ' return the formatted error string

    NewVariable = GetTokenKeyword(TK_DIM) & " Error:" & msg

End Function

'
' Pop an integer off the ReturnStack
'
Function Pop (Stack() As Integer) As Integer

    If Stack(0) > 0 Then

	Pop = Stack(Stack(0))
	Stack(0) = Stack(0) - 1

    Else

	Pop = -1

    End If

End Function

'
' This procedure clears all variable, then defines
' the labels located in the first pass
'
' The labels are stored in Labels() as 'LabelStruct's
'
Sub PresetLabels ()

  Dim ii As Integer
  Dim rc As Integer

    ZapVariables

    If ixLabel > 0 Then

	For ii = 1 To ixLabel
		
	    rc = AddVariable(Trim$(Labels(ii).Label), VTLABEL, Str$(Labels(ii).Cursor))
	    
	Next

    End If


End Sub

'
'  Push integer onto the Return Stack
'
Function Push (value As Integer, Stack() As Integer) As Integer

    If Stack(0) < MAX_STACK Then

	Push = True
	Stack(0) = Stack(0) + 1
	Stack(Stack(0)) = value

    Else

	Push = False

    End If

End Function

'
' QueryVariable()
'
'   Implements the INPUT <msg>,<varname> command
'
Sub QueryVariable (ctl As TextBox, ii As Integer, ByVal buf As String)

Dim msg     As String   ' message (or prompt)
Dim varname As String   ' variable name
Dim vdata   As String   ' variable data
Dim vtype   As Integer  ' variable type
Dim errmsg  As String   ' error message
Dim rc      As Integer  ' return code

    ' first parameter is the prompt

    msg = GetString(ctl, ii, buf)
    If token = NT_USER_ERROR Then Exit Sub

    ' now look for the comma

    GetToken ii, buf

    If Expected(ctl, ii, NT_COMMA) Then

	' collect the variable identifier

	GetToken ii, buf
	varname = KeyWord

	'
	' does it exist?
	'
	If Expected(ctl, ii, NT_VARIABLE_FOUND) Then

	    ' locate the variable's type

	    vdata = GetVariable(varname, vtype)

	    ' now collect the input using the
	    ' value of the variable as the default
	    
	    If FirstPass Then Exit Sub
	    
	    vdata = InputBox$(msg, "Data Entry", vdata)

	    '
	    ' Empty input is concidered a cancel request
	    '
	    If vdata = "" Then

		MsgBox "User cancelled or No Input", err_stop, err_title
		SetError ctl, ii, token, KeyWord
		
		Exit Sub

	    End If

	    ' based on the variable type, collect or evaluate
	    ' the input

	    Select Case vtype

		Case VTSTRING
		    
		    ' just copy

		    rc = SetVariable(varname, vdata)
		
		Case VTFLOAT, VTINTEGER
		    
		    ' evaluate the input
		    ' this allows input strings like "1 + sqr(Radius) * 4"
		    ' where Radius is a variable

		    vdata = EvalExpression(vdata, rc)

		    If Not rc Then

			errmsg = EvalErrorString(rc)
			MsgBox errmsg, err_stop, err_title
			SetError ctl, ii, token, KeyWord

		    Else
			
			rc = SetVariable(varname, vdata)
		    
		    End If

	    End Select

	End If

    End If

End Sub

'
' Create the keyword list with the default keywords.
'
Sub SetDefaultKeywords ()
Dim rc As Integer

    rc = AddKeyword("DIM", TK_DIM)          ' variable definition
    rc = AddKeyword("MSGBOX", TK_MSGBOX)    ' message box dialog
    rc = AddKeyword("INPUT", TK_INPUT)      ' input variable
    rc = AddKeyword("INTEGER", TK_INTEGER)  ' INTEGER
    rc = AddKeyword("STRING", TK_STRING)    ' STRING
    rc = AddKeyword("FLOAT", TK_FLOAT)      ' FLOAT
    rc = AddKeyword("AS", TK_AS)            ' var declaration
    rc = AddKeyword("CHR", TK_CHR)          '
    rc = AddKeyword("IF", TK_IF)            '
    rc = AddKeyword("THEN", TK_THEN)
    rc = AddKeyword("GOTO", TK_GOTO)
    rc = AddKeyword("GOSUB", TK_GOSUB)
    rc = AddKeyword("RETURN", TK_RETURN)

    rc = AddKeyword("END", TK_END)          ' end of program

End Sub

'
' This code does two things:
'
'   1.  Sets token to NT_USER_ERROR (defined for this project) and
'       reports a syntax error to the monitor list
'
'   2.  attempts to highlight the source of the error in the TextBox
'       control.  This is not always 100% accurate, but it does help
'       the user debug the program.
'
Sub SetError (program As TextBox, ByVal ii As Integer, token As Integer, ByVal KeyWord As String)

    On Error Resume Next

    If ii < 0 Then
    '
    ' Just go to the end of the program.
    '
	program.SelStart = Len(program.Text)
	program.SelLength = 0

    Else
    '
    ' highlight the token just to the left of the index
    '
	program.SelStart = ii - (Len(KeyWord))
	program.SelLength = Len(KeyWord)

    End If
    
    token = NT_USER_ERROR
    Monitor -999, "Syntax Error"

End Sub

'
' SplitExpression()
'
' This function splits the relational expression into three pieces.
'
' The VBossAPI function TestNumExpr requires Left Expr, Operator and Right Expr strings
'
' This is an example of one way you might implement relational expression parsing and
' implements only two-part truth expressions (ie a=1, myval=true etc)
'
Function SplitExpression (ByVal Expr As String, LExpr As String, Op As String, RExpr As String) As Integer
Dim ii      As Integer  ' local index/iterator
Dim inleft  As Integer  ' in-the-left-expression flag   (you can see my years of assembly)
Dim inright As Integer  ' in-the-right-expression flag  (language peeking through here.  )
Dim inop    As Integer  ' in-the-operator flag
Dim c       As String   ' the current character

    SplitExpression = False ' assume error
    
    ' clear buffers

    LExpr = ""
    Op = ""
    RExpr = ""
    
    ' initialize scanning state flags
    
    inleft = True
    inright = False
    inop = False

    ' minimum expression is something like 1=1
    
    If Len(Expr) < 3 Then Exit Function

    ' we'll scan the entire relational expression

    For ii = 1 To Len(Expr)
					
	c = Mid$(Expr, ii, 1)           ' get the character
	
	inop = InStr("=<>", c) > 0      ' set the in-operator flag
					    

	' now poke

	If inop = True Then ' the last character was a recognized operator
	    
	    If inright Then Exit Function   ' escape if already on the right side
					    ' cause that's an error

	    Op = Op & c         ' add the character (for <>. >= etc)
	    inleft = False      ' no longer on the left side!

	ElseIf inleft Then      ' add the character to the left

	    LExpr = LExpr & c

	Else

	    inright = True      ' must be on the right
	    RExpr = RExpr & c   ' so add the character

	End If

    Next

    SplitExpression = True

End Function

'
' VTypeName()
'
'   Decodes a VT* type value to a string.
'
Function VTypeName (vtype As Integer) As String

    Select Case vtype

	Case VTFLOAT
	    
	    VTypeName = "FLOAT"
	
	Case VTINTEGER
	    
	    VTypeName = "INTEGER"
	
	Case VTSTRING
	    
	    VTypeName = "STRING"
	
	Case VTLABEL
	    
	    VTypeName = "LABEL"

    End Select

End Function

