;----------------------------------------------------------------
; Sade Graphic Mode Screen Saver
; by Glynne Davis
;----------------------------------------------------------------

	      .model    small
	      .stack    100h
	      .code
	      .startup

	      jmp  begin_install

;----------------------------------------------------------------
; Defines
;----------------------------------------------------------------

CR            equ  0Ah
LF            equ  0Dh
EndMsg        equ  24h 

TSRId         equ  193

CheckIn       equ  0                    ; subfunction "check installation"
OnOff         equ  1                    ; subfunction "turn program on"
SetTime       equ  2                    ; subfunction "set to new time delay"
RepSt         equ  3                    ; subfunction "report status"

FALSE         equ  0
TRUE          equ  1

pneeded       equ  4                    ; Number of 16k EMS pages required
pagelen       equ  4000h                ; Length of EMS page (16k)
emmint        equ  67h                  ; EMM software interrupt

iwidth        equ  152                  ; Image width
iheight       equ  116                  ; Image height

;----------------------------------------------------------------
; Resident data
;----------------------------------------------------------------


AddSign       dw 1951
PresSign      dw 7654    

int09h        dd ?
int08h        dd ?
int2Fh        dd ?
int33h        dd ?

IsActive      db   FALSE                ; is the screen blanked
IsEnabled     db   TRUE                 ; is the saver switched on

timercount    db   6

mcount        dw   0                    ; nested INT 33h calls
buttons       dw   ?
vertical      dw   ?
horizontal    dw   ?
mouseflag     db   0

vmode         db   ?                    ; save the current video mode

ss_register   dw   ?                    ; contents of SS register
sp_register   dw   ?                    ; contents of SP register

mystack       db   512 dup (0)          ; internal stack

pframe        dw   0                    ; EMS page frame segment address
handle        dw   0                    ; EMS handle

offsets       dw 0,0,150,70,0,70,150,0  ; sequence of image positions on scr
cur_off       dw   0
request_exit  db   0

minutes       db   ?                    ; Time until activation
time          dw   910                  ; Countdown timer reset value
count         dw   910                  ; Countdown timer

random_val    dw   12345                ; Random number generator seed
random_mul    dw   9421                 ; Random number generator multiplier

randX         dw   ?                    ; Random X coord within the image
randY         dw   ?                    ; Random Y coord within the image

offsetX       dw   50                   ; Random X offset of image on scr
offsetY       dw   50                   ; Random Y offset of image on scr

video_segment dw   0b800h               ; video buffer segment
video_offset  dw   ?                    ; video buffer offset
small_cursor  dw   0607h                ; small cursor shape
big_cursor    dw   0407h                ; big cursor shape
cursor_mode   dw   ?                    ; cursor mode
cursor_pos    dw   ?                    ; cursor position
video_page    db   ?                    ; video page number
video_size    dw   0

log0          dw 0                      ; Logical and Physical page numbers
log1          dw 1
log2          dw 2
log3          dw 3

phy0          dw 0
phy1          dw 1
phy2          dw 2
phy3          dw 3

;----------------------------------------------------------------------
; Initialize variables on each activation
;----------------------------------------------------------------------

initialize    proc near

	      mov  cs:[cur_off],0
	      ret

initialize    endp

;----------------------------------------------------------------------
; Save mouse driver status
;----------------------------------------------------------------------

savemouse     proc near

	      mov  es,cs:[pframe]

	      mov  dx,0c000h

	      mov  ax,0016h
	      int  33h

	      ret

savemouse     endp

;----------------------------------------------------------------------
; Save mouse driver status
;----------------------------------------------------------------------

restoremouse  proc near

	      mov  es,cs:[pframe]

	      mov  dx,0c000h

	      mov  ax,0017h
	      int  33h

	      ret

restoremouse  endp

;----------------------------------------------------------------------
; Test for presence of Expanded Memory Manager
; Checks int 67h to see if it points to an EMM driver
;----------------------------------------------------------------------

EMSpres       proc near

	      xor  bx,bx                ; fetch segment of EMM int vector
	      mov  es,bx
	      mov  es,es:[(emmint*4)+2]

	      mov  di,10                ; offset of name field in drv header

	      mov  si,offset emmname    ; DS:SI = EMM driver name
	      mov  cx,8                 ; length of name field

	      cld
	      repz cmpsb                ; compare names...
	      jnz  pres_abort           ; jump if driver absent

	      clc                       ; driver detected
	      ret

pres_abort:
	      stc                       ; driver absent
	      ret

EMSpres       endp

;----------------------------------------------------------------------
; Check status of Expanded Memory Manager
; Finds total and available EMS pages
;----------------------------------------------------------------------

EMSstat       proc near

	      mov ah,40h                ; 40h: Get EMS system status
	      int 67h
	      or  ah,ah
	      jnz stat_abort

	      mov ah,46h                ; 46h: Check EMS version number
	      int 67h
	      or  ah,ah
	      jnz stat_abort

	      cmp al,32h                ; Compare version number
	      jb  stat_abort

	      mov ah,42h                ; 42h: Find total and available pages
	      int 67h
	      or  ah,ah
	      jnz stat_abort

	      mov tpages,dx             ; save total pages
	      mov apages,bx             ; save available pages

	      cmp bx,pneeded            ; compare with pages needed
	      jb  stat_abort

	      clc                       ; EMS status OK
	      ret               

stat_abort:
	      stc                       ; EMS status problem
	      ret

EMSstat       endp

;----------------------------------------------------------------------
; Allocate EMS pages
; Saves page frame address and EMS handle
;----------------------------------------------------------------------

EMSalloc      proc near

	      mov ah,43h                ; 43h: Allocate needed pages
	      mov bx,pneeded
	      int 67h
	      or  ah,ah
	      jnz alloc_abort

	      mov handle,dx             ; DX = EMM handle

	      mov ah,41h                ; 41h: Get page frame address
	      int 67h
	      or  ah,ah
	      jnz alloc_abort

	      mov pframe,bx             ; BX = page frame

	      clc                       ; EMS allocation OK
	      ret            

alloc_abort:
	      stc                       ; EMS allocation problem
	      ret

EMSalloc      endp

;----------------------------------------------------------------------
; Map EMS pages
; Maps logical pages (0 to 3) to physical pages (0 to 3)
;----------------------------------------------------------------------

EMSmap        proc near

	      mov  ah,44h               ; 44h: Map EMS pages
	      mov  bx,cs:[log0]          
	      mov  al,byte ptr cs:[phy0]
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  map_abort

	      mov  ah,44h               ; 44h: Map EMS pages
	      mov  bx,cs:[log1]
	      mov  al,byte ptr cs:[phy1]
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  map_abort

	      mov  ah,44h               ; 44h: Map EMS pages
	      mov  bx,cs:[log2]
	      mov  al,byte ptr cs:[phy2]
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  map_abort

	      mov  ah,44h               ; 44h: Map EMS pages
	      mov  bx,cs:[log3]
	      mov  al,byte ptr cs:[phy3]
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  map_abort

	      clc                       ; Mapping OK
	      ret

map_abort:
	      stc                       ; Mapping problem
	      ret

EMSmap        endp

;----------------------------------------------------------------------
; Save EMS map
;----------------------------------------------------------------------

EMSsavemap    proc near

	      mov  ah,47h
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  savemap_abort

	      clc                       ; clear carry and return
	      ret

savemap_abort:
	      stc                       ; set carry and return
	      ret

EMSsavemap    endp

;----------------------------------------------------------------------
; Restore EMS map
;----------------------------------------------------------------------

EMSrstmap     proc near

	      mov  ah,48h
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  rstmap_abort

	      clc                       ; clear carry and return
	      ret

rstmap_abort:
	      stc                       ; set carry and return
	      ret

EMSrstmap     endp

;----------------------------------------------------------------------
; Release EMS pages
;----------------------------------------------------------------------

EMSrel        proc near

	      mov  ah,45h
	      mov  dx,cs:[handle]
	      int  67h
	      or   ah,ah
	      jnz  rel_abort

	      clc                       ; clear carry and return
	      ret

rel_abort:
	      stc                       ; set carry and return
	      ret

EMSrel        endp

;----------------------------------------------------------------------
; Clear graphics screen
;----------------------------------------------------------------------

scrclr        proc near

	      push es
	      push ax
	      push cx
	      push di

	      cld

	      mov  ax,0a000h 
	      mov  es,ax
	      mov  di,0         

	      mov  al,0h                ; store 0 in every byte of video mem.

	      mov  cx,320*200
	      rep  stosb

	      pop  di
	      pop  cx
	      pop  ax

	      pop  es

	      ret

scrclr        endp

;----------------------------------------------------------------------
; Transfer image data to EMS
;----------------------------------------------------------------------

imgtrns       proc near 

	      cld

	      mov si,offset image_data  ; DS:SI points to image data

	      mov es,cs:[pframe]        ; ES:DI points to EMS pages
	      mov ax,pagelen
	      mul cs:[phy0]
	      mov di,ax

	      mov cx,768                ; read palette data
	      rep movsb

	      mov cx,iwidth * iheight   ; read image data
	      rep movsb

	      ret

imgtrns       endp

;----------------------------------------------------------------------
; Set VGA palette
;----------------------------------------------------------------------

setpal        proc near

	      push ds                   ; save our data segment

	      push ax
	      push bx
	      push cx
	      push dx
	      push si
	      push di

	      mov  ax,00ffh
	      mov  dx,3c6h
	      out  dx,ax

	      cld

	      mov ax,pagelen            ; DS:SI points to EMS pages
	      mul cs:[phy0]
	      mov si,ax

	      mov ax,cs:[pframe]
	      mov ds,ax

	      mov  cx,256               ; palette data = 256 * 3
	      mov  bx,0

pal:          mov  ax,bx
	      mov  dx,3c8h
	      out  dx,al

	      lodsb
	      sar  ax,2
	      cbw

	      mov  dx,3c9h
	      out  dx,al

	      lodsb
	      sar  ax,2
	      cbw

	      mov  dx,3c9h
	      out  dx,al

	      lodsb
	      sar  ax,2
	      cbw

	      mov  dx,3c9h
	      out  dx,al
	      
	      inc  bx
	      loop pal

	      mov  ax,1001h
	      mov  bx,0
	      int  10h

	      pop  di
	      pop  si

	      pop  dx
	      pop  cx
	      pop  bx
	      pop  ax

	      pop  ds                   ; restore our data segment

	      ret

setpal        endp

;----------------------------------------------------------------------
; Makes image fade out
;----------------------------------------------------------------------

imgfade       proc near

	      mov  ax,0a000h
	      mov  es,ax

	      mov  cx,65000

loop_fd:
	      call random
	      mov  ax,iwidth
	      push ax
	      call modulo
	      mov  cs:[randX],ax

	      call random
	      mov  ax,iheight
	      push ax
	      call modulo
	      mov  cs:[randY],ax

	      mov  ax,cs:[offsetY]      ; offset on screen
	      add  ax,cs:[randY]
	      mov  bx,320               ; 320 * randY + randX
	      mul  bx                
	      add  ax,cs:[offsetX]
	      add  ax,cs:[randX]

	      mov  di,0
	      add  di,ax

	      mov  al,20h

	      stosb

	      loop loop_fd

	      ret

imgfade       endp

;----------------------------------------------------------------------
; Write random dots of image onto screen
;----------------------------------------------------------------------

imgrnd        proc near

	      cmp  cs:[cur_off],16
	      jb   new_offset
	      
	      mov  cs:[cur_off],0

new_offset:
	      mov  bx,cs:[cur_off]

	      mov  ax,word ptr cs:[offsets+bx]
	      mov  cs:[offsetX],ax

	      mov  ax,word ptr cs:[offsets+bx+2]
	      mov  cs:[offsetY],ax

	      call random
	      mov  ax,10
	      push ax
	      call modulo
	      add  [offsetX],ax
	      add  [offsetY],ax

	      add  cs:[cur_off],4


	      mov  ax,0a000h
	      mov  es,ax

	      mov  cx,65353

loop_rnd:
	      call random
	      mov  ax,iwidth
	      push ax
	      call modulo
	      mov  cs:[randX],ax

	      call random
	      mov  ax,iheight
	      push ax
	      call modulo
	      mov  cs:[randY],ax


	      mov  ax,cs:[randY]
	      push ax

	      mov  ax,cs:[randX]
	      push ax

	      call plot

	      mov  ax,cs:[randY]
	      push ax

	      mov  ax,iwidth-1
	      sub  ax,cs:[randX]
	      push ax

	      call plot

	      loop loop_rnd

	      ret
	      
imgrnd        endp

;----------------------------------------------------------------------
; Random number generator
;----------------------------------------------------------------------

random        proc near

	      mov  ax,cs:[random_val]
	      mul  cs:[random_mul]
	      inc  ax
	      ror  ax,1
	      ror  ax,1
	      ror  ax,1
	      ror  ax,1
	      mov  cs:[random_val],ax

	      ret

random        endp

;----------------------------------------------------------------------
; Plot points on the screen
;----------------------------------------------------------------------

plot          proc near

	      push bp
	      mov  bp,sp

	      mov  ax,[bp+6]
	      mov  bx,iwidth
	      mul  bx
	      add  ax,[bp+4]

	      mov  si,768
	      add  si,ax

	      mov  ax,cs:[offsetY]      ; offset on screen
	      add  ax,[bp+6]
	      mov  bx,320               ; 320 * randY + randX
	      mul  bx                
	      add  ax,cs:[offsetX]
	      add  ax,[bp+4]

	      mov  di,ax

	      push ds

	      mov ax,cs:[pframe]
	      mov ds,ax

	      movsb

	      mov  ax,10
	      push ax
	      call sdelay

	      pop  ds

	      pop  bp

	      ret  4

endp          plot

;----------------------------------------------------------------------
; Short delay - a single loop
;----------------------------------------------------------------------

sdelay        proc

	      push bp
	      mov  bp,sp

	      push cx

	      mov  cx,[bp+4]
sdel1:
	      nop

	      loop sdel1

	      pop  cx

	      pop  bp

	      ret  2

sdelay        endp

;----------------------------------------------------------------------
; Long delay - nested loops
;----------------------------------------------------------------------

ldelay        proc

	      push bp
	      mov  bp,sp

	      push cx


	      mov  cx,[bp+6]
ldel2:
	      push cx

	      mov  cx,[bp+4]
ldel1:
	      nop

	      loop ldel1

	      pop  cx

	      loop ldel2

	      pop  cx

	      pop  bp

	      ret  4

ldelay        endp

;----------------------------------------------------------------------
; Performs modulo operation on random_val
; base value is passed on stack
; result is returned in ax
;----------------------------------------------------------------------

modulo        proc near

	      push bp                   ; setup stack frame
	      mov  bp,sp

	      mov  ax,random_val        ; random number in ax
	      mov  bx,[bp+4]            ; base value in bx
	      xor  dx,dx                ; ax sign extended to dx:ax
	      div  bx                   ; quotient in ax, remainder in dx

	      mov  ax,dx
	      pop  bp

	      ret  2

modulo        endp 

;--------------------------------------------------------------------------
; PopUp
; Main routine of active screen saver
;--------------------------------------------------------------------------

popup         proc near

	      cli                       ; switch to internal stack
	      mov  cs:[ss_register],ss
	      mov  cs:[sp_register],sp
	      push cs
	      pop  ss
	      mov  sp,offset mystack+511
	      sti

	      cld

	      push ax                   ; save all registers
	      push bx
	      push cx
	      push dx

	      push ds                   
	      push es

	      push di
	      push si

	      push bp

	      call initialize           ; reset animation sequence etc.

	      call video_info           ; find current video settings

	      jnc  popup_continue
	      jmp  popup_abort

popup_continue:
	      call EMSsavemap           ; save the EMS map

	      call EMSmap               ; map our EMS pages

	      mov  ax,0002h             ; hide the mouse
	      int  33h
	      
	      call save_text            ; save text mode screen contents

	      mov ah,0fh                ; 0fh : get current video mode
	      int 10h
	      mov cs:[vmode],al

	      call savemouse

	      mov  ah,00h               ; 00h : set video mode
	      mov  al,13h               ; video mode 13h = 320 x 200
	      int  10h

	      call setpal               ; set the VGA palette

mainloop:
	      call imgrnd

	      mov  cx,750               ; main idle
popup1:
	      push cx
	      mov  cx,65000

popup2:
	      nop
	      loop popup2

	      pop  cx
					; is there a request to exit?
	      cmp  cs:[request_exit],TRUE
	      je   restore_scr

	      loop popup1

	      call imgfade

	      cmp  cs:[request_exit],TRUE
	      je   restore_scr

	      call scrclr

	      mov  ax,1000              ; delay while image is displayed
	      push ax
	      mov  ax,5000
	      push ax
	      call ldelay

try_request:
	      cmp  cs:[request_exit],TRUE
	      jne  mainloop


restore_scr:
	      mov  ah,00h               ; 00h : set video mode
	      mov  al,cs:[vmode]        ; original video mode
	      int  10h

	      call restore_text         ; restore text mode screen contents

	      mov  ax,1003h             ; enable intensity
	      mov  bl,00h
	      int  10h

	      call restoremouse

	      mov  ax,0001h             ; show the mouse
	      int  33h

	      call EMSrstmap

	      pop  bp

	      pop  si
	      pop  di

	      pop  es
	      pop  ds

	      pop  dx
	      pop  cx
	      pop  bx
	      pop  ax

	      cli                       ; restore DOS stack
	      mov  ss,cs:[ss_register]
	      mov  sp,cs:[sp_register]
	      sti

popup_abort:

	      ret

popup         endp

;--------------------------------------------------------------------------
; Get information about the video environment
;--------------------------------------------------------------------------

video_info    proc near

	      mov  ax,40h
	      mov  es,ax

	      mov  al,es:[49h]          ; Get video mode number in AL

	      cmp  al,2                 ; Go ahead if video mode 2, 3 or 7
	      je   info1

	      cmp  al,3
	      je   info1

	      cmp  al,7
	      je   info1

info_exit:    jmp  info_abort

info1:        mov  ax,es:[4Ah]          ; number of columns on screen
	      cmp  ax,80                ; exit if less than 80 columns
	      jb   info_abort

	      mov  ax,es:[4Eh]          ; start address of current video page
	      mov  cs:[video_offset],ax
	      
					; is this a color adaptor?
	      test byte ptr es:[63h],40h    
	      jnz  info2

	      mov  cs:[video_segment],0b000h
	      mov  cs:[small_cursor],0b0ch
	      mov  cs:[big_cursor],080ch


info2:        
	      mov  ax,es:[60h]          ; cursor parameters
	      mov  cs:[cursor_mode],ax

	      mov  bl,es:[62h]          ; current video page number
	      xor  bh,bh
	      mov  cs:[video_page],bl

	      shl  bx,1
	      mov  ax,es:[bx+50h]       ; 50h: cursor position for 8 video modes
	      mov  cs:[cursor_pos],ax

	      clc
	      ret

info_abort:
	      stc
	      ret

video_info    endp

;--------------------------------------------------------------------------
; Save the text mode screen in EMS
;--------------------------------------------------------------------------

save_text     proc near

	      push ds

	      cld

	      mov  ax,40h
	      mov  es,ax

	      mov  ax,es:[4Ch]
	      mov  cs:[video_size],ax

	      mov  ax,cs:[pframe]
	      mov  es,ax

	      mov  ax,pagelen
	      mul  word ptr cs:[phy2]
	      mov  di,ax     
	      
	      mov  si,cs:[video_offset]
	      mov  ds,cs:[video_segment]

	      mov  cx,cs:[video_size]

	      rep  movsw

	      pop  ds

	      ret

save_text     endp

;--------------------------------------------------------------------------
; Restore the text mode screen from EMS
;--------------------------------------------------------------------------


restore_text  proc near

	      cld

	      push ds

	      mov  ax,cs:[pframe]
	      mov  ds,ax

	      mov  ax,pagelen
	      mul  word ptr cs:[phy2]
	      mov  si,ax

	      mov  di,cs:[video_offset]
	      mov  es,cs:[video_segment]

	      mov  cx,cs:[video_size]

	      rep  movsw

	      mov  ah,02h
	      mov  bh,cs:[video_page]
	      mov  dx,cs:[cursor_pos]
	      int  10h

	      mov  ah,01h
	      mov  cx,cs:[cursor_mode]
	      int  10h

rest_exit:

	      pop  ds

	      ret

restore_text  endp

;--------------------------------------------------------------------------
; Handler for multiplex interrupt 2Fh
;
; ah = TSRId and bx = [AddSign]
; al as a function code
;
;  00h - CheckIn   returns al = FFh and bx = [PresSign]
;  01h - OnOff     dl = Enabled (TRUE/FALSE)
;  02h - SetTime   cx = new time, dl = Enabled (TRUE/FALSE)
;  03h - RepSt     returns ah = [IsEnabled]
;
;--------------------------------------------------------------------------

mplex_int     proc near
	      pushf

	      cmp  ah,TSRId             ; Check for id and signature
	      jne  ToOld2F

	      cmp  bx,cs:[AddSign]
	      jne  ToOld2F

	      cmp  al,CheckIn           ; Installation Check?
	      jne  Other1
	      mov  al,0FFh              ; standard convention for Presence
	      mov  bx,cs:[PresSign]

Other1:       cmp  al,OnOff             ; function "Switch ON?"
	      jne  Other2            
	      
	      mov  cs:[IsEnabled],dl
	      jmp  Exit2F

Other2:       cmp  al,RepSt             ; function "Report state"?
	      jne  Other3
	      mov  ah,cs:[IsEnabled]
	      mov  cx,cs:[time]
	      jmp  Exit2F

Other3:       cmp  al,SetTime           ; function "Set Time"?          
	      jne  Other4
	      mov  cs:[IsEnabled],dl
	      mov  cs:[time],cx
	      mov  cs:[count],cx
	      mov  cs:[timercount],6
	      jmp  Exit2F

Other4:

Exit2F:       popf
	      iret

ToOld2F:      popf
	      jmp     cs:[int2Fh]

mplex_int     endp


;--------------------------------------------------------------------------
; Additional handler for interrupt 08h
;--------------------------------------------------------------------------

timer_int     proc near

	      pushf
	      call cs:[int08h]

	      sti

	      dec  cs:[timercount]
	      jz   time0
	      
	      jmp  ExitTimer

time0:
	      mov  cs:[timercount],6

	      cmp  cs:[IsActive],TRUE
	      je   blanked

not_blanked:
	      call test_mouse
	      jc   Restore

	      cmp  cs:[IsEnabled],TRUE
	      jne  ExitTimer

	      dec  cs:[count]
	      jnz  ExitTimer
	      
	      mov  cs:[IsActive],TRUE
	      mov  cs:[request_exit],FALSE

	      call popup

	      jmp  ExitTimer

blanked:      
	      call test_mouse
	      jc   Reset

	      jmp  ExitTimer

Reset:
	      mov  cs:[IsActive],FALSE
	      mov  cs:[request_exit],TRUE

Restore:
	      push cs:[time]
	      pop  cs:[count]

ExitTimer:
	      iret


timer_int     endp

;--------------------------------------------------------------------------
; Additional handler for interrupt 09h
;--------------------------------------------------------------------------

kb_int        proc    near

	      cmp  cs:[IsActive],TRUE
	      jne  ToOld9

	      mov  cs:[IsActive],FALSE
	      mov  cs:[request_exit],TRUE

ToOld9:
	      push cs:[time]
	      pop  cs:[count]
	      mov  cs:[timercount],6

	      jmp  cs:[int09h]    

kb_int        endp 

;--------------------------------------------------------------------------
; Additional handler for interrupt 33h
;--------------------------------------------------------------------------

mouse_int     proc near

	      inc  cs:[mcount]
	      pushf
	      call cs:[int33h]
	      dec  cs:[mcount]

	      iret

mouse_int     endp

;--------------------------------------------------------------------------
; Checks to see if a mouse event has occurred since last call   
;--------------------------------------------------------------------------

test_mouse    proc near
	      
	      cmp  cs:[mouseflag],1
	      jne  exit_cl
	      cmp  cs:[mcount],0
	      je   cm0
exit_cl:      
	      clc
	      ret

cm0:          push ax
	      push bx
	      push cx
	      push dx

	      pushf
	      mov  ax,03h
	      call cs:[int33h]
	      cmp  bx,cs:buttons
	      jne  cm2
	      cmp  cx,cs:horizontal
	      jne  cm2
	      cmp  dx,cs:vertical
	      jne  cm2

cm1:          clc
	      jmp  short cm3

cm2:          mov  cs:buttons,bx
	      mov  cs:horizontal,cx
	      mov  cs:vertical,dx
	      
set_carry:
	      stc

cm3:          pop  dx
	      pop  cx
	      pop  bx
	      pop  ax

	      ret

test_mouse    endp

;----------------------------------------------------------------------
; Prints a message on the screen
; Assumes that dx contains message offset
;----------------------------------------------------------------------

print         proc near

	      mov  ah,09h
	      int  21h
	      
	      ret

print         endp

;----------------------------------------------------------------------
; Scans command line for a /? switch
;----------------------------------------------------------------------

chkhlp        proc near

	      push si
chklp:
	      inc  si   
	      cmp  byte ptr es:[si],0dh
	      je   chk_exit
	      cmp  byte ptr es:[si],"?"
	      jne  chklp
	      cmp  byte ptr es:[si-1],"/"
	      jne  chklp

	      add  sp,2                 ; clear stack
	      sub  si,2                 ; throw away si which was pushed
	      stc
	      ret

chk_exit:
	      pop  si
	      clc
	      ret

chkhlp        endp

;----------------------------------------------------------------------
; Capitalizes the command line
; Assumes that es points to psp
;----------------------------------------------------------------------

caps          proc near

	      mov  si,81h               ; point SI to command line
	      mov  cl,es:[si-1]         ; cl = Number of characters
	      xor  ch,ch                ; convert byte to word in cx
	      jcxz caps3

caps1:        cmp  byte ptr es:[si],"a"
	      jb   caps2
	      cmp  byte ptr es:[si],"z"
	      ja   caps2
	      and  byte ptr es:[si],0dfh
caps2:        inc  si
	      loop caps1

caps3:
	      ret

caps          endp

;----------------------------------------------------------------------
; Parse command line
; Assumes ES points to the PSP
;----------------------------------------------------------------------

cmd_parse     proc near

	      mov  si,80h               ; reset SI

parse1:       
	      call findchar             ; find next character
	      jnc  parse2               ; branch if not end of line
	      jmp  parse_exit           ; exit the parsing loop
parse2:
	      cmp  byte ptr es:[si],"/"
	      jne  parse1

slasho:                                 ; check for /O switch
	      inc  si
	      mov  al,es:[si]

	      cmp  al,"O"
	      jne  slashm

try_off:                                ; check for "OFF"
	      inc  si
	      mov  al,es:[si]

	      cmp  al,"F"
	      jne  try_on

	      mov  [IsEnabled],FALSE    ; turn program off
	      mov  [set_onoff],TRUE
	      jmp  parse1

try_on:                                 ; check for "ON"
	      cmp  al,"N"
	      jne  parse_error

	      mov  [IsEnabled],TRUE     ; turn program on
	      mov  [set_onoff],TRUE
	      jmp  parse1

slashm:                                 ; check for /M
	      cmp  al,"M"
	      jne  parse_error

	      inc  si
	      mov  al,es:[si]

	      cmp  al,"="               ; check for "= XX"
	      jne  parse_error

	      inc  si
	      call asc2bin
	      jc   parse_error

	      cmp  al,0                 ; minimum time is 1 minute
	      ja   time_ok
	      mov  al,1
time_ok:
	      mov  [set_time],1
	      mov  [minutes],al         ; save number of minutes
	      xor  ah,ah                ; byte to word in ax
	      mov  bx,182               ; multiply by 182
	      mul  bx
	      mov  [time],ax            ; save the result
	      mov  [count],ax
	      dec  si

	      jmp  parse1

parse_exit:
	      clc
	      ret

parse_error:
	      stc
	      ret

cmd_parse     endp

;----------------------------------------------------------------------
; Converts a decimal number in ASCII to a binary value in AL.
;----------------------------------------------------------------------

asc2bin       proc near
	      xor  ax,ax
	      xor  bh,bh
	      mov  dl,10

a2b_loop:
	      mov  bl,es:[si]
	      inc  si
	      cmp  bl,09h               ; exit if tab
	      je   a2b_exit
	      cmp  bl,20h               ; exit if space
	      je   a2b_exit
	      cmp  bl,2ch               ; exit if comma
	      je   a2b_exit
	      cmp  bl,0dh               ; exit if carriage return
	      je   a2b_exit

	      cmp  bl,"0"               ; error if character is not
	      jb   a2b_error            ; a number
	      cmp  bl,"9"
	      ja   a2b_error

	      mul  dl                   ; multiply value in AL by 10
	      jc   a2b_error            ; exit on overflow
	      sub  bl,30h               ; ASCII to binary
	      add  ax,bx                ; add latest value to AX
	      cmp  ax,255               ; error if sum > 255
	      jna  a2b_loop

a2b_error:
	      dec  si                   ; set carry and exit
	      stc
	      ret

a2b_exit:
	      dec  si                   ; clear carry and exit
	      clc
	      ret

asc2bin       endp

;----------------------------------------------------------------------
; Converts a binary value in AX to ASCII form and displays it.
;----------------------------------------------------------------------

bin2asc       proc near
	      mov  bx,10
	      xor  cx,cx
b2a1:         inc  cx
	      xor  dx,dx
	      div  bx
	      push dx
	      or   ax,ax
	      jnz  b2a1
b2a2:         pop  dx
	      add  dl,30h               ; convert it to ASCII
	      mov  ah,2                 ; display it
	      int  21h
	      loop b2a2

	      ret
bin2asc       endp

;----------------------------------------------------------------------
; Advances si to the next non-white space character
;----------------------------------------------------------------------

findchar      proc near

	      inc  si
	      cmp  byte ptr es:[si],09h ; ignore tab
	      je   findchar
	      cmp  byte ptr es:[si],20h ; ignore space
	      je   findchar
	      cmp  byte ptr es:[si],2ch ; ignore comma
	      je   findchar
	      cmp  byte ptr es:[si],0d
	      je   eol

	      clc
	      ret

eol:          stc
	      ret

findchar endp

;----------------------------------------------------------------------
; Check whether the program is already installed
;----------------------------------------------------------------------

check_install proc near

	      mov     ah,TSRId          ; call install check of 2Fh
	      mov     al,CheckIn
	      mov     bx,cs:[AddSign]
	      int     2Fh
	      cmp     al,0FFh
	      jne     not_installed
	      cmp     bx,cs:[PresSign]
	      jne     not_installed

is_installed:
	      stc
	      ret

not_installed:
	      clc
	      ret

check_install endp

;----------------------------------------------------------------------
; Change the elapse time before saver activates
;----------------------------------------------------------------------

change_time   proc near

	      mov  ah,TSRId
	      mov  al,SetTime
	      mov  bx,cs:[AddSign]
	      mov  cx,cs:[time]
	      mov  dl,cs:[IsEnabled]
	      int  2Fh

	      ret

change_time   endp

;----------------------------------------------------------------------
; Turn the saver on or off
;----------------------------------------------------------------------

turn_onoff    proc near

	      mov  ah,TSRId
	      mov  al,OnOff
	      mov  bx,cs:[AddSign]
	      mov  dl,cs:[IsEnabled]
	      int  2Fh

	      ret

turn_onoff    endp

;----------------------------------------------------------------------
; Display information about installed program status
;----------------------------------------------------------------------

get_status    proc near

	      mov  ah,TSRId
	      mov  al,RepSt
	      mov  bx,cs:[AddSign]
	      int  2Fh
	      call showtime
	      mov  ax,4C00h
	      int  21h

	      ret

get_status    endp

;----------------------------------------------------------------------
; Show current time setting
;----------------------------------------------------------------------

showtime      proc near

	      cmp  ah,TRUE
	      je   st1
	      mov  dx,offset disabled
	      call print
	      ret
st1:
	      mov  dx,offset setto
	      call print
	      mov  ax,cx
	      xor  dx,dx
	      mov  bx,182
	      div  bx
	      call bin2asc
	      mov  dx,offset nomin
	      call print

	      ret

showtime      endp

;----------------------------------------------------------------
; Installation section
;----------------------------------------------------------------

GLOBAL C image_data:BYTE

begin_install:

	      mov  ax,cs                ; set DS to code segment
	      mov  ds,ax

	      mov  [PspAddr],es        ; save address of PSP

	      cld                    
	      mov  si,80h               ; SI = command line
	      call chkhlp               ; is there a /? switch
	      jnc  banner

	      mov  dx,offset hlpmsg     ; output help message
	      call print

	      jmp  normal_exit

banner:       mov dx,offset titlemsg    ; Output the program banner
	      call print

	      mov  [IsEnabled],TRUE     ; Set enabled flag

	      mov  ah,30h               ; 30h: get DOS version
	      int  21h
	      cmp  al,3
	      jae  parse

	      lea  dx,dosver
	      jmp  problem_exit

parse:
	      call caps                 ; Convert command line to upper case
	      call cmd_parse            ; Look for cmdline switches
	      jnc  installed 

	      lea  dx,syntaxmsg         ; Syntax error occurred
	      jmp  problem_exit

installed:
	      call check_install
	      jnc  install

	      cmp  [set_time],1
	      jne  try_onoff

	      call change_time

try_onoff:
	      cmp  [set_onoff],1
	      jne  show_status

	      call turn_onoff

show_status:
	      call get_status

	      jmp  normal_exit

					; Go ahead and install
Install:

init1:
	      call EMSpres              ; test for EMS manager present
	      jnc  init2

	      lea  dx,missing
	      jmp  problem_exit

init2:
	      call EMSstat              ; check EMS status
	      jnc  init3

	      lea  dx,status
	      jmp  problem_exit

init3:
	      call EMSalloc             ; allocate EMS pages
	      jnc  init4

	      lea  dx,alloc
	      jmp  problem_exit

init4:
	      call EMSmap               ; map EMS logical pages to physical
	      jnc  init5

	      lea  dx,mapping
	      jmp  problem_exit

init5:
	      call imgtrns              ; transfer image from data segment
	      jnc  init6                ; to EMS memory

	      lea  dx,transfer
	      jmp  problem_exit
init6:

	      mov  ax,00h               ; check for mouse driver
	      int  33h
	      or   ax,ax
	      jz   nomouse1
	      mov  cs:[mouseflag],1

nomouse1:
	      mov  ax,352Fh             ; Hook interrupt 2Fh
	      int  21h
	      mov  word ptr cs:[int2Fh],bx
	      mov  word ptr cs:[int2Fh+2],es

	      mov  dx,offset mplex_int

	      mov  ax,252Fh
	      int  21h


	      mov  ax,3508h             ; Hook interrupt 08h
	      int  21h
	      mov  word ptr cs:[int08h],bx
	      mov  word ptr cs:[int08h+2],es

	      mov  ax,3509h             ; Hook interrupt 09h
	      int  21h
	      mov  word ptr cs:[int09h],bx
	      mov  word ptr cs:[int09h+2],es

	      cli

	      mov  dx,offset timer_int
	      mov  ax,2508h
	      int  21h

	      mov  dx,offset kb_int
	      mov  ax,2509h
	      int  21h

	      sti

	      cmp  cs:[mouseflag],0
	      je   nomouse2

	      mov  ax,3533h             ; Hook interrupt 33h
	      int  21h
	      mov  word ptr cs:[int33h],bx
	      mov  word ptr cs:[int33h+2],es

	      mov  dx,offset mouse_int
	      mov  ax,2533h
	      int  21h

nomouse2:
	      mov  es,[PspAddr]
	      mov  ah,49h
	      mov  es,es:[2Ch]
	      int  21h

	      mov  cs:[offsetX],0
	      mov  cs:[offsetY],0  

	      lea  dx,Loaded            ; output the message
	      call print                ; "program is installed"

					; calculate size of resident part

	      mov  dx,offset begin_install
	      add  dx,110h              ; PSP length plus 16 byte (insurance) 
	      mov  cx,4                 ; set counter for shift
	      shr  dx,cl                ; 4 bits to the right - divide by 16
	      mov  ax,3100h             ; 31h: terminate and state resident
	      int  21h

normal_exit:
	      mov  ax,4C00h             ; exit normally
	      int  21h

problem_exit:
	      call print

	      mov  ax,4c01h             ; exit with error
	      int  21h

;----------------------------------------------------------------
; Data which is discarded after installation
;----------------------------------------------------------------

tpages        dw   0                    ; Total EMS pages
apages        dw   0                    ; Available EMS pages           
PspAddr       dw   0                   ; save address of PSP

emmname       db   'EMMXXXX0',0         ; device name for EMS manager

set_time      db   0                    ; indicates that a time was specified
set_onoff     db   0                    ; indicates that we need to turn

missing       db   'EMS is missing',cr,lf,'$'
status        db   'EMS status problem',cr,lf,'$'
alloc         db   'EMS allocation problem',cr,lf,'$' 
mapping       db   'EMS mapping problem',cr,lf,'$'
transfer      db   'Image transfer problem',cr,lf,'$'
dosver        db   'Requires DOS version 3.0 or above',cr,lf,'$'
syntaxmsg     db   'Syntax error',cr,lf,'$'
titlemsg      db   'Sade screen saver',cr,lf,'$'
hlpmsg        db   'Syntax: sade /? /m=XX /[off|on]',cr,lf,'$'
Loaded        db   'Installed successfully',CR,LF,EndMsg
TurnedOff     db   'Program is now inactive',CR,LF,EndMsg
disabled      db   'Program is inactive',CR,LF,EndMsg
setto         db   'Countdown timer set to ',EndMsg
nomin         db   ' minute(s)',CR,LF,EndMsg

	      end

