UNIT BTP;      { Version 2.5 - last mod 4/1/95         (C) 1995 John C. Leon }

{$X+,D+,A-}

INTERFACE
(* ------------------------------------------------------------------------ *)
(* ------------------------------------------------------------------------ *)

USES Objects, Memory;

CONST

{      Key Attributes            Key Types          Open Modes               }
{    ------------------      ----------------     ---------------            }
     Duplicates   =   1;     BString    =  0;     Normal    =  0;
     Modifiable   =   2;     BInteger   =  1;     Accel     = -1;
     Binary       =   4;     BFloat     =  2;     ReadOnly  = -2;
     Null         =   8;     BDate      =  3;     Verify    = -3;
     Segmented    =  16;     BTime      =  4;     Exclusive = -4;
     AltCol       =  32;     BDecimal   =  5;
     Descending   =  64;     BMoney     =  6;   {        File Flags        }
     Supplemental = 128;     BLogical   =  7;   { ------------------------ }
     ExtType      = 256;     BNumeric   =  8;     VarLength   =   1;
     Manual       = 512;     BBFloat    =  9;     BlankTrunc  =   2;
                             BLString   = 10;     PreAllocate =   4;
                             BZString   = 11;     DataComp    =   8;
                             BUnsBinary = 14;     KeyOnly     =  16;
                             BAutoInc   = 15;     Free10      =  64;
                                                  Free20      = 128;
                                                  Free30      = 192;

{              Btrieve Op Codes                      Error Codes             }
{  -----------------------------------------   ------------------------      }
   BOpen      =  0;       BAbortTran   = 21;   FileNotOpen      =  3;
   BClose     =  1;       BGetPos      = 22;   InvalidKeyNumber =  6;
   BInsert    =  2;       BGetDir      = 23;   DiffKeyNumber    =  7;
   BUpdate    =  3;       BStepNext    = 24;   InvalidPosition  =  8;
   BDelete    =  4;       BStop        = 25;   EndofFile        =  9;
   BGetEqual  =  5;       BVersion     = 26;   FileNotFound     = 12;
   BGetNext   =  6;       BUnlock      = 27;   BtrieveNotLoaded = 20;
   BGetPrev   =  7;       BReset       = 28;   DataBufferLength = 22;
   BGetGr     =  8;       BSetOwner    = 29;   RejectCount      = 60;
   BGetGrEq   =  9;       BClrOwner    = 30;   IncorrectDesc    = 62;
   BGetLess   = 10;       BCrSuppIdx   = 31;   FilterLimit      = 64;
   BGetLessEq = 11;       BDropSuppIdx = 32;   IncorrectFldOff  = 65;
   BGetFirst  = 12;       BStepFirst   = 33;   LostPosition     = 82;
   BGetLast   = 13;       BStepLast    = 34;
   BCreate    = 14;       BStepPrev    = 35;
   BStat      = 15;       BGetNextExt  = 36;
   BExtend    = 16;       BGetPrevExt  = 37;
   BSetDosDir = 17;       BStepNextExt = 38;
   BGetDosDir = 18;       BStepPrevExt = 39;
   BBegTran   = 19;       BInsertExt   = 40;
   BEndTran   = 20;       BGetKey      = 50;

{  Extended Ops Comp Codes/Bias           Extended Ops Logic Constants       }
{  -----------------------------       -----------------------------------   }
   Equal       : byte =   1;           NoFilter    : integer = 0;
   GreaterThan : byte =   2;           LastTerm    : byte    = 0;
   LessThan    : byte =   3;           NextTermAnd : byte    = 1;
   NotEqual    : byte =   4;           NextTermOr  : byte    = 2;
   GrOrEqual   : byte =   5;
   LessOrEqual : byte =   6;        {              Owner Access              }
   UseAltColl  : byte =  32;        { -------------------------------------- }
   UseField    : byte =  64;          RQ = 0; RO = 1; RQENC = 2; ROENC = 3;
   UseNoCase   : byte = 128;

{   Other Unit-Specific Constants   }
{ --------------------------------- }
   MaxFixedRecLength   =  4090;   {Btrieve limits fixed record length for std}
   MaxKBufferLength    =   255;   {files to 4090.  Max key size is 255.      }
   MaxExtDBufferLength = 32767;
   MaxFileSpecLength   =   665;
   MaxDBufferLength    = 32767;
   MaxNumSegments      =    24;
   KeySpecSize         =    16;
   None = 0; Drop = 1; Retain = 2;                 {Used in CloneFileFunction}


TYPE

   TOwnerName = array[0..8] of char;


(* Data types for TRecMgr object *)
(* ----------------------------- *)
   TVersion    = record
                    case integer of
                    1: (Number  : word;
                        Rev     : integer;
                        Product : char);
                    2: (Entire  : array[1..5] of char);
                    end;
   PRecMgr     = ^TRecMgr;
   TRecMgr     = object(TObject)            {Base obj handles abort/begin/end}
                    BtrieveIsLoaded: boolean;
                    PosBlk: array[1..128] of char;
                    constructor Init;
                    destructor Done; virtual;
                    function BT(OpCode, Key: integer): integer; virtual;
                    function GetVersion: string;
                    private
                    Version      : TVersion;   {tran, reset, version and stop}
                    VersionString: string;
                    end;


(* Data types for BFile object *)
(* --------------------------- *)
   BFileName   = array[1..80] of char;    {79 + blank pad required by Btrieve}
   TAltColSpec = record               {The data types for alternate collating}
                    case integer of   {sequence are used in CreateFile fcn.  }
                    1: (Header : byte;              {Header always equals $AC}
                        Name   : array[1..8] of char;
                        Table  : array[1..256] of char);
                    2: (Entire : array[1..265] of byte);
                    end;
   PAltColSeq  = ^TAltColSeq;
   TAltColSeq  = object(TObject)
                    Spec : TAltColSpec;
                    constructor Init(SpecName: FNameStr);
                    destructor Done; virtual;
                    end;
   PKeySpec    = ^TKeySpec;
   TKeySpec    = record                     {data type for a Btrieve key spec}
                    case integer of
                    1: (KeyPos     : integer;
                        KeyLen     : integer;
                        KeyFlags   : integer;             {Tho not used in a }
                        NotUsed    : array[1..4] of byte; {create call, these}
                        ExtKeyType : byte;                {4 bytes return #  }
                        NullValue  : byte;                {unique recs in key}
                        Reserved   : array[1..4] of byte);{after a stat call.}
                    2: (Irrelevant : array[1..3] of integer;
                        NumUnique  : longint);      {great after a stat call!}
                    3: (Entire     : array[1..KeySpecSize] of byte);
                    end;
   PKeyList    = ^TKeyList;
   TKeyList     = record
                     KeySpec: TKeySpec;
                     Next: PKeyList;
                     end;
   PFileSpec   = ^TFileSpec;
   TFileSpec   = record                      {Strictly speaking, the KeyArray}
                    case integer of          {and AltColSpec elements here   }
                    1: (RecLen     : word;   {only serve to reserve space for}
                        PageSize   : integer;{the buffer.                    }
                        NumKeys    : integer;
                        NumRecs    : array[1..2] of word;
                        FileFlags  : integer;
                        Reserved   : array[1..2] of char;
                        PreAlloc   : integer;
                        KeyArray   : array[0..23] of TKeySpec; {24=max # segs}
                        AltColSpec : TAltColSpec);   {here just to allow room}
                    2: (Irrelevant : array[1..14] of byte;
                        UnusedPgs  : word);         {great after a stat call!}
                    3: (SpecBuf    : integer); {used to refer to addr of spec}
                    4: (Entire     : array[1..665] of byte);
                    end;
   PFileSpecObj   = ^TFileSpecObj;
   TFileSpecObj   = object(TObject)
                       Specs: PFileSpec;
                       Keys : PKeyList;
                       constructor Init(RecLen, PageSize, NumKeys,
                                        FileFlags, PreAlloc: integer;
                                        AKeyList: PKeyList);
                       destructor Done; virtual;
                       end;
   PBFile      = ^BFile;
   BFile       = object(TObject)
                    DFileName  : FNameStr;                      {DOS filename}
                    Specs      : TFileSpec;               {Btrieve file specs}
                    SpecLength : word;            {length of actual file spec}
                    NumRecs    : longint;             {# records at Init time}
                    NumSegs    : integer;                   {total # key segs}
                    HasAltCol  : boolean;       {true if file has alt col seq}
                    AltColName : string[8];    {name of alt col seq from file}
                    IsVarLength: boolean;
                    HasOwner   : boolean;
                    OwnerName  : TOwnerName;                   {8 plus 1 null}
                    PosBlk     : array[1..128] of char;       {position block}
                    DBufferLen : word;
                    constructor Init(UserFileName: FNameStr;
                                     OpenMode: integer;
                                     OName: string);
                    function BT(OpCode, Key: integer): integer; virtual;
                    function Open(OpenMode: integer):  integer; virtual;
                    function IsOpen: boolean;
                    function Close: integer; virtual;
                    function Stat: integer; virtual;
                    function AddSuppIdx(KeyList: PKeyList;
                                        AltColFile: FNameStr): boolean; virtual;
                    destructor Done; virtual;
                    private
                    FileName : BFileName;              {Btrieve-type filename}
                    OpenFlag: boolean;
                    procedure ConvertName(UserFileName: FNameStr);
                    procedure CountSegments;
                    end;


(* Data types for BFixed object - descendant of BFile *)
(* -------------------------------------------------- *)
   TDBuffer    = array[1..MaxFixedRecLength] of byte;
   TKBuffer    = array[1..MaxKBufferLength] of byte;
   PBFixed     = ^BFixed;
   BFixed      = object(BFile)
                    DBuffer : TDBuffer;
                    KBuffer : TKBuffer;
                    constructor Init(UserFileName: FNameStr; OpenMode: integer;
                                     OName: string);
                    function BT(OpCode, Key: integer): integer; virtual;
                    destructor Done; virtual;
                    end;


(* Data types for BSized object - descendant of BFile *)
(* -------------------------------------------------- *)
   PBSized = ^BSized;
   BSized  = object(BFile)
                DBuffer    : pointer;
                KBuffer    : pointer;
                DBufferSize: integer;
                constructor Init(UserFileName: FNameStr;
                                 OpenMode: integer;
                                 OName: string; BuffSize: integer);
                function BT(OpCode, Key: integer): integer; virtual;
                destructor Done; virtual;
                end;


(* Data types for BFileExt object - descendant of BFile *)
(* ---------------------------------------------------- *)
   TByteArray  = array[1..255] of byte;
   THeader     = record
                    case integer of
                    1: (DBufferLen : integer;
                        Constant   : array[1..2] of char);
                    2: (Entire     : array[1..4] of byte);
                    end;
   TFilter     = record
	                 case integer of
                    1: (MaxSkip       : integer;
                        NumLogicTerms : integer);
                    2: (Entire        : array[1..2] of integer);
                    end;
   TLogicTerm  = record
                    case integer of
                    1: (FieldType  : byte;
                        FieldLen   : integer;
                        Offset     : integer;  {0 relative to start of record}
                        CompCode   : byte;
                        Expression : byte;{0 last term, 1 AND next, 2 OR next}
                        case FieldComp: boolean of
                           True : (CompOffset: integer);
                           False: (Value: TByteArray));{an arbitrary limit of}
                    2: (Fixed : array[1..7] of byte);  {255 on len of values }
                    end;
   PFilterSpec = ^TFilterSpec;
   TFilterSpec = object(TObject)
                    LogicTerm: TLogicTerm;
                    constructor InitF(FieldType: byte; FieldLen, Offset:
                                      integer; CompCode, Expression: byte;
                                      CompOffset: integer);
                    constructor InitV(FieldType: byte; FieldLen, Offset:
                                      integer; CompCode, Expression: byte;
                                      Value: array of byte);
                    destructor Done; virtual;
                    end;
   TExtractor  = record
                    case integer of
                    1: (NumRecords : integer;
                        NumFields  : integer);
                    2: (Entire     : array[1..2] of integer);
                    end;
   TExtRepeater= record
                    FieldLen : integer;
                    Offset   : integer;
                    end;
   PExtSpec    = ^TExtSpec;
   TExtSpec    = object(TObject)
                    ExtRepeater : TExtRepeater;
                    constructor Init(Len, Ofs: integer);
                    destructor Done; virtual;
                    end;
   PExtDBuffer = ^TExtDBuffer;
   TExtDBuffer = record
                    case integer of
                    1: (Header   : THeader;       {Buffer sent includes these}
                        Filter   : TFilter);         {types at its beginning.}
                    2: (NumRecs  : integer;               {Buffer rec'd looks}
                        Repeater : array[1..32765] of char);      {like this.}
                    {Repeater structure is: 2 for length of record image,    }
                    {                       4 for currency position of rec,  }
                    {                       n for record image itself        }
                    3: (Entire   : array[1..32767] of byte);   {Whole buffer.}
                    end;
   PBFileExt   = ^BFileExt;
   BFileExt    = object(BFile)
                    Header        : THeader;
                    Filter        : TFilter;
                    FilterSpec    : PCollection;
                    Extractor     : TExtractor;
                    ExtractorSpec : PCollection;
                    ExtDBuffer    : PExtDBuffer;
                    constructor Init(UserFileName: FNameStr; OpenMode: integer;
                                     OName: string);
                    function BTExt(OpCode, Key: integer): integer; virtual;
                    procedure SetTerms(MSkip, NumLTerms, NRecs, NumFlds: integer);
                    destructor Done; virtual;
                    private
                    procedure SetExtDBufferLen;
                    procedure MakeExtDBuffer;
                    end;


(* PUBLIC VARS *)
(* ----------- *)
VAR BStatus : integer;

(* PUBLIC FUNCTIONS *)
(* ---------------- *)

function CreateFile(UserFileName: FNameStr; UserFileSpec:PFileSpec;
                    AltColFile: FNameStr; OName: string;
                    Access: integer): integer;
function CloneFile(CurrentFile, NewFile: FNameStr; Option: integer;
                   OName: string): integer;
function NewKeySpec(KPos, KLen, KFlags: integer; EType: byte;
                    NextKey: PKeyList): PKeyList;
function IsBtrieveLoaded: boolean;
function LTrim(S: String): String;   {LTrim and RTrim were taken from one of }
function RTrim(S: String): String;   {the Turbo Vision .PAS source files.    }


IMPLEMENTATION
(* ------------------------------------------------------------------------ *)
(* ------------------------------------------------------------------------ *)
USES Strings, BTRAPID;

(* BRECMGR.INIT Constructor *)
(* ------------------------ *)
constructor TRecMgr.Init;
var
   VersionSize: word;
   BNumber,
   BRev     : string[2];
   BProduct : string[1];
   KBuffer  : array[1..255] of char;
begin
   inherited Init;                            {assures all data fields zeroed}
   VersionSize := sizeof(TVersion);
   BStatus := Btrv(BVersion, PosBlk, Version, VersionSize, KBuffer, 0);
   str(Version.Number:2, BNumber);
   BNumber := LTrim(BNumber);
   str(Version.Rev:2, BRev);
   BProduct := Version.Product;
   VersionString := BNumber + '.' + BRev + BProduct;
end;


(* BRECMGR.BT function *)
(* ------------------- *)
{Will not handle reset of other workstations as written, as no true key
 buffer is passed.   Will handle begin/end/abort transaction, reset & stop.
 Would also handle version op, but is handled by BRecMgr.Init anyway!}

function TRecMgr.BT(OpCode, Key: integer): integer;
var
   DBuffer: array[1..8] of char;
   DBufferLength: word;
   KBuffer: array[1..8] of char;
begin
   BT := Btrv(OpCode, PosBlk, DBuffer, DBufferLength, KBuffer, Key);
end;


(* BRECMGR Destructor *)
(* ------------------ *)
destructor TRecMgr.Done;
begin
   inherited Done;
end;


(* BRECMGR.GetVersion function *)
(* --------------------------- *)
function TRecMgr.GetVersion: string;
begin
   GetVersion := VersionString;
end;


(* TALTCOLSEQ.INIT Constructor *)
(* ---------------------------- *)
constructor TAltColSeq.Init(SpecName: FNameStr);
var
   AltFile: file of TAltColSpec;      {The TAltColSpec object type is used   }
begin                                 {internally by the CreateFile function.}
   inherited Init;
   assign(AltFile, SpecName);
   {$I-} reset(AltFile); {$I+}    {It's up to user program to assure that the}
   if ioresult = 0 then           {alternate collating sequence file exists  }
     begin                        {in the current directory when the         }
       read(AltFile, Spec);       {CreateFile fcn is called, and is of the   }
       close(AltFile);            {standard format expected by Btrieve.      }
     end
   else
     Fail;
end;


(* TALTCOLSEQ.DONE Destructor *)
(* --------------------------- *)
destructor TAltColSeq.Done;
begin
   inherited Done;
end;


(* TFILESPECOBJ.INIT Constructor *)
(* ----------------------------- *)
constructor TFileSpecObj.Init(RecLen, PageSize, NumKeys,
			      FileFlags, PreAlloc: integer;
			      AKeyList: PKeyList);
var
   Counter: integer;
   Key: PKeyList;
begin
   inherited Init;
   Specs := new(PFileSpec);
   Keys := AKeyList;                          {save head of list for disposal}
   fillchar(Specs^, sizeof(Specs^), 0);
   Specs^.RecLen    := RecLen;
   Specs^.PageSize  := PageSize;
   Specs^.NumKeys   := NumKeys;
   Specs^.FileFlags := FileFlags;
   Specs^.PreAlloc  := PreAlloc;
   Counter := 0;
   Key := AKeyList;
   if Key <> nil then
      repeat
         Specs^.KeyArray[Counter].KeyPos     := Key^.KeySpec.KeyPos;
         Specs^.KeyArray[Counter].KeyLen     := Key^.KeySpec.KeyLen;
         Specs^.KeyArray[Counter].KeyFlags   := Key^.KeySpec.KeyFlags;
         Specs^.KeyArray[Counter].ExtKeyType := Key^.KeySpec.ExtKeyType;
         inc(Counter);
         Key := Key^.Next;
      until Key = nil;
end;


(* TFILESPECOBJ.DONE Destructor *)
(* ---------------------------- *)
destructor TFileSpecObj.Done;

   procedure KillKeyList(x: PKeyList);
   var
      x1, x2: PKeyList;
   begin
      if x = nil then exit;
      x1 := x;
      while x1^.next <> nil do
         begin
         x2 := x1^.next;
         dispose(x1);
         x1 := x2;
         end;
      dispose(x1);
   end;

begin
   inherited Done;
   dispose(Specs);
   KillKeyList(Keys);
end;


(* BFILE.INIT Constructor *)
(* ---------------------- *)
constructor BFile.Init(UserFileName: FNameStr; OpenMode: integer;
                       OName: string);

var
                               {665 = 16 for filespec + 384 for max key specs}
   FileBufLen        : word;   {+ 265 for an alternate collating sequence.   }
   KeyBufLen,                  {Max of 24 keys * 16 bytes per key spec.}
   AltColNameOffset,
   Counter, Status   : integer;
   NumRecsWord1,
   NumRecsWord2      : word;

begin
   inherited Init;                            {assures all data fields zeroed}
                                              {665 = 16 for filespec + 384 for max key specs}
   OpenFlag := false;
   FileBufLen := MaxFileSpecLength;{+ 265 for an alternate collating sequence}
   KeyBufLen  := 384;                {Max of 24 keys * 16 bytes per key spec.}
   HasAltCol := false;            {initialize to false 'until proven guilty!'}
   AltColName := '';
   ConvertName(UserFileName);             {Sets fields DFileName and FileName}
   IsVarLength := false;
   HasOwner := false;
   fillchar(OwnerName, sizeof(TOwnerName), #0);
   DBufferLen := 0;                {Force equal to 0 ... it is expected that}
   Counter := 1;                   {overrides of virtual BT function will   }
   if OName <> '' then             {set DBufferLen to appropriate value.    }
      begin
      StrPCopy(OwnerName,OName);
      HasOwner := true;
      end;
   Status := Open(OpenMode);
   if Status = 0 then                    {if open op successful, do a stat op}
      begin
      NumSegs := Specs.NumKeys;
      Status := Stat;
      {Btrieve filespecs and key specs are now in the BFile object!}
      {Variable FileBufLen will have been changed to size of data
       buffer returned by stat call.  Save that value now.}
      if Status = 0 then     {if stat successfull, fill object data fields}
	      begin
	      SpecLength := FileBufLen;
	      NumRecsWord1 := Specs.NumRecs[1];  {get rid of sign bit!! by  }
	      NumRecsWord2 := Specs.NumRecs[2];  {converting 2 ints to words}
	      NumRecs := NumRecsWord1 + NumRecsWord2 * 65536;
	      if (Specs.FileFlags and VarLength) = VarLength then
                 IsVarLength := true;
	      {The CountSegments procedure called internally by the Stat function
          will have set the HasAltCol member.}
	      if HasAltCol then
            begin
            AltColNameOffset := (16 + KeySpecSize * NumSegs + 1);
            for Counter := 1 to 8 do
               AltColName[Counter] := chr(Specs.Entire[AltColNameOffset + Counter]);
            end;
	      BStatus := 0;                  {all went well, return a code 0}
	      end
      else
	      begin
         BStatus := Status;  {Open op succeeded but stat failed; put   }
         Status  := Close;   {error code for bad stat in global var and}
         end;                   {close the damn file quick!}
      end
   else
      BStatus := Status;             {assign err code for bad open to global var}
end;


(* BFILE.BT function *)
(* ----------------- *)
function BFile.BT(OpCode, Key: integer): integer;
begin
   Abstract;
end;


(* BFILE.OPEN function *)
(* ------------------- *)
function BFile.Open(OpenMode: integer): integer;
var
   BufferSize: word;
   DB, DBLen: word;
   Status: integer;
begin
   if HasOwner then
      begin
      BufferSize := StrLen(OwnerName)+1;    {Gotta add 1 for null terminator.}
      Status := Btrv(BOpen, PosBlk, OwnerName, BufferSize, FileName, OpenMode);
      end
   else
      Status := Btrv(BOpen, PosBlk, DB, DBLen, FileName, OpenMode);
   if Status = 0 then
      OpenFlag := true
   else
      OpenFlag := false;
   Open := Status;
end;


(* BFILE.ISOPEN function *)
(* ------------------- *)
function BFile.IsOpen: boolean;
begin
   if OpenFlag = true then
      IsOpen := true
   else
      IsOpen := false;
end;


(* BFILE.CLOSE Function *)
(* -------------------- *)
function BFile.Close: integer;
var Status: integer;
    DBuffer,
    DBufferLength,
    KBuffer: word;
begin
   Status := Btrv(BClose, PosBlk, DBuffer, DBufferLength, KBuffer, 0);
   Close := Status;
   if Status = 0 then
      OpenFlag := false;
end;


(* BFILE.STAT Function *)
(* ------------------- *)
function BFile.Stat: integer;
var
   FileBufLen: word;
   KeyBufLen,
   NumRecsWord1,
   NumRecsWord2,
   Counter,
   AltColNameOffset: integer;
begin                                {File must be open for this call to succeed.}
   fillchar(Specs.Entire, sizeof(Specs.Entire), 0);

			   {FileBufLen is 16 for filespec + 384 for max key
			    specs + 265 for an alternate collating sequence.}
   FileBufLen := MaxFileSpecLength;
   KeyBufLen  := 384;                {Max of 24 keys * 16 bytes per key spec.}

   HasAltCol := false;          {Initialize to false..reset as needed later.}
   AltColName := '';     {Initialize to empty string..reset as needed later.}

   BStatus := Btrv(BStat, PosBlk, Specs.SpecBuf, FileBufLen, KeyBufLen, 0);
   if  BStatus = 0 then
      begin
      SpecLength := FileBufLen;
      NumRecsWord1 := Specs.NumRecs[1];
      NumRecsWord2 := Specs.NumRecs[2];
      NumRecs := NumRecsWord1 + NumRecsWord2 * 65536;
      NumSegs := Specs.NumKeys;
      if (Specs.FileFlags and VarLength) = VarLength then
         IsVarLength := true;
      if Specs.NumKeys >= 1 then
         CountSegments;                    {Do if not data-only file.}
      if HasAltCol then
         begin
         AltColNameOffset := (16 + KeySpecSize * NumSegs + 1);
         for Counter := 1 to 8 do
            AltColName[Counter] := chr(Specs.Entire[AltColNameOffset + Counter]);
         end;
      end;
   Stat := BStatus;
end;


(* BFILE.ADDSUPPIDX Function *)
(* ------------------------- *)
function BFile.AddSuppIdx(KeyList: PKeyList; AltColFile: FNameStr): boolean;
type
   PBuffer = ^TBuffer;
   TBuffer =  array[0..MaxFileSpecLength] of byte;
var
   NewSegmentCount,
   Offset:           integer;
   SuppIdxHasAltCol: boolean;
   AKeyList, X1, X2: PKeyList;
   ACS:              PAltColSeq;
   Buffer:           PBuffer;
   BufferLength:     word;
   KBuffer:          array[1..255] of char;
begin
   NewSegmentCount := 1;
   SuppIdxHasAltCol := false;
   Offset := 0;
   AKeyList := KeyList;
   while (AKeyList^.Next <> nil) do  {Count # segs in new supp idx.}
      begin
      inc(NewSegmentCount);
      AKeyList := AKeyList^.Next;
      end;
   if (NewSegmentCount + NumSegs) > MaxNumSegments then
      AddSuppIdx := false
   else
      begin
      new(Buffer);
      fillchar(Buffer^, sizeof(Buffer^), 0);
      AKeyList := KeyList;
      repeat
         if (AKeyList^.KeySpec.KeyFlags and AltCol) = AltCol then
            SuppIdxHasAltCol := true;
         move(AKeyList^.KeySpec, Buffer^[Offset], KeySpecSize);
         inc(Offset, KeySpecSize);
         AKeyList := AKeyList^.Next;
      until (AKeyList = nil);
      if (KeyList <> nil) then          {Dispose of linked list of key specs.}
         begin
         X1 := KeyList;
         while (X1^.Next <> nil) do
            begin
            X2 := X1^.Next;
            dispose(X1);
            X1 := X2;
            end;
         dispose(X1);
         end;
      BufferLength := KeySpecSize * NewSegmentCount;
      {If the supp index will have an ACS, get it into data buffer, and add
       its size to DBufferLen parameter.}
      if (AltColFile <> '') and SuppIdxHasAltCol then
         begin
         ACS := new(PAltColSeq, Init(AltColFile));
         if (ACS <> nil) then
            begin
            move(ACS^.Spec, Buffer^[BufferLength], sizeof(ACS^.Spec));
            inc(BufferLength, sizeof(ACS^.Spec));
            dispose(ACS, Done);
            end;
         end;
      BStatus := Btrv(BCrSuppIdx, PosBlk, Buffer^, BufferLength, KBuffer, 0);
      dispose(Buffer);
      if BStatus = 0 then
         AddSuppIdx := true
      else
         AddSuppIdx := false;
   end;
end;


(* BFILE.DONE Destructor *)
(* --------------------- *)
destructor BFile.Done;
begin
   inherited Done;
   if OpenFlag = true then
      Close;
end;


(* BFILE.CONVERTNAME Procedure *)
(* --------------------------- *)
{this one is private to BFile}
procedure BFile.ConvertName(UserFileName: FNameStr);
begin
   DFileName := UserFileName;
   move(DFileName[1], FileName[1], length(DFileName));  {conv string to array}
   FileName[length(DFileName) + 1] := ' ';         {provide required pad char}
end;


(* BFILE.COUNTSEGMENTS Procedure *)
(* ----------------------------- *)
procedure BFile.CountSegments;
var
   Counter, Counter1: integer;
begin
   Counter  := 1;
   Counter1 := 0;
   while Counter <= Specs.NumKeys do
      repeat
         if (Specs.KeyArray[Counter1].KeyFlags and AltCol) = AltCol then
             HasAltCol := true;
         if (Specs.KeyArray[Counter1].KeyFlags and Segmented) = Segmented then
            inc(NumSegs)
         else
            inc(Counter);
         inc(Counter1);
      until (Specs.KeyArray[Counter1-1].KeyFlags and Segmented) <> Segmented;
end;

(* BFIXED.INIT Constructor *)
(* ----------------------- *)
constructor BFixed.Init(UserFileName: FNameStr; OpenMode: integer;
                        OName: String);
begin
   inherited Init(UserFileName, OpenMode, OName);
end;


(* BFIXED.BT function *)
(* ----------------- *)
function BFixed.BT(OpCode, Key: integer): integer;
begin
   BT := Btrv(OpCode, PosBlk, DBuffer, Specs.RecLen, KBuffer, Key);
end;


(* BFIXED.DONE Destructor *)
(* ---------------------- *)
destructor BFixed.Done;
begin
   inherited Done;
end;


(* BSIZED.INIT Constructor *)
(* ----------------------- *)
constructor BSized.Init(UserFileName: FNameStr; OpenMode: integer;
                        OName: String; BuffSize: integer);
begin
   inherited Init(UserFileName, OpenMode, OName);
   if BuffSize <= 0 then
      BuffSize := MaxFixedRecLength;
   DBufferSize := BuffSize;
   DBuffer := memallocseg(BuffSize);
   KBuffer := memallocseg(MaxKBufferLength);
   fillchar(DBuffer^, BuffSize, 0);
   fillchar(KBuffer^, MaxKBufferLength, 0);
end;


(* BSIZED.DONE Destructor *)
(* ---------------------- *)
destructor BSized.Done;
begin
   if DBuffer <> nil then freemem(DBuffer, DBufferSize);
   if KBuffer <> nil then freemem(KBuffer, MaxKBufferLength);
   DBuffer := nil;
   KBuffer := nil;
   inherited Done;
end;


(* BSIZED.BT Function *)
(* ------------------ *)
function BSized.BT(OpCode, Key: integer): integer;
begin
   DBufferLen := DBufferSize;
   BT := Btrv(OpCode, PosBlk, DBuffer^, DBufferLen, KBuffer^, Key);
end;


(* TFILTERSPEC.INITF Constructor *)
(* ----------------------------- *)
{Be sure to remember that the offset parameter here is 0 relative to start of
 record!!}

constructor TFilterSpec.InitF(FieldType: byte; FieldLen, Offset: integer;
			      CompCode, Expression: byte; CompOffset: integer);
begin
   inherited Init;                            {assures all data fields zeroed}
   LogicTerm.FieldType  := FieldType;
   LogicTerm.FieldLen   := FieldLen;
   LogicTerm.Offset     := Offset;
   LogicTerm.CompCode   := CompCode;
   LogicTerm.Expression := Expression;
   LogicTerm.FieldComp  := true;
   LogicTerm.CompOffset := Offset;
end;


(* TFILTERSPEC.INITV Constructor *)
(* ----------------------------- *)
{Be sure to remember that the offset parameter here is 0 relative to start of
 record!!}

constructor TFilterSpec.InitV(FieldType: byte; FieldLen, Offset: integer;
			      CompCode, Expression: byte; Value: array of byte);
begin
   inherited Init;                            {assures all data fields zeroed}
   LogicTerm.FieldType := FieldType;
   LogicTerm.FieldLen  := FieldLen;
   LogicTerm.Offset    := Offset;
   LogicTerm.CompCode  := CompCode;
   LogicTerm.Expression:= Expression;
   LogicTerm.FieldComp := false;
   move(Value[0], LogicTerm.Value[1], high(Value)+1);
end;


(* TFILTERSPEC.DONE Destructor *)
(* --------------------------- *)
destructor TFilterSpec.Done;
begin
   inherited Done;
end;


(* TEXTSPEC.INIT Constructor *)
(* ------------------------- *)
constructor TExtSpec.Init(Len, Ofs: integer);
begin
   inherited Init;                            {assures all data fields zeroed}
   ExtRepeater.FieldLen := Len;
   ExtRepeater.Offset   := Ofs;
end;


(* TEXTSPEC.DONE Destructor *)
(* ----------------------- *)
destructor TExtSpec.Done;
begin
   inherited Done;
end;


(* BFILEEXT.INIT Constructor *)
(* ------------------------- *)
{always check for a failure!}

constructor BFileExt.Init(UserFileName: FNameStr; OpenMode: integer;
                          OName: String);
begin
   inherited Init(UserFileName, OpenMode, OName);
   Header.Constant[1] := 'E';
   Header.Constant[2] := 'G';
   ExtDBuffer    := memallocseg(MaxExtDBufferLength);
   FilterSpec    := new(PCollection, Init(2,2));
   ExtractorSpec := new(PCollection, Init(5,2));
   if (ExtDBuffer = nil) or (FilterSpec = nil) or (ExtractorSpec = nil) then
      Fail;
end;


(* BFILEEXT.DONE Destructor *)
(* ------------------------ *)
destructor BFileExt.Done;
begin
   inherited Done;
   dispose(ExtDBuffer);
   dispose(ExtractorSpec, Done);
   dispose(FilterSpec, Done);
end;


(* BFILEEXT.SETEXTDBUFFERLEN function *)
(* ---------------------------------- *)
{Compute sizes of data buffers sent and returned, to determine proper size to
 specify in call.  Assumes user program has inserted proper items into the
 collections for filter terms and extractor specs.  Is private to BFileExt.}

procedure BFileExt.SetExtDBufferLen;
var
   LengthSent, LengthReturned,
   RecordLengthReturned, RecordImageReturned : integer;

   procedure MakeFilterSpecs;
      procedure CalcFilterLengths(FSpec: PFilterSpec); far;
      begin
      with FSpec^ do
         begin
         inc(LengthSent, 7);
         if (LogicTerm.CompCode and UseField) = UseField then
            inc(LengthSent, 2)
         else
            inc(LengthSent, LogicTerm.FieldLen);
         end;
      end;
   begin
      FilterSpec^.ForEach(@CalcFilterLengths);
   end;

   procedure MakeExtSpecs;
      procedure CalcExtLengths(ExtSpec: PExtSpec); far;
      begin
      with ExtSpec^ do
         begin
         inc(LengthSent, 4);
         inc(RecordLengthReturned, ExtRepeater.FieldLen);
         end;
      end;
   begin
      ExtractorSpec^.ForEach(@CalcExtLengths);
   end;

begin
   LengthSent := 8; {4 for header length, 4 for fixed filter length}

   {Work on filter logic term portion of spec.}
   if FilterSpec^.Count > 0 then       {if any filter terms in the collection}
      MakeFilterSpecs;

   {Work on extractor portion of spec.}
   inc(LengthSent, 4);                       {size of fixed part of extractor}
   RecordLengthReturned := 0;
   MakeExtSpecs;              {there must always be at least 1 extractor spec}

   {2 for count of recs, 4 for currency pos}
   RecordImageReturned := RecordLengthReturned + 6;
   {2 for count of recs}
   LengthReturned := 2 + (RecordImageReturned * Extractor.NumRecords);

   Header.DBufferLen := LengthSent;

   if LengthSent >= LengthReturned then
      DBufferLen := LengthSent
   else
      DBufferLen := LengthReturned;
end;


(* BFILEEXT.MAKEEXTDBUFFER Function *)
(* -------------------------------- *)
{Private to BFileExt, called in BFileExt.BT, which is called by each
 descendant's override of BFileExt.BT.  Assumes program has already set up
 the collections required.}

procedure BFileExt.MakeExtDBuffer;
var
   Offset : integer;

   procedure MoveFilterSpecs;
      procedure MoveSingleFilterSpec(FSpec: PFilterSpec); far;
      begin
      with FSpec^ do
         begin
         {move fixed part of logic term}
         move(LogicTerm, ExtDBuffer^.Entire[Offset], sizeof(LogicTerm.Fixed));
         inc(Offset, sizeof(LogicTerm.Fixed));
         {now need to move variable part of logic term}
         if (LogicTerm.CompCode and UseField) = UseField then
            begin
            move(LogicTerm.CompOffset, ExtDBuffer^.Entire[Offset],
            sizeof(LogicTerm.CompOffset));
            Offset := Offset + sizeof(LogicTerm.CompOffset);
            end
         else
            begin
            move(LogicTerm.Value, ExtDBuffer^.Entire[Offset],
                 LogicTerm.FieldLen);
            Offset := Offset + LogicTerm.FieldLen;
            end;
         end;
      end;
   begin
      FilterSpec^.ForEach(@MoveSingleFilterSpec);
   end;

   procedure MoveExtractorSpecs;
      procedure MoveSingleExtractorSpec(ExtSpec: PExtSpec); far;
      begin
      with ExtSpec^ do
         begin
         move(ExtSpec^.ExtRepeater, ExtDBuffer^.Entire[Offset],
         sizeof(ExtSpec^.ExtRepeater));
         Offset := Offset + sizeof(ExtSpec^);
         end;
      end;
   begin
      ExtractorSpec^.ForEach(@MoveSingleExtractorSpec);
   end;

begin
   {Move header definition into buffer.}
   move(Header, ExtDBuffer^.Header, sizeof(Header));

   {Move fixed part of filter definition into buffer.}
   move(Filter, ExtDBuffer^.Filter, sizeof(Filter));
   Offset := 1 + sizeof(Header) + sizeof(Filter);

   {Read filter logic terms into buffer.}
   if FilterSpec^.Count > 0 then
      MoveFilterSpecs;

   {Move fixed part of extractor definition into buffer.}
   move(Extractor, ExtDBuffer^.Entire[Offset], sizeof(Extractor.Entire));
   Offset := Offset + sizeof(Extractor.Entire);

   {Move extractor terms into buffer.}
   MoveExtractorSpecs;
end;


(* BFILEEXT.BTEXT function *)
(* ----------------------- *)
{Overrides of this function in BFileExt descendants MUST call
 BFileExt.BTExt, as it sets the buffer length in the header, and puts
 together the 'send' buffer.  User programs MUST have inserted filter logic
 terms and extractor specs into their respective collections before invoking
 this function, or they'll make a fine mess of things, Ollie!}

function BFileExt.BTExt(OpCode, Key: integer): integer;
begin
   SetExtDBufferLen;
   MakeExtDBuffer;
end;


(* BFILEEXT.SETTERMS procedure *)
(* --------------------------- *)
procedure BFileExt.SetTerms(MSkip, NumLTerms, NRecs, NumFlds: integer);
begin
   Filter.MaxSkip       := MSkip;
   Filter.NumLogicTerms := NumLTerms;
   Extractor.NumRecords := NRecs;
   Extractor.NumFields  := NumFlds;
end;


(* IMPLEMENTATION OF UTILITY FUNCTIONS/PROCEDURES *)
(* ------------------------------------------------------------------------ *)

(* CREATEFILE function *)
(* -------------------- *)
{Assumes a PFILESPEC variable has been instantiated and assigned its values,
 and that if you use an alternate collating sequence, it exists in the
 current directory.  No specific support for null keys, blank compression,
 data-only files.}

function CreateFile(UserFileName: FNameStr; UserFileSpec:PFileSpec;
                    AltColFile: FNameStr; OName: String;
                    Access: integer): integer;
var
   CFSpecLength    : word;
   Counter,
   Counter1,
   NumSegs         : integer;
   BtrieveFileName : BFileName;
   HasAltCol       : boolean;
   AltColObj       : PAltColSeq;
   NewFile         : PBFixed;
   PosBlk          : array[1..128] of char;

   procedure CountSegments;
   begin
      with UserFileSpec^ do
         repeat
            if (KeyArray[Counter1].KeyFlags and AltCol) = AltCol then
                HasAltCol  := true;
            if (KeyArray[Counter1].KeyFlags and Segmented) = Segmented then
               inc(NumSegs)
            else
               inc(Counter);
            inc(Counter1);
         until (KeyArray[Counter1-1].KeyFlags and Segmented) <> Segmented;
   end;

begin
   move(UserFileName[1], BtrieveFileName[1], length(UserFileName));
   BtrieveFileName[length(UserFileName) + 1] := ' ';
   Counter := 1; Counter1 := 0;
   NumSegs := UserFileSpec^.NumKeys;
   while Counter <= UserFileSpec^.NumKeys do
      CountSegments;
   CFSpecLength := 16 + (NumSegs * KeySpecSize);
   UserFileSpec^.Reserved[1] := chr(0);
   UserFileSpec^.Reserved[2] := chr(0);
   if (AltColFile <> '') and (HasAltCol = true) then  {Note the double check!}
      begin
      AltColObj   := new(PAltColSeq, Init(AltColFile));
      move(AltColObj^.Spec, UserFileSpec^.Entire[CFSpecLength+1],
           sizeof(AltColObj^.Spec));
      CFSpecLength := CFSpecLength + sizeof(AltColObj^.Spec);
      dispose(AltColObj, Done);
      end;
   BStatus := Btrv(BCreate, PosBlk, UserFileSpec^.SpecBuf, CFSpecLength,
                   BtrieveFileName, 0);
   if (BStatus = 0) and (OName <> '')  then
      begin
      NewFile := new(PBFixed, Init(UserFileName, Exclusive, ''));
      with NewFile^ do
         begin
         fillchar(DBuffer[1], 9, 0);                {Assure null termination.}
         fillchar(KBuffer[1], 9, 0);                {Assure null termination.}
         move(OName[1], DBuffer[1], length(OName));
         move(OName[1], KBuffer[1], length(OName));
         Specs.RecLen := length(OName) + 1;
         BStatus := BT(BSetOwner, Access);
         end;
      dispose(NewFile, Done);
      end;
   CreateFile := BStatus;
end;


(* CLONEFILE function *)
(* ------------------ *)
{Programmer is responsible for assuring that 'CurrentFile' exists and can be
 opened.  Function will overwrite any existing file with 'NewFile' name.
 The integer returned here can be meaningless if the current file does not
 exist or is not opened properly.  This function is as streamlined as
 possible, but puts RESPONSIBILITY on the programmer.

 It is entirely possible that this clone function will NOT return a byte for
 byte matching file, if cloning an 'empty' Btrieve file.  This would be due
 to the inability to determine the number of pages pre-allocated when a file
 was created, if preallocation had been used.  The Btrieve Stat call uses
 the 'Preallocate # of pages' bytes to return the number of unused pages!!
 Thus, the CloneFile function clears the Preallocation bit in the FileFlags
 before creating the new file.  A non-exact copy would also result if the
 source file used an owner name, as the clone will NOT bear an owner name.
 If you want an owner name in the clone, add it AFTER creating the clone.

 NOTE: This function goes beyond the capability of "BUTIL -CLONE" in that
       this function has flexible handling of supplemental indexes in the
 cloned file. It can drop, retain, or make them permanent in the clone.  In
 addition, if no permanent indexes use an alternate collating sequence, but
 one or more supplemental indexes DOES use one, the clone can retain the
 supplemental indexes WITH the collating sequence, duplicating the source
 file's structure perfectly.  This is something that "BUTIL -CLONE" simply
 CANNOT HANDLE!!
}

function CloneFile(CurrentFile, NewFile:FNameStr; Option: integer;
                   OName: String): integer;
type
   PSuppIdxList       = ^TSuppIdxList;
   TSuppIdxList       = array[0..23] of boolean;   {will hold list of indexes}
   PSuppIdx           = ^TSuppIdx;
   TSuppIdx           = array[0..23] of TKeySpec;
   PSuppIdxHasAltCol  = ^TSuppIdxHasAltCol;
   TSuppIdxHasAltCol  = array[0..23] of boolean;
var
   HasSuppIdx,
   PermKeyHasAltCol  : boolean;
   NewSpecLength     : word;
   NumberSuppSegs,
   NumberSuppIdx,
   NewOffset,
   DBuffOffset,
   Counter, Counter1,
   NewNumKeys        : integer;
   CurrentBFile      : PBFile;
   NewBFile          : PBFixed;
   NewBFileName      : BFileName;
   NewFileSpec       : PFileSpec;
   SuppIdxList       : PSuppIdxList;
   SuppIdx           : PSuppIdx;
   SuppIdxHasAltCol  : PSuppIdxHasAltCol;
   PosBlk            : array[1..128] of char;
begin
   HasSuppIdx       := false;
   PermKeyHasAltCol := false;
   NumberSuppSegs   :=  0;
   NumberSuppIdx    :=  0;
   NewOffset        := 17;
   DBuffOffset      :=  1;
   SuppIdxList      := new(PSuppIdxList);
   SuppIdx          := new(PSuppIdx);
   SuppIdxHasAltCol := new(PSuppIdxHasAltCol);
   NewFileSpec      := new(PFileSpec);
   fillchar(SuppIdxList^, sizeof(SuppIdxList^), false);
   fillchar(SuppIdx^, sizeof(SuppIdx^), 0);
   fillchar(SuppIdxHasAltCol^, sizeof(SuppIdxHasAltCol^), false);
   fillchar(NewFileSpec^, sizeof(NewFileSpec^), 0);  {initialize spec w/zeros}

   move(NewFile[1], NewBFileName[1], length(NewFile));{establish new filename}
   NewBFileName[length(NewFile) + 1] := ' ';

   CurrentBFile := new(PBFile, Init(CurrentFile, ReadOnly, OName));
   if BStatus <> 0 then        {Exit function if problem opening source file.}
     begin
     CloneFile := BStatus;
     exit;
     end;

   {Clear the PreAllocate file flag bit if it had been set in CurrentBFile.}
   CurrentBFile^.Specs.FileFlags := CurrentBFile^.Specs.FileFlags and $FD;
   CurrentBFile^.Specs.UnusedPgs := 0; {If preallocate file flag was set, the}
                                       {cloned file will have no pages pre- }
                                       {allocated...NO way to get the       }
                                       {original # of pre-allocated pages!  }

   NewSpecLength := CurrentBFile^.SpecLength;        {Initialize...may reduce}
   NewNumKeys    := CurrentBFile^.Specs.NumKeys;     {both of these later.   }
   move(CurrentBFile^.Specs, NewFileSpec^, 16);      {Get filespecs, not keys}

   {Determine if there are any supplemental indexes in source file.  If so,
    set indicator HasSuppIdx to true, set boolean in an array to true, and
    get a count of number of supplemental indexes, and count of total number
    of supplemental index segments.}
   with CurrentBFile^ do
      for Counter := 1 to NumSegs do
         with Specs.KeyArray[Counter-1] do
            begin
            if ((KeyFlags and AltCol) = AltCol) and
               ((KeyFlags and Supplemental) <> Supplemental) then
               PermKeyHasAltCol := true;
            if (KeyFlags and Supplemental) = Supplemental then
               begin
               if (KeyFlags and AltCol) = AltCol then
                  SuppIdxHasAltCol^[NumberSuppSegs] := true;
               HasSuppIdx := true;
               SuppIdxList^[Counter-1] := true;
               move(Specs.KeyArray[Counter-1], SuppIdx^[NumberSuppSegs],
                    KeySpecSize);
               SuppIdx^[NumberSuppSegs].KeyFlags :=    {Zero supplemental bit}
               SuppIdx^[NumberSuppSegs].KeyFlags and $FF7F;
               inc(NumberSuppSegs);      {inc count of supplemental segments.}
               if (Specs.KeyArray[Counter-1].KeyFlags and Segmented) <>
                   Segmented then
                  inc(NumberSuppIdx);     {inc count of supplemental indexes.}
               end;            {if (KeyFlags and Supplemental) = Supplemental}
            end;  {with Specs.KeyArray[Counter-1] do}

   if ((Option = Drop) or (Option = Retain)) and HasSuppIdx then
      begin
      Counter1 := 0;
      for Counter := 1 to CurrentBFile^.Specs.NumKeys do
         begin
         if SuppIdxList^[Counter1] = true then dec(NewNumKeys);
         repeat
            if (SuppIdxList^[Counter1] = false) then
               begin
               move(CurrentBFile^.Specs.KeyArray[Counter1],
               NewFileSpec^.KeyArray[Counter1], KeySpecSize);
               inc(NewOffset, KeySpecSize);
               end
            else
               dec(NewSpecLength, KeySpecSize);
            inc(Counter1);
         until (CurrentBFile^.Specs.KeyArray[Counter1-1].KeyFlags and Segmented)
		      <> Segmented;
         end;          {for Counter := 1 to CurrentBFile^.Specs.NumKeys do}

      NewFileSpec^.NumKeys := NewNumKeys;

      if (CurrentBFile^.HasAltCol) = true then
         move(CurrentBFile^.Specs.Entire[17 + (CurrentBFile^.NumSegs * KeySpecSize)],
              NewFileSpec^.Entire[NewOffset], 265);

      {Next line executed if source file has supplemental indexes, whether
       they are to be dropped or retained.}
      CloneFile := Btrv(BCreate, PosBlk, NewFileSpec^, NewSpecLength,
		                  NewBFileName, 0);
      end;{if ((Option = Drop) or (Option = Retain)) and HasSuppIdx}

   {If retaining the supplemental indexes, then at this point we're ready to
    add them to the newly created file.}
   if (Option = Retain) and HasSuppIdx then
      begin
      NewBFile := new(PBFixed, Init(NewFile, Accel, ''));
      Counter1 := 0;
      for Counter := 1 to NumberSuppIdx do
         begin
         repeat
            move(SuppIdx^[Counter1], NewBFile^.DBuffer[DBuffOffset], KeySpecSize);
            inc(DBuffOffset, KeySpecSize);
            inc(Counter1);
         until ((SuppIdx^[Counter1-1].KeyFlags) and Segmented) <> Segmented;
         with NewBFile^ do
            begin
            DBufferLen := Counter1 * KeySpecSize;
            if SuppIdxHasAltCol^[Counter1-1] and (PermKeyHasAltCol = false) then
               begin
               move(CurrentBFile^.Specs.Entire[CurrentBFile^.SpecLength - 264],
                    DBuffer[DBuffOffset], 265);
               inc(DBufferLen, 265);
               end;
            BStatus := Btrv(BCrSuppIdx, PosBlk, DBuffer, DBufferLen, KBuffer, 0);
            fillchar(DBuffer, sizeof(DBuffer), 0);
            end;
         inc(DBuffOffset);
         end;   {for Counter := 1 to NumberSuppIdx do}
      BStatus := NewBFile^.Close;
      CloneFile := BStatus;
      dispose(NewBFile, Done);
      end;   {if (Option = Retain) and HasSuppIdx}

   {WARNING!! If user program specified 'None' and there actually ARE one or
    more supplemental indexes in the source file, they WILL be retained in
    the target file, as permanent indexes!}
   if (Option = None) or ((Option = Retain) and (not HasSuppIdx)) or
      ((Option = Drop) and (not HasSuppIdx)) then
      begin
      BStatus := Btrv(BCreate, PosBlk, CurrentBFile^.Specs,
                      CurrentBFile^.SpecLength, NewBFileName, 0);
      CloneFile := BStatus;
      end;

   dispose(CurrentBFile, Done);
   dispose(NewFileSpec); {Note NewFileSpec is not used if HandleSupps=None}
   dispose(SuppIdxHasAltCol);
   dispose(SuppIdx);
   dispose(SuppIdxList);
end;


(* NEWKEYSPEC Function *)
(* ------------------- *)
function NewKeySpec(KPos, KLen, KFlags: integer; EType: byte;
		    NextKey: PKeyList): PKeyList;
var TheKeyList: PKeyList;
begin
   TheKeyList := new(PKeyList);
   fillchar(TheKeyList^, sizeof(TheKeyList^), 0);
   with TheKeyList^.KeySpec do
      begin
      KeyPos := KPos;
      KeyLen := KLen;
      KeyFlags := KFlags;
      ExtKeyType := EType;
      end;
   TheKeyList^.Next := NextKey;
   NewKeySpec := TheKeyList;
end;


(* IS BTRIEVE LOADED procedure *)
(* --------------------------- *)
{this is private to the unit, and is executed only during unit initialization}
function IsBtrieveLoaded: boolean;
var
   Version: TVersion;
   VersionSize: word;
   PosBlk: array[1..128] of char;
   KeyBuf: array[1..sizeof(TVersion)] of char;
begin
   VersionSize := sizeof(TVersion);
   BStatus := Btrv(BVersion, PosBlk, Version, VersionSize, KeyBuf, 0);
   if BStatus = 0 then
      IsBtrieveLoaded := true
   else
      IsBtrieveLoaded := false;
end;


(* MISCELLANEOUS Functions *)
(* ----------------------- *)

{LTrim and RTrim were taken from one of the Turbo Vision .PAS source files!}

function LTrim(S: String): String;
var
   I: integer;
begin
   I := 1;
   while (I < length(S)) and (S[I] = ' ') do inc(I);
   LTrim := copy(S, I, 255);
end;

function RTrim(S: String): String;
var
   I: integer;
begin
   while S[Length(S)] = ' ' do dec(S[0]);
   RTrim := S;
end;


(* INITIALIZATION Section *)
(* ----------------------------------------------------------------------- *)
END.

