{.he Stack Maintenance Module - %F}
(********************************************************************)
(*                             Stacks                               *)
(*                                                                  *)
(*  Author: Geoff Moehrke                                           *)
(*  Date: August 27, 1988                                           *)
(*                                                                  *)
(*  Purpose: Stack implementation, allows stacking of any data type *)
(*           using Turbo Pascal's generic pointers to reference     *)
(*           stacked data.                                          *)
(*                                                                  *)
(*  Source: F:\TP\UNIT\STACKS.PAS                                   *)
(********************************************************************)
Unit Stacks;

interface

type
  StackPtr = ^StackRec;
  StackRec = record
               Data: Pointer;   { Generic pointer to data  }
               Next: StackPtr;  { Pointer to next StackRec }
             end;

  Stack = record
             DataSize : Word;   { Size of data in this queue          }
             Top : StackPtr;    { Pointer to top of stack             }
             Height,            { Current Height of stack             }
             MaxHeight : Word;  { Maximum length stack reached so far }
           end;

 procedure InitStack (Var S: Stack; Size: Word );
   {- Initialize stack variables and set up for processing.          }
   {  Must be called for each stack before performing any operations }
   {  on it.                                                         }
   {  Example: InitStack ( MyStack, SizeOf(MyType) );                }


 function Push(Var S:Stack; Item: Pointer) : boolean;
   {- Push Item onto stack S.  Returns True if successful, false     }
   {  if insufficient memory.                                        }
   {                                                                 }
   {  Parameters:  S - stack variable                                }
   {               Item - generic pointer to item.  easiest way is   }
   {                      to use address operator (@AnyVar)          }
   {                                                                 }
   {  Example: if Push( MyStack, @MyVar) then                        }
   {              ...                                                }

 function Pop(Var S: Stack):Pointer;
   {- Pops top item off stack, returns a pointer to it, nil if       }
   {  stack is empty.                                                }
   {                                                                 }
   {  Easiest way to access data after popping it is to use          }
   {  typecasting to a pointer to the data type being stacked.       }
   {  Example:                                                       }
   {                                                                 }
   {            If Not StackEmpty(MyStack) then                      }
   {               IntVar := Integer( Pop(MyStack)^ );               }
   {                                                                 }
   {                                                                 }
   {  Where: IntPtr is defined as ^Integer & NumP is of type IntPtr. }
   {  Note that in this method the check for StackEmpty is           }
   {  necessary to prevent Pop(MyStack) from accessing a nil pointer.}

 function StackEmpty(S: Stack):Boolean;
   {- determine whether stack is empty or not }

 function StackSize(S: Stack):Word;
   {- returns current size (height) of stack S }

 function MaxStackSize(S: Stack):Word;
   {- ruturns the maximum size (height) that S has reached  }

implementation

 procedure InitStack(Var S: Stack; Size: Word);
   {- Initialize stack variables and set up for processing }

   begin
     with S do begin
       Top := nil;
       DataSize := Size;
       Height := 0;
       MaxHeight := 0;
     end;
   end;

 {$F+}
 function HeapFunc( Size : Word ): Integer;
   { Prevent New() or GetMem() from causing a run-time error if
     memory not available (see TP Manual - Chapter 15).        }

   begin
     HeapFunc := 1;
   end;
 {$F-}

 function Push(Var S:Stack; Item: Pointer): boolean;
   {- Push Item onto stack S, returns false if not enough heap space }

   var NewItem, Temp:StackPtr;

   begin
     Push := False;
     New(NewItem);
     If NewItem = nil then
       Exit;                { Not enough memory }
     GetMem(NewItem^.Data,S.DataSize);
     If NewItem^.Data = nil then
       Exit;
     Move(Item^,NewItem^.Data^,S.DataSize);
     With S do begin
       Temp := Top;
       Top := NewItem;         { add new item at end }
       Top^.Next := Temp;
       inc(Height);             { increment the height }
       if Height > MaxHeight then  { adjust max height }
         MaxHeight := Height;
       Push := True;
     end
   end;

 function Pop(Var S: Stack):Pointer;
   {- Returns a pointer item popped from S, nil if stack is empty }

   var Temp : StackPtr;

   begin
     Pop := nil;
     if S.Height = 0 then  { stack is empty - exit w/nil pointer }
       exit;
     with S do begin
       Dec(Height);
       Pop := Top^.Data;
       Top := Top^.Next;
     end;
   end;

 function StackEmpty(S: Stack):Boolean;
   {- determine whether stack S is empty or not }

   begin
     StackEmpty := S.Height = 0;
   end;

 function StackSize(S: Stack):Word;
   {- returns current size of stack }

   begin
     StackSize := S.Height;
   end;

 function MaxStackSize(S: Stack):Word;
   {- returns maximum height stack S has reached }

   begin
     MaxStackSize := S.MaxHeight;
   end;


end. { Unit Stacks }