(*************************************************************************)
(*                               DataEntry                               *)
(*                                                                       *)
(* Author:  Geoffrey Moehrke                                             *)
(* Date:    February 13, 1989                                            *)
(*                                                                       *)
(* Purpose: Allow user to input a group of values of mixed types in a    *)
(*          window, moving from field to field using arrow keys.         *)
(*                                                                       *)
(* Source: F:\TP\UNIT\DATAENTR.PAS                                       *)
(*************************************************************************)
Unit DataEntry;

Interface

    Uses
      TPCRT,
      {$IFDEF UseClock}
      TPClock,
      {$ENDIF}
      TPWindow,
      TPString,
      TPEdit,
      Keys,
      Messages;

  const

     MaxFields = 12;         { Max number of data entry fields     }

     DEWinWidth = 65;        { Default width of the window         }

     DEAcceptKey: Word = F2; { Default key to accept entered data  }

  type

     FieldType = (DE_Y,    { boolean }
                  DE_B,    { byte    }
                  DE_I,    { integer }
                  DE_W,    { word    }
                  DE_L,    { longint }
                  DE_R,    { real    }
                  DE_C,    { char    }
                  DE_S );  { string  }

     InputDesc = Record
                   Defined  : boolean;
                   Prompt   : string;
                   FldType  : FieldType;
                   FieldLen : byte;
                   MaxVal,
                   MinVal   : longint;
                   DecPlaces,
                   Size     : Byte;
                   Data     : Pointer;
                 end;


     DescArray = Array[1..MaxFields] of InputDesc;

     StrFunc = function( FieldNum : byte; Data: pointer ): string;
       { Function to alter how a field appears on the screen -
         called each time each field is drawn on the screen.    }

    var
      Fields           : DescArray;
      DefinedFlds      : byte;
      DEWinAttr,
      DEFrameAttr,
      DEHeaderAttr,
      DELoAttr,
      DEHiAttr,
      DESelectAttr    : byte;
      DefUsrFunc      : StrFunc;


    procedure DefineField( FN : byte;        { Field Number    }
                           P  : string;      { Prompt          }
                           FT : FieldType;   { Type            }
                           FL : byte;        { Field Length    }
                           Min,              { Max Val         }
                           Max: longint;     { Min Val         }
                           DP,               { Decimal Places  }
                           Sz : byte;        { Size of data    }
                           Ptr: pointer );   { Pointer to data }

      { Define the field - does not check if already defined just redefines }


    procedure UndefineField( FN : byte );
      { Undefine the field FN }

    procedure UndefineAllFields;
      { Undefine all defined data entry fields }

    function DataGet( Title: String;
                      Edit: Boolean;
                      UsrFunc: StrFunc ): Boolean;

      { Read in the defined fields, if Edit param is false, assumes that
        Title will contain a yes or no question and returns a corresponding
        True/False value of the user response                               }

  implementation

    procedure DefineField( FN : byte;        { Field Number    }
                           P  : string;      { Prompt          }
                           FT : FieldType;   { Type            }
                           FL : byte;        { Field Length    }
                           Min,              { Max Val         }
                           Max: longint;     { Min Val         }
                           DP,               { Decimal Places  }
                           Sz : byte;        { Size of data    }
                           Ptr: pointer );   { Pointer to data }


    { Define the field - does not check if already defined just redefines }

    begin
      With Fields[FN] do begin
        If Not Defined then begin
          Defined := True;
          Inc( DefinedFlds ); { Update global counter }
        end;
        Prompt := P;
        FldType := FT;
        FieldLen := FL;
        MaxVal := Max;
        MinVal := Min;
        DecPlaces := DP;
        Size := Sz;
        Data := Ptr;
      end
    end;


    procedure UndefineField( FN : byte );

    { Undefine the field }

    begin
      With Fields[FN] do
       If Defined then begin
         Fields[FN].Defined := False;
         If DefinedFlds > 0 then
           Dec( DefinedFlds )
       end
    end;

    procedure UndefineAllFields;
      { Undefine all data entry fields }

      var I : Byte;

    begin
      For I := 1 to DefinedFlds do
        UndefineField(I);
    end;

{$F+}
    function DefStrFunc( FieldNum : Byte; Data : Pointer ): String;

    begin
      Case Fields[FieldNum].FldType of
        DE_Y : If boolean( Data^ ) then
                 DefStrFunc := 'Y'
               else
                 DefStrFunc := 'N';
        DE_B  : DefStrFunc := Long2Str( byte( Data^ ) );
        DE_I  : DefStrFunc := Long2Str( integer( Data^ ) );
        DE_W  : DefStrFunc := Long2Str( word( Data^ ) );
        DE_L  : DefStrFunc := Long2Str( longint( Data^ ) );
        DE_R  : DefStrFunc := Real2Str(real( Data^ ),7,2);
        DE_C  : DefStrFunc := char( Data^ );
        DE_S  : DefStrFunc := String( Data^ )
      end
    end; { DefStrFunc }
{$F-}


    function DataGet( Title: String;
                      Edit: Boolean;
                      UsrFunc: StrFunc ): Boolean;

      { Read in the defined fields }

     var

       Escaped,
       Return,
       EditCh,
       InvKey : boolean;
       Temp   : boolean;
       Pos    : byte;
       Key    : Word;
       DEWin  : WindowPtr;
       Int    : integer;
       St     : string;
       Bool   : boolean;
       Wrd    : word;
       LInt   : longint;
       Ch     : char;
       R      : real;

    begin
    DataGet := False;
    CursorToEnd := False;

    { NOTE: If these fields are defined within the calling program - they will
            need to be re-defined after each call to DataGet                   }

      if Not AddEditCommand( RSUser0, 1, UpArrow, $0000 ) Or
         Not AddEditCommand( RSUser1, 1, DnArrow, $0000 ) Or
         Not AddEditCommand( RSUser2, 1, F2,      $0000 ) then
           Message(TitleCmd+PauseCmd+TitleCmd+'Program Error - edit key array full - DataGet procedure');

      Escaped := False;
      If Not MakeWindow(DEWin,
                        40 - (DEWinWidth div 2) - 1, 12 - (DefinedFlds Div 2) - 2,
                        40 + (DEWinWidth div 2) + 1, 12 + (DefinedFlds Div 2) + 2,
                        True, True, False,
                        DEWinAttr, DEFrameAttr, DEHeaderAttr,'') then begin
          Message(TitleCmd+PauseCmd+TitleCmd+'Not Enough Memory.');
          Exit
        end;
      If Not DisplayWindow(DEWin) then begin
        Message(TitleCmd+PauseCmd+TitleCmd+'Not Enough Memory.');
        Exit
      end;
      FastWriteWindow( Center( Title, DEWinWidth ), 1, 1, DEHeaderAttr );

      For Pos := 1 to DefinedFlds do  { Write initial data }
        With Fields[Pos] do begin
          FastWriteWindow( Pad(Prompt, DEWinWidth) , Pos+2, 1, DEHiAttr);
          FastWriteWindow( UsrFunc( Pos, Data ), Pos+2, Length( Prompt )+2 ,DELoAttr );
        end;

      Pos := 1;
      If Edit then
        Repeat
          invKey := False;
          With Fields[Pos] do begin
            FastWriteWindow( pad(Prompt, DEWinWidth), Pos+2,1,DESelectAttr );
            FastWriteWindow( UsrFunc( Pos, Data ), Pos+2,Length(Prompt)+2, DESelectAttr );
            GotoXY( Length( Prompt ) + 2, Pos+2 );

            Repeat
            until CheckKbd( Key );  { Wait for keystroke }
            EditCh :=  EditKey( Key );
            If (char(lo(Key)) In[#32..#126]) Or (EditCh) then begin
              Case FldType of
               DE_B : If (char(lo(Key)) in ['0'..'9']) Or EditCh then begin
                        Int := integer( byte( Data^ ) );
                        ReadInteger( '', Pos+2, Length(Prompt)+2, FieldLen,
                                     DESelectAttr, DESelectAttr,
                                     MinVal, MaxVal, Escaped, Int);
                        Move( byte(Int), Data^, Size )
                        end
                      else
                        InvKey := True;
               DE_Y : If UpCase(char(lo(key))) in ['Y','N'] then begin
                         Temp := ShowReadChar;
                         ShowReadChar := False;
                         bool := YesOrNo( '', Pos+2, Length(Prompt)+2,
                                          DESelectAttr, Ch );
                         ShowReadChar := Temp;
                         Move(bool, Data^, Size )
                         end
                       else
                         InvKey := True;


               DE_I : If (char(lo(Key)) in ['0'..'9','-']) Or EditCh then begin
                        Int := integer( Data^ );
                        ReadInteger( '', Pos+2, Length(Prompt)+2, FieldLen,
                                     DESelectAttr, DESelectAttr,
                                     MinVal, MaxVal, Escaped, Int);
                        Move(Int, Data^, Size)
                        end
                      else
                        Key := ReadkeyWord;
               DE_W : If (char(lo(Key)) in ['0'..'9']) Or EditCh then begin
                        Wrd := word( Data^ );
                        ReadWord( '', Pos+2, Length(Prompt)+2, FieldLen,
                                     DESelectAttr, DESelectAttr,
                                     MinVal, MaxVal, Escaped, Wrd);
                        Move( Wrd, Data^, Size )
                        end
                      else
                        InvKey := True;
               DE_L : If (char(lo(Key)) in ['0'..'9','-']) or EditCh then begin
                        LInt := longint( Data^ );
                        ReadLongInt( '', Pos+2, Length(Prompt)+2, FieldLen,
                                     DESelectAttr, DESelectAttr,
                                     MinVal, MaxVal, Escaped, LInt);
                        Move( LInt, Data^, Size );
                        end
                      else
                        InvKey := True;
               DE_R : If (char(lo(Key)) in ['0'..'9','.','-']) or EditCh then  begin
                        R := real( Data^ );
                        ReadReal( '', Pos+2, Length(Prompt)+2, FieldLen,
                                  DESelectAttr, DESelectAttr,
                                  DecPlaces, MinVal*1.0, MaxVal*1.0, Escaped, R);
                        Move( R, Data^, Size )
                      end
                      else
                        InvKey := True;

               DE_C : begin
                        Ch := char( Data^ );
                        ReadCharacter( '', Pos+2, Length(Prompt)+2,
                                       DESelectAttr, [#32..#255],
                                       Ch );
                        Move( Ch, Data^, Size )
                      end;

               DE_S : begin
                        St := string( Data^ );
                        ReadString( '', Pos+2, Length(Prompt)+2, FieldLen,
                                   DESelectAttr, DESelectAttr, DESelectAttr,
                                   Escaped, St );
                        Move( St, Data^, Size )
                     end;
              End;
              FastWriteWindow( Pad(Prompt, DEWinWidth) ,Pos+2,1,DEHiAttr);
              FastWriteWindow( UsrFunc( Pos, Data ),Pos+2,Length(Prompt)+2 ,DELoAttr );
              If RSCommand = RSUser2 then
                Key := DEAcceptKey
              else If RSCommand = RSUser0 then begin
                If Pos > 1 then Dec(Pos) else Pos := DefinedFlds;
                end
              else if InvKey Then
                Key := ReadKeyWord   { Flush invalid keystrokes }
              else                   {       otherwise          }
                Inc(Pos);            { move to next field       }
              If Pos > DefinedFlds then
                Pos := 1;
            end
          else begin
            Key := ReadKeyWord;
            FastWriteWindow( Pad(Prompt,DEWinWidth), Pos+2,1,DEHiAttr);
            FastWriteWindow( UsrFunc( Pos, Data ),Pos+2,Length(Prompt)+2 ,DELoAttr );

            If Key = Enter then
              Key := DnArrow;

            If Key = UpArrow then
              If Pos > 1 then Dec(Pos) else Pos := DefinedFlds;

            If Key = DnArrow then
              If Pos < DefinedFlds then Inc(Pos) else Pos := 1;

          end;
          If Escaped then Key := ESC;
          Return := Not Escaped
        end
        until (Key = ESC)
           Or (Key = DEAcceptKey)

      else begin
        FastWriteWindow( Pad( Title, DEWinWidth ), 1, 1, DEHeaderAttr);
        Return := YesOrNo('', 1, Length( Title )+1 , DEHeaderAttr, ' ');
      end;
      If Key = ESC then
        Return := False;
      DEWin := EraseTopWindow;
      DisposeWindow( DEWin );
      DataGet := Return;
    end;




Begin
  FillChar(Fields, SizeOf( Fields ), #0 );
  DefUsrFunc := DefStrFunc;
  DefinedFlds := 0;
  if LastMode In [2,7] then begin
    DEWinAttr    := $07;
    DEFrameAttr  := $07;
    DEHeaderAttr := $0F;
    DELoAttr     := $0F;
    DEHiAttr     := $07;
    DESelectAttr := $70;
  end
  else begin
    DEWinAttr    := $47;
    DEFrameAttr  := $47;
    DEHeaderAttr := $4F;
    DELoAttr     := $4B;
    DEHiAttr     := $47;
    DESelectAttr := $1F;
  end
end.