Unit FileObj;

{$V-}
{$R-}
{$S-}

Interface

{
 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
  FILEOBJ.PAS

    This unit contains the object FileObjType.  This object collects and makes
    available information about a given file.  It is a handy way to check
    various features of a file without a lot of coding.

    To use this object, either define a variable of the type FileObjType, or
    create a dynamic object with a call to the New procedure/function.
    Initialize the file with the name of the file of interest.  Among other
    things, the object can tell you

            - if the file exists
            - the fully qualified file name
            - the file base name
            - the file date, time and attributes
            - the file size

    Note that a given instance of this object can be used over and over
    within the same program.  Simply reinitialize the unit with a new
    file name.

    I wrote this to try out OOP.  I find it especially useful for
    command line parameter checking in my programs.

    Bill Matsoukas [ 76416,3622 ]

 ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 }

Uses Dos;

Type

  DriveStr = String[2]; { should this be standard in the DOS unit ?!? }
  DateStr = String[10];
  TimeStr = String[8];
  AttrStr = String[5];

  FileObjPtr = ^FileObjType;  { for dynamically allocated objects }

  FileObjType = Object
                          { object data }
    FPath     : PathStr;
    FDrive    : DriveStr;
    FBaseName : NameStr;
    FExt      : ExtStr;
    FDir      : DirStr;
    FLength   : LongInt;
    FExists   : Boolean;
    FDate     : DateStr;
    FTime     : TimeStr;
    FAttrStr  : AttrStr;
    FAttr     : Word;
                          { object methods }

    Constructor Init ( FileName : PathStr ); { init with fully or partially
                                               quallified file name            }
    Destructor Done; Virtual;                { empty function for deallocation }
    Function GetPath : PathStr;              { returns fully qualified name    }
    Function GetDrive : DriveStr;            { returns drive letter and colon  }
    Function GetBaseName : NameStr;          { returns stripped file name      }
    Function GetExt : ExtStr;                { returns dot plus extension      }
    Function GetDir : DirStr;                { returns path to file            }
    Function GetLength : LongInt;            { returns file length in bytes    }
    Function FGetDate : DateStr;             { returns file date in string
                                               form "YYYY-MM-DD"               }
    Function FGetTime : TimeStr;             { returns the file time in string
                                               form "HH:MM:SS"                 }
    Function FGetAttr : Word;                { returns file attribute word     }
    Function FGetAttrStr : AttrStr;          { returns the file attribute in
                                               unix-like string "RHSAX" where
                                                  R = read only
                                                  H = hidden
                                                  S = system
                                                  A = archive
                                                  X = executable
                                               a dash ("-") character replaces
                                               false attributes.               }
    Function FileExists : Boolean;           { true if file exists, else false }

  End;

Implementation

{ *******  private routines used by FileObjType methods  ******* }

Procedure SubZeros ( Var S : String );
Var
  I : Byte absolute S;
  J : Byte;
Begin
  If I = 0 then
    Exit;
  For J := 1 to I do
    If S[J] = ' ' then
      S[J] := '0';
End;

Function StrObjDate ( Yr, Mo, Da : Word ) : DateStr;
Var
  YrStr : String[4];
  MoStr : String[2];
  DaStr : String[2];
  DtStr : DateStr;
Begin
  Str ( Yr:4, YrStr );
  Str ( Mo:2, MoStr );
  Str ( Da:2, DaStr );
  DtStr := YrStr + '-' + MoStr + '-' + DaStr;
  SubZeros ( DtStr );
  StrObjDate := DtStr;
End;

Function StrObjTime ( Hr, Mi, Se : Word ) : TimeStr;
Var
  HrStr : String[2];
  MiStr : String[2];
  SeStr : String[2];
  TmStr : TimeStr;
Begin
  Str ( Hr:4, HrStr );
  Str ( Mi:2, MiStr );
  Str ( Se:2, SeStr );
  TmStr := HrStr + ':' + MiStr + ':' + SeStr;
  SubZeros ( TmStr );
  StrObjTime := TmStr;
End;

Function StrObjAttr ( Attr : Word; Ext : ExtStr ) : AttrStr;
Var
  S : AttrStr;
Begin
  S := '-----';
  If ( ( Attr and ReadOnly ) = ReadOnly ) then S[1] := 'R';
  If ( ( Attr and Hidden ) = Hidden ) then S[2] := 'H';
  If ( ( Attr and SysFile ) = SysFile ) then S[3] := 'S';
  If ( ( Attr and Archive ) = Archive ) then S[4] := 'A';
  If ( Ext = '.COM' ) or
     ( Ext = '.EXE' ) or
     ( Ext = '.BAT' ) then S[5] := 'X';
  StrObjAttr := S;
End;

{ *******  FileObjType methods  ********* }

Constructor FileObjType.Init ( FileName : PathStr );
Var
  FRec : SearchRec;
  DRec : DateTime;
Begin
  FPath := FExpand ( FileName );
  FSplit ( FPath, FDir, FBaseName, FExt );
  FDrive := Copy ( FDir, 1, 2 );
  FDir := Copy ( FDir, 3, Length ( FDir ) - 2 );
  FindFirst ( FPath, AnyFile Xor ( Directory Or VolumeId ), FRec );
  FExists := ( DosError = 0 );
  If FExists then
  Begin
    FLength := FRec.Size;
    UnPackTime ( FRec.Time, DRec );
    FDate := StrObjDate ( DRec.Year, DRec.Month, DRec.Day );
    FTime := StrObjTime ( DRec.Hour, DRec.Min, DRec.Sec );
    FAttr := FRec.Attr;
    FAttrStr := StrObjAttr ( FAttr, FExt );
  End
  Else
  Begin
    Flength := 0;
    FAttr := 0;
    FillChar ( FDate, SizeOf ( FDate ), 0 );
    FillChar ( FTime, SizeOf ( FTime ), 0 );
    FillChar ( FAttrStr, SizeOf ( FAttrStr ), 0 );
  End;
End;

Destructor FileObjType.Done;
Begin
End;

Function FileObjType.GetPath : PathStr;
Begin
  GetPath := FPath;
End;

Function FileObjType.GetDrive : DriveStr;
Begin
  GetDrive := FDrive;
End;

Function FileObjType.GetBaseName : NameStr;
Begin
  GetBaseName := FBaseName;
End;

Function FileObjType.GetExt : ExtStr;
Begin
  GetExt := FExt;
End;

Function FileObjType.GetDir : DirStr;
Begin
  GetDir := FDir;
End;

Function FileObjType.GetLength : LongInt;
Begin
  GetLength := FLength;
End;

Function FileObjType.FGetDate : DateStr;
Begin
  FGetDate := FDate;
End;

Function FileObjType.FGetTime : TimeStr;
Begin
  FGetTime := FTime;
End;

Function FileObjType.FGetAttr : Word;
Begin
  FGetAttr := FAttr;
End;

Function FileObjType.FGetAttrStr : AttrStr;
Begin
  FGetAttrStr := FAttrStr;
End;

Function FileObjType.FileExists : Boolean;
Begin
  FileExists := FExists;
End;

End.

