/*
main.c

1.0 -- Original version, 9/19/1989, Les Hancock
1.1 -- Next original version, 9/25/1989, Les Hancock
       Added edit_me.c and edit_me.h to help users add new sorts.
       Added banner.c to put up a nice header page.
       Added menus for colors
+++ -- Your changes should be documented here

*/

/***************************************************

         INCLUDES

****************************************************/
#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <graph.h>
#include <stdio.h>
#include <stdlib.h>
#include "defs.h"
#include "banner.h"
#include "edit_me.h"

/***************************************************
         PROTOCOLS FOR PRIVATE FUNCTIONS
****************************************************/
STATIC BOOLEAN filter(char c, FILTER current_filter);
STATIC DISPLAY_TYPE get_display_type(void);
STATIC char *trim(char *buf);
STATIC signed char get_input_file(char *input_file);
STATIC signed char keypress(void);
STATIC signed char menu(MENUPTR menuptr);
STATIC void cursor_control(CURSOR_COMMAND command);
STATIC void blank_line(unsigned char line_number, short int color);
STATIC void display_menu_line(MENUPTR menuptr, int i, int mode);
STATIC void fill_ram_from_file(char *ptr, unsigned int how_many_chars,
  char *input_file, FILTER current_filter, unsigned char color);
STATIC void fill_ram_at_random(char *ptr,
  unsigned int how_many_chars, FILTER current_filter, unsigned char color);
STATIC PTR_TO_FN_RET_VOID replace_int_vector(VECTOR vector_addr,
  PTR_TO_FN_RET_VOID new_vector);
STATIC void display_line(char *ptr, short int row, short int col,
  unsigned char color);
STATIC void sort(MENU menus[], int slowdown_factor, char *input_file,
  FILTER current_filter, DISPLAY display, unsigned char current_color);
STATIC void update_status_line(int slowdown_factor, char *input_file,
  FILTER current_filter, signed char current_input_mode,
  char *current_sort, char *trailer);

/***************************************************
         PROTOCOLS FOR EXTERNAL FUNCTIONS
****************************************************/
extern void ctl_brk_handler(void);
extern void save_ds(void);
extern void timer_handler(void);

/***************************************************
      PROTOCOLS FOR EXPORT (see demosort.h)
****************************************************/
unsigned int slowdown(int slowdown_factor);
void exchange(char *p, register int i, register int j);
void int_exchange(int *p, register int i, register int j);

/***************************************************
      GLOBAL VARIABLES
****************************************************/
volatile long int clock_ticks;       /* bumped by system tick, 18.2 per sec */
volatile unsigned char flag;                      /* set to 1 on cntl-break */

/***************************************************
         MAIN FUNCTION
****************************************************/
main(int argc, char **argv)
{
   DISPLAY display;
   PTR_TO_FN_RET_VOID old_ctl_brk_handler, old_timer_handler;
   FILTER current_filter = none;
   int slowdown_factor = 0;
   unsigned char current_color = COLOR(WHITE, BLACK);
   static char input_file[80];                              /* empty string */

/***************************************************
         Text for Menus
*/
   static char *main_menu_strings[] =
     {
      "EXIT",
      "set input",
      "perform sort",
      "choose sort algorithm",
      "slow motion factor",
      "choose text filter",
      "choose colors"
   }
   ;
   static char *input_menu_strings[] =
     {
      "random chars",
      "file",
      "keyboard"
   }
   ;
   static char *speed_menu_strings[] =
     {
      "1",
      "2",
      "4",
      "8",
      "16",
      "32"
   }
   ;
   static char *filter_menu_strings[] =
     {
      "no filter",
      "alphanumeric",
      "alphabetic",
      "lowercase",
      "numeric",
      "printable",
      "printablex",
      "punctuation",
      "uppercase"
   }
   ;
   static char *color_menu_strings[] =
     {
      "text chars",
      "background"
   }
   ;
   static char *colors_menu_strings[] =
     {
      "black",
      "blue",
      "green",
      "cyan",
      "red",
      "magenta",
      "brown",
      "white",
      "dark gray",
      "light blue",
      "light green",
      "light cyan",
      "light red",
      "light magenta",
      "yellow",
      "bright white"
   }
   ;

/***************************************************
         Text for Synopses
       (Explanations of Menu Choices)
*/
   static char *main_menu_synopses[] = {
      "Return to DOS.",
      "Specify source of input for page of text to be sorted.",
      "Perform sort using the parameters shown below.",
      "Pick an algorithm to sort the sample text.",
      "Pick a value to slow the sort for closer scrutiny.",
      "Select a filter to eliminate unwanted characters in sample text.",
      "Choose colors for the page of sample text."
   };

   static char *input_menu_synopses[] = {
      "Input will be a page of pseudo-random characters and numbers.",
      "Hit ENTER now, then name a text file for input to sample sort.",
      "Input will be entered from the keyboard.  (Not currently implemented.)"
   };

   static char *speed_menu_synopses[] = {"","","","","",""};

   static char *filter_menu_synopses[] = {
      "Sample text will not be filtered in any way.",
      "Sample text will contain letters A-Z, a-z, and digits 0-9.",
      "Sample text will contain letters A-Z, a-z.",
      "Sample text will contain letters a-z.",
      "Sample text will contain digits 0-9.",
      "Sample text will contain all printable characters including blank space.",
      "Sample text will contain all printable characters except blank space.",
      "Sample text will contain only punctuation marks.",
      "Sample text will contain letters A-Z."
   };

   static char *color_menu_synopses[] = {
      "Choose color of sort characters.",
      "Choose color of sort page's background."
   };

   static char *colors_menu_synopses[] = {
      "","","","","","","","","","","","","","","",""
   };

   static unsigned char main_menu_colors[] = {
      COLOR(RED, WHITE),
      COLOR(RED, WHITE),
      COLOR(RED, WHITE),
      COLOR(RED, WHITE),
      COLOR(RED, WHITE),
      COLOR(RED, WHITE),
      COLOR(RED, WHITE)
   };

   static unsigned char input_menu_colors[] = {
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE)
   };

   static unsigned char speed_menu_colors[] = {
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE)
   };

   static unsigned char filter_menu_colors[] = {
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE)
   };

   static unsigned char color_menu_colors[] = {
      COLOR(LIGHT_BLUE,WHITE),
      COLOR(LIGHT_BLUE,WHITE)
   };

   static unsigned char foreground_menu_colors[] = {
      COLOR(BRIGHT_WHITE,BLACK),       /* shd be BLACK, BLACK -- impossible */
      COLOR(BRIGHT_WHITE,BLUE),
      COLOR(BRIGHT_WHITE,GREEN),
      COLOR(BRIGHT_WHITE,CYAN),
      COLOR(BRIGHT_WHITE,RED),
      COLOR(BRIGHT_WHITE,MAGENTA),
      COLOR(BRIGHT_WHITE,BROWN),
      COLOR(BRIGHT_WHITE,BLACK),
      COLOR(BRIGHT_WHITE,BLACK),       /* shd be dark gray -- problem child */
      COLOR(BRIGHT_WHITE,LIGHT_BLUE),
      COLOR(BRIGHT_WHITE,LIGHT_GREEN),
      COLOR(BRIGHT_WHITE,LIGHT_CYAN),
      COLOR(BRIGHT_WHITE,LIGHT_RED),
      COLOR(BRIGHT_WHITE,LIGHT_MAGENTA),
      COLOR(BRIGHT_WHITE,YELLOW),
      COLOR(BRIGHT_WHITE,BLACK)
   };

   static unsigned char background_menu_colors[] = {
      COLOR(BLACK,BRIGHT_WHITE),        /* shd be BLACK, BLACK -- impossible */
      COLOR(BLUE,BRIGHT_WHITE),
      COLOR(GREEN,BRIGHT_WHITE),
      COLOR(CYAN,BRIGHT_WHITE),
      COLOR(RED,BRIGHT_WHITE),
      COLOR(MAGENTA,BRIGHT_WHITE),
      COLOR(BROWN,BRIGHT_WHITE),
      COLOR(BLACK,BRIGHT_WHITE)
   };

/***************************************************
         Definition of Array of Menus

         menus[0] = main menu
         menus[1] = input methods
         menus[2] = sort algorithms
         menus[3] = slowdown factors
         menus[4] = filters
         menus[5] = colors
         menus[6] = foreground colors
         menus[5] = background colors
*/
   static MENU menus[] =   
     {
      {
         0,                                                    /* start_row */
         0,                                                    /* start_col */
         22,                                                       /* width */
         sizeof(main_menu_strings) / sizeof(char *),              /* length */
         main_menu_strings,                                      /* choices */
         main_menu_synopses,                                    /* synopses */
         main_menu_colors,                                        /* colors */
         0                                                /* current_choice */
      }
      ,
      {
         1,                                                    /* start_row */
         23,                                                   /* start_col */
         9,                                                        /* width */
         sizeof(input_menu_strings) / sizeof(char *),             /* length */
         input_menu_strings,                                     /* choices */
         input_menu_synopses,                                   /* synopses */
         input_menu_colors,                                       /* colors */
         RANDOM                                           /* current_choice */
      }
      ,
      {
         3,                                                    /* start_row */
         23,                                                   /* start_col */
         22,                                                       /* width */
         0,                                   /* length resolved at runtime */
         sort_menu_strings,                                      /* choices */
         sort_menu_synopses,                                    /* synopses */
         sort_menu_colors,                                        /* colors */
         0                                                /* current_choice */
      }
      ,
      {
         4,                                                    /* start_row */
         23,                                                   /* start_col */
         3,                                                        /* width */
         sizeof(speed_menu_strings) / sizeof(char *),             /* length */
         speed_menu_strings,                                     /* choices */
         speed_menu_synopses,                                   /* synopses */
         speed_menu_colors,                                       /* colors */
         0                                                /* current_choice */
      }
      ,
      {
         5,                                                    /* start_row */
         23,                                                   /* start_col */
         13,                                                       /* width */
         sizeof(filter_menu_strings) / sizeof(char *),            /* length */
         filter_menu_strings,                                    /* choices */
         filter_menu_synopses,                                  /* synopses */
         filter_menu_colors,                                      /* colors */
         0                                                /* current_choice */
      },
      {
         6,                                                    /* start_row */
         23,                                                   /* start_col */
         12,                                                       /* width */
         sizeof(color_menu_strings) / sizeof(char *),             /* length */
         color_menu_strings,                                     /* choices */
         color_menu_synopses,                                   /* synopses */
         color_menu_colors,                                       /* colors */
         0                                                /* current_choice */
      },
      {
         6,
         36,
         14,
         sizeof(colors_menu_strings) / sizeof(char *),
         colors_menu_strings,
         colors_menu_synopses,
         foreground_menu_colors,
         0
      },
      {
         7,
         36,
         14,
         sizeof(background_menu_colors) / sizeof(unsigned char),
         colors_menu_strings,
         colors_menu_synopses,
         background_menu_colors,
         0
      }
   }
   ;

/***************************************************
      Executable Code Begins Here

      Some initialization to start with...
*/
   switch (display.type = get_display_type())
   {
   case ega: case vga:                                         /* case cga: */
      display.base_addr = (char *) 0xb8000000;      /* color display page 1 */
      display.mode = _TEXTC80;
      break;
#if 0
   case mda:
      display.base_addr = (char *) 0xb0000000;       /* mono display page 1 */
      display.mode = _TEXTMONO;
      break;
#endif
   case mda: case cga:
   case unknown:
      puts("Requires EGA or VGA display.");
      exit(1);
   }
   if (_setvideomode(display.mode) == 0)
   {
      printf("Can't set video mode %u.", display.mode);
      exit(2);
   }
   save_ds();     /* save DS register in code segment (for intrpt handlers) */
   /*
   Steal two hardware interrupts and take note of the old ones
   so they can be replaced when we're done.
    */
   old_ctl_brk_handler = replace_int_vector(ctl_break, ctl_brk_handler);
   old_timer_handler = replace_int_vector(timer, timer_handler);
   _clearscreen(_GCLEARSCREEN);
   if (argc == 1)            /* unless user chooses to skip it (any arg)... */
      banner();                                    /* ... put up title page */
   menus[2].length = number_of_sort_items;   /* must be resolved at runtime */

/***************************************************
         Main Event Loop
*/
   while(1)
   {
      _clearscreen(_GCLEARSCREEN);
      cursor_control(kill);
      flag = 0;          /* ctl-break will set it to 1 and break the switch */
      update_status_line(slowdown_factor, input_file, current_filter,
        menus[1].current_choice,                              /* input mode */
        menus[2].choices[menus[2].current_choice],          /* current sort */
        "");                                                  /* no trailer */
      /*
      Switch on choice from main menu.  The function menu() displays
      a menu's text and returns the choice selected: top line is #0,
      next is #1, etc.
       */
      switch(menus[MAIN].current_choice = menu(&menus[MAIN]))
      {

/***************************************************
         Exit from Program

      (This is the ONLY way out once you enter the
      main event loop.  It's essential to restore
      those stolen interrupts.)
*/
      case 0:
         _setvideomode(_DEFAULTMODE);
         /*
         Restore the original interrupt vectors.
          */
         (void) replace_int_vector(ctl_break, old_ctl_brk_handler);
         (void) replace_int_vector(timer, old_timer_handler);
         exit(0);                                            /* normal exit */

/***************************************************
         Set Input Venue

      If user chooses FILE_INPUT, get name of file and be sure it's
     possible to open it.  If not, user's prompted for a different name.
     No name at all sets input to default (random characters).
*/
      case 1:
         if (FILE_INPUT == (menus[INPUT].current_choice =
           menu(&menus[INPUT])))
            menus[INPUT].current_choice = get_input_file(input_file);
         update_status_line(slowdown_factor, input_file, current_filter,
           menus[INPUT].current_choice,                       /* input mode */
           menus[SORT].choices[menus[SORT].current_choice],
           "");                                               /* no trailer */
         break;         

/***************************************************
         Perform the Sort
*/
      case 2:
         sort(menus, slowdown_factor, input_file, current_filter,
           display, current_color);
         _clearscreen(_GCLEARSCREEN);
         break;

/***************************************************
         Set Sort Algorithm
*/
      case 3:
         menus[SORT].current_choice = menu(&menus[SORT]);
         update_status_line(slowdown_factor, input_file, current_filter,
           menus[INPUT].current_choice,                       /* input mode */
           menus[SORT].choices[menus[SORT].current_choice],
           "");                                               /* no trailer */
         break;         

/***************************************************
         Set Slowdown Factor
*/
      case 4:
         menus[SLOWDOWN].current_choice = menu(&menus[SLOWDOWN]);
         if (menus[SLOWDOWN].current_choice >= 0 &&
           menus[SLOWDOWN].current_choice <=
           sizeof(speed_menu_strings) / sizeof(char *))
         {
            /*
            Naturally, slowdown_factor is something of a fiction.
            Among other lies, the first choice in the menu, shown
            as 1, is 0 internally -- no slowdown, skip it.  The
            idea is that this slows us down by a factor of 1.
            More to the point, there's no way to implement this
            as a linear slowdown factor.  It's entirely ad hoc.
            For that reason the timings aren't done or shown when
            slowdown_factor is in play.
             */
            int i;

            if (menus[SLOWDOWN].current_choice == 0)
               slowdown_factor = 0;
            else
            {
               for (slowdown_factor = 1, i = 1;
                 i < menus[SLOWDOWN].current_choice;   ++i)
                  slowdown_factor *= 2;
               update_status_line(slowdown_factor, input_file,
                 current_filter,
                 menus[INPUT].current_choice,                 /* input mode */
                 menus[SORT].choices[menus[SORT].current_choice],
                 "");                                         /* no trailer */
            }
         }
         break;

/***************************************************
         Set Current Filter
*/
      case 5:
         /*
         For convenience, enumerated values of <current_filter>
         are set to match menu choices, indexed from 0.
          */
         current_filter = menus[FILTERS].current_choice =
           menu(&menus[FILTERS]);
         update_status_line(slowdown_factor, input_file,   current_filter,
           menus[INPUT].current_choice,                       /* input mode */
           menus[SORT].choices[menus[SORT].current_choice],         /* sort */
           "");                                               /* no trailer */
         break;

/***************************************************
         Set Current Colors
*/
      case 6:
         switch(menus[COLORS].current_choice = menu(&menus[COLORS]))
         {
         case 0:                                 /* foreground (text) color */
            current_color = (unsigned char)COLOR(BACKGROUND(current_color),
              (menus[FORE].current_choice =
              menu(&menus[FORE])));
            break;
         case 1:                                        /* background color */
            current_color = COLOR((menus[BACK].current_choice =
              menu(&menus[BACK])), FOREGROUND(current_color));
            break;
         }
         break;
      }
   }                                                           /* while (1) */
}

/***************************************************
         PUBLIC FUNCTION DEFINITIONS
****************************************************/
/*
Given char vector p[], exchange its ith and jth elements.
*/
void
exchange(char *p, register int i, register int j)
{
   char temp;

   temp = p[i];
   p[i] = p[j];
   p[j] = temp;
}

/*
Given integer vector p[], exchange its ith and jth elements.
*/
void
int_exchange(int *p, register int i, register int j)
{
   int temp;

   temp = p[i];
   p[i] = p[j];
   p[j] = temp;
}

/*
This routine is meant to be included in every sort's inner loop, or wherever
it'll do the most good.  The idea is to provide two services: (1) waste on
each invocation of slowdown() to make the sort slow enough to follow on the
screen; (2) watch for the condition flag == 1, which means somebody hit
ctl-break, triggered interrupt 0x1b, and wants to quit sorting as soon as
possible.  The return value tells the caller whether or not to quit -- if
slowdown() returns 1, quit, else don't.
*/
unsigned int
slowdown(int slowdown_factor)
{
   extern volatile unsigned char flag;            /* set to 1 on cntl-break */
   int i, j;

   for (i = 0; i < slowdown_factor; ++i)
      for (j = 0; j < 512; ++j)
         ;
   return flag;                          /* will be 1 if user hit ctl-break */
}

/***************************************************
         PRIVATE FUNCTION DEFINITIONS
****************************************************/
/*
Blank out the indicated line with your choice of background color.
Notice that we expect caller to be indexing lines from zero.
*/
STATIC void
blank_line(unsigned char line_number, short int color)
{
   int i;

   _setbkcolor((long) color);
   _settextposition(line_number + 1, 0);
   for (i = 0; i < 79; ++i)
      _outtext(" ");
}

/*
Squelch or restore the cursor.  For more on the cursor killer see
Peter Norton, PROGRAMMER'S GUIDE TO THE IBM PC, p174.
*/
STATIC void
cursor_control(CURSOR_COMMAND command)
{
   union REGS regs;

   if (command == kill)
      regs.h.ch = 0x20;                     /* old trick to turn off cursor */
   else if (command == revive)
   {
      regs.h.cl = 7;   /* set cursor ending line to default for text mode 3 */
      regs.h.ch = 6;      /* set cursor starting line to default for mode 3 */
   }
   else return;             /* if called improperly, don't mess with cursor */
   regs.h.ah = 1;                           /* ROM service: set cursor size */
   int86(0x10, &regs, &regs);
}

/*
Display text string with indicated parameters.  It seems that Microsoft's
text routines index from 1, while DOS ROM BIOS routines index from 0.  I'm
going to use 0 throughout, then add an offset of 1 in this routine.
*/
STATIC void
display_line(char *ptr, short int row, short int col, unsigned char colors)
{
   _setbkcolor((long)BACKGROUND(colors));
   _settextcolor((short)FOREGROUND(colors));
   _settextposition(row + 1, col + 1);
   _outtext(ptr);
}

/*
Display the ith line of menu's strings[] on the screen.  If mode is 1, display
in normal colors; if -1, in reverse video.

We have 23 lines at our disposal for menu display, hence the constant divisor.
*/
STATIC void
display_menu_line(MENUPTR menuptr, int i, int mode)
{
   char buf[80];
   char row, col;

   col = menuptr->start_col + (menuptr->width + 1) * 
     ((menuptr->start_row + i) / 23);   /* fold on row 23; 24 - 25 reserved */
   row = (menuptr->start_row + i) % 23;
   sprintf(buf, " %-*s", (int)menuptr->width, menuptr->choices[i]);
   display_line(buf, row, col,
     mode == 1 ? menuptr->colors[i] : COLORSWAP(menuptr->colors[i]));
}

/*
Given a starting address in video RAM, insert <n> alphanumeric characters at
random.  Even, low-memory byte of each word holds the ASCII representation of 
a character, while the odd bit has display attributes, which we keep constant.

The mask 0x7f eliminates non-ASCII characters.
*/
STATIC void
fill_ram_at_random(char *ptr, unsigned int how_many_chars,
  FILTER current_filter, unsigned char color)
{
   register int i;

   for (i = 0; i < how_many_chars; ++i)
   {
      char c;

      while (0 == filter(c = (char) rand() & 0x7f, current_filter));
      *ptr++ = c;                      /* char passed filter; display it... */
      *ptr++ = color;                                 /* ...in living color */
   }
}

/*
Read the first 24 * 80 chars from file into video page 1.  Each char is stored
as the first byte of a two-byte integer whose second byte tells how the
character is to be displayed.  We don't have to test on open() because we've
already tried the file and know it's going to open.

Note that low-memory byte of each word holds the ASCII representation of a
character, while the high-memory bit has display parameters, which we keep
constant at white-on-black.
*/
STATIC void
fill_ram_from_file(char *ptr, unsigned int how_many_chars, char *input_file,
  FILTER current_filter, unsigned char color)
{
   char c;
   register int i, fp;

   fp = open(input_file, O_RDONLY);                            /* MUST open */
   for (i = 0; i < how_many_chars;)
   {
      if (read(fp, &c, 1) != 1)     /* read() returns 0 on EOF, -1 on error */
      {
         /* File didn't fill the video page.  Keep going, but change only the
         attribute bytes -- otherwise the color of the text may change as
         it moves around the page.
         */
         for (++ptr; i < how_many_chars; ++i, ptr += 2)
            *ptr = color;
         break;
      }
      if (filter(c, current_filter) != 0)  /* if it got past the filter ... */
      {
         *ptr++ = c;                                     /* display char... */
         *ptr++ = color;                              /* ...in living color */
         i++;
      }
   }
   close(fp);
}

/*
Given a character and a filter criterion, return nonzero if the character
passes through the filter, else zero.  Can't use a list of pointers to
filter functions, since the Microsoft filters are implemented as macros.

No need for a default case here.
*/
STATIC BOOLEAN
filter(char c, FILTER current_filter)
{
   switch (current_filter)
   {
   case none:
      return 1;
   case alnum:
      return isalnum(c);
   case alpha:
      return isalpha(c);
   case lower:
      return islower(c);
   case digit:
      return isdigit(c);
   case print:
      return isprint(c);
   case graph:
      return isgraph(c);
   case punct:
      return ispunct(c);
   case upper:
      return isupper(c);
   }
}

/*
Is the current display a VGA, EGA, CGA or MDA?  There's no simple way to find
out.  The technique used here is due to Tony Rizzo, "Detecting Video Adapter
Types," PC MAGAZINE, September 26, 1989, p. 383.  (Corrected some apparent
code typos.)  Amended with reference to Roger Stevens, GRAPHICS PROGRAMMING IN
C (M&T Books).  Rizzo's methods are shorter, Stevens' probably more reliable.

Decided to restrict adaptors to EGA and VGA.
*/
STATIC DISPLAY_TYPE
get_display_type()
{
   char *ptr = NULL;
   union REGS regs;

   /*
   Is adapter VGA?  Call ROM BIOS video service 1A
   (read display combination).  If good return, it's VGA.
    */
   regs.h.ah = 0x1a;
   regs.h.al = 0;
   int86(0x10, &regs, &regs);
   if (regs.h.al == 0x1a)
      return vga;
   /*
   Not VGA.  Is there an EGA in the house?  Don't know
   how this works, to tell the truth.  See article.
    */
   regs.h.ah = 0x12;
   regs.h.bl = 0x10;
   int86(0x10, &regs, &regs);
   if (regs.h.bl != 0x10)
   {
      /*
      EGA is present, but is it the active monitor?
      Check the EGA-active bit in DOS data word that
      details state of EGA.
       */
      FP_OFF(ptr) = 0x487;                        /* segment is 0, remember */
      if ((*ptr & 8) == 0)           /* check bit 3 of EGA information byte */
         return ega;                      /* yes, ega is present AND active */
   }
   /*
   If EGA is present it's not active, so user must have
   an alternate display: CGA or MDA.
    */
   FP_OFF(ptr) = 0x463;             /* check base address of CRT controller */
   if ((*(int *)ptr & 0x03d0) == 0x3d0)
      return cga;
#if 0
   if ((*(int *)ptr & 0x03b0) == 0x3b0)
      return mda;
#endif
   return unknown;
}

/*
Handle the chore of prompting user for input file and verifying that file for
input.  If file won't open, prompt for another name or let user cancel request
by giving no name at all.  If no name, revert to RANDOM default input.

Never quite got the cursor working properly for this routine.  You try.
*/
STATIC signed char
get_input_file(char *input_file)
{
   int fp;
   char buf[160];
   struct rccoord position;

   display_line("Enter filename or pathname of input file;"
     " ENTER sans filename = random input.",
     23, 0, COLOR(BLACK, WHITE));
   cursor_control(revive);
   while(1)                      /* till user names a good file or gives up */
   {
      display_line("                                            ",
        2, 34, COLOR(BLACK, BLACK));
      display_line("name: ", 2, 28, COLOR(BLUE, WHITE));
      /*
      I've spent an evening trying to get the cursor to work
      as advertised.  Can't.  This kludge will at least show
      the user where to start typing.
       */
      position = _gettextposition();
      _outtext(" ");
      _settextposition(position.row, position.col);
      if ('\0' == *gets(input_file))         /* user hit ENTER, no filename */
      {
         cursor_control(revive);
         return RANDOM;                          /* default input selection */
      }
      /*
      Open file to be sure it's available, then close
      it again.  It'll be reopened when the time comes.
       */
      if ((fp = open(input_file, O_RDONLY)) == -1)
      {
         sprintf(buf, "Can't open %s.  Reenter name, or ENTER to cancel. ",
           input_file);
         blank_line(23, BLACK);                      /* blank synopsis line */
         display_line(trim(buf), 23, 0, COLOR(BLACK, BLINK(WHITE)));
      }
      else
      {
         cursor_control(revive);
         close(fp);
         return FILE_INPUT;
      }
   }
}

/*
Get user's keypress.  Only acceptable input here: ENTER, up-arrow, down-arrow,
right-arrow, left-arrow.

The arrow keys return a null byte followed by then an auxiliary byte, hence
the test for 0.
*/
STATIC signed char
keypress()
{
   register int c;

   while ((c = getch()) != 0xd)
   {
      if (c == 0)
         switch(getch())
         {
         case 72:
            return UP_ARROW;
         case 75:
            return LEFT_ARROW;
         case 77:
            return RIGHT_ARROW;
         case 80:
            return DOWN_ARROW;
         }
      putchar(BEL);                        /* not a valid keypress, so beep */
   }
   return COMMIT;
}

/*
Display the menu, then wait for the user to select a line (via the arrow keys
and the ENTER key) and return that line's index in the menu as displayed.

Display synopsis for the current selection on line 24.  Blank it before
leaving.
*/
STATIC signed char
menu(MENUPTR menuptr)
{
   register signed char i;                               /* smallish number */

   for (i = 0; i < menuptr->length; ++i)
      display_menu_line(menuptr, i, 1);
   i = menuptr->current_choice;
   while (1)
   {
      signed char j;

      display_menu_line(menuptr, i, -1); /* display current line rev. video */
      blank_line(23, BLACK);                         /* blank synopsis line */
      display_line(menuptr->synopses[i], 23, 0, COLOR(BLACK, WHITE));
      switch(keypress())         /* get one of the three permissible inputs */
      {
      case COMMIT:                    /* user has settled on current choice */
         blank_line(23, BLACK);
         return i;
      case UP_ARROW:                               /* user moves up in menu */
         j = max(0, i - 1);               /* respect upper boundary of menu */
         break;
      case DOWN_ARROW:                                   /* user moves down */
         j = min(menuptr->length - 1, i + 1);     /* respect lower boundary */
         break;
      case LEFT_ARROW:                                   /* user moves left */
         if (i - 23 >= 0)
            j = i - 23;
         else
            putchar(BEL);                                           /* beep */
         break;
      case RIGHT_ARROW:                                 /* user moves right */
         if (i + 23 < menuptr->length)
            j = i + 23;
         else
            putchar(BEL);                                           /* beep */
         break;
      }
      display_menu_line(menuptr, i, 1);   /* restore current line to normal */
      i = j;                                          /* reset current line */
   }
}

/*
Given an entry in the interrupt vector table (as the offset of an actual
address, page 0 assumed), replace what's there with <new_vector> and return
the old entry.
*/
STATIC PTR_TO_FN_RET_VOID
replace_int_vector(VECTOR vector_addr, PTR_TO_FN_RET_VOID new_vector)
{
   PTR_TO_FN_RET_VOID ptr0, *ptr;

   FP_SEG(ptr) = 0;
   FP_OFF(ptr) = vector_addr;
   ptr0 = *ptr;                                      /* save the old vector */
   *ptr = new_vector;
   return ptr0;                                    /* return the old vector */
}

STATIC void
sort(MENU menus[], int slowdown_factor, char *input_file,
  FILTER current_filter, DISPLAY display, unsigned char current_color)
{
   extern volatile long clock_ticks; /* bumped by system tick, 18.2 per sec */
   char buf[120];

/***************************************************
         Fill Video RAM with Sample Text
*/
   /*
   NB: we clear the screen here not only to look nice but
   because anything shown in the first 24 lines will be sorted.
    */
   _clearscreen(_GCLEARSCREEN);
   switch (menus[INPUT].current_choice)                      /* what input? */
   {
   case KEYBOARD_INPUT:
#if 0
      /*
      Mini-editor goes here.  User types chars that go
      to page 1 of video display.  Not yet implemented,
      and may never be.
       */
#endif
      break;
   case FILE_INPUT:                  /* NB: file MUST be open at this point */
      sprintf(buf, "Reading from %s into video RAM.", input_file);
      display_line(trim(buf), 24, 0, COLOR(BLACK, LIGHT_CYAN));
      fill_ram_from_file(display.base_addr, 24 * 80, input_file,
        current_filter, current_color);
      break;
   case RANDOM:
      fill_ram_at_random(display.base_addr, 24 * 80, current_filter,
        current_color);
      break;
   }

/***************************************************
       Prepare for Sorting
*/
   update_status_line(slowdown_factor, input_file, current_filter,
     menus[INPUT].current_choice,                             /* input mode */
     menus[SORT].choices[menus[SORT].current_choice],
     ": ENTER to start");
   getch();                                    /* wait for user's keystroke */
   blank_line(24, BLACK);                              /* blank status line */
   if (slowdown_factor == 0)                   /* timing only if full speed */
      clock_ticks = 0L;                                     /* start timing */

/***************************************************
       Perform the Sort
*/
   sortlist[menus[SORT].current_choice](display.base_addr, 80 * 24,
     slowdown_factor);

/***************************************************
       Recover from Sorting
*/
   /* show timing only if full speed and not interrupted by user */
   if (slowdown_factor == 0 && flag == 0)
   {
      /*
      The clock ticked <clock_ticks> times during the sort.
      And there are 18.2 ticks per second.  Best we can do is
      give tenths of a second, I suppose.  Figure like this:
      (clock_ticks * 10000) / 182 = thousandths of a second;
      but we don't have that much accuracy, so divide by 100
      to get nearest tenth.
       */
      unsigned int tenths = (unsigned int)(((clock_ticks * 10000) / 182) / 100);
      sprintf(buf, "; %d.%d sec", tenths / 10, tenths % 10);
   }
   else if (flag == 1)
      strcpy(buf, " -- BREAK");
   else
      *buf = '\0';                                 /* reset buffer to empty */
   strcat(buf, "; ENTER to continue");
   update_status_line(slowdown_factor, input_file, current_filter,
     menus[INPUT].current_choice,                             /* input mode */
     menus[SORT].choices[menus[SORT].current_choice],
     buf);
   getch();                                    /* wait for user's keystroke */
}

/*
If buf is too long for one line, scroll it off the screen to the left and
insert an ellipsis to show that something's missing.  (It's important to keep
the right-hand end of the line because that's where the ENTER prompt appears.)
*/
STATIC char *
trim(char *buf)
{
   if (strlen(buf) > 79)
   {
      memmove(buf, buf + strlen(buf) - 79, 80);             /* include '\0' */
      memmove(buf, "...", 3);                            /* to show elision */
   }
   return buf;
}

/*
Display status information on bottom line.
*/
STATIC void
update_status_line(int slowdown_factor, char *input_file,
  FILTER current_filter, signed char current_input_mode,
  char *current_sort, char *trailer)
{
   char buf[160];                              /* long enough for two lines */

   sprintf(buf, "%s; ", current_sort);
   switch (current_input_mode)
   {
   case FILE_INPUT:
      strcat(buf, "input file ");
      strcat(buf, input_file);
      break;
   case KEYBOARD_INPUT:
      strcat(buf, "keyboard input");
      break;
   case RANDOM:
      strcat(buf, "random chars");
      break;
   }
   switch (current_filter)
   {
   case alnum:
      strcat(buf, "; alphanumeric text");
      break;
   case alpha:
      strcat(buf, "; alphabetic text");
      break;
   case lower:
      strcat(buf, "; lowercase text");
      break;
   case digit:
      strcat(buf, "; numeric text");
      break;
   case print:
      strcat(buf, "; all printing chars");
      break;
   case graph:
      strcat(buf, "; all chars but ' '");
      break;
   case punct:
      strcat(buf, "; punctuation marks");
      break;
   case upper:
      strcat(buf, "; uppercase text");
      break;
   }
   if (slowdown_factor != 0)
      sprintf(buf + strlen(buf), "; slowed by %d",  2 * slowdown_factor);
   strcat(buf, trailer);
   display_line(trim(buf), 24, 0, COLOR(BLACK, YELLOW));
}

/* end of main.c */
