
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1994,95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit GViews;

{$F+,X+,I-,S-,R-}

{ An 80286+ CPU is required for raster fonts }

interface
uses
   Objects, Memory, Strings,
   GDrivers, Events, KeyMouse, Utils, Language,
   GraphDrv, GBuffer, UniFont,
   General, GRect, GPalette,
   Controls;

const
   CFrame =
   cpFrame+cpWhite + cpHFrame+cpBlack +
   cpTopIcons+cpBlack + cpTopIconsBk+cpWhite +
   cpBottomIcons+cpBlack + cpBottom+cpWhite +
   cpTitle+cpWhite + cpTitleBk+cpBlue;

   CPassiveFrame =
   cpFrame+cpLightGray + cpHFrame+cpBlack +
   cpTopIcons+cpBlack + cpTopIconsBk+cpLightGray +
   cpBottomIcons+cpBlack + cpBottom+cpLightGray +
   cpTitle+cpBlack + cpTitleBk+cpLightGray;

   CScrollBar =
   cpBar+cpLightGray + cpBarSelected+cpDarkGray + cpIndicator+cpLightGray +
   cpArrows+cpBlack;

   CListViewer =
   cpMain+cpCyan + cpNormal+cpBlack +
   cpSelected+cpWhite + cpSelectedBk+cpGreen + cpSelDisabled+cpYellow;

   CBlueWindow =
   cpMain+cpBlue + cpText+cpYellow;
   CCyanWindow =
   cpMain+cpCyan + cpText+cpBlack;
   CGrayWindow =
   cpMain+cpLightGray + cpText+cpBlack;
   CWhiteWindow =
   cpMain+cpWhite + cpText+cpBlack;

   CWindow =       CBlueWindow;

var
{ Default font }
   SystemFont      :TTextStyle;

const
{ Event masks }
   PositionalEvents:Word = evMouse;
   FocusedEvents   :Word = evKeyboard + evCommand;

{ Minimum window size }
   MinWinSize      :TGPoint= (X: 96; Y: 64);

{ TView.DragView parameters }
   DragColor       :Byte =   15;
   Grid            :TGPoint= (X: 4; Y: 4);

{ Default cursor parameters }
   CursorSize      :TGPoint= (X: 0; Y: 1); { X=0 - use horizontal cursor }
   CursorInsSize   :TGPoint= (X: 0; Y: 3); { Y=0 - use vertical cursor }

{ TScrollBar parameters }
   SBarSize        :TGPoint= (X:9; Y:9);
   SBarPartSize    :TGPoint= (X:9; Y:9);

{ Dialogs parameters }
   FrameOffset     :Integer= 2;
   ButtonShift     :TGPoint= (X: 16; Y: 8);
   MinButtonWidth  :Integer= 0;
   HistoryWidth    :Integer= 9;

const

{ TView State masks }

   sfVisible     = $0001;
   sfCursorVis   = $0002;
   sfCursorIns   = $0004;
   sfCursorOn    = $0008;
   sfActive      = $0010;
   sfSelected    = $0020;
   sfFocused     = $0040;
   sfDragging    = $0080;
   sfDisabled    = $0100;
   sfModal       = $0200;
   sfDefault     = $0400;
   sfExposed     = $0800;

{ TView Option masks }

   ofSelectable  = $0001;
   ofTopSelect   = $0002;
   ofFirstClick  = $0004;
   ofFramed      = $0008;
   ofPreProcess  = $0010;
   ofPostProcess = $0020;
   ofBuffered    = $0040;
   ofTileable    = $0080;
   ofCenterX     = $0100;
   ofCenterY     = $0200;
   ofCentered    = $0300;
   ofValidate    = $0400;
   ofVersion     = $3000;
   ofVersion10   = $0000;
   ofVersion20   = $1000;
   ofClipping    = $4000; { Draw view with clipping inside its owner's list }
   ofUpdate      = $8000; { Update TScrollBar's indicator on every mouse move }

{ TView GrowMode masks }

   gfGrowLoX =     $01;
   gfGrowLoY =     $02;
   gfGrowHiX =     $04;
   gfGrowHiY =     $08;
   gfGrowAll =     $0F;
   gfGrowRel =     $10;

{ TView DragMode masks }

   dmDragMove =    $01;
   dmDragGrow =    $02;
   dmLimitLoX =    $10;
   dmLimitLoY =    $20;
   dmLimitHiX =    $40;
   dmLimitHiY =    $80;
   dmLimitAll =    $F0;

{ TView Help context codes }

   hcNoContext =   0;
   hcDragging  =   1;

{ TScrollBar part codes }

   sbLeftArrow  =  0;
   sbRightArrow =  1;
   sbPageLeft   =  2;
   sbPageRight  =  3;
   sbUpArrow    =  4;
   sbDownArrow  =  5;
   sbPageUp     =  6;
   sbPageDown   =  7;
   sbIndicator  =  8;

{ TScrollBar options for TWindow.StandardScrollBar }

   sbHorizontal     =   $0000;
   sbVertical       =   $0001;
   sbHandleKeyboard =   $0002;

{ TWindow Flags masks }

   wfMove       =  $01;
   wfGrow       =  $02;
   wfClose      =  $04;
   wfZoom       =  $08;
   wfBottomLine =  $10;
   wfStdHorizSB =  $20;
   wfStdVertSB  =  $40;
   wfMenu       =  $80;

{ TWindow number constants }

   wnNoNumber =    0;

{ TWindow palette entries }

   wpBlueWindow =  0;
   wpCyanWindow =  1;
   wpGrayWindow =  2;
   wpWhiteWindow = 3;

{ Standard command codes }

   cmValid   =     0;
   cmQuit    =     1;
   cmError   =     2;
   cmMenu    =     3;
   cmClose   =     4;
   cmZoom    =     5;
   cmResize  =     6;
   cmNext    =     7;
   cmPrev    =     8;
   cmHelp    =     9;

{ Application command codes }

   cmCut     =     20;
   cmCopy    =     21;
   cmPaste   =     22;
   cmUndo    =     23;
   cmClear   =     24;
   cmTile    =     25;
   cmCascade =     26;

{ TDialog standard commands }

   cmOK      =     10;
   cmCancel  =     11;
   cmYes     =     12;
   cmNo      =     13;
   cmDefault =     14;

{ Standard messages }

   cmReceivedFocus    = 50;
   cmReleasedFocus    = 51;
   cmCommandSetChanged= 52;

{ TScrollBar messages }

   cmScrollBarChanged = 53;
   cmScrollBarClicked = 54;

{ TWindow select messages }

   cmSelectWindowNum  = 55;

{ TWindow popup-menu message }

   cmWindowMenu       = 57;

{ TListViewer messages }

   cmListItemSelected = 56;

{ Justify constants and flags for TView.DrawText }

   jsLeft =        $0000;
   jsCenterX =     $0001;
   jsRight =       $0002;
   jsTop =         $0000;
   jsCenterY =     $0100;
   jsBottom =      $0200;
   jsCenter =      jsCenterX+jsCenterY;

   tfNormal =      $0000;
   tfColored =     $8000;

{ Flags for TView.DrawButton & TPanel }

   dbNormal =      $0000;
   dbFillIn =      $0001;
   dbSmall =       $0002;

{ Flags for TPanel }

   dbInset =       $0010;

type

   PCommandSet =   ^TCommandSet;
   TCommandSet =   Set of Byte;

   TDrawClipProc = Procedure (var Bounds         :TGRect);

   PView =         ^TView;
   PGroup =        ^TGroup;

   TView =
   object(TObject)
      Owner        :PGroup;
      Next         :PView;
      Origin       :TGPoint;
      Size         :TGPoint;
      Cursor       :TGPoint;
      GrowMode     :Byte;
      DragMode     :Byte;
      HelpCtx      :Word;
      State        :Word;
      Options      :Word;
      EventMask    :Word;

      IsGroup      :Boolean;

      Font         :PTextStyle;
      MousePointer :Pointer;

      procedure InitClipping;
      function  ClipNotNeeded(var R    :TGRect)            :Boolean;

      procedure HideMouseLocal(R       :TGRect);

      procedure DrawPixel(Point        :TGPoint;
                          Color        :Integer);

      procedure DrawClip(Bounds        :TGRect;
                         DrawProc      :TDrawClipProc);

      procedure DrawBar(var Bounds     :TGRect;
                        Color          :Integer);

      procedure DrawHLine(Start        :TGPoint;
                          Length       :Integer;
                          Color        :Integer);

      procedure DrawVLine(Start        :TGPoint;
                          Length       :Integer;
                          Color        :Integer);

      procedure DrawRect(var Bounds    :TGRect;
                         Color         :Integer);

      procedure DrawImage(Pos, K       :TGPoint;
                          Image        :Pointer;
                          BackColor    :Integer);

      procedure DrawLine(var Bounds    :TGRect;
                         Color         :Integer);

      procedure DrawText(Pos           :TGPoint;
                         Const Text    :String;
                         Flags         :Word;
                         TFont         :PTextStyle;
                         Color         :Word);

      procedure DrawEllipse(Center     :TGPoint;
                            Angle1,
                            Angle2,
                            HorizR,
                            VertR,
                            Color      :Integer);

      procedure DrawArc    (Center     :TGPoint;
                            Angle1,
                            Angle2,
                            Radius,
                            Color      :Integer);

      procedure DrawCircle (Center     :TGPoint;
                            Radius,
                            Color      :Integer);

      procedure DrawERadius(Center     :TGPoint;
                            Angle,
                            HorizR,
                            VertR,
                            Color      :Integer);

      procedure DrawCRadius(Center     :TGPoint;
                            Angle,
                            Radius,
                            Color      :Integer);

      procedure DrawDragFrame(Bounds   :TGRect);

      procedure DrawButton(Bounds      :TGRect;
                           Flags       :Word;
                           Down        :Boolean;
                           CBody       :Word);

      procedure DrawVArrow(var Bounds  :TGRect;
                           Up          :Boolean;
                           Color       :Integer);

      procedure DrawHArrow(var Bounds  :TGRect;
                           Left        :Boolean;
                           Color       :Integer);

      procedure DrawBullet(var Bounds  :TGRect;
                           Color       :Integer);

      procedure InitFont; virtual;
      function  GetFont                                    :PTextStyle;
      function  CharWidth                                  :Integer;
      function  CharHeight                                 :Integer;

      function  GetPalette                                 :TPalette; virtual;
      function  GetColor(Color         :TColor)            :Integer;
      function  GetColors(C1, C2       :TColor)            :Word;

      function  GetMousePointer                            :Pointer;

      procedure LockDraw; virtual;
      procedure UnlockDraw; virtual;

      procedure ResetCursor; virtual;
      procedure GetCursorRect(var R    :TGRect;
                              Ins, Max :Boolean); virtual;

      constructor Init(var Bounds      :TGRect);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      procedure Awaken; virtual;
      procedure BlockCursor;
      procedure CalcBounds(var Bounds  :TGRect;
                           Delta       :TGPoint); virtual;
      procedure ChangeBounds(var Bounds:TGRect); virtual;
      procedure ClearEvent(var Event   :TEvent);
      function  CommandEnabled(Command :Word)              :Boolean;
      function  DataSize                                   :Word; virtual;
      procedure DisableCommands(Commands:TCommandSet);
      procedure DragView(Event         :TEvent;
                         Mode          :Byte;
                         var Limits    :TGRect;
                         MinSize,
                         MaxSize       :TGPoint);
      procedure Draw; virtual;
      procedure DrawView;
      procedure EnableCommands(Commands:TCommandSet);
      procedure EndModal(Command       :Word); virtual;
      function  EventAvail             :Boolean;
      function  Execute                :Word; virtual;
      function  Exposed                :Boolean;
      function  Focus                  :Boolean;
      procedure GetBounds(var Bounds   :TGRect);
      procedure GetClipRect(var Clip   :TGRect);
      procedure GetCommands(var Commands:TCommandSet);
      procedure GetData(var Rec); virtual;
      procedure GetEvent(var Event     :TEvent); virtual;
      procedure GetExtent(var Extent   :TGRect);
      function  GetHelpCtx             :Word; virtual;
      procedure GetPeerViewPtr(var S   :TStream;
                               var P);
      function  GetState(AState        :Word)              :Boolean;
      procedure GrowTo(X, Y            :Integer);
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Hide;
      procedure HideCursor;
      procedure KeyEvent(var Event     :TEvent);
      procedure Locate(var Bounds      :TGRect);
      procedure MakeFirst;
      procedure MakeGlobal(Source      :TGPoint;
                           var Dest    :TGPoint);
      procedure MakeLocal(Source       :TGPoint;
                          var Dest     :TGPoint);
      function  MouseEvent(var Event   :TEvent;
                           Mask        :Word)              :Boolean;
      function  MouseInView(Mouse      :TGPoint)           :Boolean;
      procedure MoveTo(X, Y            :Integer);
      function  NextView                                   :PView;
      procedure NormalCursor;
      function  Prev                                       :PView;
      function  PrevView                                   :PView;
      procedure PutEvent(var Event     :TEvent); virtual;
      procedure PutInFrontOf(Target    :PView);
      procedure PutPeerViewPtr(var S   :TStream;
                               P       :PView);
      procedure Select;
      procedure SetBounds(var Bounds   :TGRect);
      procedure SetCommands(Commands   :TCommandSet);
      procedure SetCmdState(Commands   :TCommandSet;
                            Enable     :Boolean);
      procedure SetCursor(X, Y         :Integer);
      procedure SetData(var Rec); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure Show;
      procedure ShowCursor;
      procedure SizeLimits(var Min, Max:TGPoint); virtual;
      procedure Store(var S            :TStream);
      function  TopView                                    :PView;
      function  Valid(Command          :Word)              :Boolean; virtual;

   private
      LockDrawCount          :Integer;
      SkipClipping           :Boolean;

      function  CheckRect(Bounds       :TGRect)            :Boolean;

      procedure DrawCursor;
      procedure DrawUnderCursor;

      procedure DrawHide(LastView      :PView);
      procedure DrawShow(LastView      :PView);

      procedure DrawUnderRect(var R    :TGRect;
                              LastView :PView);
      procedure DrawUnderView(LastView :PView);
   end;

{ TFrame types }

   TTitleStr =     String[80];
   TFramePart =    (fpLeft, fpRight, fpTop, fpBottom);

   PFrame =        ^TFrame;
   TFrame =
   object(TView)
      constructor Init(var Bounds      :TGRect);

      function  GetPalette                                 :TPalette; virtual;

      procedure DrawZoomIcon(Down      :Boolean); virtual;

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;

      function  GetPartSize(Part       :TFramePart)        :Integer; virtual;
   end;

   PScrollBar =    ^TScrollBar;
   TScrollBar =
   object(TView)
      Value,
      Min, Max,
      PgStep,
      ArStep       :Integer;

      Vert         :Boolean;

      constructor Init(var Bounds      :TGRect;
                       AVert           :Boolean);
      constructor Load(var S           :TStream);

      function  GetPalette                                 :TPalette; virtual;

      procedure GetPartRect(Part       :Integer;
                            var R      :TGRect); virtual;
      function  GetPartCode(Point      :TGPoint)           :Integer; virtual;
      function  GetPartSize                                :Integer; virtual;

      procedure DrawPart(Part          :Integer;
                         Selected      :Boolean); virtual;
      procedure DrawAll(Selected       :Integer);

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure ScrollDraw; virtual;
      function  ScrollStep(Part        :Integer)           :Integer; virtual;
      procedure SetParams(AValue,
                          AMin, AMax,
                          APgStep,
                          AArStep      :Integer);
      procedure SetRange(AMin, AMax    :Integer);
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure SetStep(APgStep,
                        AArStep        :Integer);
      procedure SetValue(AValue        :Integer);
      procedure Store(var S            :TStream);

   private
      PrevSize     :TGPoint;

      function GetPos                                      :Integer;
      function GetSize                                     :Integer;
   end;

   PScroller = ^TScroller;
   TScroller = object(TView)
      HScrollBar,
      VScrollBar   :PScrollBar;
      Delta, Limit :TGPoint;

      constructor Init(var Bounds      :TGRect;
                       AHScrollBar,
                       AVScrollBar     :PScrollBar);
      constructor Load(var S           :TStream);

      procedure ChangeBounds(var Bounds:TGRect); virtual;
      procedure DrawView;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure ScrollDraw; virtual;
      procedure ScrollTo(X, Y          :Integer);
      procedure SetLimit(X, Y          :Integer);
      procedure SetState(AState        :Word;
                Enable                 :Boolean); virtual;
      procedure Store(var S            :TStream);

   private
      RedrawNeeded :Boolean;
   end;

   PListViewer = ^TListViewer;
   TListViewer = object(TView)
      HScrollBar,
      VScrollBar   :PScrollBar;
      NumCols,
      NumRows      :Integer;
      TopItem      :Integer;
      Focused      :Integer;
      Range        :Integer;

      constructor Init(var Bounds      :TGRect;
                       ANumCols        :Word;
                       AHScrollBar,
                       AVScrollBar     :PScrollBar);
      constructor Load(var S           :TStream);

      function  GetPalette                                 :TPalette; virtual;

      procedure ChangeBounds(var Bounds:TGRect); virtual;
      procedure Draw; virtual;

      procedure DrawItem(var Bounds    :TGRect;
                         Item, Indent  :Integer); virtual;
      procedure GetItemRect(Item       :Integer;
                            var R      :TGRect); virtual;

      procedure FocusItem(Item         :Integer); virtual;
      function  GetText(Item, MaxLen   :Integer)           :String; virtual;
      function  IsSelected(Item        :Integer)           :Boolean; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure SelectItem(Item        :Integer); virtual;
      procedure SetRange(ARange        :Integer);
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure Store(var S            :TStream);

   private
      procedure FocusItemNum(Item      :Integer); virtual;
   end;

{ Selection modes (used internally) }
   SelectMode =    (NormalSelect, EnterSelect, LeaveSelect);

   TGroup = object(TView)
      Last,
      Current      :PView;
      Phase        :(phFocused, phPreProcess, phPostProcess);
      EndState     :Word;

      Clip         :TGRect;

      procedure LockDraw; virtual;
      procedure UnlockDraw; virtual;

      constructor Init(var Bounds      :TGRect);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      procedure Awaken; virtual;
      procedure ChangeBounds(var Bounds:TGRect); virtual;
      function  DataSize                                   :Word; virtual;
      procedure Delete(P               :PView);
      procedure Draw; virtual;
      procedure EndModal(Command       :Word); virtual;
      procedure EventError(var Event   :TEvent); virtual;
      function  ExecView(P             :PView)             :Word;
      function  Execute                                    :Word; virtual;
      function  First                                      :PView;
      function  FirstThat(P            :Pointer)           :PView;
      function  FocusNext(Forwards     :Boolean)           :Boolean;
      procedure ForEach(P              :Pointer);
      procedure GetData(var Rec); virtual;
      function  GetHelpCtx             :Word; virtual;
      procedure GetSubViewPtr(var S    :TStream;
                              var P);
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Insert(P               :PView);
      procedure InsertBefore(P, Target :PView);
      procedure Lock;
      procedure Unlock;
      procedure PutSubViewPtr(var S    :TStream;
                              P        :PView);
      procedure Redraw;
      procedure SelectNext(Forwards    :Boolean);
      procedure SetData(var Rec); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure Store(var S            :TStream);
      function  Valid(Command          :Word)              :Boolean; virtual;

   private
      function  At(Index               :Integer)           :PView;
      procedure DrawSubViews(P, Bottom :PView);
      function  FirstMatch(AState      :Word;
                           AOptions    :Word)              :PView;
      function  FindNext(Forwards      :Boolean)           :PView;
      function  IndexOf(P              :PView)             :Integer;
      procedure InsertView(P, Target   :PView);
      procedure RemoveView(P           :PView);
      procedure ResetCurrent;
      procedure ResetCursor; virtual;
      procedure SetCurrent(P           :PView;
                           Mode        :SelectMode);
   end;

   PPanel =        ^TPanel;
   TPanel =
   object(TGroup)
      Flags, Color :Word;

      constructor Init(var Bounds      :TGRect;
                       AFlags, AColor  :Word);

      procedure Draw; virtual;
   end;

   PWindow =       ^TWindow;
   TWindow =
   object(TGroup)
      Flags        :Byte;
      ZoomRect     :TGRect;
      Number       :Integer;
      Palette      :Integer;
      Frame        :PFrame;
      Title        :PString;

      constructor Init(Bounds          :TGRect;
                       ATitle          :TTitleStr;
                       ANumber         :Integer);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      procedure GetInterior(var I      :TGRect);
      function  GetMenu(NextItem       :Pointer)           :Pointer; virtual;
                                      { PMenuItem }       { PMenuItem }

      procedure Close; virtual;
      function  GetTitle(MaxSize       :Integer)           :TTitleStr; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure InitFrame; virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure SizeLimits(var Min, Max:TGPoint); virtual;
      function  StandardScrollBar(AOptions:Word)           :PScrollBar;
      procedure Store(var S            :TStream);
      procedure Zoom; virtual;

   private
      procedure GetInteriorSB(var I    :TGRect);
   end;

{$L drawline.obj}
procedure DrawAnyLine(x1, y1, x2, y2, c:Integer;
                      PutPixel         :Pointer;
                      Break            :Boolean;
                      Pattern          :Word   );

{ Message dispatch function }

   function  Message(Receiver          :PView;
                    What, Command      :Word;
                    InfoPtr            :Pointer) :Pointer;

{ Views registration procedure }

   procedure RegisterViews;

{ Stream Registration Records }

const
  RView: TStreamRec = (
     ObjType: 1;
     VmtLink: Ofs(TypeOf(TView)^);
     Load:    @TView.Load;
     Store:   @TView.Store
  );

  RFrame: TStreamRec = (
     ObjType: 2;
     VmtLink: Ofs(TypeOf(TFrame)^);
     Load:    @TFrame.Load;
     Store:   @TFrame.Store
  );

  RScrollBar: TStreamRec = (
     ObjType: 3;
     VmtLink: Ofs(TypeOf(TScrollBar)^);
     Load:    @TScrollBar.Load;
     Store:   @TScrollBar.Store
  );

  RScroller: TStreamRec = (
     ObjType: 4;
     VmtLink: Ofs(TypeOf(TScroller)^);
     Load:    @TScroller.Load;
     Store:   @TScroller.Store
  );

  RListViewer: TStreamRec = (
     ObjType: 5;
     VmtLink: Ofs(TypeOf(TListViewer)^);
     Load:    @TListViewer.Load;
     Store:   @TListViewer.Store
  );

  RGroup: TStreamRec = (
     ObjType: 6;
     VmtLink: Ofs(TypeOf(TGroup)^);
     Load:    @TGroup.Load;
     Store:   @TGroup.Store
  );

  RWindow: TStreamRec = (
     ObjType: 7;
     VmtLink: Ofs(TypeOf(TWindow)^);
     Load:    @TWindow.Load;
     Store:   @TWindow.Store
  );

const
   CommandSetChanged         :Boolean= False;

implementation
uses
   GMenus, GApp;

procedure DrawAnyLine; External;

type
   PFixupList = ^TFixupList;
   TFixupList = array[1..4096] of Pointer;

const
   OwnerGroup      :PGroup =           nil;
   FixupList       :PFixupList =       nil;
   TheTopView      :PView =            nil;

const

{ Bit flags to determine how to draw the frame icons }

   fmCloseClicked = $0001;
   fmZoomClicked  = $0002;

{ Current command set. All but window commands are active by default }

   CurCommandSet   :TCommandSet =
   [0..255] - [cmZoom, cmClose, cmResize, cmNext, cmPrev];

var
   SelfPtr, Common                     :Pointer;
   DrawColor                           :Integer;

function  NextVisible(P                :PView)   :PView; far;
assembler;
asm
   les  di, P
@@1:
   test es:[di].TView.State, sfVisible
   jnz  @@2
   les  di, es:[di].TView.Next
   jmp  @@1
@@2:
   mov  ax, di
   mov  dx, es
end;

function  Visible(P                    :PView;
                  Dummy                :Word)    :Boolean; far;
assembler;
asm
   les  di, P
   xor  ax, ax
   mov  cx, es:[di].TView.State
   and  cx, sfVisible
   jcxz @@1
   inc  ax
@@1:
end;

procedure TView.LockDraw;
begin
   Inc(LockDrawCount);
end;

procedure TView.UnlockDraw;
begin
   Dec(LockDrawCount);
end;

function  TView.CheckRect;
var
   Clip            :TGRect;
   Target          :PView;
begin
   CheckRect:=True;
   if (Owner <> nil) and GetState(sfVisible) then
   begin
      if Options and ofClipping <> 0 then
      begin
         Target:=Owner^.FirstThat(@Visible);
         while Target <> @Self do
         begin
            Target^.GetBounds(Clip);
            Clip.Move(-Origin.X, -Origin.Y);
            Clip.Intersect(Bounds);
            Target:=NextVisible(Target^.Next);
            if not Clip.Empty then
            begin
               CheckRect:=False; Exit;
            end;
         end;
      end;
      Bounds.Move(Origin.X, Origin.Y); CheckRect:=Owner^.CheckRect(Bounds);
   end;
end;

function  TView.ClipNotNeeded;
var
   Bounds, RN      :TGRect;
begin
   if (@Self = Application) or SkipClipping then
   begin
      GetExtent(Bounds);
      ClipNotNeeded:=Bounds.Contains(R.A) and Bounds.Contains(R.B);
   end else
   begin
      RN:=R; MakeGlobal(RN.A, RN.A); MakeGlobal(RN.B, RN.B);
      Application^.GetExtent(Bounds);
      ClipNotNeeded:=Bounds.Contains(RN.A) and Bounds.Contains(RN.B) and
         CheckRect(R);
   end;
end;

procedure TView.InitClipping;

procedure DoInitClipping(P             :PView); far;
var
   Bounds          :TGRect;
begin
   P^.SkipClipping:=False;
   P^.GetExtent(Bounds);
   P^.SkipClipping:=P^.ClipNotNeeded(Bounds);
end;

begin
   if Owner=nil then DoInitClipping(@Self) else Owner^.ForEach(@DoInitClipping);
end;

procedure TView.HideMouseLocal;
begin
   MakeGlobal(R.A, R.A); MakeGlobal(R.B, R.B);
   HideMouseRect(R);
end;

procedure TView.DrawPixel;
var
   View            :PView;
   Bounds          :TGRect;
begin
   GetClipRect(Bounds); if Bounds.Contains(Point) then
   if @Self = Application then
   begin
      Bounds.A:=Point; Bounds.B:=Point;
      HideMouseRect(Bounds);
      GraphDrv.PutPixel(Point.X, Point.Y, Color);
      ShowMouseRect;
   end
   else if (Owner <> nil) and GetState(sfVisible) then
   begin
      Inc(Point.X, Origin.X); Inc(Point.Y, Origin.Y);
      View := Owner^.FirstThat(@Visible);
      while View <> @Self do
      begin
         View^.GetBounds(Bounds);
         if Bounds.Contains(Point) then Exit;
         View := NextVisible(View^.Next);
      end;
      Owner^.DrawPixel(Point, Color);
   end;
end;

procedure TView.DrawClip;
var
   R               :TGRect;
   I               :Integer;

   procedure DoRect(var Bounds         :TGRect;
                    Target             :PView);
   var
      NoIntersections        :Boolean;
      Clip                   :TGRect;
      Part                   :TGRect;
   begin
      NoIntersections := True;
      while Target <> @Self do
      begin
         Target^.GetBounds(Clip);
         Clip.Move(-Origin.X, -Origin.Y);
         Clip.Intersect(Bounds);
         Part.Clear;
         Target := NextVisible(Target^.Next);
         if not Clip.Empty then
         with Bounds do
         begin
            if B.Y > Clip.B.Y then
            begin
               Part.Assign(A.X, Clip.B.Y+1, B.X, B.Y); DoRect(Part, Target);
            end;
            if A.Y < Clip.A.Y then
            begin
               Part.Assign(A.X, A.Y, B.X, Clip.A.Y-1); DoRect(Part, Target);
            end;
            if B.X > Clip.B.X then
            begin
               Part.Assign(Clip.B.X+1, A.Y, B.X, B.Y); DoRect(Part, Target);
            end;
            if A.X < Clip.A.X then
            begin
               Part.Assign(A.X, A.Y, Clip.A.X-1, B.Y); DoRect(Part, Target);
            end;
            NoIntersections := False; Exit;
         end;
      end;
      if NoIntersections then
      begin
         Bounds.Move(Origin.X, Origin.Y); Owner^.DrawClip(Bounds, DrawProc);
      end;
   end;

begin
   Bounds.Arrange; GetClipRect(R); R.Intersect(Bounds);
   if R.Empty then Exit;
   if ClipNotNeeded(R) then
   begin
      MakeGlobal(R.A, R.A); MakeGlobal(R.B, R.B);
      HideMouseRect(R); DrawProc(R); ShowMouseRect;
   end else
   if (Owner <> nil) and GetState(sfVisible) then
   begin
      if Options and ofClipping <> 0 then DoRect(R, Owner^.FirstThat(@Visible))
      else
      begin
         R.Move(Origin.X, Origin.Y); Owner^.DrawClip(R, DrawProc);
      end;
   end;
end;

procedure DoDrawBar(var R              :TGRect); far;
begin
   GraphDrv.QuickBar(R.A.X, R.A.Y, R.B.X, R.B.Y, DrawColor);
end;

procedure TView.DrawBar;
begin
   DrawColor:=Color; DrawClip(Bounds, DoDrawBar);
end;

procedure DoDrawHLine(var R            :TGRect); far;
begin
   GraphDrv.HorizLine(R.A.Y, R.A.X, R.B.X, DrawColor);
end;

procedure TView.DrawHLine;
var
   R               :TGRect;
begin
   R.A := Start; R.B.X := R.A.X+Length; R.B.Y := R.A.Y;
   DrawColor:=Color; DrawClip(R, DoDrawHLine);
end;

procedure DoDrawVLine(var R            :TGRect); far;
begin
   GraphDrv.Line(R.A.X, R.A.Y, R.A.X, R.B.Y, DrawColor);
end;

procedure TView.DrawVLine;
var
   R               :TGRect;
begin
   R.A := Start; R.B.Y := R.A.Y+Length; R.B.X := R.A.X;
   DrawColor:=Color; DrawClip(R, DoDrawVLine);
end;

procedure TView.DrawRect;
begin
   HideMouseLocal(Bounds);
   with Bounds do
   begin
      DrawHLine(A, B.X-A.X, Color); DrawVLine(A, B.Y-A.Y, Color);
      DrawVLine(B, A.Y-B.Y, Color); DrawHLine(B, A.X-B.X, Color);
   end;
   ShowMouseRect;
end;

procedure DoDrawImage(var R            :TGRect); far;
type
   TImageParams =
   record
      Image        :Pointer;
      K, Pos       :TGPoint;
   end;
var
   RelPos, ModPos  :TGPoint;
   Width,
   CurrentX,
   CurrentY,
   RelX, RelY,
   ModXR,
   RelX1, RelX2,
   PosX1, PosX2    :Integer;
   ImageLine       :Pointer;
   Offset          :Word absolute ImageLine;

procedure DrawImagePoint(x1, x2        :Integer;
                         Color         :Byte);
begin
   if Color=NullColor then
   if DrawColor=-1 then Exit else Color:=DrawColor;
   if x1<=x2 then GraphDrv.HorizLine(CurrentY, x1, x2, Color);
end;

procedure DrawImageBar  (x1, x2        :Integer;
                         Color         :Byte);
begin
   if Color=NullColor then
   if DrawColor=-1 then Exit else Color:=DrawColor;
   if x1<=x2 then GraphDrv.Bar(x1, CurrentY, x2, RelY, Color);
end;

begin
   with TImageParams(Common^) do
   begin
      asm
         les  di,R
         mov  si,Word Ptr Common
         mov  ax,es:[di].TGRect.A.X
         sub  ax,ss:[si].TImageParams.Pos.X
         cwd
         div  ss:[si].TImageParams.K.X
         mov  RelPos.X,ax
         mov  ModPos.X,dx

         mov  ax,es:[di].TGRect.A.Y
         sub  ax,ss:[si].TImageParams.Pos.Y
         cwd
         div  ss:[si].TImageParams.K.Y
         mov  RelPos.Y,ax
         mov  ModPos.Y,dx
      end;

      ImageLine:=Image;

      Inc(Offset, RelPos.Y * Word(Image^) + RelPos.X + 4);

      if ModPos.X<>0 then RelX:=K.X - ModPos.X else RelX:=0;
      RelY:=K.Y - ModPos.Y;

      asm
         les  di,R
         push es
         push di
         call TGRect.SizeX
         sub  ax,RelX
         mov  si,Word Ptr Common
         cwd
         idiv ss:[si].TImageParams.K.X
         mov  Width,ax
         mov  ModXR,dx
      end;

      Inc(RelX, R.A.X); RelX1:=Min(RelX-1, R.B.X);
      RelX2:=R.B.X-ModXR+1;

      CurrentY:=R.A.Y;
      while CurrentY<=R.B.Y do
      begin
         if Supported(DrawImageLineK) or
            (Supported(DrawImageLine) and (LongInt(K)=$00010001)) then
         while RelY>0 do
         begin
            if ModPos.X<>0 then
            begin
               DrawImagePoint(R.A.X, RelX1, Byte(ImageLine^));
               Inc(Offset);
            end;
            if Width>0 then
            if Supported(GraphDrv.DrawImageLine) and (K.X=1)
            then GraphDrv.DrawImageLine(RelX, CurrentY, Width, DrawColor, ImageLine)
            else GraphDrv.DrawImageLineK(RelX, CurrentY, Width, K.X, DrawColor, ImageLine);
            if ModXR>0 then
               DrawImagePoint(RelX2, R.B.X, PByteArray(ImageLine)^[Width]);
            if ModPos.X<>0 then Dec(Offset);
            Inc(CurrentY); if CurrentY>R.B.Y then Exit;
            Dec(RelY);
         end else
         begin
            Inc(RelY, CurrentY - 1);
            if RelY>R.B.Y then RelY:=R.B.Y;

            if ModPos.X<>0 then
            begin
               DrawImageBar(R.A.X, RelX1, Byte(ImageLine^));
               Inc(Offset);
            end;
            if Width>0 then
            begin
               PosX1:=RelX; PosX2:=PosX1+K.X-1;
               for CurrentX:=0 to Width-1 do
               begin
                  DrawImageBar(PosX1, PosX2, PByteArray(ImageLine)^[CurrentX]);
                  Inc(PosX1, K.X); Inc(PosX2, K.X);
               end;
            end;
            if ModXR>0 then
               DrawImageBar(RelX2, R.B.X, PByteArray(ImageLine)^[Width]);
            if ModPos.X<>0 then Dec(Offset);

            CurrentY:=RelY + 1;
         end;
         Inc(Offset, Word(Image^));
         RelY:=K.Y;
      end;
   end;
end;

procedure TView.DrawImage;
var
   Bounds          :TGRect;
   RealSize        :TGPoint;
begin
   RealSize.X:=TGPoint(Image^).X * K.X;
   RealSize.Y:=TGPoint(Image^).Y * K.Y;
   Bounds.A:=Pos; Bounds.SetSize(RealSize);
   Common:=@Image; DrawColor:=BackColor; MakeGlobal(Pos, Pos);
   DrawClip(Bounds, DoDrawImage);
end;

procedure DoGlobalPixel(X, Y, Color    :Integer); far;
assembler;
asm
   mov  cx,x
   mov  dx,y
   les  di,Common
   cmp  cx,es:[di].TGRect.A.X
   jl   @@1
   cmp  dx,es:[di].TGRect.A.Y
   jl   @@1
   cmp  cx,es:[di].TGRect.B.X
   jg   @@1
   cmp  dx,es:[di].TGRect.B.Y
   jg   @@1
   push cx
   push dx
   push Color
   call GraphDrv.PutPixel
@@1:
end;

procedure DoNormalLine(x1, y1, x2, y2, Color     :Integer); far;
assembler;
asm
   push x1
   push y1
   push x2
   push y2
   push Color
   les  di,GraphDrv.PutPixel
   push es
   push di
   xor  ax,ax
   push ax
   dec  ax
   push ax
   call DrawAnyLine
end;

procedure DoPixel(X, Y, Color          :Integer); far;
assembler;
asm
   push y
   push x
   push Color
   push word ptr SelfPtr+2
   push word ptr SelfPtr
   call TView.DrawPixel
end;

procedure TView.DrawLine;
var
   R, Clip,
   ABounds         :TGRect;
begin
   GetClipRect(Clip);
   ABounds:=Bounds; ABounds.Arrange; if Clip.Intersects(ABounds) then
   with Bounds do
   if A.X = B.X then DrawVLine(A, B.Y-A.Y, Color) else
   if A.Y = B.Y then DrawHLine(A, B.X-A.X, Color) else
   if ClipNotNeeded(Bounds) then
   begin
      MakeGlobal(A, R.A); MakeGlobal(B, R.B);
      if Clip.Contains(A) and Clip.Contains(B) then
      begin
         HideMouseRect(R);
         DoNormalLine(R.A.X, R.A.Y, R.B.X, R.B.Y, Color);
         ShowMouseRect;
      end
      else
      begin
         Common:=@Clip;
         MakeGlobal(Clip.A, Clip.A); MakeGlobal(Clip.B, Clip.B);
         HideMouseRect(R);
         DrawAnyLine(R.A.X, R.A.Y, R.B.X, R.B.Y, Color,
            @DoGlobalPixel, False, $FFFF);
         ShowMouseRect;
      end;
   end
   else
   begin
      SelfPtr:=@Self;
      HideMouseLocal(Bounds);
      DrawAnyLine(A.X, A.Y, B.X, B.Y, Color, @DoPixel, False, $FFFF);
      ShowMouseRect;
   end;
end;

procedure DoGlobalLine(x1, y1, x2, y2, Color     :Integer); far;
assembler;
asm
   push x1
   push y1
   push x2
   push y2
   push Color
   push cs
   mov  ax,offset DoGlobalPixel
   push ax
   xor  ax,ax
   push ax
   dec  ax
   push ax
   call DrawAnyLine
end;

procedure DoTextLine(x1, y1, x2, y2, Color       :Integer); far;
assembler;
asm
   mov  ax,x1
   xchg ax,y2
   mov  x1,ax
   mov  ax,y1
   xchg ax,x2
   mov  y1,ax

   push ss
   mov  ax,bp
   add  ax,offset y2
   push ax

   push Color

   push word ptr SelfPtr+2
   push word ptr SelfPtr

   call TView.DrawLine
end;

procedure DoTextRect(x1, y1, x2, y2, Color       :Integer); far;
assembler;
asm
   mov  ax,x1
   xchg ax,y2
   mov  x1,ax
   mov  ax,y1
   xchg ax,x2
   mov  y1,ax

   push ss
   mov  ax,bp
   add  ax,offset y2
   push ax

   push Color

   push word ptr SelfPtr+2
   push word ptr SelfPtr

   call TView.DrawRect
end;

procedure DoTextBar (x1, y1, x2, y2, Color       :Integer); far;
assembler;
asm
   mov  ax,x1
   xchg ax,y2
   mov  x1,ax
   mov  ax,y1
   xchg ax,x2
   mov  y1,ax

   push ss
   mov  ax,bp
   add  ax,offset y2
   push ax

   push Color

   push word ptr SelfPtr+2
   push word ptr SelfPtr

   call TView.DrawBar
end;

procedure DoTextPixel(X, Y, Color      :Integer); far;
assembler;
asm
   push ds
   mov  ax,seg @DATA
   mov  ds,ax

   push x
   push y
   push Color
   call DoPixel

   pop  ds
end;

procedure TView.DrawText;
var
   sx, sy, i, Step, c                  :Integer;
   R, Bounds                           :TGRect;
   ALine, ARectangle, ABar,
   AnInit, AShow, AnAdvance            :Pointer;
begin
   if Text='' then Exit;

   if TFont=nil then TFont:=GetFont;
   if TFont^.Font=nil then TFont^.Font:=UniFont.Fonts;

   if Flags and tfColored<>0 then sx:=CStrLen(Text) else sx:=Length(Text);
   sx:=sx*TFont^.Width;
   sy:=TFont^.Height;
   if Flags and jsRight<>0 then Dec(Pos.X, sx) else
   if Flags and jsCenterX<>0 then Dec(Pos.X, sx shr 1);
   if Flags and jsBottom<>0 then Dec(Pos.Y, sy) else
   if Flags and jsCenterY<>0 then Dec(Pos.Y, sy shr 1);

   Bounds.A:=Pos; Bounds.B.X:=Bounds.A.X+sx-1; Bounds.B.Y:=Bounds.A.Y+sy-1;
   GetClipRect(R); if not R.Intersects(Bounds) then Exit;

   if ClipNotNeeded(Bounds) then
   begin
      if R.Contains(Bounds.A) and R.Contains(Bounds.B)
      then ALine:=@DoNormalLine else
      begin
         Common:=@R;
         MakeGlobal(R.A, R.A); MakeGlobal(R.B, R.B);
         ALine:=@DoGlobalLine;
      end;
      ARectangle:=@GraphDrv.Rectangle; ABar:=@GraphDrv.Bar;

      AnInit:=@GraphDrv.Text_Init;
      AShow:=@GraphDrv.Text_Show8xX;
      AnAdvance:=@GraphDrv.Text_Advance;
      Pointer(@UText_PutPixel^):=@GraphDrv.PutPixel;

      MakeGlobal(Pos, Pos);
   end
   else
   begin
      ALine:=@DoTextLine; ARectangle:=@DoTextRect; ABar:=@DoTextBar;

      AnInit:=@GraphDrv.UText_Init;
      AShow:=@GraphDrv.UText_Show8xX;
      AnAdvance:=@GraphDrv.UText_Advance;
      Pointer(@UText_PutPixel^):=@DoTextPixel;
   end;

   Step:=TFont^.Width;

   if DriverPtr^.PutMode=XorPut then DriverPtr^.PutMode:=NormalPut;
   c:=WordRec(Color).Lo;
   SelfPtr:=@Self;
   HideMouseLocal(Bounds);

   if TFont^.Font^.Class = fcVector then
   for i:=1 to Length(Text) do
   begin
      if (Text[i]='~') and (Flags and tfColored<>0) then
      if c=WordRec(Color).Lo
      then c:=WordRec(Color).Hi else c:=WordRec(Color).Lo else
      begin
         UniFont.DrawChar(TFont^.Font, Text[i], Pos.X, Pos.Y, c,
            TFont^.SizeX, TFont^.DivX, TFont^.SizeY, TFont^.DivY,
            ALine, ARectangle, ABar);
         Inc(Pos.X, Step);
      end;
   end else
   asm
      les  di,TFont
      les  di,es:[di].TTextStyle.Font

      push es
      push di

      push Pos.X
      push Pos.Y
      push word ptr es:[di].TFontHeader.Height

      call AnInit

      pop  di
      pop  es

      push ds
      lds  si,Text

      mov  dx,Color
      cld

      xor  ax,ax
      lodsb
      mov  cx,ax

@@CharLoop:
      lodsb

      cmp  al,'~'
      jne  @@Show

      test Flags,tfColored
      jz   @@Show

      cmp  dl,byte ptr Color
      je   @@2ndColor
      mov  dl,byte ptr Color
      jmp  @@NextChar
@@2ndColor:
      mov  dl,dh
      jmp  @@NextChar

@@Show:
      cmp  al,es:[di].TFontHeader.c2
      ja   @@NextChar
      sub  al,es:[di].TFontHeader.c1
      jb   @@NextChar

      db   60h {pusha}
      push es
      push ds
      lds  si,es:[di].TFontHeader.CharOfs
      push word ptr es:[di].TFontHeader.Width
      call AShow
      pop  ax
      call AnAdvance
      pop  ds
      pop  es
      db   61h {popa}

@@NextChar:
      loop @@CharLoop

      pop  ds
   end;

   ShowMouseRect;
end;

const
   MaxSin =        MaxInt - MaxInt div 90 div 2;

procedure TView.DrawEllipse;
var
   i, n, Angle     :Integer;
   R               :TGRect;
begin
   n:=LongDiv(LongMul(Min(HorizR, VertR), Angle2-Angle1), 360);
   n:=Min(Min((n+1)*6, n+(Angle2-Angle1) div 5), Angle2-Angle1);
   for i:=0 to n do
   begin
      Angle:=Angle1+LongDiv(LongMul(i, Angle2-Angle1), n);
      R.B.X:=Center.X+LongDiv(LongMul(QSin(Angle), HorizR), MaxSin);
      R.B.Y:=Center.Y+LongDiv(LongMul(QCos(Angle), VertR ), MaxSin);
      if Angle<>Angle1 then DrawLine(R, Color);
      R.A:=R.B;
   end;
end;

procedure TView.DrawArc;
begin
   DrawEllipse(Center, Angle1, Angle2,
      Radius*DriverPtr^.AspX div DriverPtr^.AspY, Radius, Color);
end;

procedure TView.DrawCircle;
begin
   DrawArc(Center, 0, 360, Radius, Color);
end;

procedure TView.DrawERadius;
var
   R               :TGRect;
begin
   R.A:=Center;
   R.B.X:=Center.X+LongInt(QSin(Angle))*HorizR div MaxSin;
   R.B.Y:=Center.Y+LongInt(QCos(Angle))*VertR  div MaxSin;
   DrawLine(R, Color);
end;

procedure TView.DrawCRadius;
begin
   DrawERadius(Center, Angle,
      Radius*DriverPtr^.AspX div DriverPtr^.AspY, Radius, Color);
end;

procedure TView.DrawDragFrame;
begin
   if Bounds.Empty then Exit;
   GraphDrv.DriverPtr^.PutMode:=GraphDrv.XorPut;
   Desktop^.MakeLocal(Bounds.A, Bounds.A);
   Desktop^.MakeLocal(Bounds.B, Bounds.B);
   Desktop^.DrawRect(Bounds, DragColor);
   GraphDrv.DriverPtr^.PutMode:=GraphDrv.NormalPut;
end;

procedure TView.DrawButton;
var
   CFrame, CHighlight, CShadow, C      :Integer;
begin
   CFrame:=GetColor(cpButFrame);
   CHighlight:=GetColor(cpButHigh); CShadow:=GetColor(cpButShadow);
   if Down then SwapInts(CHighlight, CShadow);
   HideMouseLocal(Bounds);
   with Bounds do
   begin
      if Flags and dbSmall=0 then
      begin
         DrawRect(Bounds, CFrame); Grow(-1, -1);
      end;
      DrawVLine(B, A.Y-B.Y, CShadow);
      DrawHLine(B, A.X-B.X, CShadow);
      DrawVLine(A, B.Y-A.Y, CHighlight);
      DrawHLine(A, B.X-A.X, CHighlight);
      Grow(-1, -1);
      if Flags and dbSmall=0 then
      begin
         if not Down then DrawVLine(B, A.Y-B.Y, CShadow);
         DrawVLine(A, B.Y-A.Y, CHighlight);
         DrawHLine(A, B.X-A.X, CHighlight);
         if Down then
         begin
            Move(1, 1);
            if Flags and dbFillIn<>0 then C := CBody else C := CHighlight;
            DrawHLine(A, B.X-A.X-1, C); DrawVLine(A, B.Y-A.Y-1, C);
         end
         else DrawHLine(B, A.X-B.X+1, CShadow);
         Grow(-1, -1);
      end;
   end;
   if Flags and dbFillIn<>0 then DrawBar(Bounds, CBody);
   ShowMouseRect;
end;

procedure TView.DrawVArrow;
var
   R               :TGRect;
   D, G            :Integer;
begin
   if Up then D := 1 else D := -1;
   HideMouseLocal(Bounds);
   R.A.X := Bounds.A.X; R.B.X := Bounds.B.X;
   if Odd(R.SizeX) then Dec(R.B.X);
   if Up then
   begin
      R.A.Y := Bounds.A.Y; R.B.Y := R.A.Y;
   end
   else
   begin
      R.B.Y := Bounds.B.Y; R.A.Y := R.B.Y;
   end;
   G:=Max(1, ScreenWidth * (4*3) div (5*2) div ScreenHeight);
   R.A.X:=R.CenterX; R.B.X:=R.A.X;
   while Bounds.Contains(R.A) do
   begin
      DrawBar(R, Color);
      R.Move(0, D); R.Grow(G, 0);
   end;
   R.Move(0, -D); R.Grow(-((R.B.X-R.A.X) shr 2)-1, 0);
   if R.SizeX=0 then Inc(R.B.X);
   if Up then R.B.Y:=Bounds.B.Y else R.A.Y:=Bounds.A.Y;
   DrawBar(R, Color);
   ShowMouseRect;
end;

procedure TView.DrawHArrow;
var
   R               :TGRect;
   D, G            :Integer;
begin
   if Left then D := 1 else D := -1;
   HideMouseLocal(Bounds);
   R.A.Y := Bounds.A.Y; R.B.Y := Bounds.B.Y;
   if Odd(R.SizeY) then Dec(R.B.Y);
   if Left then
   begin
      R.A.X := Bounds.A.X; R.B.X := R.A.X;
   end
   else
   begin
      R.B.X := Bounds.B.X; R.A.X := R.B.X;
   end;
   G:=Max(1, ScreenHeight * (5*3) div (4*2) div ScreenWidth);
   R.A.Y:=R.CenterY; R.B.Y:=R.A.Y;
   while Bounds.Contains(R.A) do
   begin
      DrawBar(R, Color);
      R.Move(D, 0); R.Grow(0, G);
   end;
   R.Move(-D, 0); R.Grow(0, -((R.B.Y-R.A.Y) shr 2)-1);
   if R.SizeY=0 then Inc(R.B.Y);
   if Left then R.B.X:=Bounds.B.X else R.A.X:=Bounds.A.X;
   DrawBar(R, Color);
   ShowMouseRect;
end;

procedure TView.DrawBullet;
var
   R, R2           :TGRect;
   G               :Integer;
   Vert            :Boolean;
begin
   HideMouseLocal(Bounds);
   R := Bounds; Vert:=(R.SizeY > R.SizeX);

   if Vert then
   begin
      G:=-Max(1, ScreenHeight * (5*3) div (4*2) div ScreenWidth);
      R.A.Y:=R.CenterY; R.B.Y:=R.A.Y;
      R2:=R; R2.Move(0, -1);
      while R.SizeX>0 do
      begin
         DrawBar(R, Color); DrawBar(R2, Color);
         R.Grow(G, 0); R.Move(0, 1);
         R2.Grow(G, 0); R2.Move(0, -1);
      end;
   end else
   begin
      G:=-Max(1, ScreenWidth * (4*3) div (5*2) div ScreenHeight);
      R.A.X:=R.CenterX; R.B.X:=R.A.X;
      R2:=R; R2.Move(-1, 0);
      while R.SizeY>0 do
      begin
         DrawBar(R, Color); DrawBar(R2, Color);
         R.Grow(0, G); R.Move(1, 0);
         R2.Grow(0, G); R2.Move(-1, 0);
      end;
   end;
   ShowMouseRect;
end;

procedure TView.InitFont;
begin
   Font:=nil;
end;

function TView.GetFont;
begin
   if Font<>nil then GetFont:=Font else GetFont:=@SystemFont;
end;

function TView.CharWidth;
begin
   CharWidth:=GetFont^.Width;
end;
function TView.CharHeight;
begin
   CharHeight:=GetFont^.Height;
end;

function TView.GetPalette;
begin
   GetPalette:=nil;
end;

function TView.GetColor;
var
   RColor          :Integer;
begin
   RColor:=FindColor(Color, GetPalette);
   if (RColor=-1) and (Owner<>nil) then RColor:=Owner^.GetColor(Color);
   if RColor=-1 then
   if Color=cpError then RColor:=ErrorColor else RColor:=GetColor(cpError);
   GetColor:=RColor;
end;

function TView.GetColors;
assembler;
var
   Result          :Word;
asm
   les  di,Self
   push word ptr C2
   push es
   push di
   push word ptr C1
   push es
   push di
   call GetColor
   mov  Result,ax
   call GetColor
   xchg al,ah
   or   ax,Result
end;

function  TView.GetMousePointer;
begin
   if MousePointer<>nil then GetMousePointer:=MousePointer else
   if Owner<>nil then GetMousePointer:=Owner^.GetMousePointer else
   GetMousePointer:=nil;
end;

constructor TView.Init(var Bounds      :TGRect);
begin
   Inherited Init; InitFont;
   Owner := nil; State := sfVisible; SetBounds(Bounds);
   DragMode := dmLimitLoY; HelpCtx := hcNoContext;
   EventMask := evMouseDown + evKeyDown + evCommand;
   Options := ofClipping;
end;

constructor TView.Load(var S           :TStream);
begin
   Inherited Init;
   S.Read(Origin, SizeOf(TGPoint) * 3 + SizeOf(Byte) * 2 + SizeOf(Word) * 4);
   InitFont;
end;

destructor TView.Done;
begin
   Hide;
   if Owner <> nil then Owner^.Delete(@Self);
end;

procedure TView.ResetCursor;
var
   R               :TGRect;
begin
   if (GetState(sfVisible+sfCursorVis+sfFocused) or
      (State and (sfCursorVis+sfCursorOn)=sfCursorOn)) and
      (State and sfExposed<>0) then
   begin
      GetCursorRect(R, State and sfCursorIns<>0, False);
      DrawBar(R, GetColor(cpCursor));
      State:=State xor sfCursorOn;
   end;
end;

procedure TView.GetCursorRect;
var
   R1              :TGRect;
   S               :TGPoint absolute R1;
begin
   if Max then
   begin
      GetCursorRect(R1, False, False); GetCursorRect(R, True, False);
      R.Union(R1);
   end
   else
   begin
      if Ins then S:=CursorInsSize else S:=CursorSize;
      R.Assign(0, CharHeight-S.Y, S.X, CharHeight);
      if S.X=0 then R.B.X:=CharWidth;
      if S.Y=0 then R.A.Y:=0;
      R.Move(Cursor.X, Cursor.Y); Dec(R.B.X); Dec(R.B.Y);
   end;
end;

procedure TView.SetCursor(X, Y         :Integer);
const
   Working         :Boolean= False;
begin
   if Working or ((X=Cursor.X) and (Y=Cursor.Y)) then Exit;

   if State and sfCursorVis<>0 then
   begin
      Working:=True; DrawUnderCursor; Working:=False;
   end;

   Cursor.X := X; Cursor.Y := Y;
   ResetCursor;
end;

procedure TView.DrawCursor;
begin
   if not GetState(sfFocused + sfCursorVis + sfExposed) then Exit;
   DrawUnderCursor; ResetCursor;
end;

procedure TView.DrawUnderCursor;
var
   SaveClip        :TGRect;
begin
   if not GetState(sfFocused + sfExposed) then Exit;
   if Owner<>nil then
   begin
      SaveClip:=Owner^.Clip;
      GetCursorRect(Owner^.Clip, False, True);
      Owner^.Clip.Move(Origin.X, Origin.Y);
   end;
   Draw;
   if Owner<>nil then Owner^.Clip:=SaveClip;
end;

procedure TView.Awaken;
begin
end;

procedure TView.BlockCursor;
begin
   SetState(sfCursorIns, True);
end;

procedure TView.CalcBounds(var Bounds  :TGRect;
                           Delta       :TGPoint);
var
   S, D            :Integer;
   Min, Max        :TGPoint;

procedure Grow(var I                   :Integer);
begin
   if GrowMode and gfGrowRel = 0 then Inc(I, D) else
      I := LongDiv((LongMul(I, S) + (S - D) shr 1), (S - D));
end;

function Range(Val, Min, Max           :Integer) :Integer;
begin
   if Val < Min then Range := Min else
   if Val > Max then Range := Max else Range := Val;
end;

begin
   GetBounds(Bounds);
   S := Owner^.Size.X;
   D := Delta.X;
   if GrowMode and gfGrowLoX <> 0 then Grow(Bounds.A.X);
   if GrowMode and gfGrowHiX <> 0 then Grow(Bounds.B.X);
   S := Owner^.Size.Y;
   D := Delta.Y;
   if GrowMode and gfGrowLoY <> 0 then Grow(Bounds.A.Y);
   if GrowMode and gfGrowHiY <> 0 then Grow(Bounds.B.Y);
   SizeLimits(Min, Max);
   Bounds.B.X := Bounds.A.X + Range(Bounds.B.X - Bounds.A.X+1, Min.X, Max.X)-1;
   Bounds.B.Y := Bounds.A.Y + Range(Bounds.B.Y - Bounds.A.Y+1, Min.Y, Max.Y)-1;
end;

procedure TView.ChangeBounds(var Bounds:TGRect);
begin
   SetBounds(Bounds); DrawView;
end;

procedure TView.ClearEvent(var Event   :TEvent);
begin
   Event.What := evNothing;
   Event.InfoPtr := @Self;
end;

function TView.CommandEnabled(Command  :Word)    :Boolean;
begin
   CommandEnabled := (Command > 255) or (Command in CurCommandSet);
end;

function TView.DataSize                :Word;
begin
   DataSize := 0;
end;

procedure TView.DisableCommands(Commands:TCommandSet);
begin
   CommandSetChanged := CommandSetChanged or (CurCommandSet * Commands <> []);
   CurCommandSet := CurCommandSet - Commands;
end;

procedure TView.DragView(Event         :TEvent;
                         Mode          :Byte;
                         var Limits    :TGRect;
                         MinSize,
                         MaxSize       :TGPoint);
var
   P, S            :TGPoint;
   OldBounds,
   GlobalBounds,
   NewBounds,
   SaveBounds      :TGRect;

procedure FlipFrame(var Bounds         :TGRect);
begin
   if Bounds.Equals(OldBounds) then Exit;
   DrawDragFrame(GlobalBounds);
   OldBounds:=Bounds; GlobalBounds:=Bounds;
   with GlobalBounds do
   begin
      Owner^.MakeGlobal(A, A); Owner^.MakeGlobal(B, B);
   end;
   DrawDragFrame(GlobalBounds);
end;

procedure MoveGrow(var P, S            :TGPoint);
begin
   S.X := Min(Max(S.X, MinSize.X), MaxSize.X);
   S.Y := Min(Max(S.Y, MinSize.Y), MaxSize.Y);
   P.X := Min(Max(P.X, Limits.A.X - S.X + 1), Limits.B.X);
   P.Y := Min(Max(P.Y, Limits.A.Y - S.Y + 1), Limits.B.Y);
   if Mode and dmLimitLoX <> 0 then P.X := Max(P.X, Limits.A.X);
   if Mode and dmLimitLoY <> 0 then P.Y := Max(P.Y, Limits.A.Y);
   if Mode and dmLimitHiX <> 0 then P.X := Min(P.X, Limits.B.X - S.X);
   if Mode and dmLimitHiY <> 0 then P.Y := Min(P.Y, Limits.B.Y - S.Y);
   NewBounds.Assign(P.X, P.Y, P.X + S.X-1, P.Y + S.Y-1);
   FlipFrame(NewBounds);
end;

procedure Change(DX, DY                :Integer);
begin
   if (Mode and dmDragMove <> 0) and (GetShiftState and $03 = 0) then
   begin
      Inc(P.X, DX); Inc(P.Y, DY);
   end else
   if (Mode and dmDragGrow <> 0) and (GetShiftState and $03 <> 0) then
   begin
      Inc(S.X, DX); Inc(S.Y, DY);
   end;
end;

procedure Update(X, Y                  :Integer);
begin
   if Mode and dmDragMove <> 0 then
   begin
      P.X := X; P.Y := Y;
   end;
end;

begin
   HideMouse;
   SetState(sfDragging, True);
   OldBounds.MakeEmpty; GlobalBounds.MakeEmpty;
   if Event.What = evMouseDown then
   begin
      if Mode and dmDragMove <> 0 then
      begin
         P.X := Origin.X - Event.Where.X; P.Y := Origin.Y - Event.Where.Y;
         repeat
            Inc(Event.Where.X, P.X); Inc(Event.Where.Y, P.Y);
            MoveGrow(Event.Where, Size);
         until not MouseEvent(Event, evMouseMove);
         DrawDragFrame(GlobalBounds);
      end
      else
      begin
         P.X := Size.X - Event.Where.X; P.Y := Size.Y - Event.Where.Y;
         repeat
            Inc(Event.Where.X, P.X); Inc(Event.Where.Y, P.Y);
            MoveGrow(Origin, Event.Where);
         until not MouseEvent(Event, evMouseMove);
         DrawDragFrame(GlobalBounds);
      end;
      Locate(NewBounds);
   end else
   begin
      GetBounds(SaveBounds);
      FlipFrame(SaveBounds);
      P := Origin; S := Size;
      repeat
         repeat
            GetMouseEvent(Event);
            if Event.What and evMouseDown<>0 then
            begin
               Event.KeyCode:=kbEsc; Break;
            end;
            GetKeyEvent(Event);
         until Event.What=evKeyDown;
         case Event.KeyCode and $FF00 of
            kbLeft:  Change(-Grid.X, 0);
            kbRight: Change(Grid.X, 0);
            kbUp:    Change(0, -Grid.X);
            kbDown:  Change(0, Grid.X);
            kbHome:  Update(Limits.A.X, P.Y);
            kbEnd:   Update(Limits.B.X - S.X +1, P.Y);
            kbPgUp:  Update(P.X, Limits.A.Y);
            kbPgDn:  Update(P.X, Limits.B.Y - S.Y+1);
         end;
         MoveGrow(P, S);
      until (Event.KeyCode = kbEnter) or (Event.KeyCode = kbEsc);
      DrawDragFrame(GlobalBounds);
      if Event.KeyCode = kbEsc then Locate(SaveBounds)
      else Locate(NewBounds);
   end;
   SetState(sfDragging, False);
   ShowMouse;
end;

procedure TView.Draw;
var
   R               :TGRect;
begin
   GetClipRect(R); DrawBar(R, GetColor(cpMain));
end;

procedure TView.DrawHide(LastView      :PView);
begin
   DrawCursor; DrawUnderView(LastView);
end;

procedure TView.DrawShow(LastView      :PView);
begin
   DrawView;
end;

procedure TView.DrawUnderRect(var R    :TGRect;
                              LastView :PView);
begin
   Owner^.Clip.Intersect(R);
   Owner^.DrawSubViews(NextView, LastView);
   Owner^.GetExtent(Owner^.Clip);
end;

procedure TView.DrawUnderView(LastView :PView);
var
   R               :TGRect;
begin
   GetBounds(R); DrawUnderRect(R, LastView);
end;

procedure TView.DrawView;
begin
   if (LockDrawCount<=0) and Exposed then
   begin
      InitClipping;
      if MouseSafe then HideMouse;
      Draw; DrawCursor;
      if MouseSafe then ShowMouse;
   end;
end;

procedure TView.EnableCommands(Commands:TCommandSet);
begin
   CommandSetChanged := CommandSetChanged or
      (CurCommandSet * Commands <> Commands);
   CurCommandSet := CurCommandSet + Commands;
end;

procedure TView.EndModal(Command       :Word);
var
   P               :PView;
begin
   P := TopView;
   if TopView <> nil then TopView^.EndModal(Command);
end;

function TView.EventAvail              :Boolean;
var
   Event           :TEvent;
begin
   GetEvent(Event);
   if Event.What <> evNothing then PutEvent(Event);
   EventAvail := Event.What <> evNothing;
end;

procedure TView.GetBounds(var Bounds   :TGRect); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        ADD     SI,OFFSET TView.Origin
        LES     DI,Bounds
        CLD
        LODSW                           {Origin.X}
        MOV     CX,AX
        STOSW
        LODSW                           {Origin.Y}
        MOV     DX,AX
        STOSW
        LODSW                           {Size.X}
        ADD     AX,CX
        dec     ax
        STOSW
        LODSW                           {Size.Y}
        ADD     AX,DX
        dec     ax
        STOSW
        POP     DS
end;

function TView.Execute                 :Word;
begin
   Execute := cmCancel;
end;

function TView.Exposed                 :Boolean; assembler;
var
   Target          :Pointer;
asm
        LES     DI,Self
        TEST    ES:[DI].TView.State,sfExposed
        JE      @@2
        XOR     AX,AX
        CMP     AX,ES:[DI].TView.Size.X
        JGE     @@2
        CMP     AX,ES:[DI].TView.Size.Y
        JGE     @@2
@@1:    XOR     BX,BX
        MOV     CX,ES:[DI].TView.Size.X
        PUSH    AX
        CALL    @@11
        POP     AX
        JNC     @@3
        LES     DI,Self
        INC     AX
        CMP     AX,ES:[DI].TView.Size.Y
        JL      @@1
@@2:    MOV     AL,0
        JMP     @@30
@@3:    MOV     AL,1
        JMP     @@30
@@8:    STC
@@9:    RETN
@@10:   LES     DI,ES:[DI].TView.Owner
@@11:   MOV     Target.Word[0],DI
        MOV     Target.Word[2],ES
        ADD     AX,ES:[DI].TView.Origin.Y
        MOV     SI,ES:[DI].TView.Origin.X
        ADD     BX,SI
        ADD     CX,SI
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JE      @@9
        CMP     AX,ES:[DI].TGroup.Clip.A.Y
        JL      @@8
        CMP     AX,ES:[DI].TGroup.Clip.B.Y
        JG      @@8
        CMP     BX,ES:[DI].TGroup.Clip.A.X
        JGE     @@12
        MOV     BX,ES:[DI].TGroup.Clip.A.X
@@12:   CMP     CX,ES:[DI].TGroup.Clip.B.X
        JLE     @@13
        MOV     CX,ES:[DI].TGroup.Clip.B.X
        inc     cx
@@13:   CMP     BX,CX
        JG      @@8
        LES     DI,ES:[DI].TGroup.Last
@@20:   LES     DI,ES:[DI].TView.Next
        CMP     DI,Target.Word[0]
        JNE     @@21
        MOV     SI,ES
        CMP     SI,Target.Word[2]
        JE      @@10
@@21:   TEST    ES:[DI].TView.State,sfVisible
        JE      @@20
        MOV     SI,ES:[DI].TView.Origin.Y
        CMP     AX,SI
        JL      @@20
        ADD     SI,ES:[DI].TView.Size.Y
        CMP     AX,SI
        JGE     @@20
        MOV     SI,ES:[DI].TView.Origin.X
        CMP     BX,SI
        JL      @@22
        ADD     SI,ES:[DI].TView.Size.X
        CMP     BX,SI
        JGE     @@20
        MOV     BX,SI
        CMP     BX,CX
        JL      @@20
        STC
        RETN
@@22:   CMP     CX,SI
        JLE     @@20
        ADD     SI,ES:[DI].TView.Size.X
        CMP     CX,SI
        JG      @@23
        MOV     CX,ES:[DI].TView.Origin.X
        JMP     @@20
@@23:   PUSH    Target.Word[2]
        PUSH    Target.Word[0]
        PUSH    ES
        PUSH    DI
        PUSH    SI
        PUSH    CX
        PUSH    AX
        MOV     CX,ES:[DI].TView.Origin.X
        CALL    @@20
        POP     AX
        POP     CX
        POP     BX
        POP     DI
        POP     ES
        POP     Target.Word[0]
        POP     Target.Word[2]
        JC      @@20
        RETN
@@30:
end;

function TView.Focus                   :Boolean;
var
   Result          :Boolean;
begin
   Result := True;
   if State and (sfSelected + sfModal) = 0 then
   begin
      if Owner <> nil then
      begin
         Result := Owner^.Focus;
         if Result then
         if ((Owner^.Current = nil) or
            (Owner^.Current^.Options and ofValidate = 0) or
            (Owner^.Current^.Valid(cmReleasedFocus))) then Select
         else Result := False;
      end;
   end;
   Focus := Result;
end;

procedure TView.GetClipRect(var Clip   :TGRect);
begin
   GetBounds(Clip);
   if Owner <> nil then Clip.Intersect(Owner^.Clip);
   Clip.Move(-Origin.X, -Origin.Y);
end;

procedure TView.GetCommands(var Commands:TCommandSet);
begin
   Commands := CurCommandSet;
end;

procedure TView.GetData(var Rec);
begin
end;

procedure TView.GetEvent(var Event     :TEvent);
begin
   if Owner <> nil then Owner^.GetEvent(Event);
end;

procedure TView.GetExtent(var Extent   :TGRect); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        ADD     SI,OFFSET TView.Size
        LES     DI,Extent
        CLD
        XOR     AX,AX
        STOSW
        STOSW
        lodsw
        dec     ax
        stosw
        lodsw
        dec     ax
        stosw
        POP     DS
end;

function TView.GetHelpCtx              :Word;
begin
   if State and sfDragging <> 0 then GetHelpCtx := hcDragging
   else GetHelpCtx := HelpCtx;
end;

procedure TView.GetPeerViewPtr(var S   :TStream; var P);
var
   Index           :Integer;
begin
   S.Read(Index, SizeOf(Word));
   if (Index = 0) or (OwnerGroup = nil) then Pointer(P) := nil
   else
   begin
      Pointer(P) := FixupList^[Index]; FixupList^[Index] := @P;
   end;
end;

function TView.GetState(AState         :Word)    :Boolean;
begin
   GetState := State and AState = AState;
end;

procedure TView.GrowTo(X, Y            :Integer);
var
   R               :TGRect;
begin
   R.Assign(Origin.X, Origin.Y, Origin.X + X, Origin.Y + Y); Locate(R);
end;

procedure TView.HandleEvent(var Event  :TEvent);
begin
   if Event.What = evMouseDown then
   if (State and (sfSelected + sfDisabled) = 0) and
      (Options and ofSelectable <> 0) then
   if not Focus or (Options and ofFirstClick = 0) then ClearEvent(Event);
end;

procedure TView.Hide;
begin
   if State and sfVisible <> 0 then SetState(sfVisible, False);
end;

procedure TView.HideCursor;
begin
   SetState(sfCursorVis, False);
end;

procedure TView.KeyEvent(var Event     :TEvent);
begin
   repeat GetEvent(Event) until Event.What = evKeyDown;
end;

procedure TView.Locate(var Bounds      :TGRect);
var
   R               :TGRect;
   Min, Max        :TGPoint;

function Range(Val, Min, Max           :Integer) :Integer;
begin
   if Val < Min then Range := Min else
   if Val > Max then Range := Max else Range := Val;
end;

begin
   SizeLimits(Min, Max);
   Bounds.B.X := Bounds.A.X + Range(Bounds.B.X - Bounds.A.X+1, Min.X, Max.X)-1;
   Bounds.B.Y := Bounds.A.Y + Range(Bounds.B.Y - Bounds.A.Y+1, Min.Y, Max.Y)-1;
   GetBounds(R);
   if not Bounds.Equals(R) then
   begin
      ChangeBounds(Bounds);
      if (Owner <> nil) and (State and sfVisible <> 0) then DrawUnderRect(R, nil);
   end;
end;

procedure TView.MakeFirst;
begin
   PutInFrontOf(Owner^.First);
end;

procedure TView.MakeGlobal(Source      :TGPoint;
                           var Dest    :TGPoint); assembler;
asm
        LES     DI,Self
        XOR     AX,AX
        MOV     DX,AX
@@1:    ADD     AX,ES:[DI].TView.Origin.X
        ADD     DX,ES:[DI].TView.Origin.Y
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JNE     @@1
        ADD     AX,Source.X
        ADD     DX,Source.Y
        LES     DI,Dest
        CLD
        STOSW
        XCHG    AX,DX
        STOSW
end;

procedure TView.MakeLocal(Source       :TGPoint;
                          var Dest     :TGPoint); assembler;
asm
        LES     DI,Self
        XOR     AX,AX
        MOV     DX,AX
@@1:    ADD     AX,ES:[DI].TView.Origin.X
        ADD     DX,ES:[DI].TView.Origin.Y
        LES     DI,ES:[DI].TView.Owner
        MOV     SI,ES
        OR      SI,DI
        JNE     @@1
        NEG     AX
        NEG     DX
        ADD     AX,Source.X
        ADD     DX,Source.Y
        LES     DI,Dest
        CLD
        STOSW
        XCHG    AX,DX
        STOSW
end;

function TView.MouseEvent(var Event    :TEvent;
                          Mask         :Word)    :Boolean;
begin
   repeat GetEvent(Event) until Event.What and (Mask or evMouseUp) <> 0;
   MouseEvent := Event.What <> evMouseUp;
end;

function TView.MouseInView(Mouse       :TGPoint)  :Boolean;
var
   Extent          :TGRect;
begin
   MakeLocal(Mouse, Mouse);
   GetExtent(Extent);
   MouseInView := Extent.Contains(Mouse);
end;

procedure TView.MoveTo(X, Y            :Integer);
var
   R               :TGRect;
begin
   R.Assign(X, Y, X + Size.X, Y + Size.Y); Locate(R);
end;

function TView.NextView                :PView;
begin
   if @Self = Owner^.Last then NextView := nil else NextView := Next;
end;

procedure TView.NormalCursor;
begin
   SetState(sfCursorIns, False);
end;

function TView.Prev                    :PView; assembler;
asm
        LES     DI,Self
        MOV     CX,DI
        MOV     BX,ES
@@1:    MOV     AX,DI
        MOV     DX,ES
        LES     DI,ES:[DI].TView.Next
        CMP     DI,CX
        JNE     @@1
        MOV     SI,ES
        CMP     SI,BX
        JNE     @@1
end;

function TView.PrevView                :PView;
begin
   if @Self = Owner^.First then PrevView := nil else PrevView := Prev;
end;

procedure TView.PutEvent(var Event     :TEvent);
begin
   if Owner <> nil then Owner^.PutEvent(Event);
end;

procedure TView.PutInFrontOf(Target    :PView);
var
   P, LastView     :PView;

procedure MoveView;
begin
   Owner^.RemoveView(@Self); Owner^.InsertView(@Self, Target);
end;

begin
   if (Owner <> nil) and (Target <> @Self) and (Target <> NextView) and
      ((Target = nil) or (Target^.Owner = Owner)) then
   if State and sfVisible = 0 then MoveView else
   begin
      LastView := NextView;
      if LastView <> nil then
      begin
         P := Target;
         while (P <> nil) and (P <> LastView) do P := P^.NextView;
         if P = nil then LastView := Target;
      end;
      State := State and not sfVisible;
      if LastView = Target then DrawHide(LastView);
      MoveView;
      State := State or sfVisible;
      if LastView <> Target then DrawShow(LastView);
      if Options and ofSelectable <> 0 then
      begin
         Owner^.ResetCurrent; Owner^.ResetCursor;
      end;
   end;
end;

procedure TView.PutPeerViewPtr(var S   :TStream;
                               P       :PView);
var
   Index           :Integer;
begin
   if (P = nil) or (OwnerGroup = nil) then Index := 0
   else Index := OwnerGroup^.IndexOf(P);
   S.Write(Index, SizeOf(Word));
end;

procedure TView.Select;
begin
   if Options and ofSelectable <> 0 then
   if Options and ofTopSelect <> 0 then MakeFirst else
   if Owner <> nil then Owner^.SetCurrent(@Self, NormalSelect);
end;

procedure TView.SetBounds(var Bounds   :TGRect); assembler;
asm
        PUSH    DS
        LES     DI,Self
        LDS     SI,Bounds
        MOV     AX,[SI].TGRect.A.X
        MOV     ES:[DI].Origin.X,AX
        MOV     AX,[SI].TGRect.A.Y
        MOV     ES:[DI].Origin.Y,AX
        MOV     AX,[SI].TGRect.B.X
        SUB     AX,[SI].TGRect.A.X
        inc     ax
        MOV     ES:[DI].Size.X,AX
        MOV     AX,[SI].TGRect.B.Y
        SUB     AX,[SI].TGRect.A.Y
        inc     ax
        MOV     ES:[DI].Size.Y,AX
        POP     DS
end;

procedure TView.SetCmdState(Commands   :TCommandSet;
                            Enable     :Boolean);
begin
   if Enable then EnableCommands(Commands) else DisableCommands(Commands);
end;

procedure TView.SetCommands(Commands   :TCommandSet);
begin
   CommandSetChanged := CommandSetChanged or (CurCommandSet <> Commands);
   CurCommandSet := Commands;
end;

procedure TView.SetData(var Rec);
begin
end;

procedure TView.SetState(AState        :Word;
                         Enable        :Boolean);
var
   Command         :Word;
begin
   if Enable then State := State or AState else State := State and not AState;
   if Owner <> nil then
   case AState of
      sfVisible:
      begin
         if Owner^.State and sfExposed <> 0 then SetState(sfExposed, Enable);
         if Enable then DrawShow(nil) else DrawHide(nil);
         if Options and ofSelectable <> 0 then Owner^.ResetCurrent;
      end;
      sfCursorVis, sfCursorIns:
         DrawCursor;
      sfFocused:
      begin
         InitClipping;
         ResetCursor;
         if Enable then Command := cmReceivedFocus
         else Command := cmReleasedFocus;
         Message(Owner, evBroadcast, Command, @Self);
      end;
   end;
end;

procedure TView.Show;
begin
   if State and sfVisible = 0 then SetState(sfVisible, True);
end;

procedure TView.ShowCursor;
begin
   SetState(sfCursorVis, True);
end;

procedure TView.SizeLimits(var Min, Max:TGPoint);
begin
   LongInt(Min) := 0;
   if Owner <> nil then Max := Owner^.Size else Max := Application^.Size;
end;

procedure TView.Store(var S            :TStream);
var
   SaveState       :Word;
begin
   SaveState := State;
   State := State and not (sfActive + sfSelected + sfFocused + sfExposed);
   S.Write(Origin, SizeOf(TGPoint) * 3 + SizeOf(Byte) * 2 + SizeOf(Word) * 4);
   State := SaveState;
end;

function TView.TopView                 :PView;
var
   P               :PView;
begin
   if TheTopView = nil then
   begin
      P := @Self;
      while (P <> nil) and (P^.State and sfModal = 0) do P := P^.Owner;
      TopView := P;
   end
   else TopView := TheTopView;
end;

function TView.Valid(Command           :Word)              :Boolean;
begin
   Valid := True;
end;

{ TFrame }

constructor TFrame.Init(var Bounds     :TGRect);
begin
   Inherited Init(Bounds);
   GrowMode := gfGrowHiX + gfGrowHiY;
   EventMask := EventMask or evBroadcast;
end;

function  TFrame.GetPalette;
begin
   if State and sfActive<>0 then GetPalette:=CFrame else GetPalette:=CPassiveFrame;
end;

procedure TFrame.DrawZoomIcon;
var
   Min, Max        :TGPoint;
   R               :TGRect;
begin
   GetExtent(R);
   R.Assign(R.B.X-SBarSize.X, R.A.Y+1, R.B.X-1, R.A.Y);
   Inc(R.B.Y, CharHeight+1);
   HideMouseLocal(R);
   DrawButton(R, dbSmall+dbFillIn, Down, GetColor(cpButBody));
   if Down then R.Move(1, 1);

   while R.SizeY>SBarPartSize.Y do
   begin
      Inc(R.A.Y); if R.SizeY>SBarPartSize.Y then Dec(R.B.Y);
   end;
   R.Grow(-2, -2);
   if General.Min(R.B.X-R.A.X, R.B.Y-R.A.Y)>8 then R.Grow(-1, -1);

   Owner^.SizeLimits(Min, Max);
   DrawVArrow(R, LongInt(Owner^.Size)<>LongInt(Max), GetColor(cpTopIcons));
   ShowMouseRect;
end;

procedure TFrame.Draw;
Var
   Title                     :TTitleStr;
   IconPos, Pos              :TGPoint;
   R, BR, RI                 :TGRect;
   IconSpace, YShift         :Integer;

   procedure DrawIcon(Icon             :Char;
                      Color            :TColor);
   var
      Font         :TTextStyle;
   begin
      Font:=GetFont^;
      if Icon<' ' then
      begin
         if Font.Font<>nil then
         begin
            Font.SizeX:=Font.Width; Font.DivX:=8;
            Font.SizeY:=Font.Height; Font.DivY:=8;
         end;
         Font.Font:=@ControlsFont;
      end;

      DrawText(IconPos, Icon, jsLeft+jsTop, @Font, GetColor(Color));
      Inc(IconPos.X, IconSpace);
   end;

begin
   GetExtent(R); DrawRect(R, GetColor(cpFrame));

   R.Grow(-1, -1); R.B.Y:=R.A.Y+CharHeight;
   DrawBar(R, GetColor(cpTitleBk));
   Pos.X:=R.A.X; Pos.Y:=R.B.Y+1;
   DrawHLine(Pos, R.SizeX-1, GetColor(cpHFrame));

   IconPos:=R.A;
   YShift:=CharHeight shr 4;
   Inc(IconPos.Y, YShift);
   IconSpace:=CharWidth*3 shr 1;

   if PWindow(Owner)^.Flags and (wfBottomLine or wfGrow) <> 0 then
   begin
      BR:=R; BR.Move(0, Size.Y-CharHeight-2); DrawBar(BR, GetColor(cpBottom));
   end else BR.A.Y:=Size.Y-1;
   RI.Assign(R.A.X, R.B.Y+2, R.B.X, BR.A.Y-1);
   DrawBar(RI, Owner^.GetColor(cpMain));

   RI.A:=R.A; RI.B.X:=RI.A.X + CharWidth; RI.B.Y:=RI.A.Y + CharHeight;
   DrawBar(RI, GetColor(cpTopIconsBk));
   DrawIcon('', cpTopIcons);

   IconPos.X:=R.B.X-CharWidth+1; Inc(IconPos.Y);
   Inc(YShift);
   IconSpace:=-IconSpace;

   if PWindow(Owner)^.Flags and wfZoom <> 0 then
   begin
      DrawZoomIcon(False); Dec(IconPos.X, CharWidth+SBarSize.X);
   end;

   if (PWindow(Owner)^.Number > wnNoNumber) and
      (PWindow(Owner)^.Number < 10) then
      DrawIcon(Chr(PWindow(Owner)^.Number+Ord('0')), cpTitle);

   Title := PWindow(Owner)^.GetTitle((Size.X - SBarSize.X) div CharWidth - 6);
   if Title <> '' then
   begin
      Pos.X:=(R.A.X+R.B.X) shr 1; Pos.Y:=R.A.Y+YShift;
      DrawText(Pos, Title, jsCenterX+jsTop, nil, GetColor(cpTitle));
   end;

   if PWindow(Owner)^.Flags and wfGrow <> 0 then
   begin
      IconPos.X:=BR.B.X+1; IconPos.Y:=BR.A.Y; Dec(IconPos.X, CharWidth);
      DrawIcon('', cpBottomIcons);
   end;
end;

procedure TFrame.HandleEvent(var Event :TEvent);
var
   Mouse           :TGPoint;
   TitleRect       :TGRect;
   Tracking, Down  :Boolean;

procedure DragWindow(Mode              :Byte);
var
   Limits          :TGRect;
   Min, Max        :TGPoint;
begin
   Owner^.Owner^.GetExtent(Limits);
   Owner^.SizeLimits(Min, Max);
   Owner^.DragView(Event, Owner^.DragMode or Mode, Limits, Min, Max);
   ClearEvent(Event);
end;

procedure CallWindowMenu;
var
   P               :PMenuView;
   R               :TGRect;
   G               :PGroup;
begin
   R.Assign(1, 1, 0, 0);
   if Owner^.State and sfModal<>0 then G:=Owner else
   begin
      G:=PGroup(Application); MakeGlobal(R.A, R.A);
   end;
   P := New(PMenuPopup, Init(R, NewMenu(PWindow(Owner)^.GetMenu(nil))));
   Event.Command := G^.ExecView(PView(P));
   Dispose(P, Done);
   if (Event.Command<>0) and (Event.Command<>cmCancel) then
   begin
      Event.What := evCommand; Event.InfoPtr := Owner; PutEvent(Event);
   end;
end;

begin
   Inherited HandleEvent(Event);
   case Event.What of
      evBroadcast:
      if Event.Command = cmWindowMenu then
      begin
         CallWindowMenu; ClearEvent(Event);
      end;
      evMouseDown:
      begin
         MakeLocal(Event.Where, Mouse);
         if (Mouse.Y >= 0) and (Mouse.Y < GetPartSize(fpTop)) then
         begin
            if (State and sfActive <> 0) and
               (Mouse.X >= 0) and (Mouse.X < CharWidth) then
            begin
               repeat until not MouseEvent(Event, evMouseMove + evMouseAuto);
               MakeLocal(Event.Where, Mouse);
               if Mouse.Rect(0, 0, CharWidth-1, GetPartSize(fpTop)-1) then
               if PWindow(Owner)^.Flags and wfMenu<>0 then CallWindowMenu else
               begin
                  Event.What := evCommand; Event.Command:=cmClose;
                  Event.InfoPtr := Owner; PutEvent(Event);
               end;
               ClearEvent(Event);
            end else
            if (PWindow(Owner)^.Flags and wfZoom <> 0) and
               (State and sfActive <> 0) and (Event.Double or
               (Mouse.X >= Size.X - SBarSize.X)) then
            begin
               Tracking := True;
               if not Event.Double then
               begin
                  Down := True; DrawZoomIcon(Down);
                  repeat
                     MakeLocal(Event.Where, Mouse);
                     if Down <> Tracking then
                     begin
                        Down:=not Down; DrawZoomIcon(Down);
                     end;
                     Tracking:=Mouse.Rect(Size.X - SBarSize.X, 0,
                        Size.X, GetPartSize(fpTop)-1);
                  until not MouseEvent(Event, evMouseAuto);
                  if Down then DrawZoomIcon(False);
               end;
               if Tracking or Event.Double then
               begin
                  Event.What := evCommand; Event.Command := cmZoom;
                  Event.InfoPtr := Owner; PutEvent(Event);
               end;
               ClearEvent(Event);
            end else
            if PWindow(Owner)^.Flags and wfMove <> 0 then DragWindow(dmDragMove);
         end else
         if (State and sfActive <> 0) and
            (Mouse.X >= Size.X - CharWidth) and
            (Mouse.Y >= Size.Y - GetPartSize(fpBottom)) then
         if PWindow(Owner)^.Flags and wfGrow <> 0 then DragWindow(dmDragGrow);
      end;
   end;
end;

procedure TFrame.SetState(AState       :Word;
                          Enable       :Boolean);
begin
   TView.SetState(AState, Enable);
   if AState and sfActive <> 0 then DrawView;
end;

function  TFrame.GetPartSize;
begin
   case Part of
      fpLeft, fpRight:
         GetPartSize:=1;
      fpTop:
         GetPartSize:=CharHeight+3;
      fpBottom:
         if PWindow(Owner)^.Flags and (wfBottomLine or wfGrow) <> 0 then
            GetPartSize:=CharHeight+1 else GetPartSize:=1;
   end;
end;

{ TScrollBar }

constructor TScrollBar.Init(var Bounds :TGRect;
                            AVert      :Boolean);
begin
   Inherited Init(Bounds);
   Value := 0; Min := 0; Max := 0; PgStep := 1; ArStep := 1;
   if Size.X<=1 then Size.X:=SBarSize.X;
   if Size.Y<=1 then Size.Y:=SBarSize.Y;
   Vert:=AVert;
   if Vert then GrowMode := gfGrowLoX + gfGrowHiX + gfGrowHiY
   else GrowMode := gfGrowLoY + gfGrowHiX + gfGrowHiY;
   Options:=Options or ofUpdate;
end;

constructor TScrollBar.Load(var S      :TStream);
begin
   Inherited Load(S);
   S.Read(Value, SizeOf(Integer) * 5 + SizeOf(Boolean));
end;

function  TScrollBar.GetPalette;
begin
   GetPalette:=CScrollBar;
end;

procedure TScrollBar.Draw;
begin
   DrawAll(-1);
end;

procedure TScrollBar.DrawAll;
var
   Part            :Integer;
begin
   if MouseSafe then HideMouse;
   for Part := sbUpArrow to sbIndicator do DrawPart(Part, Part = Selected);
   if MouseSafe then ShowMouse;
end;

procedure TScrollBar.DrawPart;
var
   R               :TGRect;
   C               :Integer;
begin
   GetPartRect(Part, R);
   HideMouseLocal(R);
   if Part and 2 <> 0 then
   begin
      if Selected then C:=GetColor(cpBarSelected) else C:=GetColor(cpBar);
      DrawBar(R, C);
   end
   else
   begin
      DrawButton(R, dbFillIn+dbSmall, Selected, GetColor(cpIndicator));
      if Part <> sbIndicator then
      begin
         R.Grow(-2, -2);
         if General.Min(R.B.X-R.A.X, R.B.Y-R.A.Y)>8 then R.Grow(-1, -1);
         if Selected then R.Move(1, 1);
         if Vert
         then DrawVArrow(R, (Part and 1 = 0), GetColor(cpArrows))
         else DrawHArrow(R, (Part and 1 = 0), GetColor(cpArrows));
      end;
   end;
   ShowMouseRect;
end;

function  TScrollBar.GetPos            :Integer;
var
   R               :Integer;
begin
   R := Max - Min;
   if R = 0 then GetPos := 1 else
      GetPos := LongDiv(LongMul(Value - Min, GetSize - GetPartSize*3) + R shr 1, R) + 1;
end;

function  TScrollBar.GetSize           :Integer;
begin
   if Vert then GetSize := Size.Y else GetSize := Size.X;
end;

procedure TScrollBar.GetPartRect;
var
   I, H, W         :Integer;
begin
   I := GetPos;
   if Vert then
   begin
      H:=GetPartSize-1; W:=Size.X-1;
   end
   else
   begin
      W:=GetPartSize-1; H:=Size.Y-1;
   end;
   if Part and 2 <> 0 then
   begin
      GetExtent(R);
      if Vert then
      begin
         R.Grow(0, -H);
         if Part and 1 = 0 then
         begin
            R.B.Y := R.A.Y+I-1; Inc(R.A.Y);
         end
         else
         begin
            Inc(R.A.Y, I+H+1); Dec(R.B.Y);
         end;
         if R.A.Y>R.B.Y then R.MakeEmpty;
      end
      else
      begin
         R.Grow(-W, 0);
         if Part and 1 = 0 then
         begin
            R.B.X := R.A.X+I-1; Inc(R.A.X);
         end
         else
         begin
            Inc(R.A.X, I+W+1); Dec(R.B.X);
         end;
         if R.A.X>R.B.X then R.MakeEmpty;
      end;
   end
   else if Part = sbIndicator then
   begin
      R.Assign(0, 0, W, H);
      if Vert then R.Move(0, I+H) else R.Move(I+W, 0);
   end
   else
   if Part and 1 = 0 then R.Assign(0, 0, W, H) else
   begin
      R.B := Size; R.A.X := Size.X-W; R.A.Y := Size.Y-H; R.Move(-1, -1);
   end;
end;

function  TScrollBar.GetPartCode;
var
   Part            :Integer;
   R               :TGRect;
begin
   GetPartCode:=-1;
   for Part:=sbLeftArrow to sbIndicator do
   begin
      GetPartRect(Part, R);
      if R.Contains(Point) then
      begin
         if Vert and (Part<>sbIndicator) then Part:=Part or 4;
         GetPartCode:=Part; Exit;
      end;
   end;
end;

function  TScrollBar.GetPartSize;
begin
   if Vert then GetPartSize:=SBarPartSize.Y else GetPartSize:=SBarPartSize.X;
end;

procedure TScrollBar.HandleEvent(var Event       :TEvent);
var
   Update, Tracking                    :Boolean;
   I, D, S, Sd, P, ClickPart           :Integer;
   Mouse                               :TGPoint;
   R, Extent                           :TGRect;

   procedure Clicked;
   begin
      Message(Owner, evBroadcast, cmScrollBarClicked, @Self);
   end;

   procedure XorIndicator(Pos          :Integer);
   begin
      if Options and ofUpdate <> 0 then Exit;
      GetPartRect(sbIndicator, R); R.Assign(0, 0, R.B.X-R.A.X, R.B.Y-R.A.Y);
      if Vert then R.Move(0, GetPartSize+Pos) else R.Move(GetPartSize+Pos, 0);
      MakeGlobal(R.A, R.A); MakeGlobal(R.B, R.B); DrawDragFrame(R);
   end;

begin
   Inherited HandleEvent(Event);
   case Event.What of
      evMouseAuto:
         ClearEvent(Event);
      evMouseDown:
      begin
         Clicked;
         MakeLocal(Event.Where, Mouse);
         GetExtent(Extent);
         ClickPart := GetPartCode(Mouse);
         Update := True;
         if ClickPart <> sbIndicator then
         begin
            repeat
               MakeLocal(Event.Where, Mouse);
               if GetPartCode(Mouse) = ClickPart then
               begin
                  if Update then
                  begin
                     DrawPart(ClickPart, True); Update := False;
                  end;
                  LockDraw;
                  P := Value; SetValue(Value + ScrollStep(ClickPart));
                  UnlockDraw;
                  if P <> Value then
                  begin
                     D := ClickPart or 4;
                     for I := sbPageUp to sbIndicator do
                     if (I <> D) then DrawPart(I, False);
                  end;
               end
               else if not Update then
               begin
                  DrawPart(ClickPart, False); Update := True;
               end;
            until not MouseEvent(Event, evMouseAuto);
            DrawPart(ClickPart, False);
         end
         else
         begin
            D := -1; S := GetSize; P := GetPos-1;
            GetExtent(Extent);
            Tracking := True;
            repeat
               MakeLocal(Event.Where, Mouse);
               if (Tracking <> Extent.Contains(Mouse)) and (D >= 0) then
                  XorIndicator(I);
               Tracking := Extent.Contains(Mouse);
               if Tracking then
               begin
                  if D >= 0 then XorIndicator(I);
                  if Vert then I := Mouse.Y else I := Mouse.X;
                  if D < 0 then D := I-GetPartSize-P;
                  Dec(I, D);
                  if I < GetPartSize then I := 0 else
                  if I >= S-(GetPartSize shl 1) then I := S - 3*GetPartSize
                  else Dec(I, GetPartSize);
                  XorIndicator(I);
               end
               else I := GetPos-1;
               P := I;
               if (Options and ofUpdate <> 0) and (S > GetPartSize*3) then
               begin
                  Sd:=S-3*GetPartSize;
                  SetValue(LongDiv(LongMul(P, Max - Min) + Sd shr 1, Sd) + Min);
               end;
            until not MouseEvent(Event, evMouseMove);
            if Tracking and (Options and ofUpdate = 0) then
            begin
               XorIndicator(I);
               if S > GetPartSize*3 then
               begin
                  Sd:=S-3*GetPartSize;
                  SetValue(LongDiv(LongMul(P, Max - Min) + Sd shr 1, Sd) + Min);
               end;
            end;
         end;
         ClearEvent(Event);
      end;
      evKeyDown:
      if State and sfVisible <> 0 then
      begin
         ClickPart := sbIndicator;
         if Vert then
         case CtrlToArrow(Event.KeyCode) of
            kbUp:       ClickPart := sbUpArrow;
            kbDown:     ClickPart := sbDownArrow;
            kbPgUp:     ClickPart := sbPageUp;
            kbPgDn:     ClickPart := sbPageDown;
            kbCtrlPgUp: I := Min;
            kbCtrlPgDn: I := Max;
         else
            Exit;
         end
         else
         case CtrlToArrow(Event.KeyCode) of
            kbLeft:     ClickPart := sbLeftArrow;
            kbRight:    ClickPart := sbRightArrow;
            kbCtrlLeft: ClickPart := sbPageLeft;
            kbCtrlRight:ClickPart := sbPageRight;
            kbHome:     I := Min;
            kbEnd:      I := Max;
         else
            Exit;
         end;
         Clicked;
         if ClickPart <> sbIndicator then I := Value + ScrollStep(ClickPart);
         SetValue(I);
         ClearEvent(Event);
      end;
   end;
end;

procedure TScrollBar.ScrollDraw;
begin
   Message(Owner, evBroadcast, cmScrollBarChanged, @Self);
end;

function TScrollBar.ScrollStep(Part    :Integer) :Integer;
var
   Step            :Integer;
begin
   if Part and 2 = 0 then Step := ArStep else Step := PgStep;
   if Part and 1 = 0 then ScrollStep := -Step else ScrollStep := Step;
end;

procedure TScrollBar.SetParams(AValue,
                               AMin, AMax,
                               APgStep, AArStep  :Integer);
var
   SValue          :Integer;
   IsVisible       :Boolean;
   Part            :Integer;
begin
   if AMax < AMin then AMax := AMin;
   if AValue < AMin then AValue := AMin;
   if AValue > AMax then AValue := AMax;
   SValue := Value;
   if (SValue <> AValue) or (Min <> AMin) or (Max <> AMax) then
   begin
      Value := AValue; Min := AMin; Max := AMax;
      IsVisible := GetState(sfVisible);
      if AMin = AMax then
      if IsVisible then Hide else
      else if not IsVisible and GetState(sfActive) then Show;
      if (LockDrawCount<=0) and Exposed then
      begin
         if MouseSafe then HideMouse;
         for Part := sbPageUp to sbIndicator do DrawPart(Part, False);
         if MouseSafe then ShowMouse;
      end;
      if SValue <> AValue then ScrollDraw;
   end;
   PgStep := APgStep; ArStep := AArStep;
end;

procedure TScrollBar.SetRange(AMin, AMax:Integer);
begin
   SetParams(Value, AMin, AMax, PgStep, ArStep);
end;

procedure TScrollBar.SetState(AState   :Word;
                              Enable   :Boolean);
begin
   if (AState and sfVisible <> 0) and Enable and (Min = Max) then Exit;
   Inherited SetState(AState, Enable);
end;

procedure TScrollBar.SetStep(APgStep, AArStep    :Integer);
begin
   SetParams(Value, Min, Max, APgStep, AArStep);
end;

procedure TScrollBar.SetValue(AValue   :Integer);
begin
   SetParams(AValue, Min, Max, PgStep, ArStep);
end;

procedure TScrollBar.Store(var S       :TStream);
begin
   Inherited Store(S);
   S.Write(Value, SizeOf(Integer) * 5 + SizeOf(Boolean));
end;

{ TScroller }

constructor TScroller.Init(var Bounds  :TGRect;
                           AHScrollBar,
                           AVScrollBar :PScrollBar);
begin
   Inherited Init(Bounds);
   Options := Options or ofSelectable;
   EventMask := EventMask or evBroadcast;
   HScrollBar := AHScrollBar; VScrollBar := AVScrollBar;
end;

constructor TScroller.Load(var S       :TStream);
begin
   Inherited Load(S);
   GetPeerViewPtr(S, HScrollBar); GetPeerViewPtr(S, VScrollBar);
   S.Read(Delta, SizeOf(TGPoint)*2);
end;

procedure TScroller.ChangeBounds(var Bounds      :TGRect);
begin
   SetBounds(Bounds);
   LockDraw; SetLimit(Limit.X, Limit.Y); UnlockDraw;
   RedrawNeeded := True; DrawView;
end;

procedure TScroller.DrawView;
begin
   if (LockDrawCount <= 0) and RedrawNeeded then
   begin
      Inherited DrawView; RedrawNeeded := False;
   end;
end;

procedure TScroller.HandleEvent(var Event        :TEvent);
begin
   Inherited HandleEvent(Event);
   if (Event.What = evBroadcast) and (Event.Command = cmScrollBarChanged) and
      ((Event.InfoPtr = HScrollBar) or (Event.InfoPtr = VScrollBar)) then
      ScrollDraw;
end;

procedure TScroller.ScrollDraw;
var
   D               :TGPoint;
begin
   if HScrollBar <> nil then D.X := HScrollBar^.Value else D.X := 0;
   if VScrollBar <> nil then D.Y := VScrollBar^.Value else D.Y := 0;
   if (D.X <> Delta.X) or (D.Y <> Delta.Y) then
   begin
      SetCursor(Cursor.X + Delta.X - D.X, Cursor.Y + Delta.Y - D.Y);
      Delta := D;
      if LockDrawCount > 0 then RedrawNeeded := True else DrawView;
   end;
end;

procedure TScroller.ScrollTo(X, Y      :Integer);
begin
   LockDraw;
   if HScrollBar <> nil then HScrollBar^.SetValue(X);
   if VScrollBar <> nil then VScrollBar^.SetValue(Y);
   UnlockDraw;
   DrawView;
end;

procedure TScroller.SetLimit(X, Y      :Integer);
begin
   Limit.X := X; Limit.Y := Y;
   LockDraw;
   if HScrollBar <> nil then
      HScrollBar^.SetParams(HScrollBar^.Value, 0, X - Size.X, Size.X - 1,
      HScrollBar^.ArStep);
   if VScrollBar <> nil then
      VScrollBar^.SetParams(VScrollBar^.Value, 0, Y - Size.Y, Size.Y - 1,
      VScrollBar^.ArStep);
   UnlockDraw;
   DrawView;
end;

procedure TScroller.SetState(AState    :Word;
                             Enable    :Boolean);

procedure ShowSBar(SBar                :PScrollBar);
begin
   if (SBar <> nil) then
   if GetState(sfActive + sfSelected) then SBar^.Show else SBar^.Hide;
end;

begin
   Inherited SetState(AState, Enable);
   if AState and (sfActive + sfSelected) <> 0 then
   begin
      ShowSBar(HScrollBar); ShowSBar(VScrollBar);
   end;
end;

procedure TScroller.Store(var S        :TStream);
begin
   Inherited Store(S);
   PutPeerViewPtr(S, HScrollBar); PutPeerViewPtr(S, VScrollBar);
   S.Write(Delta, SizeOf(TGPoint)*2);
end;

{ TListViewer }

constructor TListViewer.Init(var Bounds          :TGRect;
                             ANumCols            :Word;
                             AHScrollBar,
                             AVScrollBar         :PScrollBar);
var
   ArStep, PgStep                      :Integer;
begin
   Inherited Init(Bounds);
   Options := Options or (ofFirstClick + ofSelectable);
   EventMask := EventMask or evBroadcast;
   Range := 0; NumCols := ANumCols; Focused := 0;
   HScrollBar := AHScrollBar; VScrollBar := AVScrollBar;

   Size.Y := Max(Size.Y - Size.Y mod CharHeight, CharHeight);
   Dec(Size.X, FrameOffset shl 1);
   Size.X := Max(Size.X - Size.X mod CharWidth,  CharWidth);
   Inc(Size.X, FrameOffset shl 1);
   NumRows := Size.Y div CharHeight;
   if VScrollBar <> nil then
   begin
      if NumCols = 1 then
      begin
         PgStep := NumRows - 1; ArStep := 1;
      end
      else
      begin
         PgStep := NumRows * NumCols; ArStep := NumRows;
      end;
      VScrollBar^.SetStep(PgStep, ArStep);
   end;
   if HScrollBar <> nil then
      HScrollBar^.SetStep((Size.X div NumCols) div CharWidth, 1);
end;

constructor TListViewer.Load(var S     :TStream);
begin
   Inherited Load(S);
   GetPeerViewPtr(S, HScrollBar); GetPeerViewPtr(S, VScrollBar);
   S.Read(NumCols, SizeOf(Word) * 5);
end;

function  TListViewer.GetPalette;
begin
   GetPalette:=CListViewer;
end;

procedure TListViewer.ChangeBounds(var Bounds    :TGRect);
begin
   Inherited ChangeBounds(Bounds);
   if HScrollBar <> nil then
      HScrollBar^.SetStep(Size.X div NumCols, HScrollBar^.ArStep);
   if VScrollBar <> nil then
      VScrollBar^.SetStep(Size.Y, VScrollBar^.ArStep);
end;

procedure TListViewer.Draw;
var
   R, Clip         :TGRect;
   Item, C, Indent :Integer;
begin
   GetExtent(R); C:=GetColor(cpMain);
   R.Assign(0, 0, FrameOffset-1, Size.Y-1); DrawBar(R, C);

   Item := TopItem;
   if HScrollBar <> nil then Indent := HScrollBar^.Value else Indent := 0;

   GetClipRect(Clip);
   while Item < TopItem + NumRows * NumCols do
   begin
      GetItemRect(Item, R);
      if R.Intersects(Clip) then
      if Item < Range then DrawItem(R, Item, Indent) else DrawBar(R, C);
      Inc(Item);
   end;
   R.Assign(Size.X - FrameOffset, 0, Size.X-1, Size.Y-1); DrawBar(R, C);
end;

procedure TListViewer.DrawItem(var Bounds        :TGRect;
                               Item, Indent      :Integer);
var
   CText, CBack,
   MaxLen          :Integer;
   T               :String;
   TLen            :Byte absolute T;
begin
   CBack := GetColor(cpMain);
   if GetState(sfSelected + sfActive) and IsSelected(Item) then
   begin
      CText := GetColor(cpSelected); CBack := GetColor(cpSelectedBk);
      SetCursor(Bounds.A.X, Bounds.A.Y);
   end
   else
   if IsSelected(Item) then CText := GetColor(cpSelDisabled)
   else CText := GetColor(cpNormal);

   MaxLen := (Bounds.B.X - Bounds.A.X + 1) div CharWidth;
   if Item < Range then T := GetText(Item, MaxLen + Indent) else T := '';
   T := Copy(T, Indent, MaxLen);
   FillChar(T[TLen + 1], MaxLen - TLen, ' '); TLen := MaxLen;

   DrawBar(Bounds, CBack); DrawText(Bounds.A, T, tfNormal, nil, CText);
end;

procedure TListViewer.FocusItem(Item   :Integer);
var
   R               :TGRect;
   Indent,
   SaveTopItem,
   SaveFocused     :Integer;
begin
   SaveTopItem := TopItem; SaveFocused := Focused; Focused := Item;
   if VScrollBar <> nil then VScrollBar^.SetValue(Item);

   if Exposed then
   begin
      if HScrollBar <> nil then Indent := HScrollBar^.Value else Indent := 0;

      if Item < TopItem then
      if NumCols = 1 then TopItem := Item else TopItem := Item - Item mod NumRows
      else if Item >= TopItem + (NumRows*NumCols) then
      if NumCols = 1 then TopItem := Item - NumRows + 1
      else TopItem := Item - Item mod NumRows - (NumRows*(NumCols - 1));

      if (SaveFocused <> Focused) and (SaveTopItem = TopItem) then
      begin
         GetItemRect(SaveFocused, R); DrawItem(R, SaveFocused, Indent);
         GetItemRect(Focused, R); DrawItem(R, Focused, Indent);
         DrawCursor;
      end else DrawView;
   end;
end;

procedure TListViewer.FocusItemNum(Item:Integer);
begin
   if Item < 0 then Item := 0
   else if (Item >= Range) and (Range > 0) then Item := Range-1;
   if Range <> 0 then FocusItem(Item);
end;

procedure TListViewer.GetItemRect(Item :Integer;
                                  var R:TGRect);
var
   W, H, X         :Integer;
   Extent          :TGRect;
begin
   W := (Size.X - (FrameOffset shl 1)) div NumCols; H := CharHeight;
   Dec(Item, TopItem);
   X:=Item div NumRows;
   R.A.X := W * X + FrameOffset;
   if X < (NumCols - 1) then R.B.X := R.A.X + W
   else R.B.X:=Size.X - FrameOffset - 1;
   R.A.Y := H * (Item mod NumRows); R.B.Y := R.A.Y + H - 1;
   GetExtent(Extent); R.Intersect(Extent);
end;

function TListViewer.GetText(Item      :Integer;
                             MaxLen    :Integer) :String;
begin
   Abstract;
end;

function TListViewer.IsSelected(Item   :Integer) :Boolean;
begin
   IsSelected := Item = Focused;
end;

procedure TListViewer.HandleEvent(var Event      :TEvent);
var
   H, ColWidth,
   OldItem, NewItem:Integer;
   Mouse           :TGPoint;

   procedure UpdateItems;
   var
      OldTopItem   :Integer;
   begin
      if (NewItem >= Range) and (Range > 0) then NewItem := Range-1;
      if NewItem < 0 then NewItem := 0;
      if NewItem >= Range then NewItem := OldItem;
      if OldItem <> NewItem then FocusItem(NewItem);
   end;

begin
   Inherited HandleEvent(Event);
   H := CharHeight;
   OldItem := Focused;
   if Event.What = evMouseDown then
   begin
      ColWidth := Size.X div NumCols + 1;
      MakeLocal(Event.Where, Mouse);
      NewItem := Mouse.Y div H + (NumRows * (Mouse.X div ColWidth)) + TopItem;
      repeat
         UpdateItems;
         OldItem := NewItem;
         MakeLocal(Event.Where, Mouse);
         if MouseInView(Event.Where)
         then NewItem := Mouse.Y div H + (NumRows * (Mouse.X div ColWidth)) + TopItem
         else
         begin
            if NumCols = 1 then
            begin
               if Mouse.Y < 0 then NewItem := Focused-1 else
               if Mouse.Y >= Size.Y then NewItem := Focused+1;
            end
            else
            begin
               if Mouse.X < 0 then NewItem := Focused-NumRows else
               if Mouse.X >= Size.X then NewItem := Focused+NumRows else
               if Mouse.Y < 0 then NewItem := Focused - Focused mod NumRows else
               if Mouse.Y > Size.Y then NewItem := Focused - Focused mod NumRows + NumRows - 1;
            end;
         end;
      until not MouseEvent(Event, evMouseMove + evMouseAuto);
      UpdateItems;
      if Event.Double and (Range > Focused) then SelectItem(Focused);
      ClearEvent(Event);
   end
   else if Event.What = evKeyDown then
   begin
      if (Event.CharCode = ' ') and (Focused < Range) then
      begin
         SelectItem(Focused); NewItem := Focused;
      end
      else case CtrlToArrow(Event.KeyCode) of
         kbUp:     NewItem := Focused - 1;
         kbDown:   NewItem := Focused + 1;
         kbRight:  if NumCols > 1 then NewItem := Focused + NumRows else Exit;
         kbLeft:   if NumCols > 1 then NewItem := Focused - NumRows else Exit;
         kbPgDn:   NewItem := Focused + NumRows * NumCols;
         kbPgUp:   NewItem := Focused - NumRows * NumCols;
         kbHome:   NewItem := TopItem;
         kbEnd:    NewItem := TopItem + (NumRows * NumCols) - 1;
         kbCtrlPgDn: NewItem := Range - 1;
         kbCtrlPgUp: NewItem := 0;
      else
         Exit;
      end;
      UpdateItems;
      ClearEvent(Event);
   end
   else
   if (Event.What = evBroadcast) then
   if Event.Command = cmScrollBarChanged then
   begin
      if (VScrollBar = Event.InfoPtr) then
      begin
         NewItem := VScrollBar^.Value;
         UpdateItems;
      end
      else if (HScrollBar = Event.InfoPtr) then DrawView;
   end;
end;

procedure TListViewer.SelectItem(Item  :Integer);
begin
   Message(Owner, evBroadcast, cmListItemSelected, @Self);
end;

procedure TListViewer.SetRange(ARange  :Integer);
begin
   Range := ARange;
   if VScrollBar <> nil then
   begin
      if Focused > ARange then Focused := 0;
      VScrollbar^.SetParams(Focused, 0, ARange-1, VScrollBar^.PgStep,
      VScrollBar^.ArStep);
   end;
end;

procedure TListViewer.SetState(AState  :Word;
                               Enable  :Boolean);

procedure ShowSBar(SBar                :PScrollBar);
begin
   if (SBar <> nil) then
   if GetState(sfActive) and GetState(sfVisible) then SBar^.Show
   else SBar^.Hide;
end;

begin
   Inherited SetState(AState, Enable);
   if AState and (sfSelected + sfActive + sfVisible) <> 0 then
   begin
      ShowSBar(HScrollBar); ShowSBar(VScrollBar);
      DrawView;
   end;
end;

procedure TListViewer.Store(var S      :TStream);
begin
   Inherited Store(S);
   PutPeerViewPtr(S, HScrollBar); PutPeerViewPtr(S, VScrollBar);
   S.Write(NumCols, SizeOf(Word) * 5);
end;

{ TGroup }

procedure TGroup.LockDraw;

procedure DoLockDraw(P                 :PView); far;
begin
   P^.LockDraw;
end;

begin
   Inherited LockDraw;
   ForEach(@DoLockDraw);
end;

procedure TGroup.UnlockDraw;

procedure DoUnlockDraw(P               :PView); far;
begin
   P^.UnlockDraw;
end;

begin
   Inherited UnlockDraw;
   ForEach(@DoUnlockDraw);
end;

constructor TGroup.Init(var Bounds     :TGRect);
begin
   Inherited Init(Bounds);
   Options := Options or (ofSelectable + ofClipping);
   GetExtent(Clip);
   EventMask := $FFFF;
   IsGroup:=True;
end;

constructor TGroup.Load(var S          :TStream);
var
   FixupSave       :PFixupList;
   Count, I        :Integer;
   P, Q            :^Pointer;
   V               :PView;
   OwnerSave       :PGroup;
begin
   Inherited Load(S);
   GetExtent(Clip);
   OwnerSave := OwnerGroup; OwnerGroup := @Self;
   FixupSave := FixupList;
   S.Read(Count, SizeOf(Word));
   asm
        MOV     CX,Count
        SHL     CX,1
        SHL     CX,1
        SUB     SP,CX
        MOV     FixupList.Word[0],SP
        MOV     FixupList.Word[2],SS
        MOV     DI,SP
        PUSH    SS
        POP     ES
        XOR     AL,AL
        CLD
        REP     STOSB
   end;
   for I := 1 to Count do
   begin
      V := PView(S.Get);
      if V <> nil then InsertView(V, nil);
   end;
   V := Last;
   for I := 1 to Count do
   begin
      V := V^.Next; P := FixupList^[I];
      while P <> nil do
      begin
         Q := P; P := P^; Q^ := V;
      end;
   end;
   OwnerGroup := OwnerSave;
   FixupList := FixupSave;
   GetSubViewPtr(S, V);
   SetCurrent(V, NormalSelect);
   if OwnerGroup = nil then Awaken;
end;

destructor TGroup.Done;
var
   P, T            :PView;
begin
   Hide;
   P := Last;
   if P <> nil then
   begin
      repeat
         P^.Hide; P := P^.Prev;
      until P = Last;
      repeat
         T := P^.Prev; Dispose(P, Done); P := T;
      until Last = nil;
   end;
   Inherited Done;
end;

function TGroup.At(Index               :Integer) :PView; assembler;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     CX,Index
@@1:    LES     DI,ES:[DI].TView.Next
        LOOP    @@1
        MOV     AX,DI
        MOV     DX,ES
end;

procedure TGroup.Awaken;

   procedure DoAwaken(P                :PView); far;
   begin
      P^.Awaken;
   end;

begin
   ForEach(@DoAwaken);
end;

procedure TGroup.ChangeBounds(var Bounds:TGRect);
var
   D               :TGPoint;

procedure DoCalcChange(P               :PView); far;
var
   R               :TGRect;
begin
   P^.CalcBounds(R, D); P^.ChangeBounds(R);
end;

begin
   D.X := Bounds.B.X - Bounds.A.X - Size.X +1;
   D.Y := Bounds.B.Y - Bounds.A.Y - Size.Y +1;
   SetBounds(Bounds);
   if LongInt(D) = 0 then DrawView else
   begin
      GetExtent(Clip);
      Lock; ForEach(@DoCalcChange); Unlock;
   end;
end;

function TGroup.DataSize               :Word;
var
   T               :Word;

procedure AddSubviewDataSize(P         :PView); far;
begin
   Inc(T, P^.DataSize);
end;

begin
   T := 0;
   ForEach(@AddSubviewDataSize);
   DataSize := T;
end;

procedure TGroup.Delete(P              :PView);
var
   SaveState       :Word;
begin
   SaveState := P^.State;
   P^.Hide;
   RemoveView(P);
   P^.Owner := nil; P^.Next := nil;
   if SaveState and sfVisible <> 0 then P^.Show;
end;

procedure TGroup.Draw;
begin
   GetClipRect(Clip); Redraw; GetExtent(Clip);
end;

procedure TGroup.DrawSubViews(P, Bottom:PView);
begin
   if P <> nil then
   while P <> Bottom do
   begin
      P^.DrawView; P := P^.NextView;
   end;
end;

procedure TGroup.EndModal(Command      :Word);
begin
   if State and sfModal <> 0 then EndState := Command
   else Inherited EndModal(Command);
end;

procedure TGroup.EventError(var Event  :TEvent);
begin
   if Owner <> nil then Owner^.EventError(Event);
end;

function TGroup.Execute                :Word;
var
   E               :TEvent;
begin
   repeat
      EndState := 0;
      repeat
         GetEvent(E); HandleEvent(E);
         if E.What <> evNothing then EventError(E);
      until EndState <> 0;
   until Valid(EndState);
   Execute := EndState;
end;

function TGroup.ExecView(P             :PView)   :Word;
var
   SaveOptions     :Word;
   SaveOwner       :PGroup;
   SaveTopView     :PView;
   SaveCurrent     :PView;
   SaveCommands    :TCommandSet;
begin
   if P <> nil then
   begin
      SaveOptions := P^.Options; SaveOwner := P^.Owner;
      SaveTopView := TheTopView; SaveCurrent := Current;
      GetCommands(SaveCommands);
      TheTopView := P;
      P^.Options := P^.Options and not ofSelectable;
      P^.SetState(sfModal, True);
      SetCurrent(P, EnterSelect);
      if SaveOwner = nil then Insert(P);
      ExecView := P^.Execute;
      if SaveOwner = nil then Delete(P);
      SetCurrent(SaveCurrent, LeaveSelect);
      P^.SetState(sfModal, False);
      P^.Options := SaveOptions;
      TheTopView := SaveTopView;
      SetCommands(SaveCommands);
   end else ExecView := cmCancel;
end;

function TGroup.First                  :PView;
begin
   if Last = nil then First := nil else First := Last^.Next;
end;

function TGroup.FirstMatch(AState      :Word;
                           AOptions    :Word)    :PView;

function Matches(P                     :PView)   :Boolean; far;
begin
   Matches := (P^.State and AState = AState) and
      (P^.Options and AOptions = AOptions);
end;

begin
   FirstMatch := FirstThat(@Matches);
end;

function TGroup.FirstThat(P            :Pointer) :PView; assembler;
var
   ALast           :Pointer;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@3
        MOV     WORD PTR ALast[2],ES
        MOV     WORD PTR ALast[0],DI
@@1:    LES     DI,ES:[DI].TView.Next
        PUSH    ES
        PUSH    DI
        PUSH    ES
        PUSH    DI
        PUSH    WORD PTR [BP]
        CALL    P
        POP     DI
        POP     ES
        OR      AL,AL
        JNE     @@2
        CMP     DI,WORD PTR ALast[0]
        JNE     @@1
        MOV     AX,ES
        CMP     AX,WORD PTR ALast[2]
        JNE     @@1
        XOR     DI,DI
        MOV     ES,DI
@@2:    MOV     SP,BP
@@3:    MOV     AX,DI
        MOV     DX,ES
end;

function TGroup.FindNext(Forwards      :Boolean) :PView;
var
   P               :PView;
begin
   FindNext := nil;
   if Current <> nil then
   begin
      P := Current;
      repeat
         if Forwards then P := P^.Next else P := P^.Prev;
      until ((P^.State and (sfVisible + sfDisabled) = sfVisible) and
            (P^.Options and ofSelectable <> 0)) or (P = Current);
      if P <> Current then FindNext := P;
   end;
end;

function TGroup.FocusNext(Forwards     :Boolean) :Boolean;
var
   P               :PView;
begin
   P := FindNext(Forwards);
   FocusNext := True;
   if P <> nil then FocusNext := P^.Focus;
end;

procedure TGroup.ForEach(P             :Pointer); assembler;
var
   ALast           :Pointer;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@4
        MOV     WORD PTR ALast[2],ES
        MOV     WORD PTR ALast[0],DI
        LES     DI,ES:[DI].TView.Next
@@1:    CMP     DI,WORD PTR ALast[0]
        JNE     @@2
        MOV     AX,ES
        CMP     AX,WORD PTR ALast[2]
        JE      @@3
@@2:    PUSH    WORD PTR ES:[DI].TView.Next[2]
        PUSH    WORD PTR ES:[DI].TView.Next[0]
        PUSH    ES
        PUSH    DI
        PUSH    WORD PTR [BP]
        CALL    P
        POP     DI
        POP     ES
        JMP     @@1
@@3:    PUSH    WORD PTR [BP]
        CALL    P
@@4:
end;

procedure TGroup.GetData(var Rec);
type
   Bytes =         array[0..65534] of Byte;
var
   I               :Word;
   V               :PView;
begin
   I := 0;
   if Last <> nil then
   begin
      V := Last;
      repeat
         V^.GetData(Bytes(Rec)[I]);
         Inc(I, V^.DataSize);
         V := V^.Prev;
      until V = Last;
   end;
end;

function TGroup.GetHelpCtx             :Word;
var
   H               :Word;
begin
   H:= hcNoContext;
   if Current <> nil then H := Current^.GetHelpCtx;
   if H = hcNoContext then H := Inherited GetHelpCtx;
   GetHelpCtx := H;
end;

procedure TGroup.GetSubViewPtr(var S   :TStream; var P);
var
   Index           :Word;
begin
   S.Read(Index, SizeOf(Word));
   if Index > 0 then Pointer(P) := At(Index) else Pointer(P) := nil;
end;

procedure TGroup.HandleEvent(var Event :TEvent);

procedure DoHandleEvent(P              :PView); far;
begin
   if (P = nil) or ((P^.State and sfDisabled <> 0) and
      (Event.What and (PositionalEvents or FocusedEvents) <> 0)) then Exit;
   case Phase of
      phPreProcess:  if P^.Options and ofPreProcess = 0 then Exit;
      phPostProcess: if P^.Options and ofPostProcess = 0 then Exit;
   end;
   if Event.What and P^.EventMask <> 0 then P^.HandleEvent(Event);
end;

function  ContainsMouse(P              :PView)   :Boolean; far;
begin
   ContainsMouse := (P^.State and sfVisible <> 0) and
      P^.MouseInView(Event.Where);
end;

begin
   Inherited HandleEvent(Event);
   if Event.What and FocusedEvents <> 0 then
   begin
      Phase := phPreProcess;
      ForEach(@DoHandleEvent);
      Phase := phFocused;
      DoHandleEvent(Current);
      Phase := phPostProcess;
      ForEach(@DoHandleEvent);
   end else
   begin
      Phase := phFocused;
      if Event.What and PositionalEvents <> 0 then
         DoHandleEvent(FirstThat(@ContainsMouse)) else ForEach(@DoHandleEvent);
   end;
end;

function TGroup.IndexOf(P              :PView)   :Integer; assembler;
asm
        LES     DI,Self
        LES     DI,ES:[DI].TGroup.Last
        MOV     AX,ES
        OR      AX,DI
        JE      @@3
        MOV     CX,DI
        MOV     BX,ES
        XOR     AX,AX
@@1:    INC     AX
        LES     DI,ES:[DI].TView.Next
        MOV     DX,ES
        CMP     DI,P.Word[0]
        JNE     @@2
        CMP     DX,P.Word[2]
        JE      @@3
@@2:    CMP     DI,CX
        JNE     @@1
        CMP     DX,BX
        JNE     @@1
        XOR     AX,AX
@@3:
end;

procedure TGroup.Insert(P              :PView);
begin
   InsertBefore(P, First);
end;

procedure TGroup.InsertBefore(P, Target:PView);
var
   SaveState       :Word;
begin
   if (P <> nil) and (P^.Owner = nil) and
      ((Target = nil) or (Target^.Owner = @Self)) then
   begin
      if P^.Options and ofCenterX <> 0 then
         P^.Origin.X := (Size.X - P^.Size.X) div 2;
      if P^.Options and ofCenterY <> 0 then
         P^.Origin.Y := (Size.Y - P^.Size.Y) div 2;
      SaveState := P^.State;
      P^.Hide;
      InsertView(P, Target);
      if SaveState and sfVisible <> 0 then P^.Show;
      if State and sfActive <> 0 then P^.SetState(sfActive, True);
   end;
end;

procedure TGroup.InsertView(P, Target  :PView);
begin
   P^.Owner := @Self;
   if Target <> nil then
   begin
      Target := Target^.Prev;
      P^.Next := Target^.Next;
      Target^.Next := P;
   end else
   begin
      if Last = nil then P^.Next := P else
      begin
         P^.Next := Last^.Next; Last^.Next := P;
      end;
      Last := P;
   end;
end;

procedure TGroup.Lock;
begin
   if Options and ofBuffered<>0 then LockScreen;
end;

procedure TGroup.Unlock;
begin
   if Options and ofBuffered<>0 then UnlockScreen;
end;

procedure TGroup.PutSubViewPtr(var S   :TStream;
                               P       :PView);
var
   Index           :Word;
begin
   if P = nil then Index := 0 else Index := IndexOf(P);
   S.Write(Index, SizeOf(Word));
end;

procedure TGroup.Redraw;
var
   P               :PView;
   SaveOptions     :Word;
begin
   InitClipping;
   P := Last;
   SaveOptions := P^.Options;
   P^.Options := P^.Options and not ofClipping;
   if MouseSafe then HideMouse;
   Lock; repeat P^.DrawView; P := P^.Next; until P = Last; Unlock;
   if MouseSafe then ShowMouse;
   P^.Options := SaveOptions;
end;

procedure TGroup.RemoveView(P          :PView); assembler;
asm
        PUSH    DS
        LDS     SI,Self
        LES     DI,P
        LDS     SI,DS:[SI].TGroup.Last
        PUSH    BP
        MOV     AX,DS
        OR      AX,SI
        JE      @@7
        MOV     AX,SI
        MOV     DX,DS
        MOV     BP,ES
@@1:    MOV     BX,WORD PTR DS:[SI].TView.Next[0]
        MOV     CX,WORD PTR DS:[SI].TView.Next[2]
        CMP     CX,BP
        JE      @@5
@@2:    CMP     CX,DX
        JE      @@4
@@3:    MOV     SI,BX
        MOV     DS,CX
        JMP     @@1
@@4:    CMP     BX,AX
        JNE     @@3
        JMP     @@7
@@5:    CMP     BX,DI
        JNE     @@2
        MOV     BX,WORD PTR ES:[DI].TView.Next[0]
        MOV     CX,WORD PTR ES:[DI].TView.Next[2]
        MOV     DS:WORD PTR [SI].TView.Next[0],BX
        MOV     DS:WORD PTR [SI].TView.Next[2],CX
        CMP     DX,BP
        JNE     @@7
        CMP     AX,DI
        JNE     @@7
        CMP     CX,BP
        JNE     @@6
        CMP     BX,DI
        JNE     @@6
        XOR     SI,SI
        MOV     DS,SI
@@6:    POP     BP
        PUSH    BP
        LES     DI,Self
        MOV     WORD PTR ES:[DI].TView.Last[0],SI
        MOV     WORD PTR ES:[DI].TView.Last[2],DS
@@7:    POP     BP
        POP     DS
end;

procedure TGroup.ResetCurrent;
begin
   SetCurrent(FirstMatch(sfVisible, ofSelectable), NormalSelect);
end;

procedure TGroup.ResetCursor;
begin
   if Current <> nil then Current^.ResetCursor;
end;

procedure TGroup.SelectNext(Forwards   :Boolean);
var
   P               :PView;
begin
   P := FindNext(Forwards); if P <> nil then P^.Select;
end;

procedure TGroup.SetCurrent(P          :PView;
                            Mode       :SelectMode);

procedure SelectView(P                 :PView;
                     Enable            :Boolean);
begin
   if P <> nil then P^.SetState(sfSelected, Enable);
end;

procedure FocusView(P                  :PView;
                    Enable             :Boolean);
begin
   if (State and sfFocused <> 0) and (P <> nil) then
      P^.SetState(sfFocused, Enable);
end;

begin
   if Current <> P then
   begin
      FocusView(Current, False);
      if Mode <> EnterSelect then SelectView(Current, False);
      if Mode <> LeaveSelect then SelectView(P, True);
      FocusView(P, True);
      Current := P;
   end;
end;

procedure TGroup.SetData(var Rec);
type
   Bytes = array[0..65534] of Byte;
var
   I               :Word;
   V               :PView;
begin
   I := 0;
   if Last <> nil then
   begin
      V := Last;
      repeat
         V^.SetData(Bytes(Rec)[I]);
         Inc(I, V^.DataSize);
         V := V^.Prev;
      until V = Last;
   end;
end;

procedure TGroup.SetState(AState       :Word;
                          Enable       :Boolean);

procedure DoSetState(P                 :PView); far;
begin
   P^.SetState(AState, Enable);
end;

procedure DoExpose(P                   :PView); far;
begin
   if P^.State and sfVisible <> 0 then P^.SetState(sfExposed, Enable);
end;

begin
   Inherited SetState(AState, Enable);
   case AState of
      sfActive, sfDragging:
         ForEach(@DoSetState);
      sfFocused:
         if Current <> nil then Current^.SetState(sfFocused, Enable);
      sfExposed:
         ForEach(@DoExpose);
   end;
end;

procedure TGroup.Store(var S           :TStream);
var
   Count           :Integer;
   OwnerSave       :PGroup;

procedure DoPut(P  :PView); far;
begin
   S.Put(P);
end;

begin
   Inherited Store(S);
   OwnerSave := OwnerGroup;
   OwnerGroup := @Self;
   Count := IndexOf(Last);
   S.Write(Count, SizeOf(Word));
   ForEach(@DoPut);
   PutSubViewPtr(S, Current);
   OwnerGroup := OwnerSave;
end;

function TGroup.Valid(Command          :Word)    :Boolean;

function IsInvalid(P                   :PView)   :Boolean; far;
begin
   IsInvalid := not P^.Valid(Command);
end;

begin
   Valid := True;
   if Command = cmReleasedFocus then
   begin
      if (Current <> nil) and (Current^.Options and ofValidate <> 0) then
         Valid := Current^.Valid(Command);
   end
   else
      Valid := FirstThat(@IsInvalid) = nil;
end;

{ TPanel }

constructor TPanel.Init;
begin
   Inherited Init(Bounds);
   Color:=AColor; Flags:=AFlags;
end;

procedure TPanel.Draw;
var
   R               :TGRect;
begin
   GetExtent(R);
   DrawButton(R, Flags {or dbFillIn}, Flags and dbInset<>0, Color);
   Inherited Draw;
end;

{ TWindow }

constructor TWindow.Init(Bounds        :TGRect;
                         ATitle        :TTitleStr;
                         ANumber       :Integer);
begin
   if Bounds.SizeX<MinWinSize.X then Bounds.B.X:=Bounds.A.X+MinWinSize.X-1;
   if Bounds.SizeY<MinWinSize.Y then Bounds.B.Y:=Bounds.A.Y+MinWinSize.Y-1;

   Inherited Init(Bounds);
   Options := Options or (ofSelectable + ofTopSelect + ofTileable);
   GrowMode := gfGrowAll + gfGrowRel;
   Flags := wfMove + wfGrow + wfClose + wfZoom + wfMenu;
   Title := NewStr(ATitle);
   Number := ANumber;
   Palette := wpBlueWindow;
   InitFrame;
   if Frame <> nil then Insert(Frame);
   GetBounds(ZoomRect);
end;

constructor TWindow.Load(var S         :TStream);
begin
   Inherited Load(S);
   S.Read(Flags, SizeOf(Byte) + SizeOf(TGRect) + 2 * SizeOf(Integer));
   GetSubViewPtr(S, Frame);
   Title := S.ReadStr;
end;

destructor TWindow.Done;
begin
   Inherited Done;
   DisposeStr(Title);
end;

function  TWindow.GetPalette;
const
   C     :Array [wpBlueWindow..wpWhiteWindow] of TPalette=
   (CBlueWindow, CCyanWindow, CGrayWindow, CWhiteWindow);
begin
   GetPalette:=C[Palette];
end;

procedure TWindow.GetInteriorSB;
begin
   GetExtent(I);
   if Frame<>nil then
   with I, Frame^ do
   begin
      Inc(A.X, GetPartSize(fpLeft)); Dec(B.X, GetPartSize(fpRight));
      Inc(A.Y, GetPartSize(fpTop));  Dec(B.Y, GetPartSize(fpBottom));
   end;
end;

procedure TWindow.GetInterior;
begin
   GetInteriorSB(I);
   if Flags and wfStdVertSB <> 0 then Dec(I.B.X, SBarSize.X);
   if Flags and wfStdHorizSB <> 0 then I.B.Y:=Min(I.B.Y, Size.Y-SBarSize.Y-2);
end;

function  TWindow.GetMenu;
begin
   GetMenu :=
     NewItem(StrPas(WMenuItems^[0]), 'Alt+F3', kbAltF3, cmClose, hcClose,
     NewItem(StrPas(WMenuItems^[1]), 'Ctrl+F5', kbCtrlF5, cmResize, hcResize,
     NewItem(StrPas(WMenuItems^[3]), 'F5', kbF5, cmZoom, hcZoom, NextItem)));
end;

procedure TWindow.Close;
begin
   if Valid(cmClose) then Free;
end;

function TWindow.GetTitle(MaxSize      :Integer) :TTitleStr;
begin
   if (Title <> nil) and (Byte(Title^[0])<=MaxSize) then GetTitle := Title^
   else GetTitle := '';
end;

procedure TWindow.HandleEvent(var Event:TEvent);
var
   Limits          :TGRect;
   Min, Max        :TGPoint;
begin
   Inherited HandleEvent(Event);
   if (Event.What = evCommand) then
   case Event.Command of
      cmResize:
      if Flags and (wfMove + wfGrow) <> 0 then
      begin
         Owner^.GetExtent(Limits);
         SizeLimits(Min, Max);
         DragView(Event, DragMode or (Flags and (wfMove + wfGrow)),
            Limits, Min, Max);
         ClearEvent(Event);
      end;
      cmClose:
      if (Flags and wfClose <> 0) and
         ((Event.InfoPtr = nil) or (Event.InfoPtr = @Self)) then
      begin
         ClearEvent(Event);
         if State and sfModal = 0 then Close else
         begin
            Event.What := evCommand; Event.Command := cmCancel;
            PutEvent(Event); ClearEvent(Event);
         end;
      end;
      cmZoom:
      if (Flags and wfZoom <> 0) and
         ((Event.InfoPtr = nil) or (Event.InfoPtr = @Self)) then
      begin
         Zoom; ClearEvent(Event);
      end;
   end
   else if Event.What = evKeyDown then
   case Event.KeyCode of
      kbTab:
      begin
         FocusNext(False); ClearEvent(Event);
      end;
      kbShiftTab:
      begin
         FocusNext(True); ClearEvent(Event);
      end;
      else
         if (GetAltChar(Event.KeyCode) = #240) and (Flags and wfMenu<>0) then
         if Message(Frame, evBroadcast, cmWindowMenu, nil) <> nil then
            ClearEvent(Event);
   end
   else
   if (Event.What = evBroadcast) and (Event.Command = cmSelectWindowNum) and
      (Event.InfoInt = Number) and (Options and ofSelectable <> 0) then
   begin
      Select; ClearEvent(Event);
   end;
end;

procedure TWindow.InitFrame;
var
   R               :TGRect;
begin
   GetExtent(R); Frame := New(PFrame, Init(R));
end;

procedure TWindow.SetState(AState      :Word;
                           Enable      :Boolean);
var
   WindowCommands  :TCommandSet;
begin
   Inherited SetState(AState, Enable);
   if AState = sfSelected then SetState(sfActive, Enable);
   if (AState = sfSelected) or ((AState = sfExposed) and
      (State and sfSelected <> 0)) then
   begin
      WindowCommands := [cmNext, cmPrev];
      if Flags and (wfGrow + wfMove) <> 0 then
         WindowCommands := WindowCommands + [cmResize];
      if Flags and wfClose <> 0 then
         WindowCommands := WindowCommands + [cmClose];
      if Flags and wfZoom <> 0 then
         WindowCommands := WindowCommands + [cmZoom];
      if Enable then EnableCommands(WindowCommands)
      else DisableCommands(WindowCommands);
   end;
end;

function TWindow.StandardScrollBar(AOptions      :Word)    :PScrollBar;
var
   R               :TGRect;
   S               :PScrollBar;
begin
   GetInteriorSB(R);
   if AOptions and sbVertical <> 0 then
   begin
      R.A.X:=R.B.X-SBarSize.X+1; R.B.X:=R.A.X;
      R.B.Y:=Min(R.B.Y, Size.Y-SBarSize.Y-2);
      Flags := Flags or wfStdVertSB;
   end
   else
   begin
      if Flags and wfGrow <> 0
      then Dec(R.B.X, Max(SBarSize.X, CharWidth)) else Dec(R.B.X, SBarSize.X);
      R.A.Y:=Size.Y-SBarSize.Y-1; R.B.Y:=R.A.Y;
      Flags := Flags or wfStdHorizSB;
   end;
   New(S, Init(R, AOptions and sbVertical<>0));
   Insert(S);
   if AOptions and sbHandleKeyboard <> 0 then
      S^.Options := S^.Options or ofPostProcess;
   StandardScrollBar := S;
end;

procedure TWindow.SizeLimits(var Min, Max        :TGPoint);
begin
   Inherited SizeLimits(Min, Max);
   Min := MinWinSize;
end;

procedure TWindow.Store(var S          :TStream);
begin
   Inherited Store(S);
   S.Write(Flags, SizeOf(Byte) + SizeOf(TGRect) + 2 * SizeOf(Integer));
   PutSubViewPtr(S, Frame);
   S.WriteStr(Title);
end;

procedure TWindow.Zoom;
var
   R               :TGRect;
   Max, Min        :TGPoint;
begin
   SizeLimits(Min, Max);
   if LongInt(Size) <> LongInt(Max) then
   begin
      GetBounds(ZoomRect);
      LongInt(R.A) := 0; R.B := Max;
      Locate(R);
   end else Locate(ZoomRect);
end;

{ Message dispatch function }

function Message(Receiver              :PView;
                 What, Command         :Word;
                 InfoPtr               :Pointer) :Pointer;
var
   Event           :TEvent;
begin
   Message := nil;
   if Receiver <> nil then
   begin
      Event.What := What; Event.Command := Command; Event.InfoPtr := InfoPtr;
      Receiver^.HandleEvent(Event);
      if Event.What = evNothing then Message := Event.InfoPtr;
   end;
end;

{ Views registration procedure }

procedure RegisterViews;
begin
   RegisterType(RView);
   RegisterType(RFrame);
   RegisterType(RScrollBar);
   RegisterType(RScroller);
   RegisterType(RListViewer);
   RegisterType(RGroup);
   RegisterType(RWindow);
end;

begin
   SystemFont.Defaults;
end.
