
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1994,95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit GMenus;

{$X+,I-,S-}

interface
uses
   GraphDrv,
   Objects, GViews,
   GDrivers, Events, KeyMouse, Utils,
   GRect, GPalette, Language;

const
   CWhiteMenu =
   cpMain+cpWhite + cpText+cpBlack + cpFrame+cpBlack +
   cpDisabled+cpLightGray + cpShortcut+cpLightRed +
   cpSelected+cpWhite + cpSelDisabled+cpLightGray + cpSelectedBk+cpBlue;

   CGrayMenu =
   cpMain+cpLightGray + cpText+cpBlack + cpFrame+cpBlack +
   cpDisabled+cpDarkGray + cpShortcut+cpLightRed +
   cpSelected+cpBlack + cpSelDisabled+cpDarkGray + cpSelectedBk+cpGreen;

   CMenu =         CWhiteMenu;

   mpWhiteMenu =   0;
   mpGrayMenu =    1;

type

{ TMenu types }

   TMenuStr =      String[127];
   THintStr =      String[127];

   PMenu =         ^TMenu;

   PMenuItem =     ^TMenuItem;
   TMenuItem =
   record
      Next         :PMenuItem;
      Name         :PString;
      Command      :Word;
      Disabled     :Boolean;
      KeyCode      :Word;
      HelpCtx      :Word;
      case Integer of
         0: (Param      :PString);
         1: (SubMenu    :PMenu);
   end;

   TMenu =
   record
      Items        :PMenuItem;
      Default      :PMenuItem;
   end;

   TMenuClass =    (mcOther, mcBar, mcBox);

   TMenuColorsView =
   object(TView)
      Palette      :Integer;

      constructor Load(var S           :TStream);

      procedure Store(var S            :TStream);

      function  GetPalette                                 :TPalette; virtual;
   private
      procedure GetMenuColors(IsCurrent, Disabled:Boolean;
                              var Color, CBack   :Word);
   end;

   PMenuView =     ^TMenuView;
   TMenuView =
   object(TMenuColorsView)
      ParentMenu   :PMenuView;
      Menu         :PMenu;
      Current      :PMenuItem;
      Class        :TMenuClass;

      constructor Init(var Bounds      :TGRect);
      constructor Load(var S           :TStream);

      procedure DrawView;

      function  Execute                                    :Word; virtual;
      function  FindItem(Ch            :Char)              :PMenuItem;
      procedure GetItemRect(Item       :PMenuItem;
                            var R      :TGRect); virtual;
      function  GetHelpCtx                                 :Word; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      function  HotKey(KeyCode         :Word)              :PMenuItem;
      function  NewSubView(var Bounds  :TGRect;
                           AMenu       :PMenu;
                           AParentMenu :PMenuView)         :PMenuView; virtual;
      procedure Store(var S            :TStream);

   private
      Last         :PMenuItem;
      SkipRedraw   :Boolean;
   end;

   PMenuBar =      ^TMenuBar;
   TMenuBar =
   object(TMenuView)
      constructor Init(var Bounds      :TGRect;
                       AMenu           :PMenu);
      destructor  Done; virtual;

      constructor Standard(AMenu       :PMenu);

      procedure Draw; virtual;
      procedure GetItemRect(Item       :PMenuItem;
                            var R      :TGRect); virtual;
   end;

   PMenuBox =      ^TMenuBox;
   TMenuBox =
   object(TMenuView)
      constructor Init(var Bounds      :TGRect;
                       AMenu           :PMenu;
                       AParentMenu     :PMenuView);
      procedure Draw; virtual;
      procedure GetItemRect(Item       :PMenuItem;
                            var R      :TGRect); virtual;
   end;

   PMenuPopup =    ^TMenuPopup;
   TMenuPopup =
   object(TMenuBox)
      constructor Init(var Bounds      :TGRect;
                       AMenu           :PMenu);
      destructor  Done; virtual;

      procedure HandleEvent(var Event  :TEvent); virtual;
   end;

   PStatusItem =   ^TStatusItem;
   TStatusItem =
   record
      Next         :PStatusItem;
      Text         :PString;
      KeyCode      :Word;
      Command      :Word;
   end;

   PStatusDef =    ^TStatusDef;
   TStatusDef =
   record
      Next         :PStatusDef;
      Min, Max     :Word;
      Items        :PStatusItem;
   end;

   PStatusLine = ^TStatusLine;
   TStatusLine = object(TMenuColorsView)
      Items        :PStatusItem;
      Defs         :PStatusDef;

      constructor Init(var Bounds      :TGRect;
                       ADefs           :PStatusDef);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      constructor Standard(ADefs       :PStatusDef);

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      function  Hint(AHelpCtx          :Word)    :THintStr; virtual;
      procedure Store(var S            :TStream);
      procedure Update; virtual;
   private
      procedure DrawSelect(Selected    :PStatusItem);
      procedure FindItems;
   end;

{ TMenuItem routines }

function  NewItem(Name, Param          :TMenuStr;
                 KeyCode               :Word;
                 Command               :Word;
                 AHelpCtx              :Word;
                 Next                  :PMenuItem)         :PMenuItem;

function  NewLine(Next                 :PMenuItem)         :PMenuItem;

function  NewSubMenu(Name              :TMenuStr;
                     AHelpCtx          :Word;
                     SubMenu           :PMenu;
                     Next              :PMenuItem)         :PMenuItem;

{ TMenu routines }

function  NewMenu(Items                :PMenuItem)         :PMenu;
procedure DisposeMenu(Menu             :PMenu);

{ TStatusLine routines }

function  NewStatusDef(AMin, AMax      :Word;
                       AItems          :PStatusItem;
                       ANext           :PStatusDef)        :PStatusDef;

function  NewStatusKey(Const AText     :String;
                       AKeyCode        :Word;
                       ACommand        :Word;
                       ANext           :PStatusItem)       :PStatusItem;

{ Menus registration procedure }

procedure RegisterMenus;

{ Stream registration records }

const
  RMenuBar: TStreamRec = (
     ObjType: 40;
     VmtLink: Ofs(TypeOf(TMenuBar)^);
     Load:    @TMenuBar.Load;
     Store:   @TMenuBar.Store
  );

  RMenuBox: TStreamRec = (
     ObjType: 41;
     VmtLink: Ofs(TypeOf(TMenuBox)^);
     Load:    @TMenuBox.Load;
     Store:   @TMenuBox.Store
  );

  RStatusLine: TStreamRec = (
     ObjType: 42;
     VmtLink: Ofs(TypeOf(TStatusLine)^);
     Load:    @TStatusLine.Load;
     Store:   @TStatusLine.Store
  );

  RMenuPopup: TStreamRec = (
     ObjType: 43;
     VmtLink: Ofs(TypeOf(TMenuPopup)^);
     Load:    @TMenuPopup.Load;
     Store:   @TMenuPopup.Store
  );

implementation

{ TMenuItem routines }

function  NewItem(Name, Param          :TMenuStr;
                  KeyCode              :Word;
                  Command              :Word;
                  AHelpCtx             :Word;
                  Next                 :PMenuItem)         :PMenuItem;
const
   T               :PView =  nil;
var
   P               :PMenuItem;
begin
   if (Name <> '') and (Command <> 0) then
   begin
      New(P);
      P^.Next := Next; P^.Name := NewStr(Name); P^.Command := Command;
      P^.Disabled := not T^.CommandEnabled(Command);
      P^.KeyCode := KeyCode; P^.HelpCtx := AHelpCtx;
      P^.Param := NewStr(Param);
      NewItem := P;
   end else NewItem := Next;
end;

function  NewLine(Next                 :PMenuItem)         :PMenuItem;
var
   P               :PMenuItem;
begin
   New(P);
   P^.Next := Next; P^.Name := nil; P^.HelpCtx := hcNoContext;
   NewLine := P;
end;

function  NewSubMenu(Name              :TMenuStr;
                     AHelpCtx          :Word;
                     SubMenu           :PMenu;
                     Next              :PMenuItem)         :PMenuItem;
var
   P               :PMenuItem;
begin
   if (Name <> '') and (SubMenu <> nil) then
   begin
      New(P);
      P^.Next := Next; P^.Name := NewStr(Name);
      P^.Command := 0; P^.Disabled := False; P^.HelpCtx := AHelpCtx;
      P^.SubMenu := SubMenu; NewSubMenu := P;
   end else
   NewSubMenu := Next;
end;

{ TMenu routines }

function  NewMenu(Items                :PMenuItem)         :PMenu;
var
   P               :PMenu;
begin
   New(P);
   P^.Items := Items; P^.Default := Items;
   NewMenu := P;
end;

procedure DisposeMenu(Menu             :PMenu);
var
   P, Q            :PMenuItem;
begin
   if Menu <> nil then
   begin
      P := Menu^.Items;
      while P <> nil do
      begin
         if P^.Name <> nil then
         begin
            DisposeStr(P^.Name);
            if P^.Command <> 0 then DisposeStr(P^.Param)
            else DisposeMenu(P^.SubMenu);
         end;
         Q := P; P := P^.Next; Dispose(Q);
      end;
      Dispose(Menu);
   end;
end;

{ TMenuColorsView }

constructor TMenuColorsView.Load;
begin
   Inherited Load(S);
   S.Read(Palette, SizeOf(Integer));
end;

procedure TMenuColorsView.Store;
begin
   Inherited Store(S);
   S.Write(Palette, SizeOf(Integer));
end;

function  TMenuColorsView.GetPalette;
begin
   if Palette=mpWhiteMenu
   then GetPalette:=CWhiteMenu
   else GetPalette:=CGrayMenu;
end;

procedure TMenuColorsView.GetMenuColors;
begin
   if IsCurrent then
   begin
      if Disabled
      then Color := GetColors(cpSelDisabled, cpSelDisabled)
      else Color := GetColors(cpSelected, cpShortcut);
      CBack := GetColor(cpSelectedBk);
   end
   else
   begin
      if Disabled
      then Color := GetColors(cpDisabled, cpDisabled)
      else Color := GetColors(cpNormal, cpShortcut);
      CBack := GetColor(cpMain);
   end;
end;

{ TMenuView }

constructor TMenuView.Init(var Bounds  :TGRect);
begin
   TView.Init(Bounds);
   EventMask := EventMask or evBroadcast;
end;

constructor TMenuView.Load(var S       :TStream);

function DoLoadMenu                    :PMenu;
var
   Item            :PMenuItem;
   Last            :^PMenuItem;
   Menu            :PMenu;
   Tok             :Byte;
begin
   New(Menu); Last := @Menu^.Items; Item := nil;
   S.Read(Tok, 1);
   while Tok <> 0 do
   begin
      New(Item); Last^ := Item; Last := @Item^.Next;
      with Item^ do
      begin
         Name := S.ReadStr;
         S.Read(Command, SizeOf(Word) * 3 + SizeOf(Boolean));
         if (Name <> nil) then
         if Command = 0 then SubMenu := DoLoadMenu else Param := S.ReadStr;
      end;
      S.Read(Tok, 1);
   end;
   Last^ := nil;
   Menu^.Default := Menu^.Items;
   DoLoadMenu := Menu;
end;

begin
   Inherited Load(S);
   S.Read(Class, 1);
   Menu := DoLoadMenu;
end;

procedure TMenuView.DrawView;
begin
   Inherited DrawView; SkipRedraw:=False;
end;

function  TMenuView.Execute            :Word;
type
   MenuAction =    (DoNothing, DoSelect, DoReturn);
var
   AutoSelect      :Boolean;
   Action          :MenuAction;
   Ch              :Char;
   Result          :Word;
   ItemShown, P    :PMenuItem;
   Target          :PMenuView;
   R               :TGRect;
   E               :TEvent;
   MouseActive     :Boolean;

procedure TrackMouse;
var
   Mouse           :TGPoint;
   R               :TGRect;
begin
   MakeLocal(E.Where, Mouse);
   Current := Menu^.Items;
   while Current <> nil do
   begin
      GetItemRect(Current, R);
      if R.Contains(Mouse) then
      begin
         MouseActive := True; SkipRedraw:=True; Exit;
      end;
      Current := Current^.Next;
   end;
end;

procedure TrackKey(FindNext            :Boolean);

procedure NextItem;
begin
   Current := Current^.Next;
   if Current = nil then Current := Menu^.Items;
end;

procedure PrevItem;
var
   P               :PMenuItem;
begin
   P := Current;
   if P = Menu^.Items then P := nil;
   repeat NextItem until Current^.Next = P;
end;

begin
   if Current <> nil then
   repeat
      if FindNext then NextItem else PrevItem;
   until Current^.Name <> nil;
   SkipRedraw:=True;
end;

function MouseInOwner                  :Boolean;
var
   Mouse           :TGPoint;
   R               :TGRect;
begin
   MouseInOwner := False;
   if (ParentMenu <> nil) and (ParentMenu^.Class=mcBar) then
   begin
      ParentMenu^.MakeLocal(E.Where, Mouse);
      ParentMenu^.GetItemRect(ParentMenu^.Current, R);
      MouseInOwner := R.Contains(Mouse);
   end;
end;

function MouseInMenus                  :Boolean;
var
   P               :PMenuView;
begin
   P := ParentMenu;
   while (P <> nil) and not P^.MouseInView(E.Where) do P := P^.ParentMenu;
   MouseInMenus := P <> nil;
end;

function TopMenu                       :PMenuView;
var
   P               :PMenuView;
begin
   P := @Self;
   while P^.ParentMenu <> nil do P := P^.ParentMenu;
   TopMenu := P;
end;

begin
   AutoSelect := False; Result := 0; ItemShown := nil;
   Current := Menu^.Default;
   MouseActive := False;
   repeat
      Last:=Current;
      Action := DoNothing;
      GetEvent(E);
      case E.What of
         evMouseDown:
         if MouseInView(E.Where) or MouseInOwner then
         begin
            TrackMouse;
            if Class=mcBar then AutoSelect := True;
         end else Action := DoReturn;
         evMouseUp:
         begin
            TrackMouse;
            if MouseInOwner then Current := Menu^.Default else
            if (Current <> nil) and (Current^.Name <> nil) then
               Action := DoSelect else
            if MouseActive or MouseInView(E.Where) then Action := DoReturn else
            begin
               Current := Menu^.Default;
               if Current = nil then Current := Menu^.Items;
               Action := DoNothing;
            end;
         end;
         evMouseMove:
         if E.Buttons <> 0 then
         begin
            TrackMouse;
            if not (MouseInView(E.Where) or MouseInOwner) and
               MouseInMenus then Action := DoReturn;
         end;
         evKeyDown:
         case CtrlToArrow(E.KeyCode) of
            kbUp, kbDown:
            if Class<>mcBar then
               TrackKey(CtrlToArrow(E.KeyCode) = kbDown) else
               if E.KeyCode = kbDown then AutoSelect := True;
            kbLeft, kbRight:
            if ParentMenu = nil then
               TrackKey(CtrlToArrow(E.KeyCode) = kbRight) else
               Action := DoReturn;
            kbHome, kbEnd:
            if Class<>mcBar then
            begin
               Current := Menu^.Items;
               if E.KeyCode = kbEnd then TrackKey(False);
            end;
            kbEnter:
            begin
               if Class=mcBar then AutoSelect := True;
               Action := DoSelect;
            end;
            kbEsc:
            begin
               Action := DoReturn;
               if (ParentMenu = nil) or (ParentMenu^.Class <> mcBar) then
                  ClearEvent(E);
            end;
         else
            Target := @Self;
            Ch := GetAltChar(E.KeyCode);
            if Ch = #0 then Ch := E.CharCode else Target := TopMenu;
            P := Target^.FindItem(Ch);
            if P = nil then
            begin
               P := TopMenu^.HotKey(E.KeyCode);
               if (P <> nil) and CommandEnabled(P^.Command) then
               begin
                  Result := P^.Command; Action := DoReturn;
               end
            end else
            if Target = @Self then
            begin
               if Class=mcBar then AutoSelect := True;
               Action := DoSelect; Current := P;
            end else
            if (ParentMenu <> Target) or (ParentMenu^.Current <> P) then
               Action := DoReturn;
         end;
         evCommand:
         if E.Command = cmMenu then
         begin
            AutoSelect := False;
            if ParentMenu <> nil then Action := DoReturn;
         end else Action := DoReturn;
      end;
      if ItemShown <> Current then
      begin
         ItemShown := Current; SkipRedraw:=True; DrawView;
      end;
      if (Action = DoSelect) or ((Action = DoNothing) and AutoSelect) then
      if Current <> nil then with Current^ do if Name <> nil then
      if Command = 0 then
      begin
         if E.What and (evMouseDown + evMouseMove) <> 0 then PutEvent(E);
         GetItemRect(Current, R);
         Inc(R.A.X, Origin.X); R.A.Y := R.B.Y + Origin.Y + 1;
         if Class=mcBox then
         begin
            Inc(R.A.X, CharWidth shl 1);
            if Current^.Next=nil then
            Dec(R.A.Y, CharHeight shr 1) else Inc(R.A.Y, CharHeight shr 1);
         end;
         R.B := Owner^.Size;
         Target := TopMenu^.NewSubView(R, SubMenu, @Self);
         Result := Owner^.ExecView(Target);
         if Class=mcBox then SkipRedraw:=False;
         if MouseSafe then HideMouse;
         Dispose(Target, Done);
         if Class=mcBox then DrawView;
         if MouseSafe then ShowMouse;
      end else if Action = DoSelect then Result := Command;
      if (Result <> 0) and CommandEnabled(Result) then
      begin
         Action := DoReturn;
         ClearEvent(E);
      end
      else Result := 0;
   until Action = DoReturn;
   if E.What <> evNothing then
      if (ParentMenu <> nil) or (E.What = evCommand) then PutEvent(E);
   if Current <> nil then
   begin
      Menu^.Default := Current; Current := nil;
      SkipRedraw:=ParentMenu<>nil; DrawView;
   end else SkipRedraw:=False;
   Execute := Result;
end;

function TMenuView.FindItem(Ch         :Char)    :PMenuItem;
var
   P               :PMenuItem;
   I               :Integer;
begin
   Ch := UpCase(Ch);
   P := Menu^.Items;
   while P <> nil do
   begin
      if (P^.Name <> nil) and not P^.Disabled then
      begin
         I := Pos('~', P^.Name^);
         if (I <> 0) and (Ch = UpCase(P^.Name^[I + 1])) then
         begin
            FindItem := P; Exit;
         end;
      end;
      P := P^.Next;
   end;
   FindItem := nil;
end;

procedure TMenuView.GetItemRect(Item   :PMenuItem;
                                var R  :TGRect);
begin
   Abstract;
end;

function TMenuView.GetHelpCtx          :Word;
var
   C               :PMenuView;
begin
   C := @Self;
   while (C <> nil) and
      ((C^.Current = nil) or (C^.Current^.HelpCtx = hcNoContext) or
      (C^.Current^.Name = nil)) do C := C^.ParentMenu;
   if C <> nil then GetHelpCtx := C^.Current^.HelpCtx
   else GetHelpCtx := hcNoContext;
end;

procedure TMenuView.HandleEvent(var Event        :TEvent);
var
   CallDraw        :Boolean;
   P               :PMenuItem;

procedure UpdateMenu(Menu              :PMenu);
var
   P               :PMenuItem;
   CommandState    :Boolean;
begin
   P := Menu^.Items;
   while P <> nil do
   begin
      if P^.Name <> nil then
      if P^.Command = 0 then UpdateMenu(P^.SubMenu)
      else
      begin
         CommandState := CommandEnabled(P^.Command);
         if P^.Disabled = CommandState then
         begin
            P^.Disabled := not CommandState; CallDraw := True;
         end;
      end;
      P := P^.Next;
   end;
end;

procedure DoSelect;
begin
   PutEvent(Event);
   Event.Command := Owner^.ExecView(@Self);
   if (Event.Command <> 0) and CommandEnabled(Event.Command) then
   begin
      Event.What := evCommand; Event.InfoPtr := nil; PutEvent(Event);
   end;
   ClearEvent(Event);
end;

begin
   if Menu <> nil then
   case Event.What of
      evMouseDown:
         DoSelect;
      evKeyDown:
      if (FindItem(GetAltChar(Event.KeyCode)) <> nil) then DoSelect else
      begin
         P := HotKey(Event.KeyCode);
         if (P <> nil) and (CommandEnabled(P^.Command)) then
         begin
            Event.What := evCommand; Event.Command := P^.Command;
            Event.InfoPtr := nil; PutEvent(Event); ClearEvent(Event);
         end;
      end;
      evCommand:
         if Event.Command = cmMenu then DoSelect;
      evBroadcast:
      if Event.Command = cmCommandSetChanged then
      begin
         CallDraw := False; UpdateMenu(Menu); if CallDraw then DrawView;
      end;
   end;
end;

function TMenuView.HotKey(KeyCode      :Word)              :PMenuItem;

function FindHotKey(P                  :PMenuItem)         :PMenuItem;
var
   T               :PMenuItem;
begin
   while P <> nil do
   begin
      if P^.Name <> nil then
      if P^.Command = 0 then
      begin
         T := FindHotKey(P^.SubMenu^.Items);
         if T <> nil then
         begin
            FindHotKey := T; Exit;
         end;
      end
      else
      if not P^.Disabled and (P^.KeyCode <> kbNoKey) and
         (P^.KeyCode = KeyCode) then
      begin
         FindHotKey := P; Exit;
      end;
      P := P^.Next;
   end;
   FindHotKey := nil;
end;

begin
   HotKey := FindHotKey(Menu^.Items);
end;

function TMenuView.NewSubView(var Bounds         :TGRect;
                              AMenu              :PMenu;
                              AParentMenu        :PMenuView)    :PMenuView;
begin
   NewSubView := New(PMenuBox, Init(Bounds, AMenu, AParentMenu));
end;

procedure TMenuView.Store(var S        :TStream);

procedure DoStoreMenu(Menu             :PMenu);
var
   Item            :PMenuItem;
   Tok             :Byte;
begin
   Tok := $FF;
   Item := Menu^.Items;
   while Item <> nil do
   begin
      with Item^ do
      begin
         S.Write(Tok, 1);
         S.WriteStr(Name);
         S.Write(Command, SizeOf(Word) * 3 + SizeOf(Boolean));
         if (Name <> nil) then
         if Command = 0 then DoStoreMenu(SubMenu) else S.WriteStr(Param);
      end;
      Item := Item^.Next;
   end;
   Tok := 0;
   S.Write(Tok, 1);
end;

begin
   Inherited Store(S);
   S.Write(Class, 1);
   DoStoreMenu(Menu);
end;

{ TMenuBar }

constructor TMenuBar.Init(var Bounds   :TGRect;
                          AMenu        :PMenu);
begin
   Inherited Init(Bounds);
   GrowMode := gfGrowHiX; Menu := AMenu;
   Options := Options or ofPreProcess; Class:=mcBar;
end;

destructor TMenuBar.Done;
begin
   Inherited Done;
   DisposeMenu(Menu);
end;

constructor TMenuBar.Standard(AMenu    :PMenu);
var
   R               :TGRect;
begin
   Font:=nil; R.Assign(0, -1, ScreenWidth-1, CharHeight);
   Inherited Init(R);
   GrowMode := gfGrowHiX; Menu := AMenu;
   Options := Options or ofPreProcess; Class:=mcBar;
end;

procedure TMenuBar.Draw;
var
   X, L            :Integer;
   Color, CBack    :Word;
   P               :PMenuItem;
   R               :TGRect;
begin
   GetExtent(R);
   if not SkipRedraw then
   begin
      DrawHLine(R.A,  Size.X, GetColor(cpFrame));
      DrawHLine(R.B, -Size.X, GetColor(cpFrame));
   end;
   R.Grow(0, -1); R.B.X:=CharWidth;
   if not SkipRedraw then DrawBar(R, GetColor(cpMain));
   R.A.X := R.B.X;
   if Menu <> nil then
   begin
      P := Menu^.Items;
      while P <> nil do
      begin
         if P^.Name <> nil then
         begin
            L := (CStrLen(P^.Name^)+2)*CharWidth;
            R.B.X:=R.A.X+L-1;
            if (not SkipRedraw) or (P=Current) or (P=Last) then
            begin
               GetMenuColors(P=Current, P^.Disabled, Color, CBack);
               DrawBar(R, CBack);
               DrawText(R.A, ' '+P^.Name^+' ', tfColored, nil, Color);
            end;
            R.A.X:=R.B.X+1;
         end;
         P := P^.Next;
      end;
   end;
   R.B.X:=Size.X-1; if not SkipRedraw then DrawBar(R, GetColor(cpMain));
end;

procedure TMenuBar.GetItemRect(Item    :PMenuItem;
                               var R   :TGRect);
var
   P               :PMenuItem;
begin
   R.Assign(0, 0, CharWidth, CharHeight);
   P := Menu^.Items;
   while True do
   begin
      R.A.X := R.B.X+1;
      if P^.Name <> nil then Inc(R.B.X, (CStrLen(P^.Name^)+2)*CharWidth);
      if P = Item then Exit;
      P := P^.Next;
   end;
end;

{ TMenuBox }

constructor TMenuBox.Init(var Bounds   :TGRect;
                          AMenu        :PMenu;
                          AParentMenu  :PMenuView);
var
   W, H, C, L      :Integer;
   P               :PMenuItem;
   R               :TGRect;
begin
   InitFont;
   W := CharWidth shl 2; C := CharHeight; H := C shr 1 + 2;
   if AMenu <> nil then
   begin
      P := AMenu^.Items;
      while P <> nil do
      begin
         if P^.Name <> nil then
         begin
            L := (CStrLen(P^.Name^)+2)*CharWidth+2;
            if P^.Command = 0 then Inc(L, CharWidth*3) else
            if P^.Param <> nil then Inc(L, (CStrLen(P^.Param^)+2)*CharWidth);
            if L > W then W := L;
         end;
         Inc(H, C);
         P := P^.Next;
      end;
   end;

   R.Copy(Bounds);
   Dec(R.B.X); Dec(R.B.Y);
   if R.A.X + W < R.B.X then R.B.X := R.A.X + W else R.A.X := R.B.X - W;
   if R.A.Y + H < R.B.Y then R.B.Y := R.A.Y + H else R.A.Y := R.B.Y - H;
   Inherited Init(R);
   Options := Options or ofPreProcess and not ofClipping;
   Menu := AMenu;
   ParentMenu := AParentMenu;
   Class:=mcBox;
end;

procedure TMenuBox.Draw;
var
   CBack, Color    :Word;
   S, D, L, H, PL  :Integer;
   P               :PMenuItem;
   R               :TGRect;
   IName           :TMenuStr;
   ILen            :Byte absolute IName;
begin
   if @Self <> TopView then Exit;
   H := CharHeight; L := (Size.X - 2) div CharWidth - 2;
   GetExtent(R);
   if not SkipRedraw then DrawRect(R, GetColor(cpFrame));
   if Menu <> nil then
   begin
      R.Grow(-1, -1); R.B.Y := H shr 2 + 1; DrawBar(R, GetColor(cpMain));
      R.A.Y:=R.B.Y+1; R.B.Y:=R.A.Y+H-1;
      P := Menu^.Items;
      while P <> nil do
      begin
         if (not SkipRedraw) or (P=Current) or (P=Last) then
         if P^.Name = nil then
         begin { Line }
            R.B.Y := R.A.Y + H shr 1;
            DrawBar(R, GetColor(cpMain));
            R.A.Y := R.B.Y;
            DrawHLine(R.A, Size.X, GetColor(cpFrame));
            Inc(R.A.Y); R.B.Y := R.A.Y + H - H shr 1 - 2;
            DrawBar(R, GetColor(cpMain));
            R.A.Y := R.B.Y+1; R.B.Y := R.A.Y+H-1;
         end
         else
         begin
            IName := P^.Name^;
            PL := Byte(P^.Name^[0])+1;
            D := L + ILen - CStrLen(IName);
            if (P^.Command <> 0) and (P^.Param <> nil) then
               IName := IName + P^.Param^;
            while ILen < D do Insert(' ', IName, PL);
            if P^.Command = 0 then IName[ILen] := '';
            GetMenuColors(P=Current, P^.Disabled, Color, CBack);
            IName := ' ' + IName + ' ';
            DrawBar(R, CBack); DrawText(R.A, IName, tfColored, nil, Color);
            R.Move(0, H);
         end
         else R.Move(0, H);
         P := P^.Next;
      end;
   end;
   R.A.X:=1; R.B.X:=Size.X-2; R.B.Y:=Size.Y-2;
   if not SkipRedraw then DrawBar(R, GetColor(cpMain));
end;

procedure TMenuBox.GetItemRect(Item    :PMenuItem;
                               var R   :TGRect);
var
   Y, H            :Integer;
   P               :PMenuItem;
begin
   P := Menu^.Items; H := CharHeight; Y := H shr 2 + 1;
   while P <> Item do
   begin
      Inc(Y, H); P := P^.Next;
   end;
   R.Assign(1, Y, Size.X - 2, Y + H - 1);
end;

constructor TMenuPopup.Init(var Bounds :TGRect;
                            AMenu      :PMenu);
begin
   inherited Init(Bounds, AMenu, nil);
   Origin := Bounds.A;
end;

destructor TMenuPopup.Done;
begin
   inherited Done;
   if Menu <> nil then DisposeMenu(Menu);
end;

procedure TMenuPopup.HandleEvent(var Event       :TEvent);
var
   P               :PMenuItem;
begin
   case Event.What of
      evKeyDown:
      begin
         P := FindItem(GetCtrlChar(Event.KeyCode));
         if P = nil then P := HotKey(Event.KeyCode);
         if (P <> nil) and (CommandEnabled(P^.Command)) then
         begin
            Event.What := evCommand;
            Event.Command := P^.Command;
            Event.InfoPtr := nil;
            PutEvent(Event);
            ClearEvent(Event);
         end
         else
            if GetAltChar(Event.KeyCode) <> #0 then ClearEvent(Event);
      end;
   end;
   inherited HandleEvent(Event);
end;

{ TStatusLine }

constructor TStatusLine.Init(var Bounds:TGRect;
                             ADefs     :PStatusDef);
begin
   Inherited Init(Bounds);
   Options := Options or ofPreProcess;
   EventMask := EventMask or evBroadcast;
   GrowMode := gfGrowLoY + gfGrowHiX + gfGrowHiY;
   Defs := ADefs;
   FindItems;
end;

constructor TStatusLine.Load(var S     :TStream);

function DoLoadStatusItems             :PStatusItem;
var
   Count           :Integer;
   Cur, First      :PStatusItem;
   Last            :^PStatusItem;
begin
   Cur := nil; Last := @First;
   S.Read(Count, SizeOf(Integer));
   while Count > 0 do
   begin
      New(Cur); Last^ := Cur; Last := @Cur^.Next;
      Cur^.Text := S.ReadStr;
      S.Read(Cur^.KeyCode, SizeOf(Word) * 2);
      Dec(Count);
   end;
   Last^ := nil;
   DoLoadStatusItems := First;
end;

function DoLoadStatusDefs              :PStatusDef;
var
   Cur, First      :PStatusDef;
   Last            :^PStatusDef;
   Count           :Integer;
begin
   Last := @First;
   S.Read(Count, SizeOf(Integer));
   while Count > 0 do
   begin
      New(Cur); Last^ := Cur; Last := @Cur^.Next;
      S.Read(Cur^.Min, 2 * SizeOf(Word));
      Cur^.Items := DoLoadStatusItems;
      Dec(Count);
   end;
   Last^ := nil;
   DoLoadStatusDefs := First;
end;

begin
   Inherited Load(S);
   Defs := DoLoadStatusDefs;
   FindItems;
end;

destructor TStatusLine.Done;
var
   T               :PStatusDef;

procedure DisposeItems(Item            :PStatusItem);
var
   T               :PStatusItem;
begin
   while Item <> nil do
   begin
      T := Item; Item := Item^.Next; DisposeStr(T^.Text); Dispose(T);
   end;
end;

begin
   while Defs <> nil do
   begin
      T := Defs; Defs := Defs^.Next; DisposeItems(T^.Items); Dispose(T);
   end;
   Inherited Done;
end;

constructor TStatusLine.Standard(ADefs :PStatusDef);
var
   R               :TGRect;
begin
   Font:=nil;
   R.Assign(0, ScreenHeight-CharHeight-1, ScreenWidth-1, ScreenHeight);
   Inherited Init(R);
   Options := Options or ofPreProcess;
   EventMask := EventMask or evBroadcast;
   GrowMode := gfGrowLoY + gfGrowHiX + gfGrowHiY;
   Defs := ADefs;
   FindItems;
end;

procedure TStatusLine.Draw;
begin
   DrawSelect(nil);
end;

procedure TStatusLine.DrawSelect(Selected        :PStatusItem);
var
   T               :PStatusItem;
   L               :Integer;
   Color, CBack    :Word;
   HintBuf         :THintStr;
   R               :TGRect;
begin
   GetExtent(R);
   DrawHLine(R.A, Size.X, GetColor(cpFrame));
   DrawHLine(R.B, -Size.X, GetColor(cpFrame));
   R.Grow(0, -1);
   T := Items;
   while T <> nil do
   begin
      if T^.Text <> nil then
      begin
         HintBuf := ' ' + T^.Text^ + ' ';
         L := CStrLen(HintBuf) * CharWidth;
         GetMenuColors(T=Selected, not CommandEnabled(T^.Command), Color, CBack);
         R.B.X:=R.A.X+L;
         DrawBar(R, CBack); DrawText(R.A, HintBuf, tfColored, nil, Color);
         R.A.X:=R.B.X;
      end;
      T := T^.Next;
   end;

   R.B.X:=Size.X-1; DrawBar(R, GetColor(cpMain));

   HintBuf := Hint(HelpCtx);
   if HintBuf <> '' then
   begin
      if R.A.X<>0 then DrawVLine(R.A, R.B.Y - R.A.Y, GetColor(cpFrame));
      Inc(R.A.X, CharWidth shr 1);
      DrawText(R.A, HintBuf, tfNormal, nil, GetColor(cpText));
   end;
end;

procedure TStatusLine.FindItems;
var
   P               :PStatusDef;
begin
   P := Defs;
   while (P <> nil) and ((HelpCtx < P^.Min) or (HelpCtx > P^.Max)) do P := P^.Next;
   if P = nil then Items := nil else Items := P^.Items;
end;

procedure TStatusLine.HandleEvent(var Event      :TEvent);
var
   Mouse           :TGPoint;
   T               :PStatusItem;

function ItemMouseIsIn                 :PStatusItem;
var
   I, K            :Word;
   T               :PStatusItem;
begin
   ItemMouseIsIn := nil;
   if Mouse.Y < 0 then Exit;
   I := 0; T := Items;
   while T <> nil do
   begin
      if T^.Text <> nil then
      begin
         K := I + (CStrLen(T^.Text^) + 2) * CharWidth;
         if (Mouse.X >= I) and (Mouse.X < K) then
         begin
            ItemMouseIsIn := T; Exit;
         end;
         I := K;
      end;
      T := T^.Next;
   end;
end;

begin
   Inherited HandleEvent(Event);
   case Event.What of
      evMouseDown:
      begin
         T := nil;
         repeat
            MakeLocal(Event.Where, Mouse);
            if T <> ItemMouseIsIn then
            begin
               T := ItemMouseIsIn; DrawSelect(T);
            end;
         until not MouseEvent(Event, evMouseMove);
         if (T <> nil) and CommandEnabled(T^.Command) then
         begin
            Event.What := evCommand; Event.Command := T^.Command;
            Event.InfoPtr := nil; PutEvent(Event);
         end;
         ClearEvent(Event);
         DrawView;
      end;
      evKeyDown:
      begin
         T := Items;
         while T <> nil do
         begin
            if (Event.KeyCode = T^.KeyCode) and CommandEnabled(T^.Command) then
            begin
               Event.What := evCommand; Event.Command := T^.Command;
               Event.InfoPtr := nil; Exit;
            end;
            T := T^.Next;
         end;
      end;
      evBroadcast:
         if Event.Command = cmCommandSetChanged then DrawView;
   end;
end;

function TStatusLine.Hint(AHelpCtx     :Word)    :THintStr;
begin
   Hint := '';
end;

procedure TStatusLine.Store(var S      :TStream);

procedure DoStoreStatusItems(Cur       :PStatusItem);
var
   T               :PStatusItem;
   Count           :Integer;
begin
   Count := 0;
   T := Cur;
   while T <> nil do
   begin
      Inc(Count); T := T^.Next
   end;
   S.Write(Count, SizeOf(Integer));
   while Cur <> nil do
   begin
      S.WriteStr(Cur^.Text); S.Write(Cur^.KeyCode, SizeOf(Word) * 2);
      Cur := Cur^.Next;
   end;
end;

procedure DoStoreStatusDefs(Cur        :PStatusDef);
var
   Count           :Integer;
   T               :PStatusDef;
begin
   Count := 0; T := Cur;
   while T <> nil do
   begin
      Inc(Count); T := T^.Next
   end;
   S.Write(Count, SizeOf(Integer));
   while Cur <> nil do
   begin
      with Cur^ do
      begin
         S.Write(Min, SizeOf(Word) * 2);
         DoStoreStatusItems(Items);
      end;
      Cur := Cur^.Next;
   end;
end;

begin
   Inherited Store(S);
   DoStoreStatusDefs(Defs);
end;

procedure TStatusLine.Update;
var
   H, LH           :Word;
   P               :PView;
   LastItems       :PStatusItem;
begin
   P := TopView;
   if P <> nil then H := P^.GetHelpCtx else H := hcNoContext;
   if HelpCtx <> H then
   begin
      LH:=HelpCtx; HelpCtx := H; LastItems:=Items; FindItems;
      if (Items<>LastItems) or (Hint(LH) <> Hint(H)) then DrawView;
   end;
end;

function NewStatusDef(AMin, AMax       :Word;
                      AItems           :PStatusItem;
                      ANext            :PStatusDef)        :PStatusDef;
var
   T               :PStatusDef;
begin
   New(T);
   with T^ do
   begin
      Next := ANext; Min := AMin; Max := AMax; Items := AItems;
   end;
   NewStatusDef := T;
end;

function NewStatusKey(Const AText      :String;
                      AKeyCode         :Word;
                      ACommand         :Word;
                      ANext            :PStatusItem)       :PStatusItem;
var
   T               :PStatusItem;
begin
   New(T);
   T^.Text := NewStr(AText); T^.KeyCode := AKeyCode;
   T^.Command := ACommand; T^.Next := ANext;
   NewStatusKey := T;
end;

procedure RegisterMenus;
begin
   RegisterType(RMenuBar);
   RegisterType(RMenuBox);
   RegisterType(RStatusLine);
   RegisterType(RMenuPopup);
end;

end.
