
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1994,95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit GDialogs;
{$X+,I-,S-}
interface
uses
   Objects, Memory, GRect, Events, KeyMouse, Utils, Language, General,
   GDrivers, GPalette, GViews, GValid;

const

   CInputLine =
   cpMain+cpBlue + cpText+cpWhite +
   cpSelected+cpWhite + cpSelectedBk+cpGreen;

   CButton =
   cpButDisabled+cpLightGray + cpButSelected+cpGreen + cpButDefault+cpLightGray +
   cpNormal+cpBlack + cpShortcut+cpLightRed +
   cpDisabled+cpDarkGray + cpDefault+cpLightCyan;

   CStaticText =
   cpText+cpBlack;

   CLabel =
   cpNormal+cpBlack + cpSelected+cpWhite + cpShortcut+cpYellow;

   CCluster =
   cpMain+cpLightGray + cpHighlight+cpWhite + cpShadow+cpDarkGray +
   cpText+cpBlue + cpShortcut+cpWhite +
   cpCheckBk+cpBlue + cpCheckX+cpWhite;

   CHistoryViewer =
   cpMain+cpBlue + cpNormal+cpWhite +
   cpSelected+cpWhite + cpSelectedBk+cpGreen + cpSelDisabled+cpYellow;

   CHistory =
   cpMain+cpLightGray + cpText+cpBlack;

{ TDialog palette entires }

   dpBlueDialog =  wpBlueWindow;
   dpCyanDialog =  wpCyanWindow;
   dpGrayDialog =  wpGrayWindow;
   dpWhiteDialog = wpWhiteWindow;

{ TButton flags }

   bfNormal    =   $00;
   bfDefault   =   $01;
   bfLeftJust  =   $02;
   bfBroadcast =   $04;
   bfGrabFocus =   $08;
   bfMouseSelect = $20;

{ TMultiCheckboxes flags }
{ hibyte = number of bits }
{ lobyte = bit mask }

   cfOneBit      = $0101;
   cfTwoBits     = $0203;
   cfFourBits    = $040F;
   cfEightBits   = $08FF;

type
   PDialog =       ^TDialog;
   TDialog =
   object(TWindow)
      constructor Init(var Bounds      :TGRect;
                       ATitle          :TTitleStr);

      function  GetMenu(NextItem       :Pointer)           :Pointer; virtual;

      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      function  Valid(Command          :Word)              :Boolean; virtual;
   end;

{ TSItem }

   PSItem =        ^TSItem;
   TSItem = record
      Value        :PString;
      Next         :PSItem;
   end;

   PInputLine =    ^TInputLine;
   TInputLine =
   object(TView)
      Data         :PString;
      MaxLen       :Integer;
      CurPos       :Integer;
      FirstPos     :Integer;
      SelStart     :Integer;
      SelEnd       :Integer;
      Validator    :PValidator;

      constructor Init(var Bounds      :TGRect;
                       AMaxLen         :Integer);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      procedure UpdateCursor;

      function  DataSize                                   :Word; virtual;
      procedure Draw; virtual;
      procedure GetData(var Rec); virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure SelectAll(Enable       :Boolean);
      procedure SetData(var Rec); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure SetValidator(AValid    :PValidator);
      procedure Store(var S            :TStream);
      function  Valid(Command          :Word)              :Boolean; virtual;

   private
      SkipRedraw   :Boolean;

      function  CanScroll(Delta        :Integer)           :Boolean;
   end;

   PButton =       ^TButton;
   TButton =
   object(TView)
      Title        :PString;
      Command      :Word;
      Flags        :Byte;
      AmDefault    :Boolean;

      constructor Init(var Bounds      :TGRect;
                       const ATitle    :TTitleStr;
                       ACommand        :Word;
                       AFlags          :Word);

      constructor InitAt(Pos           :TGPoint;
                         const ATitle  :TTitleStr;
                         ACommand      :Word;
                         AFlags        :Word);

      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      procedure Draw; virtual;
      procedure DrawState(Down: Boolean);
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure MakeDefault(Enable     :Boolean);
      procedure Press; virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure Store(var S            :TStream);
   end;

   PCluster =      ^TCluster;
   TCluster =
   object(TView)
      Value        :LongInt;
      Sel          :Integer;
      EnableMask   :LongInt;
      Strings      :TStringCollection;

      HandleFlag   :Byte;
      StringsLength:Integer;

      constructor Init(var Bounds      :TGRect;
                       AStrings        :PSItem);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      procedure Draw; virtual;
      procedure DrawBox(R              :TGRect;
                        Item           :Integer;
                        Pressed        :Boolean); virtual;
      procedure DrawItem(R             :TGRect;
                         Item          :Integer;
                         Pressed       :Boolean); virtual;

      procedure GetItemRect(Item       :Integer;
                            var R      :TGRect); virtual;

      function  ItemWidth                                  :Integer; virtual;
      function  ItemHeight                                 :Integer; virtual;

      function  ButtonState(Item       :Integer)           :Boolean;
      function  DataSize                                   :Word; virtual;
      procedure GetData(var Rec); virtual;
      function  GetHelpCtx                                 :Word; virtual;
      procedure HandleEvent(var Event                      :TEvent); virtual;
      function  Mark(Item              :Integer)           :Boolean; virtual;
      procedure Press(Item             :Integer); virtual;
      procedure MovedTo(Item           :Integer); virtual;
      procedure SetButtonState(AMask   :Longint;
                               Enable  :Boolean);
      procedure SetData(var Rec); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
      procedure Store(var S            :TStream);
   private
      function  FindSel(P              :TGPoint)           :Integer;
   end;

   PRadioButtons = ^TRadioButtons;
   TRadioButtons =
   object(TCluster)

      procedure DrawBox(R              :TGRect;
                        Item           :Integer;
                        Pressed        :Boolean); virtual;

      function  Mark(Item              :Integer)           :Boolean; virtual;
      procedure MovedTo(Item           :Integer); virtual;
      procedure Press(Item             :Integer); virtual;
      procedure SetData(var Rec); virtual;
   end;

   PCheckBoxes = ^TCheckBoxes;
   TCheckBoxes =
   object(TCluster)

      procedure DrawBox(R              :TGRect;
                        Item           :Integer;
                        Pressed        :Boolean); virtual;

      function  Mark(Item              :Integer)           :Boolean; virtual;
      procedure Press(Item             :Integer); virtual;
   end;

   PListBox =      ^TListBox;
   TListBox =
   object(TListViewer)
      List         :PCollection;

      constructor Init(var Bounds      :TGRect;
                       ANumCols        :Word;
                       AScrollBar      :PScrollBar);
      constructor Load(var S           :TStream);

      function  DataSize                                   :Word; virtual;
      procedure GetData(var Rec); virtual;
      function  GetText(Item           :Integer;
                        MaxLen         :Integer)           :String; virtual;
      procedure NewList(AList          :PCollection); virtual;
      procedure SetData(var Rec); virtual;
      procedure Store(var S            :TStream);
   end;

   PStaticText =   ^TStaticText;
   TStaticText =
   object(TView)
      Text         :PString;

      constructor Init(var Bounds      :TGRect;
                       const AText     :String);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      procedure Draw; virtual;
      procedure GetText(var S          :String); virtual;
      procedure Store(var S            :TStream);
   end;

   PParamText =    ^TParamText;
   TParamText =
   object(TStaticText)
      ParamCount   :Integer;
      ParamList    :Pointer;

      constructor Init(var Bounds      :TGRect;
                       const AText     :String;
                       AParamCount     :Integer);
      constructor Load(var S           :TStream);

      function  DataSize                                   :Word; virtual;
      procedure GetText(var S          :String); virtual;
      procedure SetData(var Rec); virtual;
      procedure Store(var S            :TStream);
   end;

   PLabel =        ^TLabel;
   TLabel =
   object(TStaticText)
      Link         :PView;
      Light        :Boolean;

      constructor Init(var Bounds      :TGRect;
                       const AText     :String;
                       ALink           :PView);
      constructor InitAt(Pos           :TGPoint;
                         const AText   :String;
                         ALink         :PView);
      constructor Standard(const AText :String;
                           ALink       :PView);
      constructor Load(var S           :TStream);

      function  GetPalette                                 :TPalette; virtual;

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Store(var S            :TStream);
   end;

   TBitMap =
   record
      Size         :LongInt;
      Data         :Pointer;
   end;

   PStaticIcon =   ^TStaticIcon;
   TStaticIcon =
   object(TView)
      BitMap       :TBitMap;

      constructor Init(var Bounds      :TGRect;
                       ABitMap         :Pointer);
      constructor InitAt(Point         :TGPoint;
                         ABitMap       :Pointer);
      constructor ReadAt(Point         :TGPoint;
                         var S         :TStream);
      constructor Load(var S           :TStream);
      destructor  Done; virtual;

      procedure Read(var S             :TStream);

      procedure Draw; virtual;
      procedure Store(var S            :TStream);
   end;

   PIcon =         ^TIcon;
   TIcon =
   object(TStaticIcon)
      Command      :Word;

      constructor Init(var Bounds      :TGRect;
                       ACommand        :Word;
                       ABitMap         :Pointer);
      constructor InitAt(Point         :TGPoint;
                         ACommand      :Word;
                         ABitMap       :Pointer);
      constructor ReadAt(Point         :TGPoint;
                         ACommand      :Word;
                         var S         :TStream);
      constructor Load(var S           :TStream);

      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Store(var S            :TStream);
   end;

   PHistoryViewer = ^THistoryViewer;
   THistoryViewer =
   object(TListViewer)
      HistoryId    :Word;

      constructor Init(var Bounds      :TGRect;
                       AHScrollBar,
                       AVScrollBar     :PScrollBar;
                       AHistoryId      :Word);

      function  GetPalette                                 :TPalette; virtual;
      function  GetText(Item           :Integer;
                        MaxLen         :Integer)           :String; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      function  HistoryWidth                               :Integer;
   end;

   PHistoryWindow = ^THistoryWindow;
   THistoryWindow =
   object(TWindow)
      Viewer       :PListViewer;

      constructor Init(var Bounds      :TGRect;
                       HistoryId       :Word);

      function  GetSelection           :String; virtual;
      procedure InitViewer(HistoryId   :Word); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;
   end;

   PHistory = ^THistory;
   THistory = object(TView)
      Link         :PInputLine;
      HistoryId    :Word;

      constructor Init(var Bounds      :TGRect;
                       ALink           :PInputLine;
                       AHistoryId      :Word);
      constructor Standard(ALink       :PInputLine;
                           AHistoryID  :Word);
      constructor Load(var S           :TStream);

      function  GetPalette                                 :TPalette; virtual;

      procedure DrawState(Down         :Boolean); virtual;

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      function  InitWindow(var Bounds  :TGRect)  :PHistoryWindow; virtual;
      procedure RecordHistory(const S  :String); virtual;
      procedure Store(var S            :TStream);
   end;

{ SItem routines }

function NewSItem(const Str            :String;
                  ANext                :PSItem)            :PSItem;

{ Dialogs registration procedure }

procedure RegisterDialogs;

{ Stream Registration Records }

const
  RDialog: TStreamRec = (
     ObjType: 10;
     VmtLink: Ofs(TypeOf(TDialog)^);
     Load:    @TDialog.Load;
     Store:   @TDialog.Store
  );

const
  RInputLine: TStreamRec = (
     ObjType: 11;
     VmtLink: Ofs(TypeOf(TInputLine)^);
     Load:    @TInputLine.Load;
     Store:   @TInputLine.Store
  );

const
  RButton: TStreamRec = (
     ObjType: 12;
     VmtLink: Ofs(TypeOf(TButton)^);
     Load:    @TButton.Load;
     Store:   @TButton.Store
  );

const
  RCluster: TStreamRec = (
     ObjType: 13;
     VmtLink: Ofs(TypeOf(TCluster)^);
     Load:    @TCluster.Load;
     Store:   @TCluster.Store
  );

const
  RRadioButtons: TStreamRec = (
     ObjType: 14;
     VmtLink: Ofs(TypeOf(TRadioButtons)^);
     Load:    @TRadioButtons.Load;
     Store:   @TRadioButtons.Store
  );

const
  RCheckBoxes: TStreamRec = (
     ObjType: 15;
     VmtLink: Ofs(TypeOf(TCheckBoxes)^);
     Load:    @TCheckBoxes.Load;
     Store:   @TCheckBoxes.Store
  );

const
  RListBox: TStreamRec = (
     ObjType: 16;
     VmtLink: Ofs(TypeOf(TListBox)^);
     Load:    @TListBox.Load;
     Store:   @TListBox.Store
  );

const
  RStaticText: TStreamRec = (
     ObjType: 17;
     VmtLink: Ofs(TypeOf(TStaticText)^);
     Load:    @TStaticText.Load;
     Store:   @TStaticText.Store
  );

const
  RLabel: TStreamRec = (
     ObjType: 18;
     VmtLink: Ofs(TypeOf(TLabel)^);
     Load:    @TLabel.Load;
     Store:   @TLabel.Store
  );

const
  RHistory: TStreamRec = (
     ObjType: 19;
     VmtLink: Ofs(TypeOf(THistory)^);
     Load:    @THistory.Load;
     Store:   @THistory.Store
  );

const
  RParamText: TStreamRec = (
     ObjType: 20;
     VmtLink: Ofs(TypeOf(TParamText)^);
     Load:    @TParamText.Load;
     Store:   @TParamText.Store
  );

const
  RStaticIcon: TStreamRec = (
     ObjType: 25;
     VmtLink: Ofs(TypeOf(TStaticIcon)^);
     Load:    @TStaticIcon.Load;
     Store:   @TStaticIcon.Store
  );

const
  RIcon: TStreamRec = (
     ObjType: 26;
     VmtLink: Ofs(TypeOf(TIcon)^);
     Load:    @TIcon.Load;
     Store:   @TIcon.Store
  );

const

{ Dialog broadcast commands }

  cmRecordHistory= 60;

implementation
uses
   GMenus, GApp,
   HistList, Strings;

const

{ TButton messages }
   cmGrabDefault    =   61;
   cmReleaseDefault =   62;

{ TDialog }

constructor TDialog.Init;
begin
   Inherited Init(Bounds, ATitle, wnNoNumber);
   Options := Options or ofVersion20;
   GrowMode := 0;
   Flags := wfMove + wfClose + wfMenu;
   Palette := dpGrayDialog;
end;

function  TDialog.GetMenu;
begin
   GetMenu :=
     NewItem(StrPas(WMenuItems^[0]), 'Alt+F3', kbAltF3, cmClose, hcClose,
     NewItem(StrPas(WMenuItems^[2]), 'Ctrl+F5', kbCtrlF5, cmResize, hcResize,
     NextItem));
end;

procedure TDialog.HandleEvent(var Event:TEvent);
begin
   Inherited HandleEvent(Event);
   case Event.What of
      evKeyDown:
      case Event.KeyCode of
         kbEsc:
         begin
            Event.What := evCommand; Event.Command := cmCancel;
            Event.InfoPtr := nil; PutEvent(Event);
            ClearEvent(Event);
         end;
         kbEnter:
         begin
            Event.What := evBroadcast; Event.Command := cmDefault;
            Event.InfoPtr := nil; PutEvent(Event);
            ClearEvent(Event);
         end;
      end;
      evCommand:
      case Event.Command of
         cmOk, cmCancel, cmYes, cmNo:
         if State and sfModal <> 0 then
         begin
            EndModal(Event.Command); ClearEvent(Event);
         end;
      end;
   end;
end;

procedure TDialog.SetState(AState      :Word;
                           Enable      :Boolean);
begin
   Inherited SetState(AState, Enable);
   if (AState and sfModal<>0) and Enable then DisableCommands([cmZoom]);
end;

function  TDialog.Valid(Command        :Word)    :Boolean;
begin
   if Command = cmCancel then Valid := True
   else Valid := Inherited Valid(Command);
end;

function  NewSItem(const Str           :String;
                   ANext               :PSItem)  :PSItem;
var
   Item            :PSItem;
begin
   New(Item); Item^.Value := NewStr(Str); Item^.Next := ANext; NewSItem := Item;
end;

function  HotKey(const S               :String)  :Char;
var
   P               :Word;
begin
   P := Pos('~', S);
   if P <> 0 then HotKey := UpCase(S[P+1]) else HotKey := #0;
end;

{ TInputLine }

constructor TInputLine.Init;
begin
   Inherited Init(Bounds);
   Size.Y := CharHeight;
   Size.X := Size.X - (Size.X mod CharWidth) + (FrameOffset shl 1);

   State := State or sfCursorVis;
   Options := Options or (ofSelectable + ofFirstClick + ofVersion20);
   GetMem(Data, AMaxLen + 1); Data^ := ''; MaxLen := AMaxLen;
end;

constructor TInputLine.Load(var S      :TStream);
begin
   Inherited Load(S);
   S.Read(MaxLen, SizeOf(Integer) * 5);
   GetMem(Data, MaxLen + 1);
   S.Read(Data^[0], 1); S.Read(Data^[1], Length(Data^));
   Validator := PValidator(S.Get);
end;

destructor TInputLine.Done;
begin
   FreeMem(Data, MaxLen + 1); SetValidator(nil);
   Inherited Done;
end;

function  TInputLine.GetPalette;
begin
   GetPalette := CInputLine;
end;

function  TInputLine.CanScroll(Delta   :Integer) :Boolean;
begin
   if Delta < 0 then CanScroll := FirstPos > 0 else
   if Delta > 0 then CanScroll := Length(Data^) - FirstPos >=
      (Size.X - (FrameOffset shl 1)) div CharWidth else CanScroll := False;
end;

procedure TInputLine.UpdateCursor;
begin
   SetCursor((CurPos - FirstPos) * CharWidth + FrameOffset, 0);
end;

function  TInputLine.DataSize          :Word;
var
   DSize           :Word;
begin
   if Validator <> nil then
      DSize := Validator^.Transfer(Data^, nil, vtDataSize) else DSize:=0;
   if DSize <> 0 then DataSize := DSize else DataSize := MaxLen + 1;
end;

procedure TInputLine.Draw;
var
   Color, SelColor :Word;
   DL, R, L, C     :Integer;
   P               :TGRect;
   S               :String;
   SLen            :Byte absolute S;
begin
   DL := Size.X div CharWidth; if (DL<=0) or (DL>255) then DL := 255;
   FillChar(S[1], DL, ' '); S := Copy(Data^, FirstPos + 1, DL); SLen := DL;

   P.Assign(FrameOffset, 0, Size.X - FrameOffset-1, Size.Y-1);
   if State and sfSelected <> 0 then
   begin
      R := Min(Max(SelEnd - FirstPos, 0), DL);
      L := Min(Max(SelStart - FirstPos, 0), R);
   end
   else
   begin
      L:=0; R:=0;
   end;
   if State and sfDisabled = 0 then
   begin
      Color:=GetColor(cpNormal); SelColor:=GetColor(cpSelected);
   end
   else
   begin
      Color:=GetColor(cpDisabled); SelColor:=GetColor(cpSelDisabled);
   end;

   if SkipRedraw then
   begin
      C:=CurPos; if CurPos=FirstPos then Inc(C);
      Inc(P.A.X, (C - FirstPos) * CharWidth);
      P.B.X:=P.A.X + CharWidth shl 1 - 1; Dec(P.A.X, CharWidth);
      DrawBar(P, GetColor(cpMain));
      S:=Copy(S, C - FirstPos, 3);
      Dec(L, C - FirstPos - 1); if L<0 then L:=0;
      Dec(R, C - FirstPos - 1); if R<0 then R:=0;
      if R>L+2 then
      begin
         R:=L+2; If CurPos=FirstPos then Inc(R);
      end;

      SkipRedraw:=False;
   end
   else Inherited Draw;

   DrawText(P.A, Copy(S, 1, L), tfNormal, nil, Color);
   Inc(P.A.X, L*CharWidth); P.B.X:=P.A.X+(R-L)*CharWidth-1;
   if L<R then DrawBar(P, GetColor(cpSelectedBk));
   DrawText(P.A, Copy(S, L+1, R-L), tfNormal, nil, SelColor);
   P.A.X:=P.B.X+1;
   DrawText(P.A, Copy(S, R + 1, SLen - R), tfNormal, nil, Color);
   UpdateCursor;
end;

procedure TInputLine.GetData(var Rec);
begin
   if (Validator = nil) or
      (Validator^.Transfer(Data^, @Rec, vtGetData) = 0) then
   begin
      FillChar(Rec, DataSize, #0); Move(Data^, Rec, Length(Data^) + 1);
   end;
end;

procedure TInputLine.HandleEvent(var Event       :TEvent);
const
   PadKeys =
   [$47, $4B, $4D, $4F, $73, $74];
var
   Delta, Anchor, I, L       :Integer;
   ExtendBlock, ForceRedraw  :Boolean;
   OldData                   :String;
   OldCurPos, OldFirstPos,
   OldSelStart, OldSelEnd    :Integer;
   WasAppending, CallDraw    :Boolean;

   function  MouseDelta                :Integer;
   var
      Mouse        :TGPoint;
      D            :Integer;
   begin
      D := (L + 3) shr 2;
      MakeLocal(Event.Where, Mouse);
      if Mouse.X < FrameOffset then MouseDelta := -D else
      if Mouse.X > Size.X - FrameOffset then MouseDelta := D
      else MouseDelta := 0;
   end;

   function  MousePos                  :Integer;
   var
      Mouse        :TGPoint;
   begin
      MakeLocal(Event.Where, Mouse);
      MousePos := Min(Max((Mouse.X - FrameOffset) div CharWidth + FirstPos,
         FirstPos), Min(FirstPos+L-1, Length(Data^)));
   end;

   procedure DeleteSelect;
   begin
      if SelStart <> SelEnd then
      begin
         Delete(Data^, SelStart + 1, SelEnd - SelStart);
         CurPos := SelStart;
         CallDraw := True; ForceRedraw:=True;
      end;
   end;

   procedure AdjustSelectBlock;
   begin
      if CurPos < Anchor then
      begin
         SelStart := CurPos; SelEnd := Anchor;
      end
      else
      begin
         SelStart := Anchor; SelEnd := CurPos;
      end;
   end;

   procedure SaveState;
   begin
      if Validator <> nil then
      begin
         OldData := Data^; OldCurPos := CurPos; OldFirstPos := FirstPos;
         OldSelStart := SelStart; OldSelEnd := SelEnd;
         WasAppending := Length(Data^) = CurPos;
      end;
   end;

   procedure RestoreState;
   begin
      if Validator <> nil then
      begin
         Data^ := OldData; CurPos := OldCurPos; FirstPos := OldFirstPos;
         SelStart := OldSelStart; SelEnd := OldSelEnd;
      end;
   end;

   function  CheckValid(NoAutoFill     :Boolean) :Boolean;
   var
      OldLen       :Integer;
      NewData      :String;
   begin
      if Validator <> nil then
      begin
         CheckValid := False; OldLen := Length(Data^);
         if (Validator^.Options and voOnAppend = 0) or
            (WasAppending and (CurPos = OldLen)) then
         begin
            NewData := Data^;
            if not Validator^.IsValidInput(NewData, NoAutoFill)
            then RestoreState else
            begin
               if Length(NewData) > MaxLen then NewData[0] := Char(MaxLen);
               Data^ := NewData;
               if (CurPos >= OldLen) and (Length(Data^) > OldLen) then
                  CurPos := Length(Data^);
               CheckValid := True;
            end;
         end
         else
         begin
            CheckValid := True;
            if CurPos = OldLen then
            if not Validator^.IsValidInput(Data^, False) then
            begin
               Validator^.Error; CheckValid := False;
            end;
         end;
      end
      else CheckValid := True;
   end;

begin
   Inherited HandleEvent(Event);
   L := (Size.X - (FrameOffset shl 1)) div CharWidth;
   if State and sfSelected <> 0 then
   begin
      OldFirstPos:=FirstPos; ForceRedraw:=False;
      case Event.What of
         evMouseDown:
         begin
            Delta := MouseDelta;
            if CanScroll(Delta) then
            begin
               repeat
                  if CanScroll(Delta) then
                  begin
                     Inc(FirstPos, Delta); DrawView;
                  end;
               until not MouseEvent(Event, evMouseAuto);
            end
            else
            if Event.Double then SelectAll(True) else
            begin
               Anchor := MousePos;
               repeat
                  if Event.What = evMouseAuto then
                  begin
                     Delta := MouseDelta;
                     if CanScroll(Delta) then Inc(FirstPos, Delta);
                  end;
                  OldSelStart:=SelStart; OldSelEnd:=SelEnd; OldCurPos:=CurPos;
                  CurPos := MousePos; AdjustSelectBlock;
                  if (SelStart<>OldSelStart) or (SelEnd<>OldSelEnd) then DrawView else
                  if CurPos<>OldCurPos then UpdateCursor;
               until not MouseEvent(Event, evMouseMove + evMouseAuto);
            end;
            ClearEvent(Event);
         end;
         evKeyDown:
         begin
            CallDraw:=SelStart<>SelEnd;
            SaveState;
            Event.KeyCode := CtrlToArrow(Event.KeyCode);
            if (Event.ScanCode in PadKeys) and
               (GetShiftState and (kbLeftShift or kbRightShift) <> 0) then
            begin
               Event.CharCode := #0;
               if CurPos = SelEnd then Anchor := SelStart else Anchor := SelEnd;
               ExtendBlock := True; CallDraw:=True;
            end
            else ExtendBlock := False;
            case Event.KeyCode of
               kbLeft:
                  if CurPos > 0 then Dec(CurPos);
               kbRight:
               if CurPos < Length(Data^) then
               begin
                  Inc(CurPos); CheckValid(True);
               end;
               kbHome:
                  CurPos := 0;
               kbEnd:
               begin
                  CurPos := Length(Data^); CheckValid(True);
               end;
               kbBack:
               if CurPos > 0 then
               begin
                  Delete(Data^, CurPos, 1); Dec(CurPos);
                  if FirstPos > 0 then Dec(FirstPos, L shr 1);
                  CheckValid(True);
                  CallDraw:=True;
                  SkipRedraw:=(CurPos=Length(Data^));
               end;
               kbDel:
               begin
                  if SelStart = SelEnd then
                  if CurPos < Length(Data^) then
                  begin
                     SelStart := CurPos; SelEnd := CurPos + 1;
                  end;
                  DeleteSelect;
                  CheckValid(True);
               end;
               kbIns:
                  SetState(sfCursorIns, State and sfCursorIns = 0);
               else
               case Event.CharCode of
                  ' '..#255:
                  begin
                     CallDraw:=True;
                     if State and sfCursorIns <> 0
                     then Delete(Data^, CurPos + 1, 1) else DeleteSelect;
                     if CheckValid(True) then
                     begin
                        if Length(Data^) < MaxLen then
                        begin
                           if FirstPos > CurPos then FirstPos := CurPos;
                           Inc(CurPos);
                           Insert(Event.CharCode, Data^, CurPos);
                           SkipRedraw:=(CurPos = Length(Data^)) or
                              (State and sfCursorIns <> 0);
                        end else CallDraw:=False;
                        CheckValid(False);
                     end;
                  end;
                  ^Y:
                  begin
                     Data^ := ''; CurPos := 0; CallDraw:=True;
                  end;
                  else Exit;
               end;
            end;
            if ExtendBlock then
            begin
               AdjustSelectBlock; SkipRedraw:=True;
            end else
            begin
               SelStart := CurPos; SelEnd := CurPos;
            end;

            while FirstPos > CurPos do
            begin
               if FirstPos > L shr 1 then Dec(FirstPos, L shr 1)
               else FirstPos := 0;
               CallDraw := True;
            end;
            I := CurPos - L;
            while FirstPos <= I do
            begin
               if FirstPos < Length(Data^) - L shr 1 then Inc(FirstPos, L shr 1)
               else FirstPos := Length(Data^);
               CallDraw := True;
            end;

            if (FirstPos<>OldFirstPos) or ForceRedraw then SkipRedraw:=False;
            if CallDraw then DrawView else UpdateCursor;
            ClearEvent(Event);
         end;
      end;
      if FirstPos<>OldFirstPos then SkipRedraw:=False;
   end;
end;

procedure TInputLine.SelectAll(Enable  :Boolean);
begin
   CurPos := 0; FirstPos := 0; SelStart := 0;
   if Enable then SelEnd := Length(Data^) else SelEnd := 0;
   DrawView;
end;

procedure TInputLine.SetData(var Rec);
begin
   if (Validator = nil) or
      (Validator^.Transfer(Data^, @Rec, vtSetData) = 0) then
      Move(Rec, Data^[0], DataSize);
   SelectAll(True);
end;

procedure TInputLine.SetState(AState   :Word;
                              Enable   :Boolean);
begin
   Inherited SetState(AState, Enable);
   if (AState = sfSelected) or ((AState = sfActive) and
      (State and sfSelected <> 0)) then SelectAll(Enable) else
   if AState = sfFocused then DrawView;
end;

procedure TInputLine.SetValidator(AValid:PValidator);
begin
   if Validator <> nil then Validator^.Free;
   Validator := AValid;
end;

procedure TInputLine.Store(var S       :TStream);
begin
   Inherited Store(S);
   S.Write(MaxLen, SizeOf(Integer) * 5); S.WriteStr(Data); S.Put(Validator);
end;

function TInputLine.Valid(Command      :Word)    :Boolean;
begin
   Valid := Inherited Valid(Command);
   if (Validator <> nil) and (State and sfDisabled = 0) then
   if Command = cmValid then Valid := Validator^.Status = vsOk else
   if Command <> cmCancel then
   if not Validator^.Valid(Data^) then
   begin
      Select; Valid := False;
   end;
end;

{ TButton }

constructor TButton.Init(var Bounds    :TGRect;
                         const ATitle  :TTitleStr;
                         ACommand      :Word;
                         AFlags        :Word);
begin
   Inherited Init(Bounds);
   Options := Options or (ofSelectable + ofFirstClick +
      ofPreProcess + ofPostProcess);
   EventMask := EventMask or evBroadcast;
   if not CommandEnabled(ACommand) then State := State or sfDisabled;
   Flags := AFlags;
   AmDefault:=AFlags and bfDefault <> 0;
   Title := NewStr(ATitle);
   Command := ACommand;
end;

constructor TButton.InitAt;
var
   R               :TGRect;
begin
   InitFont;
   R.Assign(0, 0,
      Max(CStrLen(ATitle)*CharWidth + ButtonShift.X, MinButtonWidth),
      CharHeight+ButtonShift.Y);
   R.Move(Pos.X, Pos.Y);
   Init(R, ATitle, ACommand, AFlags);
end;

constructor TButton.Load(var S         :TStream);
begin
   Inherited Load(S);
   Title := S.ReadStr;
   S.Read(Command, SizeOf(Word) + SizeOf(Byte) + SizeOf(Boolean));
   if not CommandEnabled(Command) then State := State or sfDisabled
   else State := State and not sfDisabled;
end;

destructor TButton.Done;
begin
   DisposeStr(Title); Inherited Done;
end;

function  TButton.GetPalette;
begin
   GetPalette:=CButton;
end;

procedure TButton.Draw;
begin
   DrawState(False);
end;

procedure TButton.DrawState(Down       :Boolean);
var
   R               :TGRect;
   CButton         :Integer;
   CText           :Word;
   TextFlags       :Word;

function  ShowSelected                           :Boolean;

function  Selectable(P                 :PView)   :Boolean; far;
begin
   Selectable:=(P<>@Self) and (P^.Options and ofSelectable <> 0);
end;

begin
   ShowSelected:=(State and sfSelected <> 0) and
      ((Flags and bfMouseSelect <> 0) or
      ((Owner<>nil) and (Owner^.FirstThat(@Selectable)<>nil)));
end;

begin
   if State and sfDisabled <> 0 then CButton := GetColor(cpButDisabled) else
   begin
      CButton := GetColor(cpButBody);
      if State and sfActive <> 0 then
      if ShowSelected then CButton := GetColor(cpButSelected) else
      if Flags and bfDefault <> 0 then CButton := GetColor(cpButDefault);
   end;
   GetExtent(R);
   HideMouseLocal(R);
   DrawButton(R, dbFillIn, Down, CButton);

   if State and sfDisabled<>0 then CText:=GetColors(cpDisabled, cpShortcut) else
   if Flags and bfDefault <> 0 then CText:=GetColors(cpDefault, cpShortcut) else
   CText:=GetColors(cpNormal, cpShortcut);

   R.A.Y:=R.CenterY+1;
   if Flags and bfLeftJust=0 then
   begin
      R.A.X:=R.CenterX+1; TextFlags:=jsCenterX;
   end
   else
   begin
      Inc(R.A.X, ButtonShift.X shr 1); TextFlags:=0;
   end;
   if Down then R.Move(1, 1);
   DrawText(R.A, Title^, tfColored+jsCenterY+TextFlags, nil, CText);
   ShowMouseRect;
end;

procedure TButton.HandleEvent(var Event:TEvent);
var
   Down            :Boolean;
   C               :Char;
   Mouse           :TGPoint;
   ClickRect       :TGRect;
begin
   GetExtent(ClickRect);
   if Flags and bfGrabFocus <> 0 then Inherited HandleEvent(Event);
   case Event.What of
      evMouseDown:
      begin
         if State and sfDisabled = 0 then
         begin
            if Flags and bfMouseSelect <> 0 then Select;
            Down := False;
            repeat
               MakeLocal(Event.Where, Mouse);
               if Down <> ClickRect.Contains(Mouse) then
               begin
                  Down := not Down; DrawState(Down);
               end;
            until not MouseEvent(Event, evMouseMove);
            if Down then
            begin
               Press; DrawState(False);
            end;
         end;
         ClearEvent(Event);
      end;
      evKeyDown:
      begin
         C := HotKey(Title^);
         if (Event.KeyCode = GetAltCode(C)) or
            (Owner^.Phase = phPostProcess) and (C <> #0) and
            (UpCase(Event.CharCode) = C) or
            (State and sfFocused <> 0) and (Event.CharCode = ' ') then
         begin
            Press; ClearEvent(Event);
         end;
      end;
      evBroadcast:
      case Event.Command of
         cmDefault:
         if AmDefault then
         begin
            Press; ClearEvent(Event);
         end;
         cmGrabDefault, cmReleaseDefault:
         if Flags and bfDefault <> 0 then
         begin
            AmDefault := Event.Command = cmReleaseDefault; DrawView;
         end;
         cmCommandSetChanged:
         begin
            SetState(sfDisabled, not CommandEnabled(Command)); DrawView;
         end;
      end;
   end;
end;

procedure TButton.MakeDefault(Enable   :Boolean);
var
   C               :Word;
begin
   if Flags and bfDefault = 0 then
   begin
      if Enable then C := cmGrabDefault else C := cmReleaseDefault;
      Message(Owner, evBroadcast, C, @Self); AmDefault := Enable;
      DrawView;
   end;
end;

procedure TButton.Press;
var
   E               :TEvent;
begin
   Message(Owner, evBroadcast, cmRecordHistory, nil);
   if Flags and bfBroadcast <> 0 then
      Message(Owner, evBroadcast, Command, @Self) else
   begin
      E.What := evCommand; E.Command := Command;
      E.InfoPtr := @Self; PutEvent(E);
   end;
end;

procedure TButton.SetState(AState      :Word;
                           Enable      :Boolean);
begin
   Inherited SetState(AState, Enable);
   if AState and (sfSelected + sfActive) <> 0 then DrawView;
   if AState and sfFocused <> 0 then MakeDefault(Enable);
end;

procedure TButton.Store(var S          :TStream);
begin
   Inherited Store(S);
   S.WriteStr(Title);
   S.Write(Command, SizeOf(Word) + SizeOf(Byte) + SizeOf(Boolean));
end;

{ TCluster }

constructor TCluster.Init(var Bounds   :TGRect;
                          AStrings     :PSItem);
var
   I, W, H         :Integer;
   P               :PSItem;
begin
   Inherited Init(Bounds);
   Options := Options or (ofSelectable + ofFirstClick + ofPreProcess +
      ofPostProcess + ofVersion20);
   I := 0; P := AStrings;
   while P <> nil do
   begin
      Inc(I); P := P^.Next;
   end;
   Strings.Init(I, 0);
   while AStrings <> nil do
   begin
      P := AStrings;
      Strings.AtInsert(Strings.Count, AStrings^.Value);
      AStrings := AStrings^.Next;
      Dispose(P);
   end;
   Value := 0; Sel := 0;
   EnableMask := $FFFFFFFF;

   EventMask := EventMask or evBroadCast;
   if Strings.Count > 0 then
   begin
      StringsLength := CStrLen(PString(Strings.At(0))^);
      for I := 1 to Strings.Count - 1 do
         StringsLength := Max(StringsLength, CStrLen(PString(Strings.At(I))^));
   end;
   W := ItemWidth+FrameOffset; H := ItemHeight;
   I := Size.X mod W;
   if I < Size.X then Dec(Size.X, I) else Size.X := W;
   I := Size.Y mod H;
   if I < Size.Y then Dec(Size.Y, I) else Size.Y := H;
   Inc(Size.X, FrameOffset shl 1); Inc(Size.Y, FrameOffset shl 1);
end;

constructor TCluster.Load(var S        :TStream);
begin
   Inherited Load(S);
   S.Read(Value, SizeOf(Longint) * 2 + SizeOf(Integer));
   Strings.Load(S);
   SetButtonState(0, True);
end;

destructor TCluster.Done;
begin
   Strings.Done;
   Inherited Done;
end;

procedure TCluster.Draw;
var
   I, LastNum, C,
   XR              :Integer;
   R               :TGRect;
begin
   C := GetColor(cpMain);
   GetExtent(R);
   DrawVLine(R.B, R.A.Y - R.B.Y + 1, GetColor(cpShadow));
   DrawHLine(R.B, R.A.X - R.B.X + 1, GetColor(cpShadow));
   DrawVLine(R.A, R.B.Y - R.A.Y, GetColor(cpHighlight));
   DrawHLine(R.A, R.B.X - R.A.X, GetColor(cpHighlight));
   R.Assign(1, Size.Y - FrameOffset, Size.X - 2, Size.Y - 2);
   DrawBar(R, C);
   R.Assign(1, FrameOffset, FrameOffset - 1, Size.Y - FrameOffset - 1);
   DrawBar(R, C);
   R.Assign(1, 1, Size.X - 2, FrameOffset - 1);
   DrawBar(R, C);

   GetItemRect(0, R);
   LastNum := ((Size.Y - (FrameOffset shl 1)) div (R.B.Y - R.A.Y)) *
              ((Size.X - (FrameOffset shl 1)) div (R.B.X - R.A.X)) - 1;
   XR:=0;
   for I := 0 to LastNum do
   begin
      GetItemRect(I, R);
      if I < Strings.Count then DrawItem(R, I, False) else DrawBar(R, C);
      if R.B.X > XR then XR:=R.B.X;
   end;
   R.Assign(XR + 1, 1, Size.X - 2, Size.Y - 2); DrawBar(R, C);
end;

procedure TCluster.DrawBox(R           :TGRect;
                           Item        :Integer;
                           Pressed     :Boolean);
begin
end;

procedure TCluster.DrawItem(R          :TGRect;
                            Item       :Integer;
                            Pressed    :Boolean);
var
   W               :Integer;
   CBack,
   CHigh, CShadow  :Integer;
   S               :String;
   SLen            :Byte absolute S;
begin
   HideMouseLocal(R);
   CBack := GetColor(cpMain);
   if HandleFlag = 0 then DrawBar(R, CBack);
   W := CharHeight;
   Inc(R.A.X, W + FrameOffset);
   FillChar(S[1], 255, ' ');
   S := PString(Strings.At(Item))^; SLen := StringsLength shl 1;
   if (Item <> Sel) and (State and sfSelected <> 0) then
      DrawRect(R, CBack);

   R.Move(FrameOffset, 2);
   DrawText(R.A, S, tfColored, nil, GetColors(cpText, cpShortcut));
   R.Move(-FrameOffset, -2);

   if (Item = Sel) and (State and sfSelected <> 0) then
   begin
      CHigh:=GetColor(cpHighlight); CShadow:=GetColor(cpShadow);
      if Pressed then SwapInts(CHigh, CShadow);
      DrawVLine(R.A, R.B.Y-R.A.Y, CHigh);
      DrawHLine(R.A, R.B.X-R.A.X, CHigh);
      DrawVLine(R.B, R.A.Y-R.B.Y+1, CShadow);
      DrawHLine(R.B, R.A.X-R.B.X+1, CShadow);
   end;
   Dec(R.A.X, W + FrameOffset); R.B.X := R.A.X + CharHeight;
   DrawBox(R, Item, Pressed);
   ShowMouseRect;
end;

procedure TCluster.GetItemRect(Item    :Integer;
                               var R   :TGRect);
var
   H, W,
   NumRows         :Integer;
begin
   LongInt(R.A) := 0;
   W := ItemWidth; H := ItemHeight;
   NumRows := Size.Y div H; if NumRows = 0 then NumRows := 1;
   R.A.X := Item div NumRows * (W + FrameOffset) + FrameOffset;
   R.A.Y := Item mod NumRows * H + FrameOffset;
   R.B.X := R.A.X + W - 1;
   R.B.Y := R.A.Y + H - 1;
end;

function  TCluster.ItemWidth                     :Integer;
begin
   ItemWidth:=StringsLength * CharWidth + CharHeight + (FrameOffset shl 1) + 1;
end;

function  TCluster.ItemHeight                    :Integer;
begin
   ItemHeight:=CharHeight + 4;
end;

function  TCluster.ButtonState(Item    :Integer) :Boolean; assembler;
asm
        XOR     AL,AL
        MOV     CX,Item
        CMP     CX,31
        JA      @@3
        MOV     AX,1
        XOR     DX,DX
        JCXZ    @@2
@@1:    SHL     AX,1
        RCL     DX,1
        LOOP    @@1
@@2:    LES     DI,Self
        AND     AX,ES:[DI].TCluster.EnableMask.Word[0]
        AND     DX,ES:[DI].TCluster.EnableMask.Word[2]
        OR      AX,DX
        JZ      @@3
        MOV     AL,1
@@3:
end;

function  TCluster.DataSize                      :Word;
begin
   DataSize := SizeOf(LongInt);
end;

procedure TCluster.GetData(var Rec);
begin
   LongInt(Rec) := Value;
end;

function  TCluster.GetHelpCtx          :Word;
begin
   if HelpCtx = hcNoContext then GetHelpCtx := hcNoContext
   else GetHelpCtx := HelpCtx + Sel;
end;

function  TCluster.GetPalette                    :TPalette;
begin
   GetPalette := CCluster;
end;

procedure TCluster.HandleEvent(var Event         :TEvent);
var
   Mouse           :TGPoint;
   I, OldSel       :Integer;
   Track, Pressed  :Boolean;
   R, E            :TGRect;
   C               :Char;

procedure Update;
begin
   if OldSel <> Sel then
   begin
      GetItemRect(OldSel, R); DrawItem(R, OldSel, False);
   end;
   GetItemRect(Sel, R); DrawItem(R, Sel, Pressed);
end;

begin
   Inc(HandleFlag);
   OldSel := Sel; Pressed := False;
   if Event.What = evMouseDown then
   begin
      MakeLocal(Event.Where, Mouse);
      I := FindSel(Mouse);
      if I <> -1 then Sel := I;
      MovedTo(Sel); Pressed := True;
      if (Options and ofSelectable <> 0) and not GetState(sfSelected) then
         Select;
      Update;
      repeat
         MakeLocal(Event.Where, Mouse);
         Track := FindSel(Mouse) = Sel;
         if Track <> Pressed then
         begin
            Pressed := not Pressed; Update;
         end;
      until not MouseEvent(Event,evMouseAuto);
      MakeLocal(Event.Where, Mouse);
      Pressed := False;
      if FindSel(Mouse) = Sel then Press(Sel);
      Update;
      ClearEvent(Event);
   end
   else
   if Event.What = evKeyDown then
   case CtrlToArrow(Event.KeyCode) of
      kbUp:
      if State and sfFocused <> 0 then
      begin
         Dec(Sel);
         if Sel < 0 then Sel := Strings.Count - 1;
         MovedTo(Sel); Update;
         ClearEvent(Event);
      end;
      kbDown:
      if State and sfFocused <> 0 then
      begin
         Inc(Sel);
         if Sel >= Strings.Count then Sel := 0;
         MovedTo(Sel); Update;
         ClearEvent(Event);
      end;
      kbRight:
      if State and sfFocused <> 0 then
      begin
         GetItemRect(Sel, R);
         repeat
            Inc(Sel); GetItemRect(Sel, E);
         until (Sel >= Strings.Count) or (R.A.Y <= E.A.Y);
         if Sel >= Strings.Count then Sel := 0;
         MovedTo(Sel); Update;
         ClearEvent(Event);
      end;
      kbLeft:
      if State and sfFocused <> 0 then
      begin
         if Sel > 0 then
         begin
            GetItemRect(Sel, R);
            repeat
               Dec(Sel); GetItemRect(Sel, E);
            until (Sel < 0) or (R.A.Y >= E.A.Y);
            if Sel < 0 then Sel := 0;
         end
         else Sel := Strings.Count - 1;
         MovedTo(Sel); Update;
         ClearEvent(Event);
      end;
      else
      begin
         for I := 0 to Strings.Count-1 do
         begin
            C := HotKey(PString(Strings.At(I))^);
            if (GetAltCode(C) = Event.KeyCode) or
               (((Owner^. Phase = phPostProcess) or (State and sfFocused <> 0))
               and (C <> #0) and (UpCase(Event.CharCode) = C)) then
            begin
               Sel := I;
               Select; Press(Sel); MovedTo(Sel); Update;
               ClearEvent(Event);
               Dec(HandleFlag); Exit;
            end;
         end;
      end;
      if (Event.CharCode = ' ') and (State and sfFocused <> 0) then
      begin
         Press(Sel); Update;
         ClearEvent(Event);
      end;
   end;
   Dec(HandleFlag);
end;

procedure TCluster.SetButtonState(AMask          :Longint;
                                  Enable         :Boolean); assembler;
asm
        LES     DI,Self
        MOV     AX,AMask.Word[0]
        MOV     DX,AMask.Word[2]
        TEST    Enable,0FFH
        JNZ     @@1
        NOT     AX
        NOT     DX
        AND     ES:[DI].TCluster.EnableMask.Word[0],AX
        AND     ES:[DI].TCluster.EnableMask.Word[2],DX
        JMP     @@2
@@1:    OR      ES:[DI].TCluster.EnableMask.Word[0],AX
        OR      ES:[DI].TCluster.EnableMask.Word[2],DX
@@2:    MOV     CX,ES:[DI].Strings.TCollection.Count
        CMP     CX,32
        JA      @@6
        MOV     BX,ES:[DI].TCluster.Options
        AND     BX,not ofSelectable
        MOV     AX,ES:[DI].TCluster.EnableMask.Word[0]
        MOV     DX,ES:[DI].TCluster.EnableMask.Word[2]
@@3:    SHR     DX,1
        RCR     AX,1
        JC      @@4
        LOOP    @@3
        JMP     @@5
@@4:    OR      BX,ofSelectable
@@5:    MOV     ES:[DI].TCluster.Options,BX
@@6:
end;

procedure TCluster.SetData(var Rec);
begin
   Value := LongInt(Rec);
   DrawView;
end;

procedure TCluster.SetState(AState     :Word;
                            Enable     :Boolean);
var
   SaveState       :Word;
   R               :TGRect;
begin
   SaveState := State;
   Inherited SetState(AState, Enable);
   if (AState = sfSelected) and (SaveState <> State) and
      (HandleFlag = 0) and (Sel < Strings.Count) and Exposed then
   begin
      GetItemRect(Sel, R); DrawItem(R, Sel, False);
   end;
end;

function  TCluster.Mark(Item           :Integer) :Boolean;
begin
   Mark := False;
end;

procedure TCluster.MovedTo(Item        :Integer);
begin
end;

procedure TCluster.Press(Item          :Integer);
begin
end;

procedure TCluster.Store(var S         :TStream);
begin
   Inherited Store(S);
   S.Write(Value, SizeOf(Longint) * 2 + SizeOf(Integer));
   Strings.Store(S);
end;

function  TCluster.FindSel(P           :TGPoint) :Integer;
var
   I               :Integer;
   R               :TGRect;
begin
   FindSel := -1;
   for I := 0 to Strings.Count - 1 do
   begin
      GetItemRect(I, R);
      if R.Contains(P) then
      begin
         FindSel := I; Exit;
      end;
   end;
end;

{ TRadioButtons }
procedure TRadioButtons.DrawBox(R      :TGRect;
                                Item   :Integer;
                                Pressed:Boolean);
begin
   HideMouseLocal(R);
   R.Grow(0, -1);
   DrawRect(R, GetColor(cpCheckX));
   R.Grow(-1, -1); DrawBar(R, GetColor(cpCheckBk));
   if Mark(Item) then
   begin
      R.Grow(-2, -2);
      while (R.SizeX>CharWidth shr 1) and
            (R.SizeY>CharHeight shr 1) do R.Grow(-1, -1);
      DrawBullet(R, GetColor(cpCheckX));
   end;
   ShowMouseRect;
end;

function TRadioButtons.Mark(Item       :Integer) :Boolean;
begin
   Mark := Item = Value;
end;

procedure TRadioButtons.Press(Item     :Integer);
begin
   Value := Item;
end;

procedure TRadioButtons.MovedTo(Item   :Integer);
begin
   Value := Item;
end;

procedure TRadioButtons.SetData(var Rec);
begin
   Inherited SetData(Rec);
   Sel := Integer(Value);
end;

{ TCheckBoxes }

procedure TCheckBoxes.DrawBox(R        :TGRect;
                              Item     :Integer;
                              Pressed  :Boolean);
var
   B, C            :Byte;
   I               :Integer;
   S               :TGPoint;
begin
   HideMouseLocal(R);
   C := GetColor(cpCheckX); B := GetColor(cpCheckBk);

   S.X:=R.B.X-R.A.X; S.Y:=R.B.Y-R.A.Y;
   if S.X > S.Y then R.B.X := R.A.X + S.Y;
   if S.X < S.Y then R.B.Y := R.A.Y + S.X;
   R.B.X:=R.A.X+((R.B.X-R.A.X) and $FFFE);
   R.B.Y:=R.A.Y+((R.B.Y-R.A.Y) and $FFFE);
   R.Move(0, (S.Y-(R.B.Y-R.A.Y)) shr 1);

   if HandleFlag = 0 then DrawRect(R, C);
   R.Grow(-1, -1); DrawBar(R, B);
   if Mark(Item) then
   begin
      R.Grow(-1, -1);
      DrawLine(R, C);
      I := R.A.Y; R.A.Y := R.B.Y; R.B.Y := I;
      DrawLine(R, C);
   end;
   ShowMouseRect;
end;

function TCheckBoxes.Mark(Item         :Integer) :Boolean;
begin
   Mark := Value and (1 shl Item) <> 0;
end;

procedure TCheckBoxes.Press(Item       :Integer);
begin
   Value := Value xor (1 shl Item);
end;

{ TListBox }

type
   TListBoxRec =
   record
      List         :PCollection;
      Selection    :Word;
   end;

constructor TListBox.Init(var Bounds   :TGRect;
                          ANumCols     :Word;
                          AScrollBar   :PScrollBar);
begin
   Inherited Init(Bounds, ANumCols, nil, AScrollBar);
   List := nil; SetRange(0);
end;

constructor TListBox.Load(var S        :TStream);
begin
   Inherited Load(S);
   List := PCollection(S.Get);
end;

function  TListBox.DataSize            :Word;
begin
   DataSize := SizeOf(TListBoxRec);
end;

procedure TListBox.GetData(var Rec);
begin
   TListBoxRec(Rec).List := List;
   TListBoxRec(Rec).Selection := Focused;
end;

function  TListBox.GetText(Item        :Integer;
                           MaxLen      :Integer) :String;
begin
   if List <> nil then GetText := PString(List^.At(Item))^ else GetText := '';
end;

procedure TListBox.NewList(AList       :PCollection);
begin
   if List <> nil then Dispose(List, Done);
   List := AList;
   if AList <> nil then SetRange(AList^.Count) else SetRange(0);
   if Range > 0 then FocusItem(0);
   DrawView;
end;

procedure TListBox.SetData(var Rec);
begin
   NewList(TListBoxRec(Rec).List);
   FocusItem(TListBoxRec(Rec).Selection);
   DrawView;
end;

procedure TListBox.Store(var S         :TStream);
begin
   Inherited Store(S);
   S.Put(List);
end;

{ TStaticText }

constructor TStaticText.Init(var Bounds:TGRect;
                             const AText:String);
begin
   Inherited Init(Bounds); Text := NewStr(AText);
end;

constructor TStaticText.Load(var S     :TStream);
begin
   Inherited Load(S); Text := S.ReadStr;
end;

destructor TStaticText.Done;
begin
   DisposeStr(Text); Inherited Done;
end;

procedure TStaticText.Draw;
var
   L, P, PS,
   H, W, SP        :Integer;
   T, S            :String;
   TLen            :Byte absolute T;
   SLen            :Byte absolute S;
   Extent, R       :TGRect;
begin
   TView.Draw;
   GetText(T);
   W := Size.X div CharWidth; H := CharHeight;
   GetExtent(Extent); R.A.Y := Extent.A.Y; R.B.Y := Extent.A.Y+H;
   repeat
      R.A.X := Extent.A.X; R.B.X := Extent.B.X;
      SP:=0;
      P := Pos(^M, T); if P = 0 then P := TLen + 1;
      PS := Pos(^S, T);
      if (PS<>0) and (PS<P) then
      begin
         P:=PS; SP:=Integer(T[PS+1])*CharHeight shr 3; Delete(T, PS, 1);
      end;
      S := Copy(T, 1, P-1); Delete(T, 1, P);
      if S <> '' then
      begin
         if S[1] = ^C then
         begin
            Delete(S, 1, 1); W := SLen*CharWidth;
            if W > R.B.X - R.A.X then W := R.B.X - R.A.X;
            W := (R.B.X - R.A.X - W) shr 1;
            R.Grow(-W, 0);
         end;
         DrawText(R.A, S, tfNormal, nil, GetColor(cpText));
      end;
      R.Move(0, H+SP); R.Intersect(Extent);
   until TLen = 0;
end;

function  TStaticText.GetPalette                 :TPalette;
begin
   GetPalette := CStaticText;
end;

procedure TStaticText.GetText(var S    :String);
begin
   if Text <> nil then S := Text^ else S := '';
end;

procedure TStaticText.Store(var S      :TStream);
begin
   Inherited Store(S); S.WriteStr(Text);
end;

{ TParamText }

constructor TParamText.Init(var Bounds :TGRect;
                            const AText:String;
                            AParamCount:Integer);
begin
   Inherited Init(Bounds, AText);
   ParamCount := AParamCount;
end;

constructor TParamText.Load(var S      :TStream);
begin
   Inherited Load(S);
   S.Read(ParamCount, SizeOf(Integer));
end;

function TParamText.DataSize           :Word;
begin
   DataSize := ParamCount * SizeOf(Longint);
end;

procedure TParamText.GetText(var S     :String);
begin
   if Text <> nil then FormatStr(S, Text^, ParamList^) else S := '';
end;

procedure TParamText.SetData(var Rec);
begin
   ParamList := @Rec; DrawView;
end;

procedure TParamText.Store(var S       :TStream);
begin
   Inherited Store(S);
   S.Write(ParamCount, SizeOf(Integer));
end;

{ TLabel }

constructor TLabel.Init(var Bounds     :TGRect;
                        const AText    :String;
                        ALink          :PView);
begin
   Inherited Init(Bounds, AText);
   Link := ALink;
   Options := Options or (ofPreProcess + ofPostProcess);
   EventMask := EventMask or evBroadcast;
end;

constructor TLabel.InitAt(Pos          :TGPoint;
                          const AText  :String;
                          ALink        :PView);
var
   Bounds          :TGRect;
begin
   InitFont;
   Bounds.A:=Pos;
   Bounds.B.X:=Pos.X+CStrLen(AText)*CharWidth-1; Bounds.B.Y:=Pos.Y+CharHeight-1;
   Init(Bounds, AText, ALink);
end;

constructor TLabel.Standard(const AText:String;
                            ALink      :PView);
var
   Pos             :TGPoint;
begin
   InitFont;
   Pos:=ALink^.Origin;
   Dec(Pos.Y, CharHeight + 1 + Ord(ALink^.Options and ofFramed <> 0));
   InitAt(Pos, AText, ALink);
end;

constructor TLabel.Load(var S          :TStream);
begin
   Inherited Load(S);
   GetPeerViewPtr(S, Link);
end;

procedure TLabel.Draw;
var
   Color           :Word;
const
   TextPos         :TGPoint= (X: 0; Y: 0);
begin
   TView.Draw;
   if Light
   then Color := GetColors(cpSelected, cpShortcut)
   else Color := GetColors(cpNormal, cpShortcut);
   if Text <> nil then DrawText(TextPos, Text^, tfColored, nil, Color);
end;

function  TLabel.GetPalette                      :TPalette;
begin
   GetPalette := CLabel;
end;

procedure TLabel.HandleEvent(var Event :TEvent);
var
   C               :Char;

   procedure FocusLink;
   begin
      if (Link <> nil) and (Link^.Options and ofSelectable <> 0) then Link^.Focus;
      ClearEvent(Event);
   end;

begin
   Inherited HandleEvent(Event);
   if Event.What = evMouseDown then FocusLink else
   if Event.What = evKeyDown then
   begin
      C := HotKey(Text^);
      if (GetAltCode(C) = Event.KeyCode) or
         ((C <> #0) and (Owner^.Phase = phPostProcess) and
         (UpCase(Event.CharCode) = C)) then FocusLink;
   end
   else
   if Event.What = evBroadcast then
   if ((Event.Command = cmReceivedFocus) or
      (Event.Command = cmReleasedFocus)) and
      (Link <> nil) then
   begin
      Light := Link^.State and sfFocused <> 0;
      DrawView;
   end;
end;

procedure TLabel.Store(var S           :TStream);
begin
   Inherited Store(S);
   PutPeerViewPtr(S, Link);
end;

{ TStaticIcon }

const
   NullBitMap      :LongInt =          0;

constructor TStaticIcon.InitAt(Point   :TGPoint;
                               ABitMap :Pointer);
var
   R               :TGRect;
begin
   R.A := Point; R.SetSize(TGPoint(ABitMap^));
   Inherited Init(R);
   Options := Options and not ofSelectable;
   BitMap.Size := 0; BitMap.Data := ABitMap;
end;

constructor TStaticIcon.ReadAt(Point   :TGPoint;
                               var S   :TStream);
var
   R               :TGRect;
   SaveBitMap      :TBitMap;
begin
   BitMap.Size:=0; Read(S);
   R.A := Point; R.SetSize(TGPoint(BitMap.Data^));
   SaveBitMap:=BitMap;
   Inherited Init(R);
   BitMap:=SaveBitMap;
   Options := Options and not ofSelectable;
end;

constructor TStaticIcon.Init(var Bounds:TGRect;
                             ABitMap   :Pointer);
begin
   Inherited Init(Bounds);
   Options := Options and not ofSelectable;
   BitMap.Size := 0; BitMap.Data := ABitMap;
end;

constructor TStaticIcon.Load(var S     :TStream);
begin
   Inherited Load(S);
   Read(S);
end;

destructor TStaticIcon.Done;
begin
   if BitMap.Size <> 0 then FreeMem(BitMap.Data, BitMap.Size);
   Inherited Done;
end;

procedure TStaticIcon.Read(var S       :TStream);
begin
   if BitMap.Size <> 0 then FreeMem(BitMap.Data, BitMap.Size);
   S.Read(BitMap.Size, 4);
   if S.Status=stOk then
   begin
      BitMap.Data := MemAlloc(BitMap.Size);
      if BitMap.Data <> nil then
      begin
         S.Read(BitMap.Data^, BitMap.Size);
         if S.Status<>stOk then
         begin
            FreeMem(BitMap.Data, BitMap.Size); BitMap.Size:=0;
         end;
      end else BitMap.Size:=0;
   end else BitMap.Size:=0;
   if BitMap.Size=0 then BitMap.Data:=@NullBitMap;
end;

procedure TStaticIcon.Draw;
var
   Pos             :TGPoint;
const
   K     :TGPoint= (X:1; Y:1);
begin
   if BitMap.Data<>@NullBitMap then
   begin
      LongInt(Pos):=0;
      DrawImage(Pos, K, BitMap.Data, GetColor(cpMain));
   end else TView.Draw;
end;

procedure TStaticIcon.Store(var S      :TStream);
begin
   Inherited Store(S);
   S.Write(BitMap.Size, 4);
   S.Write(BitMap.Data^, BitMap.Size);
end;

{ TIcon }

constructor TIcon.InitAt(Point         :TGPoint;
                         ACommand      :Word;
                         ABitMap       :Pointer);
begin
   Inherited InitAt(Point, ABitMap);
   Command := ACommand;
end;

constructor TIcon.ReadAt(Point         :TGPoint;
                         ACommand      :Word;
                         var S         :TStream);
begin
   Inherited ReadAt(Point, S);
   Command := ACommand;
end;

constructor TIcon.Init(var Bounds      :TGRect;
                       ACommand        :Word;
                       ABitMap         :Pointer);
begin
   Inherited Init(Bounds, ABitMap);
   Command := ACommand;
end;

constructor TIcon.Load(var S           :TStream);
begin
   Inherited Load(S);
   S.Read(Command, SizeOf(Word));
end;

procedure TIcon.HandleEvent(var Event  :TEvent);
begin
   if Event.What = evMouseDown then
   begin
      Event.What := evCommand; Event.Command := Command;
      Event.InfoPtr := @Self; PutEvent(Event);
      ClearEvent(Event);
   end
   else Inherited HandleEvent(Event);
end;

procedure TIcon.Store(var S            :TStream);
begin
   Inherited Store(S);
   S.Write(Command, SizeOf(Word));
end;

{ THistoryViewer }

constructor THistoryViewer.Init(var Bounds       :TGRect;
                                AHScrollBar,
                                AVScrollBar      :PScrollBar;
                                AHistoryId       :Word);
begin
   Inherited Init(Bounds, 1, AHScrollBar, AVScrollBar);
   HistoryId := AHistoryId;
   SetRange(HistoryCount(AHistoryId));
   if HScrollBar<>nil then HScrollBar^.SetRange(1, HistoryWidth);
end;

function THistoryViewer.GetPalette;
begin
   GetPalette := CHistoryViewer;
end;

function THistoryViewer.GetText(Item, MaxLen     :Integer) :String;
begin
   GetText := HistoryStr(HistoryId, Item);
end;

procedure THistoryViewer.HandleEvent(var Event   :TEvent);
begin
   if ((Event.What = evMouseDown) and (Event.Double)) or
      ((Event.What = evKeyDown) and (Event.KeyCode = kbEnter)) then
   begin
      EndModal(cmOk); ClearEvent(Event);
   end else
   if ((Event.What = evKeyDown) and (Event.KeyCode = kbEsc)) or
      ((Event.What = evCommand) and (Event.Command = cmCancel)) then
   begin
      EndModal(cmCancel); ClearEvent(Event);
   end else Inherited HandleEvent(Event);
end;

function THistoryViewer.HistoryWidth             :Integer;
var
   Width, T, Count, I                  :Integer;
begin
   Width := 0; Count := HistoryCount(HistoryId);
   for I := 0 to Count-1 do
   begin
      T := Length(HistoryStr(HistoryId, I));
      if T > Width then Width := T;
   end;
   HistoryWidth := Width;
end;

{ THistoryWindow }

constructor THistoryWindow.Init(var Bounds       :TGRect;
                                HistoryId        :Word);
begin
   Inherited Init(Bounds, '', wnNoNumber);
   InitViewer(HistoryId);
   Flags := wfClose;
end;

function  THistoryWindow.GetSelection            :String;
begin
   GetSelection := Viewer^.GetText(Viewer^.Focused, 255);
end;

procedure THistoryWindow.InitViewer(HistoryId    :Word);
var
   R               :TGRect;
   SBH, SBV        :PScrollBar;
begin
   SBH:=StandardScrollBar(sbHorizontal + sbHandleKeyboard);
   SBV:=StandardScrollBar(sbVertical + sbHandleKeyboard);
   GetInterior(R);
   Viewer := New(PHistoryViewer, Init(R, SBH, SBV, HistoryId));
   Insert(Viewer);
end;

procedure THistoryWindow.SetState(AState         :Word;
                                  Enable         :Boolean);
begin
   Inherited SetState(AState, Enable);
   if (AState and sfModal<>0) and Enable then DisableCommands([cmZoom, cmResize]);
end;

{ THistory }

constructor THistory.Init(var Bounds   :TGRect;
                          ALink        :PInputLine;
                          AHistoryId   :Word);
begin
   Inherited Init(Bounds);
   Options := Options or ofPostProcess;
   EventMask := EventMask or evBroadcast;
   Link := ALink; HistoryId := AHistoryId;
end;

constructor THistory.Standard(ALink              :PInputLine;
                              AHistoryID         :Word);
var
   R               :TGRect;
begin
   if ALink = nil then Fail;
   InitFont;
   ALink^.GetBounds(R);
   R.A.X := R.B.X + FrameOffset + 1; R.B.X := R.A.X + HistoryWidth - 1;
   Init(R, ALink, AHistoryID);
end;

constructor THistory.Load(var S        :TStream);
begin
   Inherited Load(S);
   GetPeerViewPtr(S, Link);
   S.Read(HistoryId, SizeOf(Word));
end;

function  THistory.GetPalette;
begin
   GetPalette := CHistory;
end;

procedure THistory.Draw;
begin
   DrawState(False);
end;

procedure THistory.DrawState(Down      :Boolean);
var
   R               :TGRect;
begin
   GetExtent(R);
   HideMouseLocal(R);
   DrawButton(R, dbSmall+dbFillIn, Down, GetColor(cpMain));
   if Down then R.Move(1, 1);
   while R.SizeY>SBarPartSize.Y do
   begin
      Inc(R.A.Y); if R.SizeY>SBarPartSize.Y then Dec(R.B.Y);
   end;
   R.Grow(-2, -2);
   if General.Min(R.B.X-R.A.X, R.B.Y-R.A.Y)>8 then R.Grow(-1, -1);
   DrawVArrow(R, False, GetColor(cpText));
   ShowMouseRect;
end;

procedure THistory.HandleEvent(var Event         :TEvent);
var
   W               :PHistoryWindow;
   R, P            :TGRect;
   C               :Integer;
   Rslt            :String;
   Down, Tracking  :Boolean;
begin
   Inherited HandleEvent(Event);
   Tracking := False;
   if Event.What = evMouseDown then
   begin
      Tracking := True; Down := True;
      DrawState(Down);
      repeat
         Tracking := MouseInView(Event.Where);
         if Down <> Tracking then
         begin
            Down := not Down; DrawState(Down);
         end;
      until not MouseEvent(Event, evMouseAuto);
      DrawState(False);
   end;
   if Tracking or
      ((Event.What = evKeyDown) and (CtrlToArrow(Event.KeyCode) = kbDown) and
      (Link^.State and sfFocused <> 0)) then
   begin
      Link^.Select;
      RecordHistory(Link^.Data^);
      if HistoryCount(HistoryId) > 0 then
      begin
         Link^.GetBounds(R);
         R.Grow(1, 0);
         R.B.Y := R.A.Y + 2+Max(
            CharHeight shl 3,
            CharHeight shl 1 + SBarPartSize.Y shl 2);
         PWindow(Owner)^.GetInterior(P); R.Intersect(P);
         if (R.B.X - R.A.X >= CharWidth shl 3) and
            (R.B.Y - R.A.Y >= CharHeight shl 1) then
         begin
            W := InitWindow(R);
            if W <> nil then
            begin
               C := Owner^.ExecView(W);
               if C = cmOk then
               begin
                  Rslt := W^.GetSelection;
                  if Length(Rslt) > Link^.MaxLen then Rslt[0] := Char(Link^.MaxLen);
                  Link^.Data^ := Rslt; Link^.SelectAll(True);
               end;
               Dispose(W, Done);
            end;
         end;
      end;
      ClearEvent(Event);
   end else if (Event.What = evBroadcast) then
   if ((Event.Command = cmReleasedFocus) and (Event.InfoPtr = Link)) or
      (Event.Command = cmRecordHistory) then RecordHistory(Link^.Data^);
end;

function  THistory.InitWindow(var Bounds         :TGRect)  :PHistoryWindow;
var
   W               :PHistoryWindow;
begin
   New(W, Init(Bounds, HistoryId));
   W^.HelpCtx := Link^.HelpCtx;
   InitWindow := W;
end;

procedure THistory.RecordHistory(const S         :String);
begin
   HistoryAdd(HistoryId, S);
end;

procedure THistory.Store(var S         :TStream);
begin
   Inherited Store(S);
   PutPeerViewPtr(S, Link);
   S.Write(HistoryId, SizeOf(Word));
end;

{ Dialogs registration procedure }

procedure RegisterDialogs;
begin
   RegisterType(RDialog);
   RegisterType(RInputLine);
   RegisterType(RButton);
   RegisterType(RCluster);
   RegisterType(RRadioButtons);
   RegisterType(RCheckBoxes);
   RegisterType(RListBox);
   RegisterType(RStaticText);
   RegisterType(RLabel);
   RegisterType(RHistory);
   RegisterType(RParamText);

   RegisterType(RStaticIcon);
   RegisterType(RIcon);
end;

end.
