
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1995 by Solar Designer            }
{                                                       }
{*******************************************************}

unit GDiagram;
{$V-}
interface
uses
   Objects, Strings,
   GViews,
   GRect, GPalette;

const
   cpScale =       #$A0;
   cpGraphic =     #$A1;

   CGraphic =
   cpMain+cpLightGray +
   cpText+cpBlack + cpScale+cpBlack +
   cpTitle+cpYellow +
   cpGraphic+cpLightRed;

   MaxGraphics =   4;

type
   TFloat =        Real;

   PFloatArray =   ^TFloatArray;
   TFloatArray =   Array [0..$FFF0 div SizeOf(TFloat) - 1] of TFloat;

   PCoordParams =  ^TCoordParams;
   TCoordParams =
   object
      Data         :PFloatArray;
      MaxIndex     :Integer;
      Min, Max     :TFloat;
      FmtW, FmtD   :Integer;
      ScalePoints  :Integer;

      procedure SetRange (AMin, AMax   :TFloat);
      procedure SetFormat(W, D, N      :Integer);

      procedure CalcRange(DoRound      :Boolean);
   end;

   PGraphicParams= ^TGraphicParams;
   TGraphicParams=
   object
      X, Y         :TCoordParams;
      Color        :Word;
      Title        :PChar;
   end;

   PGraphic =      ^TGraphic;
   TGraphic =
   object(TView)
      MaxGraphic   :Integer;
      Graphics     :Array [0..MaxGraphics-1] of TGraphicParams;
      Width        :Integer;
      Title        :PChar;
      BottomLines  :Integer;

      constructor Init(var Bounds      :TGRect);

      procedure SetMaxIndex(AMaxIndex  :Integer);
      procedure CalcRange(Axis         :Integer;
                          DoRound      :Boolean);

      function  GetVal (var Coord      :TCoordParams;
                        Index          :Integer) :TFloat; virtual;

      procedure GetName(var Coord      :TCoordParams;
                        Index          :Integer;
                        var Name       :String); virtual;

      function  GetPalette                       :TPalette; virtual;
      procedure Draw; virtual;
   end;

implementation

{ TCoordParams }

procedure TCoordParams.SetRange;
begin
   Min:=AMin; Max:=AMax;
end;

procedure TCoordParams.SetFormat;
begin
   FmtW:=W; FmtD:=D; ScalePoints:=N;
end;

procedure TCoordParams.CalcRange;
const
   Eps =           1E-6;

var
   i               :Integer;
   SMin, SMax      :Boolean;
   D, MinD         :TFloat;

procedure Round10(var x                :TFloat;
                  Dir                  :Boolean);
var
   i, n, k, s      :Integer;

function  Mod0(x, y                    :Integer) :Integer; assembler;
asm
   mov  ax,x
   mov  cx,y
   div  cl
   mov  ax,cx
   or   ah,ah
   jz   @@Done
   mov  al,ah
   xor  ah,ah
@@Done:
end;

begin
   if x=0 then Exit;

   i:=0;
   if x<0 then s:=-1 else s:=1;

   x:=abs(x);
   while x<10 do
   begin
      x:=x*10; Inc(i);
   end;
   while x>100 do
   begin
      x:=x*0.1; Dec(i);
   end;

   if (Trunc(x)<>x) or (Trunc(x) mod 5<>0) or
      ((x>=20) and (Trunc(x) mod 10<>0)) then
   begin
      if Dir xor (s>0) then
      begin
         k:=Trunc(x)+1;
         if k<20 then Inc(k, 5-Mod0(k, 5)) else Inc(k, 10-Mod0(k, 10));
      end else
      begin
         k:=Trunc(x);
         if k<20 then Dec(k, k mod 5) else Dec(k, k mod 10);
      end;
      x:=k*s;
   end else x:=x*s;

   for n:=1 to i do x:=x*0.1;
   for n:=-1 downto i do x:=x*10;
end;

begin
   if Data=nil then Exit;

   Min:=Data^[0]; Max:=Min;
   for i:=1 to MaxIndex do
   begin
      if Data^[i]<Min then Min:=Data^[i];
      if Data^[i]>Max then Max:=Data^[i];
   end;

   if DoRound then
   begin
      repeat
         D:=(Max-Min)*0.1;
         MinD:=Min*ScalePoints;
         for i:=1 to FmtW do MinD:=MinD*0.1;
         if D<MinD then D:=MinD;

         if Min-D>=Min then
         if D<Eps then D:=Eps else D:=Eps*Min;

         SMin:=(Min+Eps>=0); SMax:=(Max-Eps<=0);
         Min:=Min-D; Max:=Max+D;
         if SMin and (Min<0) then Min:=0 else
         if SMax and (Max>0) then Max:=0;

         D:=(Max-Min)/ScalePoints;
         for i:=1 to FmtD do D:=D*10;
      until D>1;

      Round10(Min, True);
      Round10(Max, False);
   end;
end;

{ TGraphic }

constructor TGraphic.Init;
var
   G               :Integer;
begin
   Inherited Init(Bounds);

   for G:=0 to MaxGraphics-1 do Graphics[G].Color:=GetColor(cpGraphic);
   Width:=1;
end;

procedure TGraphic.SetMaxIndex;
var
   G               :Integer;
begin
   for G:=0 to MaxGraphic do
   with Graphics[G] do
   begin
      X.MaxIndex:=AMaxIndex; Y.MaxIndex:=AMaxIndex;
   end;
end;

procedure TGraphic.CalcRange;
var
   Calc            :Boolean;
   G               :Integer;
   C               :PCoordParams;
   Min, Max        :TFloat;
begin
   for Calc:=True downto False do
   for G:=0 to MaxGraphic do
   with Graphics[G] do
   begin
      C:=PCoordParams(PChar(@X)+SizeOf(X)*Axis);

      if Calc then
      begin
         C^.CalcRange(DoRound);
         if (G=0) or (C^.Min<Min) then Min:=C^.Min;
         if (G=0) or (C^.Max>Max) then Max:=C^.Max;
      end else
      begin
         C^.Min:=Min; C^.Max:=Max;
      end;
   end;
end;

function  TGraphic.GetVal;
begin
   with Coord do
   if Data=nil
   then GetVal:=Min+(Max-Min)*Index/MaxIndex
   else GetVal:=Data^[Index];
end;

procedure TGraphic.GetName;
begin
   with Coord do
   if FmtW>0 then Str((Min+(Max-Min)*Index/ScalePoints):FmtW:FmtD, Name) else Name:='';
end;

function  TGraphic.GetPalette;
begin
   GetPalette:=CGraphic;
end;

procedure TGraphic.Draw;
var
   R, L            :TGRect;
   Pos             :TGPoint;
   CScale, CText   :Word;
   G, i, W, W2     :Integer;
   J               :Word;
   Name            :String[79];
begin
   TView.Draw;

   CScale:=GetColor(cpScale); CText:=GetColor(cpText);

   GetExtent(R);

   if Title<>nil then
   begin
      Pos.X:=R.CenterX; Pos.Y:=R.A.Y+CharHeight;
      DrawText(Pos, StrPas(Title), jsCenter, nil, GetColor(cpTitle));
      Inc(R.A.Y, CharHeight shl 1);
   end;

   R.A.X:=CharWidth*Graphics[0].Y.FmtW+FrameOffset; Dec(R.B.Y, CharHeight);
   R.Grow(-(FrameOffset shl 1), -(FrameOffset shl 1));

   if BottomLines<>0 then
   Dec(R.B.Y, CharHeight*BottomLines+FrameOffset);

   Pos.Y:=R.A.Y;
   for i:=0 to Graphics[0].X.ScalePoints do
   begin
      Pos.X:=R.A.X+LongDiv(LongMul(R.SizeX, i), Graphics[0].X.ScalePoints);
      DrawVLine(Pos, R.SizeY, CScale);

      if i<Graphics[0].X.ScalePoints then
      begin
         GetName(Graphics[0].X, i, Name);
         L.A.X:=Pos.X; L.A.Y:=R.B.Y+FrameOffset;
         if i=0 then J:=jsLeft+jsTop else J:=jsCenterX+jsTop;
         DrawText(L.A, Name, J, nil, CText);
      end;
   end;

   Pos.X:=R.A.X-1;
   for i:=0 to Graphics[0].Y.ScalePoints do
   begin
      Pos.Y:=R.B.Y-LongDiv(LongMul(R.SizeY, i), Graphics[0].Y.ScalePoints);
      DrawHLine(Pos, R.SizeX, CScale);

      GetName(Graphics[0].Y, i, Name);
      L.A.X:=R.A.X-FrameOffset; L.A.Y:=Pos.Y+1;
      if i=0 then
      begin
         J:=jsRight+jsBottom; Inc(L.A.Y, FrameOffset shl 1);
      end else J:=jsRight+jsCenterY;
      DrawText(L.A, Name, J, nil, CText);
   end;

   W2:=Width shr 1;

   for G:=0 to MaxGraphic do
   with Graphics[G] do
   for i:=0 to X.MaxIndex do
   begin
      Pos.X:=Round((GetVal(X, i)-X.Min)/(X.Max-X.Min)*(R.SizeX-W2))+R.A.X;
      Pos.Y:=R.B.Y-Round((GetVal(Y, i)-Y.Min)/(Y.Max-Y.Min)*(R.SizeY-W2))-W2;
      if i<>0 then
      begin
         LongInt(L.B):=LongInt(Pos);
         for W:=1 to Width do
         begin
            DrawLine(L, Color);
            L.Move(W and 1, (not W) and 1);
         end;
      end;
      LongInt(L.A):=LongInt(Pos);
   end;

   R.A.X:=0; R.A.Y:=Size.Y-BottomLines*CharHeight-FrameOffset+CharHeight shr 1;
   Pos.X:=CharWidth shl 2; Pos.Y:=Width;
   for G:=0 to MaxGraphic do
   with Graphics[G] do
   if Title<>nil then
   begin
      Name:=StrPas(Title);
      if Name[1]=^M then
      begin
         Delete(Name, 1, 1);
         R.A.X:=0; Inc(R.A.Y, CharHeight);
      end;

      Inc(R.A.X, CharWidth shl 1);
      R.SetSize(Pos); L:=R; L.Move(0, -(1+Width shr 1));
      DrawBar(L, Color);
      Inc(R.A.X, Pos.X+CharWidth);
      DrawText(R.A, Name, jsLeft+jsCenterY, nil, CText);
      Inc(R.A.X, Length(Name)*CharWidth);
   end;
end;

end.
