
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1994,95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit GApp;

{$X+,B-,I-,S-}

interface
uses
   GraphDrv, GDrivers, Events, KeyMouse, Utils, General, GRect, Language,
   Objects, Memory, HistList, GViews, GMenus, GDialogs, GPalette;

const

   CDefaults =
   cpMain+cpLightGray + cpText+cpBlack +         { Background/text }
   cpCursor+cpWhite +                            { Text cursor }
   cpButBody+cpLightGray + cpButFrame+cpBlack +  { for TView.DrawButton }
   cpButHigh+cpWhite + cpButShadow+cpDarkGray;

   CBackground =
   cpMain+cpDarkGray;

{ Standard application commands }

   cmNew =         30;
   cmOpen =        31;
   cmSave =        32;
   cmSaveAs =      33;
   cmSaveAll =     34;
   cmChangeDir =   35;
   cmDosShell =    36;
   cmCloseAll =    37;

   cmRefresh =     38;
   cmAbout =       40;

{ Standard application help contexts }
{ Note: range $FF00 - $FFFF of help contexts are reserved by Borland }

   hcNew         = $FF01;
   hcOpen        = $FF02;
   hcSave        = $FF03;
   hcSaveAs      = $FF04;
   hcSaveAll     = $FF05;
   hcChangeDir   = $FF06;
   hcDosShell    = $FF07;
   hcExit        = $FF08;

   hcUndo        = $FF10;
   hcCut         = $FF11;
   hcCopy        = $FF12;
   hcPaste       = $FF13;
   hcClear       = $FF14;

   hcTile        = $FF20;
   hcCascade     = $FF21;
   hcCloseAll    = $FF22;
   hcResize      = $FF23;
   hcZoom        = $FF24;
   hcNext        = $FF25;
   hcPrev        = $FF26;
   hcClose       = $FF27;

   hcRefresh     = $FF2A;
   hcAbout       = $FF2B;

type

   PBackground = ^TBackground;
   TBackground =
   object(TView)
      constructor Init(var Bounds      :TGRect);

      function  GetPalette                                 :TPalette; virtual;
   end;

   PDesktop = ^TDesktop;
   TDesktop =
   object(TGroup)
      Background             :PBackground;
      TileColumnsFirst       :Boolean;

      constructor Init(var Bounds      :TGRect);
      constructor Load(var S           :TStream);

      procedure Cascade(var R          :TGRect);
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure InitBackground; virtual;
      procedure Store(var S            :TStream);
      procedure Tile(var R             :TGRect);
      procedure TileError; virtual;
   end;

   PProgram = ^TProgram;
   TProgram = object(TGroup)
      constructor Init;
      destructor  Done; virtual;

      function  GetPalette                                 :TPalette; virtual;

      function  CanMoveFocus                               :Boolean;
      function  ExecuteDialog(P        :PDialog;
                              Data     :Pointer)           :Word;
      procedure GetEvent(var Event     :TEvent); virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Idle; virtual;
      procedure InitDesktop; virtual;
      procedure InitMenuBar; virtual;
      procedure InitScreen; virtual;
      procedure InitStatusLine; virtual;
      function  InsertWindow(P         :PWindow)           :PWindow;
      procedure OutOfMemory; virtual;
      procedure PutEvent(var Event     :TEvent); virtual;
      procedure Run; virtual;
      function  SetScreenMode(const Driver       :String;
                              Mode               :Word)    :Boolean;
      function  ValidView(P            :PView)             :PView;
   end;

   PApplication = ^TApplication;
   TApplication = object(TProgram)
      constructor Init(const Driver    :String;
                       Mode            :Word);
      destructor  Done; virtual;

      procedure Cascade; virtual;
      procedure CloseAll; virtual;
      procedure DosShell; virtual;
      procedure GetTileRect(var R      :TGRect); virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure Tile; virtual;
      procedure WriteShellMsg; virtual;
   end;

{ Standard menus and status lines }

function StdStatusKeys(Next            :PStatusItem)       :PStatusItem;

function StdFileMenuItems(Next         :PMenuItem)         :PMenuItem;
function StdEditMenuItems(Next         :PMenuItem)         :PMenuItem;
function StdWindowMenuItems(Next       :PMenuItem)         :PMenuItem;

{ App registration procedure }

procedure RegisterApp;

const

{ Public variables }

   Application     :PProgram =    nil;
   Desktop         :PDesktop =    nil;
   StatusLine      :PStatusLine = nil;
   MenuBar         :PMenuView =   nil;

{ Stream registration records }

const
   RBackground: TStreamRec = (
     ObjType: 30;
     VmtLink: Ofs(TypeOf(TBackground)^);
     Load: @TBackground.Load;
     Store: @TBackground.Store);

   RDesktop: TStreamRec = (
     ObjType: 31;
     VmtLink: Ofs(TypeOf(TDesktop)^);
     Load: @TDesktop.Load;
     Store: @TDesktop.Store);

implementation
uses
   DOS;

const

{ Private variables }

   Pending         :TEvent = (What: evNothing);

{ TBackground }

constructor TBackground.Init(var Bounds:TGRect);
begin
   TView.Init(Bounds);
   GrowMode := gfGrowHiX + gfGrowHiY;
   Options:=Options or ofClipping;
end;

function  TBackground.GetPalette;
begin
   GetPalette:=CBackground;
end;

{ TDesktop object }

constructor TDesktop.Init(var Bounds   :TGRect);
begin
   Inherited Init(Bounds);
   GrowMode := gfGrowHiX + gfGrowHiY;
   InitBackground;
   if Background <> nil then Insert(Background);
end;

constructor TDesktop.Load(var S        :TStream);
begin
   Inherited Load(S);
   GetSubViewPtr(S, Background);
   S.Read(TileColumnsFirst, SizeOf(TileColumnsFirst));
end;

function Tileable(P                    :PView)   :Boolean;
begin
   Tileable := (P^.Options and ofTileable <> 0) and
      (P^.State and sfVisible <> 0);
end;

procedure TDesktop.Cascade(var R       :TGRect);
var
   CascadeNum      :Integer;
   LastView        :PView;
   Min, Max        :TGPoint;

procedure DoCount(P                    :PView); far;
begin
   if Tileable(P) then
   begin
      Inc(CascadeNum); LastView := P;
   end;
end;

procedure DoCascade(P                  :PView); far;
var
   NR              :TGRect;
begin
   if Tileable(P) and (CascadeNum >= 0) then
   begin
      NR.Copy(R);
      Inc(NR.A.X, CascadeNum*(CharWidth+1));
      Inc(NR.A.Y, CascadeNum*(CharHeight+1));
      P^.Locate(NR);
      Dec(CascadeNum);
   end;
end;

begin
   CascadeNum := 0;
   ForEach(@DoCount);
   if CascadeNum > 0 then
   begin
      LastView^.SizeLimits(Min, Max);
      if (Min.X > R.B.X - R.A.X +1 - CascadeNum*CharWidth) or
         (Min.Y > R.B.Y - R.A.Y +1 - CascadeNum*CharHeight)
      then TileError else
      begin
         Dec(CascadeNum);
         LockDraw; ForEach(@DoCascade); UnlockDraw; Redraw;
      end;
   end;
end;

procedure TDesktop.HandleEvent(var Event         :TEvent);
begin
   Inherited HandleEvent(Event);
   if Event.What = evCommand then
   begin
      case Event.Command of
         cmNext: FocusNext(False);
         cmPrev:
         if Valid(cmReleasedFocus) then Current^.PutInFrontOf(Background);
      else
         Exit;
      end;
      ClearEvent(Event);
   end;
end;

procedure TDesktop.InitBackground;
var
   R               :TGRect;
begin
   GetExtent(R); New(Background, Init(R));
end;

function ISqr(X: Integer): Integer; assembler;
asm
	MOV	CX,X
        MOV	BX,0
@@1:    INC     BX
	MOV	AX,BX
	IMUL	AX
        CMP	AX,CX
        JLE	@@1
	MOV	AX,BX
        DEC     AX
end;

procedure MostEqualDivisors(N          :Integer;
                            var X, Y   :Integer;
                            FavorY     :Boolean);
var
   I               :Integer;
begin
   I := ISqr(N);
   if ((N mod I) <> 0) then
   if (N mod (I+1)) = 0 then Inc(I);
   if I < (N div I) then I := N div I;
   if FavorY then
   begin
      X := N div I; Y := I;
   end
   else
   begin
      Y := N div I; X := I;
   end;
end;

procedure TDesktop.Store(var S         :TStream);
begin
   Inherited Store(S);
   PutSubViewPtr(S, Background);
   S.Write(TileColumnsFirst, SizeOf(TileColumnsFirst));
end;

procedure TDesktop.Tile(var R          :TGRect);
var
   NumCols, NumRows,
   NumTileable,
   LeftOver,
   TileNum         :Integer;

procedure DoCountTileable(P            :PView); far;
begin
   if Tileable(P) then Inc(NumTileable);
end;

function DividerLoc(Lo, Hi, Num, Pos   :Integer) :Integer;
begin
   DividerLoc := LongDiv(LongMul(Hi - Lo +1, Pos), Num) + Lo;
end;

procedure CalcTileRect(Pos             :Integer;
                       var NR          :TGRect);
var
   X, Y, D         :Integer;
begin
   D := (NumCols - LeftOver) * NumRows;
   if Pos < D then
   begin
      X := Pos div NumRows; Y := Pos mod NumRows;
   end else
   begin
      X := (Pos - D) div (NumRows + 1) + (NumCols - LeftOver);
      Y := (Pos - D) mod (NumRows + 1);
   end;
   NR.A.X := DividerLoc(R.A.X, R.B.X, NumCols, X);
   NR.B.X := DividerLoc(R.A.X, R.B.X, NumCols, X+1) - 1;
   if Pos >= D then
   begin
      NR.A.Y := DividerLoc(R.A.Y, R.B.Y, NumRows+1, Y);
      NR.B.Y := DividerLoc(R.A.Y, R.B.Y, NumRows+1, Y+1) - 1;
   end else
   begin
      NR.A.Y := DividerLoc(R.A.Y, R.B.Y, NumRows, Y);
      NR.B.Y := DividerLoc(R.A.Y, R.B.Y, NumRows, Y+1) - 1;
   end;
end;

procedure DoTile(P                     :PView); far;
var
   R               :TGRect;
begin
   if Tileable(P) then
   begin
      CalcTileRect(TileNum, R); P^.Locate(R); Dec(TileNum);
   end;
end;

begin
   NumTileable := 0;
   ForEach(@DoCountTileable);
   if NumTileable > 0 then
   begin
      MostEqualDivisors(NumTileable, NumCols, NumRows, not TileColumnsFirst);
      if ((R.B.X - R.A.X + 1) div NumCols = 0) or
         ((R.B.Y - R.A.Y + 1) div NumRows = 0) then TileError
      else
      begin
         LeftOver := NumTileable mod NumCols;
         TileNum := NumTileable-1;
         LockDraw; ForEach(@DoTile); UnlockDraw; Redraw;
      end;
   end;
end;

procedure TDesktop.TileError;
begin
end;

{ TProgram }

constructor TProgram.Init;
var
   R               :TGRect;
begin
   Application := @Self;
   InitScreen;
   R.Assign(0, 0, ScreenWidth-1, ScreenHeight-1);
   Inherited Init(R);
   State := sfVisible + sfSelected + sfFocused + sfModal + sfExposed;
   Options := 0;
   InitStatusLine;
   InitMenuBar;
   InitDesktop;
   if Desktop <> nil then Insert(Desktop);
   if StatusLine <> nil then Insert(StatusLine);
   if MenuBar <> nil then Insert(MenuBar);
end;

destructor TProgram.Done;
begin
   if Desktop <> nil then Dispose(Desktop, Done);
   if MenuBar <> nil then Dispose(MenuBar, Done);
   if StatusLine <> nil then Dispose(StatusLine, Done);
   Application := nil;
   Inherited Done;
end;

function TProgram.GetPalette;
begin
   GetPalette:=CDefaults;
end;

function TProgram.CanMoveFocus: Boolean;
begin
   CanMoveFocus := Desktop^.Valid(cmReleasedFocus);
end;

function TProgram.ExecuteDialog(P      :PDialog;
                                Data   :Pointer) :Word;
var
   C               :Word;
begin
   ExecuteDialog := cmCancel;
   if ValidView(P) <> nil then
   begin
      if Data <> nil then P^.SetData(Data^);
      C := Desktop^.ExecView(P);
      if (C <> cmCancel) and (Data <> nil) then P^.GetData(Data^);
      Dispose(P, Done);
      ExecuteDialog := C;
   end;
end;

procedure TProgram.GetEvent(var Event  :TEvent);
var
   R               :TGRect;
   V               :PGroup;

function ContainsMouse(P               :PView)   :Boolean; far;
begin
   ContainsMouse := (P^.State and sfVisible <> 0) and
      P^.MouseInView(Event.Where);
end;

begin
   if Pending.What <> evNothing then
   begin
      Event := Pending; Pending.What := evNothing;
   end else
   begin
      GetMouseEvent(Event);
      if Event.What = evNothing then
      begin
         GetKeyEvent(Event);
         if Event.What = evNothing then
         begin
            Idle;
            Event.Where:=KeyMouse.MouseWhere;
         end;
      end;
   end;

   if ((Event.What=evNothing) or (Event.What=evMouseMove)) and
      Assigned(KeyMouse.SetMousePtr) then
   begin
      V:=@Self;
      repeat
         V:=PGroup(V^.FirstThat(@ContainsMouse));
      until (V=nil) or (not V^.IsGroup) or (V^.First=nil);

      if (V<>nil) and (V^.GetMousePointer<>KeyMouse.MouseImage) then
         KeyMouse.SetMousePtr(V^.GetMousePointer);
   end;

   if StatusLine <> nil then
   if (Event.What and evKeyDown <> 0) or
      (Event.What and evMouseDown <> 0) and
      (FirstThat(@ContainsMouse) = PView(StatusLine)) then
      StatusLine^.HandleEvent(Event);
end;

procedure TProgram.HandleEvent(var Event         :TEvent);
var
   I               :Word;
   C               :Char;
begin
   if Event.What = evKeyDown then
   begin
      C := GetAltChar(Event.KeyCode);
      if (C >= '1') and (C <= '9') then
      if Message(Desktop, evBroadCast, cmSelectWindowNum,
         Pointer(Byte(C) - $30)) <> nil then ClearEvent(Event);
   end;
   TGroup.HandleEvent(Event);
   if Event.What = evCommand then
   if Event.Command = cmQuit then
   begin
      EndModal(cmQuit);
      ClearEvent(Event);
   end;
end;

procedure TProgram.Idle;
begin
   if StatusLine <> nil then StatusLine^.Update;
   if CommandSetChanged then
   begin
      Message(@Self, evBroadcast, cmCommandSetChanged, nil);
      CommandSetChanged := False;
   end;
end;

procedure TProgram.InitDesktop;
var
   R               :TGRect;
begin
   GetExtent(R);
   if MenuBar <> nil then R.A.Y := MenuBar^.Origin.Y+MenuBar^.Size.Y;
   if StatusLine <> nil then R.B.Y := StatusLine^.Origin.Y-1;
   New(Desktop, Init(R));
end;

procedure TProgram.InitMenuBar;
begin
   MenuBar := New(PMenuBar, Standard(nil));
end;

procedure TProgram.InitScreen;
var
   FontSize        :TGPoint;
begin
   DragColor:=MaxColor; if DragColor>15 then DragColor:=15;

   case ScreenWidth of
      0..800:   FontSize.X:=8;
      else      FontSize.X:=ScreenWidth div (640 div 8);
   end;

   case ScreenHeight of
      0..349:   FontSize.Y:=8;
      350..399: FontSize.Y:=14;
      400..767: FontSize.Y:=16;
      else      FontSize.Y:=24;
   end;

   SystemFont.SetSize(FontSize.X shr 3, FontSize.Y shr 3);
   if FontSize.X and 7<>0 then
   begin
      SystemFont.SizeX:=FontSize.X; SystemFont.DivX:=8;
   end;
   if FontSize.Y and 7<>0 then
   begin
      SystemFont.SizeY:=FontSize.Y; SystemFont.DivY:=8;
   end;
   ErrorFont:=SystemFont;

   SBarSize.Y:=Min(Max(9, FontSize.Y), 16);
   SBarSize.X:=Min(Max(9, SBarSize.Y*DriverPtr^.AspX div DriverPtr^.AspY), 16);
   SBarPartSize:=SBarSize;

   HistoryWidth:=Max(Min(SBarSize.X, FontSize.Y), FontSize.X+1);

   MinWinSize.X:=Max(FontSize.X shl 2, SBarPartSize.X shl 2 + SBarSize.X) + 4;
   MinWinSize.Y:=FontSize.Y + SBarPartSize.Y shl 2 + Max(FontSize.Y, SBarSize.Y);

   Grid.X:=ScreenWidth shr 5; Grid.Y:=ScreenHeight shr 5;

   if CursorSize.X=0
   then CursorSize.Y:=Max(1, FontSize.Y shr 3)
   else CursorSize.X:=Max(1, FontSize.X shr 3);

   if CursorInsSize.X=0
   then CursorInsSize.Y:=Max(3, 1 + FontSize.Y shr 2)
   else CursorInsSize.X:=Max(3, 1 + FontSize.X shr 2);

   if ScreenWidth>=512 then MickeyPixelX:=1 else MickeyPixelX:=2;
   if ScreenHeight>200 then MickeyPixelY:=1 else MickeyPixelY:=2;
end;

procedure TProgram.InitStatusLine;
begin
   New(StatusLine, Standard(
     NewStatusDef(0, $FFFF,
       NewStatusKey('~Alt-X~ Exit', kbAltX, cmQuit,
       StdStatusKeys(nil)), nil)));
end;

function TProgram.InsertWindow(P       :PWindow) :PWindow;
begin
   InsertWindow := nil;
   if ValidView(P) <> nil then
   if CanMoveFocus then
   begin
      Desktop^.Insert(P); InsertWindow := P;
   end
   else Dispose(P, Done);
end;

procedure TProgram.OutOfMemory;
begin
end;

procedure TProgram.PutEvent(var Event  :TEvent);
begin
   Pending := Event;
end;

procedure TProgram.Run;
begin
   Execute;
end;

function  TProgram.SetScreenMode(const Driver    :String;
                                 Mode            :Word)    :Boolean;
var
   R               :TGRect;
begin
   SetScreenMode:=True;
   if (DriverName=Driver) and (DriverMode=Mode) then Exit;

   if not Desktop^.Valid(cmClose) then
   begin
      SetScreenMode:=False; Exit;
   end;

   if Assigned(DoneMousePtr) then DoneMousePtr;
   DoneEvents;

   if Desktop<>nil then Dispose(Desktop, Done); Desktop:=nil;
   if MenuBar<>nil then Dispose(MenuBar, Done); MenuBar:=nil;
   if StatusLine<>nil then Dispose(StatusLine, Done); StatusLine:=nil;

   DoneVideo;
   if Driver<>'' then
   begin
      InitVideo(Driver, Mode);
      InitScreen;
      R.Assign(0, 0, ScreenWidth-1, ScreenHeight-1);
      ChangeBounds(R);

      InitStatusLine;
      InitMenuBar;
      InitDesktop;

      if Desktop <> nil then Insert(Desktop);
      if StatusLine <> nil then Insert(StatusLine);
      if MenuBar <> nil then Insert(MenuBar);

      if Assigned(InitMousePtr) then InitMousePtr;
      InitEvents;
   end;
end;

function TProgram.ValidView(P          :PView)   :PView;
begin
   ValidView := nil;
   if P <> nil then
   begin
      if LowMemory then
      begin
         Dispose(P, Done); OutOfMemory; Exit;
      end;
      if not P^.Valid(cmValid) then
      begin
         Dispose(P, Done); Exit;
      end;
      ValidView := P;
   end;
end;

{ TApplication }

constructor TApplication.Init(const Driver       :String;
                              Mode               :Word);
begin
   if ProgramLanguage=plNone then RunError(210);
{ Display "Object not initialized" message if no language unit linked.
  You must put a language unit name such as English or Russian in your
  program's Uses line. }

   InitMemory;
   InitVideo(Driver, Mode);
   if Assigned(InitMousePtr) then InitMousePtr;
   InitEvents;
   InitSysError;
   InitHistory;

   Inherited Init;
end;

destructor TApplication.Done;
begin
   Inherited Done;

   DoneHistory;
   DoneSysError;
   if Assigned(DoneMousePtr) then DoneMousePtr;
   DoneEvents;
   DoneVideo;
   DoneMemory;
end;

procedure TApplication.Cascade;
var
   R               :TGRect;
begin
   GetTileRect(R);
   if Desktop <> nil then Desktop^.Cascade(R);
end;

procedure TApplication.CloseAll;

   procedure DoClose(P       :PView); far;
   begin
      Message(P, evCommand, cmClose, nil);
   end;

begin
   with Desktop^ do
   begin
      LockDraw; ForEach(@DoClose); UnlockDraw; Redraw;
   end;
end;

procedure TApplication.DosShell;
begin
   DoneSysError;
   DoneEvents;
   if Assigned(DoneMousePtr) then DoneMousePtr;
   DoneVideo;
   DoneDosMem;
   WriteShellMsg;
   SwapVectors;
   Exec(GetEnv('COMSPEC'), '');
   SwapVectors;
   InitDosMem;
   InitVideo(DriverName, DriverMode);
   if Assigned(InitMousePtr) then InitMousePtr;
   InitEvents;
   InitSysError;
   Redraw;
   if DosError=8 then OutOfMemory;
end;

procedure TApplication.GetTileRect(var R         :TGRect);
begin
   Desktop^.GetExtent(R);
end;

procedure TApplication.HandleEvent(var Event     :TEvent);
begin
   Inherited HandleEvent(Event);
   case Event.What of
      evCommand:
      begin
         case Event.Command of
            cmCloseAll  :CloseAll;
            cmTile      :Tile;
            cmCascade   :Cascade;
            cmRefresh   :Redraw;
            cmDosShell  :DosShell;
         else
            Exit;
         end;
         ClearEvent(Event);
      end;
   end;
end;

procedure TApplication.Tile;
var
   R               :TGRect;
begin
   GetTileRect(R);
   if Desktop <> nil then Desktop^.Tile(R);
end;

procedure TApplication.WriteShellMsg;
begin
   PrintStr('Type EXIT to return...');
end;

{ App registration procedure }

procedure RegisterApp;
begin
   RegisterType(RBackground);
   RegisterType(RDesktop);
end;

{ Standard menus and status lines }

function StdStatusKeys(Next            :PStatusItem)       :PStatusItem;
begin
   StdStatusKeys :=
     NewStatusKey('', kbAltX, cmQuit,
     NewStatusKey('', kbF10, cmMenu,
     NewStatusKey('', kbAltF3, cmClose,
     NewStatusKey('', kbF5, cmZoom,
     NewStatusKey('', kbCtrlF5, cmResize,
     NewStatusKey('', kbF6, cmNext,
     NewStatusKey('', kbShiftF6, cmPrev,
     Next)))))));
end;

function StdFileMenuItems(Next         :PMenuItem)         :PMenuItem;
begin
   StdFileMenuItems :=
     NewItem('~N~ew', '', kbNoKey, cmNew, hcNew,
     NewItem('~O~pen...', 'F3', kbF3, cmOpen, hcOpen,
     NewItem('~S~ave', 'F2', kbF2, cmSave, hcSave,
     NewItem('S~a~ve as...', '', kbNoKey, cmSaveAs, hcSaveAs,
     NewItem('Save a~l~l', '', kbNoKey, cmSaveAll, hcSaveAll,
     NewLine(
     NewItem('~C~hange dir...', '', kbNoKey, cmChangeDir, hcChangeDir,
     NewItem('~D~OS shell', '', kbNoKey, cmDosShell, hcDosShell,
     NewItem('E~x~it', 'Alt+X', kbAltX, cmQuit, hcExit,
     Next)))))))));
end;

function StdEditMenuItems(Next         :PMenuItem)         :PMenuItem;
begin
   StdEditMenuItems :=
     NewItem('~U~ndo', '', kbAltBack, cmUndo, hcUndo,
     NewLine(
     NewItem('Cu~t~', 'Shift+Del', kbShiftDel, cmCut, hcCut,
     NewItem('~C~opy', 'Ctrl+Ins', kbCtrlIns, cmCopy, hcCopy,
     NewItem('~P~aste', 'Shift+Ins', kbShiftIns, cmPaste, hcPaste,
     NewItem('C~l~ear', 'Ctrl+Del', kbCtrlDel, cmClear, hcClear,
     Next))))));
end;

function StdWindowMenuItems(Next       :PMenuItem)         :PMenuItem;
begin
   StdWindowMenuItems :=
     NewItem('~T~ile', '', kbNoKey, cmTile, hcTile,
     NewItem('C~a~scade', '', kbNoKey, cmCascade, hcCascade,
     NewItem('Cl~o~se all', '', kbNoKey, cmCloseAll, hcCloseAll,
     NewItem('~R~efresh display', '', kbNoKey, cmRefresh, hcRefresh,
     NewLine(
     NewItem('~S~ize/Move','Ctrl+F5', kbCtrlF5, cmResize, hcResize,
     NewItem('~Z~oom', 'F5', kbF5, cmZoom, hcZoom,
     NewItem('~N~ext', 'F6', kbF6, cmNext, hcNext,
     NewItem('~P~revious', 'Shift+F6', kbShiftF6, cmPrev, hcPrev,
     NewItem('~C~lose', 'Alt+F3', kbAltF3, cmClose, hcClose,
     Next))))))))));
end;

end.
