
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Multi-Sprites Editor Unit                       }
{                                                       }
{       Copyright (c) 1994,95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit SEditor;
interface
{$R-,B-,V-}
uses
   Objects, Memory,
   GViews, GMenus, GRect, GPalette, GMsgBox, GApp,
   GraphDrv, GDrivers, Events, KeyMouse, Utils, General,
   RGB,
   ReadFmt, Hints;

const
   FileBuffer =    1024;
   MaxImageSizeX = 1024;
   MaxImageSizeY = 768;
   MaxImageSize =  64000+4;

   DefaultName =   'UNTITLED.SPR';
   DefaultColor =  255;

   MinScale =      1;
   MaxScale =      64;

   MaxUndo =       64;

   PaletteGrid     :TGPoint= (X:4; Y:4);
   PaletteCount    :TGPoint= (X:8; Y:32);

   cpMouseLeft =   #$F0;
   cpMouseRight =  #$F1;

   cpDark =        #$F4;
   cpLight =       #$F5;

   CPaletteBox =
   cpMain+cpLightGray + cpFrame+cpWhite +
   cpMouseLeft+cpWhite + cpMouseRight+cpBlack +
   cpDark+cpBlack + cpLight+cpWhite;

   CInfoBar =
   cpMain+cpLightGray + cpFrame+cpWhite +
   cpText+cpBlack +
   cpDark+cpBlack + cpLight+cpWhite;

   cmRemoveWindow= 4000;

   cmScaleDown =   $F0;
   cmScaleUp =     $F1;

   cmSetColor =    4100;
   cmGetColor =    4101;
   cmFillArea =    4102;

   cmRectangle =   4104;
   cmBar =         4105;
   cmLine =        4106;

   cmFlipX =       $F4;
   cmFlipY =       $F5;

   cmInfo =        5000;

const
   EditMode        :Word =   cmSetColor;

type
   PGEditorImage = ^TGEditorImage;
   TGEditorImage = object

      FormatId     :Integer;
      HeaderSize   :Word;
      Header       :Pointer;
      FileSize     :LongInt;
      Size         :TGPoint;
      Data         :Array [0..320*200-1] of Byte;

      function  ValidPos(Pos           :TGPoint) :Boolean;
      function  GetColor(Pos           :TGPoint) :Integer;
      procedure SetColor(Pos           :TGPoint;
                         Color         :Integer);
      function  FlipX                            :Boolean;
      function  FlipY                            :Boolean;
   end;

   PGEditorField = ^TGEditorField;
   TGEditorField =
   object(TView)

      Image        :PGEditorImage;
      Grid         :TGPoint;
      Modified     :Boolean;
      UndoCount    :Integer;
      UndoBuffer   :Array [1..MaxUndo] of Pointer;
      UndoBufferM  :Array [1..MaxUndo] of Boolean;

      ShowSteps    :Boolean;

      constructor Init(var R           :TGRect;
                       AImage          :PGEditorImage);
      destructor Done; virtual;

      procedure ArrangeGrid(var R      :TGRect);
      procedure CalcBounds(var R       :TGRect;
                           Delta       :TGPoint); virtual;
      procedure Draw; virtual;
      procedure DrawPos(Pos            :TGPoint);
      procedure HandleEvent(var Event  :TEvent); virtual;

      procedure StoreUndo;
      procedure Undo;
   end;

   PGEditor =      ^TGEditor;
   TGEditor =
   object(TWindow)

      Field        :PGEditorField;
      Image        :PGEditorImage;
      HScrollBar,
      VScrollBar   :PScrollBar;

      constructor Init(var Bounds      :TGRect;
                       const FileName  :String;
                       ANumber         :Integer);
      constructor InitNew(var Bounds   :TGRect;
                          ANumber      :Integer;
                          ImageSize    :TGPoint;
                          FormatId     :Integer);
      destructor Done; virtual;

      function  GetMenu(NextItem       :Pointer) :Pointer; virtual;

      procedure UpdateScrollBars;
      procedure UpdateCommands;

      procedure HandleEvent(var Event  :TEvent); virtual;
      procedure SetState(AState        :Word;
                         Enable        :Boolean); virtual;

      function  GetTitle(MaxSize       :Integer) :TTitleStr; virtual;
      function  Valid(Command          :Word)    :Boolean; virtual;

      procedure FreeImage;
      procedure NewImage(ImageSize     :TGPoint;
                         FormatId      :Integer);
      function  ConvertImage(FormatId  :Integer) :Boolean;
      procedure LoadImage;
      function  SaveImage(FileName     :PString) :Boolean;
   end;

   TColorView =
   object(TView)
      procedure DrawColor(var Bounds   :TGRect;
                          Color        :Integer);
   end;

   PPaletteBox =   ^TPaletteBox;
   TPaletteBox =
   object(TColorView)

      CLeft,
      CRight,
      CCurrent     :Integer;

      constructor Init(var Right       :TGRect);

      function  GetPalette                       :TPalette; virtual;

      procedure Draw; virtual;
      procedure HandleEvent(var Event  :TEvent); virtual;

      procedure DrawCurrent(Color      :Integer);
      procedure Update(Mouse           :TGPoint);
   end;

   PInfoBar =      ^TInfoBar;
   TInfoBar =
   object(TColorView)

      Modified     :Boolean;
      Pos          :TGPoint;
      Color        :Integer;
      CorrectData  :Boolean;

      function  GetPalette                       :TPalette; virtual;

      procedure Draw; virtual;
      procedure DrawPos;
   end;

var
   PaletteBox      :PPaletteBox;
   InfoBar         :PInfoBar;
   PaletteRGB      :TRGBPalette;

implementation
uses
   LPalette;

{ TGEditorImage }

function  TGEditorImage.ValidPos;
begin
   ValidPos:=(Pos.X>=0) and (Pos.Y>=0) and (Pos.X<Size.X) and (Pos.Y<Size.Y);
end;

function  TGEditorImage.GetColor;
begin
   if ValidPos(Pos)
   then GetColor:=Data[Word(Pos.Y)*Size.X+Word(Pos.X)] else GetColor:=0;
end;

procedure TGEditorImage.SetColor;
begin
   if ValidPos(Pos) then Data[Word(Pos.Y)*Size.X+Word(Pos.X)]:=Color;
end;

function  TGEditorImage.FlipX;
var
   FlipBuf         :Pointer;
begin
   FlipBuf:=MemAlloc(FileSize);
   if FlipBuf=nil then Application^.OutOfMemory else
   begin
      GraphDrv.TurnBlockX(@Size, FlipBuf);
      Move(FlipBuf^, Size, FileSize);
      FreeMem(FlipBuf, FileSize);
   end;
end;

function  TGEditorImage.FlipY;
var
   FlipBuf         :Pointer;
begin
   FlipBuf:=MemAlloc(FileSize);
   if FlipBuf=nil then Application^.OutOfMemory else
   begin
      GraphDrv.TurnBlockY(@Size, FlipBuf);
      Move(FlipBuf^, Size, FileSize);
      FreeMem(FlipBuf, FileSize);
   end;
end;

{ TGEditorField }

constructor TGEditorField.Init;
begin
   Inherited Init(R);
   Image:=AImage; ArrangeGrid(R);
   GrowMode:=gfGrowHiX+gfGrowHiY;
   EventMask:=EventMask or (evMouseMove+evMouseDown+evMouseUp);
end;

destructor TGEditorField.Done;
begin
   while UndoCount>0 do
   begin
      if UndoBuffer[UndoCount]<>nil then DisposeCache(UndoBuffer[UndoCount]);
      Dec(UndoCount);
   end;
   Inherited Done;
end;

procedure TGEditorField.CalcBounds;
begin
   Inherited CalcBounds(R, Delta);
   PGEditor(Owner)^.UpdateScrollBars;
   PaletteBox^.InitClipping;
end;

procedure TGEditorField.ArrangeGrid;
var
   S               :TGPoint;
begin
   if Image=nil then LongInt(Grid):=$00010001 else
   begin
      R.GetSize(S);

      Grid.X:=Max(S.X div Image^.Size.X, 4);
      Grid.Y:=Max(S.Y div Image^.Size.Y, 4);

      if (Image^.Size.X>64) or (Image^.Size.Y>64) then LongInt(Grid):=$00010001;

      Grid.X:=Min(Grid.X, Grid.Y); Grid.Y:=Grid.X;
      if LongInt(Grid)<>$00010001 then
      if ScreenWidth<=ScreenHeight then Grid.Y:=Grid.Y shl 1 else
      if ScreenHeight<200 then Grid.X:=Grid.X shl 1;
   end;

   if Owner<>nil then PGEditor(Owner)^.UpdateScrollBars;
end;

procedure TGEditorField.Draw;
var
   R               :TGRect;
   Pos, ScrollPos  :TGPoint;
begin
   ScrollPos.X:=PGEditor(Owner)^.HScrollBar^.Value;
   ScrollPos.Y:=PGEditor(Owner)^.VScrollBar^.Value;

   Pos.X:=-ScrollPos.X * Grid.X; Pos.Y:=-ScrollPos.Y * Grid.Y;
   DrawImage(Pos, Grid, @(Image^.Size), NullColor);
   Pos.Move(Image^.Size.X * Grid.X, Image^.Size.Y * Grid.Y);

   GetExtent(R); R.A.Y:=Pos.Y;
   if R.A.Y<=R.B.Y then DrawBar(R, GetColor(cpMain));
   R.A.X:=Pos.X; R.A.Y:=0; R.B.Y:=Pos.Y-1;
   if R.A.X<=R.B.X then DrawBar(R, GetColor(cpMain));
end;

procedure TGEditorField.DrawPos;
var
   R, Clip         :TGRect;
begin
   if Pos.Rect(0, 0, Image^.Size.X-1, Image^.Size.Y-1) then
   begin
      R.A.X:=(Pos.X-PGEditor(Owner)^.HScrollBar^.Value)*Grid.X;
      R.A.Y:=(Pos.Y-PGEditor(Owner)^.VScrollBar^.Value)*Grid.Y;
      R.SetSize(Grid);
      DrawBar(R, Image^.GetColor(Pos));
   end;
end;

var
   CommonField     :PGEditorField;

procedure DoPutPixel(x, y, Color       :Integer); far;
var
   Pos             :TGPoint;
begin
   Pos.X:=x; Pos.Y:=y;
   with CommonField^ do
   begin
      Image^.SetColor(Pos, Color);
      if ShowSteps then DrawPos(Pos);
   end;
end;

procedure TGEditorField.HandleEvent;
var
   Pos             :TGPoint;
   CurrentColor    :Integer;
   R               :TGRect;

const
   StartPos        :TGPoint =
   (X:MaxInt; Y:MaxInt);
   LastPos         :TGPoint =
   (X:MaxInt; Y:MaxInt);

   StartButtons    :Integer =
   0;

procedure FloodFill(cf                 :Integer);
type
   TCoordArray =   Array [1..$4000] of Integer;
var
   SX, SY, SXNew, SYNew,
   Temp, MarkMem                       :^TCoordArray;
   x, y, cb,
   i, n, NNew, MaxX, MaxY, MaxIndex    :Integer;
   ArrayMem                            :Word;
   BufSize                             :LongInt;
   P                                   :TGPoint;

procedure FillPixel(x, y               :Integer);
begin
   if NNew>=MaxIndex then Exit;
   P.X:=x; P.Y:=y;
   Image^.SetColor(P, cf); DrawPos(P);
   Inc(NNew); SXNew^[NNew]:=x; SYNew^[NNew]:=y;
end;

function  GetPixel(x, y                :Integer) :Integer;
begin
   P.X:=x; P.Y:=y;
   GetPixel:=Image^.GetColor(P);
end;

begin
   x:=Pos.X; y:=Pos.Y;

   BufSize:=MaxAvail shr 1;
   if BufSize>64000 then BufSize:=64000;

   if BufSize<4096 then
   begin
      Application^.OutOfMemory; Exit;
   end;

   MaxIndex:=BufSize shr 3; ArrayMem:=MaxIndex shl 1;

   Mark(MarkMem);
   GetMem(SX, ArrayMem); GetMem(SY, ArrayMem);
   GetMem(SXNew, ArrayMem); GetMem(SYNew, ArrayMem);

   GetExtent(R);
   HideMouseLocal(R);

   Application^.HelpCtx:=hcWait;
   StatusLine^.Update;

   MaxX:=Image^.Size.X-1; MaxY:=Image^.Size.Y-1;
   NNew:=0;
   cb:=GetPixel(x, y);
   if cb<>cf then
   begin
      FillPixel(x, y);
      repeat
         n:=NNew; NNew:=0;
         Temp:=SX; SX:=SXNew; SXNew:=Temp; Temp:=SY; SY:=SYNew; SYNew:=Temp;
         for i:=1 to n do
         begin
            x:=sx^[i]; y:=sy^[i];
            if (x>0) and (GetPixel(x-1, y)=cb) then FillPixel(x-1, y);
            if (x<MaxX) and (GetPixel(x+1, y)=cb) then FillPixel(x+1, y);
            if (y>0) and (GetPixel(x, y-1)=cb) then FillPixel(x, y-1);
            if (y<MaxY) and (GetPixel(x, y+1)=cb) then FillPixel(x, y+1);
         end;
      until NNew=0;
   end;

   Application^.HelpCtx:=hcNoContext;

   ShowMouseRect;
   Release(MarkMem);

   Modified:=True;
end;

procedure DrawDragObject(Pos           :TGPoint);
var
   R               :TGRect;
   S               :TGPoint;
begin
   S.X:=PGEditor(Owner)^.HScrollBar^.Value;
   S.Y:=PGEditor(Owner)^.VScrollBar^.Value;

   R.A.X:=(StartPos.X-S.X)*Grid.X; R.A.Y:=(StartPos.Y-S.Y)*Grid.Y;
   R.B.X:=(Pos.X-S.X)*Grid.X; R.B.Y:=(Pos.Y-S.Y)*Grid.Y;
   R.Move(Grid.X shr 1, Grid.Y shr 1);

   GraphDrv.DriverPtr^.PutMode:=GraphDrv.XorPut;
   case EditMode of
      cmRectangle, cmBar:
         if (R.A.X=R.B.X) or (R.A.Y=R.B.Y)
         then DrawLine(R, DragColor) else DrawRect(R, DragColor);
      cmLine:
         DrawLine(R, DragColor);
   end;
   GraphDrv.DriverPtr^.PutMode:=GraphDrv.NormalPut;
end;

procedure DoColor(var Color            :Integer);

procedure DoLine(x1, y1, x2, y2        :Integer);
begin
   CommonField:=@Self;
   GViews.DrawAnyLine(x1, y1, x2, y2, Color, @DoPutPixel, False, $FFFF);
end;

begin
   case EditMode of
      cmSetColor:
      begin
         Image^.SetColor(Pos, Color);
         if Color<>CurrentColor then
         begin
            Modified:=True; DrawPos(Pos); CurrentColor:=Color;
         end;
      end;
      cmGetColor:
      begin
         Color:=Image^.GetColor(Pos);
         EditMode:=cmSetColor;
         InfoBar^.DrawView;
      end;
      cmFillArea:
      begin
         repeat until not MouseEvent(Event, evMouseUp);
         FloodFill(Color);
      end;
      else
      begin
         if LastPos.X<>MaxInt then DrawDragObject(LastPos);
         LastPos.X:=MaxInt;
         if StartPos.X<>MaxInt then
         case EditMode of
            cmRectangle:
            begin
               ShowSteps:=True;
               DoLine(StartPos.X, StartPos.Y, Pos.X, StartPos.Y);
               DoLine(Pos.X, StartPos.Y, Pos.X, Pos.Y);
               DoLine(Pos.X, Pos.Y, StartPos.X, Pos.Y);
               DoLine(StartPos.X, Pos.Y, StartPos.X, StartPos.Y);
            end;
            cmBar:
            begin
               Application^.HelpCtx:=hcWait;
               StatusLine^.Update;

               ShowSteps:=False;
               ArrangeInts(StartPos.Y, Pos.Y);
               while StartPos.Y<=Pos.Y do
               begin
                  DoLine(StartPos.X, StartPos.Y, Pos.X, StartPos.Y);
                  Inc(StartPos.Y);
               end;
               DrawView;

               Application^.HelpCtx:=hcNoContext;
            end;
            cmLine:
            begin
               ShowSteps:=True;
               DoLine(StartPos.X, StartPos.Y, Pos.X, Pos.Y);
            end;
         end;
         StartPos.X:=MaxInt;
         Modified:=True;
      end;
   end;
end;

begin
   Inherited HandleEvent(Event);
   if (Event.What and (evMouseDown+evMouseUp+evMouseMove)<>0) and
      (Owner^.State and sfFocused<>0) then
   begin
      if ((EditMode< cmRectangle) and (Event.What=evMouseDown)) or
         ((EditMode>=cmRectangle) and (Event.What=evMouseUp)) then StoreUndo;

      MakeLocal(Event.Where, Event.Where);
      Pos.X:=Event.Where.X div Grid.X; Pos.Y:=Event.Where.Y div Grid.Y;

      Inc(Pos.X, PGEditor(Owner)^.HScrollBar^.Value);
      Inc(Pos.Y, PGEditor(Owner)^.VScrollBar^.Value);

      if (Event.What=evMouseDown) and (not Image^.ValidPos(Pos)) then Exit;

      CurrentColor:=Image^.GetColor(Pos);

      if ((EditMode< cmRectangle) and (Event.What<>evMouseUp)) or
         ((EditMode>=cmRectangle) and (Event.What= evMouseUp)) then
      begin
         if Event.What=evMouseUp then Event.Buttons:=StartButtons;
         if Event.Buttons<>0 then
         if Event.Buttons and mbRightButton=0
         then DoColor(PaletteBox^.CLeft) else DoColor(PaletteBox^.CRight);
      end;

      if EditMode>=cmRectangle then
      case Event.What of
         evMouseDown:
         if Image^.ValidPos(Pos) then
         begin
            StartPos:=Pos; LastPos.X:=MaxInt;
            StartButtons:=Event.Buttons;
         end;
         evMouseMove:
         if (StartPos.X<>MaxInt) and (LongInt(Pos)<>LongInt(LastPos)) then
         begin
            if LastPos.X<>MaxInt then DrawDragObject(LastPos);
            DrawDragObject(Pos);
            LastPos:=Pos;
         end;
      end;

      if Image^.ValidPos(Pos) and
         ((not InfoBar^.CorrectData) or
         (LongInt(InfoBar^.Pos)<>LongInt(Pos)) or
         (InfoBar^.Color<>CurrentColor)) then
      begin
         InfoBar^.Modified:=Modified;
         InfoBar^.Pos:=Pos; InfoBar^.Color:=CurrentColor;
         InfoBar^.CorrectData:=True;
         InfoBar^.DrawPos;
      end;

      ClearEvent(Event);
   end;
end;

procedure TGEditorField.StoreUndo;
begin
   if UndoCount<MaxUndo then
   begin
      Inc(UndoCount);
      NewCache(UndoBuffer[UndoCount], Image^.Size.X*Image^.Size.Y);
      if UndoBuffer[UndoCount]=nil then Dec(UndoCount) else
      begin
         Move(Image^.Data, UndoBuffer[UndoCount]^, Image^.Size.X*Image^.Size.Y);
         UndoBufferM[UndoCount]:=Modified;
      end;
   end;
end;

procedure TGEditorField.Undo;
begin
   while (UndoCount>0) and (UndoBuffer[UndoCount]=nil) do Dec(UndoCount);
   if UndoCount>0 then
   begin
      Move(UndoBuffer[UndoCount]^, Image^.Data, Image^.Size.X*Image^.Size.Y);
      Modified:=UndoBufferM[UndoCount];
      DisposeCache(UndoBuffer[UndoCount]);
      Dec(UndoCount);
      DrawView;

      InfoBar^.Modified:=Modified;
      if not Modified then InfoBar^.DrawPos;
   end
   else MessageBox(^M^M^C'Unable to undo.', nil, mfError+mfOKButton);
end;

{ TGEditor }

constructor TGEditor.Init;
var
   R               :TGRect;
   FileSize        :LongInt;
begin
   Inherited Init(Bounds, FileName, ANumber);
   if FileName<>'' then
   begin
      LoadImage; if Image=nil then Fail;
   end else Title:=NewStr(DefaultName);

   HScrollBar:=StandardScrollBar(sbHorizontal+sbHandleKeyboard);
   VScrollBar:=StandardScrollBar(sbVertical+sbHandleKeyboard);
   HScrollBar^.SetStep(8, 1); VScrollBar^.SetStep(8, 1);

   GetInterior(R);
   New(Field, Init(R, Image)); Insert(Field);
end;

constructor TGEditor.InitNew;
var
   R               :TGRect;
begin
   Init(Bounds, '', ANumber);
   NewImage(ImageSize, FormatId);
   if Image=nil then Fail;
   Field^.Image:=Image; Field^.GetExtent(R); Field^.ArrangeGrid(R);
end;

destructor TGEditor.Done;
begin
   FreeImage;
   Inherited Done;
   Message(Application, evCommand, cmRemoveWindow, @Self);
end;

function  TGEditor.GetMenu;
begin
   GetMenu:=
     Inherited GetMenu(
     NewLine(
     NewItem('Scale ~d~own', 'Gray -', kbGrayMinus, cmScaleDown, hcScaleDown,
     NewItem('Scale ~u~p', 'Gray +', kbGrayPlus, cmScaleUp, hcScaleUp,
     NewLine(
     NewItem('~I~nformation...', '', kbNoKey, cmInfo, hcInfo,
   nil))))));
end;

procedure TGEditor.UpdateScrollBars;
var
   R               :TGRect;
begin
   if Image=nil then
   begin
      HScrollBar^.SetRange(0, 0); VScrollBar^.SetRange(0, 0); Exit;
   end;
   GetInterior(R);
   HScrollBar^.SetRange(0, Image^.Size.X - R.SizeX div Field^.Grid.X);
   VScrollBar^.SetRange(0, Image^.Size.Y - R.SizeY div Field^.Grid.Y);
   HScrollBar^.Show; VScrollBar^.Show;

   UpdateCommands;
end;

procedure TGEditor.UpdateCommands;
var
   Cmds            :TCommandSet;
begin
   if State and sfSelected = 0 then Exit;
   DisableCommands([cmScaleDown, cmScaleUp]);
   Cmds:=[];
   if Field^.Grid.X>MinScale then Cmds:=Cmds+[cmScaleDown];
   if Field^.Grid.X<MaxScale then Cmds:=Cmds+[cmScaleUp];
   EnableCommands(Cmds);
end;

procedure TGEditor.HandleEvent;
begin
   Inherited HandleEvent(Event);
   if Event.What=evBroadcast then
   case Event.Command of
      cmScrollBarChanged:
         if Field<>nil then Field^.DrawView;
      cmReceivedFocus:
         UpdateScrollBars;
   end;
end;

procedure TGEditor.SetState;
begin
   Inherited SetState(AState, Enable);
   if (AState = sfSelected) or ((AState = sfExposed) and
      (State and sfSelected <> 0)) then
   begin
      UpdateCommands;
      InfoBar^.Modified:=Field^.Modified; InfoBar^.DrawPos;
   end;
end;

function  TGEditor.GetTitle;
var
   i               :Integer;
begin
   if Title=nil then GetTitle:='Untitled' else
   if Length(Title^)<=MaxSize then GetTitle:=Title^ else
   begin
      i:=Length(Title^);
      while (i>0) and (Title^[i]<>'\') do Dec(i);
      if Length(Title^)-i<=MaxSize then GetTitle:=Copy(Title^, i+1, 255)
      else GetTitle:='';
   end;
end;

function  TGEditor.Valid;
begin
   if (Command=cmClose) or (Command=cmQuit) then
   if Field^.Modified then
   begin
      Command:=MessageBox(
         ^M^C'The image '+GetTitle(12)+
         ^M^M^C'has been modified. Save?',
         nil, mfConfirmation+mfYesButton+mfNoButton+mfCancelButton);
      if Command=cmYes then
      if not SaveImage(Title) then Command:=cmCancel;
      Valid:=(Command<>cmCancel);
   end else Valid:=True
   else Valid:=Inherited Valid(Command);
end;

procedure TGEditor.FreeImage;
begin
   if Image<>nil then
   begin
      FreeMem(Image^.Header, Image^.HeaderSize);
      FreeMem(Image, Image^.FileSize + (4 + 4+2 + 2)); Image:=nil;
   end;
end;

procedure TGEditor.NewImage;
var
   FileSize        :Word;
begin
   FreeImage;
   FileSize:=Word(ImageSize.X)*ImageSize.Y + 4;
   Image:=MemAlloc(FileSize + (4 + 4+2 + 2));
   if Image=nil then
   begin
      Application^.OutOfMemory; Exit;
   end;
   Image^.FileSize:=FileSize; Image^.Size:=ImageSize;

   if not ConvertImage(FormatId) then
   begin
      FreeMem(Image, FileSize + (4 + 4+2 + 2)); Exit;
   end;

   FillChar(Image^.Data, FileSize-4, DefaultColor);
end;

function  TGEditor.ConvertImage;
var
   Header          :PByteArray;
   FileSizeH       :Word;
   ImageSizeH      :TGPoint;
begin
   ConvertImage:=False;
   with Formats[FormatId] do
   begin
      Header:=MemAlloc(HeaderSize);
      if Header=nil then
      begin
         Application^.OutOfMemory; Exit;
      end;
      Image^.Header:=Header; Image^.HeaderSize:=HeaderSize;
      FillChar(Header^, HeaderSize, 0);
      if FormatId<>nil then Move(FormatId^[1], Header^, Length(FormatId^));
      if FSizeOfs>=0 then
      begin
         FileSizeH:=Image^.FileSize-4-FSizeInc;
         Move(FileSizeH, Header^[FSizeOfs], 2);
      end;
      ImageSizeH.X:=Image^.Size.X-SizesInc;
      ImageSizeH.Y:=Image^.Size.Y-SizesInc;
      Move(ImageSizeH.X, Header^[SizesOfs], 2);
      Move(ImageSizeH.Y, Header^[SizesOfs+SizesSize], 2);
   end;
   Image^.FormatId:=FormatId;
   ConvertImage:=True;
end;

procedure TGEditor.LoadImage;
var
   S               :TBufStream;
   FileSize        :LongInt;
   ImageSizeX,
   ImageSizeY      :LongInt;
   HeaderSize,
   i               :Integer;
   Header          :PByteArray;
   Id              :String[31];
begin
   if Title=nil then Exit;
   S.Init(Title^, stOpenRead, FileBuffer);

   if (S.GetSize<MaxHeaderSize) and (S.GetSize>0)
   then HeaderSize:=S.GetSize else HeaderSize:=MaxHeaderSize;

   Header:=MemAlloc(HeaderSize);
   if Header=nil then
   begin
      Application^.OutOfMemory; S.Done; Exit;
   end;
   S.Read(Header^, HeaderSize);
   Move(Header^, Id[1], SizeOf(Id)-1);

   i:=1;
   while i<=FormatsCount do
   with Formats[i] do
   begin
      if FormatId<>nil then Id[0]:=FormatId^[0];

      if FSizeOfs<0 then FileSize:=S.GetSize else
      begin
         FileSize:=0; Move(Header^[FSizeOfs], FileSize, FSizeSize);
      end;
      Inc(FileSize, FSizeInc);

      ImageSizeX:=0; Move(Header^[SizesOfs], ImageSizeX, SizesSize);
      ImageSizeY:=0; Move(Header^[SizesOfs+SizesSize], ImageSizeY, SizesSize);
      Inc(ImageSizeX, SizesInc); Inc(ImageSizeY, SizesInc);

      if ((FormatId=nil) or (Id=FormatId^)) and
         (FileSize>0) and (ImageSizeX>0) and (ImageSizeY>0) and
         (FileSize=ImageSizeX*ImageSizeY) then Break;

      Inc(i);
   end;
   FreeMem(Header, HeaderSize);

   if (S.Status=stOk) and (i>FormatsCount) then
   begin
      S.Done; MessageBox(^M^C'Unknown image format.', nil, mfError+mfOKButton);
      Exit;
   end;

   if (ImageSizeX>MaxImageSizeX) or (ImageSizeY>MaxImageSizeY) or
      (ImageSizeX*ImageSizeY+4>MaxImageSize) then
   begin
      S.Done; MessageBox(^M^C'Image is too large.', nil, mfError+mfOKButton);
      Exit;
   end;

   if S.Status=stOk then
   begin
      FreeImage;

      HeaderSize:=Formats[i].HeaderSize;
      Header:=MemAlloc(HeaderSize);
      if Header=nil then
      begin
         S.Done; Application^.OutOfMemory; Exit;
      end;
      S.Seek(0);
      S.Read(Header^, HeaderSize);

      Image:=MemAlloc(FileSize + (4+4 + 4+2 + 2));
      if Image=nil then
      begin
         FreeMem(Header, HeaderSize); S.Done; Application^.OutOfMemory; Exit;
      end;
      Image^.FileSize:=FileSize+4;
      Image^.Size.X:=ImageSizeX; Image^.Size.Y:=ImageSizeY;
      Image^.Header:=Header; Image^.HeaderSize:=HeaderSize;
      S.Read(Image^.Data, FileSize);
      Image^.FormatId:=i;
   end;

   with Formats[i] do
   if PaletteOfs>=0 then
   begin
      S.Seek(PaletteOfs);
      LoadPalette(S, PaletteWin);
   end;

   if S.Status<>stOk then
   begin
      FreeImage; MessageBox(^M^C'Unable to load image.', nil, mfError+mfOKButton);
   end;
   S.Done;

   if Formats[i].Reverse then
   if not Image^.FlipY then
   begin
      FreeImage; Application^.OutOfMemory;
   end;

   Field^.Modified:=False;
end;

function  TGEditor.SaveImage;
var
   S               :TBufStream;
   Ok              :Boolean;
begin
   SaveImage:=False;
   if FileName=nil then Exit;
   S.Init(FileName^, stCreate, FileBuffer);
   S.Write(Image^.Header^, Image^.HeaderSize);
   if Formats[Image^.FormatId].Reverse then
   begin
      Ok:=Image^.FlipY;
      if not Ok then Application^.OutOfMemory;
   end;
   S.Write(Image^.Data, Image^.FileSize-4);
   if Formats[Image^.FormatId].Reverse and Ok then Image^.FlipY;
   if S.Status<>stOk then
      MessageBox(^M^C'Unable to save image.', nil, mfError+mfOKButton);
   S.Done;
   Field^.Modified:=False;
   SaveImage:=True;
end;

{ TColorView }

procedure TColorView.DrawColor;
var
   P               :TGPoint;
   S               :String[7];
   TextColor       :Integer;
begin
   HideMouseLocal(Bounds);
   DrawBar(Bounds, Color);
   Str(Color, S);
   P.X:=Bounds.CenterX; P.Y:=Bounds.CenterY + 1;
   if PaletteRGB[Color].R+PaletteRGB[Color].G+PaletteRGB[Color].B > 3*64 div 2
   then TextColor:=GetColor(cpDark) else TextColor:=GetColor(cpLight);
   DrawText(P, S, jsCenter, nil, TextColor);
   ShowMouseRect;
end;

{ TPaletteBox }

constructor TPaletteBox.Init;
begin
   Right.A.X:=Right.B.X-PaletteGrid.X*PaletteCount.X-1;
   Inherited Init(Right);
   EventMask:=EventMask or evMouseMove;
   CLeft:=GetColor(cpMouseLeft); CRight:=GetColor(cpMouseRight);
   CCurrent:=-1;
end;

function  TPaletteBox.GetPalette;
begin
   GetPalette:=CPaletteBox;
end;

procedure TPaletteBox.Draw;
var
   Pos             :TGPoint;
   R               :TGRect;
   Color           :Byte;
   Colors          :Record
      Size         :TGPoint;
      Image        :Array [0..255] of Byte;
   end;
begin
   Color:=0;
   Colors.Size:=PaletteCount;
   for Pos.X:=0 to PaletteCount.X-1 do
   for Pos.Y:=0 to PaletteCount.Y-1 do
   begin
      Colors.Image[Pos.Y * PaletteCount.X + Pos.X]:=Color; Inc(Color);
   end;

   GetExtent(R);
   DrawRect(R, GetColor(cpFrame));
   R.Grow(-1, -1);
   DrawImage(R.A, PaletteGrid, @Colors, NullColor);
   R.A.Y:=PaletteCount.Y * PaletteGrid.Y + 1;
   DrawBar(R, GetColor(cpMain));

   DrawCurrent(CCurrent);
end;

procedure TPaletteBox.DrawCurrent;
var
   R               :TGRect;
begin
   GetExtent(R); R.Grow(-1, -1); R.A.Y:=R.B.Y - CharHeight - 2;
   if Color<0 then DrawBar(R, GetColor(cpMain)) else
   begin
      DrawRect(R, GetColor(cpDark)); R.Grow(-1, -1);
      DrawColor(R, Color);
   end;
   CCurrent:=Color;
end;

procedure TPaletteBox.Update;
var
   Extent          :TGRect;
begin
   if CCurrent<0 then Exit;
   GetExtent(Extent);
   MakeLocal(Mouse, Mouse);
   if not Extent.Contains(Mouse) then DrawCurrent(-1);
end;

procedure TPaletteBox.HandleEvent;
var
   MouseColor      :Integer;

function  CalcColor                    :Integer;
var
   Pos             :TGPoint;
begin
   MakeLocal(Event.Where, Event.Where); Event.Where.Move(-1, -1);
   Pos.X:=Event.Where.X div PaletteGrid.X; Pos.Y:=Event.Where.Y div PaletteGrid.Y;
   if (Pos.X<0) or (Pos.X>=PaletteCount.X) or
      (Pos.Y<0) or (Pos.Y>=PaletteCount.Y) then CalcColor:=-1
   else CalcColor:=Pos.X*PaletteCount.Y+Pos.Y;
   ClearEvent(Event);
end;

procedure SetColor(var Color           :Integer);
var
   Last            :Integer;
begin
   Last:=Color; Color:=CalcColor;
   if Color<0 then Color:=Last else
   if Color<>Last then InfoBar^.DrawView;
end;

begin
   Inherited HandleEvent(Event);
   if Event.What and evMouseDown<>0 then
   if Event.Buttons and mbRightButton=0
   then SetColor(CLeft) else SetColor(CRight) else
   if Event.What and evMouseMove<>0 then
   begin
      MouseColor:=CalcColor;
      if MouseColor<>CCurrent then DrawCurrent(MouseColor);
   end;
end;

{ TInfoBar }

function  TInfoBar.GetPalette;
begin
   GetPalette:=CInfoBar;
end;

procedure TInfoBar.Draw;
var
   R               :TGRect;
   P               :TGPoint;
   S               :String[15];
   V               :String[5];
begin
   GetExtent(R);
   DrawRect(R, GetColor(cpFrame));
   R.Grow(-1, -1);
   DrawBar(R, GetColor(cpMain));
   DrawPos;

   case EditMode of
      cmSetColor:  S:='Set';
      cmGetColor:  S:='Get';
      cmFillArea:  S:='Fill';
      cmRectangle: S:='Rectangle';
      cmBar:       S:='Bar';
      cmLine:      S:='Line';
   else
      S:='';
   end;
   P.X:=11*CharWidth; P.Y:=R.CenterY + 1;
   DrawText(P, S, jsCenterY, nil, GetColor(cpText));

   R.Grow(-1, -1); R.A.X:=R.B.X - 3*CharWidth - 3;
   DrawColor(R, PaletteBox^.CRight);
   R.Move(-R.SizeX-1, 0);
   DrawColor(R, PaletteBox^.CLeft);

   Str(DriverPtr^.SizeX, S);
   Str(DriverPtr^.SizeY, V);
   P.X:=R.A.X-CharWidth;
   DrawText(P, S+'x'+V, jsRight+jsCenterY, nil, GetColor(cpText));
end;

procedure TInfoBar.DrawPos;
var
   R               :TGRect;
   S               :String[23];
   P               :Record
      Modified,
      PosX, PosY   :LongInt;
   End;
begin
   if not Exposed then Exit;
   GetExtent(R);
   R.Grow(-1, -1);
   R.A.X:=8; R.B.X:=R.A.X + 9*CharWidth -1;
   HideMouseLocal(R);
   DrawBar(R, GetColor(cpMain));
   if CorrectData then
   begin
      if Modified then Char(P.Modified):=#15 else Char(P.Modified):=' ';
      P.PosX:=Pos.X; P.PosY:=Pos.Y;
      FormatStr(S, '%c%4d:%-3d', P);

      R.A.Y:=R.CenterY + 1;
      DrawText(R.A, S, jsCenterY, nil, GetColor(cpText));

      if Color<>PaletteBox^.CCurrent then PaletteBox^.DrawCurrent(Color);
   end;
   ShowMouseRect;
end;

begin
   MaxBufMem:=MaxHeapSize;
end.
