/******************************************************************************
Module name: SuperCls.c
Written by: Jeffrey Richter
Notices: Copyright (c) 1995 Jeffrey Richter
Purpose: Utility functions to help with window superclassing.
******************************************************************************/


#include "..\Win95ADG.h"          /* See Appendix A for details */
#include <Windows.h>
#include <WindowsX.h>
#pragma warning(disable: 4001)    /* Single-line comment */
#include "SuperCls.h"


///////////////////////////////////////////////////////////////////////////////


// Enough class extra bytes are added to the superclass to accommodate
// the data represented by the following structure. These additional data make it
// easier for a superclass to add class and window extra bytes of its own to
// the superclass. After the superclass has been registered, using the 
// functions in this module, the layout of the superclass's extra bytes is as 
// follows:
//
// Range:    0 to (cbClsExtraBaseCls - 1)                
// Contents: Base class's class extra bytes
// Range:    cbClsExtraBaseCls to (GCL_CBCLSEXTRA - SUPERCLS_BASECLSINFO - 1)
// Contents: Additional class extra bytes desired by superclass
// Range:    (GCL_CBCLSEXTRA - SUPERCLS_BASECLSINFO) to (GCL_CBCLSEXTTA - 1)
// Contents: Data represented by the following SUPERCLS_BASECLSINFO structure
//
typedef struct {
   WNDPROC pfnWndProcBaseCls;     // Base class's window procedure
   int     cbClsExtraBaseCls;     // Class extra bytes for base class
   int     cbWndExtraBaseCls;     // Window extra bytes for base class
} SUPERCLS_BASECLSINFO;


///////////////////////////////////////////////////////////////////////////////


// Helper macro is used to get the offset of a SUPERCLS_BASECLSINFO member from 
// within the class's extra bytes.
#define SuperCls_BaseClsInfoIndex(cbClsTotal, member)   \
   (cbClsTotal - sizeof(SUPERCLS_BASECLSINFO) +         \
   adgMEMBEROFFSET(SUPERCLS_BASECLSINFO, member))


///////////////////////////////////////////////////////////////////////////////


ATOM WINAPI SuperCls_RegisterClassEx (
   PWNDCLASSEX pwc,              // lpfnWndClass member is base class WndProc.
   WNDPROC pfnWndProcSuperCls,   // Address of superclass WndProc function
   int cbClsAdditional,          // # of class extra bytes for super class
   int cbWndAdditional) {        // # of window extra bytes for super class

   HWND hwnd;
   ATOM atomSuperClass;
   WNDPROC pfnWndProcBaseCls = pwc->lpfnWndProc;
   int cbClsExtraBaseCls     = pwc->cbClsExtra;
   int cbWndExtraBaseCls     = pwc->cbWndExtra;

   // Add to the cbClsExtra member the number of class extra bytes 
   // desired by the superclass plus the number of extra bytes 
   // required by the SUPERCLS_BASECLSINFO structure.
   pwc->cbClsExtra += cbClsAdditional + sizeof(SUPERCLS_BASECLSINFO);

   // Add to the cbWndExtra member the number of window extra bytes
   // desired by the superclass.
   pwc->cbWndExtra += cbWndAdditional;

   // In order for the superclass window procedure to manipulate
   // any of the superclass's class and window extra bytes successfully, the 
   // SUPERCLS_BASECLSINFO data members in the superclass's class extra bytes
   // must be initialized first.  Because there is no Win32 function that 
   // allows us to change a class's extra bytes without creating a window 
   // first, we need to hack the solution.  This is done by registering the 
   // class using DefWindowProc as the window procedure.  After the class 
   // extra bytes have been initialized, DefWindowProc is replaced by the 
   // desired superclass window procedure.
   pwc->lpfnWndProc = DefWindowProc;

   // Register the superclass.
   atomSuperClass = RegisterClassEx(pwc);
   if (atomSuperClass == INVALID_ATOM)
      return(atomSuperClass);

   // Now, we must complete the class's initialization by setting its 
   // class extra bytes.  Unfortunately, we must have a window handle 
   // to change a class's extra bytes.  So, we create a window of the class.
   // This is a dummy window whose messages are processed by DefWindowProc.
   hwnd = CreateWindowEx(0, MAKEINTATOM(atomSuperClass), NULL,
      0, 0, 0, 0, 0, NULL, NULL, pwc->hInstance, NULL);

   if (!IsWindow(hwnd)) {

      // If the window could not be created, unregister the superclass
      // and return INVALID_ATOM to the caller.
      UnregisterClass(MAKEINTATOM(atomSuperClass), pwc->hInstance);
      atomSuperClass = INVALID_ATOM;
   } else {

      // Initialize the data represented by the SUPERCLS_BASECLSINFO structure.
      SetClassLong(hwnd, 
         SuperCls_BaseClsInfoIndex(pwc->cbClsExtra, pfnWndProcBaseCls),
            (LONG) pfnWndProcBaseCls);
      SetClassLong(hwnd, 
         SuperCls_BaseClsInfoIndex(pwc->cbClsExtra, cbClsExtraBaseCls),
            cbClsExtraBaseCls);
      SetClassLong(hwnd, 
         SuperCls_BaseClsInfoIndex(pwc->cbClsExtra, cbWndExtraBaseCls),
            cbWndExtraBaseCls);

      // Now, we can set the class's window procedure to point to the 
      // desired superclass window procedure.
      SetClassLong(hwnd, GCL_WNDPROC, (LONG) pfnWndProcSuperCls);

      // NOTE: At this point, any windows of this class that are created
      // will have their messages processed by the proper superclass WndProc.

      // Because all the class extra bytes are set, we can destroy the dummy
      // window.  The WM_DESTROY/WM_NCDESTROY messages will be sent to 
      // DefWindowProc because the call to SetClassLong above does NOT affect
      // the window procedure address that was associated with the dummy 
      // window when it was created.
      DestroyWindow(hwnd);
   }

   // Return the atom of the registered class to the caller.
   return(atomSuperClass);
}


///////////////////////////////////////////////////////////////////////////////


WNDPROC WINAPI SuperCls_GetWndProcBaseCls (HWND hwnd) {

   // Return the address of the base class's WndProc from the superclass's
   // extra bytes.
   return((WNDPROC) GetClassLong(hwnd,
      SuperCls_BaseClsInfoIndex(GetClassLong(hwnd, GCL_CBCLSEXTRA), 
         pfnWndProcBaseCls)));
}


///////////////////////////////////////////////////////////////////////////////


// Function used internally by Get/SetClassWord/Long functions
static int SuperCls_ClassByteIndex (HWND hwnd, int nIndex) {

   int cbClsExtraIndexBaseCls, cbClsExtraBaseCls;

   // If nIndex is negative, the caller wants a Win32 predefined class 
   // word/long value.
   if (nIndex < 0) 
      return(nIndex);

   // Retrieve index into class extra bytes for the number of class extra 
   // bytes used by the base class.
   cbClsExtraIndexBaseCls =
      SuperCls_BaseClsInfoIndex(GetClassLong(hwnd, GCL_CBCLSEXTRA), 
         cbClsExtraBaseCls);

   // Retrieve number of class extra bytes used by the base class.
   cbClsExtraBaseCls = GetClassWord(hwnd, cbClsExtraIndexBaseCls);

   // Return (index + number) of class extra bytes used by base class.
   return(nIndex + cbClsExtraBaseCls);
}


/////////////////////// Set/GetClassWord/Long Functions ///////////////////////


WORD WINAPI SuperCls_SetClassWord (HWND hwnd, int nIndex, WORD wNewWord) {

   return(SetClassWord(hwnd, 
      SuperCls_ClassByteIndex(hwnd, nIndex), wNewWord));
}


WORD WINAPI SuperCls_GetClassWord (HWND hwnd, int nIndex) {

   return(GetClassWord(hwnd, 
      SuperCls_ClassByteIndex(hwnd, nIndex)));
}


DWORD WINAPI SuperCls_SetClassLong (HWND hwnd, int nIndex, DWORD dwNewLong) {

   return(SetClassLong(hwnd, 
      SuperCls_ClassByteIndex(hwnd, nIndex), dwNewLong));
}


DWORD WINAPI SuperCls_GetClassLong (HWND hwnd, int nIndex) {

   return(GetClassLong(hwnd, 
      SuperCls_ClassByteIndex(hwnd, nIndex)));
}


///////////////////////////////////////////////////////////////////////////////


// Function used internally by Get/SetWindowWord/Long functions.
static int SuperCls_WindowByteIndex (HWND hwnd, int nIndex) {

   int cbWndExtraIndexBaseCls, cbWndExtraBaseCls;

   // If nIndex is negative, the caller wants a Win32 predefined window 
   // word/long value.
   if (nIndex < 0) 
      return(nIndex);

   // Retrieve index into class extra bytes for the number of window extra 
   // bytes used by the base class.
   cbWndExtraIndexBaseCls = 
      SuperCls_BaseClsInfoIndex(GetClassLong(hwnd, GCL_CBCLSEXTRA), 
         cbWndExtraBaseCls);

   // Retrieve number of window extra bytes used by base class.
   cbWndExtraBaseCls = GetClassWord(hwnd, cbWndExtraIndexBaseCls);

   // Return (index + number) of window extra bytes used by base class.
   return(nIndex + cbWndExtraBaseCls);
}


/////////////////////// Set/GetWindowWord/Long Functions //////////////////////


WORD WINAPI SuperCls_SetWindowWord (HWND hwnd, int nIndex, WORD wNewWord) {

   return(SetWindowWord(hwnd, 
      SuperCls_WindowByteIndex(hwnd, nIndex), wNewWord));
}


WORD WINAPI SuperCls_GetWindowWord (HWND hwnd, int nIndex) {

   return(GetWindowWord(hwnd, 
      SuperCls_WindowByteIndex(hwnd, nIndex)));
}


DWORD WINAPI SuperCls_SetWindowLong (HWND hwnd, int nIndex, DWORD dwNewLong) {

   return(SetWindowLong(hwnd, 
      SuperCls_WindowByteIndex(hwnd, nIndex), dwNewLong));
}


DWORD WINAPI SuperCls_GetWindowLong (HWND hwnd, int nIndex) {

   return(GetWindowLong(hwnd, 
      SuperCls_WindowByteIndex(hwnd, nIndex)));
}


//////////////////////////////// End of File //////////////////////////////////
