;-----------------------------------------------------------------
;
; File Name..........: VHKD.ASM
; File Description...: Installs interrupt hooks to demonstrate the 
;  undocumented fault hooking process.
; Author.............: Victor Webber
;
;-----------------------------------------------------------------
; Build Notes:
;-----------------------------------------------------------------
;   set include=\ddk\include
;   masm5 -p -l -w2 hk.asm;
;   link386 hk,vhkd.386,,,hk.def
;   addhdr vhkd.386
;
;-----------------------------------------------------------------
; Operation Notes:
;-----------------------------------------------------------------
;  1) Add device=vhkd.386 to the [386Enh] section of the SYSTEM.INI file
;  2) Open this VxD and call functions from a DLL.  
;--------------------------------------------------------------

;--------------------------------------------------------------
;               A S S E M B L E R   D I R E C T I V E S
;--------------------------------------------------------------
.386p

;--------------------------------------------------------------
;                      I N C L U D E   F I L E S 
;--------------------------------------------------------------

INCLUDE VMM.INC

;--------------------------------------------------------------
;                     E X T E R N A L   L I N K S   
;--------------------------------------------------------------

;--------------------------------------------------------------
;                   E Q U A T E S
;--------------------------------------------------------------

; The device ID -- get your own ID from vxdid@microsoft.com
VHKD_VXD_ID           	EQU 28C2h
VERS_MAJ              	EQU 1
VERS_MIN              	EQU 0
VERSION               	EQU ((VERS_MAJ SHL 8) OR VERS_MIN)

;--- Equates used in reprogramming the 8254
; Number of reg 0 ticks which equal 1 milli sec
TMR_MTICK  				EQU 04A9h
; The count down value in Windows 3.1  
TMR_W31_CNT_DWN			EQU (TMR_MTICK*50)
; The count down value in Windows for Workgroups
TMR_WWG_CNT_DWN			EQU (TMR_MTICK*20)
TMR_OLD_CNT_DWN			EQU TMR_WWG_CNT_DWN
; Count down value for a tick every 15 milli secs
TMR_NEW_CNT_DWN			EQU (TMR_MTICK*15)

; 8254 Registers
CTRL_8254             	EQU 043h  
RWCNT_8254            	EQU 034h  
DATA_8254             	EQU 040h  

; 8259 values
VPICD_SP_EOI_0        	EQU 60h
VPICD_CMD_M8259       	EQU 20h

; Identifies interrupt to hook
TMR_IRQ0				EQU	050h

; Mode id
VMM_MODE_ID    			EQU 0
PM_MODE_ID     			EQU 1
V86_MODE_ID    			EQU 2

CFLAG					EQU 1

;--------------------------------------------------------------
;                 S T R U C T U R E S
;--------------------------------------------------------------

;--------------------------------------------------------------
;--------------------------------------------------------------
HK_ST     STRUC

;--------------------
; State of the hook system
wSt       				dw  ?
 ; Known uninitilized state
 ;HK_UNINIT      EQU 1
 ; All fault hooks initialized
 ;HK_INIT        EQU 2

;--------------------
; Frequency counters
dwNewFreq			 	dw	?	
dwOldFreq 		  		dw	?	

;--------------------
; Old 8254 count down value
dwOldTmrCntDwn			dw 	0
; New 8254 count down value
dwNewTmrCntDwn			dw 	0

;--------------------
; System VM Handle
dwCntDwnAccum      	  	dw	0

;--------------------
; System VM Handle
ddSysVM       			dd 	0  

;--------------------
; Array to hold old vectors
aVctr					dd	3 dup (?)

HK_ST     ENDS

 ; Known uninitilized state
 HK_UNINIT      EQU 1
 ; All fault hooks initialized
 HK_INIT        EQU 2

;--------------------------------------------------------------
;        V I R T U A L   D E V I C E   D E C L A R A T I O N
;--------------------------------------------------------------

Declare_Virtual_Device VHKD, VERS_MAJ, VERS_MIN, _VhkdVxDControl, \
    VHKD_VXD_ID, Undefined_Init_Order, \
    _VhkdAPIHandler, _VhkdAPIHandler,

;--------------------------------------------------------------
;           R E A L   M O D E   I N I T I A L I Z A T I O N
;--------------------------------------------------------------

VXD_REAL_INIT_SEG

_real_init proc near
    mov 	 ah, 9
    mov 	 dx, offset claim
    int 	 21h
    xor 	 ax, ax          ; set up to tell Windows it's okay 
    xor 	 bx, bx          ; to keep loading this VxD
    xor 	 si, si
    xor 	 edx, edx
    ret
_real_init endp

claim     db  'VHKD.386 -- VHKD  v. 1.0',0dh,0ah
          db  'Copyright (c) 1995 1996 Victor Webber. All Rights Reserved.'
          db  0dh,0ah,'$'

VXD_REAL_INIT_ENDS

;--------------------------------------------------------------
;                D A T A   S E G M E N T S
;--------------------------------------------------------------

VxD_DATA_SEG


;-----------------------------------------------------
; State Vars
 
sHK		HK_ST < HK_UNINIT, 0, 0, TMR_OLD_CNT_DWN >

;-----------------------------------------------------
; API jump table
VhkdAPICall    label dword

                dd  offset32 _HkGetNewFreq

                dd  offset32 _HkGetOldFreq

VHKD_API_MAX EQU ( $ - VhkdAPICall ) / 4

VxD_DATA_ENDS

;--------------------------------------------------------------
;           L O C K E D   C O D E   S E G M E N T
;--------------------------------------------------------------

VxD_LOCKED_CODE_SEG

;------------------------------------------------------
BeginProc   _VhkdAPIHandler
; --- In: ebp.Client_AX :: function index
;------------------------------------------------------
; --- Description:
;  - This is the handler which funnels all service 
;  calls.
; 
; --- Out:
;  - Calls indexed function
;
; --- Return:
;  - ebp.Client_EFlags :: CFLAG set is error
;  - ebp.Client_EFlags :: CFLAG clear is error
;  - See individual function
;
;------------------------------------------------------

    movzx    eax, [ebp.Client_AX]            ; get caller's AX register
    cmp      eax, VHKD_API_MAX               ; valid function number?
    jae      short fail
    and      [ebp.Client_EFlags], NOT CFLAG  ; clear carry for success

    call     VhkdAPICall[eax * 4]            ; call through table
    ret
fail:
    or       [ebp.Client_EFlags], CFLAG
    ret

EndProc     _VhkdAPIHandler

;------------------------------------------------------
BeginProc   _HkReload8254, PUBLIC

; --- In: Word value to write into 8254

_HkRld STRUC 

     dd ?  ;bp
     dd ?  ;ret
rl1  dd ? 

_HkRld ENDS   

;------------------------------------------------------
; --- Desc:
;  - Reload the 8254 with a new IPC
;  - Interrupts should be disabled throughout
;  this process. 
; 
; --- Out:
;  - Write value into 8254
;
; --- Return:
;  - void
;
;------------------------------------------------------
			
	push    ebp
  	mov     ebp, esp

  	mov     al, RWCNT_8254
  	out     CTRL_8254, al 
  	mov     ax, WORD PTR [ebp.rl1]
  	out     DATA_8254, al
  	shr     ax, 8   
  	out     DATA_8254, al   

  	pop     ebp
  	ret

EndProc _HkReload8254

;-----------------------------------------------------
BeginProc _VhkdVxDControl
; --- In: void
;-----------------------------------------------------
; --- Description:
;  - Process system messages
;
; --- Out:
;  - Call message handlers
;
; --- Return:
;  - void
;
;-----------------------------------------------------

	Control_Dispatch Sys_Critical_Init, _VhkdCritInitMsg
	Control_Dispatch Sys_Critical_Exit, _VhkdCritExitMsg

	clc    
	ret

EndProc _VhkdVxDControl

;------------------------------------------------------
BeginProc _VhkdCritInitMsg
; In: void
;------------------------------------------------------
; --- Description:
;  - Process Crit Init message.  This message occurs 
;  during system initialization.
; 
; --- Out:
;  - Filles database with control information
;  - Init Fault Hooks
;  - Init first stage of tmr 
;
; --- Return:
;  - cy clr :: success, everything installed
;  - cy set :: error, nothing has been installed
;
;------------------------------------------------------

    VMMcall  Get_Sys_VM_Handle
	
    mov      sHk.ddSysVM, ebx

    ; Install hooks, but keep hook system uninitialized.
    call 	 _HkInstallHooks
	jc		 CI_EXIT

    ; Increase frequency of hook system
	call	 _HkScheduleFreq
    clc  

	CI_EXIT:
    ret
    
EndProc _VhkdCritInitMsg
		  
;------------------------------------------------------
BeginProc _VhkdCritExitMsg
;------------------------------------------------------
; --- Description:
;  - Process Crit Exit message
; 
; --- Out:
;  - Uninit Fault Hooks
;
; --- Return:
;  - cy clr :: error, nothing has been installed
;
;------------------------------------------------------

    ; Reset hook system
	call	 _HkUnScheduleFreq

	clc
  	ret

EndProc _VhkdCritExitMsg

;------------------------------------------------------
BeginProc _HkInstallHooks
; --- In: Void
;------------------------------------------------------
; --- Description:
;  - Hook the Timer faults. This should be the last 
;  hook of these faults. No Hook should occur after 
;  this.     
; 
; --- Out:
;  - Call Hooking procedures
;  - Fill structure with address of old fault hooks
;
; --- Return:
;  - cy set :: error, no handlers installed
;  - cy clr :: success
;
;------------------------------------------------------
										          
    push     esi

	mov      eax, TMR_IRQ0
	mov      esi, OFFSET32 _HkPmIRQ0FaultHook
	VMMcall  Hook_PM_Fault
    jc       short HK_ERR
	mov      sHK.aVctr[PM_MODE_ID*4], esi
	
	mov      esi, OFFSET32 _HkV86IRQ0FaultHook
	VMMcall  Hook_V86_Fault
    jc       short HK_ERR
	mov      sHK.aVctr[V86_MODE_ID*4], esi
	
	mov      esi, OFFSET32 _HkVMMIRQ0FaultHook
	VMMcall  Hook_VMM_Fault
    jc       short HK_ERR
	mov      sHK.aVctr[VMM_MODE_ID*4], esi

    HK_ERR:
    pop     esi
    ret

EndProc _HkInstallHooks

;------------------------------------------------------
BeginProc _HkScheduleFreq
; --- In: void
;------------------------------------------------------
; --- Description:
;  - Reprograms the 8254 to interrupt at a higher 
;  frequency. 
;  - Changes database for higher frequency
; 
; --- Out:
;  - Reprograms 8254
;  - Sets minimum interrupt period reload variable
;
; --- Return:
;  - void
;
;------------------------------------------------------

	; Turn off interrupt
	cli

	; Initialize hook system variables
	mov		 sHK.wSt, HK_INIT
	mov		 sHK.dwCntDwnAccum, TMR_NEW_CNT_DWN

	; Load new count down into 8254
	cCall	 _HkReload8254 < TMR_NEW_CNT_DWN >

	; Write reload values into structure
	mov		 sHK.dwNewTmrCntDwn, TMR_NEW_CNT_DWN

	; Turn on interrupts
	sti
          
EndProc _HkScheduleFreq


;------------------------------------------------------
; H O O K   P R O C S
; --- In: void
;------------------------------------------------------
; --- Desc:
;  - ISRs for the PM mode, V86 mode, and the VMM mode
;  - All registers have been saved prior to the ISR
;  being called.
; 
; --- Out:
; - Call another function
;
; --- Return:
;  - void
;
; --- Uses:
;  - eax 
;------------------------------------------------------

BeginProc _HkPmIRQ0FaultHook

    mov      eax, PM_MODE_ID
    call     _HkGenIRQ0FaultHook
    ret
        
EndProc _HkPmIRQ0FaultHook

BeginProc _HkV86IRQ0FaultHook

    mov      eax, V86_MODE_ID
    call     _HkGenIRQ0FaultHook
    ret
        
EndProc _HkV86IRQ0FaultHook

BeginProc _HkVmmIRQ0FaultHook

    mov      eax, VMM_MODE_ID
    call     _HkGenIRQ0FaultHook
    ret

EndProc _HkVmmIRQ0FaultHook


;------------------------------------------------------
BeginProc _HkGenIRQ0FaultHook, PUBLIC
; eax = FAULT ID
;------------------------------------------------------
; --- Desc:
;  - Checks if critical frequency is reached.  Increment
;  variables
; 
; --- Out:
;  - Increment Variables
;
; --- Return:
;  - void
;
; --- Uses: 
; - Nothing
;
;------------------------------------------------------

  	; Check if timer initialize, exit if not
  	cmp      sHK.wSt, HK_UNINIT
  	je       short GEN_EXIT

  	;Increment the count for new frequency
  	inc      sHK.dwNewFreq

  	; Check if old frequency reached
	sub		 sHK.dwCntDwnAccum, TMR_NEW_CNT_DWN
	jc		 GEN_EXIT

	; Send EOI to 8259
  	mov   	 al, VPICD_SP_EOI_0
  	out   	 VPICD_CMD_M8259, al

    ret

	GEN_EXIT:

  	;Increment the count for old frequency
  	inc      sHK.dwNewFreq

	mov		 sHK.dwCntDwnAccum, TMR_OLD_CNT_DWN
  	call     sHK.aVctr[eax*4]
  	ret

EndProc _HkGenIRQ0FaultHook, PUBLIC

;------------------------------------------------------
BeginProc _HkUnScheduleFreq
; --- In: void
;------------------------------------------------------
; --- Description:
;  - Resets 8254 to original state
; 
; --- Out:
;  - Reprograms 8254
;  - Resets hook system control variables
;
; --- Return:
;  - void
;
;------------------------------------------------------

	; Turn off interrupt
	cli
	
  	; Uninitialize the hook system variables
	mov		 sHK.wSt, HK_UNINIT

	; Load original count down into 8254
	cCall	_HkReload8254 < TMR_OLD_CNT_DWN >

	; Turn on interrupts
	sti

EndProc _HkUnScheduleFreq

;------------------------------------------------------
BeginProc   _HkGetNewFreq
; --- In: ebp :: pointer to client register structure
;------------------------------------------------------
; --- Desc:
;  - Reports frequency count for new 8254 period
; 
; --- Out:
;  - Read hook system structure
;
; --- Return:
;  - Client_AX :: Frequency count
;
;------------------------------------------------------
			
	mov		ax, sHK.dwNewFreq
	mov		[ebp.Client_AX], ax

EndProc 	_HkGetNewFreq

;------------------------------------------------------
BeginProc   _HkGetOldFreq
; --- In: ebp :: pointer to client register structure
;------------------------------------------------------
; --- Desc:
;  - Reports frequency count for old 8254 period
; 
; --- Out:
;  - Read hook system structure
;
; --- Return:
;  - Client_AX :: Frequency count
;
;------------------------------------------------------
			
	mov		ax, sHK.dwOldFreq
	mov		[ebp.Client_AX], ax

EndProc 	_HkGetOldFreq

VxD_LOCKED_CODE_ENDS

    END
