unit Remeng;

interface

uses SysUtils;

type
   TReminderRecord = record
      Message : String[100];
      Date    : TDateTime;
      Window  : Byte;
      Active  : Boolean; {Should the message be active today?}
      Cleared : Boolean;
      Nag     : Boolean;
      end;

   TReminder = class
         Data   : TReminderRecord;
      constructor Create(Msg: String; D: TDateTime; Win: Byte; N, C: Boolean);
   {  destructor  Destroy; (not needed)}
      function    Describe: String;
      function    IsActive(Today: TDateTime): boolean;
      procedure   Replace(NewData: TReminderRecord);
      end;

   TReminderFile = file of TReminderRecord;

   TReminderReader = class
         filename : String;
         handle   : TReminderFile;
         Last     : TDateTime;
      constructor Create(filename: String);
      destructor  Destroy;
{Valid record numbers used in GetReminder and
 RemoveReminder are from 1 to NumberOfReminders.}
      function    GetReminder(Number: Integer): TReminder;
      function    NumberOfReminders: Longint;
      procedure   PutReminder(Number: Integer; Reminder: TReminder);
      procedure   RemoveReminder(Number: Integer);
      procedure   SetClear(Number: Integer; Clear: Boolean);
      function    SinceLastChecked(Number: Integer): boolean;
      end;

implementation

{The DATE function is from the SYSUTILS unit and returns the
 current date.}
constructor TReminder.Create(Msg: String; D:TDateTime; Win: Byte; N, C: Boolean);
begin
   with Data do
   begin
      Message := Msg;
      Date := D;
      Window := Win;
      Cleared := C;
      Nag := N;
      Active := IsActive(Date);
      end;
   end;

{In this version of the program, Today might just as
 well be returned by the SysUtil function Date-- that
 is the current date.  But we might want to add
 functionality later to allow the program to ask:
 Would this reminder be valid on a certain date,
 where that date wasn't today.  This could be useful,
 for example, if we were going to add a monthly
 calendar that showed the active dates for a
 reminder.}
function    TReminder.IsActive(Today: TDateTime): boolean;
var Distance, ReminderDate: TDateTime;
    Y, M, D               : Word;
begin
   {Whatever year the reminder was written in,
    it should be compared as though the date
    were the same year as today. This might
    change in a later version.}
    ReminderDate := Data.Date;
    DecodeDate(ReminderDate, Y, M, D);
    ReminderDate := ReminderDate - Y;
    DecodeDate(Today, Y, M, D);
    ReminderDate := ReminderDate + Y;

    with Data do begin
       Distance := Abs(Int(Today) - Int(Date));
        {If we've passed the Reminder's date then Nag must be true
         in order for the Reminder to be Active.}
       Data.Active := ((Today <= Date) and (Distance <= Window)) or Nag;
       end;

    IsActive := Data.Active;
    end;

function    TReminder.Describe;
begin
   with Data do
   begin
      If Active then Result := '!' else Result := '';
      If Nag then Result := Result + '(nag)';
      If Cleared then Result := Result + 'x';
      Result := Result + ' ' +DateToStr(Date)+' ';
      Result := Result + Message;
      end;
   end;

procedure   TReminder.Replace;
begin
   with Data do
   begin
      Message := NewData.Message;
      Date := NewData.Date;
      Window := NewData.Window;
      Cleared := NewData.Cleared;
      Nag := NewData.Nag;
      Active := IsActive(Date);
      end;
   end;

constructor TReminderReader.Create(filename: String);
var Error: Integer;
    First: TReminder;
begin
   Assign(Handle, filename);
   {$I-}
   reset(handle);
   {$I+}
   Error := IOResult;
   If Error<>0 then
   begin
      If Error<>2 then Fail;
      {$I-}
      Rewrite(Handle);
      {$I+}
      end;

   If NumberOfReminders<1 then
   begin
      {Create a "last access" date so that a new file may
       be treated like an existing file after being created.}
      First := TReminder.Create('Created', Date, 0, False, False);
      PutReminder(0, First);
      First.Destroy;
      end;

   First := GetReminder(0);
   Last := Trunc(First.Data.Date);
   First.Destroy;
   {Now that we know what the last access date was,
    set the last access date for the file to today.}
   First := TReminder.Create('Last access', Date, 0, False, False);
   PutReminder(0, First);
   First.Destroy;
   end;

destructor  TReminderReader.Destroy;
begin
   Close(Handle);
   end;

function    TReminderReader.GetReminder(Number: Integer): TReminder;
var R   : TReminderRecord;
    B,E : TDateTime;
begin
   Seek(Handle, Number);
   Read(Handle, R);
   Result := TReminder.Create(R.Message, R.Date, R.Window, R.Nag, R.Cleared);
   end;

function    TReminderReader.NumberOfReminders: Longint;
begin
   {Subtract 1 because first "reminder" is, in fact, last access.}
   NumberOfReminders := FileSize(handle)-1;
   end;

{Up to clinet to pass a valid record number.}
procedure   TReminderReader.PutReminder(Number: Integer; Reminder: TReminder);
begin
   Seek(Handle, Number);
   Write(Handle, Reminder.Data);
   end;

{Up to clinet to pass a valid record number.}
procedure   TReminderReader.RemoveReminder(Number: Integer);
var R: TReminderRecord;
    I: Longint;
begin
   Seek(Handle, Number);
   For I := Number to NumberOfReminders-1 do
   begin
      Seek(Handle, I+1);
      Read(Handle, R);
      Seek(Handle, I);
      Write(Handle, R);
      end;
   Truncate(Handle);
   end;

{Up to client to pass valid reminder number}
procedure   TReminderReader.SetClear(Number: Integer; Clear: Boolean);
var R: TReminder;
begin
   R := GetReminder(Number);
   R.Data.Cleared := Clear;
   PutReminder(Number, R);
   R.Destroy;
   end;

function    TReminderReader.SinceLastChecked(Number: Integer): boolean;
var T: TDateTime;
    R: TReminder;
begin
   SinceLastChecked := false;

   T := Trunc(Date); {today! (days only)}
   R := GetReminder(Number);

   while (T<>Last) do
   begin
      {If reminder was active then became inactive it s/b uncleared;
       If reminder was inactive then became active it s/b uncleared.}
      If R.Data.Active<>R.IsActive(T) then SetClear(Number, False);
      If R.Data.Active and not R.Data.Cleared then
      begin
         SinceLastChecked := true;
         break;
         end;
      T := T + 1;
      end;

   R.Destroy;
   end;
end.
