// BasicMac.cpp - demonstration of using the Prodis Parsing Engine.
//
// Notes:
//   We include BasicMac.h for our tokens, and it includes 
//   all other neccessary files.

#include "basicmac.h"

// Declare global variable, just for ease of demonstration
// flow control object for parsing input lines
gpFlowControl *fcMacEngine;

// parser object for parsing expressions
gpParser *pExpEngine;           
    
int main ( )
  {
    // Build the macro-language flow control object.
    fcMacEngine = BuildMacroEngine();
    
    // Build the expression parser object.
    pExpEngine = BuildExpEngine();

    gpString sLine;      // buffer for input lines
    int nLineNo, nToken; // line number, token
    StringList slParms;  // string list for holding parameters

    nLineNo = GetMacroLine(sLine); // Get the first line.
    
    // nLineNo will be zero when there are no more input lines.
    while(nLineNo)
      {
        // Parse the input line. The engine will return a token.
        nToken = fcMacroEngine->Parse(sLine, slParms);
        
        switch(nToken)
          {
            // no match was found for the line
            case TK_UNRECOGNIZED : 
                ErrorHandler();  //Send an error message
                break;
                
            case TK_ASSIGN :
                // first parm was the variable name
                // second parm was the value to set it to
                SetVariableToValue(slParms[1], slParms[2]);
                break; 
                
            case TK_IF :
            case TK_WHILE :
              { 
                // slParms[1] has the expression following the 'if'
                // or while. It must be parsed and evaluated.
                int IsTrue = EvalateExpression(slParms[1]);
                PostExpressionValue(IsTrue);
                break;
              }  
                
            case TK_DOMENUITEM :
                // The following function - not listed -
                // must implement the macro-language line :
                // DoCmd DoMenuItem [Parm1], [Parms2], [Parm3]
                DoMenuItem (lsParms[1], lsParms[2], lsParms[3]);
                break;
            
            case TK_MBOX :
                // The following function - not listed -
                // causes a Message Box to be displayed
                MessageBox(lsParms[1], lsParms[2], lsParms[3]);
                break;
                
            case TK_CALL :
                // The following function - not listed - 
                // Matches the first parameter with a 
                // function call and gets its arguments
                // from the second parameter.
                CallFunctionWithParms(lsParms[1], lsParms[2]);
                break;

            case TK_ENDWHILE :
                // When an ENDWHILE is returned, the first parameter
                // is the number of the line continaing the matching
                // WHILE clause, which will be re-evaluated.
                RewindMacroSource (atoi (lsParms[1]));
                break;

            case TK_ELSE :
            case TK_ENDIF :
                // We don't need to do anything here; gpFlowControl
                // will handle it all.  However, if we were 
                // interested, we could trap these lines.
                break;
                
          }
          
        nLineNo = GetMacroLine(sLine); // get the next line
      }
    return (1);           
  }
  
gpFlowControl *BuildMacroEngine( )
  {
    gpFlowControl fcNew = new FlowControl;
    
    fcNew.AddSyntax ("^ *If +&.+ +Then *$", TK_IF);
    fcNew.AddSyntax ("^ *Else *$", TK_ELSE);
    fcNew.AddSyntax ("^ *Endif *$", TK_ENDIF);
    fcNew.AddSyntax ("^ *Let &.+ *= *&.+ *$", TK_ASSIGN);
    fcNew.AddSyntax ("^ *While *&.* *$", TK_WHILE);
    fcNew.AddSyntax ("^ *WEnd *$", TK_ENDWHILE);
    fcNew.AddSyntax ("^ *Call *MessageBox *(&.+,&.+,&.+) *$"
                    , TK_MBOX);
    fcNew.AddSyntax ("^ *Call *&.+ *(&.+) *$", TK_CALL);
    fcNew.AddSyntax ("^ *DoCmd +DoMenuItem +&.+, +&.+, +&.+ *$"
                    , TK_DOMENUITEM);

    return fcNew;
  }                                 

gpParser *BuildExpEngine( )
  {
    gpParser pNew = new gpParser;

    pNew.AddSyntax ("&.+ *= *&.+", TK_EQUALS);
    pNew.AddSyntax ("&.+ *< *&.+", TK_LESS_THAN);
    pNew.AddSyntax ("&.+ *> *&.+", TK_GREATER_THAN);

    return pNew;
  }

int EvalExp (gpString &sExpress)
  {
    int nReturn = 0;     //return value
    int nToken;          //token
    StringList lsParms;  //String list for parameters

    // parse the given line and get the token
    nToken = pExpEngine->Parse(sExpress, lsParms);

    switch (nToken)
      {
        //the line is : expression = expression
        //if the two expressions are equal, return TRUE
        case TK_EQUALS :
            if CheckEquals(lsParms[1], lsParms[2])
                nReturn = 1;
            break;
                
        //the line is : expression > expression
        //if the first expression is greater, return TRUE
        case TK_GREATER_THAN :         
            if CheckGreater(lsParms[1], lsParms[2])
                nReturn = 1;
            break;
                
        //the line is : expression < expression
        //if the second expression is greater, return TRUE
        case TK_LESS_THAN :                 
            if CheckLessThan (lsParms[1], lsParms[2])
                nReturn = 1;
      }
      
    return nReturn;  
  }


