/* FileComp.cpp -- C++ for "const int", not for I/O.

   Compare two user-specified files byte-by-byte.
   Do NOT require any interaction with the user.

   Return
     Value:    When:
       0         Files match exactly.
       1         Files do not match exactly.
      11         First file specified does not exist, but second one does.
      12         First file specified exists, but second one does not.
      13         Neither file specified exists.
      20         No files specified.
      21         Only one file specified.
      99         Unknown error.

   0 if an exact match
          1 otherwise.

   R.B. Daehler-Wilking
   03/28/95

*/
#include <stdio.h>
#include <process.h>

const int MaxFileNameSize    = 240;

const int NoResultYet        =  -1;
const int ResultMatch        =   0;
const int ResultNotMatch     =   1;
const int ResultNoFirstFile  =  11;
const int ResultNoSecondFile =  12;
const int ResultNeitherFile  =  13;
const int ResultNoNames      =  20;
const int ResultOnlyOneName  =  21;
const int ResultUnknownError =  99;

void usage(void)
  {
  printf("   usage: FileComp file1 file2\n");
  }

int main(int argc, char *argv[])
  {
  int result=NoResultYet;

  FILE *file1, *file2;
  unsigned char buffer1[BUFSIZ], buffer2[BUFSIZ];
    /* BUFSIZ is defined in stdio.h for buffered I/O.  */
  unsigned char c1, c2;

  if (argc == 1)
      /* No command-line arguments.  Note that the name of the
           program ("usage") counts as the first argument (number zero),
           so we shouldn't even get into this program unless there
           is at least one argument.
      */
      {
      result = ResultNoNames;
      printf("FileComp: No filenames given.\n");
      usage();
      }
    else
      {
      if (argc == 2)
          /* Only one command-line argument */
          {
          result = ResultOnlyOneName;
          printf("FileComp: Only one filename given ('%s').\n",
                 argv[1]);
          usage();
          }
        else
          /* We have enough arguments.  Just ignore any extras
               and treat the second and third as our filenames.
               Remember that in C/C++, all arrays start with
               element zero, so argv[1] is the second argument
               of the command line, and argv[2] is the third.
          */
          {
          file1=fopen(argv[1], "rb");
          file2=fopen(argv[2], "rb");
          setbuf(file1, buffer1);  /* set up fully-buffered I/O. */
          setbuf(file2, buffer2);

          if ( (file1==NULL) && (file2 != NULL) )
              {
              result = ResultNoFirstFile;
              printf("FileComp cannot open file '%s'.\n", argv[1]);
              }
            else
              {
              if ( (file1 != NULL) && (file2 == NULL) )
                  {
                  result = ResultNoSecondFile;
                  printf("FileComp cannot open file '%s'.\n",
                         argv[2]);
                  }
                else
                  {
                  if ( (file1 == NULL) && (file2 == NULL) )
                      {
                      result = ResultNeitherFile;
                      printf("FileComp cannot open either '%s' or '%s'.\n",
                             argv[1], argv[2]);
                      }
                    else
                      /* Both files are opened OK. */
                      {
                      result = NoResultYet;
                      while (   (!feof(file1))
                             && (!feof(file2))
                             && (result == NoResultYet)
                            )
                          {
                          c1=fgetc(file1);
                          c2=fgetc(file2);
                          if (c1!=c2)
                              result = ResultNotMatch;
                          /* endif */
                          }
                      /*endwhile */

                      fclose(file1);
                      fclose(file2);

                      if (result == NoResultYet)
                          /* No trouble found, so it's a match. */
                          {
                          result = ResultMatch;
                          }
                      /*endif NotMatch */

                      switch (result)
                          {
                          case ResultMatch:
                              printf("FileComp: %s and %s match.\n",
                                     argv[1], argv[2]);
                              break;
                          case ResultNotMatch:
                              printf("FileComp: %s and %s do NOT match.\n",
                                     argv[1], argv[2]);
                              break;
                          default:
                              result = ResultUnknownError;
                              printf("FileComp: Unknown error.\n");
                              break;
                          }
                      /* end switch */
                      }
                  /* endif NeitherFile -- else */
                  }
              /* endif NoSecondFile -- else */
              }
          /* endif NoFirstFile -- else */
          }
      /* endif argc == 2 -- else */

      }
  /* endif argc < 1 -- else */

  exit(result);
  return(result);
    /* Just to keep the compiler happy, since it doesn't see
       exit() as returning a value.
    */

  }













