/*

This program takes a .LIB file and breaks it out into seperate .OBJ modules. 
Although the standard LIB program does this, I've come across some .LIB's
which LIB won't recognise because library headers are missing, etc.  Hence
this program.

A successful compile & link can be achieved with MSC 7 using:

CL /AL /Za /W4 /WX libtoobj.c

See the acompanying .TXT file for more information

Lee Brown      Compuserve: 100325,3304
*/

#include <fcntl.h>
#include <io.h>
#include <malloc.h>
#include <memory.h>
#include <process.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>

/* main() return codes */

#define EXIT_OK         0
#define EXIT_ERROR     20
#define EXIT_ERROR     20


/* library object identifier manifests */

#define THEADR          0x80
#define MODEND          0x8a


/* filename information manifests */

#define MAX_PATH        _MAX_PATH               /* supplied by msc */
#define EXTENSION_LEN   3


/* readability */

#define FALSE           0
#define TRUE            !FALSE

#define SkipRecord( fin, Hdr )   _lseek( (fin), (Hdr)->Size, SEEK_CUR )
#define WriteRecord( fout, Buffer, Hdr )  _write( (fout), (Buffer),\
      (Hdr)->Size + sizeof( *(Hdr) ) )

/* dependent types (ensure that structures in this section are byte aligned):
   -------------------------------------------------------------------------*/
typedef unsigned char ubyte;
typedef unsigned int uword;      

typedef struct _RecordHeader {
   ubyte Type;
   uword Size;
} RecordHeader;

/* end of dependent types section
   ------------------------------ */

void *SafeMalloc( size_t size ) {
   void *Ret;

   if ( (Ret = malloc( size )) != NULL )
      return Ret;
   printf( "Memory allocation failure\n." );
   exit( EXIT_ERROR );
}

void help( void ) {
   fprintf( stderr, "usage: libsplit <libname.LIB>\n" );
   exit( EXIT_ERROR );
}


/* This simply reads the record header structure. */

RecordHeader *ReadRecordHeader( int fin, RecordHeader *Hdr ) {
   _read( fin, Hdr, sizeof( *Hdr ) );
   return Hdr;
}


/* Fill a buffer with the record data including header. */

ubyte *ReadRecord( int fin, RecordHeader *Hdr ) {
   ubyte *Buffer = SafeMalloc( Hdr->Size + sizeof( *Hdr ) );

   *(RecordHeader *)Buffer = *Hdr;                    /* copy header */
   _read( fin, Buffer + sizeof( *Hdr ), Hdr->Size );  /* read record info */
   return Buffer;
}


/* Copy a record from one file to another. */

void CopyRecord( int fout, int fin, RecordHeader *Hdr ) {
   ubyte *Buffer = ReadRecord( fin, Hdr );

   _write( fout, Buffer, Hdr->Size + sizeof( *Hdr ) );
   free( Buffer );
}


/* Pascal to C type string conversion (1st byte is string length).
   This function allocates memory for the new string. */

char *PtoCString( char *PString ) {
   int StringSize = *(ubyte *)PString;
   char *CString = SafeMalloc( StringSize + 1 );

   memcpy( CString, PString + 1, StringSize );
   *(CString + StringSize) = '\0';
   return CString;
}


/* FullSpec - path to split
   Drive    - Storage for drive (typ. 3 chars)
   Dir      - Storage dir dir...
   any component except FullSpec may be NULL to ignore it */

void SplitPath( char *FullSpec, char *Drive, char *Dir, char *Name, 
                char *Extension ) {
   _splitpath( FullSpec, Drive, Dir, Name, Extension );
}

/* Create a unique filename for the object module.
   This strips the filename and extension parts from a module name which may
   be full specified.  If the file already exists, the extension is cycled
   through .000, .001 etc. until either 999 is reached(!) or a file can be
   sucessfully created.
   Memory is allocated for the filename generated. */

char *CreateFilename( char *Modulename, int *fout ) {
   /* search backward for last '\' or ':' (in that order) which will give 
      start of filename part, or if none found use entire string. */

   char *Filename,
        *Extension;
   int Ordinal;

   Filename = SafeMalloc( MAX_PATH );
   SplitPath( Modulename, NULL, NULL, Filename, NULL );  /* get name only */
   Extension = Filename + strlen( Filename ) + 1;        /* start of extension */
   strcat( Filename, ".obj" );                           /* append extension */
   
   /* Try to create a file until the ordinal reaches 999! */

   for ( *fout = -1, Ordinal = 0; *fout == -1 && Ordinal < 999; Ordinal++ )
      if ( (*fout = _open( Filename, _O_BINARY | _O_RDONLY | _O_CREAT | 
            _O_EXCL | _O_WRONLY, _S_IREAD | _S_IWRITE )) == -1 )
         sprintf( Extension, "%03d", Ordinal++ );        /* build new ext. */
   return Filename;                                      /* return new name */
}


/* Creates the .LNK file based on the .LIB name */

FILE *CreateLNK( char *LibFilename ) {
   char *Name;
   FILE *fout;

   Name = SafeMalloc( MAX_PATH );
   SplitPath( LibFilename, NULL, NULL, Name, NULL );  /* get name only */
   strcat( Name, ".lnk" );                            /* append '.lnk' */
   if ( (fout = fopen( Name, "w" )) == NULL )         /* try creating */
      fprintf( stderr, "Can't create '%s'\n", Name );
   free( Name );
   return fout;                                       /* return handle */
}


/* Creates a unique file based on the module name in a THEADR record.  Once
   this is done, the THEADR record is written to it. */

int CreateFile( int fin, RecordHeader *Hdr, FILE *lnk ) {
   char *Buffer = (char *)ReadRecord( fin, Hdr ),
        *Modulename,
        *Filename;
   int fout;
   static int First = TRUE;

   Modulename = PtoCString( Buffer + sizeof( *Hdr ) );
   Filename = CreateFilename( Modulename, &fout );
   if ( fout == -1 )
      printf( "Warning: Can't create '%s'\n", Filename );
   WriteRecord( fout, Buffer, Hdr );
   printf( "File %-12s formed from '%s'\n", Filename, Modulename );
   if ( lnk ) {
      fprintf( lnk, "%s+%s", First ? "" : "&\n", Filename );
      if ( First )
         First = !First;
   }
   free( Modulename );
   free( Filename );
   free( Buffer );
   return fout;
}


/* The main loop.  This gets sucessive records and creates a new file if the
   record type is a THEADR.  A MODEND record closes the record while all other
   record types are copied. */

void SplitLib( int fin, FILE *lnk ) {
   int fout = -1;
   RecordHeader Hdr;

   while ( !_eof( fin ) ) {
      ReadRecordHeader( fin, &Hdr );
      switch( Hdr.Type ) {
         case THEADR:
            fout = CreateFile( fin, &Hdr, lnk );
            break;
         case MODEND:
            CopyRecord( fout, fin, &Hdr );
            _close( fout );
            fout = -1;
            break;
         default:
            if ( fout != -1 )
               CopyRecord( fout, fin, &Hdr );
            else
               SkipRecord( fin, &Hdr );
      }
   }
}

/* parse the command line, open the library file, do the work... */

int main( int argc, char **argv ) {
   int fin;
   FILE *lnk;

   if ( argc != 2 )
      help();
   if ( (fin = _open( *++argv, _O_BINARY | _O_RDONLY )) == -1 ) {
      fprintf( stderr, "can't open '%s'\n", *argv );
      exit( EXIT_ERROR );
   }
   lnk = CreateLNK( *argv );
   SplitLib( fin, lnk );
   _close( fin );
   fclose( lnk );
   return EXIT_OK;
}
