UNIT I14;


{ Modul zur Ausgabe einer Datei ber die serielle Schnittstelle }


{ Ullrich von Bassewitz am 16.10.1992 }

{ Diverse Compiler-Switches: }

{$F-     Force Far Calls Off  }
{$O-     No Overlays Allowed  }
{$A+     Align Data at Word Boundary }
{$B-     Short Circuit Boolean Evaluation }
{$I-     I/O Checking Off     }
{$D+     Debug Information On }
{$L+     Local Symbols On     }
{$G-     No 286-Code }

{$IFDEF Debug }
  {$R+   Range Checking On    }
  {$S+   Stack Checking On    }
{$ELSE}
  {$R-   Range Checking Off   }
  {$S-   Stack Checking Off   }
{$ENDIF}



{ nderungsliste:

16.10.1992              Erstellt


}


INTERFACE



USES
  DOS;




CONST
  { Anzahl Wiederholungsversuche bei Timeout }
  InitTimeoutRetries = 20;



TYPE
  { Definitionen fr Parity und Baudrate }
  TParity     = (pNone, pOdd, pEven);
  TBaudRate   = (bd110, bd150, bd300, bd600, bd1200, bd2400, bd4800, bd9600);
  TStopBits   = 1..2;
  TDataBits   = 5..8;
  TPortNr     = 1..4;

  { Objekt fr einen seriellen Port, *** abstrakt, nicht verwenden *** }
  PSerialPort  = ^TSerialPort;
  TSerialPort  = OBJECT

    PortNr     : WORD;                  { Nummer des Ausgabeports, 0..3 (!) }
    MaxTries   : WORD;                  { Anzahl Versuche wenn Timeout }

    { Konstruktor }
    CONSTRUCTOR Init (Port: TPortNr;
                      Baudrate: TBaudRate;
                      Parity: TParity;
                      StopBits: TStopBits;
                      DataBits: TDataBits);

    { Destruktor }
    DESTRUCTOR Done; VIRTUAL;

    { Virtuelle Fehlerbehandlungsfunktion }
    PROCEDURE Error (Errno: WORD); VIRTUAL;

    { Prft ob Ausgabe frei, ergibt TRUE wenn mglich }
    FUNCTION HandshakeOk : BOOLEAN; VIRTUAL;           { Abstrakte Funktion !! }

    { Anzahl Ausgabeversuche bei Timeout setzen }
    PROCEDURE SetTimeoutRetries (Tries : WORD);

    { Ausgabe eines Zeichens }
    PROCEDURE OutByte (B : BYTE); VIRTUAL;

    { Ausgabe eines Strings }
    PROCEDURE OutString (S : STRING);

    { Eingabe eines Zeichens }
    FUNCTION InByte : BYTE; VIRTUAL;

    { Ergibt TRUE wenn ein Zeichen verfgbar ist }
    FUNCTION ByteAvail: BOOLEAN; VIRTUAL;

    { Prft den bergebenen Status und ruft den Fehlerhandler auf wenn ntig }
    PROCEDURE CheckPortStatus (PortStatus: WORD); VIRTUAL;

  END;




  PHardWiredPort = ^THardWiredPort;
  THardWiredPort = OBJECT (TSerialPort)

    { Neue HandShakeOk-Funktion }
    FUNCTION HandShakeOk : BOOLEAN; VIRTUAL;

  END;


  PXonXoffPort = ^TXonXoffPort;
  TXonXoffPort = OBJECT (TSerialPort)

    Stopped    : BOOLEAN;               { Angehalten wg. Handshake }


    { Konstruktor }
    CONSTRUCTOR Init (Port: TPortNr;
                      Baudrate: TBaudRate;
                      Parity: TParity;
                      StopBits: TStopBits;
                      DataBits: TDataBits);

    { Neue HandShakeOk-Funktion }
    FUNCTION HandShakeOk : BOOLEAN; VIRTUAL;

  END;



{ ------------------------------------------------------------------------- }

FUNCTION Plot (FileName   : PathStr;
               PreString  : STRING;
               Port       : PSerialPort) : INTEGER;

{ ------------------------------------------------------------------------- }
{                                                                           }
{ Die Prozedur gibt die Datei auf die serielle Schnittstelle aus. Benutzt   }
{ wird dazu das ROM-BIOS, also der Interrupt 14h. Die bergebenen Parameter }
{ haben folgende Bedeutung:                                                 }
{                                                                           }
{   FileName        ist der Name der auszugebenden Datei.                   }
{   PreString       ist ein String, der vor dieser Datei noch auf die       }
{                   Schnittstelle ausgegeben wird. Dies kann z.B. ein       }
{                   Steuerstring sein, der dem Plotter das Protokoll        }
{                   mitteilt.                                               }
{   Port            ist ein Zeiger auf ein bereits initialisiertes          }
{                   Schnittstellenobjekt.                                   }
{                                                                           }
{ Das Ergsbnis ist 0 wenn die Datei erfolgreich ausgegeben werden konnte    }
{ und entspricht bei Fehlern IOResult. Bei Fehlern bei der Ausgabe auf die  }
{ Schnittstelle erfolgt ein Programmabbruch, es sei denn, es wird ein       }
{ Objekt bergeben, bei dem die Error-Funktion berdefiniert worden ist.    }
{                                                                           }
{ ------------------------------------------------------------------------- }




{ ************************************************************************* }


IMPLEMENTATION


CONST
  IBaudrate : ARRAY [TBaudRate] OF BYTE = (
    $00, $20, $40, $60, $80, $A0, $C0, $E0
  );

  IParity : ARRAY [TParity] OF BYTE = (
    $00, $08, $18
  );

  IStopBits : ARRAY [TStopBits] OF BYTE = (
    $00, $04
  );

  IDataBits : ARRAY [TDataBits] OF BYTE = (
    $00, $01, $02, $03
  );


  { XON/XOFF }
  Xon  = $11;       { ^Q }
  Xoff = $13;       { ^S }





{ ------------------------------------------------------------------------- }
{ Hilfsprozeduren & Kleinkruscht }





{ ------------------------------------------------------------------------- }
{ Methoden von TSerialPort }




CONSTRUCTOR TSerialPort.Init (Port: TPortNr;
                              Baudrate: TBaudRate;
                              Parity : TParity;
                              StopBits: TStopBits;
                              DataBits: TDataBits);

VAR
  InitVal : BYTE;

BEGIN
  { Wert fr Portnummer bernehmen }
  PortNr := Port - 1;

  { Anzahl Sendeversuche bei Timeout setzen }
  MaxTries := InitTimeoutRetries;

  { Wert fr AL-Register zusammenbauen }
  InitVal := IBaudRate [BaudRate] OR IParity [Parity] OR
             IStopBits [StopBits] OR IDataBits [DataBits];

  { Port initialisieren }
  ASM
    les     di, [Self]
    mov     ah, 0
    mov     al, [InitVal]
    mov     dx, [(TSerialPort PTR es:di).PortNr]
    int     14h
  END;

END;







DESTRUCTOR TSerialPort.Done;
{ Virtueller Destruktor sorgt dafr, da sptere auch virtuell sein mssen }
BEGIN
END;





FUNCTION TSerialPort.HandShakeOk : BOOLEAN;
{ Abstrakte Funktion, mu berladen werden }
BEGIN
  RunError (211);               { Call to abstract method }
END;




PROCEDURE TSerialPort.SetTimeoutRetries (Tries: WORD);
{ Setzt die Anzahl Ausgabeversuche bei Timeout }
BEGIN
  MaxTries := Tries;
END;





PROCEDURE TSerialPort.OutByte (B: BYTE);
{ Gibt ein Zeichen aus }
VAR
  PortStatus : WORD;
  Tries      : WORD;

BEGIN
  { Warten bis die Ausgabe frei ist }
  WHILE (NOT HandShakeOk) DO ;

  Tries := MaxTries;
  REPEAT
    { Zeichen ausgeben }
    ASM
      les     di, [Self]
      mov     dx, [(TSerialPort PTR es:di).PortNr]
      mov     ah, 01h
      mov     al, [B];
      int     14h
      mov     al, 0               { Unteres Byte ist ungltig }
      mov     [PortStatus], ax
    END;
    Dec (Tries);
  UNTIL (Tries = 0) OR ((PortStatus AND $8000) = 0);

  { Portstatus prfen }
  CheckPortStatus (PortStatus);
END;






PROCEDURE TSerialPort.OutString (S: STRING);
{ Gibt einen String auf die serielle aus }
VAR
  I : WORD;

BEGIN
  FOR I := 1 TO Length (S) DO OutByte (BYTE (S [I]));
END;






FUNCTION TSerialPort.InByte : BYTE;
{ Liest ein Zeichen vom seriellen Port. }

VAR
  PortStatus : WORD;

BEGIN
  ASM
    les     di, [Self]
    mov     dx, [(TSerialPort PTR es:di).PortNr]
    mov     ah, 02h
    int     14h
    mov     [@Result], al
    mov     al, 0
    mov     [PortStatus], ax
  END;

  { Portstatus prfen }
  CheckPortStatus (Portstatus);
END;




FUNCTION TSerialPort.ByteAvail : BOOLEAN; ASSEMBLER;
{ Ergibt TRUE wenn ein Eingabebyte verfgbar ist }
ASM
  les     di, [Self]
  mov     dx, [(TSerialPort PTR es:di).PortNr]
  mov     ah, 03h
  int     14h
  mov     al, ah
  and     al, 01h               { Ergebnis direkt in al }
END;






PROCEDURE TSerialPort.Error (Errno: WORD);
{ Bricht in der aktuellen Version das Programm unter Angabe einer
  Fehlermeldung ab. Mu fr andere Reaktionen berladen werden.
}

BEGIN
  Write ('TSerialPort: Port #', PortNr + 1, ': ');
  CASE Errno OF

    1 :  Writeln ('Kein Handshake. Ausgang offen ?');
    2 :  Writeln ('Timeout.');
    3 :  Writeln ('bertragungsfehler.');
    ELSE Writeln ('Unbekannter Fehler, Code = ', Errno);

  END;
  Halt (Errno);
END;




PROCEDURE TSerialPort.CheckPortStatus (PortStatus : WORD);
{ Prft den Status und ruft wenn notwendig Error auf }

BEGIN
  IF ((PortStatus AND $8000) <> 0) THEN BEGIN
    Error (2);
  END ELSE IF ((PortStatus AND $0E00) <> 0) THEN BEGIN
    Error (3);
  END;
END;



{ ------------------------------------------------------------------------- }
{ Methoden von THardWiredPort }




FUNCTION THardWiredPort.HandShakeOk : BOOLEAN; ASSEMBLER;
ASM
  les     di, [Self]
  mov     dx, [(THardWiredPort PTR es:di).PortNr]
  mov     ah, 03h
  int     14h
  xor     bx, bx              { FALSE annehmen }
  and     al, 30h             { DSR & CTS prfen }
  cmp     al, 30h
  jne     @@L1
  inc     bx                  { Ergebnis ist TRUE }
@@L1:
  xchg    ax, bx              { Ergebnis nach ax }
END;





{ ------------------------------------------------------------------------- }
{ Methoden von TXonXoffPort }




CONSTRUCTOR TXonXoffPort.Init (Port: TPortNr;
                               Baudrate: TBaudRate;
                               Parity : TParity;
                               StopBits: TStopBits;
                               DataBits: TDataBits);

BEGIN
  { Stopped-Flag auf FALSE setzen }
  Stopped := FALSE;

  { Und Vorgnger aufrufen ... }
  TSerialPort.Init (Port, Baudrate, Parity, StopBits, DataBits);
END;








FUNCTION TXonXoffPort.HandShakeOk: BOOLEAN;

VAR
  B : BYTE;

BEGIN
  WHILE (ByteAvail) DO BEGIN
    B := InByte;
    CASE B OF

      Xon  :  Stopped := FALSE;
      Xoff :  Stopped := TRUE;
      ELSE    ;                     { Ingorieren }

    END;
  END;

  { Wert von Stopped zurckgeben }
  HandShakeOk := Stopped;
END;



{ ------------------------------------------------------------------------- }
{ Plot-Funktion }


FUNCTION Plot (FileName   : PathStr;
               PreString  : STRING;
               Port       : PSerialPort) : INTEGER;

{ ------------------------------------------------------------------------- }
{                                                                           }
{ Die Prozedur gibt die Datei auf die serielle Schnittstelle aus. Benutzt   }
{ wird dazu das ROM-BIOS, also der Interrupt 14h. Die bergebenen Parameter }
{ haben folgende Bedeutung:                                                 }
{                                                                           }
{   FileName        ist der Name der auszugebenden Datei.                   }
{   PreString       ist ein String, der vor dieser Datei noch auf die       }
{                   Schnittstelle ausgegeben wird. Dies kann z.B. ein       }
{                   Steuerstring sein, der dem Plotter das Protokoll        }
{                   mitteilt.                                               }
{   Port            ist ein Zeiger auf ein bereits initialisiertes          }
{                   Schnittstellenobjekt.                                   }
{                                                                           }
{ Das Ergebnis ist 0 wenn die Datei erfolgreich ausgegeben werden konnte    }
{ und entspricht bei Fehlern IOResult. Bei Fehlern bei der Ausgabe auf die  }
{ Schnittstelle erfolgt ein Programmabbruch, es sei denn, es wird ein       }
{ Objekt bergeben, bei dem die Error-Funktion berdefiniert worden ist.    }
{                                                                           }
{ ------------------------------------------------------------------------- }


CONST
  BufSize = 512;

VAR
  F     : FILE;
  Buf   : ARRAY [1..BufSize] OF BYTE;
  Res   : INTEGER;
  FM    : BYTE;
  Size  : LONGINT;
  Count : WORD;
  I     : WORD;


BEGIN
  { Datei ffnen }
  Assign (F, FileName);
  Reset (F, 1);
  Res := IOResult;
  IF (Res <> 0) THEN BEGIN
    Plot := Res;
    Exit;
  END;

  { Vorstring ausgeben }
  Port^.OutString (PreString);

  { Dateigre ermitteln }
  Size := FileSize (F);

  { Datei ausgeben }
  WHILE (Size > 0) DO BEGIN

    { Puffer lesen }
    IF (Size > BufSize) THEN BEGIN
      Count := BufSize;
    END ELSE BEGIN
      Count := Size;
    END;
    Dec (Size, Count);
    BlockRead (F, Buf, Count);

    { Pufferinhalt ausgeben }
    FOR I := 1 TO Count DO Port^.OutByte (Buf [I]);

  END;

  { Datei wieder schlieen }
  Close (F);

  { Ergebnis auf Ok setzen }
  Plot := 0;
END;




END.


