/* This file is PROCESS.C
**
** contains :
**
**	- process handling
**
** Copyright (c) Rainer Schnitker '92 '93
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "DPMI.H"
#include "PROCESS.H"
#include "SIGNALS.H"
#include "START32.H"
#include "CDOSX32.H"
#include "ADOSX32.H"
#include "EXCEP32.H"
#include "COPY32.H"
#include "RSX.H"
#include "LOADPRG.H"
#include "DEBUG.H"
#include "FPU.H"

/* Global Var */
NEWPROCESS process[N_PRZ + 1];	/* 4 processes ; 0=extender/emu387 */
NEWPROCESS *npz;		/* current running process */
unsigned int current_pid = 1;

/*
** set signal for one process
*/
int send_signal(NEWPROCESS * p, int signal)
{
    if (!p || signal < 0 || signal >= MAX_SIGNALS)
	return EINVAL;
    p->sig_raised |= 1L << signal;
    return 0;
}

/*
** find empty processtable
*/
static NEWPROCESS *find_empty_process(void)
{
    NEWPROCESS *p;

    for (p = &FIRST_PROCESS; p <= &LAST_PROCESS; p++)
	if (p->p_status == PS_EMPTY)
	    return p;
    return NULL;
}

/*
** init processes called from extender 16bit prg
*/
void init_this_process()
{
    NEWPROCESS *p;

    for (p = &FIRST_PROCESS; p <= &LAST_PROCESS; p++)
	p->p_status = PS_EMPTY;

    npz = &RSX_PROCESS;
    npz->pid = current_pid++;
    npz->p_status = PS_RUN;
    npz->p_flags = PF_EXTENDER;
}

/*
** check illegal arguments
*/
int verify_illegal(NEWPROCESS * p, DWORD where, DWORD lenght)
{
    if (where <= p->entry || where + lenght >= p->brk_value)
	return 1;
    else
	return 0;
}

/*
** find processtable
*/
NEWPROCESS *find_process(unsigned pid)
{
    NEWPROCESS *p;

    for (p = &FIRST_PROCESS; p <= &LAST_PROCESS; p++)
	if (p->pid == pid)
	    return p;
    return (NEWPROCESS *) 0;
}

/*
** get wait_status
*/
unsigned sys_wait(unsigned *status)
{
    NEWPROCESS *p;
    int pid = -1;

    for (p = &LAST_PROCESS; p >= &FIRST_PROCESS; p--)
	if (p->pptr == npz && (p->p_flags & PF_WAIT_WAIT)) {
	    *status = p->wait_return;
	    pid = p->pid;
	    p->p_flags &= ~PF_WAIT_WAIT;
	    if (p->p_status == PS_ZOMBIE)
		clean_processtable(p);
	    break;
	}
    return pid;
}

/*
** free process memory and selectors from DPMI-Server
*/
void free_process(NEWPROCESS * p)
{
    if (npz->code32sel == 0)	/* already cleaned ? */
	return;
    FreeMem(p->memhandle);
    FreeLDT(p->code32sel);
    FreeLDT(p->data32sel);
    p->code32sel = 0;
}

/*
** clean processtable
*/
void clean_processtable(NEWPROCESS * p)
{
    memset(p, 0, sizeof(NEWPROCESS));
}

/*
** switch to next program, save mathe state, set npz
*/
int switch_to_process(NEWPROCESS * nextp)
{
    /* if math used, save 387 regs */
    if (npz->p_flags & PF_MATH_USED) {
	if (copro == 3)
	    cpy32_16(emu_sel, copro_struct, &(npz->npx),
		     sizeof(union i387_union));
	else if (copro == 1)
	    do_fnsave((unsigned) &(npz->npx));
    }

    /* change process table */
    npz = nextp;
    cbrkcall = 0;

    /* load 387 regs (init 387) */
    if (copro == 3) {
	if (npz->npx.soft.cwd)	/* emulation done ? */
	    npz->p_flags |= PF_MATH_USED;
	cpy16_32(emu_sel, copro_struct, &(npz->npx), sizeof(union i387_union));

	if (npz->p_flags & PF_MATH_USED)
	    emu_switch(MATH_USED, npz->p_flags & PF_DEBUG);
	else
	    emu_switch(MATH_NEW, npz->p_flags & PF_DEBUG);
    } else if (copro == 1)
	if (npz->p_flags & PF_MATH_USED)
	    do_frstor((unsigned) &(npz->npx));
	else {
	    do_fninit();
	    npz->p_flags |= PF_MATH_USED;
	}

    return 0;
}

/* real mode spawn prototype */
int spawnvpe(int mode, char *name, char **argv , char **env);

/*
** call a real mode program, must be far!
*/
static int far do_spawnve(char *name, char **argp, char **envp)
{
    return spawnvpe(P_WAIT, name, argp, envp);
}

/*
** execute a real-mode program
*/
int realmode_prg(char *filename, char **argv, char **env)
{
    int stack;

    tr.ds = tr.es = tr.ss = tr.fs = tr.gs = ds16real;
    tr.cs = cs16real;
    tr.flags = FLAGS;
    tr.ip = (WORD) (DWORD) do_spawnve;
    tr.reserved = 0;
    tr.sp = (WORD) & stack - 40;

    CallRMprocFar(0, 3, &tr, filename, argv, env);

    if ((int) tr.eax == -1)    /* error real mode prg */
	return errno;
    else {
	EAX = tr.eax & 0xff;	/* return value */
	current_pid++;
	return 0;
    }
}

/*
** load, init and switch to another 32bit program
*/
int exec32(unsigned mode, char *name, int argc, char **argv, int envc, char **env)
{
    NEWPROCESS *child;
    int ret;

    /* look for a empty slot */
    child = find_empty_process();
    if (child == NULL)
	return EAGAIN;

    /* try load a a.out program */
    if ((ret = load_protected_program(name, child)) != 0)
	return ret;

    /* copy arguments,environment */
    argvenv(argc, argv, envc, env, child);

    /* setup new process table */
    child->pid = current_pid++;
    child->pptr = npz;
    if (mode == P_DEBUG)
	child->p_flags |= PF_DEBUG;
    /* start values */
    child->regs.eip = child->entry;
    child->regs.esporg = child->stackp32;
    child->regs.esp = child->stackp32 - 12L;	/* iret frame */
    child->regs.eax = 0;
    child->regs.ebx = 0;
    child->regs.ecx = 0;
    child->regs.edx = 0;
    child->regs.esi = 0;
    child->regs.edi = 0;
    child->regs.ebp = 0;
    child->regs.cs = child->code32sel;
    child->regs.ds = child->data32sel;
    child->regs.es = child->data32sel;
    child->regs.ss = child->data32sel;
    child->regs.fs = child->data32sel;
    child->regs.eflags = 0x3202;
    if (emu_sel)
	child->regs.gs = emu_sel;
    else
	child->regs.gs = child->data32sel;
    child->time_tic = time_tic;

    npz->cptr = child;

    /* switch to prg or debugger, if first emx-porgram */
    if (npz->p_flags & PF_EXTENDER) {
	switch_to_process(child);
	npz->p_status = PS_RUN;
	if (opt_debug)
	    debugger(name, DEB_INIT);
	else
	    back_from_syscall();
    }
    /* else change running prz or wait for debug */

    if (mode == P_WAIT) {
	MarkPageDemand(npz->memaddress, npz->membytes);
	npz->p_status = PS_SYS_SPAWN;
	switch_to_process(child);
	npz->p_status = PS_RUN;
    } else if (mode == P_DEBUG) {
	EAX = child->pid;	/* return process no */
	child->p_status = PS_STOP;
    } else if (mode == P_OVERLAY) {
	NEWPROCESS *this = npz;
	npz->p_flags &= ~PF_MATH_USED;	/* don't save mathe state */
	switch_to_process(npz->pptr);	/* switch to parent */
	free_process(this);	/* free process memory */
	clean_processtable(this);	/* free process table */
	npz->cptr = child;	/* new child */
	child->pptr = npz;	/* new parent */
	switch_to_process(child);	/* switch to new child */
	npz->p_status = PS_RUN;
    } else
	return EINVAL;

    return 0;
}

/*
** fork current process
*/
int sys_fork(void)
{
    NEWPROCESS *child;

    child = find_empty_process();
    if (child == NULL)
	return EAGAIN;

    memcpy(child, npz, sizeof(NEWPROCESS));
    child->p_status = PS_EMPTY;	/* if error, leave empty */

    /* MEMORY per DPMI besorgen */
    if (AllocMem(npz->membytes, &(child->memhandle), &(child->memaddress)))
	return ENOMEM;
    child->dsbase = child->memaddress - child->entry;
    AllocLDT(2, &(child->code32sel));
    child->data32sel = child->code32sel + 8;

    SetBaseAddress(child->code32sel, child->dsbase);
    SetBaseAddress(child->data32sel, child->dsbase);
    SetAccess(child->code32sel, APP_CODE_SEL, DEFAULT_BIT | GRANULAR_BIT);
    SetAccess(child->data32sel, APP_DATA_SEL, DEFAULT_BIT | GRANULAR_BIT);
    SetLimit(child->code32sel, lsl32(npz->code32sel));
    SetLimit(child->data32sel, lsl32(npz->data32sel));

    child->regs.cs = child->code32sel;
    child->regs.ds = child->data32sel;
    child->regs.es = child->data32sel;
    child->regs.ss = child->data32sel;
    child->regs.fs = child->data32sel;
    if (emu_sel)
	child->regs.gs = emu_sel;
    else
	child->regs.gs = child->data32sel;
    child->pid = current_pid++;
    child->pptr = npz;
    child->cptr = NULL;
    child->p_status = PS_STOP;
    child->time_alarm = 0;
    child->time_tic = time_tic;
    npz->cptr = child;

    cpy32_32(npz->data32sel, npz->entry,
	     child->data32sel, child->entry, npz->membytes);

    MarkPageDemand(npz->memaddress, npz->membytes);
    return 0;
}

/*
** global clean-up for extender
*/
void shut_down(int exit_code)
{
    NEWPROCESS *p;

    /* if built-in debugger, clear breakpoints,etc */
    if (opt_debug)
	debugger(NULL, DEB_EXIT);

    /* free memory,selectors */
    for (p = &FIRST_PROCESS; p <= &LAST_PROCESS; p++)
	free_process(p);

    /* clean_up ints,exceptions,... */
    clean_up();

    /* leave protected mode */
    protected_to_real(exit_code);
}

/*
** get more memory from DPMI-Server
*/
DWORD getmem(DWORD bytes, NEWPROCESS * p)
{				/* ret: old break value */
    DWORD retv, pagealign;
    DWORD newaddress, newhandle;

    if (bytes <= p->pagefree) {
	retv = p->brk_value;
	p->brk_value += bytes;
	p->pagefree -= bytes;
    } else {
	pagealign = (bytes + 0xFFF) & 0xFFFFF000L;
	if (ResizeMem(p->membytes + pagealign, p->memhandle,
		      &newhandle, &newaddress))
	    return -1L;
	p->membytes += pagealign;
	retv = p->brk_value;
	p->brk_value += bytes;
	p->pagefree = pagealign - bytes;
	if (!opt_memaccess)
	    SetLimit(p->data32sel, p->membytes + p->entry - 1);
	if (p->memhandle != newhandle) {
	    p->memhandle = newhandle;
	}
	if (p->memaddress != newaddress) {
	    p->memaddress = newaddress;
	    p->dsbase = newaddress - p->entry;
	    SetBaseAddress(p->code32sel, p->dsbase);
	    SetBaseAddress(p->data32sel, p->dsbase);
	    if (opt_debug && p == &FIRST_PROCESS)
		debug_update_breakpoints();
	}
    }
    return retv;
}

char *sigtext[] =
{
    NULL,
    "SIGHUP",
    "SIGINT",
    "SIGQUIT",
    "SIGILL",
    "SIGTRAP",
    "SIGABRT",
    "SIGEMT",
    "SIGFPE",
    "SIGKILL",
    "SIGBUS",
    "SIGSEGV",
    "SIGSYS",
    "SIGPIPE",
    "SIGALRM",
    "SIGTERM",
    "16"
    "17"
    "SIGCLD",
    "19"
    "20"
    "SIGBREAK"};

/*
** exit child and switch to father
*/
int do_exit4c(int bysig)
{
    NEWPROCESS *father;
    WORD ret;

    /* get parent */
    father = npz->pptr;

    if (bysig)
	printf("\nProgram terminate by signal %d , %s\n"
	       ,bysig, sigtext[bysig]);

    /* if father process is Extender */
    if (father->p_flags & PF_EXTENDER)
	shut_down(0);

    if (bysig) {
	ret = 3;		/* return spawn */
	npz->wait_return = bysig;	/* wait: term by sig */
    } else {			/* process terminate */
	if (npz->p_flags & PF_DEBUG)
	    ret = npz->pid;
	else
	    ret = AX & 0xff;
	npz->wait_return = (AX & 0xFF) << 8;
    }

    npz->p_status = PS_ZOMBIE;
    npz->p_flags |= PF_WAIT_WAIT;	/* allow wait() for father */
    npz->p_flags &= ~PF_MATH_USED;	/* don't save mathe state */

    free_process(npz);
    switch_to_process(father);

    if (npz->p_status == PS_SYS_SPAWN) {
	EFLAGS &= ~1L;		/* clear carry, spawn,exec ok */
	EAX = (DWORD) ret;	/* return value to caller */
    } else if (npz->p_status == PS_SYS_PTRACE) {
	EAX = ECX = 0;
    } else if (npz->p_status == PS_SYS_KILL) {
	EAX = ECX = 0;
    }
    npz->p_status = PS_RUN;

    send_signal(npz,SIGCLD);
    return CARRY_OFF;
}
