/* script.c - script functions
	Copyright (C) 1990 Brad K. Clements, All Rights Reserved
*/
#include 	<stdio.h>
#include	<string.h>
#include	<ctype.h>
#include	<stdarg.h>
#ifdef	_MSC_
#include	<malloc.h>
#else
#include	<alloc.h>
#endif
#include	<stdlib.h>
#ifdef	_MSC_
#include	<stdlib.h>
#else
#include	<dir.h>
#endif
#include	"config.h"
#include	"windat.h"
#include	"newwin.h"
#include 	"vsdata.h"
#include	"mem.h"
#include 	"protocol.h"
#include	"data.h"
#include	"netevents.h"
#include	"hostform.h"
#ifdef	TN3270
#include	"screen.h"
#include 	"globals.h"
#endif
#ifdef	SCRIPT

typedef	struct SCRIPT_STRUCT {
	struct twin	*S_tw;
	unsigned long	S_nextevent_time;
	int		S_Token;	/* current Token */

	FILE		*S_scriptfile;
	char		*S_scriptfile_name;
	int		S_lineno;
	int		S_timeout;	/* seconds */
	int		S_vector;
	int		S_timeout_action;
#define	SCRIPT_TOACTION_IGNORE	0
#define	SCRIPT_TOACTION_RETRY	1
#define	SCRIPT_TOACTION_ABORT	2
#define	SCRIPT_TOACTION_JUMP	3
	unsigned long	S_flags;
#define	SFLAG_DEBUG	1
#define	SFLAG_CHECKED	2
#define	SFLAG_ERRORS	4
#define	SFLAG_TIMERON	8
#define	SFLAG_WDATA	16	/* waiting for data to arrive */
#define	SFLAG_DELAY	32	/* in a delay */
#define	SFLAG_FAILME	64	/* set if next timeout should fail */
#define	SFLAG_NOCASE	128	/* ignore case during matches */

	unsigned long	S_jumpto;
#define	MAX_BUFFER_SIZE	128
	char	S_inputbuffer[MAX_BUFFER_SIZE];
#define	MAX_ARGS	20
	char	*S_argv[MAX_ARGS];
	char	*S_ask;
	struct machinfo *S_domain;	/* domain structure we are looking for */
} Script;
#define MAX_SCRIPTS	8
static	Script	*Scripts[MAX_SCRIPTS];
#define	DEFAULT_TIMEOUT 	15
#define	DEFAULT_ACTION		SCRIPT_TOACTION_ABORT

#define	SDEBUG 		if(S->S_flags & SFLAG_DEBUG) nprintf(CONSOLE,

typedef	struct {
	char	*T_keyword;
	int	T_keyvalue;
	int	T_args;		/* number of args allowed */
	int	T_flags;
#define	TFLAGS_NEEDCON	1	/* must have a connection before proceeding */
#define	TFLAGS_DATAOK	2	/* receipt of data can restart this token */
} Token;

#define	TOKEN_IGNORE	0
#define	TOKEN_EOF	1
#define	TOKEN_WRITE	2
#define	TOKEN_LOOKFOR	3
#define	TOKEN_TIMEOUT	4
#define	TOKEN_MOVETO	5
#define	TOKEN_FLAGS	6
#define	TOKEN_LOOKAT	7
#define	TOKEN_WAITFOR	8
#define	TOKEN_LABEL	9
#define	TOKEN_ECHO	10
#define	TOKEN_MESSAGE	11
#define	TOKEN_PRESSKEY	12
#define	TOKEN_ONERROR	13
#define	TOKEN_TELNET	14
#define	TOKEN_RLOGIN	15
#define	TOKEN_RSHELL	16
#define	TOKEN_DELAY	17
#define	TOKEN_END	18
#define	TOKEN_MARK	19
#define	TOKEN_CLOSE	20
#define	TOKEN_DETACH	21
#define	TOKEN_ASK	22
#define	TOKEN_SEND	23
#define	TOKEN_SHOW	24
#define	TOKEN_ATTACH	25
#define	TOKEN_TFLAGS	26
#define	TOKEN_SETSNAME	27
#define	TOKEN_NEWKEYMAP	28
#define	TOKEN_NETSTAT	29
#define	TOKEN_GOTO	30

static	Token	Tokens[]={
	{";",TOKEN_IGNORE,0,0},
       {"\032",TOKEN_EOF,0,0},
       {"write",TOKEN_WRITE,1,TFLAGS_NEEDCON},
       {"lookfor",TOKEN_LOOKFOR,1,TFLAGS_NEEDCON|TFLAGS_DATAOK},
       {"lookat",TOKEN_LOOKAT,3,TFLAGS_NEEDCON|TFLAGS_DATAOK},
       {"timeout",TOKEN_TIMEOUT,1,0},
       {"moveto",TOKEN_MOVETO,2,TFLAGS_NEEDCON},
       {"flags",TOKEN_FLAGS,1,0},
       {"waitfor",TOKEN_WAITFOR,1,TFLAGS_NEEDCON|TFLAGS_DATAOK},
	{"label",TOKEN_LABEL,1,0},
       {"echo",TOKEN_ECHO,1,TFLAGS_NEEDCON},
       {"message",TOKEN_MESSAGE,1,0},
       {"presskey",TOKEN_PRESSKEY,1,TFLAGS_NEEDCON},
       {"onerror",TOKEN_ONERROR,1,0},
       {"telnet",TOKEN_TELNET,1,0},
       {"rlogin",TOKEN_RLOGIN,1,0},
       {"rshell",TOKEN_RSHELL,1,0},
       {"delay",TOKEN_DELAY,1,0},
       {"end",TOKEN_END,0,0},
       {"mark",TOKEN_MARK,0,0},
       {"close",TOKEN_CLOSE,0,TFLAGS_NEEDCON},
       {"detach",TOKEN_DETACH,0,TFLAGS_NEEDCON},
       {"ask",TOKEN_ASK,1,0},
       {"send",TOKEN_SEND,1,TFLAGS_NEEDCON},
       {"show",TOKEN_SHOW,1,0},
       {"attach",TOKEN_ATTACH,1,0},
	{"tflags",TOKEN_TFLAGS,1,TFLAGS_NEEDCON},
       {"setsname",TOKEN_SETSNAME,1,TFLAGS_NEEDCON},
       {"keymap",TOKEN_NEWKEYMAP,2,TFLAGS_NEEDCON},
       {"netstat",TOKEN_NETSTAT,1,0},
       {"goto",TOKEN_GOTO,1,0},
	{NULL,0,0}	
};

#define	RES_OK		-1
#define	RES_CLOSED	-2
#define	RES_FAILED	-3
#define	RES_ABORT	-4	/* failed immediate */
#define	RES_RETURN	0

#define	SARGV(x)	S->S_argv[x]
extern	struct	twin	*addsess();
extern	struct	twin	*current;
extern	int	viewmode;

static	char	*flag_modes[]={
		"debug",
		"nodebug",
	        "case",
	        "nocase",
		NULL
};

static	char	*send_modes[]={
		"ask",
	        "myip",
		"password",
		NULL
};

static char	*show_modes[]={
		"console",
		"current",
		NULL
};

static char	*waitfor_modes[]={
		"data",
		"connection",
		NULL
};

static int
Script_Lookup(char *vals[],char *keyword)
{
	int	x;

	for(x=0; vals[x]; x++)
	       	if(!stricmp(vals[x],keyword))
		       	return(x+1);
	return(0);
}

Token
*Script_GetToken(int token)
{
	int x;

	for(x=0;Tokens[x].T_keyword; x++)
	       	if(Tokens[x].T_keyvalue == token)
		       	return(&Tokens[x]);
	return(NULL);
}

void
Script_Log(Script *S, int mode, char *fmt, ...)
{
	 va_list	list;
	 char	xbuff[128];

	 if(!(S->S_flags & SFLAG_DEBUG) && !mode)
		return;

	 va_start(list,fmt);
	 vsprintf(xbuff,fmt, list);
	 va_end(list);
	 nprintf(CONSOLE,"Script (%s): %s\n",S->S_scriptfile_name, xbuff);
}



int
Script_Event(int type, Script *SS, int data)
{
	Script	*S = NULL;
	struct machinfo *m;
	int	x;

	if(!SS)  {
	       	if(data > -1 && data < MAX_SCRIPTS)
		       	S = Scripts[data];
		else
		       	nprintf(CONSOLE,"Script_Event: Invalid Data Ptr\n");
	}
	else if(type != SCRIPT_DOMAIN) {
	        if(SS == Scripts[SS->S_vector])
		       	S = SS;
		else
		       	nprintf(CONSOLE,"Script_Event: Invalid SS ptr\n");
	}
	if(!S && type != SCRIPT_DOMAIN)
	       	return(0);

	switch (type) {

	        case SCRIPT_EVENT :	/* we timed out */
		        if(S->S_timeout_action != SCRIPT_TOACTION_IGNORE &&
			        (S->S_flags & SFLAG_FAILME) &&
				(S->S_flags & SFLAG_CHECKED)) {
			      	/* either jump or abort */
				if(S->S_timeout_action == SCRIPT_TOACTION_ABORT) {
					Script_Log(S,0,"Command (%s) timed out, aborting.",SARGV(0));
					Script_Close(S->S_vector);
					return(0);
				}
			 	Script_Log(S,0,"Command (%s) timed out, jumping",SARGV(0));
			 	fseek(S->S_scriptfile, S->S_jumpto, 0);
				S->S_timeout_action = SCRIPT_TOACTION_ABORT;
				Script_ReadLine(S);
			}
		 	if(S->S_timeout_action == SCRIPT_TOACTION_IGNORE)
			       	Script_ReadLine(S);
			Script_Process(S);
			break;
		case SCRIPT_DATA :
			if(Script_GetToken(S->S_Token)->T_flags & TFLAGS_DATAOK) {
				/* something might have changed */
				Script_Process(S);
			}
		       	break;

		case SCRIPT_PROC :

		       	break;

		case SCRIPT_CLOSE :
		case SCRIPT_FORCE :	/* forced closed by user */

		       	S->S_tw->Script = NULL;
		       	S->S_tw = NULL;
			if(type == SCRIPT_FORCE) {
			       	Script_Close(S->S_vector);
				statline();
				return(0);
			}
		 	Script_ReadLine(S);
		 	Script_Process(S);
			break;
		case SCRIPT_DOMAIN :	/* a domain name lookup was ok */
		  	m = (struct machinfo *) SS;
		 	for(x=0; x < MAX_SCRIPTS; x++) {
			       	if(Scripts[x] && (m == Scripts[x]->S_domain)) {
				       	Script_Process(Scripts[x]);
					m->flags &= ~MFLAGS_SCRIPT;
					return(0);
				}
			 }
		  	nprintf(CONSOLE,"Script: Invalid domain lookup handle\n");
			break;
		default :;
	}
	return(0);
}

Script			
*Script_Spawn(char *f, struct twin *t)
/* start up new script file */
{
	FILE	*file;
	Script	*S;
	int	x;
	int	round = 0;
	char	filename[80],*ff,xbuff[128],fname[13],extension[5];
	struct	machinfo	*mp;

	for(x=0; x < MAX_SCRIPTS; x++) {
	       	if(!Scripts[x])
		       	break;
	}
	if(x == MAX_SCRIPTS) {
	       	nprintf(CONSOLE,"Script: Too many script files open\n");
		return(NULL);
	}
	strcpy(filename,f);
#ifndef	_MSC_
	if(!(EXTENSION & fnsplit(f,NULL,NULL,NULL,NULL)))
#else
	extension[0] = 0;
	_splitpath(f,fname,xbuff,filename,extension);
	if(extension[0] == 0)
#endif
	       	strcat(filename,".scr");
	ff = filename;
retry:;
	if(file = fopen(ff,"r")) {
		S = (Script *) mem_calloc(sizeof(Script), 1);
		if(!S) {
		       	nprintf(CONSOLE,"Script: Not enough memory \n");
			fclose(file);
			return(NULL);
		}
	 	S->S_scriptfile = file;
		S->S_scriptfile_name = strdup(filename);
		S->S_vector = x;
		S->S_timeout = DEFAULT_TIMEOUT;
		S->S_timeout_action = DEFAULT_ACTION;
		S->S_lineno = 0;
		Scripts[x] = S;
		if(t) {
		       	S->S_tw = t;
			t->Script = S;
		}
		Stimerset(USERCLASS,SCRIPT_FUNC, x, 0);	/* ready to spawn */
       }
	else {
	       	if(!round) {
		       	mp = Shostlook("script_dir");
			if(mp) {
#ifndef	_MSC_
			       	fnsplit(ff,NULL,NULL,fname,extension);
#else
				_splitpath(ff,filename,xbuff,fname,extension);
#endif
			       	sprintf(xbuff,"%s%s%s",mp->hname,fname,extension);
				ff= xbuff;
				round = 1;
				goto retry;
			}
		}
	        nprintf(CONSOLE,"Script: Unable to open file (%s)\n",ff);
		return(NULL);
	}
	return(S);
}


int
Script_Close(int vector)
{
	 Script *S;

	 if(vector < 0 || vector > MAX_SCRIPTS)
		return(-1);
	 S = Scripts[vector];
	 if(!S)
		return(-1);
	 fclose(S->S_scriptfile);
	 Script_Log(S,1,"closed");
	 Stimerunset(USERCLASS,SCRIPT_FUNC,vector); /* clear any pending timers */
	 if(S->S_tw) {
	 	S->S_tw->Script = NULL;
		statline();
	 }
 	 if(S->S_ask)
		mem_free(S->S_ask);
	 mem_free(S->S_scriptfile_name);
	 mem_free(S);
	 Scripts[vector] = NULL;
	 return(0);
}

int
Script_CloseAll()
{
	int	t;

	for(t=0; t < MAX_SCRIPTS; t++)
	       	if(Scripts[t] && Scripts[t]->S_scriptfile) {
		       	fclose(Scripts[t]->S_scriptfile);
			nprintf(CONSOLE,"Scriptfile (%s): killed\n",
			       Scripts[t]->S_scriptfile_name);
		}
	return(0);
}


int
Script_Process(Script *S)
{
	int	rc;

	if(!(S->S_flags & SFLAG_CHECKED)) {

		while(Script_ReadLine(S) != TOKEN_EOF);
		if(S->S_flags & SFLAG_ERRORS) {
		        Script_Close(S->S_vector);
			return(0);
		}
	 	S->S_flags |= SFLAG_CHECKED;
	 	rewind(S->S_scriptfile);
		Script_ReadLine(S);
	}

	while(1) {
	       	Script_Log(S,0,"%s ",SARGV(0));
		S->S_flags &= ~SFLAG_FAILME;
		rc = Script_Token(S);
		switch (rc) {
		        case RES_OK :
			        if(S->S_flags & SFLAG_TIMERON) {
				        Stimerunset(USERCLASS,SCRIPT_FUNC, S->S_vector);
					S->S_flags &= ~SFLAG_TIMERON;
					Script_Log(S,0,"Timer off");
				}
		       		Script_ReadLine(S);
			        continue;
			case RES_CLOSED :
			       	return(0);
			case RES_FAILED : /* failed, delay */
			 	Script_Log(S,0,"%s Failed",SARGV(0));
			 	S->S_flags |= SFLAG_FAILME;
			       	if(!(S->S_flags & SFLAG_TIMERON)) {
				        Stimerset(USERCLASS,SCRIPT_FUNC, S->S_vector, S->S_timeout);
					S->S_flags |= SFLAG_TIMERON;
					Script_Log(S,0,"Timer on %d Seconds",S->S_timeout);
				}
				return(0);
			case RES_ABORT :	/* failed immediate */
			       	S->S_flags |= SFLAG_FAILME;
			       	Script_Log(S,0,"%s Failed Imm",SARGV(0));
			 	Stimerunset(USERCLASS,SCRIPT_FUNC, S->S_vector);
				Stimerset(USERCLASS,SCRIPT_FUNC, S->S_vector, 0);
				return(0);
			default:;	  /* sleep this many seconds */
			       	Script_Log(S,0,"Sleeping %d Seconds",rc);
			       	Stimerunset(USERCLASS,SCRIPT_FUNC, S->S_vector);
			 	Stimerset(USERCLASS,SCRIPT_FUNC, S->S_vector, rc);
				return(0);
		} /* end switch */
	} /* end while */
}


long
Script_FindLabel(Script *S, char *label)
{
	long	here, there;
	char	xbuff[32];
	int	Token;

	strcpy(xbuff,label);
	there = 0L;
	here = ftell(S->S_scriptfile);
	rewind(S->S_scriptfile);
	while(1) {
	       	Token = Script_ReadLine(S);
		if(Token == TOKEN_EOF)
		       	break;
		if(Token == TOKEN_LABEL) {
		       	if(!stricmp(xbuff,SARGV(1))) {
			       	there = ftell(S->S_scriptfile);
				break;
			}
		 }
	}
	fseek(S->S_scriptfile, here, 0);
	return(there);
}

int
Script_Token(Script *S)
{
	 struct twin *tw;
	 int port,x;
	 long  l;

	if(!(S->S_tw) && (Script_GetToken(S->S_Token)->T_flags & TFLAGS_NEEDCON)) {
		Script_Log(S,0,"Function (%s) Needs open connection, failed",SARGV(0));
/*		Script_Close(S->S_vector); */
		return(RES_ABORT);
	}
	switch(S->S_Token) {

		case TOKEN_IGNORE:
			break;

		case TOKEN_EOF:
		case TOKEN_END:
			Script_Close(S->S_vector);
			return(RES_CLOSED);

		case TOKEN_WRITE :	/* write to connection */
write_stuff:;

		       {
		          	unsigned char *c;
			 	struct twin *t;
				int 	x = 255;
				t = current;
				current = S->S_tw;
			  	c = SARGV(1);
				while(*c && x) {
					vt100key(*c & 0xff);
					c++;
					x--;
				}
				current = t;
			}
		 	break;
		case TOKEN_LOOKFOR :
			if(Script_Match(S,0,0,SARGV(1)))
			       	return(RES_FAILED);
		       	break;

		case TOKEN_TIMEOUT :
		        S->S_timeout = atoi(SARGV(1));
		 	break;

		case TOKEN_MOVETO :
#ifdef	TN3270
		       	if(!S->S_tw->tnmode) {
#else
			{
#endif
				Script_Log(S,0,"Command (%s) aborted, connection not in 3270 mode",SARGV(0));
				return(RES_ABORT);
			}
#ifdef	TN3270
			CursorAddress = SetBufferAddress((atoi(SARGV(1))-1), (atoi(SARGV(2))-1));
		 	RefreshScreen();
		 	
#endif
		        break;
		case TOKEN_FLAGS :

		       	switch (Script_Lookup(flag_modes,SARGV(1))) {

				case 1 :		/* debug */
				       	S->S_flags |= SFLAG_DEBUG;
				 	break;
				case 2 :		/* nodebug */
				       	S->S_flags &= ~SFLAG_DEBUG;
				 	break;
				case 3 :		/* case */
				       	S->S_flags &= ~SFLAG_NOCASE;
				 	break;
				case 4 :		/* nocase */
				       	S->S_flags |= SFLAG_NOCASE;
				 	break;
			} /* end switch */
		 	break;
		case TOKEN_LOOKAT :
		       {
		       		int row, col;

				row = atoi(SARGV(1));
				col = atoi(SARGV(2));
				if(Script_Match(S,col,row,SARGV(3)))
				       	return(RES_FAILED);
			}
		       	break;
		case TOKEN_WAITFOR :
		       	switch (Script_Lookup(waitfor_modes, SARGV(1))) {

				case 1 :	/* data */
					if(S->S_flags & SFLAG_WDATA) {
						S->S_flags &= ~SFLAG_WDATA;
						return(RES_OK);
					}
				 	S->S_flags |= SFLAG_WDATA;
					return(RES_FAILED);
				case 2 :	/* connect */
#ifndef	_MSC_
				       	if(portlist[S->S_tw->pnum]->state != SEST)
#else
					if(netest(S->S_tw->pnum))
#endif
					       	return(RES_FAILED);
					break;
				default :;
					Script_Log(S,0,"Invalid keyword (%s) in waitfor command",SARGV(1));
					return(RES_ABORT);
			} /* end switch */
			break;
		case TOKEN_LABEL :
		       	break;
		case TOKEN_ECHO :
		       	VSwrite(S->S_tw->vs,SARGV(1), strlen(SARGV(1)));
		 	break;
		case TOKEN_MESSAGE :
		       	nprintf(CONSOLE,SARGV(1));
		 	break;
		case TOKEN_PRESSKEY :
		       {
			       	extern struct twin *current;
			 	struct twin *t;
				unsigned int x;

				t = current;
				current = S->S_tw;
				sscanf(SARGV(1),"%x",&x);
				vt100key(x);
				current = t;
			}
			break;
		case TOKEN_ONERROR :
		       if(!(stricmp(SARGV(1),"ignore"))) {
			      	S->S_timeout_action = SCRIPT_TOACTION_IGNORE;
				break;
			}
		       else
			      if(!(stricmp(SARGV(1),"abort"))) {
				      S->S_timeout_action = SCRIPT_TOACTION_ABORT;
				      break;
			      }
		       	/* ### FALL THROUGH to GOTO */

		case TOKEN_GOTO :
		       { 	/* its a label, look for it */
			      	long offset;
			 	int oldToken = S->S_Token;

			 	if(offset = Script_FindLabel(S,SARGV(1))) {
				        if(oldToken == TOKEN_GOTO) {

						fseek(S->S_scriptfile, offset,0);
						Script_Log(S,0,"Goto %s at %ld",SARGV(1),offset);
						return(RES_OK);
				        }
				        S->S_timeout_action = SCRIPT_TOACTION_JUMP;
					S->S_jumpto = offset;
				}
			 	else {
				        Script_Log(S,1,"Missing target of onerror/goto jump");
					Script_Close(S->S_vector);
					return(RES_CLOSED);
				}
			}
		 	break;
		case TOKEN_TELNET :
			port = 23;
		 	goto process;
		case TOKEN_RLOGIN :
		        port = 513;
		 	goto process;
		case TOKEN_RSHELL:
		        port = 514;
process:;
			if(S->S_tw) {	/* session already open */
			    Script_Log(S,0,"Session already open, command (%s) ignored",
				       	SARGV(0));
			    break;
		        }
		 	if(*SARGV(1) != '@') { /* not a script, resolve name */
			       	if(!Sgethost(SARGV(1))) { /* see if its in our config.tel file */
				        int i;
					if((i = Sdomain(SARGV(1))) > 0)  { /* domain lookup */
					       	S->S_domain = Slooknum(i);
					 	S->S_domain->flags |= MFLAGS_SCRIPT;
					       	return(RES_FAILED);
					}
					return(RES_ABORT);
				 }
		  	}
		 	if(tw = addsess(SARGV(1),port)) {
			       	S->S_tw = tw;
				tw->Script = S;
				current = tw;
			       	viewmode = 10;
			}
		 	else
			       	return(RES_ABORT);
		 	break;
		case TOKEN_DELAY :
		       	if(S->S_flags & SFLAG_DELAY) {
				S->S_flags &= ~SFLAG_DELAY;
			       	break;
		        }
		 	S->S_flags |= SFLAG_DELAY;
			return(atoi(SARGV(1)));
			break;
		case TOKEN_CLOSE :
		       	netclose(S->S_tw->pnum);
			Stask();
			netputuev(CONCLASS,CONCLOSE,S->S_tw->pnum);
		 	break;
		case TOKEN_DETACH :	/* disconnect connection */
		       	if(S->S_tw) {
				S->S_tw->Script = NULL;
				S->S_tw = NULL;
			}
		 	break;
		case TOKEN_ASK :	/* ask the user for something */
		       {
		       		struct twin *t;
				int 	i;

			       	if(!(S->S_ask))
				       	S->S_ask = mem_calloc(80,1);

				t = current;
				*S->S_ask = 0;	/* trunc preallocated ask string */
				wrest(console);
				nprintf(CONSOLE,"%s",SARGV(1));
				while (0 >= (i = RSgets(console->vs,S->S_ask,79)))
					Stask();
				nprintf(CONSOLE,"\n");
				wrest(t);
				if(i == 27)
				       	return(RES_ABORT);
			 }
			break;
		case TOKEN_SEND :	/* send something */
		        {
				extern unsigned char myipnum[];
	     			extern unsigned long ftppassword;

		       		switch (Script_Lookup(send_modes,SARGV(1))) {


					case 1 : /* send the ask data */
					        if(!S->S_ask)
						       	return(RES_ABORT);
						SARGV(1) = S->S_ask;
						break;
					case 2 :	/* ip address */
						SARGV(1) = SARGV(0);
					 	sprintf(SARGV(0),"%d.%d.%d.%d",
						        myipnum[0],myipnum[1],myipnum[2],myipnum[3]);
						break;
					case 3 :
					       	SARGV(1) = SARGV(0);
		                                ftppassword = (( ( (long) rand() <<  16L) + (long) rand()) | 1L) & 0x5f5e0ff;
			  			/* above & is to be sure that no passwords are
						   longer than 8 characters */

                                                /* can't allow it to be 0 */
      						sprintf(SARGV(0),"%ld\r",ftppassword);
					 	break;
					default :;
						if(*SARGV(1) == '$') {	/* dos environment lookup */
						        if(getenv(SARGV(1)+1)) {
							       	strcpy(SARGV(0),getenv(SARGV(1)+1));
							}
						 	else	{
								*SARGV(0) = 0;
								Script_Log(S,0,"Unknown DOS Environment Variable '%s' in send command",SARGV(1)+1);
							}
						 	SARGV(1) = SARGV(0);
							goto write_stuff;
					  	}
						Script_Log(S,0,"Invalid arg (%s) in send command",
						       	SARGV(1));
						return(RES_ABORT);
				}	/* end switch */
				goto write_stuff;

		        }
			break;
		case TOKEN_SHOW :	/* display a certain screen */
		case TOKEN_NETSTAT :	/* write netstat to screen */
		       {
		       	  struct twin *t,*s;

			  switch (Script_Lookup(show_modes,SARGV(1))) {

				 case 1 :	/* console */
					if(S->S_Token == TOKEN_NETSTAT) {
						t = console;
						goto netstat;
					}
					wrest(console);
				 	break;
				 case 2 :	/* current */
					if(S->S_tw) {
						if(S->S_Token == TOKEN_NETSTAT) {
							t = S->S_tw;
							goto netstat;
						}
					       	wrest(S->S_tw);
					}
					else {
					       	Script_Log(S,0,"Show/ netstat command failed. No current connection");
						return(RES_ABORT);
					}
				 	break;
				default :;	/* look for this name */
attach_me:;
				       	s = current;
				 	t = s->next;
					while(t && t != s) {
					       	if(!stricmp(t->mname,SARGV(1))) {
							if(S->S_Token == TOKEN_NETSTAT) {
							       goto netstat;
						        }
						       	wrest(t);
							if(S->S_Token == TOKEN_ATTACH) {
							       	t->Script = S;
								S->S_tw = t;
							}
							return(RES_OK);
						}
					 	t = t->next;
					}
				 	if(stricmp(current->mname,SARGV(1))) {
						Script_Log(S,0,"Show/netstat couldn't find session '%s'",
						       	SARGV(1));
						return(RES_ABORT);
					}
					if(S->S_Token == TOKEN_ATTACH) {
					       	current->Script = S;
						S->S_tw = current;
					}
				 	else
					       	t = current;

			   }	/* end switch */
netstat:;
			if(S->S_Token == TOKEN_NETSTAT) {
				statcheck(t);
				return(RES_OK);
 		        }

		      }
			   break;
		case TOKEN_ATTACH :
		       	if(S->S_tw) {
				Script_Log(S,0,"Attach Failed. Already attached");
				return(RES_ABORT);
			}
		 	goto attach_me;
		case TOKEN_TFLAGS :
			sscanf(SARGV(1),"%ld",&l);
			S->S_tw->flags |= l;
			break;
		case TOKEN_SETSNAME :
		       	strncpy(S->S_tw->mname,SARGV(1),13);
		 	S->S_tw->mname[14] = 0;
			break;
		case TOKEN_NEWKEYMAP :
		       	if(!stricmp(SARGV(1),"default")) {
				Key_Free_Lib(Default_Key_Config);
				Default_Key_Config = Key_Load_File(SARGV(2));
				break;
			}
		       	if(S->S_tw->key_map)
			       	Key_Free_Lib(S->S_tw->key_map);
			S->S_tw->key_map = NULL;
			S->S_tw->key_map = Key_Load_File(SARGV(2));
			break;
		default:;
		 	nprintf(CONSOLE,"Script (%s): Unsupported Command (%s), line %d, ignored\n",
			       	S->S_scriptfile_name, SARGV(0),S->S_lineno);
			break;
	}	/* end switch */
	return(RES_OK);
}

int
Script_ReadLine(Script *S)
{
  	int count;
	int	token;
	Token	*T;


	if(feof(S->S_scriptfile))
	       	return(S->S_Token = TOKEN_EOF);
	if(fgets(S->S_inputbuffer,MAX_BUFFER_SIZE,S->S_scriptfile) == NULL)
	       	return(S->S_Token  = TOKEN_EOF);
	S->S_lineno++;
	if((count = Script_Argify(S->S_inputbuffer,S->S_argv,S)) < 0) 
		goto abort_me;
	if(count < 1 || !strlen(SARGV(0)))
	       	return(S->S_Token = TOKEN_IGNORE);
	count--;
	for(T = Tokens; T->T_keyword; T++) {
	        if(!strnicmp(T->T_keyword,SARGV(0),strlen(T->T_keyword))) {
			if(count != T->T_args && T->T_keyvalue != TOKEN_IGNORE) {
		        nprintf(CONSOLE,"ScriptFile (%s): Invalid # of args: %d found, wanted %d, line %d\n",
			        S->S_scriptfile_name,count,T->T_args,S->S_lineno);
abort_me:;
			 	S->S_flags |= SFLAG_ERRORS;
				S->S_Token = TOKEN_EOF;
			 	return(TOKEN_EOF);
			}
		 	return(S->S_Token = T->T_keyvalue);
	        }
	}

	nprintf(CONSOLE,"ScriptFile (%s): Unknown keyword (%s), line %d\n",
	        S->S_scriptfile_name,SARGV(0),S->S_lineno);
	S->S_flags |= SFLAG_ERRORS;
	return(S->S_Token = TOKEN_EOF);
}

int
Script_Argify(char *string, char **argv, Script *S)
{
	char	*c = string;
	int	str = 0;
	char	*i,*o;
	int	base;

	while(1) {
		while(*c && isspace(*c))
		       	c++;

		if(!*c || *c == ';')
		       	return(str);
		if(*c == '"') {
		       	c++;
		       	argv[str++] = c;
			while(*c) {
			       	if(*c == '"' && *(c-1) != '\\')
				       	break;
			       	c++;
			}
			*c = 0;
			o = i = argv[str-1];
			while(*i) {
			       if(*i == '\\') {
					i++;
					switch (*i) {
					        case '\\' :
						       break;
						case 'e'  :
						       *i = 27;
							break;
						case 'r' :
						       *i = 13;
							break;
						case 'n' :
						       *i = 10;
							break;
						case ';' :
						       	break;
						default :;
							if(*i <= '3' && *i >= '0' ) {/* octal escape */
							        base = (*i++ - '0') * 64;
								if(*i) {
								       	base += 8 * (*i++ - '0');
									if(*i) {
									       	base += (*i - '0');
									}
								 }
							         if(*i)
								       	*i = base;
							 }
						  	 else {
								nprintf(CONSOLE,"Script: Invalid Escape Sequence line %d, position %d char %d '%c'\n",
								       	S->S_lineno, i - string + 1,*i,*i);
								return(-1);
							 }
							 break;
					 } /* end switch */
			       }
			       *o++ = *i++;	
		        } /* end while */
			*o++  = 0;
			continue;
		}
		argv[str++] = c;
		while(*c && !isspace(*c)) {
		       	if(*c == ';' && *(c-1) != '\\')  {
			       	*c = 0;
				break;
			}
		       	c++;
		}
		if(*c) {
		        *c = 0;
			c++;
		}
	}
}

int
Script_Match(Script *S, int col, int row, char *string)
/* returns 0 if matched string at this location */
{
	char	*c,CC,cc;
	int rc, len,x,y;

	if(VSvalids(S->S_tw->vs))
	       	return(-1);
	len = strlen(string);
	if(!len)
	       	return(0);	/* null length always matches */
	if(!col) {	/* search entire screen */
	       	CC = *string;
	 	if((S->S_flags & SFLAG_NOCASE) && isupper(CC))
		       	CC = tolower(CC);
		for(y=0; y < rows; y++) {
		        c = &VSIw->linest[y]->text[0];
			for(x = 0; x <cols - len; x++, c++) {
			       	if((S->S_flags & SFLAG_NOCASE) && isupper(*c))
				       	cc = tolower(*c);
				else
				        cc = *c;
			       	if(CC == cc) {
					if(S->S_flags & SFLAG_NOCASE)
					       	rc = strnicmp(c,string,len);
					else
					       	rc = strncmp(c,string,len);
					if(!rc)
					       	return(0);
				 }
			  }
		  }
	   	  return(-1);
	}
	if(!row)
	       	row = rows - 1;
       c = &VSIw->linest[row-1]->text[col-1];

	if(S->S_flags & SFLAG_NOCASE)
	       	rc = strnicmp(c,string,len);
	else
	       	rc = strncmp(c,string,len);
	if(rc)
	       	Script_Log(S,0,"Match failed at (row %d,col %d) found '%*.*s'",
		       	row,col,len,len, c);
	return(rc);
}

#endif
