/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       ICMP.C
**     SYSTEM   NAME:       IP
**     ORIGINAL AUTHOR(S):  Wim van Campen
**     VERSION  NUMBER:     1.0
**     CREATION DATE:       1990/6/18
**
** DESCRIPTION: Contains the ICMP implementation
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   ICMP.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/ICMP.C_V  $
**              
**                 Rev 1.1   21 Nov 1990 14:23:20   etstjan
**              No explicit note
**              
**                 Rev 1.0   20 Nov 1990 16:14:36   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/UDPIP/IP/VCS/ICMP.C_V   1.1   21 Nov 1990 14:23:20   etstjan  $";
#endif

#include      <stdio.h>
#include      <stdlib.h>
#include      <string.h>
#include      <power3.h>
#include      <beholder.h>

#include      "ip.h"                   /* general ip defines */
#include      "ipcodes.h"              /* and ip return codes */
#include      "iplib.h"                /* include ip library */
#include      "iplayer.h"

/**************************************************************
** NAME:        ICMPRec
** SYNOPSIS:    int ICMPRec(PRREQ ThisReq, CIPHEAD *IPHeader,
**                          DATALINK *IPPacket, int BCast,
**                          int ECode);
**              
** DESCRIPTION: Processes reception of ICMP messages. In case
**              of an 'ECHO' or 'TIME' packet, a response is
**              sent. In case of an error code, the error
**              code is passed to the corresponding protocol
**              handler. A redirect message is handled by
**              modifying the associated routing table entry.
**              A a final step, the ICMP message is passed up
**              for raw processing.
** RETURNS:     NO_ERR --> always
**         
**************************************************************/
int ICMPRec(PRREQ ThisReq, CIPHEAD *IPHeader,
            DATALINK *IPPacket, int BCast, int ECode)
{
  CICMPHEAD      TempHead;
  CICMPHEAD      *ICMPHeader = &TempHead;
  CIPHEAD        RecIPHeader;
  DATALINK       *NewPacket;
  DATALINK       *SaveForRaw;
  ULONG          TimeSt;
  BYTE           SaveHead[8];       /* save header of responded messages   */
  int            ErrCode = NOERR;   /* error code to be reported           */
  PRREQ          Command = ERRREQ;  /* request to be passed to upper layer */
  int            LinkFreed = 0;

  if (ThisReq == RECREQ) {
    if (IPPacket->Length < 8) {
      IPStat.NrICMPShort++;
      IPReport(ICMPSHORT, IPHeader, IPPacket);
      return NO_ERR;
      }

    ByteToCicmp(ICMPHeader, IPPacket->DataStart);
    if (CompCheck(IPPacket->DataStart, IPPacket->Length) != 0) {
        IPStat.NrICMPChkErr++;
        IPReport(ICMPCHERR, IPHeader, IPPacket);
        return NO_ERR;
        }

     /* prepare a datalink to save contents of packet for raw processing */
     if ((SaveForRaw = IPBufGet(sizeof(DATALINK))) == NULL) {
       IPReport(NOSPACE, IPHeader, IPPacket);
       return NO_ERR;
       }
    *SaveForRaw = *IPPacket;
    memcpy(SaveHead, IPPacket->DataStart, 8); /* save this message's header */

    IPPacket->SaveData = 1;             /* this packet's data must be saved */
    IPPacket->DataStart += 8;
    IPPacket->Length -= 8;

    if (BCast)                          /* set specific destination address */
      IPHeader->Destin = IPHeader->ExtraAddress;

    switch (ICMPHeader->Type) {
      case ECHO      :  IPStat.NrEchoReqR++;
                        ICMPHeader->Type = ECHOREPLY;
                        MakeRespondOptions(IPHeader);
                        ICMPSend(IPHeader, ICMPHeader,
                                 IPPacket, HOPS_ALLOWED);
                        LinkFreed = 1;
                        break;
      case TIME      :  IPStat.NrTimeStReqR++;
                        ICMPHeader->Type = TIMEREPLY;
                        if (IPPacket->Length < 12) {
                          if ((NewPacket = IPDataLinkGet(12)) == NULL) {
                            IPReport(NOSPACE, IPHeader, IPPacket);
                            IPBufFree(SaveForRaw);
                            return NO_ERR;
                            }
                          NewPacket->ThisType = DATAHD;
                          memcpy(NewPacket->DataStart,
                                 IPPacket->DataStart, 4);
                          DiscardAll(IPPacket);
                          IPPacket = NewPacket;
                          }
                        TimeSt = TimeMilSec();
                        *(ULONG *)(IPPacket->DataStart + 4) = htonl(TimeSt);
                        *(ULONG *)(IPPacket->DataStart + 8) = htonl(TimeSt);
                        MakeRespondOptions(IPHeader);
                        ICMPSend(IPHeader, ICMPHeader,
                                 IPPacket, HOPS_ALLOWED);
                        LinkFreed = 1;
                        break;
      case DSTUNRCH  :  UPDSTAT(UnRchR, 6);
                        ErrCode = ENETUNREACH - ICMPHeader->Code;
                        break;
      case PARAMPROB :  IPStat.NrParamProbR++;
                        ErrCode = EPARAMPROB;
                        break;
      case SRCQUENCH :  IPStat.NrSrcQueR++;
                        ErrCode = ESRCQUENCH;
                        Command = SRCQNCHREQ;
                        break;
      case REDIRECT  :  UPDSTAT(RedirecR, 4);
                        ErrCode = DUMMYCODE;  /* just set an error code */
                        Command = REDIRREQ;
                        break;
      case TIMEX     :  UPDSTAT(TimExR, 3);
                        ErrCode = ETTLEXC - ICMPHeader->Code;
                        break;
      case ECHOREPLY :  IPStat.NrEchoRepR++;
                        break;
      case TIMEREPLY :  IPStat.NrTimeStRepR++;
                        break;
      case AMREQ     :
      case AMREPLY   :  break;
      }

    if (!LinkFreed) {                    /* all switches equal right now */
      memcpy(SaveForRaw->DataStart, SaveHead, 8);        /* restore data */
      IPBufFree(IPPacket);
      }

    if (ErrCode != NOERR) {             /* ErrCode means further processing */
      if (SaveForRaw->Length < 28) {
        IPStat.NrICMPIncom++;
        }
      else {
        if (ByteToCip(&RecIPHeader, SaveForRaw->DataStart + 8) == NOERR) {
          if (SaveForRaw->Length < (USHORT)(RecIPHeader.HLength) + 8) {
            IPStat.NrICMPIncom++;
            }
          else {
            if (Command == REDIRREQ) {        /* process a redirect message */
              ModRoute(IPHeader, &RecIPHeader,
                       ICMPHeader->MoreInfo.GateWayAdd);
              }
            else {                              /* process an error message */
              Protoc[RecIPHeader.Protocol](Command, &RecIPHeader,
                                           IPPacket, BCast, ErrCode);
              }
            }
          }
        }
      }
    /* pass up to raw socket handling */
    if (IPRawRec(RECREQ, IPHeader, SaveForRaw, BCast, 0) != NO_ERR) {
      IPBufFree(IPHeader);
      DiscardAll(SaveForRaw);
      }
    return NO_ERR;
    }
  return ILL_REQUEST;
}


/**************************************************************
** NAME:        ICMPSend
** SYNOPSIS:    void ICMPSend(CIPHEAD *IPHeader,
**                            CICMPHEAD *ICMPHeader,
**                            DATALINK *IPPacket,
**                            DIR_CHOICE Direct);
**              
** DESCRIPTION: Sends an ICMP message, whose header should be
**              present in ICMPHeader. The ICMP checksum will
**              be added before sending the packet. The header
**              of the IP packet is formed out of IPHeader,
**              taking into account that the source and
**              destination address are swapped and the Tos,
**              Ident, Flag, Time and Protocol field are
**              set to ICMP values.
**              IPPacket is freed on sending the packet.  
** RETURNS:     
**************************************************************/
void ICMPSend(CIPHEAD *IPHeader, CICMPHEAD *ICMPHeader,
              DATALINK *IPPacket, DIR_CHOICE Direct)
{
  DATALINK       *ICMPHeadBuf;
  CIPHEAD        NewIPHead;
  static USHORT  ICMPIdent = 0;

  if ((ICMPHeadBuf = IPDataLinkGet(8)) == NULL) {
      IPReport(NOSPACE, NULL, IPPacket);
      return;
      }
  /* update statistics */
  switch (ICMPHeader->Type) {
    case ECHO      :  IPStat.NrEchoReqS++;
                      break;
    case ECHOREPLY :  IPStat.NrEchoRepS++;
                      break;
    case DSTUNRCH  :  UPDSTAT(UnRchS, 6);
                      break;
    case TIMEX     :  UPDSTAT(TimExS, 2);
                      break;
    case PARAMPROB :  IPStat.NrParamProbS++;
                      break;
    case SRCQUENCH :  IPStat.NrSrcQueS++;
                      break;
    case REDIRECT  :  IPStat.NrRedirecS++;
                      break;
    case TIME      :  IPStat.NrTimeStReqS++;
                      break;
    case TIMEREPLY :  IPStat.NrTimeStRepS++;
                      break;
    default        :  IPStat.NrOtherS++;
                      break;
    }
    
  ICMPHeadBuf->NextOne = IPPacket;
  ICMPHeadBuf->ThisType = ICMPHD;
  CicmpToByte(ICMPHeadBuf->DataStart, ICMPHeader);
  *(USHORT *)(ICMPHeadBuf->DataStart + 2) = OnesSum(LinkCheckSum(ICMPHeadBuf));

  /* make new IP header from old header */
  NewIPHead = *IPHeader;
  NewIPHead.Tos = ICMPTOS;
  NewIPHead.Ident = ICMPIdent++;
  NewIPHead.Flag = 0;
  NewIPHead.Time = TTL;
  NewIPHead.Protocol = ICMP;
  AddressSwap(&(NewIPHead.Source), &(NewIPHead.Destin));
  IPSend(&NewIPHead, ICMPHeadBuf, Direct);
}

/**************************************************************
** NAME:        SendProb
** SYNOPSIS:    void SendProb(BYTE Code, BYTE Type,
**                            BYTE Pointer, CIPHEAD *ThisHeader,
**                            DATALINK *Packet);
** 
** DESCRIPTION: Sends an ICMP message with Code as code,
**              Type as type and with Pointer as problem
**              pointer (should be zero if not used).
**              Packet will be discarded.
**              This function currently is in use for sending
**              'Parameter Problem' and 'Destination
**              Unreachable' messages.
** RETURNS:     
**************************************************************/
void SendProb(BYTE Type, BYTE Code, BYTE Pointer,
              CIPHEAD *ThisHeader, DATALINK *Packet)
{
  CICMPHEAD  ICMPHead;

  ICMPHead.MoreInfo.GateWayAdd = 0l;           /* initialize union to 0 */
  ICMPHead.Type = Type;
  ICMPHead.Code = Code;
  ICMPHead.MoreInfo.Pointer = Pointer;
  Packet->DataStart -= ThisHeader->HLength;
  Packet->Length = ThisHeader->HLength + 8;
  ThisHeader->OLength = 0;
  ICMPSend(ThisHeader, &ICMPHead, Packet, HOPS_ALLOWED);
}

/**************************************************************
** NAME:        SendParamProb
** SYNOPSIS:    void SendParamProb(BYTE Pointer,
**                                 CIPHEAD *ThisHeader,
**                                 DATALINK *Packet);
** 
** DESCRIPTION: Sends an ICMP 'Parameter Problem' message.
**              Pointer is the problem pointer.
**              Packet will be discarded and ThisHeader
**              is freed.
**              IP statistics are updated.
** RETURNS:     
**************************************************************/
void SendParamProb(BYTE Pointer, CIPHEAD *ThisHeader, DATALINK *Packet)
{
  SendProb(PARAMPROB, PNTINDIC, Pointer, ThisHeader, Packet);
  IPStat.NrIllOptions++;
  IPReport(ILLOPTION, ThisHeader, NULL);
}
