{
  cNesting.pas    Copyright (c) 1991   by Georg Post

 - part of PCPC
 - handles the procedure & function calling and parameter passing
 - all nested Pascal procedures transformed into global C functions
 - auxiliary parameters are added as required

BUGS:
 -   duplicate inner procedure names aren't separated?
 -   all ? local Const & Types should be made Global and Unique
 -   only secondary aux parameters need _p prefix.
 -   recursive flag often redundant for globals containing nested procedures.
 -   implicit semilocal record types: aux parameter typing is impossible.
}

unit cNesting;  { compiler: Turbo Pascal 4.0 }
{$S+,R+}

interface

uses pcpcdata,
     pascannr, { the source code scanner }
     cDeclara; {output formatter, declaration parts}

const
  maxNest=15; {max. nesting of procedure declarations}
var
    procStack: array[1..maxNest] of pide;  {points to proc/funct id}
    lastProc: integer; {index to procStack of declared proc/funct names }

procedure initNesting;
procedure termNesting;
procedure CreatePlist(pi:pide; lv:integer; imported:boolean);
procedure pushIdentL(pi:pide);  {uses global vars of scope}
procedure extendHeader(pid:pide; lv: integer);
procedure insertGlobId(var name:symbol; delta:integer);
procedure concatParam(fp,q:ptpel); { add proc/funct call parameter}
procedure callCode(var stat:scanStatus; fp: char); {fUnct or proceD }
procedure finishCall;

implementation

const
  maxCall=10; {maximum of nested (function) calls in expressions}

var
    recursive: array[1..maxNest] of boolean; {True= called from inside itself}

    callStack : array[1..maxCall] of pide;  {called proc/func identifiers }
    levlStack : array[1..maxCall] of integer; {proc/func ident. level is... }
      { one less than the level of local variables to the proc/funct ! }
    typeStack: array[1..maxCall] of ptpel;
    lastCall: integer; {for proc calls, indexes call-,levl- AND typeStack }
    callContext, {mark if Procedure of Function}
    paraVariant: {signal weak param. variants} array[1..maxCall] of char;

{
see PCPCCODE.TXT on the "theory of AuxParams" for procedure unnesting.
Where to call the following actions ?

 initNesting:    once at program start
 CreatePlist:    _procEnd, _functEnd  IF scope.actual > 2.
                  Recall that scope.actual goes up @ _blockEntry and
                  goes down @ _blockEnd or _forwDecl
 pushIdentL:     _getAddr _fileAddr _parAddr _valpAddr
 extendHeader:   _blockEnd after destroying inner block procedures...
 extendParList:  _doCall

Bracket codes adorn the C Output, for the Reorder postprocessor:
- [ statement part | aux Var declarations ]   to shift things in front
- ( < proc A > ( subprocedure B ) body A )   recursively extract B
- ( < proc A > (....) body >    echo A header before extractions.
- ( (subproc..) body <proc A) > echo A and move it before body, too.

Prepare here for the macro post-post-processor part in Reorder:
- the AuxParam  Definition and Call sequences are "macro declarations"
- at the places of use, there is a Macro Call

All Procedure and Type identifiers (why not the Const ? ) (of level>=globScope)
  go into a special Symbol Table so that we can detect their re-use and append
  "serial numbers" to them, in order to create distinct globals.
}

const
  maxNewParam=40; {never add more AuxParams}
  maxLocals=60;   {max number of local functions}
  maxCaller=20;   {loc. proc. caller list, not only ITSELF or its subproc}
  globScope=3;    {actual scope > globScope INSIDE a non-global procedure}

type
  pGlobId=^GlobIdData; {for auxiliary symbol table of "escaped" local symbols}
  GlobIdData=record
    key:symbol;
    SerNumb:integer;
    rg,lf: pGlobId; {for binary tree storage of escaped GlobIds}
  end;
  pAuxParam=^AuxParam;
  AuxParam=record {data  for auxiliary parameter: 30 bytes}
    befor: pAuxParam; {linked list hooks}
    auxType: ptpel;
      {param. type pointers. may be Nil for local(=volatile) type names}
    locType: pGlobId; {if auxType is invalid, find type Id here }
    auxTypeCl, {copy of auxType^.cl}
    auxValVar: char; {L or A if auxType derived from val/var param}
    auxName: symbol; { parameter name }
    auxClass,   {0,2: Var  1,3: Const  >=2: has a LOCAL type ! }
    auxReuse,   { keep the reuse number of identifier }
    auxLev: byte;  { scope level}
  end;
  pNewParam=^NewParam;
  NewParam=record
    { describe the list of AuxParams for any non-global procedure: 300 bytes}
    next,prev: pNewParam; {double linked list}
    nParam,    {total number of added parameters}
    nPrimary,  {nb of primary AuxParams, <= nParam}
    nestLevel,
    serial: integer;   {serial number for synonyms }
    myName:pide;  {volatile? }
    procName: symbol; {= myName^.name}
    hasParam: boolean; {tells if the procedure has regular parameters}
    auxP: array[1..maxNewParam] of pAuxParam;  {list of AuxParams}
    caller: array[1..maxCaller] of pNewParam;  {list of nonglobal callers}
    nCallers: integer; {number of non-global procedures calling this here}
  end;
var
  pFirst,pLast,pReserv: pNewParam;
  auxLast,auxReserv: pAuxParam;

{ BUG: "parType" entries may point to type elements whose Name pointers
  become invalid after the end of the scope of a procedure (local types).
  -> difficulty to reconstruct parameter types at call time.
}

{------- escaped symbols table : ------------

  table of all globalized local symbols
  The purpose is to tag Serial Number on non-unique globalised symbols.
  New Proc/Funct entries are made, or old counters incremented, by CreatePlist.
  The counter value is extracted there, too.
  Local Types and Consts (level>2) enter here, too.
  the 2  legal calls are:  "InsertGlobId","SerialNumber"
}

var  GIRoot: pGlobId;

procedure killGlobId(p:pGlobId); {call it with p=GIroot }
begin
  if p<>Nil then begin
    killGlobId(p^.rg);
    killGlobId(p^.lf);
    dispose(p);
  end;
end;

procedure searchGlobId(var name:symbol; var p:pGlobId;
  var found,small: boolean);
{ ACTION: search for  name in the "escaped" symbol table
  CALLER: insertGlobId, serialNumber
  INPUT : symbol name
  OUTPUT: p= result if Found. If not found, p=insertion pointer;
          small= True if Name is smaller than the ID at p^.key
}
var q:pGlobId;
begin
  found:=false; small:=false;
  p:=GIRoot;
  q:=nil;  { tree search loop}
  while p<>nil do begin {=nil, end of search}
    q:=p;
    found:=(name=p^.key);
    if not found then begin
      small:=(name<p^.key);
      if small then p:=p^.lf else p:=p^.rg;
    end else p:=nil; {and q is result pointer}
  end;
  p:=q; {the insertion point if NOT found, the good pointer if found ! }
end;

procedure insertGlobI(var name:symbol; delta:integer; var p:pGlobId);
{ ACTION: inserts Name into the Escaped Id list
  CALLER: CreatePlist
  INPUT : delta= increment for serial number attached to Name
  OUTPUT: p = pointer to inserted GlobId
to re-use any old GlobId , increment serial number by delta
Delta is normally 1 except for Forward declaration where it's -1 !
}
var q:pGlobId;  found,small: boolean;
begin
  searchGlobId(name,p,found,small);
  if not found then begin {must create a new heap object}
    if delta<0 then begin {error message}
      error('unknown Forward '+name);
    end else begin
      q:=p; new(p);
      with p^ do begin key:=name; SerNumb:=1 {0?} ;
(*       better version: SerNumb:=delta ?         *)
        lf:=Nil;rg:=Nil;
      end;
      if q=Nil then GIRoot:=p {p was the first GlobId at all} else begin
        if small then q^.lf:=p else q^.rg:=p; {tree linking}
      end;
    end;
  end else begin
    p^.SerNumb:=p^.SerNumb + delta;
  end;
end;

procedure insertGlobId(var name:symbol; delta:integer);
var p:pGlobId;
begin InsertGlobI(name,delta,p) end;

function serialNumber(var name:symbol): integer;
{ ACTION: returns serial Number >=1 if name is an Escaped Id, else return 0
  CALLER: createPlist, callCode
}
var p:pGlobId;  found,small: boolean;
begin
  searchGlobId(name,p,found,small);
  if found then serialNumber:=p^.SerNumb else serialNumber:=0;
end;

{-----------  aux parameter data manager ------------}

procedure NewAuxParam(var p:pAuxParam);
begin
  if auxReserv=Nil then new(p) else begin
    p:=auxReserv; auxReserv:=p^.befor;
  end;
  p^.befor:=auxLast; auxLast:=p;
end;

procedure scratchAuxParams;
{ to be called as we fall back to a global level}
var p,q:pAuxParam;
begin p:=auxLast;
  while p<>Nil do begin
    q:=p^.befor; p^.befor:=auxReserv; auxReserv:=p; p:=q;
  end;
end;

procedure CreatePlist(pi:pide; lv:integer; imported:boolean);
{ ACTION: write header for global function, prepare stack entry for local one.
  CALLER: Cbulk.CompilerAction: _ProcEnd, _FunctEnd
  INPUT : pi = proc/funct ID, lv = nesting level, imported=from some "Uses".

 Is called whenever a new procedure/function header has been parsed
 Can make a global procedure's parameter list directly. Later if nonglobal.
 Make new auxPar entry if lv is high enough (=level of local symb here >=4)
}
var p:pNewParam;  snb: char; piName: symbol; k:integer;
    myType,pt:ptpel;
begin
 piName:=pi^.name;
 recursive[lv]:=false; {init the recursion indicator }
 if lv> globScope then begin {we have some local (nested) procedure}
   insertGlobId(piname,1);   {new global identifier needed}
   {nested local procedure, prepare the aux list }
   if pReserv=Nil then new(p) else begin {from the reserve}
     p:=pReserv; pReserv:=pReserv^.prev;
   end;
   rangeCheck(p,'Cnesting');
   p^.serial:=SerialNumber(piname); {should be at least 1 ? }
   with p^ do begin
     nParam:=0; nCallers:=0;
     myName:=pi; {the procedure identifier}
     myType:=pi^.typof;
     hasParam:=myType^.q<>Nil;
     pt:=myType^.p; {result type}
     if pt<>Nil then hasParam:=hasParam or (pt^.cl='s'); {string-valued}
     {idName[0]} procName:=piname; { = pi^.name}
     nestLevel:=lv; next:=Nil; prev:=pLast;
   end;
   if pLast<>Nil then pLast^.next:=p else pFirst:=p;
   pLast:=p;
   outC('/*(*/`');
   headerDeclared:=true; {opening bracket and uglyTrick, here}
 end else begin {global proc, may directly write ANSI header }
   if (pi^.class<>unitId) then begin {else, parasitic from Prog header !}
     outC('/*(*/`');  { bracket for 2nd pass: nested procedure extraction}
     outC('/*<*/`');
     ANSIheader(pi, 0, true,imported); {write the complete header}
     outC('/*>*/`');
     headerDeclared:=true;
   end;
 end;
end;

procedure DestroyPlist;
{ ACTION: delete the last procedure list entry
  CALLER: BackToLevel, when we exit from some global-level function body.
}
var p:pNewParam;
begin
  if pLast<>Nil then begin p:=pLast;
    pLast:=p^.prev;
    if pLast<>Nil then pLast^.next:=Nil else pFirst:=Nil; {list empty}
    p^.prev:=pReserv; pReserv:=p;
  end;
end;

procedure killPlist;
var p,q: pNewParam;
    a,b: pAuxParam;
begin
  p:=pReserv; while p<>Nil do begin q:=p; p:=p^.prev; dispose(q) end;
  p:=pLast;   while p<>Nil do begin q:=p; p:=p^.prev; dispose(q) end;
  a:=auxReserv; while a<>Nil do begin b:=a; a:=b^.befor; dispose(b) end;
  a:=auxLast;   while a<>Nil do begin b:=a; a:=b^.befor; dispose(b) end;
end;

procedure reconstruct(ap:pAuxParam; var pref,suff: str80);
{ local type name reconstruction
  auxType may belong to a deleted local scope , perhaps lost or invalid
 CALLER: dumpInfo, for AuxClass>=2
}
var k:integer; ck:char;  s:str20;
begin
  suff:=''; pref:='';
  with ap^ do begin
    if locType<>Nil then begin {easy case, we kept a name}
      s:='_t';
      for k:=1 to lenIde do begin ck:=locType^.key[k];
        if ck>' ' then s:=s+ck;
      end;
      outC(s+' ');
    end else begin { perhaps pointer to or array of global type?}
      if auxTypeCl='s' then outC('String ')
      else if auxTypeCl='e' then outC('Set ')
      else if auxTypeCl='E' then outC('short ') {enum}
      else if auxTypeCl='F' then outC('File ')
      else if auxType<>Nil then
        fullTypeName(auxType, 0, pref,suff)
      else outC({'void '} 'char ');  {records for example: non-recoverable}
    end;
    if auxValVar='Y' then suff:=suff+'[]';
    if auxValVar='P' then pref:='*'+pref;
  end;
end;

procedure makeAuxParam(var ap:pAuxParam; pid:pide; lv:integer);
{difficult cases are: semilocal implicit arrays/records...}
var tid:pide;
begin
  newAuxParam(ap); {init with locType=Nil?}
  with ap^ do begin
    auxName:=pid^.name;
    auxLev:=lv;
    auxReuse:=pid^.reUse;
    if pid^.class=constId then auxClass:=1  else auxClass:=0;
    auxType:=pid^.typof; auxValVar:=auxType^.cl;
    if (auxValVar='L')or(auxValVar='A') then auxType:=auxType^.p;
    if auxType^.cl='S' then auxType:=auxType^.p; {reduce subrange}
    auxTypeCl:=auxType^.cl;
    tid:=auxType^.tname; {type identifier for named type,
       const/var/field Id for no-name type }
    if tid^.defLevel > 2 then begin
      auxClass:=auxClass+2; {local type gets special translation}
      if tid^.class=typeId then begin {named semilocal type}
        insertGlobI(tid^.name,0,locType); auxValVar:=' ';
      end else begin  {anonymous semilocal type}
        if auxValVar='Y' then auxType:=auxType^.q; {save some arrays}
        if auxValVar='P' then auxType:=auxType^.p; {save some pointers}
        tid:=auxType^.tName;
        if tid^.defLevel>2 then begin {unrecoverable loss ?}
          if tid^.class=typeId then begin
            insertGlobI(tid^.name,0,locType);  {named type ptr is saved}
          end else begin
            error('Semilocal no-name type.');
            locType:=Nil;
          end;
          auxTypeCl:=auxType^.cl; auxType:=Nil;
        end;
      end;
    end;
  end; {with}
end;

procedure DumpInfo(p:pNewParam);
{ ACTION: output Declare and Call macro definitions for the delayed prototype
          of a nested function declaration, at the end of a global procedure.
  CALLER: BackToLevel.
  INPUT : p points to a procedure list entry

  If the header's list of recursive callers isn't empty,
  the aux params are completely known HERE, not before.
  The REORDER postprocessor does the final work.
  Const Ids never get the * prefix.
  Macros may be multiline up to 512 characters!
}
var i,j,k: integer;
    option: byte;
    pref,suff,s:str80;
    macroId: str40;
    arg,numb: str20; {for the 2 versions of macro identifier}
    valp,varp,ok: boolean;
    pt:ptpel;  nk,ch,cla:char;
begin
  with p^ do { if nCallers>0 then ? } begin
    {macro expansion for extendHeader }
    macroId:='';
    for k:=1 to lenIde do begin nk:=procName[k];
      if nk>' ' then macroId:=macroId+nk;
    end;
    outC('/*#'+macroId+'D ');  {D=Define procedure header}
    if (not hasParam)and(nParam<=0) then outC('void');
    for i:=1 to nParam do with auxP[i]^ do begin
      if (i>1)or(hasParam) then outC(',');
      if auxClass>=2 then begin  {option:=2}
        reconstruct(auxP[i],pref,suff);
      end else begin {option:=0;}
        fullTypeName(auxType, 0, pref,suff);
      end;
        {C allows explicit no-name types for ANSI parameters}
      if (not odd(auxClass)) and not (auxTypeCl in ['Y','s','e']) then begin
        {glue the * for non-constants, non-arrays}
        {Bug here for auxTypeCl propagated from Val/Var ? }
        if suff<>'' then begin pref:='*('+pref; suff:=suff+')' end
        else pref:='*'+pref;
      end;
      s:=pref+'_p';  {all aux params have underbar prefix}
      if auxReuse=1 then s:=s+'_';
      for k:=1 to lenIde do begin nk:=auxName[k]; if nk>' ' then s:=s+nk; end;
      s:=s+suff; outC(s);
    end;
    outC('*/`'); {end of Define macro }
     { the 2nd one is the Call macro
       Syntax of each argument entry, where [  ]  means option:
        , Number [ & ] [ Prefix . ] Name
      }
    outC('/*#'+macroId+'C ');
    s:='';
    for i:=1 to nParam do with auxP[i]^ do begin
      j:=1; arg:='';
      if auxReuse=1 then arg:='_';
      repeat ok:=(j<=lenIde);
        if ok then begin ch:=auxName[j]; ok:= (ch > ' ');
          if ok then arg:=arg+ch;
          j:=succ(j);
        end;
      until not ok;
      pt:=auxType;
      valp:=(auxValVar='L'); varp:=(auxValVar='A');
      {valp:=(pt^.cl='L'); varp:=(pt^.cl='A');
       if valp or varp then pt:=pt^.p;}
      cla:=auxTypeCl;
      if valp and ((cla='Y')or(cla='s')or(cla='e')or(cla='R')) then
        arg:=auxPrefix + '.'+arg;  {local copy option for postprocessor}
      if not (odd(auxClass) or varp or ((cla='Y')or(cla='s')or(cla='e')))
        then arg:='&'+arg;  {const,array, var params: no address prefix}
      str(auxLev,numb);  {the argument level is vital information}
      if (i>1)or(hasParam) then numb:=','+numb;
      outC(numb+arg);
    end; {for i}
    outC('*/`'); {end of 2nd macro }
  end;
end;

procedure propagate(pid:pide; pp:pNewParam; lv:integer; var didIt:boolean);
{ ACTION: add identifier Pid to the AuxParam list in Pp, with level lv.
  CALLER: checkPlist
  INPUT : pid= an identifier, lv=its level
  OUTPUT: didIt=true if added, false if Pid already there before.
}
var found,over: boolean; i:integer;
begin  found:=false; i:=0;
  with pp^ do begin
    repeat i:=succ(i); over:=(i>nParam);
      if not over then found:=
        (auxP[i]^.auxName = pid^.name) and (auxP[i]^.auxLev=lv);
    until found or over;
    didIt:=over;
    if over then begin
      rangeCheck(pp,'propagat');
      if i>maxNewParam then begin
        error('too many Primary Aux Params!');
        didIt:=false;
      end else begin
        nParam:=i;
        makeAuxParam(auxP[i], pid,lv);
      end;
    end;
  end;
end;

procedure oneMoreParam(into,from: pNewParam; i:integer; var done:boolean);
{ ACTION: copy AuxParam from one list into another: idClass,idLev,parType...
  CALLER: transitiveClose
  INPUT : take parameter From^.idName[i], append to lists inside Into^.
}
var lv:byte; idn:symbol;
    k:integer; found,over:boolean;
begin
  idn:=from^.auxP[i]^.auxName; lv:=from^.auxP[i]^.auxLev;
  with into^ do begin k:=0; found:=false;
    repeat k:=succ(k); over:=(k>nParam);
      if not over then found:=
        (auxP[k]^.auxName=idn) and (auxP[k]^.auxLev=lv);
    until found or over;
    done:=over;
    if over then begin
      rangeCheck(into,'into:chk');
      if k>maxNewParam then begin
        error('too many secondary AuxParams!');
        done:=false;
      end else begin
        nParam:=k;
        auxP[k]:=from^.auxP[i]; {propagate pointers only}
      end;
    end;
  end;
end;

procedure transitiveClose(lv: integer; pb: pNewParam);
{ ACTION: complete the network of auxiliary parameters (AuxParams).
          After that, we may forget the parser's semantic data for levels >lv.
  CALLER: BackToLevel, at end of a global procedure.
  INPUT : lv= highest excluded level, pb=ptr to last NewParam entry

  Run through all calling chains to get the "transitive closure"
  of AuxParam lists of inner procedures, for semilocal parameters.
  This is done at the end of a procedure belonging to global level lv.
  At that point, the inner procedures have their "primary" AuxParams,
  i.e. semilocals from outside their scope which they directly use.
  They still need the "secondary and higher" AuxParams which they do not
  access BUT must hand along to some other inner procedure they call.

  Algorithm (brute force):
    For all nested procedures P             (of level lp > lv)
     For all nested procedures K calling P  (of level lk > lv)
       For all aux. parameters x of P       (of level lv <=lx < lk)
         Add x to parameter list of K       (because x is external to K)
  Repeat this triple loop until saturation occurs.

  BUG: it may propagate through the network some semilocal identifiers which
    happen to be local Ids of some innocent host function. To avoid the
    collision, I prefix any propagated Id with _p. If a semilocal variable
    has some implicit (no-name) record type, the AuxParam ANSI header entry
    will have a bogus type reference.
}
var pprev,pa: pNewParam; done,yes: boolean;
    nLinks,i,k,n,lk: integer;
begin   {in general, pa= first and pb= last list element }
  pa:=pb; n:=0;
  repeat
    pa^.nPrimary:=pa^.nParam; {fix number of gathered primary aux params }
    pprev:=pa^.prev; {look backwards}
    done:=(pprev=Nil); {then pa = 1st list element}
    if not done then done:=(pprev^.nestLevel) <=lv;
    if not done then pa:=pprev;
    {debug counter} n:=n+1;
  until done or (n>maxLocals);
  if n>maxLocals then error('Too many local functions.');
  n:=0;
  repeat {brute-force link saturation loop}
    nLinks:=0; {no new links created}
    pprev:=pb; {subloop from pb downto pa }
    repeat
      with pprev^ do begin {double loop: all callers, all parameters}
        for k:=1 to nCallers do begin
          lk:=(caller[k]^.nestLevel); {level of caller}
          yes:= (lk > lv);
          if yes then for i:=1 to nParam do with auxP[i]^ do begin
            yes:=(auxLev>=lv)and (auxLev<lk);
            {parameter is semilocal, but outside of caller}
            if yes then { propagate(parTable[i],caller[k],lv, yes); }
              oneMoreParam(caller[k],pprev,i, yes);
            if yes then nLinks:=nLinks+1; {something was missing somewhere}
          end;
        end;
      end;
      done:=(pprev=pa);
      if not done then pprev:=pprev^.prev;
    until done;
    {beware of infinite loop !! } n:=n+1;
  until (nLinks=0)or(n>maxLocals);
  if n>maxLocals then error('AuxParam propagation Bug');
  {no more links added, self-consistency achieved }
end;

procedure BackToLevel(lv: integer; var pp:pNewParam);
{ ACTION: if back to global scope, calls transitiveClose, writes macro
          definitions, and destroys procedure list entries.
  CALLER: extendHeader
  INPUT : lv some level >= globScope,
  OUTPUT: pp=Nil if all done.

  returns to last local procedure of level lv, in Plist.
  If lv <= globScope, may forget all procedures of level > lv.
  Do not forget those procedures until level goes down to 3!
  This enables us to adjust procedure headers when recursions are at work.
  Make transitive closure for level lv if forgetIt is allowed !
}
var done,forgetIt,loop:boolean;
begin
  done:=(pLast=Nil);
  forgetIt:=(lv<=globScope);
  if forgetIt and (not done) then begin
    transitiveClose(lv,plast);
  end;
  pp:=pLast;
  while not done do begin
    done:=(pp^.nestLevel <= lv);
    if not done then begin
      if forgetIt then begin
        dumpInfo(pp); {write the definite procedure header macros }
        DestroyPlist; pp:=pLast;
      end else pp:=pp^.prev;
      done:=(pp=Nil);
    end;
  end;{while}
end;

procedure checkPlist(pid:pide; lv,markLevel: integer);
{ ACTION: check if the semilocal identifier pid is in the last list of
          level lv. Else add it with Id scope = markLevel.
  CALLER: pushIdentL

  later must check all of the dependent caller list:
  TransitiveClose will propagate  aux. parameters to all relevant callers
}
var i:integer; pp: pNewParam;  ok:boolean;
begin
  pp:=pLast;  { has level > lv  if other more nested headers survive }
  while pp^.nestLevel>lv do begin pp:=pp^.prev;
    rangeCheck(pp,'chkPlist');
  end;
  propagate(pid,pp, markLevel, ok);
end;

{---------------------------------------}

procedure pushIdentL(pi:pide);
{ ACTION: pushes Const or Var IDs, and  VAL or VAR parameters on expr. stack
          Implements the columns 2 and 3 of the above Coding Table.
          If pi^ is a Parameter, add some * as required.
  CALLER: Cbulk.compilerAction _constRef, _getAddr, _parAddr, _valpAddr ...
  INPUT : uses globals idLevel (level of last parsed Id) ... from  scope
}
var pt:ptpel; isConst,valp,varp:boolean;
    cla:char;
begin pushIdent(pi);  {the naked Id, may have preceding _ if reused}
  isConst:=(pi^.class=constId);
  pt:=pi^.typof;  valp:=(pt^.cl='L'); varp:=(pt^.cl='A');
  if valp or varp then pt:=pt^.p; cla:=pt^.cl;
  with scope do if (idLevel>=globScope) and (idLevel<actual) then begin
    { here a SEMILOCAL var is used as an Aux.Param. : column 3 .
      pointer indirection required if NOT (VarParameter/ Const/ Array) }
(*  if expStack[lastExp][1]='_' then begin {delete the system underbar}
      delete(expStack[lastExp], 1,1);      {this is a BUG: undeclared Id}
    end;
*)
    unaryOp('_p','',0,0); { auxparam underbar prefix}
    if not (isConst or varp or (cla='Y')or(cla='s')or(cla='e')) then
      unaryOp('*','',1,1);
    { now make sure that this ident is already in expanded param list}
    checkPlist(pi,actual,idLevel);
  end else begin {use at origin level: case 2 }
    { if array/record VAL par, we work with a local copy! }
    if valp and ((cla='Y')or(cla='s')or(cla='e')or(cla='R')) then
      unaryOp(AuxPrefix,'',0,0);
  end;
end;

procedure extendHeader(pid:pide; lv: integer);  {column 4 of coding table}
{ ACTION: output the ANSI procedure header for "pLast" with AuxParam macro .
          this is called instead of ANSIheader , for nonglobal functions .
          any id here gets a * if NOT array/string/set
  CALLER: Cbulk.compilerAction_blockEnd ( for non-global procedures )
  INPUT : lv= function's nesting level, pid points to its name Ide record.
  BUG   : Assume that all semilocal Type declarations are globalised !
}
var pp:pide; pt: ptpel; i,k:integer;  nk:char;
    pnp: pNewParam;
    var pref,suff:str80;
    macro:str40;
begin
  BackToLevel(lv,pnp); {may scratch higher level aux headers IF lv = glob.+1 }
  if lv> globScope then begin {else no header revision required }
    rangeCheck(pnp,'extHeadr');
    with pnp^ do { always, not only  if nParam>0 then } begin
      {debug  code }
      if pid<>myName then begin {proc identifiers don't match !!!}
        writeln('BUG in Extended header routine: proc. identifiers ');
        writeln(pid^.name,'<>',myName^.name);
      end; {debug code}
      outC('/*<*/`');
      ANSIheader(myName, serial, {complete=} false, {imported=} false);
      { ANSIheader will prefix _f_SerialNumber to the nested proc Id }
      {now the aux param declaration: a macro call }
      macro:='/*@';
      for k:=1 to lenIde do begin nk:=procName[k];
        if nk>' ' then macro:=macro+nk;
      end;
      macro:=macro+'D*/';
      outC(macro);
      outC(')`');
      outC('/*)*/`'); { bracket construct is " < ... ) " }
    end; {with,if}
  end; {if lv...}
  if recursive[lv] then outC('`/*>*/`') else outC('`/*)*/`');
    {recursive means: some inner procedure called the main procedure }
end;

procedure insertAuxPars(var st:str255);
{ ACTION: Verify that expr. on TOS ends with ")" .  Insert st before the ")"
  CALLER: extendParlist
}
var le: integer;
begin
  le:=length(expStack[lastExp]);
  if expStack[lastExp,le]=')' then begin {insertion point Ok }
    insert(st,expStack[lastExp], le);
  end else error('AuxParam insertion BUG');
end;

procedure extendParList(called,from:pide);  {cases 5 and 6 }
{ ACTION: a nested procedure has been called:
          update caller network for aux parameter propagation
          output the aux-parameter-passing macro
  CALLER: finishCall
  INPUT : "called" is name of a procedure in the stack of locals.
          "from" is the name of the caller (local or global).

  must add the aux parameters for that procedure/function call
  put a macro reference to the definite parameter list.
  Macro parameter is scope.actual, to put the & etc..
  The second pass does all macro substitutions:  the macro holds info of
  origin scope for each parameter.
  If "from" is local, too,  side effect on "called":
  remember, for transitiveClosure, all non-global callers.
}
var ppa,pcalled,pfrom:pNewParam;
    pi:pide;
    over, found, valp, varp: boolean;
    arg:str40;  pt:ptpel;
    ok: boolean; {for copy loop}
    i,j,k: integer; ch,cla,nk:char;
    var st,numb:str255;
begin ppa:=pLast; pcalled:=Nil; pfrom:=Nil;
  repeat   {list scan, from tail to head, find matching identifiers}
    if ppa<>Nil then begin
      pi:= ppa^.myName;
      {Check the pointer to Id first. If that will match, MUST check the
       identifier itself since pi may have been recycled. The ppa-linked list
       contains parTable pointers to some dead  out-of-scope local Id's
      }
      if pi=called then
        if (ppa^.procName)=called^.name then pcalled:=ppa;
      if pi=from then
        if (ppa^.procName)=from^.name then pfrom:=ppa;
      found:= (pcalled<>Nil) and (pfrom<>Nil);
      if not found then begin  {if from=called, pfrom=pcalled}
        ppa:=ppa^.prev;
      end;
    end;
  until (found or (ppa=Nil));
  if pcalled=Nil then error('[Cnesting.ExtendParList]!')
  else with pcalled^ do begin
    if pfrom<>Nil then begin {insert in the local caller list, if necessary}
      i:=0; found:=false;
      repeat i:=succ(i); over:=(i>nCallers);
        if not over then found:=(caller[i]=pfrom);
      until over or found;
      if over then begin
        if i>maxCaller then error('Cnesting.MaxCaller too small!')
        else begin nCallers:=i; caller[i]:=pfrom end;
      end;
    end;
    st:='/*@'; {macro for AuxParam passing}
    for k:=1 to lenIde do begin nk:=procName[k];
      if nk>' ' then st:=st+nk;
    end;
    str(scope.actual,numb);
    st:=st+'C '+numb+'*/';
    insertAuxPars(st); {squeeze st into last expression on stack}
  end;  {with pcalled}
end;

procedure lookPrefix;
{ ACTION: within a proc/funct call, add separators/prefixes for next param.
          for string-valued funct, throw in the result pointer argument.
          set up trigger if next arg is a weak number
  CALLER: concatParam callCode
}
var  pt:ptpel; class,resultCl:char;
     noArgs: boolean; {not yet seen any arguments}
     s:str80;
begin
  s:='';
  pt:=typeStack[lastCall];  class:=pt^.cl;
  noArgs:= (class='U')or(class='D');
  if class='U' then begin  {special code for structure-valued functions }
    resultCl:=pt^.p^.cl;   {may be simple or string type}
    if resultCl='s' then begin
      pushAuxString; binaryOp('','(','', 0);
      noArgs:=false;
    end;
  end;
  { if class='A' then s:=s+')';  end of &(....) structure }
  pt:=pt^.q; {chain;}
  if pt<>Nil then begin {there is one more parameter}
    if noArgs then s:='(' else s:=s+', ';
    if pt^.cl='A' then s:=s+'&'; {addr - operator required, why not &( ? }
    if (pt^.p^.cl='r')and(pt^.m=1) then paraVariant[lastCall]:='r';
    {a weak real encountered: short, long or double}
  end else begin
    if noArgs then s:='()' {no params for funct/proc}  else s:=s+')' ;
  end;
  typeStack[lastCall]:=pt; unaryOp('',s, 0,0);
end;

procedure concatParam(fp,q:ptpel);
{ ACTION: Procedure/function call: add one more parameter to the par. list.
          Make sure that array par. NEVER, record/set par. ALWAYS, get a & .
          Eliminate &* and &Array redundancy and adjust types.
          Is called the 1st time with a procedure type on TOS
          Will walk through a pointer chain stored at top of type stack.
          Adjust slight mismatches (convert actual Char to formal String...)
          Marks true type of a weak numeric parameter.
  CALLER: Cbulk.compilerAction_pushParam
  INPUT : fp and fp^.p:  type of the expected formal parameter.
          q:  type of the last actual parameter; is at lastExp on stack .
          Previous parameter at lastExp-1 has already suffix like ,& or so.

 We cannot know the formal parameter identifiers here:
 SEMANTIC  scratches old fuction declarations
 when new ones come in, the parameter names are LOCAL !
 We need a stack since there are functions called within function calls ...
 Bug: too dumb to reduce  &(*X) to X !
 Note that e.g. &A->B means &(A->B) in C: no parentheses required.
}
var  la,lb: integer; arrayLike:boolean;
begin
    {first: handle & and type bending on expression stack}
  la:=length(expStack[lastExp-1]);
  if (q^.cl='R') {or(q^.cl='e')} then begin   {record: always as pointer}
    if (expStack[lastExp-1][la]<>'&') then begin
      expStack[lastExp-1]:=expStack[lastExp-1]+'&'; la:=la+1;
    end;
  end;
  if (expStack[lastExp-1][la]='&') then begin {a VAR param}
    arrayLike:=(q^.cl='Y')or(q^.cl='s')or(q^.cl='e');
    if arrayLike then begin { drop  address level: &X->X, X = &*X -> *X}
      delete(expStack[lastExp-1],la,1);
    end else if (expStack[lastExp][1] ='*') then begin {&* redundancy}
      delete(expStack[lastExp-1],la,1); delete(expStack[lastExp],1,1);
    end;
  end;
  if fp^.p^.cl='s' then begin {string params do not like char values!}
    if expStack[lastExp][1]='''' then begin {replace with " }
      lb:=length(expStack[lastExp]);
      expStack[lastExp][1]:='"'; expStack[lastExp][lb]:='"';
    end else if q^.cl='c' then begin {force Char expression into String}
      pushAuxString; binaryOp('_sL(',',',')',0); {_sL = converter function}
    end;
  end else if (fp^.cl='L')and(fp^.p^.cl='l') then begin
    {longint value parameter: must typecast if actual type isn't long}
    if q^.cl<>'l' then unaryOp('(long)','',1,1);
  end;
  if paraVariant[lastCall]='r' then begin {weak numeric type trigger}
    paraVariant[lastCall]:=upcase(q^.cl); {avoid 2nd trigger with upcase}
  end;
  binaryOp('','','',0);  {add last parameter now}
  lookPrefix; {look ahead to next argument if any }
end;

procedure callCode(var stat:scanStatus; fp: char); {fUnct or proceD }
{ ACTION: push a proc/funct call onto the call stack, name onto expr. stack.
  CALLER: Cbulk.compilerAction _functCall _procCall
}
var nam:symbol; s:str40; k:integer;
begin
  pushIdent(stat.pp);
  nam:=stat.pp^.name;  k:=SerialNumber(nam);
  if k>0 then begin
    { str(k,s); unaryOp('',s,0,0); }
    if k>1 then begin str(k,s); s:='_f'+s end else s:='_f';
    unaryOp(s,'',0,0); { if local proc, must prefix _f, _f2, _f3... }
  end;
  { now must push typeElement, needed for scanning the parameter chain}
  lastCall:=succ(lastCall);
  typeStack[lastCall]:=stat.pp^.typof;
  callStack[lastCall]:=stat.pp;
  levlStack[lastCall]:=scope.idLevel;
  callContext[lastCall]:=fp;
  paraVariant[lastCall]:=' '; {no variant parameters}
  lookPrefix; {check if the 1st parameter needs a prefix like & }
  { for string-valued functions, lookPrefix adds 1st aux string argument }
end;

procedure missingParam(var s:str255; weakNumber: char);
{ ACTION: handle system calls with missing optional - or variant - parameters
          if a System proc or fct call ends with "," NOT ")":  correct and
          make last letter of its name UPPERCASE to invoke a different version
          if weakNumber<>' ', adjust last 2 letters for short/long/double vers.
  CALLER: finishCall
  INPUT : s is the entire function call: Name(Par1,Par2,...)
}
var i,ls: integer; missing:boolean;
begin
  ls:=length(s);
  missing:=s[ls]<>')';
  if missing then begin
    if s[ls]='(' then s:=s+')' else s[ls]:=')';
    if s[ls-1]=',' then s[ls-1]:=' '; {kludge}
  end;
  if missing or (weakNumber<>' ') then begin
    i:=1;
    repeat i:=succ(i) until s[i]='('; i:=pred(i);
    if missing or (weakNumber='R')or(weakNumber='D') then s[i]:=upcase(s[i]);
       {Real version or missing param version: last char is  upper case}
    if weakNumber='L' then s[i-1]:=upcase(s[i-1]); {Longint version}
  end;
end;

procedure finishCall;
{ ACTION: add the extra parameters to the pending proc/funct call.
          Output a procedure call, leave funct. call on expression stack.
  CALLER: Cbulk.compilerAction_doCall
}
var k:integer;
begin {Function results MUST be kept on expression stack !}
  k:= levlStack[lastCall]; {level of the callee identifier}
  if k>=globScope then extendParlist(callStack[lastCall],procStack[lastProc]);
    {call of nested function, write the aux parameter macro call }
  k:=k+1; {internal level of callee}
  if k< scope.actual then recursive[k]:=true;   { needed for ">" bracket }
  missingParam(expStack[lastExp],paraVariant[lastCall]);
  if CallContext[lastCall]<>'U' then outCexp(1);
    {dump Procedure call only}
  lastCall:=pred(lastCall);
end;

{------------------------------------------}

procedure initNesting;
{ ACTION: clear stack pointers and Escaped Symbol table
  CALLER: Cbulk.initCbulk
}
begin
  pFirst:=Nil; pLast:=Nil; pReserv:=Nil;
  auxLast:=Nil; auxReserv:=Nil;
  lastProc:=0; lastCall:=0;
  GIRoot:=Nil;
end;

procedure termNesting;
{ ACTION: heap garbage collection.
}
begin
  killGlobId(GIroot);
  killPlist;
end;

end. {cNesting}

