{    getunits.pas        Copyright (c) 1991 by Georg Post

  Unit dependency list for the PCPC Pascal-C  translator

GetUses:
  Reads the interface part of all units on which a .PAS file depends (USES) .
  Makes an ordered list of all units (direct or indirect use: a "transitive
  closure") such that any list entry depends only on the preceding ones.
  Some "recursion cycle" in the USES reference graph is strictly forbidden
  and will probably shoot this algorithm.
Parameters:
  parses the command line
}

unit getUnits;
{$R+,S+}
interface

const maxUnit=30; {no more than 30 units..}
      maxUses=20; {each unit may use 20 others...}

type str40=string[40];
     str80=string[80];

var fileName, {'Uses' file list for the current file, all lower case}
    unitName: array[1..maxUnit] of str40; { internal $U name ...}
    sourceName: array[1..maxUses] of str40; {command line file list}
    useList: array[1..maxUnit, 0..maxUses] of integer;
      {useList[i,j]=k if unit i uses unit k with precedence j }
      {useList[i,0]= number of units used by unit i }
    nbUnits: integer;

procedure transferNames(nbSysFiles,nbUnits: integer; var nU:integer);
procedure getUses(dataDir,nf: str40; var lastNew,nbSysFiles:integer);
procedure parameters(var pc: integer; var codeDir,dataDir: str40;
  var buildAll,traceMode,AnsiKill,Ubar: boolean);

implementation

var visited: array[1..maxUnit] of boolean;

procedure transferNames(nbSysFiles,nbUnits: integer; var nU:integer);
{ ACTION: transfer  the current Uses list to command-line-name list
  CALLER: pcpc.translate if buildAll option
  OUTPUT: nU = new limit of sourceName list
}
var p:integer; path:str40;
begin {nonstandard Units must be processed only}
  nU:=1;
  for p:=nbSysFiles+1 to nbUnits-1 do begin
    path:=fileName[p]; path:=copy(path,1,length(path)-4); {strip '.PAS'}
    nU:=nU+1; sourceName[nU]:=path;
  end;
end;

procedure getToken(var f:text; var s: str40;
  var directive:str80; var c:char; var block:boolean);
{ ACTION: read next alphanumeric token in file f, skip comments.
          Intercepts the latest $U compiler directive preceding a token.
  CALLER: usesThings
  INPUT : c = the last character read by a previous call.
  OUTPUT: s. If s nonempty, c = the separator following the token.
          Separators ; and , following the token are recognized.
          block=TRUE if we get stuck at end of file (should never happen)
}
var d:char;
    ldir: integer;
    noToken, token, valid, separator, paren, getDirec: boolean;
begin
  s:=''; d:=upcase(c);
  token:=((d>='A') and (d<='Z'));
  directive:=''; {intercepts $U etc. }
  paren:=false;
  separator:=(c in [')',';',',']);
  if not (separator or token) then begin
   repeat
    if not eof(f) then read(f,d);
    { if eoln(f) then begin readln(f); d:=' ' end else read(f,d); }
    if (c='(') then begin
      if (d='*') then begin {skip all until *) appears}
        if not eof(f) then read(f,d);
        getDirec:=(d='$');
        repeat c:=d;
          if not eof(f) then read(f,d);
          if getDirec then directive:=directive+d;
        until ((c='*')and(d=')')) or eof(f);
        if not eof(f) then read(f,d);
      end else paren:=true;
    end else if c='{' then begin
      getDirec:=(d='$');
      repeat c:=d;
        if not eof(f) then read(f,d);
        if getDirec then directive:=directive+d;
      until (c='}') or eof(f);
    end;
    d:=upcase(d);
    notoken:= eof(f);
    token:=((d>='A') and (d<='Z'));
    separator:=paren or (d in [')',';',',']);
    c:=d;
   until notoken or token or separator; {token starts here ! }
  end;
  if token then begin s:=d;
    repeat
      if eof(f) then c:=' ' else read(f,c); c:=upcase(c);
      valid:= ((c>='A')and(c<='Z')) or ((c>='0')and(c<='9'));
      if valid then s:=s+c;
    until not valid;
  end else if separator then begin
    if paren then begin
      s:='('; c:=d;
    end else begin
      s:=d;
      if not eof(f) then read(f,c); {c = next char after separator}
    end;
  end;
  block:=eof(f);
  ldir:=length(directive);
  if ldir>=2 then directive:=copy(directive,1,ldir-2);
end;

procedure usesThings(var f:text; var number:integer; var ok:boolean);
{ ACTION: parses file f for keyword USES and list of names that follow.
  CALLER: getUses
  INPUT : number = current index of unitName/fileName tables
  OUTPUT: increments number, adds entries to unitName/fielName
}
var c:char; s:str40; block,does,last: boolean;
    i:integer;
    dr: str80;
begin c:=' ';
  ok:=true;
  getToken(f,s,dr,c, block);
  if s='PROGRAM' then begin  {but TP is sloppy: allow omission of PROGRAM? }
    repeat getToken(f,s,dr,c,block)
    until (s=';') or block;
    getToken(f,s,dr,c,block);
    does:=(s='USES');
  end else if s='UNIT' then begin
    repeat getToken(f,s,dr,c,block)
    until (s='INTERFACE') or (s='IMPLEMENTATION') or block;
    if s='INTERFACE' then getToken(f,s,dr,c,block);
    does:=(s='USES');
  end;
  if does then begin {else assume it's non-Pascal file}
    repeat
      number:=succ(number);
      ok:=(number<=maxUnit); {room in the table}
      getToken(f,s,dr,c,block);
      if ok then begin
        for i:=1 to length(s) do begin {convert s to lower case}
          if (s[i]>='A')and(s[i]<='Z') then
            s[i]:=chr(ord(s[i])-ord('A')+ord('a'));
        end;
        unitName[number]:=s;
        fileName[number]:=s+'.pas';
      end;
      getToken(f,s,dr,c,block); last:=(s=';');
      if (not last) and (s<>',') then writeln('Comma not found ! ');
    until block or last or (not ok);
  end;
end;

procedure tableUpdate(user,first,last: integer; var lastNew:integer;
  var ok:boolean);
{ ACTION: updates the useList tables  useList [User, *].
  CALLER: getUses
  INPUT : User is a module that uses the others from First to Last.
  OUTPUT: if User's name is already listed, erase double entry and
          update the references
          lastNew <=last is last index of really new stuff.
}
var i,j,k,nbu: integer; oldie: boolean;
    ident:str40;
begin
  visited[user]:=true;
  nbu:=0; { nbu must be kept <=maxUses}
  k:=first-1;
  i:=first; ok:=true;
  while (i<=last) and ok {for i:=first to last} do begin
    ident:=fileName[i];
    nbu:=succ(nbu);
    ok:=(nbu<=maxUses);
    j:=0;
    repeat j:=succ(j); oldie:=(fileName[j]=ident)
    until oldie or (j>=(first-1));
    if oldie then begin
      if ok then useList[user,nbu]:=j;
    end else begin
      k:=succ(k); fileName[k]:=ident;
      visited[k]:=false;
      if ok then useList[user,nbu]:=k;
    end;
    i:=succ(i);
  end;
  useList[user,0]:=nbu;
  lastNew:=k;
end;

procedure unitSwap(i,j, last: integer);
{ ACTION: swap useList table entries i and j. Update all reference tables.
  CALLER: DosCrtGraph, UnitSort
  INPUT : i,j = swap pair. last= high index for references
}
var tmp:str40; ti, k,n,x:integer;
begin
  tmp:=fileName[i]; fileName[i]:=fileName[j]; fileName[j]:=tmp;
  for k:=0 to maxUses do begin
    ti:=useList[i,k]; useList[i,k]:=useList[j,k]; useList[j,k]:=ti;
  end;
  for n:=1 to last do begin
    for k:=1 to useList[n,0] do begin
      x:=useList[n,k];
      if x=i then useList[n,k]:=j else if x=j then useList[n,k]:=i;
    end;
  end;
end;

function maxiOfUses(i:integer): integer;
{ ACTION: compute maximum index of things used by line i of useList
  CALLER: unitSort
}
var mx,n,k: integer;
begin
  mx:=0; n:=useList[i,0];
  for k:=1 to n do begin
    if useList[i,k]>mx then mx:=useList[i,k];
  end;
  maxiOfUses:=mx;
end;

procedure DosCrtGraph(Index,Last: integer; var nSys:integer);
{ ACTION : Borland Units CRT DOS GRAPH must go in front of "Uses" list!
  CALLER : unitSort
}
var i,fr,ic,id,ig: integer;
begin
  fr:=1; {frontier for system units}
  ic:=0; for i:=fr to index do if fileName[i]='crt.pas' then ic:=i;
  if ic>fr then unitSwap(ic,fr,last); if ic>0 then fr:=fr+1;
  id:=0; for i:=fr to index do if fileName[i]='dos.pas' then id:=i;
  if id>fr then unitSwap(id,fr,last); if id>0 then fr:=fr+1;
  ig:=0; for i:=1 to index do if fileName[i]='graph.pas' then ig:=i;
  if ig>fr then unitSwap(ig,fr,last);
  nsys:=0; if ic>0 then nsys:=nsys+1; if id>0 then nsys:=nsys+1;
  if ig>0 then nsys:=nsys+1; {count the standard units}
end;

procedure unitSort(last: integer; var nSys:integer);
{ ACTION: sort the unit list such that any entry depends only preceding ones
  CALLER: getUses
  INPUT : last valid index
  OUTPUT: updated global table useList, nSys = nbr of system files
}
var frontier,lastIndex,i: integer;
begin
  frontier:=0; lastIndex:=frontier;
  for i:=frontier+1 to last do begin {put those without USES in front!}
    if useList[i,0]=0 then begin {no USES list}
      lastIndex:=succ(lastIndex);
      unitSwap(i,lastIndex, last);
    end;
  end; {for i}
  DosCrtGraph(lastIndex,last,nSys);
  repeat {try to advance everything that depends only on things < lastIndex}
    frontier:=lastIndex;
    for i:=frontier+1 to last do begin
      if maxiOfUses(i)<=frontier then begin
        lastIndex:=succ(lastIndex);
        unitSwap(i,lastIndex, last);
      end;
    end;
  until frontier=lastIndex; {no more move}
  if frontier<last then begin
    writeln('Error:  Frontier=',frontier,'  last=',last);
  end;
end;

procedure unitTable(last:integer);
{ ACTION: print the sorted unit table
  CALLER: getUses
}
var nb,k: integer;
begin
  for nb:=1 to last do begin
    write(nb:2,' ',fileName[nb]:14,'  ');
    for k:=1 to useList[nb,0] do write(useList[nb,k]:3);
    writeln;
  end;
end;

procedure getUses(dataDir,nf: str40; var lastNew,nbSysFiles:integer);
{ ACTION: examine the USES part of a Turbo Pascal file
  CALLER: pcpc.translate
  INPUT : dataDir= dir for ALL unit files *.PAS
          nf = file name of main prog, must be in dataDir, too
  OUTPUT: lastNew= nb of detected unit references + 1 (the main file)
          ordered list of units in the global fileName table
          units 1..nbSysFiles are standard units (Dos Crt Graph)
}
var f:text;
    uName: str40;
    i,nb, user,first,last: integer;
    c,rep:char;
    saturate,ok:boolean;
begin
  nb:=0;
  for i:=1 to length(nf) do begin c:=nf[i];
    if (c>='A')and(c<='Z') then nf[i]:=chr(ord(c)-ord('A')+ord('a'));
  end;
  nb:=succ(nb); fileName[nb]:=nf; visited[nb]:=false;
  user:=0;
  lastNew:=nb;
  repeat
    user:=succ(user); uName:=fileName[user];
    if ((uName='dos.pas')or(uName='crt.pas')or(uName='graph.pas')) then begin
      visited[user]:=true; {these are known}
      useList[user,0]:=0; {they use nothing else}
    end else if not (visited[user]) then begin  {non-system units }
      assign(f,dataDir+fileName[user]);
      {$I-} reset(f); {$I+}
      if IOresult<>0 then begin
        writeln('GETUNITS: Cannot open ',dataDir,fileName[user]);
        halt;
      end;
      first:=lastNew+1; last:=lastNew;
      usesThings(f,last,ok); {read Uses part of f, fill table, increment Last}
      close(f);
      if ok then tableUpdate(user, first,last, lastNew, ok);
      if not ok then begin
        writeln('GETUNITS: unitName table overflow at ',fileName[user]);
        halt;
      end;
    end;
    saturate:=(user>=lastNew);
  until saturate;
  writeln('         Unit      uses:');
  unitSort(lastNew,nbSysFiles);
  unitTable(lastNew);
end;

procedure parameters(var pc: integer; var codeDir,dataDir: str40;
  var buildAll,traceMode,AnsiKill,Ubar: boolean);
{ ACTION: gets Command Line entries to the program
  CALLER: pcpc.main
  OUTPUT: source/destination file names, file count, some debug flags.
}
var argu:str40; j,k,la: integer;
    choice:char;  endOptions: boolean;
begin
  buildAll :=false; {don't translate used Units}
  traceMode:=false; {don't make debug noise}
  AnsiKill :=false;{no step by step}
  Ubar:=false; {undocumented U option}
  pc:=paramCount;   endOptions:=false;
  codeDir:='';
  dataDir:='';
  if pc>0 then begin j:=1;
    repeat
      argu:=paramStr(j); la:=length(argu);
      if (la>=2) and ((argu[1]='/')or(argu[1]='-')) then begin
        {cmd-line option: currently supported are PCTABU (A in Reord2) }
        j:=j+1; choice:=upcase(argu[2]);
        if (choice='P') and (la>2) then begin
          dataDir:=copy(argu,3,la-2);
          if argu[la]<>'\' then dataDir:=dataDir+'\';
        end else if (choice='C') and (la>2) then begin
          codeDir:=copy(argu,3,la-2);
          if argu[la]<>'\' then codeDir:=codeDir+'\';
        end else if choice='T' then begin
          TraceMode:=true;
        end else if choice='B' then begin
          BuildAll:=true;
        end else if choice='A' then begin
          AnsiKill:=true;
        end else if choice='U' then begin
          Ubar:=true;
        end;
      end else endOptions:=true;
    until endOptions or (j>pc);
    pc:=pc-j+1;
    for k:=1 to pc do begin sourceName[k]:=paramStr(j); j:=j+1 end;
  end;
end;

end. {getunits}
