'****************************************************************************
'****************************************************************************
'* PASSWORD.BAS by Duane Paulson. First released into the Public Domain on  *
'* 4/23/91. Version 2.1 released 04/21/91.                                  *
'* This version developed using Microsoft BASIC Professional Development    *
'* System version 7.1. Also compatible with QuickBASIC version 4.5.         *
'*                                                                          *
'* PURPOSE: Generates a string of random characters, 5 to 8 characters long,*
'* suitable for use as a password.                                          *
'*                                                                          *
'* INPUT: (1) Accepts single keystroke from user via radio button interface.*
'*        (2) System timer provides random function seed.                   *
'*        (3) BASIC random function provides random variables.              *
'*                                                                          *
'* OUTPUT: Random password string displayed to screen.                      *
'*                                                                          *
'* XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX *
'* COMPILE AND LINK INSTRUCTIONS for BASIC PDS 7.1:                         *
'*                                                                          *
'* bc /O /Ot /Lr /FPa /T /C:512 password;                                   *
'* link /NOE password.obj+nofltin.obj+nolpt.obj+smallerr.obj+noedit.obj+nocom.obj,,,bcl71anr.lib;
'*                                                                          *
'* COMPILING UNDER QUICKBASIC 4.5                                           *
'*                                                                          *
'* Select the "Make EXE" option from the "Run" menu. The executable will be *
'*  significantly larger than the one supplied with the distribution        *
'*  package.                                                                *
'****************************************************************************
'****************************************************************************

DEFINT A-Z  ' Declare integer as default variable type.

'****************************************************************************
'* SUBprocedure and FUNCTION procedure declarations.                        *
'****************************************************************************
DECLARE SUB A.ShowTime (Row, Col) ' Display the current time
DECLARE SUB Title ()             ' Opening title
DECLARE SUB SetUpDisplay ()      ' Initializes screen display
DECLARE SUB PrintPassword ()     ' Prints the password to screen
DECLARE SUB ClearBox ()          ' Clears the password display
DECLARE SUB SelectPassword ()    ' Generates the password
DECLARE FUNCTION AcceptKey ()    ' Processes user input
DECLARE SUB ShowInfo ()          ' Displays information about program

'****************************************************************************
'* Global constant and variable declarations.                               *
'****************************************************************************
CONST False% = 0, True% = NOT False
DIM SHARED password AS STRING       ' The random string
DIM SHARED LenFrame AS INTEGER      ' Length of password+2
DIM SHARED StartColumn AS INTEGER   ' The column where the password display
                                    '  will start.

'****************************************************************************
'* Main Procedure processing begins here.                                   *
'*                                                                          *
'* PURPOSE: To control program flow                                         *
'*                                                                          *
'* INPUT: Current button from FUNCTION AcceptKey                            *
'*                                                                          *
'* OUTPUT: None                                                             *
'*                                                                          *
'* VARIABLES USED:                                                          *
'*  AcceptKey        Indicates current button                               *
'*  Finished         Exit flag                                              *
'****************************************************************************
RANDOMIZE TIMER                  ' Seed the random function

'****************************************************************************
'* Initialize display and select first password                             *
'****************************************************************************
Title
SetUpDisplay
SelectPassword

'****************************************************************************
'* Main Loop                                                                *
'****************************************************************************
DO WHILE NOT Finished
   SELECT CASE AcceptKey
      CASE 1
         ClearBox
         SelectPassword
      CASE 2
         Finished = True
      CASE 3
         ShowInfo
         PrintPassword
   END SELECT
LOOP

'****************************************************************************
'* Exit routine                                                             *
'****************************************************************************
LOCATE 25, 1
END

'****************************************************************************
'* End of Main Procedure                                                    *
'****************************************************************************

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB A.ShowTime                                                           *
'*  PURPOSE: Displays the current time in am/pm format on the top line      *
'*                                                                          *
'*  INPUT: (1) From calling procedure: current cursor position              *
'*         (2) From user: None                                              *
'*         (3) From BIOS clock (if found) or System Timer: time of day      *
'*                                                                          *
'*  OUTPUT: (1) To calling procedure: None                                  *
'*          (2) To user: Time of day in am/pm format in upper right corner  *
'*                       of screen.                                         *
'*                                                                          *
'*  VARIABLES USED:                                                         *
'*  tim$     Time of day input buffer.                                      *
'*  Hr$,Hr   Hour of day as value, for processing of time display.          *
'*  m$       A.M./P.M. string for time display.                             *
'****************************************************************************
'----------------------------------------------------------------------------
SUB A.ShowTime (Row, Col)

'****************************************************************************
'* Create variable containing current hour (for am/pm conversion)           *
'****************************************************************************
tim$ = TIME$
Hr$ = LEFT$(tim$, 2)
Hr = VAL(Hr$)

'****************************************************************************
'* Convert to am/pm format                                                  *
'****************************************************************************
SELECT CASE Hr
   CASE IS > 12                     ' P.M.
      Hr = Hr - 12
      Hr$ = LTRIM$(STR$(Hr))
      m$ = "p.m."
   CASE 0                           ' Midnight
      m$ = "a.m."
      Hr$ = "12"
   CASE ELSE                        ' A.M.
      Hr$ = LTRIM$(STR$(Hr))
      m$ = "a.m."
END SELECT

tim$ = Hr$ + RIGHT$(tim$, 6) + " " + m$      ' Finish the conversion
IF LEN(tim$) = 12 THEN tim$ = " " + tim$     '  and adjust for length

'****************************************************************************
'* Display the time                                                         *
'****************************************************************************
LOCATE 1, 68
PRINT tim$;
LOCATE Row, Col

END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* FUNCTION AcceptKey                                                       *
'*                                                                          *
'* PURPOSE: Processes user input                                            *
'*                                                                          *
'* INPUT: (1) From calling procedure: None.                                 *
'*        (2) From user: Single keystroke.                                  *
'*                                                                          *
'* OUTPUT: (1) To calling procedure: AcceptKey                              *
'*         (2) To user: Prompt strings, highlighting of radio buttons.      *
'*                                                                          *
'* LOCAL SUBROUTINES:                                                       *
'*  AgainOn,AgainOff,ExitOn,ExitOff,InfoOn,InfoOff                          *
'*                Turns on or off highlighting of radio buttons.            *
'*                                                                          *
'* VARIABLES USED:                                                          *
'*  a$            Input buffer.                                             *
'*  AcceptKey     Passes current back to main procedure.                    *
'*  Button        Keeps track of current radio button. Must be STATIC.      *
'*  Finished      Controlls processing loop.                                *
'*  cp            Current cursor position. For finding current radio button *
'*                      after displaying time.                              *
'*  Esc$,Enter$,TabKey$,ShiftTab$,Spacebar$,LeftArrow$,RightArrow$          *
'*                ASCII substitution strings.                               *
'****************************************************************************
'----------------------------------------------------------------------------
FUNCTION AcceptKey STATIC

Esc$ = CHR$(27): Enter$ = CHR$(13): TabKey$ = CHR$(9)
ShiftTab$ = CHR$(0) + CHR$(15): Spacebar$ = CHR$(32):
LeftArrow$ = CHR$(0) + CHR$(75): RightArrow$ = CHR$(0) + CHR$(77)

IF Button = 0 THEN Button = 1 ' Initialize Button the first time this
                              '   Function is called.

Finished = False  ' Reset exit flag


'****************************************************************************
'* Input processing loop.                                                   *
'****************************************************************************
DO WHILE Finished = False
   SELECT CASE Button                  ' Determine current button and
      CASE 1                           '  display the cursor in appropriate
         LOCATE 16, 23, 1              '  location.
      CASE 2
         LOCATE 16, 38, 1
      CASE 3
         LOCATE 16, 53, 1
   END SELECT

   A$ = ""                 ' Clear the input buffer.
   COLOR 0, 7              ' Black on light gray.
   cp = POS(0)             ' Store cursor position.

'****************************************************************************
'* Accept keystroke from user.                                              *
'****************************************************************************
   DO WHILE A$ = ""
      A$ = INKEY$
         ' If tenths of second=0 then display the time on the top line.
         ' Doing it this way cuts down on cursor flicker.
      IF INT(TIMER * 10) MOD 10 = 0 THEN A.ShowTime 16, cp
   LOOP
'****************************************************************************
'* End of keystoke capture loop.                                            *
'****************************************************************************

   SELECT CASE A$                      ' Now that we have the keystroke,
      CASE Esc$                        '  process it if it's an acceptable
         AcceptKey = 2                 '  one. Shift radio buttons, show
         Finished = True               '  info, or set exit flag, as
                                       '  appropriate.

      CASE TabKey$, RightArrow$    ' Shift buttons right (or wrap around)
         AcceptKey = Button
         SELECT CASE Button
            CASE 1
               GOSUB AgainOff
               GOSUB ExitOn
               Button = 2
            CASE 2
               GOSUB ExitOff
               GOSUB InfoOn
               Button = 3
            CASE 3
               GOSUB InfoOff
               GOSUB AgainOn
               Button = 1
         END SELECT

      CASE ShiftTab$, LeftArrow$    ' Shift buttons left (or wrap around)
         AcceptKey = Button
         SELECT CASE Button
            CASE 1
               GOSUB AgainOff
               GOSUB InfoOn
               Button = 3
            CASE 2
               GOSUB ExitOff
               GOSUB AgainOn
               Button = 1
            CASE 3
               GOSUB InfoOff
               GOSUB ExitOn
               Button = 2
         END SELECT

      CASE Enter$, Spacebar$       ' Action key.
         AcceptKey = Button
         Finished = True
   END SELECT

LOOP
'****************************************************************************
'* End of input processing loop.                                            *
'****************************************************************************

LOCATE , , 0         ' Turn off cursor.
EXIT FUNCTION        ' End of main part of function.

'****************************************************************************
'* Local SUBROUTINES for Function AcceptKey                                 *
'****************************************************************************
AgainOff:
   COLOR 0, 7
   LOCATE 16, 22
   PRINT "<";
   LOCATE 16, 30
   PRINT ">";
RETURN

AgainOn:
   COLOR 15, 7
   LOCATE 16, 22
   PRINT "<";
   LOCATE 16, 30
   PRINT ">";
   COLOR 0, 3
   LOCATE 24, 1
   PRINT SPACE$(80);
   LOCATE 24, 3
   PRINT "Generate another password."
RETURN

ExitOff:
   COLOR 0, 7
   LOCATE 16, 37
   PRINT "<";
   LOCATE 16, 44
   PRINT ">";
RETURN

ExitOn:
   COLOR 15, 7
   LOCATE 16, 37
   PRINT "<";
   LOCATE 16, 44
   PRINT ">";
   COLOR 0, 3
   LOCATE 24, 1
   PRINT SPACE$(80);
   LOCATE 24, 3
   PRINT "Exit to DOS."
RETURN

InfoOff:
   COLOR 0, 7
   LOCATE 16, 52
   PRINT "<";
   LOCATE 16, 59
   PRINT ">";
RETURN

InfoOn:
   COLOR 15, 7
   LOCATE 16, 52
   PRINT "<";
   LOCATE 16, 59
   PRINT ">";
   COLOR 0, 3
   LOCATE 24, 1
   PRINT SPACE$(80);
   LOCATE 24, 3
   PRINT "Display information about program."
RETURN

END FUNCTION

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB ClearBox                                                             *
'*                                                                          *
'*  PURPOSE: Clears the display box that the password appears in.           *
'*                                                                          *
'*  INPUT: None                                                             *
'*                                                                          *
'*  OUTPUT: (1) To calling procedure: None                                  *
'*          (2) To user: Spaces blank the old password display box.         *
'*                                                                          *
'*  VARIABLES USED:                                                         *
'*    LenFrame       Length of password+2. This is the length of the top    *
'*                   and bottom lines for the password display box. It does *
'*                   not include the length of the corner characters,       *
'*                   representing instead the number of straight line       *
'*                   characters that will have to be drawn by               *
'*                   SUB PrintPassword. Taking into account the corner      *
'*                   characters and the two shadow characters, a total of   *
'*                   4 spaces has to be added to blank the entire display.  *
'*                   LenFrame is a global variable.                         *
'****************************************************************************
'----------------------------------------------------------------------------
SUB ClearBox

IF LenFrame = 0 THEN EXIT SUB

COLOR 0, 7                    ' Black on light gray
FOR A = 10 TO 14
   LOCATE A, StartColumn
   PRINT SPACE$(LenFrame + 4);
NEXT A

END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB PrintPassword                                                        *
'*                                                                          *
'*  PURPOSE: Print the password string to the screen.                       *
'*                                                                          *
'*  INPUT: Global variables Password, LenPassword, StartColumn.             *
'*                                                                          *
'*  OUTPUT: Password string displayed to screen.                            *
'*                                                                          *
'*  VARIABLES USED: [No local variables]                                    *
'*                                                                          *
'****************************************************************************
'----------------------------------------------------------------------------
SUB PrintPassword
'****************************************************************************
'* Calculate the length of the 'frame' and center the display.              *
'****************************************************************************

COLOR 7, 4              ' Light gray on red.
LOCATE 10, StartColumn - length
PRINT ""; STRING$(LenFrame, ""); ""   ' Top of frame"


'****************************************************************************
'* Print the password.                                                      *
'****************************************************************************
LOCATE 11, StartColumn - length
COLOR 7, 4: PRINT " "; : COLOR 15, 4: PRINT password; : COLOR 7, 4: PRINT " ";
COLOR 7, 0: PRINT SPACE$(2)   ' Print 2 "shadow" characters.

LOCATE 12, StartColumn - length
COLOR 7, 4: PRINT ""; STRING$(LenFrame, ""); "";   ' Bottom of frame
COLOR 7, 0: PRINT SPACE$(2)         ' 2 "shadow" characters

LOCATE 13, StartColumn + 2 - length          ' bottom "shadow" characters
PRINT SPACE$(LenFrame + 2)

END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB SelectPassword                                                       *
'*                                                                          *
'*  PURPOSE: Generate a 5 to 8 character long random string.                *
'*                                                                          *
'*  INPUT: None                                                             *
'*                                                                          *
'*  OUTPUT: Global variables Password, LenFrame, Startcol.                  *
'*                                                                          *
'*  VARIABLES USED:                                                         *
'*    length         Length of password                                     *
'*    char, char$    Buffer to hold each individual character as it is      *
'*                      generated.                                          *
'****************************************************************************
'----------------------------------------------------------------------------
SUB SelectPassword

'****************************************************************************
'* Initialize string variable                                               *
'****************************************************************************
password = ""

'****************************************************************************
'* Select random password length between 5 and 8 characters using the       *
'* following formula:                                                       *
'* INT ((upperbound - lowerbound + 1)*RND + lowerbound)                     *
'****************************************************************************
length = INT((8 - 5 + 1) * RND + 5)

'****************************************************************************
'* Fill in each character using 0-9 and a-z (total of 36 characters)        *
'****************************************************************************
  FOR A = 1 TO length
    char = INT((35 - 0 + 1) * RND + 0)

'****************************************************************************
'* Convert to ASCII character                                               *
'*    Values 0-9 become "0"-"9"                                             *
'*    Values 10-35 become "A"-"Z"                                           *
'****************************************************************************
    IF char < 10 THEN
        char$ = CHR$(char + ASC("0"))
    ELSE
        char$ = CHR$(char - 10 + ASC("a"))
    END IF

'****************************************************************************
'* Add to end of the string and pad with a space to make reading easier.    *
'****************************************************************************
    password = password + char$ + " "

  NEXT A

'****************************************************************************
'* Remove the final space                                                   *
'****************************************************************************

password = LEFT$(password, LEN(password) - 1)

LenFrame = LEN(password) + 2
StartColumn = 40 - ((LenFrame \ 2) + (LenFrame MOD 2))

PrintPassword

END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB SetUpDisplay                                                         *
'*                                                                          *
'*  PURPOSE: Initialize or restore the screen display.                      *
'*                                                                          *
'*  INPUT: None                                                             *
'*                                                                          *
'*  OUTPUT: Background display to screen.                                   *
'*                                                                          *
'*  VARIABLES USED: None.                                                   *
'****************************************************************************
'----------------------------------------------------------------------------
SUB SetUpDisplay

'****************************************************************************
'* Display active keys for main process                                     *
'****************************************************************************
COLOR 0, 7                    ' Black on light gray
LOCATE 25, 1
PRINT SPACE$(80);             ' Clear and print active keys line.
LOCATE 25, 3
PRINT "Move with <Tab>, <Shift+Tab>, <-, ->.   Select with <Enter>, <Spacebar>.";

'****************************************************************************
'* Draw main display "box" in center of screen.                             *
'****************************************************************************
LOCATE 7, 20                        ' Draw main display box
PRINT ""; STRING$(40, ""); ""    ' Top of box

FOR A = 8 TO 14                  ' Body of box
   LOCATE A, 20
   PRINT ""; SPACE$(40); "";
   COLOR 1, 0                    ' Blue on black
   PRINT SPACE$(2)               '  Draw "shadow"
   COLOR 0, 7                    ' Black on light gray
NEXT A

COLOR 0, 7
LOCATE 15, 20
PRINT ""; STRING$(40, ""); "";   ' Line for Button "window"
COLOR 1, 0                             ' Blue on black
PRINT SPACE$(2)                        '  Draw "shadow"

COLOR 0, 7                          ' Black on light gray
LOCATE 17, 20                       '  Bottom of main display box.
PRINT ""; STRING$(40, ""); "";
COLOR 1, 0                          ' Blue on black
PRINT SPACE$(2)                     '  shadow
LOCATE 18, 22
PRINT SPACE$(42)

'****************************************************************************
'* Display permanent text for the box                                       *
'****************************************************************************
COLOR 0, 7                          ' Black on light gray
LOCATE 8, 28
PRINT "Here Is Your New Password"

'****************************************************************************
'* Display the three "radio buttons"                                        *
'****************************************************************************
LOCATE 16, 20
PRINT " ";
COLOR 15, 7                         ' White on light gray
PRINT "<";                          ' Highlighted angle bracket
COLOR 0, 7                          ' Black on light gray
PRINT " Again ";                    ' Text of default button
COLOR 15, 7                         ' White on light gray
PRINT ">";                          ' Highlighted angle bracket

COLOR 0, 7                                      ' Black on light gray
PRINT SPACE$(6); "< Exit >       < Info > ";   ' Non-selected buttons don't
                                                '  have highlighted brackets
COLOR 1, 0                 ' Blue on black
PRINT SPACE$(2)            '  "shadow" for button line.


END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB ShowInfo                                                             *
'*                                                                          *
'*  PURPOSE: Display information about program to screen.                   *
'*                                                                          *
'*  INPUT: None.                                                            *
'*                                                                          *
'*  OUTPUT: Informational display to screen.                                *
'*                                                                          *
'*  VARIABLES USED: None.                                                   *
'****************************************************************************
'----------------------------------------------------------------------------
SUB ShowInfo

'****************************************************************************
'* Display the information                                                  *
'****************************************************************************
COLOR 15, 5       ' White on magenta
LOCATE 5, 1

PRINT ""; STRING$(78, ""); ""
PRINT "PASSWORD is a public domain program. It generates a random alphanumeric string"
PRINT "between 5 and 8 characters long suitable for use as a password. It only uses  "
PRINT "the characters 0-9 and a-z, in order to maintain compatibility with a wide    "
PRINT "range of systems. If a system requires a delimiting character, feel free to   "
PRINT "drop one in somewhere. Working with a 5 to 8 character string using 0-9 and   "
PRINT "a-z (36 characters) gives an extremely large total of possible combinations:  "
PRINT "(36^5)+(36^6)+(36^7)+(36^8), or 3,901,711,320,064. Over 3 billion passwords.  "
PRINT "That's enough to keep anyone guessing for a while.                            "
PRINT "                                                                              "
PRINT "Experts say that you should choose a password that has no relationship to     "
PRINT "anything in your life, so what better than a random generator to pick your    "
PRINT "passwords for you? Just remember, the more meaningless the password is        "
PRINT "the better for you. Remember: Be safe, change your password often.            "
PRINT ""; STRING$(78, ""); "";
PRINT "                                   "; : COLOR 14, 5: PRINT "<";
COLOR 15, 5: PRINT " OK "; : COLOR 14, 5: PRINT ">"; : COLOR 15, 5
PRINT "                                     "
PRINT ""; STRING$(78, ""); "";

COLOR 1, 0           ' Blue on black
LOCATE 22, 3
PRINT SPACE$(78);    ' "shadow"

'****************************************************************************
'* User prompt line                                                         *
'****************************************************************************
LOCATE 20, 38, 1

COLOR 0, 3        ' Black on cyan
LOCATE 24, 1
PRINT SPACE$(80);
LOCATE 24, 3
PRINT "Press a key to continue."

COLOR 0, 7     ' Black on light gray
LOCATE 25, 1
PRINT SPACE$(80);

LOCATE 20, 38, 1

'****************************************************************************
'* Wait for a keystroke                                                     *
'****************************************************************************
DO
   COLOR 0, 7
   IF INT(TIMER * 10) MOD 10 = 0 THEN A.ShowTime 20, 38
LOOP WHILE INKEY$ = ""

'****************************************************************************
'* Restore main screen and return                                           *
'****************************************************************************
COLOR 0, 1           ' Black on blue
VIEW PRINT 2 TO 23   ' Restore background
CLS

VIEW PRINT

LOCATE 25, 1
PRINT SPACE$(80);

SetUpDisplay

COLOR 0, 7              ' Fix the bracket highlights for the buttons to
LOCATE 16, 22           '  show "info" button as active.
PRINT "<";
LOCATE 16, 30
PRINT ">";
COLOR 15, 7
LOCATE 16, 52
PRINT "<";
LOCATE 16, 59
PRINT ">";
COLOR 0, 3
LOCATE 24, 1
PRINT SPACE$(80);
LOCATE 24, 3
PRINT "Display information about program.";  ' Fix user prompt

END SUB

'----------------------------------------------------------------------------
'****************************************************************************
'* SUB Title                                                                *
'*                                                                          *
'*  PURPOSE: Display opening screen.                                        *
'*                                                                          *
'*  INPUT: None.                                                            *
'*                                                                          *
'*  OUTPUT: Opening screen.                                                 *
'*                                                                          *
'*  VARIABLES USED:  tim&     Long integer to hold system timer value.      *
'****************************************************************************
'----------------------------------------------------------------------------
SUB Title

'****************************************************************************
'* Set background color.                                                    *
'****************************************************************************
COLOR 0, 1                       ' Black on blue
CLS                              '  clear to blue background.

'****************************************************************************
'* Display title bar on screen line 1                                       *
'****************************************************************************
COLOR 0, 7                       ' Black on light gray
LOCATE 1, 1                      '  Title bar across top of screen.
PRINT SPACE$(80);
LOCATE 25, 1
PRINT SPACE$(80);
LOCATE 1, 1
PRINT "PASSWORD version 2.1;"

'****************************************************************************
'* Display "active key" list on line 25                                     *
'****************************************************************************
LOCATE 25, 1                     '  Acitve key list on line 25
PRINT SPACE$(80);

'****************************************************************************
'* Display user prompt on line 25                                           *
'****************************************************************************
COLOR 0, 3                       ' Black on cyan
LOCATE 24, 1                     '  Prompt bar on line 24
PRINT SPACE$(80);
LOCATE 24, 3
PRINT "Press a key to continue.";

'****************************************************************************
'* Opening screen                                                           *
'****************************************************************************
LOCATE 8, 17                        ' Opening screen display.
PRINT ""; STRING$(47, ""); ""
LOCATE 9, 17
PRINT "     PASSWORD version 2.1, by Duane Paulson    ";
COLOR 1, 0
PRINT SPACE$(2)
COLOR 0, 3
LOCATE 10, 17
PRINT " Portions (C) 1982-1990 Microsoft Corporation. ";
COLOR 1, 0
PRINT SPACE$(2)
COLOR 0, 3
LOCATE 11, 17
PRINT "               All rights reserved.            ";
COLOR 1, 0
PRINT SPACE$(2)
COLOR 0, 3
LOCATE 12, 17
PRINT ""; STRING$(47, ""); "";
COLOR 1, 0
PRINT SPACE$(2)
LOCATE 13, 19
PRINT SPACE$(49)

'****************************************************************************
'* Wait for time delay or key hit                                           *
'****************************************************************************
tim& = TIMER + 5                             ' Wait for 5 seconds or keypress
DO: LOOP WHILE TIMER < tim& AND INKEY$ = ""

'****************************************************************************
'* Clear opening display                                                    *
'****************************************************************************
VIEW PRINT 2 TO 23         ' Clear middle portion of screen to blue background
COLOR 0, 1                 ' Black on blue
CLS
VIEW PRINT

'****************************************************************************
'* Display first user prompt of main program                                *
'****************************************************************************
COLOR 0, 3                             ' Black on cyan
LOCATE 24, 3                           '  Display opening prompt string.
PRINT "Generate another password.";

END SUB

