/*   tvscreen.c    TVFS screen implementation.
 *
 *   (c) Copyright International Business Machines Corporation 1994.
 *   All rights reserved.
 *
 *   11/04/91   T. Francis   Creation.
 *   11/11/91   M. Leitch    TVREMOTE processing added.
 *   07/07/92   T. Francis   Delayed deletion, conversion to 32 bit.
 *   07/09/92   M. Leitch    Cleanup.
 *   07/24/92   M. Leitch    Fixed delay flag on menu pulldown.
 *   09/08/92   M. Leitch    Completed 32 bit conversion.
 */

#define INCL_DOSERRORS
#define INCL_DOSPROCESS
#define INCL_GPILCIDS
#define INCL_WINTIMER
#define INCL_WINMENUS
#define INCL_WINERRORS
#define INCL_WINFRAMEMGR
#define INCL_WINSHELLDATA
#define INCL_WINSWITCHLIST
#define INCL_WINSYS

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "tvremote.h"
#include "tvscreen.h"


#define       DELAY_TIMER_ID                1
#define       DELAY_TIMER_TICK              500
#define       DELETE_AFTER_TICKS            6
#define       SHARABLE_BY_DOSGETSEG         2
#define       TVSCREEN_CLASS_NAME          "TvScreen Client"
#define       TVSCREEN_DELAY_TIME          "Delay"
#define       TVSCREEN_INVALIDATE           TVREMOTE_MSG_USER
#define       TVSCREEN_NAME                "TvScreen"
#define       TVSCREEN_RECYCLE_OPEN_FILES   1
#define       TVSCREEN_TITLE               "TvScreen: Open TVFS Files"


#ifdef __32BIT__                           
   #define    TVSCREEN_SIZE_NAME           "Size32"
   #define    QueryParent(hwnd)             WinQueryWindow( hwnd, QW_PARENT )
   #define    WNDMSG                        ULONG
#else                                      
   #define    TVSCREEN_SIZE_NAME           "Size"
   #define    QueryParent(hwnd)             WinQueryWindow( hwnd, QW_PARENT, FALSE )
   #define    WNDMSG                        USHORT
#endif                                     

#define QueryParentMenu( hwnd )             WinWindowFromID( QueryParent( hwnd ), FID_MENU )


/************************************************************************/
/* Function prototypes                                                  */
/************************************************************************/

MRESULT EXPENTRY ShowOpenWindowProc(HWND hwnd, WNDMSG msg, MPARAM mp1, MPARAM mp2 );
static void Paint( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL fWantDelay  );
static void TimerTick( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL fWantDelay );
static void TVRemoteClose( HWND hwnd, OPENFILELIST **ppOpenFilesHead, MPARAM mp1, MPARAM mp2 );
static void TVRemoteOpen( HWND hwnd, OPENFILELIST **ppOpenFilesHead, MPARAM mp1, MPARAM mp2 );
static void WMCreate( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL *pfWantDelay );


                        


/************************************************************************/
/* Globals                                                              */
/************************************************************************/

CHAR          szClientClass[] = TVSCREEN_CLASS_NAME;
HAB           hab;


/**********************  Start of main procedure  ***********************/

int cdecl main( void )
{
   HMQ        hmq;                       /* Message queue handle         */
   HSWITCH    hSwitch;
   HWND       hwndClient;                /* Client area window handle    */
   HWND       hwndFrame;                 /* Frame window handle          */
   QMSG       qmsg;                      /* Message from message queue   */
   SWCNTRL    PgmEntry;
   SWP        swp;
   ULONG      flCreate;                  /* Window creation control flags*/
   ULONG      ulDataLength;

   hab = WinInitialize( (USHORT) NULL ); /* Initialize PM                */
   hmq = WinCreateMsgQueue( hab, 0 );    /* Create a message queue       */

   WinRegisterClass(                     /* Register window class        */
      hab,                               /* Anchor block handle          */
      szClientClass,                     /* Window class name            */
      (PFNWP) ShowOpenWindowProc,        /* Address of window procedure  */
      CS_SIZEREDRAW,                     /* Class style                  */
      0                                  /* No extra window words        */
      );

   flCreate = ((FCF_STANDARD & (~FCF_TASKLIST) & (~FCF_SHELLPOSITION)) | FCF_NOBYTEALIGN);

   hwndFrame = WinCreateStdWindow(
                  HWND_DESKTOP,          /* Desktop window is parent     */
                  FS_STANDARD,           /* No frame styles              */
                  &flCreate,             /* Frame control flag           */
                  szClientClass,         /* Client window class name     */
                  "",                    /* No window text               */
                  WS_VISIBLE,            /* No special class style       */
                  (HMODULE) NULL,        /* Resource is in .EXE file     */
                  ID_WINDOW,             /* Frame window identifier      */
                  &hwndClient            /* Client window handle         */
                  );


   WinSetWindowText(hwndFrame, TVSCREEN_TITLE);

   ulDataLength = sizeof(swp);           /* Initialize the screen size (profiled). */
   if (!PrfQueryProfileData(HINI_USERPROFILE,
                            TVSCREEN_NAME,
                            TVSCREEN_SIZE_NAME,
                            &swp,
                            &ulDataLength))
   {
      swp.x  = 100;      swp.y  = 100;
      swp.cx = 600;      swp.cy = 250;
   }

   WinSetWindowPos(hwndFrame,
                   HWND_TOP,
                   swp.x,
                   swp.y,
                   swp.cx,
                   swp.cy,
                   SWP_SIZE | SWP_MOVE | SWP_SHOW | SWP_ACTIVATE | SWP_ZORDER);


   /* Give us a nice name in the task list          */
   {
#ifdef __32BIT__
      PTIB     ptib;  /* Address of pointer to TIB */
      PPIB     ppib;  /* Address of pointer to PIB */

      DosGetInfoBlocks(&ptib, &ppib);
      hSwitch = WinQuerySwitchHandle( hwndFrame, ppib->pib_ulpid );
#else
      PIDINFO    ProcessID;

      DosGetPID(&ProcessID);
      hSwitch = WinQuerySwitchHandle(hwndFrame, ProcessID.pid);
#endif
      WinQuerySwitchEntry(hSwitch, &PgmEntry);
      strcpy(PgmEntry.szSwtitle, TVSCREEN_NAME);
      WinChangeSwitchEntry(hSwitch, &PgmEntry);
   }

   TVREMOTE_AddDefault(hmq,              /* Register the program with the TVFS. */
                       TVSCREEN_NAME); 

   /************************************************************************/
   /* Get and dispatch messages from the application message queue         */
   /* until WinGetMsg returns FALSE, indicating a WM_QUIT message.         */
   /************************************************************************/
   while( WinGetMsg( hab, &qmsg, 0, 0, 0 ) )
      WinDispatchMsg( hab, &qmsg );

   TVREMOTE_Delete(hmq);                 /* Unregister the program with the TVFS. */

   WinDestroyWindow( hwndFrame );        /* Tidy up...                   */
   WinDestroyMsgQueue( hmq );            /* and                          */
   WinTerminate( hab );                  /* terminate the application    */

   return(0);
}

/***********************  End of main procedure  ************************/

/*********************  Start of window procedure  **********************/

MRESULT EXPENTRY ShowOpenWindowProc(HWND hwnd, WNDMSG msg, MPARAM mp1, MPARAM mp2 )
{
   static OPENFILELIST *pOpenFilesHead;   /* The set of open files. */
   static BOOL          fWantDelay=TRUE;

   switch( msg )
   {
      case WM_CREATE:
         WMCreate( hwnd, &pOpenFilesHead, &fWantDelay );
         break;
 

      case TVREMOTE_MSG_OPEN:
         TVRemoteOpen( hwnd, &pOpenFilesHead, mp1, mp2 );
         break;


      case TVREMOTE_MSG_CLOSE:
         TVRemoteClose( hwnd, &pOpenFilesHead, mp1, mp2);
         break;


      case WM_TIMER:
         {
            if( LONGFROMMP( mp1 ) == DELAY_TIMER_ID )
               TimerTick( hwnd, &pOpenFilesHead, fWantDelay );

            /* Must return via here, as some timer ticks require further processing */
            return WinDefWindowProc( hwnd, msg, mp1, mp2 );
         }


      case TVSCREEN_INVALIDATE:
         WinInvalidateRect( hwnd, NULL, FALSE );
         break;


      case WM_PAINT:
         Paint( hwnd, &pOpenFilesHead, fWantDelay );
         break;


      case WM_SAVEAPPLICATION:
      {
         SWP     swp;
      
         /*
         ** Do we want to delay deletes?
         */
         WinQueryWindowPos( QueryParent( hwnd ),
                            &swp );

         PrfWriteProfileData( HINI_USERPROFILE,
                              TVSCREEN_NAME,
                              TVSCREEN_SIZE_NAME,
                              &swp,
                              (ULONG)sizeof(swp));

         PrfWriteProfileData( HINI_USERPROFILE,
                              TVSCREEN_NAME,
                              TVSCREEN_DELAY_TIME,
                              &fWantDelay,
                              (ULONG)sizeof(fWantDelay));
      }
      break;

      case WM_COMMAND:
      {
         switch (SHORT1FROMMP(mp1))
         {
            case ID_EXITPROG:
            {
               WinPostMsg( hwnd, WM_QUIT, 0L, 0L );  /* Cause termination */
            }
            break;

            case ID_WANTDELAY:
            {
               fWantDelay = !fWantDelay;

               WinSendMsg( QueryParentMenu( hwnd ),
                           MM_SETITEMATTR,
                           MPFROM2SHORT( ID_WANTDELAY, TRUE ),
                           MPFROM2SHORT( MIA_CHECKED, fWantDelay ? MIA_CHECKED : 0 ) );

               /*
               ** Redraw the screen
               */
               WinPostMsg( hwnd, TVSCREEN_INVALIDATE, (MPARAM)NULL, (MPARAM)NULL );
            }
            break;
         } /* endswitch */
      }
      break;


      default:                     
         return( WinDefWindowProc( hwnd, msg, mp1, mp2 ) );
   }

   return FALSE;
}
/**********************  End of window procedure  ***********************/


static void TimerTick( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL fWantDelay )
{
   OPENFILELIST       *pCurrent;
   OPENFILELIST       *pPrev;
   BOOL                fChangesMade = FALSE;

   /*
   ** Walk the linked list, decrementing the counter of all closed files.
   ** If any file's count has reached 0, delete it.
   */

   pPrev = NULL;
   pCurrent = *ppOpenFilesHead;
   while (pCurrent)
   {
      if( !pCurrent->FileOpen )
      {
         if( !pCurrent->TicksLeft )
         {
            OPENFILELIST *pNext;

            /*
            ** Time's up -- delete the entry!
            */
            pNext = pCurrent->Next;

            if( pPrev )
               pPrev->Next = pNext;
            else
               *ppOpenFilesHead = pNext;

            /* Delete the entry. */
            free(pCurrent->pszTvfsName);
            free(pCurrent->pszFileName);
            free(pCurrent);

            /* Update the display */
            fChangesMade = TRUE;

            /* process the next entry. */
            pCurrent = pNext;
            continue;
         }
         else
            pCurrent->TicksLeft--;

      }

      /* process the next entry. */
      pPrev    = pCurrent;
      pCurrent = pCurrent->Next;
   }

   /*
   ** Do not update the screen if delays are not even being shown
   ** Try to avoid unecessary flickers
   */
   if( fChangesMade & fWantDelay )
      WinPostMsg( hwnd, TVSCREEN_INVALIDATE, (MPARAM)NULL, (MPARAM)NULL );
}



static void WMCreate( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL *pfWantDelay )
{
   ULONG DataLength = sizeof( BOOL );

   if (!PrfQueryProfileData( HINI_USERPROFILE,
                             TVSCREEN_NAME,
                             TVSCREEN_DELAY_TIME,
                             pfWantDelay,
                            &DataLength
                           )
      ) *pfWantDelay = TRUE;

   if( *pfWantDelay )
      WinSendMsg( QueryParentMenu( hwnd ),
                  MM_SETITEMATTR,
                  MPFROM2SHORT( ID_WANTDELAY, TRUE ),
                  MPFROM2SHORT( MIA_CHECKED, MIA_CHECKED ) );

   /*
   ** Start a timer going for the delayed delete
   */
   WinStartTimer( hab,
                  hwnd,
                  DELAY_TIMER_ID,
                  DELAY_TIMER_TICK );

   *ppOpenFilesHead = NULL;           /* Set the file list to empty. */
}





static void TVRemoteOpen( HWND hwnd, OPENFILELIST **ppOpenFilesHead, MPARAM mp1, MPARAM mp2 )
{
   OPENFILELIST      *pNewFileData;
   TVREMOTE_OPEN_PARM TvremoteParm; /* The passed paramter area. */
   BOOL               AlreadyInList = FALSE;

   /* Extract the passed parameter. */
   TvremoteParm = TVREMOTE_MsgParmGet(mp1,mp2);

   /* Add the file to the open list of it was opened successfully. */
   if (TVREMOTE_OpenRc(TvremoteParm) == NO_ERROR)
   {
#ifdef TVSCREEN_RECYCLE_OPEN_FILES

      OPENFILELIST  *pCurrent;

      /* First, see if the file recently closed (and is in the list, half-toned) */
      pCurrent = *ppOpenFilesHead;
      while (pCurrent)
      {
         if( !stricmp(TVREMOTE_OpenTvfsName(TvremoteParm), pCurrent->pszTvfsName) &&
             !stricmp(TVREMOTE_OpenFileName(TvremoteParm), pCurrent->pszFileName) )
         {
            pCurrent->FileHandle  = TVREMOTE_OpenFileHandle(TvremoteParm);
            pCurrent->FileOpen    = TRUE;
            AlreadyInList = TRUE;
            break;
         }

         /* process the next entry. */
         pCurrent = pCurrent->Next;
      }
#endif

      if (!AlreadyInList)
      {
         /* Allocate the latest open file entry. */
         pNewFileData = malloc( sizeof(OPENFILELIST) );
         pNewFileData->pszTvfsName = strdup(TVREMOTE_OpenTvfsName(TvremoteParm));
         pNewFileData->pszFileName = strdup(TVREMOTE_OpenFileName(TvremoteParm));
         pNewFileData->FileHandle  = TVREMOTE_OpenFileHandle(TvremoteParm);
         pNewFileData->FileOpen    = TRUE;

         /* Link in the latest open file entry. */
         pNewFileData->Next = *ppOpenFilesHead;
         *ppOpenFilesHead   = pNewFileData;
      }
   }

   /* Free the passed parameter. */
   TVREMOTE_MsgParmUnblock(TvremoteParm);
   TVREMOTE_MsgParmFree(TvremoteParm);

   /* Update the display. */
   WinPostMsg( hwnd, TVSCREEN_INVALIDATE, (MPARAM)NULL, (MPARAM)NULL );
}



static void TVRemoteClose( HWND hwnd, OPENFILELIST **ppOpenFilesHead, MPARAM mp1, MPARAM mp2 )
{
   OPENFILELIST       *pCurrent;
   TVREMOTE_CLOSE_PARM TvremoteParm; /* The passed paramter area. */

   /* Extract the passed parameter. */
   TvremoteParm = TVREMOTE_MsgParmGet(mp1,mp2);

   /* Locate the entry to close. */
   pCurrent = *ppOpenFilesHead;
   while (pCurrent)
   {
      if ( (pCurrent->FileHandle == TVREMOTE_CloseFileHandle(TvremoteParm) ) &&
           (pCurrent->FileOpen) )
      {
         pCurrent->FileOpen  = FALSE;
         pCurrent->TicksLeft = DELETE_AFTER_TICKS;
         break;
      }

      /* process the next entry. */
      pCurrent = pCurrent->Next;
   }

   /* Free the passed parameter. */
   TVREMOTE_MsgParmUnblock(TvremoteParm);
   TVREMOTE_MsgParmFree(TvremoteParm);

   /* Update the display. */
   WinPostMsg( hwnd, TVSCREEN_INVALIDATE, (MPARAM)NULL, (MPARAM)NULL );
}




static void Paint( HWND hwnd, OPENFILELIST **ppOpenFilesHead, BOOL fWantDelay )
{
   CHAR           Output[CCHMAXPATH*2 + 5];   /* filename+filename+" -> "+\0  */
   FONTMETRICS    Fm;
   LONG           Halftone;
   OPENFILELIST  *pCurrent;
   RECTL          rclWindow;
   RECTL          rclText;
   HPS            hps;                            /* Presentation Space handle    */

   /* Obtain a cached micro PS. */
   hps = WinBeginPaint( hwnd, 0, 0 );

   /* Extract window parameters. */
   GpiQueryFontMetrics( hps, sizeof(Fm), &Fm );
   WinQueryWindowRect( hwnd, &rclWindow );


   /* Display the set of open TVFS files. */
   for( pCurrent = *ppOpenFilesHead; pCurrent; pCurrent = pCurrent->Next )
   {
      sprintf(Output, "%s -> %s", pCurrent->pszTvfsName, pCurrent->pszFileName );

      if( pCurrent->FileOpen )
         Halftone = 0;
      else
      {
         if( fWantDelay )
            Halftone = DT_HALFTONE;
         else
            continue;
      }

      rclText = rclWindow;
      rclText.yBottom = rclText.yTop - Fm.lMaxBaselineExt;

      /* Draw the text */
      WinDrawText( hps,
                   strlen(Output),
                   Output,
                  &rclText,
                   0L,
                   0L,
                   (USHORT) (DT_LEFT | DT_TOP | DT_ERASERECT | DT_TEXTATTRS | Halftone ));

      /* Now, find out the space that the text took (so we can blank the rest) */
      WinDrawText( hps,
                   strlen(Output),
                   Output,
                  &rclText,
                   0L,
                   0L,
                   (USHORT) (DT_LEFT | DT_TOP | DT_QUERYEXTENT | DT_TEXTATTRS | Halftone) );

      rclText.xLeft  = rclText.xRight;
      rclText.xRight = rclWindow.xRight;
      WinFillRect(hps, &rclText, CLR_WHITE);   /* Erase the rest of the line */

      rclWindow.yTop -= Fm.lMaxBaselineExt;  /* Move down a line */
      if (rclWindow.yTop <= rclWindow.yBottom)
         break;

      
   }

   if (rclWindow.yTop >= rclWindow.yBottom)
      WinFillRect(hps, &rclWindow, CLR_WHITE);   /* Erase the rest of the screen */

   /* The drawing is complete. */
   WinEndPaint( hps );
}
