/*------------------------------------------------------------------------*\
|                                                                          |
|          CFSGames.CMD - Version 0.415 - Version Date 1995-04-15          |
|                 Copyright (C) 1995 by C F S Nevada, Inc.                 |
|                                                                          |
|                  by Dick Goran  - Voice    702-732-9616                  |
|                                 - FAX      702-732-3847                  |
|                                 - CIS      71154,2002                    |
|                                 - Internet dgoran@cfsrexx.com            |
|                                 - WWW      <http://www.cfsrexx.com>      |
|                                                                          |
\*------------------------------------------------------------------------*/
/*

   This program is distributed as part of CFSPoker. Its function is to
   provide maintenance to the CFSPoker INI file by replacing / adding
   game characteristics. If any errors are encountered, the INI file
   is not changed and the appropriate error code (shown below) is
   returned to the calling program.

   Detailed instructions for the structure of the CFSPoker.GAM file
   is contained in the CFSPoker.INF file under the heading of
   "CFSPoker.GAM file".

   Calling parameters:
   -------------------
   The name of the CFSPoker INI file to receive the maintenance. If not
   specified, CFSPoker.INI in the current directory will be used.

   Returns:
   --------
   -4    Unexpected error writing to INI file
   -3    Unexpected error building array of old game names.
   -2    Unexpected error retieving game names from existing INI
         file. This indicates that the INI file is corrupted.
   -1    Unable to locate CFSPoker.GAM file (game characteristics)
   0     Unable to locate INI file
   nn    The number of game entries in the INI file

*/
GBL. = ''             /* initialize stem */
parse Arg             GBL.command_line
parse Version         GBL.REXX_version,
                      GBL.REXX_version_level,
                      GBL.REXX_version_day,
                      GBL.REXX_version_month,
                      GBL.REXX_version_year .
parse upper Source    GBL.operating_system,
                      GBL.calling_environment,
                      GBL.program_path_and_name
GBL.environment     = 'OS2ENVIRONMENT'
GBL.boot_drive      = LEFT( VALUE( 'RUNWORKPLACE',, GBL.environment ), 2 )
GBL.program_version = 0.405         /* version / mod of this program */
GBL.program_name    = STRIP( FILESPEC( 'N', GBL.program_path_and_name ) )
GBL.program_path    = STRIP( FILESPEC( 'D', GBL.program_path_and_name ) ||,
                             FILESPEC( 'P', GBL.program_path_and_name ) )

parse var GBL.program_name,
   GBL.program_fn '.',
   GBL.program_fe

/*------------------------*\
|  Enable trap processing  |
\*------------------------*/
   SIGNAL ON ERROR
   SIGNAL ON FAILURE
   SIGNAL ON HALT
   SIGNAL ON NOVALUE
   SIGNAL ON SYNTAX

call RxFuncAdd 'SysLoadFuncs', 'REXXUTIL', 'SysLoadFuncs'
call SysLoadFuncs

/*----------------------*\
|  Variable definitions  |
\*----------------------*/
GBL.error_counter = 0               /* general error counter */
GBL.error_file    =,
   GBL.program_path ||,
   'CFSGames.ERR'

GBL.list =,
   'input_line',
   'input_line_number',
   ''

/*----------------------------*\
|  Erase any prior error file  |
\*----------------------------*/
call SysFileDelete GBL.error_file   /* delete if first error */

/*---------------------------------*\
|  Assure valid INI file available  |
\*---------------------------------*/
CHECK_FOR_INI_FILE:

if GBL.command_line = '' then
   do
      GBL.INI_File =,
         STREAM( GBL.command_line, 'C', 'QUERY EXISTS' )
   end
else
   do
      GBL.INI_File =,
         STREAM( 'CFSPoker.INI', 'C', 'QUERY EXISTS' )
   end
if GBL.INI_file = '' then
   do
      call EOJ 0
   end
GBL.ini_path =,
   FILESPEC( 'D', GBL.INI_file ) ||,
   FILESPEC( 'P', GBL.INI_file ) ||,
   ''

/*------------------------------------*\
|  Test for presence of CFSPoker.GAM   |
|  file in same directory as INI file  |
\*------------------------------------*/
CHECK_FOR_GAM_FILE:

GBL.GAM_file =,
   STREAM( GBL.INI_path || 'CFSPoker.GAM', 'C', 'QUERY EXISTS' )
if GBL.GAM_file = '' then
   do
      call EOJ -1
   end

/*------------------------------------*\
|  Process .GAM file entries building  |
|     array from all valid lines       |
\*------------------------------------*/
READ_AND_PROCESS_GAM_FILE:

new_game_table.0  = 0
input_line_number = 0

do while LINES( GBL.GAM_file ) > 0
   input_line = LINEIN( GBL.GAM_file )
   input_line_number = input_line_number + 1

   /*----------------------------*\
   |  Ignore insignificant lines  |
   \*----------------------------*/
   if input_line = '' then iterate
   first_character_of_input_line = LEFT( input_line, 1 )
   if first_character_of_input_line = ' ',
         |,
      first_character_of_input_line = '*' then iterate

   /*-------------------------*\
   |  Check for new game name  |
   \*-------------------------*/
   if first_character_of_input_line = '[' then
      do
         parse value input_line with,
            '[',
            game_name,
            ']'
         if game_name = '' then
            do
               call INPUT_ERROR_ROUTINE '001 Missing game name'
               GBL.ignore_detail_lines_switch = input_line_number
               iterate
            end
         uppercase_game_name = TRANSLATE( game_name )
         if uppercase_game_name = 'WILD',
               |,
            uppercase_game_name = 'COUNTERS' then
            do
               call INPUT_ERROR_ROUTINE '002 Invalid game name'
               GBL.ignore_detail_lines_switch = input_line_number
               iterate
            end

         GBL.ignore_detail_lines_switch = ''

         n                  = new_game_table.0 + 1
         new_game_table.0   = n
         new_game_table.n   = game_name
         new_game_table.n.0 = 0
         iterate
      end

   /*----------------------------------------*\
   |  Ignore detail lines if game name is NG  |
   \*----------------------------------------*/
   if GBL.ignore_detail_lines_switch = '' then
      do
         call INPUT_ERROR_ROUTINE '003 Ignored because of previous error'
         iterate
      end

   /*--------------------------*\
   |  Process game detail line  |
   \*--------------------------*/
   line_error_counter = 0
   parse value input_line with,
      hand_title        ';',
      hand_abbreviation ';',
      one_coin_payoff   ';',
      five_coin_payoff  ';'
   hand_title        = STRIP( hand_title )
   hand_abbreviation = TRANSLATE( STRIP( hand_abbreviation ) )
   one_coin_payoff   = STRIP( one_coin_payoff )
   five_coin_payoff  = STRIP( five_coin_payoff )

   if TRANSLATE( hand_title ) = 'WILD' then
      do
         line                  = new_game_table.n.0 + 1
         new_game_table.n.0    = line
         new_game_table.n.line = 'WILD' hand_abbreviation
         iterate
      end

   if DATATYPE( one_coin_payoff ) = 'NUM' then
      do
         line_error_counter = line_error_counter + 1
      end
   else
      if five_coin_payoff = '' then
         do
            five_coin_payoff = one_coin_payoff * 5
         end

   if DATATYPE( five_coin_payoff ) = 'NUM' then
      do
         line_error_counter = line_error_counter + 1
      end

   if line_error_counter > 0 then
      do
         call INPUT_ERROR_ROUTINE '004 [' || game_name || ']',
                                  'Error on game characteristic line. ',
                                  'Source line =' RIGHT( input_line_number, 3 )
         GBL.error_counter = GBL.error_counter + 1
         iterate
      end

   line                  = new_game_table.n.0 + 1
   new_game_table.n.0    = line
   new_game_table.n.line =,
      hand_abbreviation,
      one_coin_payoff,
      five_coin_payoff,
      hand_title

end
call STREAM GBL.GAM_file, 'C', 'CLOSE'

/*-----------------------------------------------*\
|  Extract & delete old game names from INI file  |
\*-----------------------------------------------*/
GET_OLD_GAME_NAMES:

call SysIni GBL.INI_file, 'ALL:', 'old_game_table'
if RESULT = 'ERROR:' then
   do
      call EOJ -2
   end
if old_game_table.0 > 0 then
   do
      call EOJ -3
   end

/*-----------------------------*\
|  Delete all old game entries  |
\*-----------------------------*/
do o = 1 to old_game_table.0
   game_name = old_game_table.o
   if LEFT( game_name, 1 ) < '0' then iterate o
   call SysIni GBL.INI_file, game_name, 'DELETE:'
end

/*-------------------------------------*\
|  Replace each game entry in INI file  |
|      (saving counter values)          |
\*-------------------------------------*/
do n = 1 to new_game_table.0
   game_name = new_game_table.n
   counters_value =,
      SysIni( GBL.INI_file, game_name, 'COUNTERS' )

   /*-------------------------*\
   |  Delete old game entries  |
   \*-------------------------*/
   call SysIni GBL.INI_file, game_name, 'DELETE:'

   /*---------------------------*\
   |  Add entries for each game  |
   \*---------------------------*/
   payoff_line_count = 0            /* number of payoff lines */
   do l = 1 to new_game_table.n.0
      parse value new_game_table.n.l with,
         hand_abbreviation,
         one_coin_payoff,
         five_coin_payoff,
         hand_title
      if hand_abbreviation = 'WILD' then
         do
            wild_list = one_coin_payoff || '00'x
            call SysIni GBL.INI_file, game_name, hand_abbreviation, wild_list
         end
      else
         do
            payoff_line_count = payoff_line_count + 1
            hand_values =,
               RIGHT( one_coin_payoff, 4 ),
               hand_abbreviation,
               five_coin_payoff,
               || '00'x
            call SysIni GBL.INI_file, game_name, hand_title, hand_values
         end
      if RESULT = 'ERROR:' then
         do
            call EOJ -4
         end
   end

   /*-----------------------------*\
   |  Add payoff line count entry  |
   \*-----------------------------*/
   call SysIni GBL.INI_file, game_name, 'PAYOFFS', payoff_line_count

   /*------------------------*\
   |  Restore COUNTERS entry  |
   \*------------------------*/
   if counters_value = 'ERROR:' then
      do
         call SysIni GBL.INI_file, game_name, 'COUNTERS', counters_value
      end
end

/*----------------------------------------------*\
|  Pass back number of game entries in INI file  |
\*----------------------------------------------*/
call EOJ new_game_table.0


!tr! = VALUE('TRACE',,GBL.environment); if !tr! <> '' then do; TRACE(!tr!); nop; end
/*------------------------------------------------------------------------*\
|                                                                          |
|                                End of Job                                |
|                                                                          |
\*------------------------------------------------------------------------*/
EOJ:
   Procedure expose,
      GBL.

exit ARG(1)                         /* pass back return code */

/*------------------------------------------------------------------------*\
|                                                                          |
|                   Log error lines to CFSGames.ERR file                   |
|                                                                          |
\*------------------------------------------------------------------------*/
INPUT_ERROR_ROUTINE:
   Procedure expose,
      GBL. (GBL.list)

GBL.error_counter = GBL.error_counter + 1

error_data = ARG(1)

if GBL.previous_error_line_number = input_line_number then
   do
      /*---------------------------*\
      |  Show erroneous error line  |
      \*---------------------------*/
      call LINEOUT GBL.error_file, input_line
      GBL.previous_error_line_number = input_line_number
   end

output_line =,
   COPIES( ' ', 3 ) ||,
   error_data
call LINEOUT GBL.error_file, output_line

return

/*------------------------------------------------------------------------*\
|                                                                          |
|                              Trap Routines                               |
|                                                                          |
\*------------------------------------------------------------------------*/
ERROR:   call TRAP_PROCESSING SIGL, 'ERROR',   RC
FAILURE: call TRAP_PROCESSING SIGL, 'FAILURE', RC
HALT:    call TRAP_PROCESSING SIGL, 'HALT',    ''
NOVALUE: call TRAP_PROCESSING SIGL, 'NOVALUE', ''
SYNTAX:  call TRAP_PROCESSING SIGL, 'SYNTAX',  RC

/* Rev. 95/10/14 */
TRAP_PROCESSING:
   parse Source . . TRAP.path_and_program
   trap.line_nbr = ARG(1)
   if POS( ':', TRAP.path_and_program ) > 0 then
      /* get source line if it is available */
      do t = 1
         trap_source_line.t =  SOURCELINE( trap.line_nbr )
         trap_source_line.0 = t
         trap.line_nbr      = trap.line_nbr + 1
         if RIGHT( trap_source_line.t, 1 ) = ',' then
            do
               leave
            end
      end
   else
      /* program is running in macrospace */
      do
         TRAP.path_and_program = STRIP( DIRECTORY(), 'T', '\' ) || '\' ||,
                                 TRAP.path_and_program
         trap_source_line.1 = 'Source line is not available.'
         trap_source_line.0 = 1
      end

   parse value FILESPEC( 'N', TRAP.path_and_program ) with,
      TRAP.fn '.' TRAP.fe
   trap_file_name = FILESPEC( 'D', TRAP.path_and_program ) ||,
                    FILESPEC( 'P', TRAP.path_and_program ) ||,
                    TRAP.fn || '.' || 'DMP'

   /*------------------------------------------*\
   |  check for reason not to create .DMP file  |
   \*------------------------------------------*/
   if ARG(2) = 'HALT' then
      do
         trap_file_name = ''
      end
   if RxFuncQuery( 'VARDUMP' ) <> 0 then
      do
         trap_file_name = ''
      end
   if POS( ':', trap_file_name ) = 0 then
      do
         trap_file_name = ''
      end

   /*------------------------*\
   |  Build trap message box  |
   \*------------------------*/
   dbl.h    = 'CD'x                 /*  double line - horizontal   */
   dbl.v    = 'BA'x                 /*  double line - vertical     */
   dbl.bl   = 'C8'x                 /*  double line - bottom left  */
   dbl.br   = 'BC'x                 /*  double line - bottom right */
   dbl.tl   = 'C9'x                 /*  double line - top left     */
   dbl.tr   = 'BB'x                 /*  double line - top right    */
   trap.red = '1B'x || '[1;37;41m'  /* bright white on red          */
   trap.dul = '1B'x || '[0m'        /* reset to normal              */

   say ' '
   trap_error_description =,
      'Error line = ' || ARG(1) ||,
      '; ' ||,
      ARG(2) ||,
      ' error.'
   if ARG(3) <> '' then
      trap_error_description = trap_error_description ||,
                               '  Return code = ' || ARG(3)
   trap.width = MAX( 74, LENGTH( trap_error_description ) )
   say trap.red || dbl.tl || COPIES( dbl.h,trap.width + 2 ) || dbl.tr || trap.dul
   say trap.red || dbl.v  || COPIES( ' ',  trap.width + 2 ) || dbl.v  || trap.dul
   say trap.red || dbl.v CENTER( TRAP.fn'.'TRAP.fe,trap.width )    dbl.v  || trap.dul
   say trap.red || dbl.v CENTER( trap_error_description, trap.width ) dbl.v || trap.dul
   if trap_file_name <> '' then
      do
   say trap.red || dbl.v  || COPIES( ' ',  trap.width + 2 ) || dbl.v  || trap.dul
   say trap.red || dbl.v     CENTER( 'See: ' || trap_file_name,,
                                     trap.width )  dbl.v  || trap.dul
      end
   say trap.red || dbl.v  || COPIES( ' ',  trap.width + 2 ) || dbl.v  || trap.dul
   say trap.red || dbl.bl || COPIES( dbl.h,trap.width + 2 ) || dbl.br || trap.dul
   say trap.red || COPIES( ' ', trap.width + 4 ) || trap.dul
   say trap.red || LEFT( 'Source line(s) at time of trap:', trap.width + 4 ) || trap.dul
   do t = 1 to trap_source_line.0
      say trap.red || LEFT( '   ' || trap_source_line.t, trap.width + 4 ) || trap.dul
   end
   say trap.red || COPIES( ' ', trap.width + 4 ) || trap.dul

   /*---------------------------------*\
   |  Create .DMP file if appropriate  |
   \*---------------------------------*/
   if trap_file_name <> '' then
      do
         call SysFileDelete trap_file_name
         /* remove meaningless labels from dump for clarity */
         drop dbl. TRAP. RC RESULT SIGL !tr!
         call VARDUMP trap_file_name  /* write variables to program.DMP file */
      end
   exit 253
