//-< FILE.CXX >------------------------------------------------------*--------*
// POST++                     Version 1.0        (c) 1998  GARRET    *     ?  *
// (Persistent Object Storage)                                       *   /\|  *
//                                                                   *  /  \  *
//                          Created:      2-Feb-98    K.A. Knizhnik  * / [] \ *
//                          Last update: 22-Feb-98    K.A. Knizhnik  * GARRET *
//-------------------------------------------------------------------*--------*
// Mapped on memory file class implementation
//-------------------------------------------------------------------*--------*

#include "storage.h"

file* file::chain;

bool file::handle_page_modification(void* addr) 
{
    for (file* fp = chain; fp != NULL; fp = fp->next) { 
	if (size_t((char*)addr - fp->base) < fp->mapped_size) { 
	    return fp->create_shadow_page(addr);
	}
    }  
    return false;
} 

#if defined(__unix__)

#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <sys/resource.h>

#ifdef SA_SIGINFO
#include <sys/siginfo.h>
#else
#define SA_SIGINFO 0
#endif

#ifndef MAP_FAILED
#define MAP_FAILED (-1)
#endif

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

#ifndef MAP_VARIABLE
#define MAP_VARIABLE 0
#endif

#ifndef MAP_ANONYMOUS 
#define MAP_ANONYMOUS 0
#define USE_SHMEM     1
#endif

#ifndef MAP_NO_RESERVEX
// This attrbute used in Solaris to avoid reservation space in swap file
#define MAP_NO_RESERVE 0
#endif

#ifdef USE_SHMEM 
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#define USE_WRITEV 1 // most modern Unix-s have this function
#ifdef USE_WRITEV
#include <sys/uio.h>
#endif

#define USE_FDATASYNC 1 // more efficient version of fsync()


// minimal number of pages appended to the file at each file extensin
#define FILE_EXTENSION_QUANT 16

//
// mlock(), munlock() functions can be used in transaction mode to 
// provide buffering of shadow pages writes to transaction log.
// In Unix unly superuser can use this functions. Method file::open()
// will check if it is possible to use mlock() function.
// 
#if !defined(__osf__)
// In Digital Unix standard C library has no definition of mlock(), munlock()
#define USE_MLOCK 1
#endif

#ifndef USE_MLOCK
#undef mlock
#undef munlock
#define mlock(p, s) -1
#define munlock(p, s) -1
#endif

#if defined(__linux__)
//
// In Linux it is possible to create copy on write memory mapping object with 
// size greater than size of the mapped file and access pages beyond end 
// of file. In most other Unix-s access beyond end of file causes protection 
// violation.
//
#define ACCESS_BEYOND_END_OF_FILE 1
#else
#define ACCESS_BEYOND_END_OF_FILE 0
#endif

//
// It will takes a lot of time and disk space to produce core file
// for program with huge memory mapped segment. This option allows you
// to automatically start debugger when some exception is happened in program.
//
#define CATCH_SIGNALS 1

#ifdef CATCH_SIGNALS
#include <signal.h>

#define WAIT_DEBUGGER_TIMEOUT 10

static void fatal_error_handler(int signo) { 
    char  buf[64];
    char* exe_name = getenv("_");
    fprintf(stderr, "\
Program catch signal %d.\n\
Input path to executable file to debug or press Ctrl/C to terminate program:\n\
[%s] ", 
	    signo, exe_name);
    int pid = getpid();
    if (fgets(buf, sizeof buf, stdin)) { 
	char pid_str[16];
	int len = strlen(buf);
	if (len > 1) { 
	    buf[len-1] = '\0'; // truncate '\n'
	    exe_name = buf;
	}
	sprintf(pid_str, "%d", pid); 
	if (fork()) { 
	    sleep(WAIT_DEBUGGER_TIMEOUT);
	} else { 
	    execlp("gdb", "gdb", "-q", "-s", exe_name, exe_name, pid_str, 0);
	}
    } else { 
	kill(pid, SIGKILL);
    }
}
#endif


inline bool file::flush_log_buffer()
{
#if USE_FDATASYNC
    if (fdatasync(log) != ok) { 
#else
    if (fsync(log) != ok) { 
#endif
	error_code = errno;
	TRACE(("file::flush_log_buffer: fsync failed: %s\n", 
	       strerror(error_code)));
	return false;
    }
    for (int i = n_locked_pages; --i >= 0;) { 
	if (munlock(locked_page[i], page_size) != ok) { 
	    error_code = errno;
	    TRACE(("file::flush_log_buffer: munlock %p failed: %s\n",
		   locked_page[i], strerror(error_code)));
	    return false;
	}
    }
    n_locked_pages = 0;
    return true;
}

bool file::create_shadow_page(void* addr)
{
    char* page = (char*)(long(addr) & ~(page_size-1));
    if (mprotect(page, page_size, PROT_READ|PROT_WRITE) != ok) { 
	TRACE(("file::create_shadow_page: mprotect failed for %p: %s\n", 
	       page, strerror(errno)));
	return false;
    }
#if USE_WRITEV
    iovec iov[2];
    long offs = page - base;
    iov[0].iov_base = (char*)&offs;
    iov[0].iov_len = sizeof offs;
    iov[1].iov_base = page;
    iov[1].iov_len = page_size;
    if ((size_t)writev(log, iov, 2) != sizeof(long) + page_size) { 
	TRACE(("file::create_shadow_page: writev failed: %s\n",
	       strerror(errno)));
	return false;
    }
#else
    *(long*)log_buffer = page - base;
    memcpy(log_buffer + sizeof(long), page, page_size);
    size_t trans_size = page_size + sizeof(long);
    if (write(log, log_buffer, trans_size) != trans_size) { 
	TRACE(("file::create_shadow_page: failed to write log file: %s\n", 
	       strerror(errno)));
	return false;
    }
#endif
    if (max_locked_pages != 0) { 
	if (n_locked_pages >= max_locked_pages) { 
	    return flush_log_buffer();
	}  
	if (mlock(page, page_size) != ok) { 
	    TRACE(("file::create_shadow_page: mlock %p failed: %s\n", 
		   page, strerror(errno)));
	    if (n_locked_pages != 0) { 
		max_locked_pages = n_locked_pages;
	    }
	    return flush_log_buffer();
	}  
	locked_page[n_locked_pages++] = page;
    }
    return true;
}


#if defined(__linux__)
struct rtm_linux_exception_frame {
    long	signr;
    long	gs, fs, es, ds;
    long	edi, esi, ebp, efp;
    long	ebx, edx, ecx, eax;
    long	tss_trap_no, tss_error_code;
    long	eip, cs, eflags, esp, ss;
    long	fpp_stat;
    // Two following fields are not provided by iBCS2
    long	oldmask, tss_cr2;
};

static void sigsegv_handler(int signo)
{
    rtm_linux_exception_frame* efp;
    void* addr;

    efp = (rtm_linux_exception_frame*)&signo;
    addr = (void*)efp->tss_cr2;

    if (!file::handle_page_modification(addr)) {
	kill(getpid(), SIGABRT);
    }
}
#else
static void sigsegv_handler(int, struct siginfo *info)
{
    if (!file::handle_page_modification(info->si_addr)) {
	kill(getpid(), SIGABRT);
    }
}
#endif

file::file(const char* name, size_t max_file_size, size_t max_locked_pages)
{
    // reserver one char for backup/log file names construction
    this->name = new char[strlen(name)+2]; 
    strcpy(this->name+1, name); 
#ifndef USE_MLOCK
    max_locked_pages = 0;
#else
    if (max_locked_pages != 0) { 
	// Check if we can use mlock()
	if (mlock((char*)this, page_size) != ok) { 
	    max_locked_pages = 0;
	    TRACE(("file::open: mlock test failed: %s\n",strerror(errno)));
	} else { 
	    munlock((char*)this, page_size);
	}
    }
#endif
    struct rlimit mem_limit;
    if (getrlimit(RLIMIT_DATA, &mem_limit) == ok) { 
	if (max_file_size > size_t(mem_limit.rlim_cur)) { 
	    max_file_size = mem_limit.rlim_cur & ~(allocation_granularity-1);
	    TRACE(("file::file: set max_file_size to %ld bytes\n", 
		   max_file_size));
	}
    } else {
	TRACE(("file::file: getrlimit failed: %s\n", strerror(errno)));
    }

    allocation_granularity = page_size = getpagesize();
    this->max_file_size = ALIGN(max_file_size, allocation_granularity);
    this->max_locked_pages = max_locked_pages;

    log_buffer = new char[page_size+sizeof(long)];
    locked_page = new char*[max_locked_pages];
    file_extension_granularity = FILE_EXTENSION_QUANT*page_size;

    static struct sigaction sigact; 
    sigact.sa_handler = (void(*)(int))sigsegv_handler;
    sigact.sa_flags = SA_SIGINFO;
    sigaction(SIGSEGV, &sigact, NULL);
#ifdef CATCH_SIGNALS
    sigact.sa_flags = 0;
    sigact.sa_handler = fatal_error_handler;
    sigaction(SIGBUS, &sigact, NULL);
    sigaction(SIGILL, &sigact, NULL);
    sigaction(SIGABRT, &sigact, NULL);
#endif
    error_code = ok;
}

file::~file() 
{
    delete[] name;
    delete[] log_buffer;
    delete[] locked_page;
}

char* file::get_error_text(char* buf, size_t buf_size)
{
    char* err_txt;
    switch (error_code) {
      case ok: 
	err_txt = "no error";
	break;
      case file_size_not_aligned:
	err_txt = "size of file is not aligned on page boundary";
	break;
      case file_mapping_size_exceeded:
	err_txt = "attempt to exeed the limit on mapped object size";
	break;
      case not_in_transaction:
	err_txt = "operation is possible only inside transaction"; 
	break;
      case end_of_file:
	err_txt = "unexpected end of file"; 
	break;
      default: 
	err_txt = strerror(error_code);
    }
    return strncpy(buf, err_txt, buf_size);
}

inline bool file::recover_file()
{
    ssize_t rc, trans_size = page_size + sizeof(long);
    int n_recovered_pages = 0;
    TRACE(("file::recover_file: recover data file from transaction log\n"));
    while ((rc = read(log, log_buffer, trans_size)) == trans_size) { 
	memcpy(base + *(long*)log_buffer, log_buffer+sizeof(long), page_size);
	n_recovered_pages += 1;
    }
    if (rc != 0) {
	if (rc < 0) { 
	    error_code = errno;
	    TRACE(("file::recover: log read failed: %s\n", 
		   strerror(error_code)));
	} else { 
	    error_code = end_of_file;	
	    TRACE(("file::recover: expected end of log\n"));
	}
	return false;
    }
    allocated_size = size = ALIGN(((file_header*)base)->file_size, page_size);
    TRACE(("recover::file: recover %d pages, set file size to %ld\n", 
	   n_recovered_pages, size));
    error_code = ok;
    return true;
}


bool file::open(open_mode mode, access_prot prot) 
{
    assert(name != NULL);
    if (prot == read_only) { 
	mode = copy_on_write_map;
    } 
#ifdef USE_SHMEM
    else if (mode == copy_on_write_map) { 
	mode = load_in_memory; // I don't want to mix IPC and mmap mechanisms
    } 
#endif
    this->mode = mode;
    this->prot = prot;

    if (mode == shadow_pages_transaction) {
	fd = ::open(name+1, O_RDWR|O_CREAT|O_SYNC, 0777);
	if (fd < 0) { 
	    error_code = errno;
	    TRACE(("file::open: failed to open file '%s': %s\n",
		   name+1, strerror(error_code)));
	    return false;
	}
	file_header hdr;
	hdr.base_address = NULL;
	hdr.file_size = 0;
	read(fd, &hdr, sizeof hdr);
	base = (char*)hdr.base_address;
	allocated_size = size = ALIGN(hdr.file_size, page_size);
	mapped_size = size > max_file_size ? size : max_file_size;
	void* p = mmap(base, mapped_size, PROT_READ|PROT_WRITE,
		       MAP_VARIABLE|MAP_SHARED|MAP_FILE, fd, 0);
	if (p == (char*)MAP_FAILED) { 
	    error_code = errno;
	    TRACE(("file::open: mmap failed: base=%p, size=%ld: %s\n",
		   base, mapped_size, strerror(error_code)));
	    ::close(fd);
	    return false;
	}
	base = (char*)p;
	name[0] = '.'; // transaction log file name prefix
	n_locked_pages = 0;

	int log_flags = O_RDWR;
	if (max_locked_pages == 0) { 
	    log_flags |= O_SYNC;
	}
	log = ::open(name, log_flags, 0); 
	if (log < 0) { 
	    if (errno != ENOENT || 
		(log = ::open(name, log_flags|O_CREAT, 0777)) < 0)
	    {
		error_code = errno;
		TRACE(("file::open: failed to open log file '%s': %s\n",
		       name, strerror(error_code)));
		munmap(base, mapped_size);
		::close(fd);
		return false;
	    }
	    if (size != 0 && mprotect(base, size, PROT_READ) != ok) {
		error_code = errno;
		TRACE(("file::open: mprotect failed: address=%p, size=%ld: "
		       "%s\n", base, size, strerror(error_code)));
		munmap(base, mapped_size);
		::close(fd);
		::close(log);
		return false;
	    }
	} else { 
	    recover_file();
	    if (!commit()) { 
		munmap(base, mapped_size);
		::close(fd);
		::close(log);
		return false;
	    }
	}
	next = chain;
	chain = this;
    } else { // mapping file in non-transaction mode
	fd = ::open(name+1, O_RDONLY, 0);
	int mmap_attr = MAP_VARIABLE|MAP_PRIVATE;
	base = NULL;
	int d = fd;
	if (fd >= 0) { 
	    size_t rc = read(fd, &base, sizeof base);
	    if (rc != 0 && rc != sizeof base) { 
		error_code = errno;
		TRACE(("file::open: failed to read from file '%s': %s\n",
		       name+1, strerror(error_code)));
		::close(fd);
		return false;
	    }
	    size = ALIGN(lseek(fd, 0, SEEK_END), allocation_granularity);
	    TRACE(("file::open: file '%s' exists: size=%ld, base=%p\n", 
		   name+1, size, base));
	    if (mode == copy_on_write_map && ACCESS_BEYOND_END_OF_FILE) { 
		mmap_attr |= MAP_FILE;
	    } else { 
		d = -1;
		mmap_attr |= MAP_ANONYMOUS;
	    }
	} else { 
	    error_code = errno;
	    TRACE(("file::open: can't open file '%s', %s\n", 
		   name+1, strerror(error_code)));
	    if (error_code != ENOENT) { 
		return false;
	    }
	    if (prot == read_only) {
		TRACE(("file::open: failed to open storage in read_only mode "
		       "because file '%s' doesn't exist\n", name+1));
		return false;
	    }
	    size = 0;
	    mmap_attr |= MAP_ANONYMOUS;
	}
	if (prot != read_only) { 
	    if (size >= max_file_size) { 
		TRACE(("file::open: size of file >= max_file_size=%ld\n",
		       max_file_size));
		//
		// Map all file to the memory, no more space is reserved
		//
		mapped_size = size;
		if (size == 0) { 
		    TRACE(("file::open: empty file is mapped\n"));
		    return true;
		}
		void* p = mmap(base, size, PROT_READ|PROT_WRITE, 
			       MAP_VARIABLE|MAP_PRIVATE|MAP_FILE, fd, 0);
		if (p == (void*)MAP_FAILED) { 
		    error_code = errno;
		    TRACE(("file::open: failed to map file on %p: %s\n", 
			   base, strerror(error_code)));
		    ::close(fd);
		    return false;
		}
		TRACE(("file::open: map file to address %p\n", p));
		base = (char*)p;
	    } else { 
		//
		// Map the whole segment to memory (with swap as back storage)
		//
		mapped_size = max_file_size;
#ifdef USE_SHMEM
		int shmid = shmget(IPC_PRIVATE, mapped_size, IPC_CREAT|0777);
		if (shmid < 0) { 
		    error_code = errno;
		    if (fd >= 0) { 
			::close(fd);
		    }
		    TRACE(("file::open: failed to get SHM key: %s\n", 
			   strerror(error_code)));
		    return false;
		}
		void* p= shmat(shmid, base, SHM_RND);
		if (p == (void*)MAP_FAILED) { 
		    error_code = errno;
		    TRACE(("file::open: shmat at %p failed: %s\n",
			   base, strerror(error_code)));
		    shmctl(shmid, IPC_RMID, NULL);
		    if (fd >= 0) { 
			::close(fd);
		    }
		    return false;
		}
		TRACE(("file::open: create shared memory segment at %p\n", p));
		base = (char*)p;
		shmctl(shmid, IPC_RMID, NULL); // will be removed after detach
#else
		void* p = mmap(base, mapped_size, PROT_READ|PROT_WRITE, 
			       mmap_attr, d, 0);
		if (p == (void*)MAP_FAILED) { 
		    error_code = errno;
		    TRACE(("file::open: failed to map %ld bytes: %s\n", 
			   mapped_size, strerror(error_code)));
		    if (fd >= 0) { 
			::close(fd);
		    }
		    return false;
		}
		TRACE(("file::open: map %ld bytes to address %p\n", 
		       mapped_size, p));
		base = (char*)p;
#endif
		if (fd >= 0) { // file exists
		    if (mode == load_in_memory) { 
			//
			// Read file to memory
			//
			lseek(fd, 0, SEEK_SET);
			if ((size_t)read(fd, base, size) != size) { 
			    error_code = errno;
			    TRACE(("file::open: failed to read file in memory:"
				   " %s\n", strerror(error_code)));
#ifdef USE_SHMEM
			    shmdt(base);
#else
			    munmap(base, mapped_size);
#endif
			    ::close(fd);
			    return false;
			}
			::close(fd); // file is nor more needed
			fd = -1;
		    } else if (!ACCESS_BEYOND_END_OF_FILE) {
			//
			// Map file at the beginning of the same segment. 
			// With MAP_FIXED attribute mmap will unmap previous 
			// mapping first. 
			//
			void* p = mmap(base, size,
				       PROT_READ|PROT_WRITE, 
				       MAP_FIXED|MAP_PRIVATE|MAP_FILE, fd, 0);
			if (p == (void*)MAP_FAILED) {
			    error_code = errno;
			    TRACE(("file::open: failed to map file to address "
				   "%p: %s\n", p, strerror(error_code)));
			    close();
			    return false;
			}
			TRACE(("file::open: map file at address %p\n", p));
			assert((char*)p == base);
		    }
		}
	    }
	} else { // read only file mapping
	    mapped_size = size;
	    void* p = mmap(base, size, PROT_READ, 
			   MAP_VARIABLE|MAP_PRIVATE|MAP_FILE|MAP_NO_RESERVE,
			   fd, 0);
	    if (p == (void*)MAP_FAILED) { 
		error_code = errno;
		TRACE(("file::open: failed to readonly map file: %s\n",
		       strerror(error_code)));
		::close(fd);
		return false;
	    }
	    TRACE(("file::open: readonly map file at address %p\n", p));
	    base = (char*)p;
	}
    } 
    error_code = ok;
    return true;
}

bool file::set_size(size_t new_size)
{
    new_size = ALIGN(new_size, allocation_granularity);
    if (new_size > mapped_size) { 
	error_code = file_mapping_size_exceeded;
	return false;
    } 
    if (mode == shadow_pages_transaction) { 
	if (new_size > allocated_size) { 
	    allocated_size = ALIGN(new_size, file_extension_granularity);
	    if (ftruncate(fd, allocated_size) != ok) { 
		error_code = errno;
		return false;
	    }
	}
    }
    error_code = ok;
    size = new_size;
    return true;
}

bool file::set_protection(access_prot prot) 
{
    if (mprotect(base, mapped_size, 
		 prot == read_only ? PROT_READ : PROT_READ|PROT_WRITE) != ok)
    {
	error_code = errno;
	return false;
    } else { 
	error_code = ok;
	this->prot = prot;
	return true;
    }
}

bool file::commit()
{
    if (mode != shadow_pages_transaction) { 
	error_code = not_in_transaction;
	return false;
    }
    if (n_locked_pages != 0 && !flush_log_buffer()) { 
	return false;
    }
    if (size > 0 && msync(base, size, MS_SYNC) != ok) { 
	error_code = errno;
	TRACE(("file::commit: msync failed for address %p size %ld: %s\n", 
	       base, size, strerror(error_code)));
	return false;
    }
    if (lseek(log, 0, SEEK_SET) != 0 ||
	ftruncate(log, 0) != ok) 
    {
	error_code = errno;
	TRACE(("file::commit: failed to truncate log file: %s\n",
	       strerror(error_code)));
	return false;
    }
    if (size > 0 && mprotect(base, size, PROT_READ) != ok) {
	error_code = errno;
	TRACE(("file::commit: mprotect failed for address %p size %ld: %s\n",
	       base, size, strerror(error_code)));
	return false;
    }
    error_code = ok;
    return true;
}

bool file::rollback()
{
    if (mode != shadow_pages_transaction) { 
	error_code = not_in_transaction;
	return false;
    }
    if (lseek(log, 0, SEEK_SET) != 0) { 
	error_code = errno;
	TRACE(("file::rollback; failed to set position in log file: %s\n", 
	       strerror(error_code)));
	return false;
    }
    return recover_file();
}

bool file::flush() 
{
    if (prot == read_only) {
	return true;
    }
    if (mode == shadow_pages_transaction) { 
	return commit();
    } else { 
	name[0] = '.'; // temporary file prefix
	int new_fd = ::open(name, O_WRONLY|O_CREAT|O_TRUNC, 0666);
	if (new_fd < 0) { 
	    error_code = errno;
	    return false;
	}
	if ((size_t)write(new_fd, base, size) != size) { 
	    error_code = errno;
	    ::close(new_fd);
	    return false;
	}
	::close(new_fd);
	if (rename(name, name+1) != ok) { 
	    error_code = errno;
	    return false;
	}
	error_code = ok;
	return true;
    }
}

bool file::close()
{
    if (base == NULL) { 
	if (fd >= 0 && ::close(fd) != ok) { 
	    error_code = errno;
	    TRACE(("file::close: failed to close file: %s\n",
		   strerror(error_code)));
	    return false;
	}
	return true;
    }
    if (mode == shadow_pages_transaction) { 
	file *fp, **fpp = &chain;
	while ((fp = *fpp) != this) { 
	    fpp = &fp->next;
	}
	*fpp = fp->next;
	    
	if (n_locked_pages != 0 && !flush_log_buffer()) { 
	    return false;
	}
    }
#ifdef USE_SHMEM
    if (fd < 0) {
	//
	// We can use IPC mechanism only when file was not existed
	// or it was read to memory. In both cases fd == -1
	//
	if (shmdt(base) != ok) { 
	    error_code = errno;
	    TRACE(("file::close: failed to detach memory segment: %s\n",
		   strerror(error_code)));
	    return false;
	}
	return true;
    }
#endif
    if (munmap(base, mapped_size) != ok) { 
	error_code = errno;
	TRACE(("file::close: failed to unmap memory segment: %s\n",
	       strerror(error_code)));
	return false;
    }
    if (mode == shadow_pages_transaction) { 
	if (ftruncate(fd, size) != ok) {
	    error_code = errno;
	    TRACE(("file::close: failed to truncate file: %s\n",
		   strerror(error_code)));
	    return false;
	}
    }
    if (fd >= 0 && ::close(fd) != ok) {
	error_code = errno;
	TRACE(("file::close: failed to close file: %s\n",
	       strerror(error_code)));
	return false;
    }
    if (mode == shadow_pages_transaction) { 
	if (::close(log) != ok) { 
	    TRACE(("file::close: failed to close transaction log file: %s\n", 
		   strerror(errno)));
	}
	if (unlink(name) != ok) {
	    TRACE(("file::close: failed to remove transaction log file: %s\n", 
		   strerror(errno)));
	}
    }
    error_code = ok;
    return true;	
}

#elif defined(_WIN32)


static LONG WINAPI AccessViolationHandler(LPEXCEPTION_POINTERS ep)
{
    return ep->ExceptionRecord->ExceptionCode == EXCEPTION_ACCESS_VIOLATION
	&& file::handle_page_modification
	         ((void*)ep->ExceptionRecord->ExceptionInformation[1])
	? EXCEPTION_CONTINUE_EXECUTION
	: EXCEPTION_CONTINUE_SEARCH;
}


file::file(const char* name, size_t max_file_size, size_t max_locked_pages)
{
    SYSTEM_INFO sysinfo;
    MEMORYSTATUS memstat;

    OSVERSIONINFO osinfo;
    osinfo.dwOSVersionInfoSize = sizeof osinfo;
    GetVersionEx(&osinfo);
    platform = osinfo.dwPlatformId;
    
    GetSystemInfo(&sysinfo);
    allocation_granularity = sysinfo.dwAllocationGranularity;
    page_size = sysinfo.dwPageSize;

    if (platform == VER_PLATFORM_WIN32_NT) {     
	DWORD MinimumWorkingSetSize, MaximumWorkingSetSize;
	GetProcessWorkingSetSize(GetCurrentProcess(),
				 &MinimumWorkingSetSize, 
				 &MaximumWorkingSetSize);

	if (max_locked_pages > MinimumWorkingSetSize/page_size) { 
	    MinimumWorkingSetSize = max_locked_pages*2*page_size;
	    if (MaximumWorkingSetSize < MinimumWorkingSetSize) { 
		MaximumWorkingSetSize = MinimumWorkingSetSize;
	    }
	    if (!SetProcessWorkingSetSize(GetCurrentProcess(),
					  MinimumWorkingSetSize,
					  MaximumWorkingSetSize)) 
	    {
		const size_t max_nt_locked_pages = 30;
		TRACE(("file::file: failed to extend process working set size "
		       "to %ld bytes, set max_locked_pages to %ld\n", 
		       MinimumWorkingSetSize, max_nt_locked_pages));
		max_locked_pages = max_nt_locked_pages;
	    } else { 
		TRACE(("file::file: extend process working set size "
		       "to %ld bytes\n", MinimumWorkingSetSize));
	    }
	}
    } else { 
	max_locked_pages = 0;
    }
    memstat.dwLength = sizeof(memstat);
    GlobalMemoryStatus(&memstat);
    if (memstat.dwAvailVirtual < max_file_size) { 
	max_file_size = memstat.dwAvailVirtual & ~(allocation_granularity-1);
	TRACE(("file::file: set max_file_size to %ld bytes\n",max_file_size));
    }

    this->name = new char[strlen(name)+2];
    strcpy(this->name+1, name);
    this->max_file_size = ALIGN(max_file_size, allocation_granularity);
    this->max_locked_pages = max_locked_pages;

    log_buffer = new char[page_size+sizeof(long)];
    locked_page = new char*[max_locked_pages];
    
    SetUnhandledExceptionFilter(AccessViolationHandler);
    error_code = ok;
}

file::~file() 
{
    delete[] name;
    delete[] log_buffer;
    delete[] locked_page;
}


inline bool file::recover_file()
{
    DWORD read_bytes;
    size_t trans_size = page_size + sizeof(long);
    int n_recovered_pages = 0;
    TRACE(("file::recover_file: recover data file from transaction log\n"));
    while (ReadFile(log, log_buffer, trans_size, &read_bytes, NULL)) { 
	if (read_bytes != trans_size) { 
	    if (read_bytes == 0) { 
		size = ALIGN(((file_header*)base)->file_size, page_size);
		TRACE(("recover::file: recover %d pages, set file size "
		       "to %ld\n", n_recovered_pages, size));
		error_code = ok;
		return true;	    
	    } else { 
		error_code = end_of_file;
		TRACE(("file::recover_file: read %ld bytes from log instead "
		       "of %ld\n", read_bytes, trans_size));
		return false;
	    }
	}    
	size_t offs = *(long*)log_buffer;
	assert(offs < size);
	memcpy(base + offs, log_buffer+sizeof(long), page_size);
	n_recovered_pages += 1;
    }
    error_code = GetLastError();
    TRACE(("file::recover_file: failed to read log file: %d\n", 
	   error_code));
    return false;
}

inline bool file::flush_log_buffer()
{
    if (!FlushFileBuffers(log)) { 
	error_code = GetLastError();
	TRACE(("file::flush_log_buffer: FlushFileBuffers failed: %d\n", 
	       error_code));
	return false;
    }
    for (int i = n_locked_pages; --i >= 0;) { 
	if (!VirtualUnlock(locked_page[i], page_size)) { 
	    error_code = GetLastError();
	    TRACE(("file::flush_log_buffer: VirtualUnlock %p failed: %ld\n",
		   locked_page[i], error_code));
	    return false;
	}
    }
    n_locked_pages = 0;
    return true;
}


bool file::create_shadow_page(void* addr)
{
    DWORD old_prot;
    DWORD written_bytes;
    char* page = (char*)(long(addr) & ~(page_size-1));
    if (!VirtualProtect(page, page_size, PAGE_READWRITE, &old_prot)) { 
	TRACE(("file::create_shadow_page: VirtualProtect %p failed: %ld\n", 
	       page, GetLastError()));
	return false;
    }
    *(long*)log_buffer = page - base;
    memcpy(log_buffer+sizeof(long), page, page_size);
    size_t trans_size = page_size + sizeof(long);
    if (!WriteFile(log, log_buffer, trans_size, &written_bytes, NULL)
	|| written_bytes != trans_size)
    {
	TRACE(("file::create_shadow_page: WriteFile failed: %d\n", 
	       GetLastError()));
	return false;
    }
    if (platform != VER_PLATFORM_WIN32_NT) {     
	unsigned page_no = (page - base) / page_size;
	dirty_page_map[page_no >> 5] |= 1 << (page_no & 0x1F);
    } else if (max_locked_pages != 0) { 
	if (n_locked_pages >= max_locked_pages) { 
	    return flush_log_buffer();
	} 
	if (!VirtualLock(page, page_size)) { 
	    TRACE(("file::create_shadow_page: VirtualLock %p failed,"
		   "number of locked pages %ld: %ld\n",
		   page, n_locked_pages, GetLastError()));
	    if (n_locked_pages != 0) { 
		max_locked_pages = n_locked_pages;
	    }
	    return flush_log_buffer();
	} 
	locked_page[n_locked_pages++] = page;
    }
    return true;
}


bool file::read_file_in_memory()
{
#if DEBUG >= DEBUG_TRACE
    msgbuf buf;
#endif
    DWORD read_bytes;
    vmem = (char*)VirtualAlloc(base, mapped_size, 
			       MEM_RESERVE, PAGE_READWRITE);
    if (vmem == NULL) { 
	error_code = GetLastError();
	TRACE(("file::read_file_in_memory: failed to virtual alloc at address"
	       " %p: %s\n", base, get_error_text(buf, sizeof buf)));
	vmem = (char*)VirtualAlloc(NULL, mapped_size, 
				   MEM_RESERVE, PAGE_READWRITE);
    }
    if (vmem == NULL) { 
	error_code = GetLastError();
	TRACE(("file::read_file_in_memory: failed to virtual alloc: %s\n", 
	       get_error_text(buf, sizeof buf)));
	CloseHandle(fd);
	return false;
    }
    base = vmem;
    if (size != 0) { 
	TRACE(("file::read_file_in_memory: virtual alloc: address=%p, "
	       "mapped_size=%ld, size=%ld\n", vmem, mapped_size, size));
	if (base != (char*)VirtualAlloc(base, size, MEM_COMMIT, 
					PAGE_READWRITE) 
	    || SetFilePointer(fd, 0, NULL, FILE_BEGIN) != 0
	    || ReadFile(fd, base, size, &read_bytes, NULL) == false)
	{
	    error_code = GetLastError();
	    TRACE(("file::read_file_in_memory: failed to read file in memory:"
		   " %s\n", get_error_text(buf, sizeof buf)));
	    CloseHandle(fd);
	    VirtualFree(vmem, 0, MEM_RELEASE);
	    return false;
	} else if (read_bytes != size) { 
	    error_code = file_size_not_aligned;
	    TRACE(("file::read_file_in_memory:read %ld bytes instead of %ld\n",
		   read_bytes, size));
	    CloseHandle(fd);
	    VirtualFree(vmem, 0, MEM_RELEASE);
	    return false;
	}	    
    }
    return true;
}

bool file::write_dirty_pages_in_file()
{
    size_t offs = 0; 
    unsigned page_no = 0;
    size_t updated_size = 0;
    for (page_no = 0; offs < size; page_no += 1) { 
	int mask = 1 << (page_no & 0x1F);
	if (dirty_page_map[page_no >> 5] & mask) {  
	    dirty_page_map[page_no >> 5] &= ~mask;
	    DWORD written_bytes;
	    if (SetFilePointer(fd, offs, NULL, FILE_BEGIN) != offs
		|| !WriteFile(fd, base+offs, page_size, &written_bytes, 0)
		|| written_bytes != page_size)
	    {
		error_code = GetLastError();
		TRACE(("file::write_dirty_pages_in_file: failed to write"
		       " page in file: %d\n", error_code));
		return false;
	    }
	}
	offs += page_size;
    }
    if (!FlushFileBuffers(fd)) { 
	error_code = GetLastError();
	TRACE(("file::write_dirty_pages_in_file: FlushFileBuffers failed:%d\n",
	       error_code));
	return false;
    }
    return true;
}

bool file::open(open_mode mode, access_prot prot) 
{
#if DEBUG >= DEBUG_TRACE
    msgbuf buf;
#endif
    DWORD old_prot;
    assert(name != NULL);
    if (prot == read_only) { 
	mode = copy_on_write_map;
    } else if (mode == copy_on_write_map && platform != VER_PLATFORM_WIN32_NT){
	mode = load_in_memory; // copy on write not working in Windows 95
    }

    this->mode = mode;
    this->prot = prot;
    vmem = NULL;
    md = NULL;

    if (mode == shadow_pages_transaction) { 
	int flags = (platform == VER_PLATFORM_WIN32_NT)
	    ? FILE_FLAG_WRITE_THROUGH|FILE_FLAG_RANDOM_ACCESS
	    : FILE_FLAG_RANDOM_ACCESS;
	fd = CreateFile(name+1, GENERIC_READ|GENERIC_WRITE, 0, 
			NULL, OPEN_ALWAYS, flags, NULL);
	if (fd == INVALID_HANDLE_VALUE) { 
	    error_code = GetLastError();
	    TRACE(("file::open: failed to open file '%s': %s\n", 
		   name+1, get_error_text(buf, sizeof buf)));
	    return false;
	}
	DWORD read_bytes;
	file_header hdr;
	hdr.base_address = NULL;
	hdr.file_size = 0;
        if (!ReadFile(fd, &hdr, sizeof hdr, &read_bytes, NULL)
	    || (read_bytes != 0 && read_bytes != sizeof hdr))
	{
	    error_code = GetLastError();
	    TRACE(("file::open: failed to read file: %s\n", 
		   get_error_text(buf, sizeof buf)));
	    CloseHandle(fd);
	    return false;
	}
	base = (char*)hdr.base_address;
	allocated_size = size = ALIGN(hdr.file_size, page_size);
	mapped_size = size > max_file_size ? size : max_file_size;
	if (platform == VER_PLATFORM_WIN32_NT) {     
	    md = CreateFileMapping(fd,NULL,PAGE_READWRITE,0,mapped_size,NULL);
	    if (md == NULL) { 
		error_code = GetLastError();
		TRACE(("file::open: failed to create file mapping: base=%p, "
		       "size=%ld: %s\n", base, mapped_size, 
		       get_error_text(buf, sizeof buf)));
		CloseHandle(fd);
		return false;
	    }
	    void* p = MapViewOfFileEx(md, FILE_MAP_ALL_ACCESS, 0, 0, 0, base);
	    if (p == NULL) { 
		error_code = GetLastError();
		TRACE(("file::open: failed to map view of file on address "
		       "%p: %s\n", base, get_error_text(buf, sizeof buf)));
		p = MapViewOfFileEx(md, FILE_MAP_ALL_ACCESS, 0, 0, 0, NULL);
		if (p == NULL) { 
		    error_code = GetLastError();
		    TRACE(("file::open: failed to map view of file: %s\n",
			   get_error_text(buf, sizeof buf)));
		    CloseHandle(fd);
		    CloseHandle(md);
		    return false;
		}
	    }
	    base = (char*)p;
	    dirty_page_map = NULL;
	} else { 
	    //
	    // VirtualProtect doesn't work in Winfows 95 with 
	    // mapped on file memory. We have to use VirtualAlloc instead 
	    // and read file in memory.
	    //
	    if (!read_file_in_memory()) { 
		return false;
	    }
	    size_t page_map_size = ((mapped_size / page_size) + 31) >> 5;
	    dirty_page_map = new int[page_map_size];
	    memset(dirty_page_map, 0, sizeof(int)*page_map_size);
	}
	name[0] = '.'; // transaction log file name prefix
	n_locked_pages = 0;
	if (max_locked_pages != 0) {
	    // check if we can use VirtualLock()
	    if (VirtualLock(base, page_size)) { 
		VirtualUnlock(base, page_size);
	    } else { 
		error_code = GetLastError();
		TRACE(("file::open: VirtualLock test failed: %s\n",
		       get_error_text(buf, sizeof buf)));
		max_locked_pages = 0;
	    }		
	}
	int log_flags = FILE_FLAG_SEQUENTIAL_SCAN;
	if (max_locked_pages == 0 && platform == VER_PLATFORM_WIN32_NT) { 
            log_flags |= FILE_FLAG_WRITE_THROUGH;
	}
	log = CreateFile(name, GENERIC_READ|GENERIC_WRITE, 0, NULL, 
			 OPEN_EXISTING, log_flags, NULL); 
	if (log == INVALID_HANDLE_VALUE) { 
	    if (GetLastError() == ERROR_FILE_NOT_FOUND) {
		log = CreateFile(name, GENERIC_READ|GENERIC_WRITE, 0, NULL, 
				 CREATE_ALWAYS, log_flags, NULL); 
	    }
	    if (log == INVALID_HANDLE_VALUE) { 
		error_code = GetLastError();
		TRACE(("file::open: failed to create log file '%': %s\n", 
		       name, get_error_text(buf, sizeof buf)));
	      return_error:
		if (md != NULL) { 
		    UnmapViewOfFile(base);
		    CloseHandle(md);
		} else { 
		    VirtualFree(base, 0, MEM_RELEASE);
		}
		CloseHandle(fd);
		CloseHandle(log);
		delete[] dirty_page_map;
		return false;
	    }
	    if (size != 0 
		&& !VirtualProtect(base, size, PAGE_READONLY, &old_prot)) 
            {
		error_code = GetLastError();
		TRACE(("file::open: VirtualProtect failed for base=%p, "
		       "size=%ld: %s\n", 
		       base, size, get_error_text(buf, sizeof buf)));
		goto return_error;
	    }
	} else { 
	    recover_file();	
	    if (!commit()) { 
		goto return_error;
	    }
	}
	next = chain;
	chain = this;
    } else { // non-shadow_pages_transaction mode
	fd = CreateFile(name+1, GENERIC_READ, FILE_SHARE_READ, 
			NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL); 
	size = 0;
	if (fd != INVALID_HANDLE_VALUE) { 
	    DWORD read_bytes;
	    if (!ReadFile(fd, &base, sizeof base, &read_bytes, NULL)
		|| (read_bytes != 0 && read_bytes != sizeof base))
	    {
		error_code = GetLastError();
		TRACE(("file::open: failed to read file '%s': %s\n", 
		       name+1, get_error_text(buf, sizeof buf)));
		CloseHandle(fd);
		return false;
	    }
	    size = GetFileSize(fd, NULL);
	    if ((size & (allocation_granularity-1)) != 0
		&& prot != read_only 
		&& mode == copy_on_write_map 
		&& size < max_file_size)
	     {
		error_code = file_size_not_aligned;
		TRACE(("file::open: size of file '%s' is not aligned on %ld\n",
		       name+1, allocation_granularity));
		return false;
	    }
	    if (mode == copy_on_write_map || size >= max_file_size) { 
		mapped_size = size;
		md = CreateFileMapping(fd, NULL, PAGE_WRITECOPY, 0, size,NULL);
		if (md == NULL) { 
		    error_code = GetLastError();
		    TRACE(("file::open: failed to create file mapping: %s\n",
			   get_error_text(buf, sizeof buf)));
		    CloseHandle(fd);
		    return false;
		}
		TRACE(("file::open: create file mapping: size=%ld\n", size));
		void* p = MapViewOfFileEx(md, prot == read_only  
					  ? FILE_MAP_READ : FILE_MAP_COPY, 
					  0, 0, size, base);
		if (p == NULL) { 
		    error_code = GetLastError();
		    TRACE(("file::open: failed to map view of file on address "
			   "%p: %s\n", base, get_error_text(buf, sizeof buf)));
		    p = MapViewOfFileEx(md, prot == read_only  
					? FILE_MAP_READ : FILE_MAP_COPY,
					0, 0, size, NULL);
		}
		if (p == NULL) { 
		    error_code = GetLastError();
		    TRACE(("file::open: failed to map view of file: %s\n", 
			   get_error_text(buf, sizeof buf)));
		    CloseHandle(md);
		    CloseHandle(fd);
		    return false;
		}
		base = (char*)p;
		TRACE(("file::open: map view of file on %p\n", base));
		if (prot != read_only && size < max_file_size) { 
		    mapped_size = max_file_size;
		    vmem = (char*)VirtualAlloc(base + size, mapped_size - size,
					       MEM_RESERVE, PAGE_READWRITE);
		    if (vmem == NULL) { 
			error_code = GetLastError();
			TRACE(("file::open: failed to allocate end of region:"
			       "base=%p, size=%ld: %s\n", 
			       base+size, mapped_size-size, 
			       get_error_text(buf, sizeof(buf))));
			
			UnmapViewOfFile(p);
			CloseHandle(md);
			CloseHandle(fd);
			return false;
		    }
		    TRACE(("file::open: virtual alloc: address=%p, size=%ld\n",
			   vmem, mapped_size));
		    assert(vmem == base + size);
		}  
	    } else { // read file to memory
		mapped_size = max_file_size;
		if (!read_file_in_memory()) { 
		    return false;
		}
		CloseHandle(fd);
		fd = INVALID_HANDLE_VALUE;
	    }
	} else { 
	    error_code = GetLastError();
	    if (error_code != ERROR_FILE_NOT_FOUND) { 
		TRACE(("file::open: failed to open file '%s': %s\n", 
		       name+1, get_error_text(buf, sizeof buf)));
		return false;
	    }
	    TRACE(("file::open: file '%s' not found\n", name+1));
	    if (prot == read_only) {
		TRACE(("file::open: failed to open in read only mode unexisted"
		       " file '%s'\n", name+1));
		return false;
	    }
	    mapped_size = max_file_size;
	    base = vmem = (char*)VirtualAlloc(NULL, mapped_size, 
					      MEM_RESERVE, PAGE_READWRITE);
	    if (base == NULL) {
		error_code = GetLastError();
		TRACE(("file::open: failed to virtual alloc: %s\n", 
		       get_error_text(buf, sizeof buf)));
		return false;
	    }
	}
    }
    error_code = ok;
    return true;
}

bool file::set_size(size_t new_size)
{
    if (new_size > mapped_size) { 
	error_code = file_mapping_size_exceeded;
	return false;
    } 
    if (mode == shadow_pages_transaction) {
	new_size = ALIGN(new_size, page_size);
	if (platform != VER_PLATFORM_WIN32_NT) { // Windows 95, uhh...
	    if (new_size > allocated_size) { 
		allocated_size = ALIGN(new_size, allocation_granularity);
		if (!VirtualAlloc(base+size, allocated_size-size, MEM_COMMIT,
				  PAGE_READWRITE))
		{
		    error_code = GetLastError();
		    return false;
		}
		unsigned page_no = size / page_size;
		size_t offs = size; 
		while (offs < allocated_size) {
		    dirty_page_map[page_no >> 5] |= 1 << (page_no & 0x1F);
		    offs += page_size;
		    page_no += 1;
		} 
	    }
	}
    } else { 
	new_size = ALIGN(new_size, allocation_granularity);
	if (new_size > size) { 
	    if (!VirtualAlloc(base+size, new_size-size, MEM_COMMIT,
			      PAGE_READWRITE))
	    {
		error_code = GetLastError();
		return false;
	    }
	}
    }
    size = new_size;
    error_code = ok;
    return true;
}

bool file::set_protection(access_prot prot)
{
    DWORD old_prot;
    if (!VirtualProtect(base, mapped_size, 
			prot == PAGE_READONLY ? PAGE_READONLY : PAGE_WRITECOPY,
			&old_prot))
    {
	error_code = GetLastError();
	return false;
    }
    this->prot = prot;
    error_code = ok;
    return true;
}


bool file::commit()
{
    DWORD old_prot;
    if (mode != shadow_pages_transaction) { 
	error_code = not_in_transaction;
	return false;
    }
    if (n_locked_pages != 0 || platform != VER_PLATFORM_WIN32_NT) { 
	if (!flush_log_buffer()) { 
	    return false;
	}
    }
    if (platform == VER_PLATFORM_WIN32_NT) { 
	if (size > 0 && !FlushViewOfFile(base, size)) { 
	    error_code = GetLastError();
	    TRACE(("file::commit: FlushViewOfFile(%p, %ld) failed: %d\n",
		   base, size, error_code));
	    return false;
	}
    } else { 
	if (!write_dirty_pages_in_file()) { 
	    return false;
	}
    } 
    if (SetFilePointer(log, 0, NULL, FILE_BEGIN) != 0 || !SetEndOfFile(log)) { 
	error_code = GetLastError();
	TRACE(("file::commit: failed to truncate lof file: %d\n", error_code));
	return false;
    }
    if (size > 0 && !VirtualProtect(base, size, PAGE_READONLY, &old_prot)){
	error_code = GetLastError();
	TRACE(("file::commit: VirtualProtect(%p, %ld) failed: %d\n",
	       base, size, error_code));
	return false;
    }
    error_code = ok;
    return true;
}

bool file::rollback()
{
    if (mode != shadow_pages_transaction) { 
	error_code = not_in_transaction;
	return false;
    }
    SetFilePointer(log, 0, NULL, FILE_BEGIN);
    return recover_file();
}

bool file::flush() 
{
    if (prot == read_only) {
	return true;
    }
    if (mode == shadow_pages_transaction) { 
	return commit();
    } else { // non-shadow_pages_transaction mode
	name[0] = '~'; // temporary file prefix
	HANDLE tmp_fd = CreateFile(name, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS,
				   FILE_FLAG_SEQUENTIAL_SCAN, NULL);
	if (tmp_fd == INVALID_HANDLE_VALUE) { 
	    error_code = GetLastError();
	    return false;
	}
	DWORD written_bytes;
	if (!WriteFile(tmp_fd, base, size, &written_bytes, NULL)
	    || written_bytes != size)
	{
	    error_code = GetLastError();
	    TRACE(("file::flush: failed to write %ld bytes to file '%s': %d\n",
		   size, name, error_code));
	    CloseHandle(tmp_fd);
	    return false;
	}
	CloseHandle(tmp_fd);
	if (platform == VER_PLATFORM_WIN32_NT) { 
	    if (fd != INVALID_HANDLE_VALUE) { 
		//
		// NT bug ??? By some reasons closing fd can cause loose of 
		// data at some pages mapped to this file after MoveFileEx.
		// The following loop modify all pages causing copy on write
		// mechanism to create copies of all pages.
		//
		TRACE(("file::flush: force copy on write page creation\n"));
		for (char* p = base; p < vmem; p += page_size) { 
		    *(volatile int*)p = *(volatile int*)p;
		}
		CloseHandle(fd);
		fd = INVALID_HANDLE_VALUE;
	    }
	    if (!MoveFileEx(name, name+1, MOVEFILE_REPLACE_EXISTING|
			    MOVEFILE_WRITE_THROUGH) != 0)
	    { 
		error_code = GetLastError();
		return false;
	    }
	} else { // windows 95
	    if (fd != INVALID_HANDLE_VALUE) { 
		CloseHandle(fd);
		fd = INVALID_HANDLE_VALUE;
	    }
	    name[0] = '#'; // backup copy
	    bool backup = MoveFile(name+1, name); // may be exists, may be not
	    if (!backup && GetLastError() == ERROR_ALREADY_EXISTS) {
		DeleteFile(name); // remove previous backup file
		backup = MoveFile(name+1, name);
		if (!backup) { 
		    error_code = GetLastError();
		    TRACE(("file::flush: failed to make backup copy of file"
			   " '%s'\n",  name+1));
		    return false;
		}
	    }
	    name[0] = '~';
	    if (!MoveFile(name, name+1)) { 
		error_code = GetLastError();
		TRACE(("file::flush: failed to rename file '%s' to '%s'\n", 
		       name, name+1));
		return false;
	    }
	    if (backup) { 
		name[0] = '#'; 
		DeleteFile(name); 
	    }
	}
	error_code = ok;
	return true;
    }
}

bool file::close()
{
    if (mode == shadow_pages_transaction) { 
	file *fp, **fpp = &chain;
	while ((fp = *fpp) != this) { 
	    fpp = &fp->next;
	}
	*fpp = fp->next;
	    
	if (n_locked_pages != 0 || platform != VER_PLATFORM_WIN32_NT) { 
	    if (!flush_log_buffer()) { 
		return false;
	    }
	}
	if (platform != VER_PLATFORM_WIN32_NT) { 
	    if (!write_dirty_pages_in_file()) {
		return false;
	    }
	    delete[] dirty_page_map;
	}
    }
    if (vmem != NULL) { 
	if (!VirtualFree(vmem, 0, MEM_RELEASE)) { 
	    error_code = GetLastError();
	    TRACE(("file::close: failed to free %p: %d\n", vmem, error_code));
	    return false;
	}
    }
    if (md != NULL) { 
        if (!UnmapViewOfFile(base)) { 
	    error_code = GetLastError();
	    TRACE(("file::close: failed to unmap %p: %d\n", 
		   base, error_code));
	    return false;
	}
	if (!CloseHandle(md)) { 
	    error_code = GetLastError();
	    TRACE(("file::close: failed to close file mapping: %d\n",
		   error_code));
	    return false;
	}	    
    }
    if (mode == shadow_pages_transaction && platform == VER_PLATFORM_WIN32_NT){
	if (SetFilePointer(fd, size, NULL, FILE_BEGIN) != size ||
	    !SetEndOfFile(fd)) 
	{ 
	    error_code = GetLastError();
	    TRACE(("file::close: failed to change size of file to %ld: %d\n", 
		   size, error_code));
	    return false;
	}	    	    
    }
    if (fd != INVALID_HANDLE_VALUE) { 
	if (!CloseHandle(fd)) { 	    
 	    error_code = GetLastError();
	    TRACE(("file::close: failed to close file: %d\n", error_code));
	    return false;
	}
    }
    if (mode == shadow_pages_transaction) { 
	if (!CloseHandle(log)) { 
	    TRACE(("file::close: CloseHandle for log failed: %d\n", 
		   GetLastError()));
	}
	if (!DeleteFile(name)) { 
	    TRACE(("file::close: failed to remove log file: %d\n", 
		   GetLastError()));
	}
    }
    error_code = ok;
    return true;
}

char* file::get_error_text(char* buf, size_t buf_size)
{
    char* err_txt;
    char errbuf[64];

    switch (error_code) {
      case ok: 
	err_txt = "no error";
	break;
      case file_size_not_aligned:
	err_txt = "size of file is not aligned on segment boundary";
	break;
      case file_mapping_size_exceeded:
	err_txt = "attempt to exeed the limit on mapped object size";
	break;
      case not_in_transaction:
	err_txt = "operation is possible only inside transaction"; 
	break;
      case end_of_file:
	err_txt = "unexpected end of file"; 
	break;
      default: 
	if (FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
			  NULL, error_code, 0,
			  buf, buf_size, NULL) > 0)
	{
	    return buf;
	} else { 
	    sprintf(errbuf, "unknown error code %u", error_code);
	    err_txt = errbuf;
	}
    }
    return strncpy(buf, err_txt, buf_size);
}

#endif

#if DEBUG >= DEBUG_CHECK
//
// Sometimes it is more convenient to catch SIGSEGV instead of SIGABRT. 
// For example in Digital Unix debugger failed to unroll stack after
// assertion failure. And in Windows NT assertion failure will not cause
// invocation of debugger if program is not started from MSDEV. That is
// why this "strange" version of abort() was implemented. 
//
extern "C" 
void abort() { while(1) *(int*)0 = 0; /* do not return */ }
#endif


