/*
    Program Name:   Cipher.C
    Author:         Curtis Little

    Copyright (c) 1990, By Curtis Little
    ------------------------- ALL RIGHTS RESERVED ------------------------
    Created:  10/31/1990 at 10:06 am

    Compiler: ------------------------------------------------------------
    Turbo C 2.0

    Description: ---------------------------------------------------------
    This program is a tool that automates the creation of tables to
    perform file encryption/decryption with.  This is a sample program
    distributed with Turbo SNR to help generate tables that demonstrate
    the use of multiple contexts.

    A list of substitution strings will be listed to standard out (which
    can be redirected to any file you desire) that can be used both to
    encrypt a file and decrypt a file.  When you redirect output to a file
    you will need to go in the file and manually write out the decryption
    table portion of the file.  (You can search for the word 'Decryption'
    to locate the start of the decryption table portion).

    The file created has the following format:

        Header to indicate the start of the encryption portion of strings
        Encryption strings
        Header to indicate the start of the decryption portion of strings
        Decryption strings

    You can specify how many context settings to use (1-10) by including
    a parameter on the command line.  For example to use 4 contexts and
    direct output to a file called MYFILE.TAB you would use the following
    command line:

        cipher 4 > myfile.tab

    At default CIPHER uses three contexts when you don't specify the
    number.

    Modification History: ------------------------------------------------

    Revision:  1.0  Last Revised:  10/31/1990 at 10:06 by CBL
    Description:  Original Creation.
    ----------------------------------------------------------------------
*/


#include <stdio.h>
#include <time.h>
#include <stdlib.h>

#define MAX_CONTEXT 10      /* maximum number of contexts allowd */
#define MAX_TRIES   256     /* maximum attempts to try to get a unique */
                            /* random number before using a linear search */
#define MAX_BYTE_VAL 255    /* maximum number a single character can hold */


int *uses[MAX_CONTEXT];     /* pointers to the table entries we used so far */


/*
    findone()

    This function uses random numbers to pick the character to replace
    with.  The original character we're replacing is put in the uses[x]
    array so we know which characters we transposed.

    Returns:    The character to substitute with.
*/
int findone( int use[], int c )
{
    register int i, j=0;

    /* loop to try and generate a random number that hasn't already been */
    /* used in the current context */
    while (j++ < MAX_TRIES) {
        i=random(256);
        if (use[i]==-1) {            /* is the character already used? */
            use[i] = c;             /* if not we'll use it now */
            return(i);
        }
    }

    /* search up to the end character now... */
    while (++i < 256) {
        if (use[i]==-1) {
            use[i]=c;
            return(i);
        }
    }

    /* search down now... */
    while (i-- > 0)
        if (use[i]==-1) {
            use[i]=c;
            return(i);
        }

    /* we have an internal error if execution gets here */
    printf( "\n\a\aError:  Can't map to a unique character!  Program halted.\n\n" );
    exit(1);
}

/*
    fixchar()

    This function creates a string out of the specified character.  This
    is used so we can add the leading backslash for special characters as
    well as support the NULL character.

    The buffer must be at least 5 bytes long.

    Returns:  None.
*/
void fixchar( char *buf, int c )
{

    buf[1]=0;               /* set up the most common case */

    switch (c) {
    case ':': case ';': case '\\': case '=': case '"':
        sprintf( buf, "\\%c", c );
        break;
    default:
        if (c < 32 || c > 126)
            sprintf( buf, "\\%d", c );
        else
            buf[0] = c;
    }
}

void usage( char *message )
{
    fprintf( stderr, "Error:  %s\n", message );
    fprintf( stderr, "Calling Syntax: CIPHER [num_contexts]\n" );
    exit(1);
}
int main( int argc, char *argv[] )
{
    register int i, j;
    int k, l;
    int contexts=3;           /* default number of contexts */
    char buf[5];
    char buf1[5];
    extern void *malloc( unsigned size );

    fprintf( stderr, "CIPHER - Copyright (C)1990, By Curtis Little   All Rights Reserved.\n" );
    fprintf( stderr, "Create CIPHER/DECIPHER tables for use by Turbo SNR.\n\n" );

    /* make sure the command line looks ok. */
    if (argc > 2) {
        usage( "Too many parameters specified!" );
    }

    if (argc == 2) {
        contexts = atoi(argv[1]);
        if (contexts < 1 || contexts > MAX_CONTEXT)
            usage( "Invalid number of contexts specified!" );
    }

    /* seed the random number generator using the system clock */
    randomize();

    /* loop to allocate the required array for each context */
    for (i=0; i < contexts; i++) {
        if ((uses[i] = (int *) malloc(sizeof(int)*(MAX_BYTE_VAL+1)))==NULL)
            usage( "Not enough memory!" );
    }

    /* now loop to initialize all the table elements */
    for (i=0; i < contexts; i++) {
        for (j=0; j < MAX_BYTE_VAL + 1; j++)
            uses[i][j] = -1;
    }

    /* now output the encryption table */
    printf( ";\n; File Encryption Table for Turbo SNR.\n" );
    printf(    "; ------------------------------------\n;\n\n\n" );

    /* now loop to create the tables as required */
    for (i=0; i < contexts; i++ ) {
        for (j=0; j < MAX_BYTE_VAL + 1; j++) {
            fixchar(buf, j);
            fixchar( buf1, findone(uses[i], j));
            printf( "\"%d:%s=%d:%s\"\n", i, buf,
                i == contexts - 1 ? 0 : i + 1, buf1 );
        }
    }

    /* now output the "cure" */
    printf( "\n\n;-------------------------------------------------------------\n" );
    printf( ";\n; File Decryption Table for Turbo SNR.\n" );
    printf(    "; ------------------------------------\n;\n\n\n" );

    for (i=0; i < contexts; i++ ) {
        for (j=0; j < MAX_BYTE_VAL + 1; j++) {
            fixchar( buf, j );
            fixchar( buf1, uses[i][j] );
            printf( "\"%d:%s=%d:%s\"\n", i, buf, 
                i == contexts - 1 ? 0 : i + 1, buf1 );
        }
    }
	return(0);
}
