/**************************************************************************
           TSGREP ... A regular expression line extractor, v5.0
 **************************************************************************/

/*
Known Bugs:
               -- none --

Changes:

     To support the 'include context' option, I have made
     <alt N> and <alt P> go to the next/previous hit either
     in the hit list or in a file that was searched.

     In the "hit list," <ctrl N> and <ctrl P> now go to the beginning of
     the listing for the next/previous file.

     \_ only functions (and is only needed) in command line mode.

7-25-93

     When loading files from a list file, TSGREP
     nows displays the list file name.

See TSGREP.DOC for other changes.

*/

/**************************************************************************
          Forward Proc Declarations
 **************************************************************************/
forward INTEGER proc          ask_input()         // Displays help; asks inputs
forward proc                  check_for_abort()   // aborts between file loads
                                                  // if <escape>
forward proc                  end_processing      // cleanup AND view
     (INTEGER MODE)
forward proc                  goto_hit_list(INTEGER MODE)   // goes to outfile
forward proc                  goto_hit()          // goes to another file from outfile
forward proc                  hilite_filename()   // highlights in out file
forward proc                  interactive_setUp() // Setup for interactive operation
forward proc                  help_display()      // show legal keystrokes
forward help                  keystrokes
forward proc                  load_recursively    // loads files from all sub-
     (STRING arg)                                 // directories below current
forward proc                  load_file_list()    // loads list of files from  @file
forward proc                  make_other_hit_lists()   // for use by toggle_hists
forward INTEGER proc          multiple_filespecs  // loads files separated by ^
    (VAR STRING filespec)
forward proc                  NextFile_if_NOT_excluded()    // loads file if NOT on excluded extension list
forward proc                  reload_files()      // interactive mode : reload files
forward proc                  process_and()       // processes -and- in search STRING
forward proc                  remove_file_from_hit_List()   // removes a file & hits
forward STRING proc           repl_char           // replaces chars in a STRING
    (STRING find_string,
     STRING repl_string,
     VAR STRING target)
forward proc                  toggle_hits()       //  toggles display of hit lines
forward proc                  undo_all_removes () // undoes removes
forward proc                  unload_TSGREP()     // unloads TSGREP.

/**************************************************************************
          Variable & CONSTANT Declarations
 **************************************************************************/
CONSTANT
     INTERACTIVE = 1,
     CMDLINE = 0,
     MAXFN = 60,                        // :U: largest file name TSGREP can handle
     MAXSPEC = 120,                     // :U: largest input file spec
     ALWAYS = 1,                        // used by save_hit_list
     NEVER = 2,                         // no save unless you edit file
     ASKSAVE = 3                         // TSE will ask

INTEGER
     save_hit_list = NEVER,             // :U: controls save of hit list file
     ILBA,                              // holds InsertLineBlocksAbove value
     MSGLVL,                            // holds MsgLevel value
     normal_Attr,                       // Attribute for normal text
     hilite_Attr,                       // Attribute for highlighting
                                        // (change these at the beginning of
                                        // Main(), NOT here.)
     query_saves = TRUE,                // :U: whether to query saves (interactive)
     tsgrep_mode = CMDLINE,               // initial mode
     done_executed_once = FALSE,        // for determining non-interactive
     cid = 0,                           // first file loaded
     oid = 0,                           // buffer to become output file
     dironlyfile,                           // buffer for dirs_only file
     copy_of_list,                      // id for copy of above
     files_only,                        // id for copy with only files
     cfid = 0,                          // id for buffer with current files
     id = 0,                            // id of file being processed
     tid = 0,                           // a temp buffer used for
                                        // searchstring processing
     load_file =  TRUE,                 // load outfile into editor
     linenumbers = TRUE,                // to include (OR NOT)
     hits = 0,                          // hits counter
     write_hits_flag = TRUE,            // write hits in outfile
     no_hitters_flag = FALSE,           // were there any?
     dirfile_only = FALSE,              // produce only list of dirs
     curr_file_hits = 0,                // hits counter
     name_line,                         // line filename placed on
     files = 0,                         // file counter
     lndigits = 5,                      // :U: max digits in a line number
     linenumber = 0,                    // line number of find,
     last_line_gotten = 0,              // last line copied to outfile
     first_line_gotten = 0,             // first line copied to outfile
     auto,                              // original AutoIndent state
     msgs,                              // original msg level state
     wild,                              // original loadwild state
     im  ,                              // original insert state
     km  ,                              // original killbuffmax state
     context = 0,                       // number of lines context
     col = 1,                           // for gotohit
     first_file_line = 12,              // for gotohit
     isloaded = 0,                      // for restoring curr files
     sstring_hist_buff = 0,             // history buffers for Ask()s
     infile_hist_buff = 0,
     options_hist_buff = 0,
     outfile_hist_buff = 0,
     getting_dirs_line =     9,          // display lines
     loading_files_line =    9,
     getting_files_line =    9,
     processing_line =       9,
     files_loaded_line =    10,
     files_processed_line = 10,
     dirs_found_line =      10,
     hits_line =            11,
     press_any_key_line =   12,
     val2 = 0

STRING
     pad[60] = ' ',
     name[MAXFN] = '',                  // used for name of file being processed
     v[3]= '5.0',                       // :U: TSGREP version
     name_prefix[4] = ' ',           // :U: inserted into outfile before name
     name_suffix[1] = ':',              // :U: inserted into outfile after name
     name_prefix_re[SizeOf(name_prefix)*2],
                                        // version of above with, double length
     options[25]  ='',                  // search options used
     options_entered[25]  ='',          // search/TSGREP options entered
     infile[MAXSPEC]  ='',              // filespec for input
     sstring[255] ='',                  // search STRING
     outfile[MAXFN]  ='',               // filespec for output
     default_name[MAXFN] ="TSGREP.TMP", // :U: default for above
     default_name_2[MAXFN] ="",         //
     default_ext_2[4] = ".$MP",         // :U: ext for files-only list
     or_operator[4] = '-or-',           // :U: represents | in searchstring
     and_operator[5] = '-and-',         // :U: represents AND in searchstring
     eq_char[2] = '\-',                 // :U: represents = in searchstring
     space_char[1] = '_',               // :U: represents   in searchstring
     less_than[2] = '[[',               // :U: represents < in searchstring
     greater_than[2] = ']]',            // :U: represents > in searchstring
     one_hit_flag[2] = '\1',            // :U: flag for stop after 1 hit
     dont_write_hits[2] = '\w',         // :U: flag for NOT writing hits in outfile
     no_line_numbers[2] = '\#',         // :U: flag for NOT writing hits in outfile
     dont_decode_spacechar[2] = '\_',   // :U: flag for NOT decoding space_char
     dont_load_outfile[2] = '\l',       // :U: flag for NOT loading output file
     no_warning_flag[2] = '\q',         // :U: flag for NOT showing any TSE msgs
                                        //     *** use above with care ***
     no_re_flag[2] = '\x',              // :U: flag for disable auto regexp mode
     dirfile_flag[2]='\d',              // :U: flag for directory list only
     context_flag[2]='\c',              // :U: flag for specifying lines of context
     files_only_flag[2]='\f',           // :U: flag for file listing only
     file_sep_char[1] = '^',            // :U: file separator character
     recurse_flag[1] = '+' ,            // :U: recursive search indicator
     list_file_flag[1] = '@' ,          // :U: list file flag
     ramdrive[3] = '',                  // :U: format:  'd:\'  OR '\'
     dir_spec[2] = '*.',                // :U: change to '' if you use directories
                                        //     with an extension as part of name
     line[255] = ''                     //  used variously

/**************************************************************************
          KEY DEFINITIONS
 **************************************************************************/

keydef file         // for while in  searched files
     <alt enter>    Disable(file) goto_hit_list(1)
                    BegLine() UpdateDisplay() Find(sstring, options)
     <ctrl enter>   Disable(file) goto_hit_list(1)
                    BegLine() UpdateDisplay() Find(sstring, options)
     <alt escape>   if tsgrep_mode == CMDLINE Exit() endif
     <alt n>        lFind(sstring, options + '+')
     <alt p>        lFind(sstring, options + 'b')
     <alt h>        help_display()
     <alt ->        unload_TSGREP()
end

keydef hit_file     // for while in  the hit list
     <alt enter>    Disable(hit_file) goto_hit()
                    BegLine() UpdateDisplay() Find(sstring, options)
     <ctrl enter>   Disable(hit_file) goto_hit()
                    BegLine() UpdateDisplay() Find(sstring, options)
     <alt escape>   if tsgrep_mode == CMDLINE
                    GotoBufferId(oid) Exit() endif
     <alt r>        remove_file_from_hit_List()
     <alt u>        undo_all_removes()
     <alt f>        toggle_hits()
     <ctrl n>       if lFind(name_prefix, '^+')  hilite_filename() endif
     <ctrl p>       if lFind(name_prefix, '^b') hilite_filename() endif
     <alt n>        lFind(sstring, options + '+') HiLiteFoundText()
     <alt p>        lFind(sstring, options + 'b') HiLiteFoundText()
     <alt h>        help_display()
     <alt ->        unload_TSGREP()
end

/**************************************************************************
          PROCEDURES
 **************************************************************************/

// :U: Choose one of below to set style for input of
// :U: options, search string, etc.:

// #INCLUDE ['tsgrepa1.inc']    // prompt-style entry
   #INCLUDE ['tsgrepa2.inc']    // menu-style entry

/**************************************************************************
          Check for Abort
 **************************************************************************/
proc                          check_for_abort()
     if KeyPressed() AND GetKey() == <escape>
          PushKey(<enter>)              // bypasses 'press any key'
          end_processing(1)
          halt                          // Avoid going back to where
     endif                              // check_for_abort was called
end

/**************************************************************************
          end Processing (including list)
 **************************************************************************/
proc                          end_processing(INTEGER mode)
                                             // mode 1 = interrupted
     GotoBufferId(oid)                       // mode 0 = normal end
     GotoLine(9)  GotoPos(25)
     if mode == 1
          InsertText(Str(files)+']' + name_prefix + 'interrupted' + name_suffix)
     else
          InsertText(Str(files)+']')
     endif
     GotoLine(10) GotoPos(25)
     InsertText(Str(hits)+']')

     EndFile()
     if no_hitters_flag
          repeat
               Up()
          until CurrLineLen() == 0
          CReturn()
          InsertText( SubStr(pad,2,30) + 'Other Files Searched' +
                      SubStr(pad,2,30), _insert_)
     endif
     BegFile()

     if NOT SaveAs(outfile, 1)               // ,1 == overlay
          Message('cannot save as ', outfile)
     endif
     if (tsgrep_mode == CMDLINE) AND (NOT load_file)
               VGotoXY(3,press_any_key_line)
               Set(Attr,hilite_attr)
               PutLine(Format("Press any key to continue": 61), 61)
               Alarm()
               GetKey()
               AbandonEditor()
     else
          Message('Preparing list ... ')
          /*
               Get rid of all files other than the output file
          */
          if NOT GotoBufferId(oid)
             Alarm() Alarm()
             Warn('You started TSGREP with only its own output files loaded.')
             EditFile()
             Return()
          endif
          repeat
               NextFile(_dont_load_)
               if GetBufferID() <> oid
                    AbandonFile()
                    GotoBufferId(oid)
               endif
          until GetBufferID() == oid

          Message('Reloading files ...')

          if tsgrep_mode == INTERACTIVE
             reload_files()
               PopWinClose()              // fullscreen window
          endif

          PopWinClose()                           // output area

          /*
               Reset editor
          */

          Set(InsertLineBlocksAbove, ILBA)
          Set(MsgLevel, MSGLVL)
          Set(LoadWildFromInside, wild)
          Set(AutoIndent,auto)
          Set(Insert,im)
          AbandonFile(cid)
          AbandonFile(oid)
          make_other_hit_lists()    // ****
          goto_hit_list(0)
          SetGlobalInt('Bypass_WhenSwitchToFile', FALSE)
          SetGlobalInt('Bypass_WhenFirstEdited', FALSE)
          Set(KillMax,km)
          Set(MsgLevel, Msgs)
          Set(Cursor,On)
          ShowMouse()

     endif
     done_executed_once = TRUE
     UpdateDisplay()
     hilite_filename()
     Message('Alt+H for help.')
     EraseDiskFile(default_name_2)  /* XXX */
     if save_hit_list <> ALWAYS
          EraseDiskFile(outfile)
     endif
     if save_hit_list == ASKSAVE
          ForceChanged(TRUE)
     endif
end

/**************************************************************************
          Go To Hit
 **************************************************************************/
proc                          goto_hit()
     STRING
          fn[MAXFN] = ''
top:
     if GetBufferID() == files_only
          toggle_hits()
     else
          EditFile(outfile)
     endif
     BegLine()

     if NumLines() == 2
          Warn('No files matched file spec ....')
          AbandonEditor()
     endif

     if CurrLine() < first_file_line            // get to first hit if at top of file
          GotoLine(first_file_line)
     endif

     if (CurrLine() < first_file_line)          // get to first hit if at top of file
          OR (CurrLineLen() == 0)
          GotoLine(first_file_line)
     endif

     if (GetText(1,1) == '') OR (CurrLineLen() == 0)
          repeat
               Down()
          until GetText(1,Length(name_prefix)) == name_prefix OR
                (NumLines() == CurrLine())
     endif

     if NumLines() == CurrLine()
          repeat
               Up()
          until GetText(1,Length(name_prefix)) == name_prefix OR
                (CurrLine() <= first_file_line)
     endif

     col = CurrCol()               // get currcol in case search was used
     BegLine()

     PushPosition()                // remember line

     if GetText(1,Length(name_prefix)) == name_prefix
          linenumber = 0
     else                               // get line number
          linenumber = Val( GetText(1,lndigits))
          while NOT (GetText(1,Length(name_prefix)) == name_prefix)
          AND
                (CurrLine() >= first_file_line)
               Up()
          endwhile                      // get file name if NOT on it
     endif

     fn  =   ( GetText( Length(name_prefix) + 1,
               CurrLineLen() - Length(name_prefix)
               - Length(name_suffix) ) )
     PopPosition()                      // go back to line
     if FileExists(fn)
          EditFile( fn )
          if linenumber
               GotoLine(linenumber)
               GotoColumn(col - 9)      // if isrch was used
          endif
          if write_hits_flag == FALSE   // hits were NOT written
              lFind(sstring, options)   // so goto first
          endif
          Message("'ALT-ENTER' to reselect")
          Disable(hit_file)
          Enable(file)

          UpdateDisplay()
          pad = SubStr(pad, 1, 44 - Length(CurrFileName()))
          ScrollToRow(12)
     else
          if Length(fn)
               Warn("Can't find file [" , fn, ']    '  )
               Enable(hit_file)
               Return()
          endif
          goto top
     endif
end

/**************************************************************************
          Goto Hit List
 **************************************************************************/
proc                          goto_hit_list(INTEGER mode)
     EditFile(outfile)
     oid = GetBufferID()
     Enable(hit_file)
     if mode == 0
          BegFile()
          linenumber = 1
          col = 1
          if lFind(name_prefix,'')
               first_file_line = CurrLine()
               ScrollToRow( Query(ScreenRows) )
          else
               first_file_line = 12
          endif
     else
          Down()
     endif
end

/**************************************************************************
          display help message
 **************************************************************************/
proc                          help_display()
     Message('Press any key to continue')
     ShowHelp(keystrokes)
     UpdateDisplay()
end

/**************************************************************************
          Keystroke Help
 **************************************************************************/

Help keystrokes
     title = 'Keystrokes for TSGREP:'
     width = 78
     height = 18
     x = 1 y = 2
' While you are in the file listing hits:'
''
'     <alt F>            Toggle showing lines with hits or only filenames'
'     <alt N>, <alt P>   Go to next/prev hit in listing'
'     <ctrl N>, <ctrl P> Go to next/prev file in listing'
'     <alt R>, <alt U>   Remove the entries for a file/undo all removes'
'     <alt enter>        Edit file AND go to line the cursor is on'
'     <alt h>            Pop up this help window.'
'     <alt ->            Unload TSGREP. (All keys revert to normal use.)'
''
' While editing one of the files listed'
''
'     <alt N>, <alt P>   Goto next/prev hit in this file'
'     <alt enter>        Go back to the file listing hits'
'     <alt h>            Pop up this help window.'
'     <alt ->            Unload TSGREP. (All keys revert to normal use.)'
''
'                     Press any key when you have memorized this screen.'
end

/**************************************************************************
          hilite the found text
 **************************************************************************/
proc                          hilite_filename()
    UpdateDisplay()                         // seems needed twice sometimes
    UpdateDisplay()
     if PosLastNonWhite() -  CurrPos() + 1
        PutAttr(Query(HiLiteAttr), PosLastNonWhite() -  CurrPos() + 1)
     endif
end

/**************************************************************************
          Setup Stuff for Interactive Operation
**************************************************************************/
proc                          interactive_setUp()

     cid = GetBufferID()
          repeat
          if IsChanged() and
             query_saves and
             (NumLines() <> 0)
               UpdateDisplay()
               case YesNo( 'Save changes?')
                    when 0, 2, 3           // Escape OR Cancel
                         if GetBufferID() == cid    // reset cid
                              PrevFile(_DONT_LOAD_)
                              cid = GetBufferID()
                              NextFile(_DONT_LOAD_)
                         endif
                         name = CurrFileName()
                         linenumber = CurrLine()
                         Message('Reloading ' + name)
                         AbandonFile()
                         EditFile(name)
                         GoToLine(linenumber)
                         UpdateDisplay(_STATUSLINE_REFRESH_)
                     when 1                   // Yes
                         SaveFile()      //
                   endcase
          endif
          NextFile(_DONT_LOAD_)
          until cid == GetBufferID()

    PopWinOpen( 01,01, Query(ScreenCols), Query(ScreenRows), 0, '',
       Query(OtherWinBorderAttr))    // to blank screen
    ClrScr()

     if NOT ask_input()
          AbandonFile(cfid)
          PopWinClose()            // full screen window
          AbandonFile(GetBufferID('++unnamed++'))
          halt
     endif

     msgs = Set(MsgLevel,_NONE_)
     if GetBufferID(outfile)
          AbandonFile(GetBufferID(outfile))
     endif
     Set(MsgLevel,msgs)

     if infile == ''               // seems to required
          infile = 'none.$$$'           // else a file is dropped!
     endif

     /*
          List currently-loaded files
     */
     PrevFile(_DONT_LOAD_)                   // currently-loaded files
     NextFile(_DONT_LOAD_)                   // start from a file
     cid = GetBufferID()

     if cfid
          AbandonFile(cfid)
     endif
     cfid = CreateTempBuffer()               // buffer to hold list
     GotoBufferID(cid)
     repeat
          id = GetBufferID()
          name = CurrFileName()              // file name
          isloaded = iif(NumLines(), 1, 0)   // whether to reload
          linenumber = CurrLine()            // line to which to go
          GotoBufferId(cfid)
          AddLine(Format(name : -MAXFN) +
                  Format(isloaded) +
                  Str(linenumber) )
          GotoBufferId(id)
          NextFile(_DONT_LOAD_)
     until GetBufferID() == cid
end

/**************************************************************************
          Load File List
 **************************************************************************/
proc                          load_file_list()
                              // note: this doesn't load nested files
                              // that is done in main processing
     INTEGER
          cid = GetBufferID()
     STRING
          name[MAXFN] = CurrFileName()
     EndFile()                // to force processing in the order
     BegLine()                //   files appear in the list file
     repeat
          if CurrLineLen() AND
             Pos(recurse_flag, GetText(1, CurrLineLen()))
               load_recursively( GetText(1, CurrLineLen() -1 ))
//             VGotoXY(1,loading_files_line) PutLine("   Loading files from: [" + GetText(1, CurrLineLen()) + ']' ,54)
               VGotoXY(1,loading_files_line) PutLine("   Loading files from: [" + name + ']' ,54)
               VGotoXY(1,files_loaded_line)  PutLine("    Files Inventoried: ["+Str(NumFiles())+"] " ,54)
          elseif CurrLineLen() AND
             (FileExists( GetText(1, CurrLineLen()) ))
//             VGotoXY(1,loading_files_line) PutLine("   Loading files from: [" + GetText(1, CurrLineLen()) + ']' ,54)
               VGotoXY(1,loading_files_line) PutLine("   Loading files from: [" + name + ']' ,54)
               VGotoXY(1,files_loaded_line)  PutLine("    Files Inventoried: ["+Str(NumFiles())+"] " ,54)
               EditFile( '-a none.$$$ ' + GetText(1, CurrLineLen()) )
          endif
          GotoBufferId(cid)
          check_for_abort()
     until NOT Up()
     if NumFiles() == 1
          Warn('Cannot locate ANY files')
     endif
     GotoBufferId(cid)
     AbandonFile()
     NextFile()
     VGotoXY(1,files_processed_line)    PutLine("      Files Completed: [ ]" ,54)
     return()
end

/**************************************************************************
          Load Recursively
 **************************************************************************/
proc                          load_recursively(STRING arg)
     /*
          Creates AND loads a list file that has
          one line for each subdirectory under the
          specified directory. Each line included the filespec
          to be searched.
     */
     STRING
          dir[MAXSPEC]     = SplitPath(arg, _DRIVE_ | _PATH_ ),
          filespec[MAXSPEC]= SplitPath(arg, _NAME_  | _EXT_  ),
          dir_name[MAXSPEC - 12] = '',
          dirfile[SizeOf(ramdrive) + 12] = ramdrive + '$temp$.$$$'
     INTEGER
           listfile = 0,                     // bufferid for listfile
           dirlist = 0,                      // bufferid for dir list
           dir_list_buff = 0,
           dir_counter = 0

     if listfile
          AbandonFile(listfile)
     endif
     if dirlist
          AbandonFile(dirlist)
     endif
warn(numfiles())
if (numfiles() < 1) OR
   ((numfiles() ==1) AND (CurrFileName() == '++unnamed++'))
   warn('aha')
endif

     listfile = CreateTempBuffer()     // bufferid for listfile
     dirlist = CreateTempBuffer()      // bufferid for dir list

     if FileExists(arg)                       // if there are files in arg
         GotoBufferId(listfile)               // add to list file
         AddLine(arg)
     endif

     dir_name = dir
     VGotoXY(1,getting_dirs_line)  PutLine("   Finding Dirs Under: [" + dir + ']' ,54)
     Delay(2)
     VGotoXY(1,dirs_found_line)    PutLine("           Dirs Found: [ ]" ,54)
     Delay(2)
     EndFile()                // to force processing in the order

top:
     /*
          Create buffer to hold a directory AND get
          it in there.
     */
     if dir_list_buff
          AbandonFile(dir_list_buff)
     endif
     dir_list_buff = CreateTempBuffer()

     DOS('dir ' + dir + dir_spec + ' > ' + dirfile, _dont_clear_ )
     GotoXY(1,1)                   // avoids some DR DOS pita-ness
     InsertFile (dirfile)

     VGotoXY(1,getting_dirs_line)  PutLine("   Finding Dirs Under: [" + dir + ']' ,54)
     repeat                        // for each line in directory
          check_for_abort()
          BegLine()
          while (                  // delete lines we don't want
                 CurrChar() == Asc(' ') OR   //
                 CurrChar() == Asc('.') OR
                 CurrLineLen() == 0
                )
                AND
                (NumLines() <> CurrLine())
               DelLine()
          endwhile

          line = GetText(1, CurrLineLen())   // work with line we do

          if NOT Pos('<DIR>', line)          // if NOT a directory
               DelLine()                     // delete it
          else
               dir_counter = dir_counter + 1
               VGotoXY(1,dirs_found_line)    PutLine("           Dirs Found: ["+Str(dir_counter)+"] "
                         + dir_name ,54)
               dir_name = dir + GetText(1,Pos(' ', line)-1)
               if GetText(12,1) <> ' '       // check for .nnn dir name
                    dir_name = dir_name + '.' + GetText(10,3)
               elseif GetText(11,1) <> ' '   // check for .nn dir name
                    dir_name = dir_name + '.' + GetText(10,2)
               elseif GetText(10,1) <> ' '   // check for .n dir name
                    dir_name = dir_name + '.' + GetText(10,1)
               endif

               GotoBufferId(dirlist)         // add to the list of dirs
               AddLine(dir_name)             // to process
               if dirfile_only
                    GotoBufferID(dironlyfile)
                    AddLine(dir_name)
                    SaveFile()
               endif

               if FileExists(dir_name + '\' + filespec)
                    GotoBufferId(listfile)        // add to list file
                    AddLine(dir_name + '\' + filespec)
               endif

               GotoBufferId(dir_list_buff)
          endif
     until NOT Down()
     AbandonFile()                                // abandon dir_list_buff

     /*
          repeat for any directories left
          to process
     */

     GotoBufferId(dirlist)
     BegFile()
     if NumLines() AND CurrLineLen()
          dir = GetText(1,CurrLineLen()) + '\'    // set next dir to process
          DelLine()                               // delete it from list
          goto top                                // loop around
     endif
     AbandonFile(dirlist)
     GotoBufferId(listfile)                       // mark all of listfile
     BegFile()
     MarkLine()
     EndFile()
     MarkLine()
     if FileExists('' + list_file_flag + '')
          EraseDiskFile('' + list_file_flag + '')
     endif
     CreateBuffer(list_file_flag)                 // copy to an @file file
     CopyBlock()                                  // which will be processed
                                                  // AND abandoned by TSGREP
     AbandonFile(listfile)
     EraseDiskFile(dirfile)
     Message('')
end

/**************************************************************************
          Processing for  multiple filespecs
 **************************************************************************/
INTEGER proc                  multiple_filespecs(VAR STRING filespec)
     INTEGER
          files = 0,
          passes = 1
     STRING
          file_to_load[MAXSPEC]='',
          prev_path[MAXFN - 14]='',
          prev_drive[2]='',
          orig_file_sep_char[1] = file_sep_char

     repeat                             // repeat loop 2x, once for
                                        // unmodified file_sep_char
                                        // once for it set to  ' '
          while Pos(file_sep_char, filespec)
               file_to_load = SubStr(
                                     filespec,
                                     1,
                                     Pos(file_sep_char, filespec)-1 )
               /*
                  Assign previous filespec's path if none
               */
               if SplitPath(file_to_load, _drive_ | _path_) == ''
                    file_to_load = prev_drive + prev_path + file_to_load
               endif
               /*
                  Assign previous filespecs drive if none
               */
               if SplitPath(file_to_load , _drive_) == ''
                    file_to_load = prev_drive + file_to_load
               endif

               prev_drive = SplitPath(file_to_load , _drive_)
               prev_path = SplitPath(file_to_load , _path_)

               if Pos(recurse_flag, file_to_load)
                    load_recursively(SubStr(file_to_load,
                                             1,
                                             Length(file_to_load)-1))
               else
                    if FileExists(file_to_load)
                         VGotoXY(1,loading_files_line) PutLine("   Loading files from: [" + GetText(1, CurrLineLen()) + ']' ,54)
                         VGotoXY(1,files_loaded_line)  PutLine("    Files Inventoried: ["+Str(NumFiles())+"] " ,54)
                         EditFile('-a none.$$$ ' + file_to_load) // edit first spec
                    endif
               endif

               filespec = SubStr(                 // remove first spec
                              filespec,           // from filespec
                              Pos(file_sep_char, filespec)+1,
                              Length(filespec)
                                 )
               files = files + 1
          endwhile

          passes = passes + 1
          file_sep_char = ' '
     until passes == 2

     file_sep_char = orig_file_sep_char      // reset

     /*
       Assign previous filespecs path if none
     */
          if SplitPath(filespec, _drive_ | _path_) == ''
               filespec = prev_drive + prev_path + filespec
          endif
          /*
             Assign previous filespecs drive if none
          */
          if SplitPath(filespec , _drive_) == ''
               filespec = prev_drive + filespec
          endif

     if Pos(recurse_flag, filespec)
         load_recursively(SubStr(filespec, 1, Length(filespec)-1))
     else
         EditFile('-a none.$$$  ' + filespec) // edit last spec
     endif
     return(files)
end

/**************************************************************************
          MAIN
 **************************************************************************/
proc                          MAIN()
     normal_Attr = Query(TextAttr)           // Attribute for normal text
     hilite_Attr = Query(HiLiteAttr)         // Attribute for highlighting
     hits = 0                                // balls = 2
     files = 0                               // runs =  0 <g>
     context = 0                             // will be set by optionstring
     last_line_gotten = 0                    //
                                             // :U: defaults:
     load_file =  TRUE                       // :U:  load outfile into editor
     linenumbers = TRUE                      // :U:  to include (OR NOT)
     write_hits_flag = TRUE                  // :U:  write hits in outfile

     ILBA = Set(InsertLineBlocksAbove, ON)
     MSGLVL = Query(MsgLevel)

     if sstring_hist_buff == 0               // set up histories for Asks
        sstring_hist_buff = GetFreeHistory()
         infile_hist_buff = GetFreeHistory()
        options_hist_buff = GetFreeHistory()
        outfile_hist_buff = GetFreeHistory()
        AddHistoryStr(default_name, outfile_hist_buff)
        AddHistoryStr('i',  options_hist_buff)
     endif

     if NOT done_executed_once               // don't get after 1st time
          options = GetEnvStr('OPTIONS')
               AddHistoryStr(options, options_hist_buff)
               options_entered = options     // options is what TSGREP will work with

          infile  = GetEnvStr('INFILE')
               AddHistoryStr(infile, infile_hist_buff)
          Lower(infile)
          outfile = GetEnvStr('OUTFILE')
               AddHistoryStr(outfile, outfile_hist_buff)
          sstring = GetEnvStr('STRING')
               AddHistoryStr(sstring, sstring_hist_buff)
     endif

     /*
          if no Env STRINGs, go interactive
     */
     if (sstring == '')                      // tests whether running
          OR done_executed_once              // from cmdline
          OR (NumFiles() > 1)
          OR (NumFiles() == 1 AND CurrFileName() <> '++unnamed++')
          tsgrep_mode = INTERACTIVE
          interactive_setUp()
     endif

     /*
          Set editor states
     */

     Set(Break, ON)
     im   = Set(Insert,ON)
     auto = Set(AutoIndent,OFF)
//   msgs = Set(MsgLevel,_NONE_)
//   msgs = Set(MsgLevel,_ALL_MESSAGES_)
     msgs = Set(MsgLevel,_warnings_only_)
     wild = Set(LoadWildFromInside, ON)
     km   = Set(KillMax, 0)

     /*
        The two lines below are unique for my setup only all my
        when(event) macros check this variable. if TRUE, they do no
        processing. This prevents any states from being set that I don't
        want, e.g., AutoIndent.
     */
     SetGlobalInt('Bypass_WhenSwitchToFile', TRUE)
     SetGlobalInt('Bypass_WhenFirstEdited', TRUE)

     /*
     Set environment.
     Parse option string AND set flags.
     */
     if oid                        // to ensure minimal # of buffers
          AbandonFile(oid)
     endif
     oid = CreateTempBuffer()      // buffer to become output file

     if Length(options) == 0                 // set default options
          options = 'i'
     else

          /*
               Set whether to unload/reload files

          if Pos(unload_flag, options)
               unload = TRUE
               options = repl_char(unload_flag, '', options) // get rid of option
          endif
          */

          /*
               Set 'files only' entries
          */
          if Pos(files_only_flag, options)
               options = dont_write_hits        // \1 mode automtically pops in
               options_entered = files_only_flag
               sstring = ''
          endif

          /*
               Set 'don't write lines' flag
          */
          if Pos(dont_write_hits, options)
               options = repl_char(dont_write_hits, '', options) // get rid of option
               write_hits_flag = FALSE
               if NOT Pos(one_hit_flag, options)     // set 'one hit' mode
                    options = options + one_hit_flag
               endif
          endif

          /*
               Set dirfile_only flag
          */
          if Pos(dirfile_flag, options)
               options = repl_char(dirfile_flag, '', options) // get rid of option
               dirfile_only = TRUE
               write_hits_flag = FALSE
               dironlyfile = createbuffer('dir_list')
               if NOT Pos(one_hit_flag, options)     // set 'one hit' mode
                    options = options + one_hit_flag
               endif
          endif

          /*
               Set flag for loading the output file
          */
          if Pos(dont_load_outfile, options)
               load_file = FALSE             // strip from options
               options = repl_char(dont_load_outfile, '', options)
          endif

          /*
               Set flag for line number suppression
          */
          if Pos(no_line_numbers, options)
               linenumbers = FALSE          // strip from options
               options = repl_char(no_line_numbers, '', options)
          endif

          /*
               Set flag for turning ALL TSE msgs off !!!
          */
          if Pos(no_warning_flag, options)
               Set(MsgLevel,_NONE_)
               options = repl_char(no_warning_flag, '', options)
          endif

          /*
               Set number of context lines to include
          */
          if Pos(context_flag, options)
               context = Val(SubStr(options,Pos(context_flag,options)+2,1))
               options = repl_char(context_flag, '', options)
               options = repl_char(Str(context), '', options)
          else
               context = 0
          endif
          if context == 1
               warn('Setting context to 2 (minimum value)')
               context = 2
          endif

     endif

     /*
          Set reg exp mode if certain STRINGs in search STRING,
          unless no_re_flag option used.
          :U: Edit here to change to string used to cause automatic re search
     */
     line = sstring                          // to preserve state
     Lower(sstring)
     Lower(or_operator)                      // in case someone defines
     if (
         Pos (or_operator, sstring)        // variable with upper case
         OR Pos(and_operator, sstring)
         OR Pos(".*",        sstring)
         OR (
             Pos("[" ,        sstring)
             AND Pos("]" ,        sstring)
            )
        )
        AND NOT Pos(no_re_flag, options)    // if NOT overridden
        AND NOT Pos('x', options)           // if NOT present already
          options = options + 'x'
          if NOT Pos('x', options_entered)
               options_entered = options_entered + 'x'
          endif
     endif                                   // then
     if Pos(no_re_flag, options)             // strip out no_re_flag
          options = repl_char(no_re_flag, '', options)
     endif
     sstring = line                          // to restore state

     /*
          Get rid of unusable options.
     */
     if Pos('g', options)
          options = repl_char('g', '', options)
     endif
     if Pos('b', options)              // get rid of option we can't use
          options = repl_char('b', '', options)
     endif
     if Pos('+', options)              // get rid of option we can't use
          options = repl_char('+', '', options)
     endif
     if Pos('l', options)              // get rid of option we can't use
          options = repl_char('l', '', options)
     endif

     /*
     Decode Searchstring

          Substitute < for less_than
          Substitute > for greater_than
     */
     sstring = repl_char(less_than, '<', sstring)
     sstring = repl_char(greater_than, '>', sstring)
     /*
          Substitute | for or_operator
     */
     if Pos('x', options)                    // regexp only
          sstring = repl_char(or_operator, '|', sstring)
     endif
     /*
          Substitute | for or_operator
     */
     if Pos('x', options)                    // regexp only
          sstring = repl_char(or_operator, '|', sstring)
     endif

     /*
          Substitute = for eq_char
     */

     sstring = repl_char(eq_char, '=', sstring) // decode equal signs

     /*
          Set up AND
     */
     if Pos(and_operator, sstring)
          process_and()
     endif

     /*
          Substitute spaces for space char
     */
     if NOT Pos(dont_decode_spacechar, options)             // decode spaces
        AND tsgrep_mode == CMDLINE
          sstring = repl_char(space_char,  ' ', sstring)
     else
          options = repl_char(dont_decode_spacechar, '', options)
     endif

/*
     Provide Default Braces if | is used with regular expr.
*/
     if Pos('|', sstring) AND Pos('x', options)
          tid = CreateTempBuffer()
          InsertText(sstring)                // put STRING in buffer
          BegFile()
          while lFind ('[\}]\|\c[~\{]', 'gx') // process {...}|...
               InsertText('{')               // becomes {...}|{...}
               EndLine()
               InsertText('}')
          endwhile
          while lFind ('[~\}]\c\|[\{]', 'gx')// process ...|{...}
               InsertText('}')               // becomes {...}|{...}
               BegLine()
               InsertText('{')
          endwhile
          if lFind ('[~\}]\|[~\{]', 'x')     // process ...|...
             Replace('{[~\}]}{\|}{[~\{]}', '\1}\2{\3', 'gnx')
             BegLine()                       // becomes {...}|{...}
             InsertText('{')
             EndLine()
             InsertText('}')
          endif
          BegLine()
          sstring = GetText(1,CurrLineLen())
          AbandonFile(tid)
          GotoBufferId(oid)
     endif

/*
     Set default outfile
*/
     if Length(outfile) == 0
          outfile = default_name
     endif
     if SplitPath(outfile, _drive_ | _path_) == ''
          outfile = ramdrive + outfile
     endif
     default_name_2 = SplitPath(outfile, _DRIVE_ | _NAME_) + default_ext_2
     if GetBufferID(ExpandPath(default_name_2))
          AbandonFile(GetBufferID(ExpandPath(default_name_2)))
     endif
     if GetBufferID(ExpandPath(outfile))
          AbandonFile(GetBufferID(ExpandPath(outfile)))
     endif
     if FileExists(default_name_2)      // overkill <g>
          EraseDiskFile(default_name_2)
     endif

/*
     Place header in outfile.
*/
     GotoBufferId(oid)
     AddLine('Ŀ')
     AddLine(' TSGREP v' + v +
             ' ... David Marcus' +
             Format(GetDateStr() + " " + GetTimeStr():18) +" ")
     AddLine('')
     AddLine("         Searchstring: [" + sstring + "]")
     if tsgrep_mode == CMDLINE
          AddLine("      Input File Spec: " + GetEnvStr('INFILE'))
     else
          AddLine("      Input File Spec: LOADED FILES + " + infile )
     endif
     AddLine("         File created: " + outfile)
     AddLine("              Options: [" + options_entered + ']')
     AddLine()
     AddLine("              # Files: [")
     AddLine("               # Hits: [")
     AddLine()
     AddLine( SubStr(pad,2,30) + '  Files With Hits   ' +
                      SubStr(pad,2,30) )
     AddLine()

/*
     Screen display
*/
     Message('')
     Set(Cursor,Off)
     HideMouse()
     PopWinOpen( 08,12, 73,25, 1, '', normal_attr)
     ClrScr()
     Set(Attr, normal_attr)
     VGotoXY(1,2)                       WriteLine(Format('  Ŀ':-54))
                                        WriteLine(Format('   TSGREP v'+v+' ... David Marcus' +
                   Format(GetDateStr() + ' ' + GetTimeStr() :18)  +' ':-54))
                                        WriteLine(Format('  ':-54))
                                        WriteLine(Format("         Searchstring: [" + sstring + "]": -99))
     if tsgrep_mode == CMDLINE
                                        WriteLine(Format("      Input File Spec: " + GetEnvStr('INFILE'):-99))
     else
                                        WriteLine(Format("      Input File Spec: LOADED FILES + " + infile:-99 ))
     endif
                                        WriteLine(Format("         File created: " + outfile:-54))
                                        WriteLine(Format("              Options: [" + options_entered + ']':-54))
     VGoToXY(1, files_processed_line)   WriteLine(Format("      Files Completed: [ ]":-54))
     VGoToXY(1, hits_line)              WriteLine(Format("               # Hits: [ ]":-54))
     Set(Attr,hilite_attr)
     PutLine  ("                            <escape> to abort", 50)
     Set(Attr, normal_attr)

     /*
          Process multiple file input (OR single)
     */
     multiple_filespecs(infile)
     AbandonFile(GetBufferID('++unnamed++'))
     cid = GetBufferID()           // set 1st file ID

     /*
          Perform listfile loading.
     */
     GotoBufferId(cid)
     if Pos(list_file_flag ,CurrFileName())
          load_file_list()
     endif
     NextFile_if_NOT_excluded()
     cid = GetBufferID()

/* --------------------------------------------------------------------------
     MAIN PROCESSING LOOP
*/
     repeat
          id = GetBufferID()                 // set id of file being processed

          if Pos(outfile, CurrFileName()) OR // don't process outfiles
             Pos(default_name_2, CurrFileName())
               NextFile_if_NOT_excluded()
               id = GetBufferID()
          endif

          /*
               Check for whether is a list file and process accordingly.
          */
          if Pos('' + list_file_flag + '',CurrFileName())
               name = CurrFileName()
               VGotoXY(1,getting_files_line) PutLine("   Getting files from: [" + name + ']' ,42)
               load_file_list()
               GotoBufferId(id)
               if id == cid
                    NextFile_if_NOT_excluded()
                    id = GetBufferID()
                    cid = id
               else
                    abandonfile()
                    NextFile_if_NOT_excluded()
                    id = GetBufferID()
               endif
          endif

          name = CurrFileName()              // get name of file
          GotoBufferId(oid)                  // go to output buffer
          GotoColumn(1)
          if Length(name)                    // prevent blank at end
                InsertText(name_prefix + name + name_suffix)
          endif                              // insert file name
          name_line = CurrLine()             // later, go back to this line
          VGotoXY(1,processing_line)    PutLine("           Processing: [" + name + ']' ,54)
          CReturn()                          // AND creturn

          GotoBufferId(id)                   // goto file being processed
          BegFile()
          last_line_gotten = 0
          while lFind(sstring, options)
               AND CurrLine() <> NumLines()  // stops processing if EOF
               hits = hits + 1
               curr_file_hits = TRUE
               VGotoXY(25,hits_line)    PutLine(Str(hits)+']',10)
               if write_hits_flag
                    linenumber = CurrLine()            // remember line number
                    UnMarkBlock()
                    PushPosition()                     // remember start
                    GotoLine(CurrLine() - context)     // goto top of context
                    if CurrLine() <= last_line_gotten  // don't repeat context
                         GotoLine(last_line_gotten + 1)
                    endif
                    first_line_gotten = CurrLine()     // for numbering
                    MarkLine()                         // mark beg of block
                    PopPosition()
                    GotoLine(CurrLine() + context)     // goto end of context
                    MarkLine()                         // mark end of block
                    EndLine()
                    while NOT Pos(one_hit_flag, options) AND
                       lFind(sstring, options + 'b')   // i.e., if >1 hit
                         AND CurrLine() > linenumber   // in block
                         linenumber = currline()       // need to
                         GotoLine(CurrLine() + context)// add more lines
                         MarkLine()
                         EndLine()
                    endwhile
                    gotoblockend()

                    last_line_gotten = CurrLine()      // remember for next time
                    if context
                         GotoBlockBegin()              // adjust hits counter
                         hits = hits - 1               // for all hits in block
                         while lFind(sstring, options + 'l')
                              hits = hits + 1
                              endline()
                         endwhile
                    endif
                    linenumber = CurrLine()       // remember line number

                    GotoBufferId(oid)             // goto output buffer
                    CopyBlock()                   // copy block
                    GotoBlockBegin()
                    BegLine()                     // insert line number
                    if linenumbers               // if so flagged
                         InsertText(Format(first_line_gotten:lndigits) +  '  ')
                              Down() BegLine()
                         while IsCursorInBlock()
                              first_line_gotten = first_line_gotten + 1
                              InsertText(Format(first_line_gotten:lndigits) +  '  ')
                              Down() BegLine()
                         endwhile
                    endif
                    if context                    // flag actual hits
                         GotoBlockBegin()
                         Up()
                         while Down() and IsCursorInBlock() and
                              lFind(iif(Pos('^', options),
                                        Format('':lndigits:'.') + '...' + sstring,
                                        sstring
                                       ),
                                     iif(Pos('^', options),
                                         options + 'lx',
                                         options + 'l'
                                        )
                                    )
                              BegLine()
                              InsertText(GetText(1,lndigits) + ' * ', _OVERWRITE_)
                         endwhile

                         GotoBlockBegin()
                         if GetText(lndigits + 1, 3) == '  '
                              BegLine()
                              InsertText(GetText(1,lndigits) + '  ', _OVERWRITE_)
                         endif
                         GotoBlockEnd()
                         BegLine()
                         if GetText(lndigits + 1, 3) == '  '
                              InsertText(GetText(1,lndigits) + '  ', _OVERWRITE_)
                         endif
                         GotoBlockBegin()
                         BegLine()
                         val2 = val(gettext(1, lndigits))
                         Up()
                         if val(gettext(1, lndigits)) == val2 + 1
                              Down()
                              InsertText(GetText(1,lndigits) + '  ', _OVERWRITE_)
                         endif
                    endif
                    GotoBLockEnd()
                    Down()
               endif
               GotoBufferId(id)              // goto file being processed
               if Pos(one_hit_flag, options)
                    EndFile()
               else
                    Down()                   // AND down
                    BegLine()                // to beginning of next line
               endif
               check_for_abort()
          endwhile
          BegFile()                          // reposition to top of file
          files = files + 1
          VGotoXY(25,files_processed_line)   PutLine(Str(files)+'] ~' + Str(NumFiles()-2) + ' to go',20)
          NextFile_if_NOT_excluded()
          id = GetBufferID()

          if curr_file_hits == FALSE        // if no hits, push file name
               no_hitters_flag = TRUE
               GotoBufferId(oid)            //
               DelLine()                    // get rid of blank line bottom
               GotoLine(name_line)
               InsertLine()
          endif
          curr_file_hits = FALSE
          GotoBufferId(id)
          if GetBufferID() <> cid            // this processing unloads processed
               PrevFile()                    // files so we don't get VM slows
               if GetBufferID() <> cid
                    AbandonFile()     // abandon regardless
               endif
             GotoBufferId(id)
          endif

          check_for_abort()
     until GetBufferID() == cid
     PrevFile()
     AbandonFile()
     VGotoXY(25,files_processed_line)   PutLine(Str(files)+']',22)              // get rid of 'to go'

     end_processing(0)
end

/**************************************************************************
          Create Files-Only List AND Create Backup Complete List
 **************************************************************************/
proc                          make_other_hit_lists()
     msgs = Set(MsgLevel,_warnings_only_)
     if copy_of_list
          AbandonFile(copy_of_list)
     endif
     copy_of_list = CreateTempBuffer()
     InsertFile(outfile)
     if files_only
          AbandonFile(files_only)
     endif
     files_only = CreateTempBuffer()
     Message(default_name_2)
     SaveAs(default_name_2,1)
     EditFile(default_name_2)
     AbandonFile(files_only)
     files_only = GetBufferID()
     InsertFile(outfile)
     Set(KillMax,0)
     repeat
          if ( Val(GetText(1, lndigits)))
               DelLine()
          else
               Down()
          endif
     until NumLines() == CurrLine()
     Set(KillMax,km)
     ForceChanged(FALSE)
     GotoBufferId(oid)
     PopBlock()
     Set(MsgLevel,msgs)
end

/**************************************************************************
          NextFile if NOT Excluded
 **************************************************************************/
proc                          NextFile_if_NOT_excluded()
     STRING
            exclude[255] = GetEnvStr('EXCLUDE')
             + ' .img .exe .com .dll .tif .pcx .$hp .$sc .cif .vgr', // :U:
             // note in doc if this changes
             fn[8] = '',
             fe[4] = ''

     /*
          Goes to next file if NOT on excluded .ext list.
     */
     check_for_abort()
top:
     NextFile(_DONT_LOAD_)              // goto bufferid w/o load
     fn = SplitPath(CurrFileName(),_NAME_)
     fe = SplitPath(CurrFileName(),_EXT_)

     if Pos(Format( fe : -4) , exclude)
          OR ((fn + fe) == 'none.$$$')
          OR ((fn + fe) == OUTFILE)
          OR ((fn + fe) == default_name_2)
          AbandonFile()
          goto top
     endif
     if Length(CurrFileName()) and FileExists((CurrFileName()))
          VGotoXY(1,processing_line)    PutLine("              Loading: [" + CurrFileName() + ']' ,54)

          EditFile(CurrFileName())      // to load file
     endif
end

/**************************************************************************
          Process the 'and' operator in search string
 **************************************************************************/
proc                          process_and()
     STRING
          part1[MAXSPEC] = '',
          part2[MAXSPEC] = ''

     part1 = SubStr(sstring, 1, Pos(and_operator,sstring) - 1)
     part2 = SubStr(sstring,    Pos(and_operator,sstring) + Length(and_operator), Length(sstring))
     sstring = '{'+ part1 + '.*' + part2 + '}|{' +
                    part2 + '.*' + part1 + '}'
end

/**************************************************************************
          Reload Files (Interactive Mode)
 **************************************************************************/
proc                          reload_files()
     INTEGER count = 0
     cid = GetBufferID()
     if NOT cfid                   // in case current file list NOT built
          return()
     endif
     GotoBufferId(cfid)
     BegFile()
     repeat
          count = count + 1
          GotoBufferId(cfid)
          if GetText(1,11) == '++unnamed++'
                                             // do nothing

          elseif GetText(MAXFN + 1, 1) == '1'
               Message('Reloading ' + GetText(1,MAXFN))
               EditFile(SubStr(
                                GetText(1,MAXFN), 1, Pos(' ', GetText(1,MAXFN))
                              ) +
                         '-n' + GetText(MAXFN + 2, lndigits)
                       )
          else
               Message('Reloading ' + GetText(1,MAXFN))
               EditFile(outfile + ' ' + GetText(1,MAXFN))
          endif
          GotoBufferId(cfid)
     until NOT Down()
     AbandonFile(cfid)
end

/**************************************************************************
          Remove File From Hit List
 **************************************************************************/
proc                          remove_file_from_hit_List()
     INTEGER
          startbuffer = GetBufferID(),
          counter = 0
     Set(KillMax, 0)
     PushBlock()
     PushPosition()
     UnMarkBlock()
     EndLine()
     if startbuffer == files_only
          if GetText(1,Length(name_prefix)) <> name_prefix
               Warn('NOT on file name')
                  return()
          endif
          line = GetText(1,80)
          DelLine()
          ForceChanged(FALSE)
          GotoBufferId(oid)
          begfile()
          if NOT lFind(line, '^')
               Warn("can't find ", line)
               Set(KillMax, km)
               return()
          endif
     endif

     name_prefix_re = ''                // build an escaped reg_exp version
     repeat                             // of name_prefix (because RE is used)
     counter = counter + 1              // in alt-p and alt-n)
          if Pos(name_prefix[counter], '.^$\|?[]*+@#{}')
               name_prefix_re = name_prefix_re + '\'
          endif
          name_prefix_re = name_prefix_re + name_prefix[counter]
     until counter == Length(name_prefix)

     if (GetText(1,Length(name_prefix)) == name_prefix) AND Down() AND
        (GetText(1,Length(name_prefix)) == name_prefix) OR
        (NumLines() <= (CurrLine() + 1))
          Up()
          line = GetText(1,80)
          DelLine()
          PopBlock()
     elseif lFind('{'+ name_prefix_re + '}|{}', '^x') AND
        Up() AND
        MarkLine() AND
        lFind(name_prefix, '^b') AND
        MarkLine()
          GotoBlockBegin()
          line = GetText(1,80)
          GotoBlockEnd()
          Down()
          DelBlock()
     endif
     PopBlock()

     if startbuffer == oid
          GotoBufferId(files_only)
          if lFind(line, '^')
               DelLine()
               ForceChanged(FALSE)
          endif
     endif
     PopPosition()
     BegLine()
     Set(KillMax, km)
     case save_hit_list
          when NEVER     ForceChanged(FALSE)
     endcase
     UpdateDisplay()
     hilite_filename()
end

/**************************************************************************
          Replace character with a string -- utility proc
 **************************************************************************/
string proc                   repl_char(STRING find_string, STRING repl_string, VAR STRING target)
     while Pos(find_string, target)
          target = SubStr(target,
                          1,
                          Pos(find_string, target)-1
                          ) +
                    repl_string +
                    SubStr(target,
                           Pos(find_string, target)+
                               Length(find_string),
                           Length(target)
                           )
     endwhile
     return(target)
end

/**************************************************************************
          Toggle From Showing Detail to NOT Showing It
 **************************************************************************/
proc                          toggle_hits()
     INTEGER
          row = CurrRow()
     if GetBufferID() == oid
          PushPosition()
          EndLine()
          lFind(name_prefix, '^b')
          row = CurrRow()
          line=GetText(1,CurrLineLen())
          PopPosition()
          GotoBufferId(files_only)
          BegFile()
          lFind(line,'^')
          ScrollToRow(row)
     elseif GetBufferID() == files_only
          EndLine()
          lFind(name_prefix, '^b')
          line=GetText(1,CurrLineLen())
          ForceChanged(FALSE)
          GotoBufferId(oid)
          BegFile()
          lFind(line,'^')
          ScrollToRow(row)
     endif
     UpdateDisplay()
     hilite_filename()
end

/**************************************************************************
          Undo All "Remove File From Hit List"
 **************************************************************************/
proc                          undo_all_removes()
     INTEGER
          row = 0
     PushBlock()
     UnMarkBlock()
     GotoBufferId(oid)
     line = GetText(1,80)
     row = CurrRow()
     BegFile()      MarkLine()
     EndFile()      MarkLine()
     Set(KillMax,0)
     DelBlock()
     Set(KillMax,km)
     GotoBufferId(copy_of_list)
     BegFile() MarkLine()
     EndFile() MarkLine()
     GotoBufferId(oid)
     CopyBlock()
     PopBlock()
     lFind(line, '^')
     PushPosition()
     PopPosition()
     ScrollToRow(row)
     if save_hit_list == NEVER
               ForceChanged(FALSE)
     endif
     UpdateDisplay()
//   hilite_filename()
end

/**************************************************************************
          Unload TSGREP
 **************************************************************************/
proc                          unload_TSGREP()
     cid = GetBufferID()
     AbandonFile(copy_of_list)
     AbandonFile(files_only)
     if EditFile(default_name_2)
         KillFile()
         AbandonFile()
     endif
     gotobufferid(oid)
     UpdateDisplay()
     QuitFile()
     GotoBufferID(cid)
     PurgeMacro('tsgrep')
     Message("TSGREP purged.")
     Alarm()
end