/*   One percent resistor calculator program
     by Andrew Dart
	Box 380508
	Duncanville, TX   75138

     C version, revison 1.1
     6/10/93

     This program finds the two best values of standard 1% resistors
     to use in a simple two-resistor voltage divider like this:

     V1  ----------/\/\/\/\/------*--------/\/\/\/\/-----------  V2
				  |
		      R1          |            R2
				  |

				  V3

     You specify the voltage at V1 and V2, and the voltage you
     want at V3.  Usually V1 is a positive supply and V2 is
     ground, or zero volts, but V1 need not be greater than V2.
     If V3 is midway between V1 and V2, then you could use any
     two identical resistors, and you don't need this program,
     but you may still find it useful even then, because it will
     adjust the values according to the power rating of the
     resistors (see below).

     One percent resistor values are 96 evenly spaced powers
     of the 96th root of 10.  That is the key to the algorithm
     used here.  Refer to EIA standard RS-385.   While I've
     never seen RS-385 myself, I understand it has something
     to do with this topic.

     Updated features since the first BASIC version:
     *   The result is printed immediately.  There is no delay.

     Updated features since the last BASIC version:
     *   The previous version would select the correct values to
	 achieve the desired output voltage, but typically each
	 resistor was only a few ohms.  You probably are not going
	 to use 100-watt resistors, so this program asks you what
	 size resistors you're using (any reasonable size; for example,
	 quarter-watt resistors) and multiplies the result by some power
	 of ten, to bring the current down to a level that the selected
	 resistors can handle.
     *   This program draws the diagram above and shows the voltages
	 and resistor values, and the power dissipated by each
	 resistor, instead of simply listing the voltages and
	 resistor values.  This makes it a little easier to use.

     Future modifications:
     *   With a little extra programming, I could draw the resistors
	 in color and let you see what the stripes should look like.
     *   After selecting the ideal resistor pair, you could increment
	 or decrement each resistor value (by one place in the table
	 of standard values) and come out with almost exactly the same
	 voltage at V3, but a little higher or lower total resistance.
	 So after the program computes the result, you should be able
	 to hit the up or down arrow keys and have this adjustment made
	 for you, and the values instantly recomputed.  Well, I've tried
	 that, and in many cases it works, but to cover every situation,
	 like one value rolling over to a new order of magnitude (from
	 9.76 to 10.00 and beyond), there are a number of flags and
	 special conditions to keep track of, and it hasn't yet been
	 demonstrated to me that it would be worthwhile.  Keep this in
	 mind, however, if you are using your two-resistor divider to
	 replace a trim-pot and you need the total resistance (R1+R2)
	 to equal the value of the trim-pot (e.g., 1000 ohms).  Adjust
	 the two resistor values up or down (by the same number of steps)
	 in the table, and output voltage will be almost the same as
	 with the ideal combination of values.  Another application for
	 this is in the event you have run out of 715 ohm resistors but
	 still have plenty of 732's.  Still another would be the case
	 where you know R1 must be a particular value, because you have
	 a thousand of them, or because you're modifying an existing design
	 and only want to change one value.  You could then increment or
	 decrement the value of R1 until it was the value you wanted.
	 There's nothing to prevent you from doing this manually, or
	 modifying the program to include this feature, or motivating
	 me to append to the program.

     This program was written with Microsoft QuickC / Quick Assembler
     and probably will run through other compilers as well with little
     or no modification.  The most likely exception would be cursor-
     positioning routines, clearscreen, header files, etc.

     Naturally, the author assumes no responsibility for the use of
     this software, and while every effort has been made to keep it
     free of bugs and test it thoroughly, you may use it at your own
     risk.  Also, I'll take no credit or blame for any circuit or
     system designed (partially) with the aid of this program.

     These remarks are included in RESISTOR.DOC in case you would
     like to cut them out of this file or delete the .DOC file.

     Even though the software is provided to you as is, at no charge,
     if you find it to be of value, or you find the source code to be
     educational, or the comments to be entertaining, or you would
     simply like to support my efforts, please feel free to send a
     generous token of your appreciation, in umarked U.S. currency,
     to the author at the address above.  Job offers will be accepted
     in lieu of donations.
*/

#include <stdio.h>
#include <graph.h>
#include <math.h>
#include <stdlib.h>

main()
{
   float table[] = {1.00,1.02,1.05,1.07,1.10,1.13,1.15,1.18,1.21,1.24,
     1.27,1.30,1.33,1.37,1.40,1.43,1.47,1.50,1.54,1.58,1.62,1.65,1.69,
     1.74,1.78,1.82,1.87,1.91,1.96,2.00,2.05,2.10,2.15,2.21,2.26,2.32,
     2.37,2.43,2.49,2.55,2.61,2.67,2.74,2.80,2.87,2.94,3.01,3.09,3.16,
     3.24,3.32,3.40,3.48,3.57,3.65,3.74,3.83,3.92,4.02,4.12,4.22,4.32,
     4.42,4.53,4.64,4.75,4.87,4.99,5.11,5.23,5.36,5.49,5.62,5.76,5.90,
     6.04,6.19,6.34,6.49,6.65,6.81,6.98,7.15,7.32,7.50,7.68,7.87,8.06,
     8.25,8.45,8.66,8.87,9.09,9.31,9.53,9.76};

   float power[] = { 1, 10, 100, 1000, 10000, 100000, 1000000, 10000000,
		     100000000, 1000000000};

   int flag1 = 0, flag2 = 0, flag3 = 0, flag4 = 0;
   int multiplier = 0, steps, scale, x, y;
   float ratio, best_match, V1, V2, V3, temp, trial;
   float total_power, j, k, result, R1_power, R2_power, size, temp1, temp2;
   char buffer[10], Omega = 234;

   _clearscreen( _GCLEARSCREEN );

   /* - - - - - - - - - - - - - - Introduction - - - - - - - - - - - - - */
   _settextposition(9,5);
   printf(" V1  -/\\/\\/\\/\\/*-/\\/\\/\\/\\/-  V2\n");
   printf("                        R1                         R2\n");
   printf("                                      \n");
   printf("                                      \n\n");
   printf("                                     V3 \n");
   _settextposition(1,1);
   printf("One percent resistor calculator\n\n");

   /* - - - - - - - - - - - - - - Get voltages V1 and V2 - - - - - - - - */
   printf("What is the voltage at V1?   ");
   scanf("%f",&V1);
   printf("What is the voltage at V2?   ");
   scanf("%f",&V2);
   if (V1 == V2)
   {
      /* Wise guy, huh? */
      printf("V1 and V2 are both %g volts.\n", V1);
      _settextposition(22,1);
      exit(0);
   }

   /* - - - - - - - - - - - - Swap V2 and V1 if V2 > V1 - - - - - - - - - */
   if (V2 > V1)
   {
      temp = V1;
      V1 = V2;
      V2 = temp;
      flag1 = 1;
   }

   /* - - - - - - - - - - - -  Get Voltage V3 - - - - - - - - - - - - - - */
   do
   {
      V3 = 0;
      printf("What voltage do you want at V3?   ");
      scanf("%f",&V3);

      /* Check for V1 > V3 > V2 */
      if ((V3 < V1) && (V3 > V2))
	 break;
      /* otherwise... */
      printf("Be sure V3 is between V1 and V2.\n");
   }  while ((V1 <= V3) || (V3 <= V2));  /* continue until V1 > V3 > V2 */

   /* - - - - - - - - - - - - Get resistor size - - - - - - - - - - - - - */
   _settextposition(18,1);
   printf("What is the maximum power (in watts) \n");
   printf("that can be safely dissipated by the resistors?\n");
   scanf("%f",&size);

   /* optional feedback: */
   /* printf("Using %g watt resistors\n",size); */

   ratio = (V1 - V3) / (V3 - V2);   /* (ratio) is the ratio of R1:R2  */

   /* - - - - - - - - - - Watch out for special case R1=R2 - - - - - - - - */
   if (ratio == 1.0)
   {
      /* Special case if R1 = R2 */
      j = 1;
      k = 1;
      goto skip_it;
      /* Yes, I know, it says "goto", but it couldn't be easily avoided. */
   }

   /* - - - - - - flag2 is set if V3 is closer to V1 than to V2 - - - - - - */
   if (ratio < 1)
   {
      ratio = 1 / ratio;
      flag2 = 1;
   }

   /* - - - - - - Assure 1 < Ratio < 10 by dividing by a power of ten: - - - */
   scale = (int) floor ( log10( ratio ) );
   ratio /= power[ scale ];

   /* - - - - - - - - - - - - Here is the key to it all: - - - - - - - - - */
   steps = (int) (( log( ratio ) * 41.69227 ) + 0.5 );

   /* (steps) is the log of (ratio) to the base of the
      96th root of 10.

      This is the number of steps in the resistor value table
      between the two values of R1 and R2, whatever they are.
      This narrows the selection down to 96 combinations, any
      one of which is probably close enough, unless you have access
      to any resistor value.

      The optimum combination of standard values (which we are about to
      find) will always result in an output voltage which is well within
      one percent of the target.  In other words, the main source of error
      will be the deviation of the resistors used, from the ideal values.
   */

   /* Now try each of the 96 values, and its mate, and see which pair
      fits best: */

   best_match = 9999.9;
   for (x=0; x<96; x++)
   {
      flag3 = 0;
      y = x + steps;

      /* But there are only 96 values in the table, so if y tries to point
	 to the 97th value, flag3 is set to 1, and 96 is subtracted from y.
	 The alternative to this would be to have 192 values in the table,
	 and have the second half just like the first half, multiplied
	 by ten.  But that's not necessary, since this comment is here
	 and we all understand the algorithm perfectly.
      */

      if (y > 95)
      {
	 y -= 96;
	 /* y = x + steps % 96 */
	 flag3 = 1;
	 /* flag3 indicates y > 96 */
      }

      trial = (power[ flag3 ] * table[ y ]) / table[ x ];
      /* power[ flag3 ] is either 1 or 10 because flag3 is 0 or 1. */

      if (trial == ratio)
      {
	 /* Discontinue loop if a perfect match is found, */
	 /* but first, specify the values for R1 and R2. */
	 j = table[ y ];
	 k = table[ x ];
	 break;
      }
      /* (best_match) gets smaller and smaller as better matches are found. */
      if ( fabs( ratio - trial ) < best_match )
      {
	 best_match = fabs( ratio - trial );
	 j = table[ y ];
	 k = table[ x ];
	 flag4 = flag3; /* flag4 indicates y was > 96 and has been reduced.*/
      }
   }
   j *= power[ scale + flag4 ];

/* Jump here if R1=R2 and no comparisons are required: */
skip_it:
   if ( flag1 != flag2 )
   {
      temp = j;
      j = k;
      k = temp;
   }
   if (flag1)
   {
      temp = V1;
      V1 = V2;
      V2 = temp;
   }

   result = (k * (V1-V2) / (j+k) ) + V2 ;

   /* - - - - - - Calculate the power dissipated by R1 and R2: - - - - - */
   total_power = (fabs(V1-V2) *  fabs(V1-V2)) / (j+k);
   while (total_power > (size * 2.0) )
   {
      multiplier++;
      total_power /= 10.0;
   }

   /* Scale the values up by a power of ten to avoid burning them up */
   j *= power[ multiplier ];
   k *= power[ multiplier ];
   temp1 = (float) fabs( (double) (V1-V3) );
   temp2 = (float) fabs( (double) (V2-V3) );

   R1_power = (temp1 * temp1) / j;
   R2_power = (temp2 * temp2) / k;

   while ((R1_power > size) || (R2_power > size))
   {
      multiplier++;
      total_power /= 10.0;
      j *= power[ multiplier ];
      k *= power[ multiplier ];
      temp1 = (float) fabs( (double) (V1-V3) );
      temp2 = (float) fabs( (double) (V2-V3) );
      R1_power = (temp1 * temp1) / j;
      R2_power = (temp2 * temp2) / k;

   }

   /* - - - - - - - - - - - - -  Now print the result:  - - - - - - - -  */

   _clearscreen( _GCLEARSCREEN );

   printf("\n\nWith %g volts at V1 and %g volts at V2,\n", V1, V2 );

   /* The program used to say...
      printf("if R1 = %g ohms and R2 = %g ohms,\n", j, k );
      Now there is a new approach:
      Values are adjusted so that instead of printing
      "102000 ", for example, the program will print "102 k".
   */

   /* First R1 is formatted for k or M :                               */
   printf("if R1 = ");
   if (j>1000)
   {
      if (j >= 1000000)
      {
	 printf("%gM%c", (j/1000000), Omega);        /* M  */
      }
      else
      {
	 printf("%gk%c", (j/1000), Omega);           /* k  */
      }
   }
   else
   {
      printf("%g%c", j, Omega);                      /*    */
   }
   /* Now R2 is formatted for k or M : */
   printf(" and R2 = ");
   if (k>1000)
   {
      if (k >= 1000000)
      {
	 printf("%gM%c", (k/1000000), Omega);        /* M  */
      }
      else
      {
	 printf("%gk%c", (k/1000), Omega);           /* k  */
      }
   }
   else
   {
      printf("%g%c", k, Omega);                      /*    */
   }
   printf(", \n");
   printf("then V3 will be %g volts.\n", result );

   /* Now draw the diagram again with resistances and voltages shown: */
   _settextposition(8,5);
   printf("V1");
   _settextposition(8,71);
   printf("V2");
   _settextposition(9,9);
   printf(" -/\\/\\/\\/\\/*-/\\/\\/\\/\\/-  ");
   _settextposition(9,4);
   printf("%3.1f", V1);
   _settextposition(9,71);
   printf("%3.1f\n", V2);
   printf("   volts                R1                         R2               volts");
   _settextposition(11,24);
   /* Formatting R1 again as above */
   if (j>1000)
   {
      if (j >= 1000000)
      {
	 printf("%gM%c", (j/1000000), Omega);       /* M  */
      }
      else
      {
	 printf("%gk%c", (j/1000), Omega);          /* k  */
      }
   }
   else
   {
      printf("%g%c", j, Omega);                     /*    */
   }
   _settextposition(11,39);
   printf("");
   _settextposition(11,52);
   /* Now format R2 as above */
   if (k>1000)
   {
      if (k >= 1000000)
      {
	 printf("%gM%c", (k/1000000), Omega);       /* M  */
      }
      else
      {
	 printf("%gk%c", (k/1000), Omega);          /* k  */
      }
   }
   else
   {
      printf("%g%c", k, Omega);                     /*    */
   }
   _settextposition(12,39);
   printf("");
   _settextposition(13,33);
   printf("%3.2f volts", result);
   _settextposition(14,38);
   printf("V3");

   /* Show the power dissipated by each resistor. */
   /* Formatted to show watts or milliwatts, as appropriate. */
   _settextposition(19,1);
   if (R1_power < 0.1)
   {
      printf("R1 will dissipate %3.0f milliwatts\n", (R1_power * 1000));
   }
   else
   {
      printf("R1 will dissipate %1.2f watts\n", R1_power);
   }
   if (R2_power < 0.1)
   {
      printf("R2 will dissipate %3.0f milliwatts\n", (R2_power * 1000));
   }
   else
   {
      printf("R2 will dissipate %1.2f watts\n", R2_power);
   }
   printf("assuming that V3 is connected to a high impedance.\n");

}  /* End of main() */
