/*
 * File......: ATTACH.PRG
 * Author....: Steve Larsen
 * CIS ID....: 76370,1532
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This is an original work by Steve Larsen and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */

#include "ftint86.ch"
#include "netto.ch"


/*  $DOC$
 *  $FUNCNAME$
 *      fn_attchFS()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      Attach to file server
 *  $SYNTAX$
 *
 *      fn_attchFS( <cFileServerName> ) -> lSuccess
 *
 *  $ARGUMENTS$
 *
 *      <cFileServerName> is the file server to attach to
 *
 *  $RETURNS$
 *
 *      A logical indicating whether this workstation successfully attached
 *      to the designated file server.
 *
 *  $DESCRIPTION$
 *
 *      This function allows the workstation to attach to a designated file
 *      server.  A workstation may be attached to up to 8 file servers.  To
 *      use this function, the workstation must already be attached to at
 *      least one file server.
 *
 *      Check fn_Error() upon return for any information regarding why
 *      an attempt to attach failed.  Also, if you attempted to attach to
 *      a server to which you were already attached, fn_attchFS() will return
 *      TRUE, indicating that you are still attached, however will set the
 *      error value returned by fn_Error() to EALREADY_ATTACHED.
 *
 *      Please note that fn_attchFS() simply attaches you to a file server.
 *      To log in to that server, you must make it the preferred server,
 *      then login with a name and password.
 *
 *  $EXAMPLES$
 *
 *      IF !fn_attchFS( "SALES" )          // attach to server
 *         <... unable to attach ...>
 *      ELSE
 *
 *         aFSNameTable := fn_FSName()     // get list of attached servers
 *                                         //  to find ConnectionID
 *
 *         nConnID := ASCAN( aFSNameTable, {|e| e == "SALES" })
 *
 *         fn_SPFCID( nConnID )            // set preferred server
 *                                         //  and log in to it
 *
 *         fn_loginFS( "SLARSEN", OT_USER, "dork" )
 *
 *      ENDIF
 *
 *  $SEEALSO$
 *    fn_detchFS() fn_connID() fn_connINF() fn_loginFS() fn_SPFCID() fn_FSName()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_attchFS( cServerName )
	LOCAL aRegs[ INT86_MAX_REGS ], lResult := .F.
	LOCAL cServerAddress, aConnTable, aSrvrTable, nSlotNo, nOrderNo
	LOCAL nNetNo, nNodeAddr, nSocketNo, aTempCT, aTempST, i, j
	LOCAL aAddr[ 3 ]

	cServerName := UPPER( cServerName )

	/* test bindery object ID for server presence */

	IF ( i:= fn_gBndOID( cServerName, OT_FILE_SERVER )) == 0 // test for server present
		_fnSetErr( EUNKNOWN_SERVER )
		RETURN lResult
	ENDIF

	/* fetch server internet address */

	cServerAddress := fn_rdProVa( cServerName, OT_FILE_SERVER, "NET_ADDRESS" )

	/* verify that the server actually exists */

	IF EMPTY( cServerAddress ) .OR. VALTYPE( cServerAddress ) <> "C"
		_fnSetErr( EUNKNOWN_SERVER )
		RETURN lResult
	ENDIF

	/* got it, parse out the address */

	nNetNo      := PADL( fn_bin2hex( SUBSTR( cServerAddress,  1, 4 ) ),  8, '0' )
	nNodeAddr   := PADL( fn_bin2hex( SUBSTR( cServerAddress,  5, 6 ) ), 12, '0' )
	nSocketNo   := PADL( fn_bin2hex( SUBSTR( cServerAddress, 11, 2 ) ),  4, '0' )

	aConnTable  := fn_connID()         						// fetch our WS' connection table
	aSrvrTable  := fn_FSName()									// and our FS table

	aTempCT     := ACLONE( aConnTable )                // save tables for restoration
	aTempST     := ACLONE( aSrvrTable )                //  upon failure

	/* determine if we are already attached, if so return TRUE but set error */

	IF ( ASCAN( aConnTable, {|a| ( ( a[3] == nNetNo ) .AND. ( a[4] == nNodeAddr ) ) } ) != 0 )
		_fnSetErr( EALREADY_ATTACHED )
		RETURN !lResult
	ENDIF

	/* find an empty slot, if none available return error */

	IF ( nSlotNo := ASCAN( aConnTable, {|a| ( a[ 1 ] == 0 ) } ) ) == 0
		_fnSetErr( ENOFREE_SLOTS )
		RETURN lResult
	ENDIF

	aConnTable[ nSlotNo ][ 1 ] := 255				// mark this slot used
	aConnTable[ nSlotNo ][ 2 ] := nSlotNo
	aConnTable[ nSlotNo ][ 3 ] := nNetNo			// fill server's address
	aConnTable[ nSlotNo ][ 4 ] := nNodeAddr		//        "
	aConnTable[ nSlotNo ][ 5 ] := nSocketNo		//        "

	aSrvrTable[ nSlotNo ]      := cServerName    // fill server name

	fn_wrConnID( aConnTable )							// write the new tables
	fn_wrFSName( aSrvrTable )

	/* attempt to attach to the server */

	aRegs[ AX ] := ft_Hex2Dec( "F100" )				// Netware Service F1h, Function 00
	aRegs[ DX ] := nSlotNo								//  "AttachToServer"

	lResult := ft_int86( INT21, aRegs )

	IF ( !lResult .OR. ( LowByte( aRegs[ AX ] ) != 0 ) )
		fn_wrConnID( aTempCT )							// on failure restore tables
		fn_wrFSName( aTempST )
		_fnSetErr( LowByte( aRegs[ AX ] ) )
		RETURN lResult
	ENDIF

	/* attach successful, re-order server order No.s */

	aConnTable  := fn_connID()							// refresh connection table
	nOrderNo := 1

	FOR i := 1 TO 8                              // initialize server order no.s
		aConnTable[ i ][ 2 ] = IIF( aConnTable[ i ][ 1 ] == 255, i, 0 )
	NEXT

	FOR i := 1 TO 8                              // main loop to test each server

		IF aConnTable[ i ][ 1 ] != 255				// if slot unused, skip it
			LOOP
		ENDIF

		FOR j := 1 TO 8                           // sub-loop to test remaining servers
			IF ( ( i == j ) .OR. ( aConnTable[ j ][ 1 ] != 255 ) )
            LOOP
			ENDIF

			IF (((  aConnTable[ i ][ 3 ] > aConnTable[ j ][ 3 ] ) .OR.	;
				  (  aConnTable[ i ][ 4 ] > aConnTable[ j ][ 4 ] )).AND. ;
				  (  aConnTable[ i ][ 2 ] < aConnTable[ j ][ 2 ] ))
				  nOrderNo := aConnTable[ i ][ 2 ]  // out of order, reverse order no.s
				  aConnTable[ i ][ 2 ] := aConnTable[ j ][ 2 ]
				  aConnTable[ j ][ 2 ] := nOrderNo
			ENDIF

		NEXT

	NEXT

	fn_wrConnID( aConnTable )                     // write the new table

RETURN( lResult )


/*  $DOC$
 *  $FUNCNAME$
 *      fn_detchFS()
 *  $CATEGORY$
 *      Connection/Workstation
 *  $ONELINER$
 *      Detach from file server
 *  $SYNTAX$
 *
 *      fn_detchFS( <cServerName> ) -> lSuccess
 *
 *  $ARGUMENTS$
 *
 *      <cServerName> is the server to detach from.
 *
 *  $RETURNS$
 *
 *      A logical indicating success (.T.), else that the connection does
 *      not exist.  Either way, you're detached.
 *
 *  $DESCRIPTION$
 *
 *      This function detaches the current workstation from the designated
 *      file server.
 *
 *      There is an important difference between detaching and logging out 
 *      of a file server.  Detaching relinquishes the file server connection
 *      number the workstation was using and breaks the connection.  To send
 *      any more requests to that file server, a new attachment must be
 *      created.  Logging out on the other hand, does not relinquish the
 *      connection number; it is retained for continued use.
 *
 *      When this call is made, the shell removes all drives mapped to the
 *      detached file server.
 *
 *      NOTE:  ACCESSING THE NETWORK REQUIRES THAT YOU BE ATTACHED TO
 *      AT LEAST ONE FILE SERVER ON THE NETWORK:  DETACHING FROM ALL FILE
 *      SERVERS RENDERS THE NETWORK PERMANENTLY INACCESSIBLE.  BE CAREFUL
 *      NOT TO DETACH FROM YOUR LAST REMAINING FILE SERVER LEST YOU BE SAD!
 *
 *  $EXAMPLES$
 *
 *      /* detach from a file server */
 *
 *      fn_detchFS( "Sales" )
 *
 *  $SEEALSO$
 *      fn_attchFS()
 *  $INCLUDE$
 *
 *  $END$
 */

FUNCTION fn_detchFS( cServerName )
	LOCAL aRegs[ INT86_MAX_REGS ], lResult := .T.
	LOCAL aSrvrTable, nSlotNo

	aSrvrTable  := fn_FSName()                    // fetch server name table

	/* find our server in the name table */

	IF ( nSlotNo := ASCAN( aSrvrTable, {|a| ( a == cServerName ) } ) ) == 0
		_fnSetErr( ENOT_ATTACHED )
		RETURN ( !lResult )                        // not attached, don't continue
	ENDIF

	aRegs[ AX ] := ft_Hex2Dec( "F101" )   			 // Netware Service F1h, Function 01
	aRegs[ DX ] := nSlotNo                        //  "DetachFromFileServer"

	IF !ft_int86( INT21, aRegs )                  // do the deed
		_fnSetErr( EINT86 )
	ENDIF

RETURN( lResult )
