/*	Copyright (c) 1989 Citadel	*/
/*	   All Rights Reserved    	*/

/* #ident	"@(#)cvtss.c	1.4 - 90/06/22" */

/*#define huge		/* include this definition if NOT using MS-DOS */

#include <blkio.h>	/* ansi compatibility macros */

/* ansi headers */
#include <ctype.h>
#include <errno.h>
/*#include <stddef.h>*/
#include <stdio.h>
/*#include <stdlib.h>*/
/*#include <string.h>*/

/* local headers */
#include "basstr.h"

/* macros */
#define max(a,b)	((a) > (b) ? (a) : (b))
#define min(a,b)	((a) < (b) ? (a) : (b))

/* macro to preserve characters within single and double quotes */
#define QUOTES {							\
	char c = NUL;							\
	if (quotes && (*ps == '\'' || *ps == '\"')) {			\
		c = *ps;						\
		*pt++ = *ps++;						\
		while (pt < ttend) {					\
			*pt++ = *ps;					\
			if (*ps == c || *ps == NUL) break;		\
		}							\
		continue;						\
	}								\
}

/*man---------------------------------------------------------------------------
NAME
     cvtss - convert string

SYNOPSIS
     int cvtss(t, s, m, n)
     char *t;
     const char *s;
     int m;
     int n;

DESCRIPTION
     The cvtss function takes the source string pointed to by s,
     performs the conversions indicated by m, and places the result in
     the target string pointed to by t.  n is the size of the target
     string.  At most n characters are placed in t.

     Values for m are constructed by bitwise OR-ing flags from the
     following list.  These macros are defined in basstr.h.

     CVT_TP               Trim the parity bit.
     CVT_XSP              Discard spaces and tabs.
     CVT_XCTL             Discard control characters.
     CVT_XLEADSP          Discard leading spaces and tabs.
     CVT_1SP              Reduce spaces and tabs to one space.
     CVT_UPPER            Convert lowercase to uppercase.
     CVT_BTOP             Convert [ to ( and ] to ).
     CVT_XTRAILSP         Discard trailing spaces and tabs.
     CVT_QUOTES           Do not alter characters inside single or
                          double quotes except for parity bit
                          trimming.

     cvtss will fail if one or more of the following is true:

     [EINVAL]       t or s is the NULL pointer.
     [EINVAL]       n is less than 0.

DIAGNOSTICS
     Upon successful completion, a value of 0 is returned.  Otherwise,
     a value of -1 is returned, and errno set to indicate the error.

NOTES
     cvtss is adapted directly from the BASIC function CVT$$.

------------------------------------------------------------------------------*/
int cvtss(t, s, m, n)
char *t;
const char *s;
int m;
int n;
{
	char *ts = NULL;	/* temporary source string */
	char *tt = NULL;	/* temporary target string */
	char huge *ttend = NULL;	/* first char past end of tt */
	int ns = 0;		/* size of ts */
	int quotes = 0;		/* preserve quoted characters flag */
	int bit = 0;		/* bit number */
	char huge *ps = NULL;	/* pointer into ts */
	char huge *pt = NULL;	/* pointer into tt */
	int flag = 0;		/* gp flag */

	/* validate arguments */
	if (t == NULL || s == NULL || n < 0) {
		errno = EINVAL;
		return -1;
	}
	if (n < 1) {
		errno = 0;
		return 0;
	}

	/* find size for ts */
	ns = max(strlen(s) + 1, n);

	/* create temporary strings */
	ts = (char *)calloc((size_t)ns, sizeof(*ts));
	if (ts == NULL) {
		errno = ENOMEM;
		return -1;
	}
	tt = (char *)calloc((size_t)n, sizeof(*tt));
	if (tt == NULL) {
		free(ts);
		errno = ENOMEM;
		return -1;
	}
	ttend = tt + n;

	/* initialize ts with s */
	strcpy(ts, s);

	/* perform conversions */
	quotes = m & CVT_QUOTES;	/* set preserve quoted chars flag */
	for (bit = 0; m != 0; ++bit, m >>= 1) {
		if (!(m & 1)) {
			continue;
		}
		switch (bit) {
		case 0:	/* trim the parity bit */
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				*pt++ = *ps & 0x7F;
				if (*ps == NUL) break;
			}
			break;	/* case 0: */
		case 1:	/* discard all spaces and tabs */
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				if (!(*ps == ' ' || *ps == '\t')) {
					*pt++ = *ps;
				}
				if (*ps == NUL) break;
			}
			break;	/* case 1: */
		case 2:	/* discard all control characters */
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				if (*ps == '\t') {
					*pt++ = ' ';
				} else if (!iscntrl(*ps) || *ps == NUL) {
					*pt++ = *ps;
				}
				if (*ps == NUL) break;
			}
			break;	/* case 2: */
		case 3:	/* discard leading spaces and tabs */
			for (ps = ts; *ps == ' ' || *ps == '\t'; ++ps) {
			}
			strncpy(tt, ps, n);
			break;	/* case 3: */
		case 4:	/* reduce spaces and tabs to one space */
			flag = 0;
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				if (flag) {
					if (*ps != ' ' && *ps != '\t') {
						*pt++ = *ps;
						flag = 0;
					}
				} else {
					if (*ps == ' ' || *ps == '\t') {
						*pt++ = ' ';
						flag = 1;
					} else {
						*pt++ = *ps;
					}
				}
				if (*ps == NUL) break;
			}
			break;	/* case 4: */
		case 5:	/* convert lowercase to uppercase */
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				*pt++ = toupper(*ps);
				if (*ps == NUL) break;
			}
			break;	/* case 5: */
		case 6:	/* convert [ to ( and ] to ) */
			for (ps = ts, pt = tt; pt < ttend; ++ps) {
				QUOTES;
				if (*ps == '[') *pt = '(';
				else if (*ps == ']') *pt = ')';
				else *pt = *ps;
				++pt;
				if (*ps == NUL) break;
			}
			break;	/* case 6: */
		case 7:	/* discard trailing spaces and tabs */
			for (ps = ts + strlen(ts) - 1;
				ps >= ts && (*ps == ' ' || *ps == '\t'); ps--) {
			}
			strncpy(tt, ts, n);
			tt[min(n - 1, ps + 1 - ts)] = NUL;
			break;	/* case 7: */
		case 8:	/* do not alter characters inside quotes */
			continue;
			break;	/* case 8: */
		default:
			free(tt);
			free(ts);
			errno = EINVAL;
			return -1;
			break;
		}
		strncpy(ts, tt, ns);
	}

	/* load target string */
	strncpy(t, ts, n);

	/* free temporary strings */
	free(tt);
	free(ts);

	errno = 0;
	return 0;
}
