/*
? ENVED.REX is A DOS Environment Editor.  It displays a window with a copy of
? the master environment.  Standard RXWINDOW editing keys can be used to modify
? the environment variables.  Modified values are updated in the master
? environment.  Personal REXX 1.60 or later is required.
?
? - To delete environment variables, change the value to blanks.
? - Variables with blank values are removed from the environment.
? - ENVED creates a temporary file called "enved.'.  If a TMP=path
?   environment variable if found, the file is created in that directory.
?   Otherwise it is created in the current directory.
? - EXECIO, RXWINDOW, and RXUNLOAD are run.  If a RXDIR=path environment
?   variable is found, these programs are run from that directory.
?   Otherwise the default PATH= is used.
? - Ideas for future enhancements include
?   - handle RXNEWCOM=YES (probably by creating a temporary BAT file)
?   - handle very long environment variables and more than 23 variables
?   - handle blank environment variables
?   - don't create a temporary file if program stack is installed
?   - allow editing of variable names (not just the variables)
?   - make it possible to add an environment variable
?   - an option to sort the environment variable names
 
On-line help is provided by displaying all lines starting with '?' if any
arguments are specified by the user.
 
Please note changes here:
 
Version History:
1.0  - Nico Mak, 09/87 - original version
1.1  - Nico Mak, 10/87 - added logic to insure environment was changed
1.2  - Charles Daney, 11/87 - use linein instread of execio
*/
 
signal on halt
signal on novalue
signal on syntax
 
/* misc initialization and validation */
call initialize arg(1)

/* put output from "set" command into array "env.N" */
'set >' temp_fileid
do i = 1 by 1
    env.i = linein(temp_fileid)
    if env.i = '' then leave
    end
env.0 = i - 1
call dosdel(temp_fileid)
if env.0 > scr_height - 2 then call emsg 'Too many environment variables'
 
/* open window and set border */
start_line = format((scr_height - env.0) / 2, 2, 0)
w = w_open(start_line, 1, env.0 + 2, scr_width)
if w = '' then call emsg 'Error opening a window'
call w_border w, , , , ,color_border
call w_put w, 1, format((scr_width - length(top_header)) / 2, 2, 0), ,
 top_header, , color_prefix
call w_put w, env.0 + 2, ,
 format((scr_width - length(bot_header)) / 2, 2, 0), bot_header, , color_prefix
 
/* setup window with environment variables */
do i = 1 to env.0
   parse var env.i name.i '=' value.i
   call w_put w, i + 1, 2, name.i'=', ,color_prefix
   field_width = scr_width - 3 - length(name.i)
   if length(value.i) > field_width then
      call w_put w, i + 1, 2 + length(name.i) + 1, ,
       ' ... Variable too long for ENVED ...', , color_prefix
   else
      call w_field w, 'value.'i, i + 1, 2 + length(name.i) + 1, ,
       field_width, color_data
end
 
/* make a copy of the original values */
do i = 1 to env.0
   original.i = value.i
end
 
/* display variables and wait for user to finish editing */
call w_read w
call w_close w
w = ''
 
/* update the environment with any changed variables */
changes = 0
do i = 1 to env.0
   if original.i \= value.i then do
      'set' name.i'='value.i
      if dosenv(name.i) \= value.i then
         say 'Unable to set environment variable' name.i
      else
         changes = changes + 1
      end
end
if changes = 0 then say 'Environment unchanged'
else say changes 'environment variable(s) changed'
 
/* unload rxwindow if we loaded it and exit */
call unload
exit 0

/* initialize - misc initialization and validation */
initialize:
   /* inititalize w (window number) and rxwindow_loaded */
   w = ''
   rxwindow_loaded = 0
 
   /* display help info if any arguments were specified */
   if arg(1) \= '' then signal explain
 
   /* validate rexx version and rxnewcom setting */
   parse version . v .
   if v < 1.6 then call emsg 'Personal REXX 1.60 or later required'
   if lower(dosenv('rxnewcom')) = 'yes' then
      call emsg 'ENVED does not support RXNEWCOM=YES'
 
   /* set name of temporary fileid and get rxdir setting */
   parse source with . . my_name .
   temp_fileid = envdir('ramdisk')my_name'.tmp'
   rxdir = envdir('rxdir')
 
   /* load rxwindow if necessary */
   if \fcnpkg('rxwindow') then do
      rxdir'rxwindow /q'
      if \fcnpkg('rxwindow') then call emsg 'Error installing RXWINDOW'
      rxwindow_loaded = 1
      end
 
   /* set misc values for window handling */
   scr_height = 25
   scr_width  = 80
   top_header = 'DOS Environment Editor Version 1.1'
   bot_header = 'Press Enter to make all changes, Esc to abort all changes'
   if pcvideo() = 3 then do
       color_prefix = 15
       color_data = 7
       color_border = 7
       end
   else do
       color_prefix = 14
       color_data = 10
       color_border = 6
       end
   return
 

/* envdir - get directory name from environment - add \ if necessary */
envdir: procedure
   dir = dosenv(arg(1))
   if dir \= '' then
      if substr(dir, length(dir)) \= '\' then dir = dir'\'
   return dir
 
/* unload rxwindow if we loaded it */
unload:
   if rxwindow_loaded then do
      rxdir'rxunload rxwindow'
      if rc \= 0 then say 'RXUNLOAD RXWINDOW failed - rc' rc
      end
   return
 
/* ctrl-break handler */
halt:
   call emsg 'Ctrl-Break detected - aborting'
 
/* emsg - general purpose fatal error message routine */
emsg:
   if w /= '' then call w_close w
   say arg(1)
   call unload
   exit 99
 
/* explain - when user asks for help */
explain:
   do i = 1 to sourceline()
      parse value sourceline(i) with firstcol +1 rest
      if firstcol = '?' then say rest
   end
   exit
 
/* syntax - error handler based on Wally Pro's syntax routine */
syntax:
   do stax = max(1, sigl - 7) to min(sigl + 3, sourceline())
      call scrwrite , 1, '', 80
      say right(word(stax '===>', 1 + (stax = sigl)), 4) sourceline(stax)
   end
   say '===> REXX error' rc 'in line' sigl':' errortext(rc)
   if queued() > 0 then say queued() 'lines queued. Stackstatus=' stackstatus()
   trace '?r'; nop
   call unload
   exit
 
/* novalue - error handler */
novalue:
   call emsg 'novalue error detected on line' sigl



