/*
 * floppycontrol.c
 *
 * This program illustrates the new FDGETEMSTRESH, FDSETMAXERRS, FDGETMAXERRS,
 * FDGETDRVTYP ioctl added by A. Knaff
 */

#include <stdio.h>
#include <sys/types.h>
#include <linux/fd.h>

#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>

struct option longopts[] =
{
 /* { name  has_arg  *flag  val } */
    {"help",       0, 0, 'h'}, /* print help message */
    {"print",      0, 0, 'p'}, /* print current settings */
    {"abort",      1, 0, 'a'}, /* set abortion treshold */
    {"readtrack",  1, 0, 't'}, /* set read track treshold */
    {"track",      1, 0, 't'}, /* set read track treshold */
    {"reporting",  1, 0, 'e'}, /* set error reporting treshold */
    {"flush",      0, 0, 'f'}, /* flush buffer */
    {"drive",      1, 0, 'd'}, /* set drive */
    {"formatend",  0, 0, 'F'}, /* emit format end ioctl */
    {"type",	   0, 0, 'T'}, /* print only drive type */
    { 0, 0, 0, 0 }
};

void print_usage(char *progname)
{
  fprintf(stderr,
	  "Usage: %s [-h] [-p] [-d drive] [-e reporting_treshold] "
	  "[-t readtrack_treshold] [-a abort_treshold] [-f] [-F] [-T]\n",progname);
}


void help(char *progname)
{
  print_usage(progname);
  fprintf(stderr,
 " -h --help        prints this help message\n"
 " -a --abort       set operation abortion treshold (if there are more error\n"
 "                  than this treshold, the driver gives up to read/write the\n"
 "                  requested sector and reports an I/O error to the user\n"
 "                  program)\n"
 " -t --readtrack   set read track treshold (if there are less errors than\n"
 "                  this treshold, an entire track is read at once)\n"
 " -e --reporting   set error reporting treshold (if there are more errors\n"
 "                  than this treshold, error messages are printed to the\n"
 "                  console)\n"
 " -p --print       print the current error tresholds and the drive type\n"
 " -f --flush       flush the floppy buffers\n"
 " -d --drive       specify the target drive (default is /dev/fd0)\n"
 " -F --formatend   emits an FDFMTEND ioctl. This is needed when, after\n"
 "                  interrupting a formatting program, the drive light\n"
 "                  stays on\n"
 " -T --type        prints the drive type\n"
 );
  exit(0);
}


int eioctl(int fd, int command,void * param, char *emsg)
{
  int r;
  if ((r=ioctl(fd,command,param))<0 )
    {
      perror(emsg);
      exit(1);
    }
  return r;
}

void main( int argc, char **argv)
{
  int fd;
  struct floppy_max_errors max_errors;
  short reporting_treshold;

  int ch;
  int printflag=0;
  int print_type_flag=0;
  char *drive;
  extern char *optarg;
  char *progname = argv[0];
  char drivtyp[17];

  char set_drive=0, set_abort=0;
  char set_reporting_treshold=0, set_readtrack=0,do_flush=0,do_fmtend=0;
  short new_reporting_treshold=0;
  int new_abort=0, new_readtrack=0;

  drive = "/dev/fd0";
  drivtyp[17]='\0';

  while ((ch = getopt_long (argc, argv, "hpfFd:e:t:a:T", 
			    longopts, (int *)0)) != EOF) 
    {
      switch(ch) {
      case 'h':
	help(progname);
      case 'f':
	do_flush=1;
	break;
      case 'F':
	do_fmtend=1;
	break;
      case 'p' :
	printflag = 1;
	break;
      case 'T':
	print_type_flag = 1;
	break;
      case 'd':
	set_drive = 1;
	drive = optarg;
	break;
      case 'e':
	/* message printing treshold */
	set_reporting_treshold = 1;
	new_reporting_treshold = strtoul(optarg,0,0)& 0xffff ;
	break;
      case 't':
	/* read track treshold */
	set_readtrack = 1;
	new_readtrack = strtoul(optarg,0,0);
	break;
      case 'a':
	/* abort treshold */
	set_abort = 1;
	new_abort = strtoul(optarg,0,0);
	break;       	
      case '?':
      default:
	print_usage(progname);
	exit(1);
      }
    }
 	
  /* default drive */
  if ( optind < argc - 1 + set_drive){
    print_usage(progname);
    exit(1);
  }

  if ( optind < argc ){
    set_drive = 1;
    drive = argv[optind];
  }

  if ( !set_drive )
    fprintf(stderr,"Using default drive %s\n", drive);

  fd = open (drive,3);
  if ( fd < 0 )
    {
      perror(drive);
      exit(1);
    }
  
  if (do_fmtend )
    eioctl(fd,FDFMTEND,0,"format end");

  if (do_flush)
    eioctl(fd,FDFLUSH,0,"flush buffers");

  if ( printflag )
    eioctl(fd,FDGETEMSGTRESH, (void *) &reporting_treshold, 
	   "Get error reporting treshold");
  if ( printflag || print_type_flag )
      eioctl(fd,FDGETDRVTYP,(void *) drivtyp, "Get drive type");

  if ( printflag || set_abort || set_readtrack )
    eioctl(fd,FDGETMAXERRS, (void *) & max_errors, "Get max errors");

  if ( printflag )
    printf("error reporting treshold = %d\n"
	   "abort treshold           = %d\n"
	   "read track treshold      = %d\n"
	   "drive type               = %s\n"
	   "autodetected types       = %d\n",
	   reporting_treshold, 
	   max_errors.abort, 
	   max_errors.read_track,
	   drivtyp,
	   FLOPPY_NR_PROBE);
  if ( print_type_flag )
    printf("%s\n", drivtyp);

  if (set_reporting_treshold && 
      ioctl(fd,FDSETEMSGTRESH, new_reporting_treshold))
      perror("Set error reporting treshold");

  if ( set_abort )
    max_errors.abort = new_abort;

  if ( set_readtrack )
    max_errors.read_track = new_readtrack;

  if ( set_abort || set_readtrack ){
    if ( set_readtrack && new_readtrack < FLOPPY_NR_PROBE )
      fprintf(stderr,
	      "Warning: setting readtrack to a value less than %d may peturb autodetection !\n", FLOPPY_NR_PROBE);
    if ( max_errors.abort - max_errors.read_track < FLOPPY_NR_PROBE)
      fprintf(stderr,
	      "Warning: setting abort to a value less than readtrack + %d may perturb autodetection !\n", FLOPPY_NR_PROBE);      
    eioctl(fd,FDSETMAXERRS, (void *) &max_errors, "Set max errors");
  }


  exit(0);
}

