/*+-------------------------------------------------------------------------
	zcurses.c -- tip file transfer program curses interface
	wht%n4hgf@emory.mathcs.emory.edu

  000000000011111111112222222222333333333344444444445555555550
  012345678901234567890123456789012345678901234567890123456789
00.-[ prog+rev ]-- <dir> ------------------------------------.
01|  ZMODEM_6____  _40_____________________________________  |
02|  File ### of ###: _38__________________________________  |
03|  File position:  _8______ length: _8______  -rwxrwxrwx   |
04|  _55____________________________________________________ | transaction
05|  _55____________________________________________________ | last rx/tx hdr
06|  Comm I/O: rx _8______  tx _8______ bytes                |
07|  Baud rate: _5___ BINARY blklen: _____ comm mode: RAW-g  |
08|  Time:    started: __:__:__ this file: __:__:__ window:  |
09|  __:__:__ elapsed: __:__:__            __:__:__ ________ |
10|  Errors: this file: _3_ total: _4__ files skipped: _3_   |
11|  _55____________________________________________________ |  err str
12|  _55____________________________________________________ |  comment str
13|  _55____________________________________________________ |  remote info
14`----------------------------------------------------------'

  Defined functions:
	clear_area(win,row,col,len)
	clear_area_char(win,row,col,len,fillchar)
	get_elapsed_time(elapsed_seconds)
	get_tod(type,tod)
	mode_map(mode,mode_str)
	report_comm_baud_rate(baud_rate)
	report_error_count()
	report_file_byte_io(count)
	report_file_close()
	report_file_open_length(length)
	report_file_open_mode(file_mode)
	report_file_open_tod()
	report_file_rcv_started(filename,length,last_mod_time,file_mode)
	report_file_send_open(filename,filestat)
	report_init(title)
	report_last_rxhdr(rptstr,error_flag)
	report_last_txhdr(rptstr,error_flag)
	report_mode(mode)
	report_protocol_crc_type(str)
	report_protocol_type(str)
	report_rx_ind(status)
	report_rx_tx_count()
	report_rxblklen(blklen)
	report_rxpos(rxpos)
	report_str(rptstr,error_flag)
	report_top_line(topstr)
	report_transaction(str)
	report_tx_ind(status)
	report_txblklen(blklen)
	report_txpos(txpos)
	report_uninit(sig)
	report_window()
	report_xfer_mode(str)

------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:08-23-1990-13:55-wht@tridom-enable rx/tx ind */
/*:05-21-1990-16:00-wht@tridom-adapt ecu xfer protocols for tipwht */

#include <curses.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>
#include <sys/timeb.h>

#include "zlint.h"

long time();
extern char *tzname[];
struct tm *localtime();

#ifdef IBMPC
#define sTL		0xDA
#define sTR		0xBF
#define sBL		0xC0
#define sBR		0xD9
#define sLT		0xC3	/* left hand T */
#define sRT		0xB4	/* right hand T */
#define sVR		0xB3	/* vertical rule */
#define sHR		0xC4	/* horizontal rule */
#else
#define sTL		'.'
#define sTR		'.'
#define sBL		'`'
#define sBR		'\''
#define sLT		'+'
#define sRT		'+'
#define sVR		'|'
#define sHR		'-'
#endif

#define WIN_LINES	15
#define WIN_COLS	60
#define WIN_TOPY	2
#define WIN_LEFTX	8

extern char curr_dir[];
extern char s128[];
extern char *bottom_label;
extern int Filcnt;
extern int tipsz_flag;	/* tipsz == 1, tiprz == 0 */
extern int skip_count;
extern int npats;
extern long rxpos;
extern int log_packets;
extern long Txpos;
extern long Rxpos;

WINDOW	*win;
int (*original_sigint_handler)();
int (*original_sigquit_handler)();
int (*original_sigterm_handler)();
int curses_installed = 0;		/* curses not yet active */
int this_file_errors;
int total_errors;
int show_window = 0;
long current_seconds = 0;
long start_seconds = 0;
long this_file_start_seconds = 0;
long elapsed_seconds = 0;
unsigned long total_data_chars_xfered = 0L;
unsigned int zcurses_baud_rate = 0;
char s256[256];

char *win_template[] =
{
/*00000000001111111111222222222233333333334444444444555555555 */
/*01234567890123456789012345678901234567890123456789012345678 */
/*.----------------------------------------------------------. */
  "                                                          ",	/* 1 */
  "  File ### of ###: _____________________________________  ",	/* 2 */
  "  File position:  ________ length: ________               ",	/* 3 */
  "                                                          ",	/* 4 */
  "  tx: ______________________  rx: ______________________  ",	/* 5 */
  "  Comm I/O: rx ________  tx ________ bytes                ",	/* 6 */
  "  Baud rate: _____ ______ blklen: _____ comm mode: ______ ",	/* 7 */
  "  Time:    started: __:__:__ this file: __:__:__          ", /* 8 */
  "  __:__:__ elapsed: __:__:__            __:__:__          ", /* 9 */
  "  Errors: this file: ___ total: ____ files skipped: ___   ", /* 10 */
  "                                                          ",	/* 11 */
  "                                                          ",	/* 12 */
  "                                                          ",	/* 13 */
/*`----------------------------------------------------------' */
(char *)0
};

/*+-----------------------------------------------------------------------
	char *get_elapsed_time(elapsed_seconds)
	hh:mm:ss returned
  static string address is returned
------------------------------------------------------------------------*/
char *get_elapsed_time(elapsed_seconds)
long elapsed_seconds;
{
	static char elapsed_time_str[10];
	long hh,mm,ss;

	hh = elapsed_seconds / 3600;
	elapsed_seconds -= hh * 3600;
	mm = elapsed_seconds / 60L;
	elapsed_seconds -= mm * 60L;
	ss = elapsed_seconds;

	sprintf(elapsed_time_str,"%02ld:%02ld:%02ld",hh,mm,ss);
	return(elapsed_time_str);
}	/* end of get_elapsed_time */

/*+-----------------------------------------------------------------------
	char *get_tod(type,tod)

  time of day types:
	0		hh:mm
	1		hh:mm:ss
	2		mm-dd-yyyy hh:mm

  static string address is returned
  if tod != (char *)0, time is returned there too
------------------------------------------------------------------------*/
char *
get_tod(type,tod)
int type;
char *tod;
{
	long cur_time = 0;
	struct tm *lt;			/* local time */
	static char tod_str[32];
#if defined(M_SYS5)
	struct timeb tp;
#endif

	cur_time = time((long *)0);
	lt = localtime(&cur_time);

	switch(type)
	{
	case 0:
		sprintf(tod_str,"%02d:%02d",lt->tm_hour,lt->tm_min);
		break;

	default:
	case 1:
		sprintf(tod_str,"%02d:%02d:%02d",lt->tm_hour,lt->tm_min,lt->tm_sec);
		break;

	case 2:
		sprintf(tod_str,"%02d-%02d-%04d %02d:%02d",
		    lt->tm_mon + 1,lt->tm_mday,lt->tm_year + 1900,
		    lt->tm_hour,lt->tm_min);
		break;
	}

	if(tod != (char *)0)
		strcpy(tod,tod_str);

	return(tod_str);
}	/* end of get_tod */

/*+-----------------------------------------------------------------------
	mode_map(mode,mode_str)	build drwxrwxrwx string
------------------------------------------------------------------------*/
char *
mode_map(mode,mode_str)
unsigned short mode;
char *mode_str;
{
register unsigned ftype = mode & S_IFMT;
register char *rtn;
static char result[12];

	rtn = (mode_str == (char *)0) ? result : mode_str;

	/*               drwxrwxrwx */
	/*               0123456789 */
	strcpy(rtn,"----------");

	switch(ftype)
	{
		case S_IFIFO:	*rtn = 'p'; break; /* FIFO (named pipe) */
		case S_IFDIR:	*rtn = 'd'; break; /* directory */
		case S_IFCHR:	*rtn = 'c'; break; /* character special */
		case S_IFBLK:	*rtn = 'b'; break; /* block special */
		case S_IFREG:	*rtn = '-'; break; /* regular */

#if defined(V7) | defined(BSD4)
		case S_IFLNK:	*rtn = 'l'; break; /* symbolic link */
		case S_IFSOCK:	*rtn = 's'; break; /* socket */
#endif

#if defined(M_SYS5)
		case S_IFNAM:						/* name space entry */
			if(mode & S_INSEM)				/* semaphore */
			{
				*rtn = 's';
				break;
			}
			if(mode & S_INSHD)				/* shared memory */
			{
				*rtn = 'm';
				break;
			}
#endif

		default:		*rtn = '?'; break;	/* ??? */
	}

	if(mode & 000400) *(rtn + 1) = 'r';
	if(mode & 000200) *(rtn + 2) = 'w';
	if(mode & 000100) *(rtn + 3) = 'x';
	if(mode & 004000) *(rtn + 3) = 's';
	if(mode & 000040) *(rtn + 4) = 'r';
	if(mode & 000020) *(rtn + 5) = 'w';
	if(mode & 000010) *(rtn + 6) = 'x';
	if(mode & 002000) *(rtn + 6) = 's';
	if(mode & 000004) *(rtn + 7) = 'r';
	if(mode & 000002) *(rtn + 8) = 'w';
	if(mode & 000001) *(rtn + 9) = 'x';
	if(mode & 001000) *(rtn + 9) = 't';

	return(rtn);

}	/* end of mode_map */

/*+-------------------------------------------------------------------------
	clear_area(win,row,col,len)
--------------------------------------------------------------------------*/
clear_area(win,row,col,len)
WINDOW	*win;
int row;
int col;
int len;
{
	wmove(win,row,col);
	while(len-- > 0)
		waddch(win,' ');
	wmove(win,row,col);

}	/* end of clear_area */

/*+-------------------------------------------------------------------------
	clear_area_char(win,row,col,len,fillchar)
--------------------------------------------------------------------------*/
clear_area_char(win,row,col,len,fillchar)
WINDOW	*win;
int row;
int col;
int len;
char fillchar;
{
	wmove(win,row,col);
	while(len-- > 0)
		waddch(win,fillchar);
	wmove(win,row,col);

}	/* end of clear_area_char */

/*+-------------------------------------------------------------------------
	report_top_line(topstr)
   top line: row 1 col 17 length 42
--------------------------------------------------------------------------*/
void
report_top_line(topstr)
char *topstr;
{
char s42[42];
	clear_area(win,1,17,42);
	if(strlen(topstr) < 40)
		waddstr(win,topstr);
	else
	{
		strncpy(s42,topstr,40);
		s42[40] = 0;
		waddstr(win,s42);
	}
}	/* end of report_top_line */

/*+-------------------------------------------------------------------------
	report_xfer_mode(modestr)  BINARY/ASCII
   protocol xfer type: row 7 col 20 length 6
--------------------------------------------------------------------------*/
report_xfer_mode(str)
char *str;
{
char s10[10];

	if(strlen(str) > 6)
	{
		strncpy(s10,str,6);
		s10[7] = 0;
		str = s10;
	}
	clear_area(win,7,20,6);
	waddstr(win,str);
	wrefresh(win);

}	/* end of report_xfer_mode */

/*+-------------------------------------------------------------------------
	report_protocol_type(str)

  protocol type:  row 1 col 3 length 6 string
--------------------------------------------------------------------------*/
report_protocol_type(str)
register char *str;
{
char s10[10];

	if(strlen(str) > 6)
	{
		strncpy(s10,str,6);
		s10[7] = 0;
		str = s10;
	}
	clear_area(win,1,3,6);
	waddstr(win,str);
	wrefresh(win);

}	/* end of report_protocol_type */

/*+-------------------------------------------------------------------------
	report_protocol_crc_type(str)

  protocol crc type:  row 1 col 9 length 6
--------------------------------------------------------------------------*/
report_protocol_crc_type(str)
register char *str;
{
char s8[8];

	if(strlen(str) > 6)
	{
		strncpy(s8,str,6);
		s8[7] = 0;
		str = s8;
	}
	clear_area(win,1,9,6);
	waddstr(win,str);
	wrefresh(win);

}	/* end of report_protocol_crc_type */

/*+-------------------------------------------------------------------------
	report_uninit(sig)
--------------------------------------------------------------------------*/
void
report_uninit(sig)
int sig;		/* if -1, called by normal code, else kill() value */
{
	float rate = 0.0;
	float eff = 0.0;

	if(curses_installed)
	{
		elapsed_seconds = current_seconds - start_seconds;
		if(elapsed_seconds)
		{
			rate = (float)total_data_chars_xfered / (float)elapsed_seconds;
			if(zcurses_baud_rate)
				eff  = 100 * (rate / ((float)zcurses_baud_rate / 10.0));
		}
		if(rate > 0.01)
		{
			sprintf(s128,"Transfer rate ~= %.0f ch/sec (%.0f%%)",
			    rate,(eff > 0.5) ? eff : 0.0);
			if(log_packets)
			{
				write(log_packets,"info: ",6);
				write(log_packets,s128,strlen(s128));
				write(log_packets,"\n",1);
			}
			report_top_line(s128);
		}
		report_file_byte_io(0L);
		report_rx_tx_count();
		wmove(win,WIN_LINES - 1,WIN_COLS - 1);
		wrefresh(win);
		endwin();
		fprintf(stderr,"\r\n\r\n\r\n");
		fflush(stderr);
		curses_installed = 0;
	}

}	/* end of report_uninit */

/*+-------------------------------------------------------------------------
	report_init(title)
  "top line": row 1 col 11 len 21
  file quan:  row 2 col 15 len  3
              row 2 col 12 len  7 clear "of ###"
  start time: row 8 col 21 len  8
  "window:"   row 8 col 50 len  7
--------------------------------------------------------------------------*/
void
report_init(title)
char *title;
{
	register int itmp;
	register char *cptr;
	char buf[80];

	if(curses_installed)
		return;

	initscr();
	crmode();
	noecho();
	nonl();
	clear();
	curses_installed = 1;
	win = newwin(WIN_LINES,WIN_COLS,WIN_TOPY,WIN_LEFTX);
	box(win,sVR,sHR);
	wmove(win,0,0); waddch(win,sTL);
	wmove(win,win->_maxy - 1,0); waddch(win,sBL);
	wmove(win,win->_maxy - 1,win->_maxx - 1); waddch(win,sBR);
	wmove(win,0,win->_maxx - 1); waddch(win,sTR);
	wmove(win,0,2);
	wstandout(win);
	waddch(win,'[');
	waddch(win,' ');
	waddstr(win,title);
	waddch(win,' ');
	waddch(win,']');
	wstandend(win);
	waddch(win,sHR);
	waddch(win,sHR);
	waddch(win,' ');
	itmp = WIN_COLS - 2 - 7 - strlen(title);
	curr_dir[itmp] = 0;
	waddstr(win,curr_dir);
	waddch(win,' ');
	if(bottom_label)
	{
		strncpy(buf,bottom_label,WIN_COLS - 6);
		buf[WIN_COLS - 6] = 0;
		wmove(win,WIN_LINES - 1,2);
		waddch(win,' ');
		waddstr(win,buf);
		waddch(win,' ');
	}

	itmp = 0;
	while(1)
	{
		if(win_template[itmp] == (char *)0)
			break;
		wmove(win,itmp + 1,1);
		waddstr(win,win_template[itmp++]);
	}
	if(tipsz_flag)
	{
		clear_area(win,2,15,3);
		sprintf(s128,"%3d",npats);
		waddstr(win,s128);
	}
	else	/* tiprz */
	{
		clear_area(win,2,11,8);	/* clear "of ###" */
		waddstr(win,":");
	}

	clear_area(win,1,11,21);
	report_error_count();
	clear_area(win,8,21,8);		/* starting time */
	waddstr(win,get_tod(1,(char *)0));
	start_seconds = time((long *)0);
	current_seconds = start_seconds;

	if(show_window)
	{
		wmove(win,8,50);
		waddstr(win,"window:");
		wmove(win,9,50);
		waddstr(win,"+0");
	}

	wrefresh(win);

}	/* end of report_init */

/*+-------------------------------------------------------------------------
	report_rx_ind(status)
--------------------------------------------------------------------------*/
void
report_rx_ind(status)
int status;
{
	wmove(win,1,54);
	waddch(win,(status) ? 'R' : ' ');
	wmove(win,1,54);
	wrefresh(win);
}	/* end of report_rx_ind */

/*+-------------------------------------------------------------------------
	report_tx_ind(status)
--------------------------------------------------------------------------*/
void
report_tx_ind(status)
int status;
{
	wmove(win,1,56);
	waddch(win,(status) ? 'T' : ' ');
	wmove(win,1,56);
	wrefresh(win);
}	/* end of report_tx_ind */

/*+-------------------------------------------------------------------------
	report_window() - if enable, show open widow size
--------------------------------------------------------------------------*/
void
report_window()
{
	if(show_window)
	{
	long ltmp;
		wmove(win,9,50);
		if((ltmp = (Txpos - Rxpos)) > 999999L)
			waddstr(win,">+999999");
		else if(ltmp < -999999L)
			;
		else
		{
			sprintf(s128,"%+-8ld",ltmp);
			waddstr(win,s128);
			if(log_packets)
			{
				write(log_packets,"window: ",8);
				write(log_packets,s128,strlen(s128));
				write(log_packets,"\n",1);
			}
		}
	}
}	/* end of report_window */

/*+-------------------------------------------------------------------------
	report_rx_tx_count()

  This procedure may be counted upon to perform wrefresh(win)

  rx char count:          row 6 col 16 len 8 unsigned long
  tx char count:          row 6 col 29 len 8 unsigned long
  session elapsed time:   row 9 col 21 len 8
  this file elapsed time: row 9 col 41 len 8
  current tod:            row 9 col  3 len 8
  window:                 row 9 col 50 len 8
--------------------------------------------------------------------------*/
report_rx_tx_count()
{
	extern unsigned long rx_char_count;
	extern unsigned long tx_char_count;

	register char *cptr;

	sprintf(s128,"%8ld",rx_char_count);
	wmove(win,6,16);
	waddstr(win,s128);
	sprintf(s128,"%8ld",tx_char_count);
	wmove(win,6,29);
	waddstr(win,s128);

	/* now time of day */
	wmove(win,9,3);
	cptr = get_tod(1,(char *)0);
	waddstr(win,cptr);
	current_seconds = time((long *)0);
	elapsed_seconds = current_seconds - start_seconds;
	cptr = get_elapsed_time(elapsed_seconds);
	wmove(win,9,21);
	waddstr(win,cptr);
	if(this_file_start_seconds)
		elapsed_seconds = current_seconds - this_file_start_seconds;
	else
		elapsed_seconds = 0;
	cptr = get_elapsed_time(elapsed_seconds);
	wmove(win,9,41);
	waddstr(win,cptr);

	report_window();

	wrefresh(win);		/* calling procs expect this to occur always */

}	/* end of report_rx_tx_count */

/*+-------------------------------------------------------------------------
	report_mode(mode)

 comm mode row 7 col 52 length 6
   3: save old tty stat, set raw mode with flow control
   2: set XON/XOFF for sb/sz with ZMODEM or YMODEM-g
   1: save old tty stat, set raw mode 
   0: restore original tty mode
--------------------------------------------------------------------------*/
void
report_mode(mode)
int mode;
{
char *cptr;
char tmp[8];

	clear_area(win,7,52,6);
	switch(mode)
	{
	case 0:
		cptr = "NORMAL";
		break;
	case 1:
		cptr = "RAW";
		break;
	case 2:
		cptr = "RAW-g";
		break;
	case 3:
		cptr = "RAW-f";
		break;
	default:
		sprintf(tmp,"%5u",mode);
		cptr = tmp;
	}
	waddstr(win,cptr);
	wrefresh(win);
	if(log_packets)
	{
		write(log_packets,"mode: ",6);
		write(log_packets,cptr,strlen(cptr));
		write(log_packets,"\n",1);
	}

}	/* end of report_mode */

/*+-------------------------------------------------------------------------
	report_rxblklen(blklen) 
row 7 col 35 5 chars
--------------------------------------------------------------------------*/
void
report_rxblklen(blklen)
int blklen;
{
	char tmp[10];

	sprintf(tmp,"%5u",blklen);
	clear_area(win,7,35,5);
	waddstr(win,tmp);
	wrefresh(win);
}	/* end of report_rxblklen */

/*+-------------------------------------------------------------------------
	report_txblklen(blklen) 
row 7 col 35 5 chars
--------------------------------------------------------------------------*/
void
report_txblklen(blklen)
int blklen;
{
	report_rxblklen(blklen);
}	/* end of report_txblklen */

/*+-------------------------------------------------------------------------
	report_rxpos(rxpos) row 3 col 19 len 8
--------------------------------------------------------------------------*/
void
report_rxpos(rxpos)
long rxpos;
{
int i;
char tmp[16];
char rdchar;
	if(
#if defined(M_SYS5)
		rdchk(0)
#else
		!ioctl(0,FIONREAD,&i) && i
#endif
			)
	{
		read(0,&rdchar,1);
		rdchar &= 0x7f;
		if(rdchar == 0x0C || rdchar == 0x012)	/* ^L or ^R */
		{
			touchwin(stdscr);
			wrefresh(stdscr);
			touchwin(win);
			wrefresh(win);
		}
	}

	if((rxpos > 99999999L) || (rxpos < 0L))
		return;

	sprintf(tmp,"%8lu",rxpos);
	wmove(win,3,19);
	waddstr(win,tmp);
	wrefresh(win);
	report_rx_tx_count();	/* which will do a refresh */
}	/* end of report_rxpos */

/*+-------------------------------------------------------------------------
	report_txpos(txpos)
--------------------------------------------------------------------------*/
void
report_txpos(txpos)
long txpos;
{
	report_rxpos(txpos);
}	/* end of report_txpos */

/*+-------------------------------------------------------------------------
	report_error_count()
	DOES NOT PERFORM A REFRESH CYCLE

  this file: row 10 col 22 len 3
  total:     row 10 col 33 len 4
  skipped:   row 10 col 53 len 3
--------------------------------------------------------------------------*/
report_error_count()
{
	char tmp[16];

	wmove(win,10,22);
	sprintf(tmp,"%3d",this_file_errors);
	if(this_file_errors)
		wstandout(win);
	waddstr(win,tmp);
	if(this_file_errors)
		wstandend(win);

	wmove(win,10,33);
	sprintf(tmp,"%4d",total_errors);
	if(total_errors)
		wstandout(win);
	waddstr(win,tmp);
	if(total_errors)
		wstandend(win);

	wmove(win,10,53);
	sprintf(tmp,"%3d",skip_count);
	waddstr(win,tmp);
	wrefresh(win);

}	/* end of report_error_count */

/*+-------------------------------------------------------------------------
	report_last_txhdr(rptstr,error_flag)
	5,7,22
--------------------------------------------------------------------------*/
void
report_last_txhdr(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s24[24];

	if(log_packets)
	{
		write(log_packets,"tx:   ",6);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

	if(strlen(rptstr) > 22)
	{
		strncpy(s24,rptstr,22);
		s24[23] = 0;
		rptstr = s24;
	}
	clear_area(win,5,7,22);
	waddstr(win,rptstr);

	if(error_flag)
	{
		++this_file_errors;
		++total_errors;
		report_error_count();
	}

}	/* end of report_last_txhdr */

/*+-------------------------------------------------------------------------
	report_last_rxhdr(rptstr,error_flag)
	5,35,22
--------------------------------------------------------------------------*/
void
report_last_rxhdr(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s24[24];
extern int log_packets;

	if(log_packets)
	{
		write(log_packets,"rx:   ",6);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

	if(strlen(rptstr) > 22)
	{
		strncpy(s24,rptstr,22);
		s24[23] = 0;
		rptstr = s24;
	}
	clear_area(win,5,35,22);
	waddstr(win,rptstr);

	if(error_flag)
	{
		++this_file_errors;
		++total_errors;
		report_error_count();
	}
	report_window();

}	/* end of report_last_rxhdr */

/*+-------------------------------------------------------------------------
	report_str(rptstr,error_flag) row 11/12 col 3 len 55

  error_flag == 0 for status/progress message
             == 1 for bump error count, unless rptstr is null
                  in which case, merely clear error string area
             == 2 write string on bottom line (not an error)
             == 3 write string on transaction line (not an error)
             == -1 use error line but do not bump error count
--------------------------------------------------------------------------*/
void
report_str(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s60[60];
extern int log_packets;

	if(strlen(rptstr) > 55)
	{
		strncpy(s60,rptstr,55);
		s60[55] = 0;
		rptstr = s60;
	}

	switch(error_flag)
	{
		case 0:
			clear_area(win,12,3,55);
			break;
		case 1:
			++this_file_errors;
			++total_errors;
			report_error_count();
		case -1:
			clear_area(win,11,3,55);
			break;
		case 2:
			clear_area(win,13,3,55);
			break;
		case 3:
			clear_area(win,4,3,55);
			break;
	}

	waddstr(win,rptstr);
	wrefresh(win);

	if(log_packets)
	{
		write(log_packets,"info: ",6);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

}	/* end of report_str */

/*+-------------------------------------------------------------------------
	report_transaction()
--------------------------------------------------------------------------*/
void
report_transaction(str)
char *str;
{
	report_str(str,3);
}	/* end of report_transaction */

/*+-------------------------------------------------------------------------
	report_file_open_tod() -- time of start of this file

  this file open time: row 8 col 41 length 8
--------------------------------------------------------------------------*/
void
report_file_open_tod()
{
	clear_area(win,8,41,8);
	this_file_start_seconds = time((long *)0);
	waddstr(win,get_tod(1,(char *)0));
	wrefresh(win);
}	/* end of report_file_open_tod */

/*+-------------------------------------------------------------------------
	report_file_open_mode(file_mode)
  mode map: row 4 col 46 len 10
--------------------------------------------------------------------------*/
report_file_open_mode(file_mode)
unsigned short file_mode;
{
	clear_area(win,3,46,10);
	waddstr(win,mode_map(file_mode,(char *)0));
	wrefresh(win);
}	/* end of report_file_open_mode */

/*+-------------------------------------------------------------------------
	report_file_open_length(long_length)
  length:   row 3 col 36 len  8
--------------------------------------------------------------------------*/
report_file_open_length(length)
long length;
{
	clear_area(win,3,36,8);
	if(length <= 0)
		waddstr(win,"unknown");
	else
	{
		sprintf(s128,"%8lu",length);
		waddstr(win,s128);
	}
	wrefresh(win);
}	/* end of report_file_open_length */

/*+-------------------------------------------------------------------------
	report_file_send_open(filename,filestat)

  filename: row 2 col 20 len 38
  number:   row 2 col 8 len 3
  length:   row 3 col 36 len  8
  mode:     row 3 col 46 len 10
  time of start of this file: row 4 col 47 length 8 hh:mm:ss
--------------------------------------------------------------------------*/
void
report_file_send_open(filename,filestat)
char *filename;
struct stat *filestat;
{
char s50[50];
register char *cptr = filename;

	if(log_packets)
	{
		write(log_packets,"file: ",6);
		write(log_packets,filename,strlen(filename));
		write(log_packets,"\n",1);
	}

	/* number */
	clear_area(win,2,8,3);
	sprintf(s50,"%3d",Filcnt);
	waddstr(win,s50);

	/* filename */
	if(strlen(filename) > 38)
	{
		strncpy(s50,filename,38);
		s50[39] = 0;
		cptr = s50;
	}
	clear_area(win,2,20,38);
	waddstr(win,cptr);

	/* length */
	report_file_open_length(filestat->st_size);

	/* mode */
	report_file_open_mode(filestat->st_mode);

	/* time of start of this file */
	report_file_open_tod();

	this_file_errors = 0;
	report_error_count();
}	/* end of report_file_send_open */

/*+-------------------------------------------------------------------------
	report_file_rcv_started(filename,length,last_mod_time,file_mode)

  filenumber: row 2 col  8 len  3
              row 2 col 12 len  7 clear "of ###"
  filename:   row 2 col 20 len 38
--------------------------------------------------------------------------*/
report_file_rcv_started(filename,length,last_mod_time,file_mode)
char *filename;
long length;					/* if < 0, "UNKNOWN" */
long last_mod_time;			/* not currently displayed */
unsigned short file_mode;		/* UNIX file modifier or zero */
{
register char *cptr;
char s50[50];

	if(log_packets)
	{
		write(log_packets,"file: ",6);
		write(log_packets,filename,strlen(filename));
		write(log_packets,"\n",1);
	}

	/* filename */
	if(strlen(filename) > 38)
	{
		strncpy(s50,filename,38);
		s50[39] = 0;
		cptr = s50;
	}
	else
		cptr = filename;

	clear_area(win,2,20,38);
	waddstr(win,cptr);

	/* file number */
	clear_area(win,2,8,3);
	sprintf(s50,"%3d",Filcnt);	/* rz uses as file number 1-n */
	waddstr(win,s50);

/* if remote sender provides a file count, display it */
	if(npats)
	{
		clear_area(win,2,12,7);	/* clear "of ###" */
		sprintf(s50,"of %3d:",npats);
		waddstr(win,s50);
	}

	/* length */
	report_file_open_length(length);

	/* mode */
	report_file_open_mode(file_mode);

	/* time of start of this file */
	report_file_open_tod();

	this_file_errors = 0;
	report_error_count();
}	/* end of report_file_rcv_started */

/*+-------------------------------------------------------------------------
	report_file_close()
--------------------------------------------------------------------------*/
void report_file_close()
{
	if(show_window)
	{
		clear_area(win,9,50,8);
		waddstr(win,"+0");
		Txpos = 0;
		Rxpos = 0;
	}

	report_str("End of file",0);
	wrefresh(win);
	this_file_start_seconds = 0;

}	/* end of report_file_close */

/*+-------------------------------------------------------------------------
	report_comm_baud_rate(baud_rate)

 baud rate: row 7 col 14 length 5
--------------------------------------------------------------------------*/
report_comm_baud_rate(baud_rate)
unsigned int baud_rate;
{
	char tstr8[8];

	zcurses_baud_rate = baud_rate;
	clear_area(win,7,14,5);
	if(baud_rate == 0)
		waddstr(win,"?");
	else

	{
		sprintf(tstr8,"%5u",baud_rate);
		waddstr(win,tstr8);
	}
	wrefresh(win);

}	/* end of report_comm_baud_rate */

/*+-------------------------------------------------------------------------
	report_file_byte_io(count)
--------------------------------------------------------------------------*/
report_file_byte_io(count)
long count;
{

	total_data_chars_xfered += (long)count;
	if(total_data_chars_xfered)
	{
		sprintf(s128,"Total file bytes transferred: %lu",
			total_data_chars_xfered);
		report_str(s128,-1);
	}

}	/* end of report_file_byte_io */

/* end of zcurses.c */
/* vi: set tabstop=4 shiftwidth=4: */
