/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user or with the express written consent of
 * Sun Microsystems, Inc.
 *
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
#if !defined(lint) && defined(SCCSIDS)
static char sccsid[] = "@(#)rpc_generic.c 1.33 91/03/11 Copyr 1988 Sun Micro";
#endif

/*
 * rpc_generic.c, Miscl routines for RPC. 
 *
 */

#include <stdio.h>
#include <rpc/rpc.h>
#include <rpc/nettype.h>
#include <sys/param.h>
#include <ctype.h>
#include <sys/resource.h>
#include <netconfig.h>
#include <malloc.h>

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

static struct handle {
	NCONF_HANDLE *nhandle;
	int nflag;		/* Whether NETPATH or NETCONFIG */
	int nettype;
};

static struct _rpcnettype {
	char *name;
	int type;
} _rpctypelist[] = {
	"netpath", _RPC_NETPATH,
	"visible", _RPC_VISIBLE,
	"circuit_v", _RPC_CIRCUIT_V,
	"datagram_v", _RPC_DATAGRAM_V,
	"circuit_n", _RPC_CIRCUIT_N,
	"datagram_n", _RPC_DATAGRAM_N,
	"tcp", _RPC_TCP,
	"udp", _RPC_UDP,
	0, _RPC_NONE
};

/*
 * Cache the result of getrlimit(), so we don't have to do an
 * expensive call every time.
 */
int
_rpc_dtbsize()
{
	static int tbsize = 0;
	struct rlimit	rl;

	if (tbsize)
		return (tbsize);
	if ( getrlimit(RLIMIT_NOFILE, &rl) == 0 ) 
		return (tbsize = rl.rlim_max);
	/*
	 * Something wrong.  I'll try to save face by returning a
	 * pessimistic number.
	 */
	return (32);
}

/*
 * Cache the result of gethostname(), so we don't have to do an
 * expensive system call every time.
 */
char *
_rpc_gethostname()
{
	static char *host;

	if (host == (char *)NULL) {
		host = (char *)calloc(1, MAXHOSTNAMELEN);
		if (host == (char *)NULL)
			return ((char *)NULL);
		gethostname(host, MAXHOSTNAMELEN);
	}
	return (host);
}

/*
 * Find the appropriate buffer size
 */
u_int
_rpc_get_t_size(size, bufsize)
	int size;	/* Size requested */
	long bufsize;	/* Supported by the transport */
{
	if (bufsize == -2)
		/* transfer of data unsupported */
		return ((u_int)0);
	if (size == 0) {
		if ((bufsize == -1) || (bufsize == 0))
			/*
			 * bufsize == -1 : No limit on the size
			 * bufsize == 0 : Concept of tsdu foreign. Choose
			 *			a value.
			 */
			return ((u_int)MAXTR_BSIZE);
		else
			return ((u_int)bufsize);
	}
	if ((bufsize == -1) || (bufsize == 0))
		return ((u_int)size);
	/* Check whether the value is within the upper max limit */
	return (size > bufsize ? (u_int)bufsize : (u_int)size);
}

/*
 * Find the appropriate address buffer size
 */
u_int
_rpc_get_a_size(size)
	long size;	/* normally tinfo.addr */
{
	if (size >= 0)
		return ((u_int)size);
	if (size <= -2)
		return ((u_int)0);
	/*
	 * (size == -1) No limit on the size. we impose a limit here.
	 */
	return ((u_int)MAXTR_BSIZE);
}

static char *
strlocase(p)
	char *p;
{
	char *t = p;

	for (; *p; p++)
		if (isupper(*p))
			*p = tolower(*p);
	return (t);
}

/*
 * Returns the type of the network as defined in <rpc/nettype.h>
 * If nettype is NULL, it defaults to NETPATH.
 */
static int
getnettype(nettype)
	char *nettype;
{
	int i;

	if ((nettype == NULL) || (nettype[0] == NULL))
		return (_RPC_NETPATH);	/* Default */

	nettype = strlocase(nettype);
	for (i = 0; _rpctypelist[i].name; i++)
		if (strcmp(nettype, _rpctypelist[i].name) == 0)
			return (_rpctypelist[i].type);
	return (_rpctypelist[i].type);
}

/*
 * For the given nettype (tcp or udp only), return the first structure found.
 * This should be freed by calling freenetconfigent()
 */
struct netconfig *
_rpc_getconfip(nettype)
	char *nettype;
{
	char *netid;
	static char *netid_tcp;
	static char *netid_udp;

	if (!netid_udp && !netid_tcp) {
		struct netconfig *nconf;
		extern char *strdup();
		void *confighandle;

		if (!(confighandle = setnetconfig()))
			return (NULL);
		while (nconf = getnetconfig(confighandle)) {
			if (strcmp(nconf->nc_protofmly, NC_INET) == 0) {
				if (strcmp(nconf->nc_proto, NC_TCP) == 0)
					netid_tcp = strdup(nconf->nc_netid);
				else if (strcmp(nconf->nc_proto, NC_UDP) == 0)
					netid_udp = strdup(nconf->nc_netid);
			}
		}
		endnetconfig(confighandle);
	}
	if (strcmp(nettype, "udp") == 0)
		netid = netid_udp;
	else if (strcmp(nettype, "tcp") == 0)
		netid = netid_tcp;
	else
		return ((struct netconfig *)NULL);
	if ((netid == NULL) || (netid[0] == NULL))
		return ((struct netconfig *)NULL);
	return (getnetconfigent(netid));
}


/*
 * Returns the type of the nettype, which should then be used with
 * _rpc_getconf().
 */
void *
_rpc_setconf(nettype)
	char *nettype;
{
	struct handle *handle;

	handle = (struct handle *) malloc(sizeof (struct handle));
	if (handle == NULL)
		return (NULL);
	switch (handle->nettype = getnettype(nettype)) {
	case _RPC_NETPATH:
	case _RPC_CIRCUIT_N:
	case _RPC_DATAGRAM_N:
		if (!(handle->nhandle = setnetpath())) {
			free(handle);
			return (NULL);
		}
		handle->nflag = TRUE;
		break;
	case _RPC_VISIBLE:
	case _RPC_CIRCUIT_V:
	case _RPC_DATAGRAM_V:
	case _RPC_TCP:
	case _RPC_UDP:
		if (!(handle->nhandle = setnetconfig())) {
			free(handle);
			return (NULL);
		}
		handle->nflag = FALSE;
		break;
	default:
		return (NULL);
	}

	return (handle);
}

/*
 * Returns the next netconfig struct for the given "net" type.
 * _rpc_setconf() should have been called previously.
 */
struct netconfig *
_rpc_getconf(handle)
	struct handle *handle;
{
	struct netconfig *nconf;

	if (handle == NULL)
		return (NULL);
	while (1) {
		if (handle->nflag)
			nconf = getnetpath(handle->nhandle);
		else
			nconf = getnetconfig(handle->nhandle);
		if (nconf == (struct netconfig *)NULL)
			break;
		if ((nconf->nc_semantics != NC_TPI_CLTS) &&
			(nconf->nc_semantics != NC_TPI_COTS) &&
			(nconf->nc_semantics != NC_TPI_COTS_ORD))
			continue;
		switch (handle->nettype) {
		case _RPC_VISIBLE:
			if (!(nconf->nc_flag & NC_VISIBLE))
				continue;
			/* falls through */
		case _RPC_NETPATH:	/* Be happy */
			break;
		case _RPC_CIRCUIT_V:
			if (!(nconf->nc_flag & NC_VISIBLE))
				continue;
			/* falls through */
		case _RPC_CIRCUIT_N:
			if ((nconf->nc_semantics != NC_TPI_COTS) &&
				(nconf->nc_semantics != NC_TPI_COTS_ORD))
				continue;
			break;
		case _RPC_DATAGRAM_V:
			if (!(nconf->nc_flag & NC_VISIBLE))
				continue;
			/* falls through */
		case _RPC_DATAGRAM_N:
			if (nconf->nc_semantics != NC_TPI_CLTS)
				continue;
			break;
		case _RPC_TCP:
			if (((nconf->nc_semantics != NC_TPI_COTS) &&
		  	   (nconf->nc_semantics != NC_TPI_COTS_ORD)) ||
				strcmp(nconf->nc_protofmly, NC_INET) ||
				strcmp(nconf->nc_proto, NC_TCP))
				continue;
			break;
		case _RPC_UDP:
			if ((nconf->nc_semantics != NC_TPI_CLTS) ||
				strcmp(nconf->nc_protofmly, NC_INET) ||
				strcmp(nconf->nc_proto, NC_UDP))
				continue;
			break;
		}
		break;
	}
	return (nconf);
}

void
_rpc_endconf(handle)
	struct handle *handle;
{
	if (handle == NULL)
		return;
	if (handle->nflag) {
		endnetpath(handle->nhandle);
	} else {
		endnetconfig(handle->nhandle);
	}
	free(handle);
}

/*
 * Used to ping the NULL procedure for clnt handle.
 * Returns NULL if fails, else a non-NULL pointer.
 */
void *
rpc_nullproc(clnt)
	CLIENT *clnt;
{
	static char res;
	struct timeval TIMEOUT;

	TIMEOUT.tv_sec = 25;
	TIMEOUT.tv_usec = 0;
	if (clnt_call(clnt, NULLPROC, xdr_void, (char *)NULL,
		xdr_void, (char *)NULL, TIMEOUT) != RPC_SUCCESS)
		return (NULL);
	return ((void *)&res);
}
