/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user or with the express written consent of
 * Sun Microsystems, Inc.
 *
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
#if !defined(lint) && defined(SCCSIDS)
static char sccsid[] = "@(#)clnt_generic.c 1.38 91/03/11 Copyr 1988 Sun Micro";
#endif
/*
 * Copyright (C) 1987, Sun Microsystems, Inc.
 */

#include <stdio.h>
#include <rpc/rpc.h>
#include <sys/errno.h>
#ifdef SYSLOG
#include <sys/syslog.h>
#else
#define LOG_ERR 3
#endif
#include <rpc/nettype.h>

extern int errno;
extern int t_errno;
extern char *strdup();

/*
 * Generic client creation with version checking the value of
 * vers_out is set to the highest server supported value
 * vers_low <= vers_out <= vers_high  AND an error results
 * if this can not be done.
 */
CLIENT *
clnt_create_vers(hostname, prog, vers_out, vers_low, vers_high, nettype)
	char *hostname;
	unsigned prog;
	unsigned *vers_out;
	unsigned vers_low;
	unsigned vers_high;
	char *nettype;
{
	CLIENT *clnt;
	struct timeval to;
	enum clnt_stat rpc_stat;
	struct rpc_err rpcerr;

	clnt = clnt_create(hostname, prog, vers_high, nettype);
	if (clnt == NULL)
		return (NULL);
	to.tv_sec = 10;
	to.tv_usec = 0;
	rpc_stat = clnt_call(clnt, NULLPROC, xdr_void, (char *) NULL,
		    xdr_void, (char *) NULL, to);
	if (rpc_stat == RPC_SUCCESS) {
		*vers_out = vers_high;
		return (clnt);
	}
	if (rpc_stat == RPC_PROGVERSMISMATCH) {
		int minvers, maxvers;
		struct netconfig *nconf;

		clnt_geterr(clnt, &rpcerr);
		minvers = rpcerr.re_vers.low;
		maxvers = rpcerr.re_vers.high;
		if (maxvers < vers_high)
			vers_high = maxvers;
		if (minvers > vers_low)
			vers_low = minvers;
		if (vers_low > vers_high) {
			goto error;
		}
		nconf = getnetconfigent(clnt->cl_netid);
		if (nconf == NULL) {
			/* Use the same error numbers. Nothing appropriate */
			goto error;
		}
		clnt_destroy(clnt);
		clnt = clnt_tp_create(hostname, prog, vers_high, nconf);
		freenetconfigent(nconf);
		if (clnt == NULL)
			return (NULL);
		rpc_stat = clnt_call(clnt, NULLPROC, xdr_void, (char *) NULL,
			    xdr_void, (char *) NULL, to);
		if (rpc_stat == RPC_SUCCESS) {
			*vers_out = vers_high;
			return (clnt);
		}
	}
	clnt_geterr(clnt, &rpcerr);
error:	rpc_createerr.cf_stat = rpc_stat;
	rpc_createerr.cf_error = rpcerr;
	clnt_destroy(clnt);
	return (NULL);
}

/*
 * Top level client creation routine.
 * Generic client creation: takes (servers name, program-number, nettype) and
 * returns client handle. Default options are set, which the user can 
 * change using the rpc equivalent of ioctl()'s.
 * 
 * It tries for all the netids in that particular class of netid until
 * it succeeds.
 * XXX The error message in the case of failure will be the one
 * pertaining to the last create error.
 *
 * It calls clnt_tp_create();
 */
CLIENT *
clnt_create(hostname, prog, vers, nettype)
	char *hostname;				/* server name */
	u_long prog;				/* program number */
	u_long vers;				/* version number */
	char *nettype;				/* net type */
{
	struct netconfig *nconf;
	CLIENT *clnt = NULL;
	void *handle;

	if ((handle = _rpc_setconf(nettype)) == NULL) {
		rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
		return ((CLIENT *)NULL);
	}
	rpc_createerr.cf_stat = RPC_SUCCESS;
	while (clnt == (CLIENT *)NULL) {
		if ((nconf = _rpc_getconf(handle)) == NULL) {
			if (rpc_createerr.cf_stat == RPC_SUCCESS)
				rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
			break;
		}
		clnt = clnt_tp_create(hostname, prog, vers, nconf);
		if (clnt)
			break;
	}	
	_rpc_endconf(handle);
	return (clnt);
}

/*
 * Generic client creation: takes (servers name, program-number, netconf) and
 * returns client handle. Default options are set, which the user can
 * change using the rpc equivalent of ioctl()'s : clnt_control()
 * It finds out the server address from rpcbind and calls clnt_tli_create()
 */
CLIENT *
clnt_tp_create(hostname, prog, vers, nconf)
	char *hostname;				/* server name */
	u_long prog;				/* program number */
	u_long vers;				/* version number */
	register struct netconfig *nconf;	/* net config struct */
{
	struct netbuf *svcaddr;			/* servers address */
	CLIENT *cl;				/* client handle */
	struct t_bind *tbind;			/* bind info */
	int fd;					/* end point descriptor */

	if (nconf == (struct netconfig *)NULL) {
		rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
		return ((CLIENT *)NULL);
	}
	if ((fd = t_open(nconf->nc_device, O_RDWR, NULL)) == -1) {
		rpc_createerr.cf_stat = RPC_TLIERROR;
		rpc_createerr.cf_error.re_terrno = t_errno;
		return ((CLIENT *)NULL);
	}

	/*
	 * Get the address of the server
	 */
	tbind = (struct t_bind *)t_alloc(fd, T_BIND, T_ADDR);
	if (tbind == (struct t_bind *)NULL) {
		rpc_createerr.cf_stat = RPC_SYSTEMERROR;
		rpc_createerr.cf_error.re_errno = errno;
		rpc_createerr.cf_error.re_terrno = t_errno;
		(void) t_close(fd);
		return ((CLIENT *)NULL);
	}
	svcaddr = &tbind->addr;
	if (rpcb_getaddr(prog, vers, nconf, svcaddr, hostname) == FALSE) {
		(void) t_free((char *)tbind, T_BIND);
		/* appropriate error number is set by rpcbind libraries */
		(void) t_close(fd);
		return ((CLIENT *)NULL);
	}
	cl = clnt_tli_create(fd, nconf, svcaddr, prog, vers, 0, 0);
	(void) t_free((char *)tbind, T_BIND);
	if (cl == (CLIENT *)NULL) {
		(void) t_close(fd);
		return ((CLIENT *)NULL);
	}
	/*
	 * The fd should be closed while destroying the handle.
	 */
	(void) CLNT_CONTROL(cl, CLSET_FD_CLOSE, (char *)NULL);
	return (cl);
}

/*
 * Generic client creation:  returns client handle.
 * Default options are set, which the user can 
 * change using the rpc equivalent of ioctl()'s : clnt_control().
 * If fd is RPC_ANYFD, it will be opened using nconf.
 * It will be bound if not so.
 * If sizes are 0; appropriate defaults will be chosen.
 */
CLIENT *
clnt_tli_create(fd, nconf, svcaddr, prog, vers, sendsz, recvsz)
	register int fd;		/* fd */
	struct netconfig *nconf;	/* netconfig structure */
	struct netbuf *svcaddr;		/* servers address */
	u_long prog;			/* program number */
	u_long vers;			/* version number */
	u_int sendsz;			/* send size */
	u_int recvsz;			/* recv size */
{
	CLIENT *cl = NULL;		/* client handle */
	struct t_info tinfo;		/* transport info */
	bool_t madefd = FALSE;		/* whether fd opened here */

	if (fd == RPC_ANYFD) {
		if (nconf == (struct netconfig *)NULL) {
			rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
			return ((CLIENT *)NULL);
		}
		fd = t_open(nconf->nc_device, O_RDWR, &tinfo);
		if (fd == -1)
			goto err;
		madefd = TRUE;
		if (t_bind(fd, (struct t_bind *)NULL,
			(struct t_bind *)NULL) == -1)
				goto err;
	} else {
		int state;		/* Current state of provider */

		/*
		 * Sync the opened fd.
		 * Check whether bound or not, else bind it
		 */
		if (((state = t_sync(fd)) == -1) ||
		    ((state == T_UNBND) && (t_bind(fd, (struct t_bind *)NULL,
				(struct t_bind *)NULL) == -1)) ||
		    (t_getinfo(fd, &tinfo) == -1))
			goto err;
	}

	switch (tinfo.servtype) {
	case T_COTS:
	case T_COTS_ORD:
		cl = clnt_vc_create(fd, svcaddr, prog, vers, sendsz, recvsz);
		break;
	case T_CLTS:
		cl = clnt_dg_create(fd, svcaddr, prog, vers, sendsz, recvsz);
		break;
	default:
		goto err;
	}

	if (cl == (CLIENT *)NULL)
		goto err;
	if (nconf) {
		cl->cl_netid = strdup(nconf->nc_netid);
		cl->cl_tp = strdup(nconf->nc_device);
	} else {
		cl->cl_netid = "";
		cl->cl_tp = "";
	}
	if (madefd)
		(void) CLNT_CONTROL(cl, CLSET_FD_CLOSE, (char *)NULL);
	return (cl);
err:
	if (madefd)
		(void) t_close(fd);
	rpc_createerr.cf_stat = RPC_TLIERROR;
	rpc_createerr.cf_error.re_errno = 0;
	rpc_createerr.cf_error.re_terrno = t_errno;
	return ((CLIENT *)NULL);
}
