.\" @(#)rpcprog.ms 1.5 91/03/11 TIRPC 1.0
.\"
.\" Must use -- tbl -- with this one
.\"
.ND
.if t .po 1.25i
.\" prevent excess underlining in nroff
.if n .fp 2 R
.OF '''%'
.EF '%'''
.OH '\*(Tl''\&Remote Procedure Call Programming Guide'
.EH '\&Remote Procedure Call Programming Guide''\\*(Tl'
.if !\n%=1 \{\
.br
.bp \}
.SH
\&Remote Procedure Call Programming Guide
.XS
\&Remote Procedure Call Programming Guide
.XE
.IX "RPC Programming Guide"
.LP
.NH 0
\&Introduction
.XS
    \&Introduction
.XE
.LP
This document assumes some knowledge of network theory.  It is
intended for programmers who wish to write network applications using
remote procedure calls (explained below), and who want to understand
the RPC mechanisms usually hidden by the
.I rpcgen(1) 
protocol compiler.
.I rpcgen 
is described in detail in the previous chapter, the
\fI\f(LBrpcgen\fP Programming Guide\fR\. 
.LP
.IX rpcgen "" \fIrpcgen\fP
In most cases, 
.I rpcgen 
and its facilities will be more than adequate to the task at hand, and
the detailed knowledge of the material in this chapter will not be
necessary.  Still, you may want to review this chapter before attempting 
to write a network application, or to convert an existing non-network 
application to run over the network.  The
\fIOne Real Example\fR
section of the
\fI\f(LBrpcgen\fP Programming Guide\fR
contains the complete source for a working RPC service\(ema remote 
directory listing service which uses
.I rpcgen 
to generate XDR routines as well as client and server stubs.
.NH 2
\&Layers of RPC
.IX "layers of RPC"
.IX "RPC" "layers"
.LP
The RPC interface can be seen as being divided into a number of distinct
layers.  The top ones are quite general, and provide for no fine control 
of any kind.  The lower levels \(em three can be usefully distinguished 
\(em are available for use as necessary, and provide increasingly 
detailed levels of control.\**
.FS
For a complete specification of the routines in the RPC library, see the
.I rpc(3N) 
and related manual pages.
.FE
.LP
.I "The Simplified Interface:"
.IX RPC "The simplified interface"
Here, the user doesn't need to consider the characteristics of the 
underlying transport,
.UX ,
or other low-level implementation mechanisms.  
In fact, they don't even have to create client and server communications 
handles.  They simply make remote procedure calls to routines on other 
machines, and need specify only the 
.I type 
of transport that they wish to use.  The selling point here is simplicity.
It's this top layer that allows RPC to pass the \*Qhello world\*U test 
\(em that simple things should be simple.  The top-layer routines are 
used for most applications.
.LP
Included in the simplified interface are only three basic RPC routines:
.IX rpc_reg() "" \fIrpc_reg()\fP
.IP \*(SQ
.B rpc_reg() 
The most fundamental of all the RPC calls, 
.I rpc_reg() 
registers a routine as an RPC routine and obtains a unique, system-wide
procedure-identification number for it.
.IX rpc_call() "" \fIrpc_call()\fR
.IP \*(SQ
.B rpc_call() 
Given such a unique, system-wide procedure-identification number, 
.I rpc_call() 
actually uses it to make a remote call to that routine on a specified
host.
.IX rpc_broadcast() "" \fIrpc_broadcast()\fR
.IP \*(SQ
.B rpc_broadcast() 
Like 
.I rpc_call (),
except that it broadcasts its call message across all networks of the 
specified type.
.LP
.I "The Top Level:"
.IX RPC "The top level"
At the top level the interface is still simple, but the user does have
to create client and server handles before making a call.  Still, like
the routines in the simplified interface, the routines here take a 
.I nettype
argument which specifies only a general class of transports.
.LP
The top level essentially consists of two routines:
.IX clnt_create() "" \fIclnt_create()\fR
.IP \*(SQ
.B clnt_create() 
The generic client creation.  The user tells 
.I clnt_create() 
where the server is located and the type of transport to use to get to it.
.IX svc_create() "" \fIsvc_create()\fR
.IP \*(SQ
.B svc_create()
Creates server handles for all the transports of the specified 
.I nettype .
The user tells 
.I svc_create()
which dispatch function should be used.
.LP
The top levels of RPC, being simple, are also inflexible.  They do not
allow or the choice of a specific transport \(em at these levels, all
routines just take a
.I nettype
argument, which serves to define the class of transport which the user
wishes to use.  On the client side, they involve network selection, and
hence may be slightly inefficient depending on the 
.I nettype .
On the server side, they may have to listen on many transports, and 
hence may waste system resources.  In both of these cases, however,
efficiency can be improved by judicious selection of 
.I NETPATH 
environment variables.  If the programmer wishes the application to
run on all transports, this is the interface that should be used.
.LP
.I "The Intermediate Level:"
.IX RPC "The intermediate level"
The intermediate interface of RPC, and the two even lower
interfaces below it, allow many details to be controlled by the
programmer, and for that reason their use is necessary for special
applications.  Programmers should only go down to the level necessary 
for the control needed.  Programs written at these lower levels are 
also more efficient.
.LP
The intermediate differs from the two levels above it in that it allows
the user to directly specify the transport that they wish to use.  It
consists of two routines:
.IX clnt_tp_create() "" \fIclnt_tp_create()\fR
.IP \*(SQ
.B clnt_tp_create() 
Creates a client handle for a specified network.
.IX svc_tp_create() "" \fIsvc_tp_create()\fR
.IP \*(SQ
.B svc_tp_create()
Likewise,
.I svc_tp_create 
creates a server handle for a specified network.
.LP
.I "The Expert Level:"
.IX "The expert level"
The expert layer consists of a larger set of routines with which
the programmer can specify more parameters, but those parameters are 
still all directly transport related.  It includes the following routines:
.IX clnt_tli_create() "" \fIclnt_tli_create()\fR
.IP \*(SQ
.B clnt_tli_create() 
Creates a client handle for a specified network, allowing fine control
of the server characteristics.
.IX svc_tli_create() "" \fIsvc_tli_create()\fR
.IP \*(SQ
.B svc_tli_create() 
Creates a server handle for a specified network, allowing fine control 
of the client characteristics. 
.IX rpcb_set,() "" \fIrpcb_set,()\fR
.IP \*(SQ
.B rpcb_set()
Provides a programmatic interface to
.I rpcbind ,
one which establishes a mapping between a RPC service and a network 
address.
.IX rpcb_unset() "" \fIrpcb_unset()\fR
.IP \*(SQ
.B rpcb_unset() 
Destroys a mapping of the type established by
.I rpcb_set .
.IX rpcb_getaddr() "" \fIrpcb_getaddr()\fR
.IP \*(SQ
.B rpcb_getaddr() 
Provides a programmatic interface to 
.I rpcbind ,
one which returns the network address of specified RPC service.
.IX svc_reg() "" \fIsvc_reg()\fR
.IP \*(SQ
.B svc_reg() 
Associates a given program and version number pair with a given 
dispatch routine.
.IX svc_unreg() "" \fIsvc_unreg()\fR
.IP \*(SQ
.B svc_unreg()
Destroys an association of the type established by
.I svc_reg .
.LP
.I "The Bottom Level:"
.IX "The bottom level"
The bottom layer consists of routines called when the programmer 
requires full control, even down to the smallest details of transport
options.  It consists of the following routines:
.IX clnt_dg_create() "" \fIclnt_dg_create()\fR
.IP \*(SQ
.B clnt_dg_create() 
Creates a RPC client for the specified remote program, using a 
connectionless transport.
.IX svc_dg_create() "" \fIsvc_dg_create()\fR
.IP \*(SQ
.B svc_dg_create() 
Creates a RPC server handle, using a connectionless transport.
.IX clnt_vc_create() "" \fIclnt_vc_create()\fR
.IP \*(SQ
.B clnt_vc_create() 
Creates a RPC client for the specified remote program, using a 
connection-oriented transport.
.IX svc_vc_create() "" \fIsvc_vc_create()\fR
.IP \*(SQ
.B svc_vc_create() 
Creates a RPC server handle, using a connection-oriented transport.
.NH 1
\&The Higher Layers of RPC
.XS
    \&The Higher Layers of RPC
.XE
.NH 2
\&RPC-Library Based Network Services
.IX "RPC-library based network services"
.LP
Imagine you're writing a program that needs to know how many users 
are logged into a remote machine.  You can do this by calling the RPC 
library routine
.I rnusers()
as illustrated below:
.DS
.ft L
#include <stdio.h>

main(argc, argv)
	int argc;
	char **argv;
{
	int num;
.DE
.DS
.ft L
	if (argc != 2) {
		fprintf(stderr, "usage: rnusers hostname\en");
		exit(1);
	}
	if ((num = rnusers(argv[1])) < 0) {
		fprintf(stderr, "error: rnusers\en");
		exit();
	}
	printf("%d users on %s\en", num, argv[1]);
	exit(0);
}
.DE
RPC library routines such as
.I rnusers()
are in the RPC services library
.I librpcsvc.a
Thus, the program above should be compiled with
.DS
.ft L
example% cc \fBprogram.c -lrpcsvc\fP
.DE
.LP
Here are some of the RPC service library routines available to the
C programmer:
.TS
box tab (&) ;
cfI cfI
lfL l .
Routine&Description
_
.sp.5
rnusers&Return number of users on remote machine
rusers&Return information about users on remote machine
havedisk&Determine if remote machine has disk
rstats&Get performance data from remote kernel
rwall&Write to specified remote machines
.TE
.LP
Other RPC services \(em for example
.I ether()
.I mount
.I rquota()
and
.I spray
\(em are not available to the C programmer as library routines.
They do, however,
have RPC program numbers so they can be invoked with
.I rpc_call()
which will be discussed in the next section.  Most of them also
have compilable
.I rpcgen(1) 
protocol description files.  (The
.I rpcgen
protocol compiler radically simplifies the process of developing
network applications.  See the
\fI\f(LBrpcgen\fP Programming Guide\fR
chapter for detailed information about
.I rpcgen 
and
.I rpcgen 
protocol description files).
.IX "simplified interface to RPC"
.IX "RPC" "simplified interface"
.LP
The simplest interface to the RPC functions is based upon the routines
.I rpc_call
.I rpc_reg
and 
.I rpc_broadcast
which provide direct access the RPC facilities appropriate for programs 
which do not require fine levels of control.  
.LP
Using the simplified interface, the number of remote users can be gotten 
as follows:
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <rpcsvc/rusers.h>

main(argc, argv)
	int argc;
	char **argv;
{
	unsigned long nusers;
	int clnt_stat;
.sp .5
.DE
.DS
.ft L
	if (argc != 2) {
		fprintf(stderr, "usage: nusers hostname\en");
		exit();
	}
.DE
.DS
.ft L
	if (clnt_stat = rpc_call(argv[1],
	  RUSERSPROG, RUSERSVERS, RUSERSPROC_NUM,
	  xdr_void, 0, xdr_u_long, &nusers, "visible") != 0) {
		clnt_perrno(clnt_stat);
		exit(1);
	}
	printf("%d users on %s\en", nusers, argv[1]);
	exit(0);
}
.DE
The simplest way of making remote procedure calls is with the RPC
library routine
.I rpc_call()
It has nine parameters.  The first is the name of the remote server
machine.  The next three parameters are the program, version, and procedure
numbers\(emtogether they identify the remote procedure to be called.  The
fifth and sixth parameters are an XDR filter for encoding and an argument
which has to be passed to the remote procedure.  The next two parameters
are an XDR filter for decoding the results returned by the remote procedure
and a pointer to the place where the procedure's results are to be stored.
Finally, there is the nettype specifier.  Multiple arguments and results
are handled by embedding them in structures.  If
.I rpc_call() 
completes successfully, it returns zero; else it returns a nonzero value.
The return codes (of type
.I "enum clnt_stat"
.IX "enum clnt_stat (in RPC programming)" "" "\fIenum clnt_stat\fP
here cast upon return into an integer) are found in
.I <rpc/clnt.h> .
.LP
Since data types may be represented differently on different machines,
.I rpc_call() 
needs both the type of, and a pointer to, the RPC argument (similarly 
for the result).  For
.I RUSERSPROC_NUM ,
the return value is an
.I "unsigned long"
so
.I rpc_call() 
has
.I xdr_u_long() 
as its first return parameter, which says that the result is of type
.I "unsigned long"
and
.I &nusers 
as its second return parameter,
which is a pointer to where the long result will be placed.  Since
.I RUSERSPROC_NUM 
takes no argument, the argument parameter of
.I rpc_call() 
is
.I xdr_void ().
.LP
After trying a few times to deliver a message, if
.I rpc_call() 
gets no answer, it returns with an error code.  In the example, it tries 
all of the transports listed in 
.I /etc/netconfig 
which are flagged as visible until one of them succeeds.  Methods for 
adjusting the number of retries require you to use the lower layer of 
the RPC library, discussed later in this document.  The remote server 
procedure corresponding to the above might look like this:
.DS
.ft L
char *
nuser()
{
	unsigned long nusers;

.ft I
	/*
	 * Code here to compute the number of users
	 * and place result in variable \fInusers\fP.
	 */
.ft L
	return((char *)&nusers);
}
.DE
.LP
It takes one argument, which is a pointer to the input
of the remote procedure call (ignored in our example),
and it returns a pointer to the result.
In the current version of C,
character pointers are the generic pointers,
so both the input argument and the return value are cast to
.I "char *"
.LP
Normally, a server registers all of the RPC calls it plans to handle, 
and then goes into an infinite loop waiting to service requests.  If 
.I rpcgen() 
is used to provide this functionality, it will generate a lot of code,
including a server dispatch function and support for portmonitors.  
But users can also write servers themselves using
.I rpc_reg (),
and it is appropriate that they do so if they have very simple 
applications, like the one shown as an example here.  In this example,
there is only a single procedure to register, so the main body of the
server would look like this:
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <rpcsvc/rusers.h>

char *nuser();

main()
{

	if (rpc_reg(RUSERSPROG, RUSERSVERS, RUSERSPROC_NUM,
	  nuser, xdr_void, xdr_u_long, "visible")) == -1 {
		fprintf(stderr, "Couldn't Register\en");
		exit(1);
		}
	svc_run();		/* \fINever returns\fP */
	fprintf(stderr, "Error: svc_run returned!\en");
	exit(1);
}
.DE
.LP
The
.I rpc_reg()
routine registers a C procedure as corresponding to a given RPC 
procedure number.  The registration is done for each of the 
transports of the specified type, or if the type parameter is
.I NULL ,
for all the transports named in 
.I NETPATH .
The first three parameters,
.I RUSERPROG ,
.I RUSERSVERS ,
and
.I RUSERSPROC_NUM 
are the program, version, and procedure numbers of the remote 
procedure to be registered;
.I nuser() 
is the name of the local procedure that implements the remote
procedure; and
.I xdr_void() 
and
.I xdr_u_long() 
are the XDR filters for the remote procedure's arguments and
results, respectively.  (Multiple arguments or multiple results
are passed as structures).  The last parameter specifies the
desired nettype.  Note that, when using 
.I rpc_reg (),
the users are not required to write their own dispatch routines.
.SH
.I
NOTE:
.I svc_run() 
is not really one of the simplified-level routines. It is, in 
fact, usable at 
.I all
levels.
.LP
.LP
After registering the local procedure, the server program's
main procedure calls
.I svc_run (),
the RPC library's remote procedure dispatcher.  It is this
function that calls the remote procedures in response to RPC
call messages.  Note that the dispatcher in
.I rpc_reg() 
takes care of decoding remote procedure arguments and encoding 
results, using the XDR filters specified when the remote procedure 
was registered.
.NH 3
\&Passing Arbitrary Data Types
.IX "arbitrary data types"
.LP
In the previous example, the RPC call returns a single
.I "unsigned long"
RPC can handle arbitrary data structures, regardless of
different machines' byte orders or structure layout conventions,
by always converting them to a network standard called
.I "External Data Representation"
(XDR) before
sending them over the wire.
The process of converting from a particular machine representation
to XDR format is called
.I serializing ,
and the reverse process is called
.I deserializing .
The type field parameters of
.I rpc_call() 
and
.I rpc_reg() 
can be a built-in procedure like
.I xdr_u_long() 
in the previous example, or a user supplied one.
XDR has these built-in type routines:
.IX RPC "built-in routines"
.DS
.ft L
xdr_int()      xdr_u_int()      xdr_enum()
xdr_long()     xdr_u_long()     xdr_bool()
xdr_short()    xdr_u_short()    xdr_wrapstring()
xdr_char()     xdr_u_char()
.DE
Note that the routine
.I xdr_string()
exists, but cannot be used with
.I rpc_call() 
and
.I rpc_reg (),
which only pass two parameters to their XDR routines.
.I xdr_wrapstring() 
has only two parameters, and is thus OK.  It calls
.I xdr_string ().
.LP
As an example of a user-defined type routine,
if you wanted to send the structure
.DS
.ft L
struct simple {
	int a;
	short b;
} simple;
.DE
then you would call
.I rpc_call()
as
.DS
.ft L
rpc_call(hostname, PROGNUM, VERSNUM, PROCNUM,
        xdr_simple, &simple ...);
.DE
where
.I xdr_simple()
is written as:
.DS
.ft L
#include <rpc/rpc.h>
#include "simple.h"

bool_t
xdr_simple(xdrsp, simplep)
	XDR *xdrsp;
	struct simple *simplep;
{
	if (!xdr_int(xdrsp, &simplep->a))
		return (FALSE);
	if (!xdr_short(xdrsp, &simplep->b))
		return (TRUE);
	return (TRUE);
}
.DE
.LP
An XDR routine returns nonzero (true in the C sense) if it completes 
successfully, and zero otherwise.  A complete description of XDR is
in the
.I "XDR Protocol Specification"
section of this manual, only few implementation examples are given 
here.  Note that the above routine could have been generated
automatically by using
.I rpcgen ().
.LP
In addition to the built-in primitives, there are also some
prefabricated building blocks:
.DS
.ft L
xdr_array()       xdr_bytes()     xdr_reference()
xdr_vector()      xdr_union()     xdr_pointer()
xdr_string()      xdr_opaque()
.DE
To send a variable array of integers,
you might package them up as a structure like this
.DS
.ft L
struct varintarr {
	int *data;
	int arrlnth;
} arr;
.DE
and make an RPC call such as
.DS
.ft L
rpc_call(hostname, PROGNUM, VERSNUM, PROCNUM,
        xdr_varintarr, &arr...);
.DE
with
.I xdr_varintarr()
defined as:
.DS
.ft L
bool_t
xdr_varintarr(xdrsp, arrp)
	XDR *xdrsp;
	struct varintarr *arrp;
{
	return (xdr_array(xdrsp, &arrp->data, &arrp->arrlnth,
		MAXLEN, sizeof(int), xdr_int));
}
.DE
This routine takes as parameters the XDR handle,
a pointer to the array, a pointer to the size of the array,
the maximum allowable array size,
the size of each array element,
and an XDR routine for handling each array element.
.LP
If the size of the array is known in advance, one can use
.I xdr_vector (),
which serializes fixed-length arrays.
.DS
.ft L
int intarr[SIZE];

bool_t
xdr_intarr(xdrsp, intarr)
	XDR *xdrsp;
	int intarr[];
{
	return (xdr_vector(xdrsp, intarr, SIZE, sizeof(int),
		xdr_int));
}
.DE
.LP
XDR always converts quantities to 4-byte multiples when serializing.
Thus, if either of the examples above involved characters
instead of integers, each character would occupy 32 bits.
That is the reason for the XDR routine
.I xdr_bytes()
which is like
.I xdr_array()
except that it packs characters;
.I xdr_bytes() 
has four parameters, similar to the first four parameters of
.I xdr_array ().
For null-terminated strings, there is also the
.I xdr_string()
routine, which is the same as
.I xdr_bytes() 
without the length parameter.
On serializing it gets the string length from
.I strlen (),
and on deserializing it creates a null-terminated string.
.LP
Here is a final example that calls the previously written
.I xdr_simple() 
as well as the built-in functions
.I xdr_string() 
and
.I xdr_reference (),
which chases pointers:
.DS
.ft L
struct finalexample {
	char *string;
	struct simple *simplep;
} finalexample;
.DE
.DS
.ft L
bool_t
xdr_finalexample(xdrsp, finalp)
	XDR *xdrsp;
	struct finalexample *finalp;
{
	if (!xdr_string(xdrsp, &finalp->string, MAXSTRLEN))
		return (FALSE);
	if (!xdr_reference(xdrsp, &finalp->simplep,
	  sizeof(struct simple), xdr_simple);
		return (FALSE);
	return (TRUE);
}
.DE
Note that we could as easily call
.I xdr_simple() 
here instead of
.I xdr_reference ().
.NH 1
\&The Lower Levels of RPC
.XS
    \&The Lower Levels of RPC
.XE
.IX "lower levels of RPC"
.IX "RPC" "lower levels"
.LP
In the examples given so far, RPC takes care of almost as many details 
as would 
.I rpcgen .
It does so by choosing defaults for almost everything, including the
transport protocol.  In this section, we'll show you how you can control
the details by using lower layers of the RPC library.  It is assumed that 
you are familiar with Transport Level Interface.  
.LP
There are several reasons why you may need to use lower layers of RPC.  
First, you may need to directly control the selection of the transport 
protocol, which at the simplified interface level you can do only by use 
of the 
.I NETPATH 
variable.  Second, you may want to allocate and free memory while 
serializing or deserializing with XDR routines.  There is no facilities
for doing so available at the higher level.  For details, see the
\fIMemory Allocation with XDR\fR
section below.  
.LP
Note that 
.I clnt_call (),
.I clnt_destroy (),
.I clnt_control (),
.I clnt_perrno (),
.I clnt_pcreateerror (),
.I clnt_perror ()
and
.I svc_destroy() 
can be used at any of the lower levels of RPC.  They are not available
at the simplified-interface level only because they require the caller
to have a transport handle.
.NH 2
\&The Top Level
.LP
At the top level, the application can specify the 
.I type 
of transport that it wants to use, but not an individual transport.  It 
differs from the simplified interface to RPC in that the application is 
responsible for creating its own transport handles, on both the client 
and server sides.
.
.NH 3
\&The Client Side
.LP
The following code implements the client side of a trivial date
service, written at the top level.
.DS
.ft L

#include <stdio.h>
#include <rpc/rpc.h>
#include "time_prot.h"

#define TOTAL (30)
.ft I
/*
 *  Caller of trivial date service
 *  usage: calltime hostname
 */
.ft L
main(argc,argv)
	int argc;
	char *argv[];
{
	struct timeval timeout;
	CLIENT *client;
	enum clnt_stat stat;
	struct timev timev;
	char *nettype;

	if (argc != 2 && argc != 3) {
		fprintf(stderr,"usage: %s host [nettype]\en",argv[0]);
	}
.DE
.DS
.ft L
	if (argc == 2)
		nettype = "netpath"; 	/* \fIDefault\fP */	
	else
		nettype = argv[2];
	client = clnt_create(argv[1], TIME_PROG, TIME_VERS,
	  nettype);
	if (client == NULL) {
		clnt_pcreateerror("Couldn't create client");
		exit(1);
	}
.DE
.DS
.ft L
	timeout.tv_sec = TOTAL;
	timeout.tv_usec = 0;
	stat = clnt_call(client, TIME_GET, xdr_void, NULL, 
	  xdr_timev, &timev, timeout);
	if (stat != RPC_SUCCESS) {
		clnt_perror(client, "Call failed");
		exit(1);
	}
	printf("%s: %02d:%02d:%02d GMT\en", nettype, timev.hour,
	  timev.minute, timev.second);
	exit(0);	
}
.DE
Note that, if 
.I nettype 
is not explicitly specified, the code assigns it to point to the 
string \*Qnetpath\*U.  Whenever the routines in the RPC libraries 
encounter this string, they consult the 
.I NETPATH 
environment variable for the user's list of acceptable network
identifiers.  If the client handle cannot be created, the reason 
for the failure can be printed using 
.I clnt_pcreateerror (),
or the error status can be obtained via the global variable 
.I rpc_createerr .
After the client handle is created, 
.I client_call() 
is used to make the actual remote call.  It takes as arguments
the remote procedure number, and XDR filter for the input argument
and the argument pointer, and XDR filter for the result and the 
result pointer, and the time-out period of the call.  Normally, 
this last should not be 0.  In this particular example there are
no arguments, and thus
.I xdr_void()
has been used.
.NH 3
\&The Server Side
.LP
Here's the code for the time server.
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include "time_prot.h"

static void time_prog();

main(argc,argv)
	int argc;
	char *argv[];
{
	int transpnum;
	char *nettype;
.sp .5
.DE
.DS
.ft L
	if (argc == 2)
		nettype = argv[1];
	else
		nettype = "netpath";	/* \fIDefault\fP */
	transpnum = svc_create(time_prog, TIME_PROG, TIME_VERS, 
	  nettype);
	if (transpnum == 0) {
		fprintf(stderr,"%s: cannot create %s service.\en",
		  argv[0], nettype);
		exit(1);
	} 
	svc_run();
}
.DE
.DS
.ft I
/* 
 * The server dispatch function
 */
.ft L
static void
time_prog(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
	struct timev rslt;
	long thetime;
.sp .5
.DE
.DS
.ft L
	switch (rqstp->rq_proc) {
	case NULLPROC:
		svc_sendreply(transp, xdr_void, NULL);
		return;
.sp .5
	case TIME_GET:
		break;
.sp .5
	default:
		svcerr_noproc(transp);
		return;
	}
.DE
.DS
.ft L
	thetime = time(0);
	rslt.second = thetime % 60;
	thetime /= 60;
	rslt.minute = thetime % 60; 
	thetime /= 60;
	rslt.hour = thetime % 24;
	if (! svc_sendreply(transp, xdr_timev, &rslt)) {
		svcerr_systemerr(transp);
	}
}
.DE
.I svc_create() 
returns the number of transports on which it could create server
handles.  
.I time_prog() 
is the dispatch function which is called by
.I svc_run() 
whenever there's a request for its given program and version number.  
In this particular case there were no arguments, otherwise 
.I "svc_getargs (transport,
could be used to deserialize (XDR decode) the arguments.  In such 
cases, 
.I svc_freeargs() 
should be used to free up the arguments after the actual call has 
been made.  The server reply results are sent back to the client using
.I svc_sendreply ().
It is recommended that 
.I rpcgen 
be used to generate the dispatch function which can later be
customized.  Note that 
.I rslt 
is not declared as static because 
.I svc_sendreply() 
is called from within this function.  If 
.I rpcgen 
is used to generate the dispatch function,
.I svc_sendreply() 
is called only after the actual procedure has returned, and hence it
is essential to have 
.I rslt 
declared as static.
.NH 2
\&The Intermediate Level
.LP
At the intermediate level, the application directly choses the
transport which it wishes to use, factoring the value of 
.I NETPATH 
and the contents of 
.I /etc/netconfig 
into its decision as it sees fit.  
.NH 3
\&The Client Side
.LP
The following code implements the client side of the same time
service shown above, but this time written to interface to the
RPC service at the intermediate level.
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>		/* \fIFor netconfig structure\fP */
#include "time_prot.h"
.sp .5
.ft I
/*
 *  Caller of trivial date service
 *  usage: calltime hostname netid
 */
.ft L
main(argc,argv)
	int argc;
	char *argv[];
{
	struct netconfig *nconf;
.sp .5
.DE
.DS
.ft L
	if (argc != 3) {
		fprintf(stderr,"usage: %s host netid\en",argv[0]);
	}
.DE
.DS
.ft L
	netid = argv[2];
	if ((nconf = getnetconfigent(netid)) == NULL) {
		fprintf(stderr, "Illegal netid type %s\en", netid);
		exit(1);
	}
.DE
.DS
.ft L
	client = clnt_tp_create(argv[1], TIME_PROG, TIME_VERS,
	  nconf);
	if (client == NULL) {
		clnt_pcreateerror("Couldn't create client");
		exit(1);
	}
.sp .5
	/* \fISame as previous example after this point\fP */
.sp .5
}
.DE
Here the user has chosen the transport over which the call will
be made.  The
.I netconfig 
structure can be obtained via a call to
.I getnetconfigent (netid).
See
.I netconfig (3N)
for more details.  At this level the user must explicitly make
all decisions about network-selection.
.NH 3
\&The Server Side
.LP
Here's the corresponding server.
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>		/* \fIFor netconfig structure\fP */
#include "time_prot.h"
.sp .5
static void time_prog();
.sp .5
.ft I
/*
 * Service to supply Greenwich mean time
 * usage: server netid ...
 */
.ft L
.sp .5
main(argc,argv)
	int argc;
	char *argv[];
{
	SVCXPRT *transp;
	struct netconfig *nconf;
.sp .5
.DE
.DS
.ft L
	if (argc == 1) {
		fprintf(stderr, "usage: server netid ... \en");
		exit(1);
	}
.DE
.DS
.ft L
	if ((nconf = getnetconfigent(argv[1])) == NULL) {
		fprintf(stderr, "Could not find info on %s\en", 
		  argv[1]);
		exit(1);
	}
	transp = svc_tp_create(time_prog, TIME_PROG, TIME_VERS,
	  nconf);
	if (transp == NULL) {
		fprintf(stderr,"%s: cannot create %s service.\en",
		  argv[0], argv[1]);
		exit(1)
		}
	svc_run();
}
.DE
.DS
.ft L
.sp .5
static void
time_prog(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
.sp .5
	/* \fICode identical to Top Level version\fP */
.sp .5
}
.DE
.LP
.NH 2
\&The Expert Level
.LP
At the expert level, network selection is exactly the same as it is 
at the intermediate level.  The only difference here is in the level of
control which the application has over the details of the transport's
configuration, which is much greater.  These examples demonstrate that
control, which is based upon
.I clnt_tli_create() 
and
.I svc_tli_create ().
.NH 3
\&The Client Side
.LP
Here is the client side of some code that implements a version of
.I clntudp_create() 
(the client-side creation routine for the UDP transport) in terms of 
.I clnt_tli_create ().
It is a real example, and also shows how to do network selection based 
upon the family of the transport one wishes to use.  
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>
#include <netinet/in.h>
.ft I
/*
 * In an earlier implementation of RPC, the only transports supported
 * were TCP/IP and UDP/IP.  Here they are shown based on the Berkeley
 * socket, but implemented on the top of TLI/Streams.
 */
.ft L
CLIENT *
clntudp_create(raddr, prog, vers, wait, sockp)
	struct sockaddr_in *raddr;      /* \fIRemote address\fP */
	u_long prog;                    /* \fIProgram number\fP */
	u_long vers;                    /* \fIVersion number\fP */
	struct timeval wait;            /* \fITime to wait\fP */
	int *sockp;                     /* \fIfd pointer\fP */
{
	CLIENT *cl;                     /* \fIClient handle\fP */
	int madefd = FALSE;             /* \fIIs fd opened here\fP */
	int fd = *sockp;                /* \fIfd\fP */
	struct t_bind *tbind;           /* \fIbind address\fP */
	struct netconfig *nconf;        /* \fInetconfig structure\fP */
.DE
.DS
.ft L
.sp .5
	if (setnetconfig() == 0) {      /* \fINo transports available\fP */
		rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
		return ((CLIENT *)NULL);
	}
.ft I
	/*
	 * Try all the transports till it gets one which is
	 * a connection less, family is INET and name is UDP
	 */
.ft L
	while (nconf = getnetconfig()) {
		if ((nconf->nc_flag & NC_TRANSPORT) &&
		    (nconf->nc_semantics == NC_TPI_CLTS) &&
		    (strcmp(nconf->nc_protofmly, NC_INET) == 0) &&
		    (strcmp(nconf->nc_proto, NC_UDP) == 0))
			break;
	}
.DE
.DS
.ft L
	if (nconf == NULL) {
		rpc_createerr.cf_stat = RPC_UNKNOWNPROTO;
		goto err;
	}
.DE
.DS
.ft L
	if (fd == RPC_ANYSOCK) {
		fd = t_open(nconf->nc_device, O_RDWR, NULL);
		if (fd == -1) {
			rpc_createerr.cf_stat = RPC_SYSTEMERROR;
			goto err;
		}
		madefd = TRUE;          /* \fIThe fd was opened here\fP */
	}
	if (raddr->sin_port == 0) {     /* \fIremote addr unknown\fP */
		u_short sport;
.DE
.DS
.sp .5
.ft I
		/*
		 * rpcb_getport() is a user provided routine
		 * which will call rpcb_getaddr and translate
		 * the netbuf address to port number.
		 */
.ft L
		sport = rpcb_getport(raddr, prog, vers, nconf);
		if (sport == 0) {
			rpc_createerr.cf_stat = RPC_PROGUNAVAIL;
			goto err;
		}
		raddr->sin_port = sport;
	}
.sp .5
.DE
.DS
.ft L
	/* \fITransform sockaddr_in to netbuf\fP */
	tbind = (struct t_bind *)t_alloc(fd, T_BIND, T_ADDR);
	if (tbind == NULL) {
		rpc_createerr.cf_stat = RPC_SYSTEMERROR;
		goto err;
	}
.DE
.DS
.ft L
	(void) memcpy(tbind->addr.buf, (char *)raddr,
	  (int)tbind->addr.maxlen);
	tbind->addr.len = tbind->addr.maxlen;

	/* \fIBind endpoint to a reserved address\fP */
	(void) bind_clnt_resv(fd);
	cl = clnt_tli_create(fd, nconf, &(tbind->addr), prog, 
	  vers, 8800, 8800);
	(void) endnetconfig();	/* \fIClose the netconfig file\fP */
	(void) t_free((char *)tbind, T_BIND);
.DE
.DS
.ft L
	if (cl) {
		*sockp = fd;
		if (madefd == TRUE) {
			/* \fIfd should be closed while destroying the handle\fP */
			(void) CLNT_CONTROL(cl, CLSET_FD_CLOSE, NULL);
		}
		/* \fISet the retry time\fP */
		(void) CLNT_CONTROL(cl, CLSET_RETRY_TIMEOUT, &wait);
		return (cl);
	}
.sp .5
.DE
.DS
.ft L
err:
	if (madefd == TRUE)
		(void) t_close(fd);
	(void) endnetconfig();
	return ((CLIENT *)NULL);
}
.DE
The network selection was done using the library functions
.I setnetconfig (),
.I getnetconfig (),
and
.I endnetconfig ().
Note that 
.I endnetconfig() 
was called only after the call to
.I clnt_tli_create ().
.I clntudp_create() 
can be passed and open
.I fd , 
but it is not it will open its own using the 
.I netconfig 
structure for UDP.  If the remote address is not known,
.I "(raddr->sin_port == 0)" ,
then it is gotten from the remote server.  We then transform
.I sockaddr 
to netbuf/TLI format.  Note the call to 
.I bind_clnt_resv (),
which serves to bind a transport endpoint to a reserved address.  This 
call is necessary because there is no notion of a reserved address in 
RPC under TLI, as there is in both TCP and UDP.  The implementation of 
this routine is of no interest here, because it is entirely transport 
specific.  What is of interest is the context, the scaffolding necessary 
to call it.  
.I clnt_tli_create() 
is normally used to create a client handle when:
.IP  1.
The application must sometimes be passed an open 
.I fd ,
which may or may not be bound.
.IP  2.
The user wants to feed the server's address to the client.
.IP  3.
The user want to specify the send and receive buffer size (8800 bytes in 
this case).
.LP
After the client handle has been created, the user can suitably
customize it using calls to
.I clnt_control ().
In this case, we wanted the RPC library to close the fd while 
destroying the handle (as it usually does when it opens the fd itself, 
and we wanted to set the retry timeout period.
.NH 3
\&The Server Side
.LP
Here's the corresponding server code, which implements 
.I svcudp_create()
in terms of
.I svc_tli_create ().
It calls the user provided
.I bind_resv_svc()
to bind the transport endpoint to a reserved address.
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>
#include <netinet/in.h>
.sp .5
.ft I
/*
 * On the server side
 */
.ft L
SVCXPRT *
svcudp_create(fd)
	register int fd;
{
	struct netconfig *nconf;
	SVCXPRT *svc;
	int madefd = FALSE;
	int port;
.sp .5
.DE
.DS
.ft L
	if (setnetconfig() == 0) {      /* \fINo transports available\fP */
		fprintf(stderr, "No transports available\en");
		return ((SVCXPRT *)NULL);
	}
.ft I
	/*
	 * Try all the transports till it gets one which is
	 * a connection less, family is INET and name is UDP
	 */
.ft L
	while (nconf = getnetconfig()) {
		if ((nconf->nc_flag & NC_TRANSPORT) &&
		    (nconf->nc_semantics == NC_TPI_CLTS) &&
		    (strcmp(nconf->nc_protofmly, NC_INET) == 0) &&
		    (strcmp(nconf->nc_proto, NC_UDP) == 0))
			break;
	}
.DE
.DS
.ft L
	if (nconf == NULL) {
		endnetconfig();
		fprintf(stderr, "UDP transport not available\n");
		return ((SVCXPRT *)NULL);
	}
	if (fd == RPC_ANYFD) {
		fd = t_open(nconf->nc_device, O_RDWR, NULL);
		if (fd == -1) {
			(void) endnetconfig();
			(void) fprintf(stderr,
			"svcudp_create: could not open connection\n");
			return ((SVCXPRT *)NULL);
		}
		madefd = TRUE;
	}
.DE
.DS
.sp .5
.ft I
	/*
	 * Bind Endpoint to a reserved address
	 */
.ft L
	port = bind_resv_svc(fd, 8);    /* \fIqlen is 8\fP */
	svc = svc_tli_create(fd, nconf, (struct t_bind *)NULL,
	  8800, 8800);
	(void) endnetconfig();
	if (svc == (SVCXPRT *)NULL) {
		if (madefd)
			(void) t_close(fd);
		return ((SVCXPRT *)NULL);
	}
.DE
.DS
.ft L
	if (port == -1)
		/* \fISpecifically set xp_port now\fP */
		svc->xp_port = 
		  ((struct sockaddr_in *)svc->xp_ltaddr.buf)->sin_port;
	else
		svc->xp_port = port;
	return (svc);
}
.DE
The network selection here is done in a similar way as in 
.I clntudp_create ().
.I svcudp_create() 
is also set up to receive an open fd, and will open one itself 
using the selected 
.I netconfig 
structure if it does not.
.I bind_resv_svc() 
is a user-provided function which will bind the fd to a reserved 
port if the caller was root.  
.I svc_tli_create() 
is normally used when the user needs a fine degree of control, and
specifically if it is necessary to:
.IP  1.
Pass an open file descriptor (\fIfd\fP) to the application.
.IP  2.
Pass the user's bind address.  Here, for example, 
.I fd
may be unbound when passed in.  If it is, then it is bound to a given 
address, and the address is stored away in the handle.  Here, we have 
set the bind address to 
.I NULL ,
so if the fd is initially unbound it will be bound to any suitable
address.
.IP  3.
Set the send and receive buffer sizes (here being set to 8800 bytes).
.NH 2
\&The Bottom Level
.LP
With the bottom-level interface to RPC, the application can control all
options, transport-related and otherwise.
.I clnt_tli_create (),
and the other expert-level RPC interface routines are implemented on top 
of these bottom-level routines.   The user should not normally be using
these low-level routines.  These routines are normally responsible for
creating their own data structures, their own buffer management, the
creation of their own RPC headers, etc.  Callers of these routines (like
.I clnt_tli_create ()),
are responsible for initializing the
.I cl_netid 
and 
.I cl_tp
fields within the client handle, while the bottom level routines 
.I clnt_dg_create() 
and
.I clnt_vc_create ()
are themselves responsible for filling out the 
.I clnt_ops 
and
.I cl_private
fields.
.I cl_netid 
is the network identifier (e.g. udp) of the transport on which this
handle was created and
.I cl_tp
is the device name of that transport (e.g.
.I /dev/udp ).
.NH 3
\&The Client Side
.LP
All we are showing here is that we can use local variables to control
the exact details of the calls to 
.I clnt_vc_create() 
and
.I clnt_dg_create ().
Thus, these routines allow control of the transport to the very lowest
level.
.DS
.ft L
 	switch (tinfo.servtype) {
		case T_COTS:
		case T_COTS_ORD:
			cl = clnt_vc_create(fd, svcaddr, prog, vers, 
			  sendsz, recvsz);
			break;
		case T_CLTS:
			cl = clnt_dg_create(fd, svcaddr, prog, vers, 
			  sendsz, recvsz);
			break;
		default:
			goto err;
		}
.DE
.NH 3
\&The Server Side
.LP
And, again, on the server side:
.DS
.ft L
	/*
	 * call transport specific function.
	 */
	switch(tinfo.servtype) {
		case T_COTS_ORD:
		case T_COTS:
			xprt = svc_vc_create(fd, sendsz, recvsz);
			break;

		case T_CLTS:
			xprt = svc_dg_create(fd, sendsz, recvsz);
			break;
		default:
			 goto err;
        }
.DE
.NH 1
\&Client and Server Structures
.XS
    \&Client and Server Structures
.XE
.IX RPC "client-side structure"
.IX RPC "server-side structure"
.LP
Here, for your reference, are the client- and server-side RPC handles,
as well as an authentication structure.  
.DS
.ft I
/*
 * Client rpc handle.
 * Created by individual implementations
 * Client is responsible for initializing auth 
 */
.ft L
typedef struct {
    AUTH    *cl_auth;                 /*\fI authenticator \fP*/
    struct clnt_ops {
        enum clnt_stat  (*cl_call)(); /*\fI call remote procedure \fP*/
        void        (*cl_abort)();    /*\fI abort a call \fP*/
        void        (*cl_geterr)();   /*\fI get specific error code \fP*/
        bool_t      (*cl_freeres)();  /*\fI frees results \fP*/
        void        (*cl_destroy)();  /*\fI destroy this structure \fP*/
        bool_t      (*cl_control)();  /*\fI the ioctl() of rpc \fP*/
    } *cl_ops;
    caddr_t         cl_private;       /*\fI private stuff \fP*/
    char            *cl_netid;        /*\fI network token \fP*/
    char            *cl_tp;           /*\fI device name \fP*/
} CLIENT;

.DE
.DS
.ft I
/*
 * Auth handle, interface to client side authenticators.
 */
.ft L
typedef struct {
    struct  opaque_auth     ah_cred;	/* \fIcredentials\fP */
    struct  opaque_auth     ah_verf;	/* \fIverifier\fP */
    union   des_block       ah_key;	/* \fIDES key\fP */
    struct auth_ops {
        void    (*ah_nextverf)();
        int     (*ah_marshal)();        /*\fI nextverf & serialize \fP*/
        int     (*ah_validate)();       /*\fI validate varifier \fP*/
        int     (*ah_refresh)();        /*\fI refresh credentials \fP*/
        void    (*ah_destroy)();        /*\fI destroy this structure \fP*/
    } *ah_ops;
    caddr_t ah_private;                 /* \fIPrivate data\fP */
} AUTH;
.DE
.LP
As you can see, the client-side handle contains an authentication 
structure.  If the user wants to authenticate the client, they must
initialize the
.I cl_auth 
field to an appropriate authentication structure.  Within that 
structure, 
.I ch_cread 
contains the caller's actual credentials, and
.I ah_verf 
contains the information necessary to verify those credentials.
See the 
\fIAuthentication\fR
section for more details.
.DS
.ft I
/*
 * Server side transport handle
 */
.ft L
typedef struct {
    int         xp_fd;
    u_short     xp_port;             /*\fI associated port number \fP*/
    struct xp_ops {
        bool_t      (*xp_recv)();    /*\fI receive incoming requests \fP*/
        enum xprt_stat (*xp_stat)(); /*\fI get transport status \fP*/
        bool_t      (*xp_getargs)(); /*\fI get arguments \fP*/
        bool_t      (*xp_reply)();   /*\fI send reply \fP*/
        bool_t      (*xp_freeargs)();/*\fI free mem allocated for args \fP*/
        void    (*xp_destroy)();     /*\fI destroy this struct \fP*/
    } *xp_ops;
    int         xp_addrlen;     /*\fI length of remote addr. Obsoleted \fP*/
    char        *xp_tp;         /*\fI transport provider device name \fP*/
    char        *xp_netid;      /*\fI network token \fP*/
    struct netbuf   xp_ltaddr;  /*\fI local transport address \fP*/
    struct netbuf   xp_rtaddr;  /*\fI remote transport address \fP*/
    char        xp_raddr[16];   /*\fI remote address. Now obsoleted \fP*/
    struct opaque_auth xp_verf; /*\fI raw response verifier \fP*/
    caddr_t     xp_p1;          /*\fI private: for use by svc ops \fP*/
    caddr_t     xp_p2;          /*\fI private: for use by svc ops \fP*/
    caddr_t     xp_p3;          /*\fI private: for use by svc lib \fP*/
} SVCXPRT;
.DE
.LP
.I xp_fd 
is the file descriptor associated with the handle.  Two or more 
server handles can share the same file descriptor.  
.I xp_netid 
is the network identifier (e.g. udp) of the transport on which this
handle was created and 
.I xp_tp 
is the device name associated with that transport.  
.I xp_ltaddr 
is the server's own bind address, while
.I xp_rtaddr 
is the address of the remote caller and hence may change from call
to call.  
.I xp_netid ,
.I xp_tp 
and
.I xp_ltaddr 
are initialized by 
.I svc_tli_create() 
and other expert-level routines while the rest of the fields are
initialized by the bottom-level server routines
.I svc_dg_create() 
and  
.I svc_vc_create ().
.NH 1
\&Raw RPC
.XS
    \&Raw RPC
.XE
.LP
Finally, there are two pseudo-RPC interface routines which are intended
only for testing purposes.  These routines, 
.I clnt_raw_create() 
and 
.I svc_raw_create (),
don't actually involve the use of any real transport at all.  They exist
to help the developer debug and test the non-communications oriented 
aspects of their application before running it over a real network.
Here's an example of their use:
.DS
.ft I
/*
 * A simple program to increment the number by 1
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <rpc/raw.h>
.sp .5
struct timeval TIMEOUT = {0, 0};
static void server();
.sp .5
main()
{
	CLIENT *cl;
	SVCXPRT *svc;
	int num = 0, ans;
.sp .5
.DE
.DS
.ft L
	if (argc == 2)
		num = atoi(argv[1]);
	svc = svc_raw_create();
	if (svc == NULL) {
		fprintf(stderr, "Could not create server handle\en");
		exit(1);
	}
	svc_reg(svc, 200000, 1, server, 0);
	cl = clnt_raw_create(200000, 1);
	if (cl == NULL) {
		clnt_pcreateerror("raw");
		exit(1);
	}
.DE
.DS
.ft L
	if (clnt_call(cl, 1, xdr_int, &num, xdr_int, &num1,
	  TIMEOUT) != RPC_SUCCESS) {
		clnt_perror(cl, "raw");
		exit(1);
	}
	printf("Client: number returned %d\en", num1);
	exit(0) ;
}

.DE
.DS
.ft L
static void
server(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
	int num;

	switch(rqstp->rq_proc) {
	case 0:
		if (svc_sendreply(transp, xdr_void, 0) == NULL) {
			fprintf(stderr, "error in null proc\n");
			exit (1);
		}
		return;
	case 1:
		break;
	default:
		svcerr_noproc(transp);
		return;
	}
.DE
.DS
.ft L
	if (!svc_getargs(transp, xdr_int, &num)) {
		svcerr_decode(transp);
		return;
	}
	num++;
	if (svc_sendreply(transp, xdr_int, &num) == NULL) {
		fprintf(stderr, "error in sending answer\en");
		exit (1);
	}
	return;
}
.DE
.LP
Note the following points:
.IP  1.
The server is not registered with
.I rpcbind ,
and 
.I svc_run() 
is not called.
.IP  2.
All the RPC calls occur within the same thread of control.
.IP  3.
It is necessary that the server be created before the client.
.IP  4.
.I svc_raw_create() 
takes no parameters.
.IP  5.
The server dispatch routine is the same as it is for normal RPC
servers.
.IP  6.
The last parameter to
.I svc_reg() 
is 0, which means that it will not register with
.I rpcbind .
.NH 1
\&Other RPC Features
.XS
    \&Other RPC Features
.XE
.IX "RPC" "miscellaneous features"
.IX "miscellaneous RPC features"
.LP
This section discusses some other aspects of RPC
that are occasionally useful.
.
.NH 2
\&Select on the Server Side
.IX RPC select() RPC \fIselect()\fP
.IX select() "" \fIselect()\fP "on the server side"
.LP
Suppose a process is processing RPC requests
while performing some other activity.
If the other activity involves periodically updating a data structure,
the process can set an alarm signal before calling
.I svc_run()
But if the other activity
involves waiting on a file descriptor, the
.I svc_run()
call won't work.
The code for
.I svc_run()
is as follows:
.DS
.ft L
void
svc_run()
{
	fd_set readfds;
	extern int errno;
.sp .5
.DE
.DS
.ft L
	for (;;) {
		readfds = svc_fdset;
		switch (select(_rpc_dtbsize(), &readfds,
		  (fd_set *)0, (fd_set *)0, (struct timeval *)0)) {

		case -1:
			if (errno == EINTR) {
				continue;
			}
			(void) syslog(LOG_ERR,
			  "svc_run: select failed: %m");
			return;
		case 0:
			continue;
		default:
			svc_getreqset(&readfds);
		}
	}
}
.DE
.LP
You can bypass
.I svc_run()
and call
.I svc_getreqset()
(the actual dispatcher) yourself.  All you need to know are the file 
descriptors of the transport endpoints associated with the programs you 
are waiting on.  Thus you can have your own
.I select() 
.IX select() "" \fIselect()\fP
that waits on both the RPC file descriptors, and your own descriptors.  
Note that
.I svc_fdset 
is a bit mask of all the file descriptors that RPC is using for services.  
It can change every time that
.I any
RPC library routine is called, because descriptors are constantly being 
opened and closed.
.NH 2
\&Broadcast RPC
.IX "broadcast RPC"
.IX RPC "broadcast"
.LP
.I rpcbind 
is a daemon that converts RPC program numbers into network addresses
comprehensible to any transport provider.  Without it, broadcast RPC 
is impossible.  Here are the main differences between broadcast RPC 
and normal RPC calls:
.IP  1.
Normal RPC expects one answer, whereas broadcast RPC expects many 
answers (one or more answer from each responding machine).
.IP  2.
Broadcast RPC can only be preformed upon connectionless protocols which
support broadcasting.
.IP  3.
The implementation of broadcast RPC treats all unsuccessful responses 
as garbage by filtering them out.  Thus, if there is a version mismatch 
between the broadcaster and a remote service, the user of broadcast RPC
never knows.
.IP  4.
All broadcast messages are sent to 
.I rpcbind 's
network address.  Thus, only services that register themselves with 
.I rpcbind 
are accessible via the broadcast RPC mechanism.
.IP  5.
Broadcast requests are limited in size to the MTU (Maximum Transfer
Unit) of the local network.  For Ethernet, the MTU is 1500 bytes.
.LP
.KS
.NH 3
\&Broadcast RPC Synopsis
.IX "broadcast RPC" synopsis
.IX "RPC" "broadcast synopsis"
.DS
.ft L
#include <rpc/clnt.h>
#include <rpc/rpcb_clnt.h>
	. . .
enum clnt_stat	clnt_stat;
	. . .
clnt_stat = rpc_broadcast(prognum, versnum, procnum,
  inproc, in, outproc, out, eachresult, nettype)
	u_long    prognum;        /* \fIprogram number\fP */
	u_long    versnum;        /* \fIversion number\fP */
	u_long    procnum;        /* \fIprocedure number\fP */
	xdrproc_t inproc;         /* \fIxdr routine for args\fP */
	caddr_t   in;             /* \fIpointer to args\fP */
	xdrproc_t outproc;        /* \fIxdr routine for results\fP */
	caddr_t   out;            /* \fIpointer to results\fP */
	resultproc_t   eachresult;/* \fIcall with each result gotten\fP */
	char      *nettype;       /* \fItransport type\fP */
.DE
.KE
.LP
The procedure
.I eachresult()
is called each time a valid result is obtained.
It returns a boolean that indicates
whether or not the user wants more responses.
.DS
.ft L
bool_t done;
	. . .
done = eachresult(resultsp, raddr, nconf)
	caddr_t resultsp;
	struct netbuf *addr;     /* \fIAddr of responding machine\fP */
	struct netconfig *nconf; /* \fITransport which responded\fP */
.DE
If
.I done
is
.I TRUE ,
then broadcasting stops and
.I rpc_broadcast()
returns successfully.  Otherwise, the routine waits for another response.
The request is rebroadcast after a few seconds of waiting.  If no responses 
come back, the routine returns with
.I RPC_TIMEDOUT
.NH 2
\&Batching
.IX "batching"
.IX RPC "batching"
.LP
The RPC architecture is designed so that clients send a call message,
and wait for servers to reply that the call succeeded.
This implies that clients do not compute
while servers are processing a call.
This is inefficient if the client does not want or need
an acknowledgement for every message sent.
It is possible for clients to continue computing
while waiting for a response,
using RPC batch facilities.
.LP
RPC messages can be placed in a \*Qpipeline\*U of calls
to a desired server; this is called batching.
Batching assumes that:
1) each RPC call in the pipeline requires no response from the server,
and the server does not send a response message; and
2) the pipeline of calls is transported on a reliable
byte stream transport such as TCP/IP.
Since the server does not respond to every call,
the client can generate new calls in parallel
with the server executing previous calls.
Furthermore, the TCP/IP implementation can buffer up
many call messages, and send them to the server in one
.I write()
system call.  This overlapped execution
greatly decreases the interprocess communication overhead of
the client and server processes,
and the total elapsed time of a series of calls.
.LP
Since the batched calls are buffered,
the client should eventually do a nonbatched call
in order to flush the pipeline.
.LP
A contrived example of batching follows.
Assume a string rendering service (like a window system)
has two similar calls: one renders a string and returns void results,
while the other renders a string and remains silent.
The service (using the TCP/IP transport) may look like:
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include "windows.h"
.sp .5
void windowdispatch();
.sp .5
main()
{
	int num;
.sp .5
.DE
.DS
.ft L
	num = svc_create(windowdispatch, WINDOWPROG,
	  WINDOWVERS, "tcp");
	if (transp == NULL){
		fprintf(stderr, "can't create an RPC server\en");
		exit(1);
	}
	svc_run();  /* \fINever returns\fP */
	fprintf(stderr, "should never reach this point\en");
}
.sp .5
.DE
.DS
.ft L
void
windowdispatch(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
	char *s = NULL;
.sp .5
	switch (rqstp->rq_proc) {
	case NULLPROC:
		if (!svc_sendreply(transp, xdr_void, 0))
			fprintf(stderr, "can't reply to RPC call\en");
		return;
	case RENDERSTRING:
		if (!svc_getargs(transp, xdr_wrapstring, &s)) {
			fprintf(stderr, "can't decode arguments\en");
.ft I
			/*
			 * Tell caller he screwed up
			 */
.ft L
			svcerr_decode(transp);
			break;
		}
.ft I
		/*
		 * Code here to render the string \fIs\fP
		 */
.ft L
		if (!svc_sendreply(transp, xdr_void, NULL))
			fprintf(stderr, "can't reply to RPC call\en");
		break;
.DE
.DS
.ft L
	case RENDERSTRING_BATCHED:
		if (!svc_getargs(transp, xdr_wrapstring, &s)) {
			fprintf(stderr, "can't decode arguments\en");
.ft I
			/*
			 * We are silent in the face of protocol errors
			 */
.ft L
			break;
		}
.ft I
		/*
		 * Code here to render string s, but send no reply!
		 */
.ft L
		break;
	default:
		svcerr_noproc(transp);
		return;
	}
.ft I
	/*
	 * Now free string allocated while decoding arguments
	 */
.ft L
	svc_freeargs(transp, xdr_wrapstring, &s);
}
.DE
Of course the service could have one procedure
that takes the string and a boolean
to indicate whether or not the procedure should respond.
.LP
In order for a client to take advantage of batching,
the client must perform RPC calls on a TCP-based transport
and the actual calls must have the following attributes:
1) the result's XDR routine must be zero
.I NULL ),
and 2) the RPC call's timeout must be zero.
.LP
Here is an example of a client that uses batching to render a
bunch of strings; the batching is flushed when the client gets
a null string (EOF):
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include "windows.h"

main(argc, argv)
	int argc;
	char **argv;
{
	struct timeval pertry_timeout, total_timeout;
	register CLIENT *client;
	enum clnt_stat clnt_stat;
	char buf[1000], *s = buf;
.sp .5
.DE
.DS
.ft L
	if ((client = clnt_create(argv[1], WINDOWPROG,
	  WINDOWVERS, "tcp")) == NULL) {
		clnt_pcreateerror("clnt_create");
		exit(1);
	}
	total_timeout.tv_sec = 0;
	total_timeout.tv_usec = 0;
	while (scanf("%s", s) != EOF) {
		clnt_stat = clnt_call(client, RENDERSTRING_BATCHED,
			xdr_wrapstring, &s, xdr_void, NULL, total_timeout);
		if (clnt_stat != RPC_SUCCESS) {
			clnt_perror(client, "batched rpc");
			exit(1);
		}
	}
.sp .5
.DE
.DS
.ft L
	/* \fINow flush the pipeline\fP */
.sp .5
	total_timeout.tv_sec = 20;
	clnt_stat = clnt_call(client, NULLPROC, xdr_void, NULL,
		xdr_void, NULL, total_timeout);
	if (clnt_stat != RPC_SUCCESS) {
		clnt_perror(client, "rpc");
		exit(1);
	}
	clnt_destroy(client);
	exit(0);
}
.DE
Since the server sends no message,
the clients cannot be notified of any of the failures that may occur.
Therefore, clients are on their own when it comes to handling errors.
.LP
The above example was completed to render
all of the (2000) lines in the file
.I /etc/termcap .
The rendering service did nothing but throw the lines away.
The example was run in the following four configurations:
1) machine to itself, regular RPC;
2) machine to itself, batched RPC;
3) machine to another, regular RPC; and
4) machine to another, batched RPC.
The results are as follows:
1) 50 seconds;
2) 16 seconds;
3) 52 seconds;
4) 10 seconds.
Running
.I fscanf()
on
.I /etc/termcap 
only requires six seconds.
These timings show the advantage of protocols
that allow for overlapped execution,
though these protocols are often hard to design.
.
.NH 2
\&Authentication
.IX "authentication"
.IX "RPC" "authentication"
.LP
In the examples presented so far,
the caller never identified itself to the server,
and the server never required an ID from the caller.
Clearly, some network services, such as a network filesystem,
require stronger security than what has been presented so far.
.LP
In reality, every RPC call is authenticated by
the RPC package on the server, and similarly,
the RPC client package generates and sends authentication parameters.
Just as different transports (TCP/IP or UDP/IP)
can be used when creating RPC clients and servers,
different forms of authentication can be associated with RPC clients;
the default authentication type used is type
.I none .
.LP
The authentication subsystem of the RPC package is open ended.
That is, numerous types of authentication are easy to support.
.IP "\fIThe Client Side\fP"
.LP
When a caller creates a new RPC client handle as in:
.DS
.ft L
clnt = clnt_create(host, prognum, versnum, nettype)
.DE
the appropriate transport instance defaults the associated
authentication handle to be
.DS
.ft L
clnt->cl_auth = authnone_create();
.DE
Since the RPC user created this new style of authentication,
the user is responsible for destroying it with:
.DS
.ft L
auth_destroy(clnt->cl_auth);
.DE
This should be done in all cases, to conserve memory.
.sp
.IP "\fIThe Server Side\fP"
.LP
Service implementors have a harder time dealing with authentication issues
since the RPC package passes the service dispatch routine a request
that has an arbitrary authentication style associated with it.
Consider the fields of a request handle passed to a service dispatch routine:
.DS
.ft I
/*
 * An RPC Service request
 */
.ft L
struct svc_req {
	u_long	rq_prog;			/* \fIservice program number\fP */
	u_long	rq_vers;			/* \fIservice protocol vers num\fP */
	u_long	rq_proc;			/* \fIdesired procedure number\fP */
	struct opaque_auth rq_cred; /* \fIraw credentials from wire\fP */
	caddr_t rq_clntcred;		/* \fIcredentials (read only)\fP */
};
.DE
The
.I rq_cred
is mostly opaque, except for one field of interest:
the style or flavor of authentication credentials:
.DS
.ft I
/*
 * Authentication info.  Mostly opaque to the programmer.
 */
.ft L
struct opaque_auth {
	enum_t	oa_flavor;          /* \fIstyle of credentials\fP */
	caddr_t	oa_base;            /* \fIaddress of more auth stuff\fP */
	u_int	oa_length;          /* \fInot to exceed \fIMAX_AUTH_BYTES */
};
.DE
.IX RPC guarantees
The RPC package guarantees the following
to the service dispatch routine:
.IP  1.
That the request's
.I rq_cred
is well formed.  Thus the service implementor may inspect the request's
.I rq_cred.oa_flavor
to determine which style of authentication the caller used.
The service implementor may also wish to inspect the other fields of
.I rq_cred
if the style is not one of the styles supported by the RPC package.
.IP  2.
That the request's
.I rq_clntcred
field is either
.I NULL 
or points to a well formed structure
that corresponds to a supported style of authentication credentials.
Remember that only\fI
.UX
.\".I UNIX
\fPand
.I DES
styles are currently supported, so (currently)
.I rq_clntcred
could be cast only as a pointer to an
.I authsys_parms
or
.I authdes_cred
structure.  If
.I rq_clntcred
is
.I NULL ,
the service implementor may wish to inspect the other (opaque) fields of
.I rq_cred
in case the service knows about a new type of authentication
that the RPC package does not know about.
.NH 3
\&UNIX Authentication
.IX "UNIX Authentication"
.LP
The RPC client can choose to use
.I UNIX
style authentication by setting
.I clnt\->cl_auth 
after creating the RPC client handle:
.DS
.ft L
clnt->cl_auth = authsys_create_default();
.DE
This causes each RPC call associated with
.I clnt
to carry with it the following authentication credentials structure:
.DS
.ft I
/*
 * UNIX style credentials.
 */
.ft L
struct authsys_parms {
	u_long 	aup_time;               /* \fIcredentials creation time\fP */
	char 	*aup_machname;          /* \fIhost name where client is\fP */
	int 	aup_uid;                /* \fIclient's UNIX effective uid\fP */
	int 	aup_gid;                /* \fIclient's current group id\fP */
	u_int	aup_len;                /* \fIelement length of aup_gids\fP */
	int 	*aup_gids;              /* \fIarray of groups user is in\fP */
};
.DE
These fields are set by
.I authsys_create_default()
by invoking the appropriate system calls.
.LP
Our remote users service example can be extended so that
it computes results for all users except UID 16:
.ne 2i
.DS
.ft L
nuser(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
	struct authsys_parms *unix_cred;
	int uid;
	unsigned long nusers;
.sp .5
.ft I
	/*
	 * we don't care about authentication for null proc
	 */
.ft L
	if (rqstp->rq_proc == NULLPROC) {
		if (!svc_sendreply(transp, xdr_void, 0)) {
			fprintf(stderr, "can't reply to RPC call\en");
			return (1);
		 }
		 return;
	}
.DE
.DS
.ft I
	/*
	 * now get the uid
	 */
.ft L
	switch (rqstp->rq_cred.oa_flavor) {
	case AUTH_SYS:
		unix_cred =
			(struct authsys_parms *)rqstp->rq_clntcred;
		uid = unix_cred->aup_uid;
		break;
	case AUTH_NULL:
	default:
		svcerr_weakauth(transp);
		return;
	}
.DE
.DS
.ft L
	switch (rqstp->rq_proc) {
	case RUSERSPROC_NUM:
.ft I
		/*
		 * make sure caller is allowed to call this proc
		 */
.ft L
		if (uid == 16) {
			svcerr_systemerr(transp);
			return;
		}
.ft I
		/*
		 * Code here to compute the number of users
		 * and assign it to the variable \fInusers\fP
		 */
.ft L
		if (!svc_sendreply(transp, xdr_u_long, &nusers)) {
			fprintf(stderr, "can't reply to RPC call\en");
			return (1);
		}
		return;
	default:
		svcerr_noproc(transp);
		return;
	}
}
.DE
A few things should be noted here.
First, it is customary not to check
the authentication parameters associated with the
.I NULLPROC
(procedure number zero).
Second, if the authentication parameter's type is not suitable
for your service, you should call
.I svcerr_weakauth() .
And finally, the service protocol itself should return status
for access denied; in the case of our example, the protocol
does not have such a status, so we call the service primitive
.I svcerr_systemerr()
instead.
.LP
The last point underscores the relation between
the RPC authentication package and the services;
RPC deals only with
.I authentication
and not with individual services'
.I "access control" .
The services themselves must implement their own access control policies
and reflect these policies as return statuses in their protocols.
.NH 3
\&DES Authentication
.IX RPC DES
.IX RPC authentication
.LP
UNIX authentication is quite easy to defeat.  Instead of using
.I authsys_create_default (),
you can call
.I authsys_create() 
and then modify the RPC authentication handle to whatever user ID and
hostname you wish the server to think you have.  DES authentication is
recommended for people who want more security than UNIX authentication
offers.
.LP
The details of the DES authentication protocol are complicated and
are not explained here.  See the
\fIRemote Procedure Calls: Protocol Specification\fR
section for the details.
.LP
In order for DES authentication to work, the
.I keyserv(8c) 
daemon must be running on both the server and client machines.  The
users on these machines need public keys assigned by the network
administrator in the
.I publickey(5) 
database.  And, they need to have decrypted their secret keys
using their login password.  This automatically happens when one
logs in using
.I login(1) ,
or can be done manually using
.I keylogin(1) .
.sp
.IP "\fIClient Side\fP"
.LP
If a client wishes to use DES authentication, it must set its
authentication handle appropriately.  Here is an example:
.DS
.ft L
cl->cl_auth =
	authdes_seccreate(servername, 60, server, NULL);
.DE
The first argument is the network name or \*Qnetname\*U of the owner of
the server process.  Typically, server processes are root processes
and their netname can be derived using the following call:
.DS
.ft L
char servername[MAXNETNAMELEN];

host2netname(servername, rhostname, NULL);
.DE
Here,
.I rhostname
is the hostname of the machine the server process is running on.
.I host2netname() 
fills in
.I servername
to contain this root process's netname.  If the
server process was run by a regular user, one could use the call
.I user2netname()
instead.  Here is an example for a server process with the same user
ID as the client:
.DS
.ft L
char servername[MAXNETNAMELEN];

user2netname(servername, getuid(), NULL);
.DE
The last argument to both of these calls,
.I user2netname() 
and
.I host2netname (),
is the name of the naming domain where the server is located.  The
.I NULL 
used here means \*Quse the local domain name.\*U
.LP
The second argument to
.I authdes_seccreate() 
is a lifetime for the credential.  Here it is set to sixty
seconds.  What that means is that the credential will expire 60
seconds from now.  If some mischievous user tries to reuse the
credential, the server RPC subsystem will recognize that it has
expired and not grant any requests.  If the same mischievous user
tries to reuse the credential within the sixty second lifetime,
he will still be rejected because the server RPC subsystem
remembers which credentials it has already seen in the near past,
and will not grant requests to duplicates.
.LP
The third argument to
.I authdes_seccreate() 
is the name of the host to synchronize with.  In order for DES
authentication to work, the server and client must agree upon the
time.  Here we pass the hostname of the server itself, so the
client and server will both be using the same time: the server's
time.  The argument can be
.I NULL ,
which means \*Qdon't bother synchronizing.\*U You should only do this
if you are sure the client and server are already synchronized.
.LP
The final argument to
.I authdes_seccreate()
is the address of a DES encryption key to use for encrypting
timestamps and data.  If this argument is
.I NULL ,
as it is in this example, a random key will be chosen.  The client
may find out the encryption key being used by consulting the
.I ah_key 
field of the authentication handle.
.sp
.IP "\fIServer Side\fP"
.LP
The server side is simpler than the client side.  Here is the
previous example rewritten to use
.I AUTH_DES
instead of
.I AUTH_SYS :
.DS
.ft L
#include <rpc/rpc.h>
	. . .
	. . .
nuser(rqstp, transp)
	struct svc_req *rqstp;
	SVCXPRT *transp;
{
	struct authdes_cred *des_cred;
	int uid;
	int gid;
	int gidlen;
	int gidlist[10];
.DE
.DS
.ft I
	/*
	 * we don't care about authentication for null proc
	 */
.ft L
.sp .5
	if (rqstp->rq_proc == NULLPROC) {
		/* \fIsame as before\fP */
	}
.sp .5
.ft I
	/*
	 * now get the uid
	 */
.ft L
	switch (rqstp->rq_cred.oa_flavor) {
	case AUTH_DES:
		des_cred =
		  (struct authdes_cred *) rqstp->rq_clntcred;
		if (! netname2user(des_cred->adc_fullname.name,
		  &uid, &gid, &gidlen, gidlist)) {
			fprintf(stderr, "unknown user: %s\en",
			  des_cred->adc_fullname.name);
			svcerr_systemerr(transp);
			return;
		}
		break;
.DE
.DS
.ft L
	case AUTH_NULL:
	default:
		svcerr_weakauth(transp);
		return;
	}

.ft I
	/*
	 * The rest is the same as before
 	 */	
.ft L
.DE
Note the use of the routine
.I netname2user (),
the inverse of
.I user2netname ():
it takes a network ID and converts to a
.UX
ID.
.I netname2user()
also supplies the group IDs which we don't use in this example,
but which may be useful to other UNIX programs.
.NH 2
\&Using Portmonitors
.IX portmonitors
.LP
An RPC server can be started from portmonitors such as 
.I inetd 
and
.I listener .
These portmonitors listen to requests on the behalf of the services, and 
spawn servers in response to those requests.  The forked server process is 
passed the file descriptor 0 on which the request has been accepted.  
After the server is done servicing the request, it may exit immediately, 
or wait a given interval of time for another service request to come in.  
Remember that if you want to exit from the server process and return
control to portmonitor, you need to explicitly exit, since
.I svc_run ()
never returns.
.LP
The following server create routine can be used to create such a service:
.DS
.ft L
transp = svc_tli_create(0, nconf, NULL, 0, 0);
.DE
.I nconf
is the 
.I netconfig 
structure of the transport on which the request came in.  One could however 
specify 
.I NULL 
instead of 
.I nconf ,
in which case the server will have to find its own transport as a part of 
the process of accepting the connection and getting started.
This may lead to some inefficiency.
.LP
Since, the portmonitors have already registered the given service with 
.I rpcbind ,
there is no need for the service to register itself.  However, it must
call 
.I svc_reg()
in the following form:
.DS
.ft L
svc_reg(transp, PROGNUM, VERSNUM, dispatch, NULL);
.DE
The
.I netconfig
structure here is
.I NULL .
.LP
For connection-oriented transports, the following server create
routine can also be used in the following form:
.DS
.ft L
transp = svc_fd_create(0, recvsize, sendsize);
.DE
The file descriptor passed here is 0.  The user may set the value of
.I recvsize
or
.I sendsize
to any appropriate buffer size.   If they use a 0 in either case,
a system default size will be chosen.  This routine should be used
by those application servers which do not do any listening of their
own, i.e. servers which simply do their job and return.
.NH 3
\&Using Inetd
.LP
The format of entries in
.I /etc/inetd.conf 
for RPC services is in one of the following two forms:
.DS
.ft L
p_name/version dgram  rpc/udp wait/nowait user server args
p_name/version stream rpc/tcp wait/nowait user server args
.DE
where
.I p_name
is the symbolic name of the program as it appears in
.I rpc(5) ,
.I server
is the program implementing the server, and
.I program
and
.I version
are the program and version numbers of the service.  
For more information, see
.I inetd.conf(5) .
.LP
If the same program handles multiple versions, then the version number 
can be a range, as in this example:
.DS
.ft L
rstatd/1-2 dgram rpc/udp wait root /usr/sbin/rpc.rstatd
.DE
.NH 3
\&Using Listener
.IX listen "" "Using \fIlistener\fP"
.LP
We will assume here that the reader already knows the details of setting 
up the listener process and of using 
.I nlsadmin .
This example simply shows the user how to use
.I nlsadmin 
to add RPC services.
.DS
.ft L
example% \fBnlsadmin -a svc_code -c command -D \e
		-R prog:vers -y "rpc service" netid\fP
.DE
Here
.I svc_code 
is the server's identifying code, 
.I command 
is the program implementing the server, 
.I prog 
and
.I vers 
are the program number and the version number of the service, and
.I netid 
is the network id of the transport on which the service will run.  
The address of the server can be specified either statically or
dynamically (with the 
.I "-D" 
option):
.DS
.ft L
example% \fBnlsadmin -a 2 -c /usr/sbin/rpc.rstatd \e
         -D -R rstatd:1 -y "rpc service" npack\fP
.DE
For more information, see the
.I listener 
man page.
.KS
.NH 1
\&More Examples
.XS
    \&More Examples
.XE
.NH 2
\&Versions
.IX "versions"
.IX "RPC" "versions"
.LP
By convention, the first version number of program
.I PROG
is
.I PROGVERS_ORIG
and the most recent version is
.I PROGVERS
Suppose there is a new version of the
.I user
program that returns an
.I "unsigned
rather than a
.I long
If we name this version
.I RUSERSVERS_SHORT
then a server that wants to support both versions
would do a double register.  The same server handle would be used
for both of these registrations.
.KE
.DS
.ft L
if (!svc_reg(transp, RUSERSPROG, RUSERSVERS_ORIG,
  nuser, nconf)) {
	fprintf(stderr, "can't register RUSER service\en");
	exit(1);
}
if (!svc_reg(transp, RUSERSPROG, RUSERSVERS_SHORT,
  nuser, nconf)) {
	fprintf(stderr, "can't register RUSER service\en");
	exit(1);
}
.DE
Both versions can be handled by the same C procedure:
.DS
.ft L
nuser(rqstp, transp)
    struct svc_req *rqstp;
    SVCXPRT *transp;
{
    unsigned long nusers;
    unsigned short nusers2;
.sp .5
.DE
.DS
.ft L
    switch (rqstp->rq_proc) {
    case NULLPROC:
        if (!svc_sendreply(transp, xdr_void, 0)) {
            fprintf(stderr, "can't reply to RPC call\en");
            return (1);
        }
        return (0);
.DE
.DS
.ft L
    case RUSERSPROC_NUM:
.ft I
             /*
              * Code here to compute the number of users
              * and assign it to the variable \fInusers\fP
              */
.ft L
        nusers2 = nusers;
        switch (rqstp->rq_vers) {
        case RUSERSVERS_ORIG:
            if (!svc_sendreply(transp, xdr_u_long, 
              &nusers)) {
                fprintf(stderr,"can't reply to RPC call\en");
            }
            break;
        case RUSERSVERS_SHORT:
            if (!svc_sendreply(transp, xdr_u_short,
              &nusers2)) {
                fprintf(stderr,"can't reply to RPC call\en");
            }
            break;
        }
.DE
.DS
.ft L
    default:
        svcerr_noproc(transp);
        return;
    }
    return (0);
}
.DE
.
.NH 2
\&Connection Oriented Transports
.IX "connection-oriented transports"
.LP
Here is an example that is essentially
.I rcp .
The initiator of the RPC
.I snd
call takes its standard input and sends it to the server
.I rcv
which prints it on standard output.
This also illustrates an XDR procedure that behaves differently
on serialization than on deserialization.
.DS
.ft I
/*
 * The xdr routine:
 *		on decode, read from wire, write onto fp
 *		on encode, read from fp, write onto wire
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>

xdr_rcp(xdrs, fp)
	XDR *xdrs;
	FILE *fp;
{
	unsigned long size;
	char buf[BUFSIZ], *p;
.sp .5
.DE
.DS
.ft L
	if (xdrs->x_op == XDR_FREE)/* nothing to free */
		return 1;
	while (1) {
		if (xdrs->x_op == XDR_ENCODE) {
			if ((size = fread(buf, sizeof(char), BUFSIZ,
			  fp)) == 0 && ferror(fp)) {
				fprintf(stderr, "can't fread\en");
				return (1);
			}
		}
		p = buf;
		if (!xdr_bytes(xdrs, &p, &size, BUFSIZ))
			return 0;
		if (size == 0)
			return 1;
		if (xdrs->x_op == XDR_DECODE) {
			if (fwrite(buf, sizeof(char), size,
			  fp) != size) {
				fprintf(stderr, "can't fwrite\en");
				return (1);
			}
		}
	}
}
.DE
.LP
Note that the actual serializing and deserializing here is 
done only by 
.I xdr_bytes ().
.DS
.ft I
/*
 * The sender routines
 */
.ft L
#include <stdio.h>
#include <netdb.h>
#include <rpc/rpc.h>
#include <sys/socket.h>
#include <sys/time.h>

main(argc, argv)
	int argc;
	char **argv;
{
	int xdr_rcp();
.sp .5
.DE
.DS
.ft L
	if (argc < 2) {
		fprintf(stderr, "usage: %s servername\en", argv[0]);
		exit(1);
	}
	if (callcots(argv[1], RCPPROG, RCPPROC, RCPVERS,
	  xdr_rcp, stdin, xdr_void, 0) != 0) {
		exit(1);
	}
	exit(0);
}

.DE
.DS
.ft L
callcots(host, prognum, procnum, versnum, inproc,
  in, outproc, out)
	char *host, *in, *out;
	xdrproc_t inproc, outproc;
{
	enum clnt_stat clnt_stat;
	register CLIENT *client;
	struct timeval total_timeout;
.sp .5
.DE
.DS
.ft L
	if ((client = clnt_create(host, prognum, versnum,
	  "circuit_v")) == NULL) {
		perror("clnt_create");
		return (-1);
	}
	total_timeout.tv_sec = 20;
	total_timeout.tv_usec = 0;
	clnt_stat = clnt_call(client, procnum,
		inproc, in, outproc, out, total_timeout);
	clnt_destroy(client);
	if (clnt_stat != RPC_SUCCESS) {
		clnt_perror("callcots");
	}
	return ((int)clnt_stat);
}
.DE
.DS
.ft I
/*
 * The receiving routines
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>

main()
{
	int rcp_service(), xdr_rcp();
.sp .5
.DE
.DS
.ft L
	if (svc_create(rpc_service, RCPPROG, RCPVERS,
	  "circuit_v") == 0) {
		fprintf("svc_create: error\en");
		exit(1);
	}
	svc_run();  /* \fInever returns\fP */
	fprintf(stderr, "svc_run should never return\en");
}
   
.DE
.DS
.ft L
rcp_service(rqstp, transp)
	register struct svc_req *rqstp;
	register SVCXPRT *transp;
{
	switch (rqstp->rq_proc) {
	case NULLPROC:
		if (svc_sendreply(transp, xdr_void, 0) == 0) {
			fprintf(stderr, "err: rcp_service");
			return (1);
		}
		return;
.DE
.DS
.ft L
	case RCPPROC:
		if (!svc_getargs(transp, xdr_rcp, stdout)) {
			svcerr_decode(transp);
			return (1);
		}
		if (!svc_sendreply(transp, xdr_void, 0)) {
			fprintf(stderr, "can't reply\en");
			return (1);
		}
		return (0);
	default:
		svcerr_noproc(transp);
		return (1);
	}
}
.DE
.LP
Note that on the server side no explicit action was taken after
receiving the arguments.  This is because
.I xdr_rcp() 
did all the necessary dirty work automatically.
.NH 2
\&Callback Procedures
.IX RPC "callback procedures"
.LP
Occasionally, it is useful to have a server become a client, and make 
an RPC call back to the process which is its client.  An example is 
remote debugging, where the client is a window system program, and the 
server is a debugger running on the remote machine.  Most of the time,
the user clicks a mouse button at the debugging window, which converts 
this to a debugger command, and then makes an RPC call to the server
(where the debugger is actually running), telling it to execute that 
command.  However, when the debugger hits a breakpoint, the roles are 
reversed, and the debugger wants to make an rpc call to the window 
program, so that it can inform the user that a breakpoint has been reached.
.LP
In order to do an RPC callback, you need a program number to make the RPC
call on.  Since this will be a dynamically generated program number, it
should be in the transient range, \fL0x40000000 - 0x5fffffff\fP.
The routine
.I gettransient()
returns a valid program number in the transient range, and registers it with 
.I rpcbind .
The call to
.I rpcb_set()
is a test and set operation, in that it indivisibly tests whether a 
program number has already been registered, and if it has not, then 
reserves it.  
.DS
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>

gettransient(vers, nconf, address)
	int vers;
	struct netconfig *nconf;
	struct netbuf *address;
{
	static int prog = 0x40000000;
.sp .5
.DE
.DS
.ft L
	while (! rpcb_set(prog++, vers, nconf, address))
		continue;
	return (prog - 1);
}
.DE
.LP
The following contrived program illustrates how to use the
.I gettransient()
routine.
The client makes an RPC call to the server, passing it a transient
program number.  Then the client waits around to receive a callback
from the server at that program number.  The server registers the program
.I EXAMPLEPROG
so that it can receive the RPC call informing it of the callback
program number.  Then at some random time (on receiving an
.I ALRM
signal in this example), it sends a callback RPC call, using the 
program number it received earlier.
.DS
.ft I
/*
 * client
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <netconfig.h>
#include "example.h"
.sp .5
main(argc, argv)
	int argc;
	char **argv;
{
	SVCXPRT *xprt;
	struct netconfig *nconf;
	int prognum;
	enum clnt_stat stat;
.sp .5
.DE
.DS
.ft L
	if (argc != 3) {
		fprintf(stderr, "usage: clnt host transport\en");
		exit (1);
	}
	nconf = getnetconfigent(argv[2]);
	if (nconf == NULL) {
		fprintf(stderr, "unknown transport\en");
		exit (1);
	}
	xprt = svc_tli_create(RPC_ANYFD, nconf, 
	  (struct t_bind *)NULL, 0, 0);
	if (xprt == (SVCXPRT *)NULL) {
		fprintf(stderr, "could not create server handle\en");
		exit (1);
	}
.DE
.DS
.ft L
	prognum = gettransient(1, nconf, &xprt->xp_ltaddr);
	fprintf(stderr, "client gets prognum %d\en", prognum);
	if (svc_reg(xprt, prognum, 1, callback, NULL) 
	  == FALSE) {
		fprintf(stderr, "could not register service\en");
		exit(1);
	}
	stat = rpc_call(argv[1], EXAMPLEPROG, EXAMPLEVERS,
	  EXAMPLEPROC_CALLBACK, xdr_int, &prognum, xdr_void, 
	  NULL, NULL);
	if (stat != RPC_SUCCESS) {
		clnt_perrno(stat);
		exit(1);
	}
	svc_run();
	exit(1);
}

.DE
.DS
.ft L
callback(rqstp, transp)
	register struct svc_req *rqstp;
	register SVCXPRT *transp;
{
	switch (rqstp->rq_proc) {
		case 0:
			if (!svc_sendreply(transp, xdr_void, 0)) {
				fprintf(stderr, "err: exampleprog\en");
				return (1);
			}
			return (0);
.DE
.DS
.ft L
		case 1:
			if (!svc_getargs(transp, xdr_void, 0)) {
				svcerr_decode(transp);
				return (1);
			}
			fprintf(stderr, "client got callback\en");
			if (!svc_sendreply(transp, xdr_void, 0)) {
				fprintf(stderr, "err: exampleprog");
				return (1);
			}
	}
	return (1);
}
.DE
.LP
This example again demonstrates the usefulness of
.I svc_tli_create ()
in cases where we want to explicitly chose the program number 
by calling
.I rpcb_set() 
until it succeeds.  Here it was not required that we register a 
service on a given transport, and we could simply have used a
\*Qgeneric\*U network type.  After creating the handle we called
.I svc_reg ()
(with the last parameter given as 
.I NULL )
to register the dispatch function with the dispatcher.  Once the
server side is ready, it then notifies the actual server of its
dynamic program number with 
.I rpc_call ().
On success it then waits for requests from the remote server.
.DS
.ft I
/*
 * server
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>
#include <sys/signal.h>
#include "example.h"
.sp .5
char *getnewprog();
char *hostname;
int docallback();
int pnum;		/* \fIprogram number for callback routine\fP */
.sp .5
main()
{
	if (argc != 2) {
		fprint(stderr, "usage: server hostname\en");
		exit (1);
	}
.DE
.DS
.ft L
	hostname = argv[1];
	rpc_reg(EXAMPLEPROG, EXAMPLEVERS, EXAMPLEPROC_CALLBACK, 
	  getnewprog, xdr_int, xdr_void, NULL, NULL);
	signal(SIGALRM, docallback);
	alarm(10);
	svc_run();
	fprintf(stderr, "Error: svc_run shouldn't return\en");
}

char *
getnewprog(pnump)
	char *pnump;
{
	pnum = *(int *)pnump;
	return NULL;
}

.DE
.DS
.ft L
docallback()
{
	int ans;
.sp .5
	if (pnum == 0) {
		alarm (10);
		return;
	}
	ans = rpc_call(hostname, pnum, 1, 1, xdr_void, 0,
	  xdr_void, 0, NULL);
	if (ans != 0) {
		fprintf(stderr, "server: ");
		clnt_perrno(ans);
	}
}
.DE
.LP
In this contrived example, the server makes an RPC call to the client
on an
.I ALARM
signal, but only if the client has passed the program number to the 
server.  This server example also demonstrates the simplicity of the 
code when one is using 
.I rpc_reg() .
.NH 2
\&Memory Allocation with XDR
.IX "memory allocation with XDR"
.IX XDR "memory allocation"
.LP
XDR routines not only do input and output, they also do memory 
allocation.  The second parameter of
.I xdr_array()
is a pointer to an array, rather than the array itself.\**
.FS
This is true for most XDR routines.  The indirection is necessary
because these routines often allocate memory.
.FE
If it is
.I NULL ,
then
.I xdr_array()
allocates space for the array and returns a pointer to it,
putting the size of the array in the third argument.
As an example, consider the following XDR routine
.I xdr_chararr1()
which deals with a fixed array of bytes with length
.I SIZE :
.DS
.ft L
xdr_chararr1(xdrsp, chararr)
	XDR *xdrsp;
	char chararr[];
{
	char *p;
	int len;
.sp .5
	p = chararr;
	len = SIZE;
	return (xdr_bytes(xdrsp, &p, &len, SIZE));
}
.DE
If space has already been allocated in
.I chararr ,
it can be called from a server like this:
.DS
.ft L
char chararr[SIZE];

svc_getargs(transp, xdr_chararr1, chararr);
.DE
If you want XDR to do the allocation,
you would have to rewrite this routine in the following way:
.DS
.ft L
xdr_chararr2(xdrsp, chararrp)
	XDR *xdrsp;
	char **chararrp;
{
	int len;
.sp .5
	len = SIZE;
	return (xdr_bytes(xdrsp, charrarrp, &len, SIZE));
}
.DE
Then the RPC call might look like this:
.DS
.ft L
char *arrptr;

arrptr = NULL;
svc_getargs(transp, xdr_chararr2, &arrptr);
.ft I
/*
 * Use the result here
 */
.ft L
svc_freeargs(transp, xdr_chararr2, &arrptr);
.DE
Note that, after being used, the character array should normally be 
freed with
.I svc_freeargs()
.I svc_freeargs() 
will not attempt to free any memory if the variable indicating it
is NULL.  For example, in the routine
.I xdr_finalexample (),
given earlier, if
.I finalp->string 
was NULL, then it would not be freed.  The same is true for
.I finalp->simplep .
.LP
To summarize, each XDR routine is responsible
for serializing, deserializing, and freeing memory.
When an XDR routine is called from
.I rpc_call()
the serializing part is used.  When called from
.I svc_getargs()
the deserializer is used.  And when called from
.I svc_freeargs()
the memory deallocator is used.  When building simple examples like those
in this section, a user doesn't have to worry about the three modes.  
.LP
