.\" @(#)rpcgen.ms 1.4 91/03/11 TIRPC 1.0
.NH 1
\&\fBrpcgen\fP Programming Guide
.LP
.NH 2
\&The \fBrpcgen\fP Protocol Compiler
.LP
.IX RPC "" "" \fIrpcgen\fP
The details of programming applications to use Remote Procedure Calls
can be overwhelming.  Perhaps most daunting is the writing of the XDR
routines necessary to convert procedure arguments and results into
their network format and vice-versa.
.LP
Fortunately,
.I rpcgen(1) 
exists to help programmers write RPC applications simply and directly.
.I rpcgen 
does most of the dirty work, allowing programmers to debug
the main features of their application, instead of requiring them to
spend most of their time on their network interface code.
.LP
.I rpcgen 
is a compiler.  It accepts a remote program interface definition written in
a language, called RPC Language, which is similar to C.  It produces a C
language output for RPC programs able to run over the TLI.  This output
includes stub versions of the client routines, a server skeleton, XDR
filter routines for both parameters and results, a header file that
contains common definitions and, optionally, dispatch tables which the
server can use to check authorizations and then invoke service routines.
The client stubs interface with the RPC library and effectively hide the
network from their callers.  The server stub similarly hides the network
from the server procedures that are to be invoked by remote clients.
.I rpcgen 's
output files can be compiled and linked in the usual way, and support
port monitors like
.I listener 
and
.I inetd .
.LP
The developer writes server procedures\(emin any language that observes 
system calling conventions\(emand links them with the server skeleton 
produced by
.I rpcgen 
to get an executable server program.  To use a remote program, the
programmer writes an ordinary main program that makes local procedure 
calls to the client stubs produced by
.I rpcgen .
Linking this program with
.I rpcgen 's
stubs creates an executable program.  (At present the main program must be
written in C).
.I rpcgen 
options can be used to suppress stub generation and to specify the
transport to be used by the server stub.
.LP
Like all compilers,
.I rpcgen 
reduces the development time that would otherwise be spent coding and 
debugging low-level routines.  All compilers, including
.I rpcgen ,
do this at a small cost in efficiency and flexibility.  However, many 
compilers allow escape hatches for programmers to mix low-level code with 
high-level code.
.I rpcgen 
is no exception.  In speed-critical applications, hand-written routines
can be linked with the
.I rpcgen 
output without any difficulty.  Also, one may proceed by using
.I rpcgen 
output as a starting point, and then rewriting it as necessary.
(For a discussion of RPC programming without
.I rpcgen ,
see the next chapter, the
\fI@TitleOf(rpcpg)\fR\)\.
.NH 3
\&Converting Local Procedures into Remote Procedures
.IX rpcgen "local procedures" \fIrpcgen\fP
.IX rpcgen "remote procedures" \fIrpcgen\fP
.LP
Assume an application that runs on a single machine, one which we want
to convert to run over the network.  Here we will demonstrate such a
conversion by way of a simple example\(ema program that prints a
message to the console:
.DS
.ft I
/*
 * printmsg.c: print a message on the console
 */
.ft L
#include <stdio.h>
.sp .5
main(argc, argv)
	int argc;
	char *argv[];
{
	char *message;
.sp .5
	if (argc != 2) {
		fprintf(stderr, "usage: %s <message>\en", argv[0]);
		exit(1);
	}
	message = argv[1];
.sp .5
	if (!printmessage(message)) {
		fprintf(stderr, "%s: couldn't print your message\en",
			argv[0]);
		exit(1);
	}
	printf("Message Delivered!\en");
	exit(0);
}
.ft I
/*
 * Print a message to the console.
 * Return a boolean indicating whether the message was actually printed.
 */
.ft L
printmessage(msg)
	char *msg;
{
	FILE *f;
.sp .5
	f = fopen("/dev/console", "w");
	if (f == NULL) {
		return (0);
	}
	fprintf(f, "%s\en", msg);
	fclose(f);
	return (1);
}
.DE
.LP
And then, of course:
.DS
example%  \fBcc printmsg.c -o printmsg\fP
example%  \fBprintmsg "Hello, there."\fP
Message delivered!
example%
.DE
.LP
If
.I printmessage() 
was turned into a remote procedure,
then it could be called from anywhere in the network.
Ideally, one would just like to stick a keyword like
.I remote 
in front of a
procedure to turn it into a remote procedure.  Unfortunately,
we have to live within the constraints of the C language, since
it existed long before RPC did.  But even without language
support, it's not very difficult to make a procedure remote.
.LP
In general, it's necessary to figure out what the types are for
all procedure inputs and outputs.  In this case, we have a
procedure
.I printmessage() 
which takes a string as input, and returns an integer
as output.  Knowing this, we can write a protocol specification in RPC
language that describes the remote version of
.I printmessage ().
Here it is:
.DS
.ft I
/*
 * msg.x: Remote message printing protocol
 */
.ft L
.sp .5
program MESSAGEPROG {
	version MESSAGEVERS {
		int PRINTMESSAGE(string) = 1;
	} = 1;
} = 99;
.DE
.LP
Remote procedures are part of remote programs, so we actually declared
an entire remote program here which contains the single procedure
.I PRINTMESSAGE .
This procedure was declared to be in version 1 of the
remote program.  No null procedure (procedure 0) is necessary because
.I rpcgen 
generates it automatically.  (By convention, all RPC services provide 
for a procedure 0.  It is normally used only for pinging purposes).
.LP
Notice that everything is declared with all capital letters.  This is
not required, but is a good convention to follow.
.LP
Notice also that the argument type is \*Qstring\*U and not \*Qchar *\*U.  This
is because a \*Qchar *\*U in C is ambiguous.  Programmers usually intend it
to mean a null-terminated string of characters, but it could also
represent a pointer to a single character or a pointer to an array of
characters.  In RPC language, a null-terminated string is
unambiguously called a \*Qstring\*U.
.LP
There are just two more things to write.  First, there is the remote
procedure itself, and the main client program that calls it.  Here's the 
definition of a remote procedure to implement the
.I PRINTMESSAGE 
procedure we declared above.
.DS
.ft I
/*
 * msg_proc.c: implementation of the remote procedure "printmessage"
 */
.ft L
.sp .5
#include <stdio.h>
#include <rpc/rpc.h> 	/* \fIalways needed\fP  */
#include "msg.h"	 	/* \fImsg.h will be generated by rpcgen\fP */
.sp .5
.ft I
/*
 * Remote verson of "printmessage"
 */
.ft L
int *		
printmessage_1(msg)
	char **msg;	
{
	static int result; /* \fImust be static!\fP */
	FILE *f;
.sp .5
	f = fopen("/dev/console", "w");
	if (f == NULL) {
		result = 0;
		return (&result);
	}
	fprintf(f, "%s\en", *msg);
	fclose(f);
	result = 1;
	return (&result);
}
.DE
.LP
Notice here that the declaration of the remote procedure
.I printmessage_1() 
differs from that of the local procedure
.I printmessage() 
in three ways:
.IP  1.
It takes a pointer to a string instead of a string itself.  This
is true of all remote procedures: they always take pointers to their
arguments rather than the arguments themselves.
.IP  2.
It returns a pointer to an integer instead of an integer itself.  This is
also characteristic of remote procedures \(em they return pointers to 
their results.
.IP  3.
It has an \*Q_1\*U appended to its name.  In general, all remote
procedures called by
.I rpcgen 
are named by the following rule: the procedure name in the program 
definition (here
.I PRINTMESSAGE )
is converted to all
lower-case letters, an underbar (\*Q_\*U) is appended to it, and
finally the version number (here 1) is appended.
.LP
The last thing to do is declare the main client program that will call
the remote procedure.  Here it is:
.DS
.ft I
/*
 * rprintmsg.c: remote version of "printmsg.c"
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>     /* \fIalways needed\fP  */
#include "msg.h"         /* \fImsg.h will be generated by rpcgen\fP */
.sp .5
main(argc, argv)
	int argc;
	char *argv[];
{
	CLIENT *cl;
	int *result;
	char *server;
	char *message;
.sp .5
	if (argc != 3) {
		fprintf(stderr, 
		"usage: %s host message\en", argv[0]);
		exit(1);
	}
.sp .5
.ft I
	/*
	 * Save values of command line arguments
	 */
.ft L
	server = argv[1];
	message = argv[2];
.sp .5
.ft I
	/*
	 * Create client "handle" used for calling \fIMESSAGEPROG\fP on the
	 * server designated on the command line.  
	 */
.ft L
	cl = clnt_create(server, MESSAGEPROG, MESSAGEVERS, 
	  "visible");
	if (cl == NULL) {
.ft I
		/*
		 * Couldn't establish connection with server.
		 * Print error message and die.
		 */
.ft L
		clnt_pcreateerror(server);
		exit(1);
	}
	
.ft I
	/*
	 * Call the remote procedure "printmessage" on the server
	 */
.ft L
	result = printmessage_1(&message, cl);
	if (result == NULL) {
.ft I
		/*
		 * An error occurred while calling the server.
		 * Print error message and die.
		 */
.ft L
		clnt_perror(cl, server);
		exit(1);
	}
.sp .5
.ft I
	/*
	 * Okay, we successfully called the remote procedure.
	 */
.ft L
	if (*result == 0) {
.ft I
		/*
		 * Server was unable to print our message.
		 * Print error message and die.
		 */
.ft L
	fprintf(stderr, "%s: %s couldn't print your message\en", 
		argv[0], server);    
		exit(1);
	}
.sp .5
.ft I
	/*
	 * The message got printed on the server's console
	 */
.ft L
	printf("Message delivered to %s!\en", server);
	exit(0);
}
.DE
There are four things to note here:
.IP  1.
.IX "client handle, used by rpcgen" "" "client handle, used by \fIrpcgen\fP"
First a client \*Qhandle\*U is created using the RPC library routine
.I clnt_create ().
This client handle will be passed to the stub routines
which call the remote procedure.  (The client handle can be created in
other ways as well, see the 
\fI@TitleOf(rpcpg)\fR
chapter for details).
.IP  2.
.IX "nettype, and rpcgen" "" "\fInettype\fP, and \fIrpcgen\fP"
The last parameter to
.I clnt_create 
is \*Qvisible\*U, which indicates that any transport noted as
visible (almost all transports) in 
.I /etc/netconfig 
can be used.
.IP  3.
The remote procedure
.I printmessage_1() 
is called exactly the same way as it is declared in
.I msg_proc.c 
except for the inserted client handle as the second argument.  It also
returns a pointer to the result instead of the result itself.
.IP  4.
The remote procedure call can fail in two ways.  The RPC mechanism itself
can fail or, alternatively, there can be an error in the execution of the
actual remore procedure.  In the former case, the remote procedure (in this
case
.I print_message_1 ())
returns with a
.I NULL .
In the later case, however, the details of error reporting are application 
dependent.  Here, the error is being reported via
.I "*result" .
.LP
Here's how to put all of the pieces together:
.DS
example%  \fBrpcgen msg.x\fP
example%  \fBcc rprintmsg.c msg_clnt.c -o rprintmsg\fP
example%  \fBcc msg_proc.c msg_svc.c -o msg_server\fP
.DE
Two programs were compiled here: the client program
.I rprintmsg 
and the server program
.I msg_server .
Before doing this,
.I rpcgen 
was used to fill in the missing pieces.
.LP
Here is what
.I rpcgen 
(called without any flags) did with the input file
.I msg.x :
.IP  1.
It created a header file called
.I msg.h 
that contained
.I #define 's
for
.I MESSAGEPROG ,
.I MESSAGEVERS 
and
.I PRINTMESSAGE 
for use in the other modules.
.IP  2.
It created the client \*Qstub\*U routines in the
.I msg_clnt.c 
file.  In this case there is only one, the
.I printmessage_1() 
that was referred from the
.I printmsg 
client program.  If the name of the input file is 
.I FOO.x ,
the client stubs output file is called
.I FOO_clnt.c .
.IP  3.
It created the server program in 
.I msg_svc.c 
which calls
.I printmessage_1() 
from
.I msg_proc.c .
The rule for naming the server output file is similar to the
previous one: for an input file called
.I FOO.x ,
the output server file is named
.I FOO_svc.c .
.LP
(Note that, given the
.I -T 
argument,
.I rpcgen 
creates an additional output file which contains index information used 
for the dispatching of service routines).
.LP
Now we're ready to have some fun.  First, copy the server to a remote 
machine and run it.  For this example, the machine is called \*Qmoon\*U.  
.SH
Server processes created with 
.I rpcgen ,
like 
.I msg_server ,
always run in the background.  Thus,
it's not necessary to follow the server's invocation with a \*Q&\*U.
Also note that servers generated by
.I rpcgen 
can be invoked with portmonitors like
.I listen 
and 
.I inetd 
as well as from the command line.
.LP
.DS
moon% \fBmsg_server\fP
.DE
Then on our local machine (\*Qsun\*U) we can print a message on \*Qmoon\*Us
console.
.DS
sun% \fBrprintmsg moon "Hello, moon."\fP
.DE
The message will get printed on \*Qmoon\*Us console.  You can print a
message on anybody's console (including your own) with this program if
you can copy the server to their machine and run it.
.NH 3
\&One Real Example
.IX RPC "one real example"
.LP
The previous example only demonstrated the automatic generation of
client and server RPC code.
.I rpcgen 
may also be used to generate XDR routines, that is, the routines
necessary to convert local data
structures into network format and vice-versa.  This example presents
a complete RPC service\(ema remote directory listing service, which uses
.I rpcgen 
not only to generate stub routines, but also to generate the XDR
routines.  Here is the protocol description file.
.DS
.ft I
/*
 * dir.x: Remote directory listing protocol
 */
.ft L
const MAXNAMELEN = 255;		/* \fImaximum length of a directory entry\fP */
.sp .5
typedef string nametype<MAXNAMELEN>;	/* \fIa directory entry\fP */
.sp .5
typedef struct namenode *namelist;		/* \fIa link in the listing\fP */
.sp .5
.ft I
/*
 * A node in the directory listing
 */
.ft L
struct namenode {
	nametype name;		/* \fIname of directory entry\fP */
	namelist next;		/* \fInext entry\fP */
};
.sp .5
.ft I
/*
 * The result of a READDIR operation.
 */
.ft L
union readdir_res switch (int errno) {
case 0:
	namelist list;	/* \fIno error: return directory listing\fP */
default:
	void;		/* \fIerror occurred: nothing else to return\fP */
};
.sp .5
.ft I
/*
 * The directory program definition
 */
.ft L
program DIRPROG {
	version DIRVERS {
		readdir_res
		READDIR(nametype) = 1;
	} = 1;
} = 76;
.DE
.SH
.I
Types (like
.I readdir_res 
in the example above) can be defined using
the \*Qstruct\*U, \*Qunion\*U and \*Qenum\*U keywords, but those keywords
should not be used in subsequent declarations of variables of those types.
For example, if you define a union \*Qfoo\*U, you should declare using
only \*Qfoo\*U and not \*Qunion foo\*U.  In fact,
.I rpcgen 
compiles
RPC unions into C structures and it is an error to declare them using the
\*Qunion\*U keyword.
.LP
.sp
Running
.I rpcgen 
on
.I dir.x 
creates four output files.  First are the basic three itemized above:
those containing the header file, client stub routines and server
skeleton.  The fourth contains the XDR routines necessary for converting
the data types we declared into XDR format and vice-versa.  These are
output in the file
.I dir_xdr.c .
For each data type used in the \*Q.x\U file, 
.I rpcgen 
assumes that the RPC/XDR library has a routine defined with the name of
that datatype prepended by \*Qxdr_\*U (e.g. \fIxdr_int\fP).  If the data 
type was defined in the \*Q.x\*U file, then
.I rpcgen 
will generate the required xdr routine.  If there are no such data
types, then the file (e.g. msg.x) will not be generated.  If the data 
types was used but not defined, then the user has to provide that xdr
routine themselves.  This is a way for users to provide their own
customized xdr routines.  See the 
\fI@TitleOf(parb)\fP
section in the
\fI@TitleOf(rpcpg)\fP
chapter of this manual for more details.
.LP
Here is the server-side implementation of the
.I READDIR 
procedure.
.DS
.ft I
/*
 * dir_proc.c: remote readdir implementation
 */
.ft L
#include <rpc/rpc.h>		/* \fIAlways needed\fP */
#include <sys/dir.h>
#include "dir.h"			/* \fICreated by\fP rpcgen */
.sp .5
extern int errno;
extern char *malloc();
extern char *strdup();
.sp .5
readdir_res *
readdir_1(dirname)
	nametype *dirname;
{
	DIR *dirp;
	struct direct *d;
	namelist nl;
	namelist *nlp;
	static readdir_res res; /* \fImust be static\fP! */
	
.ft I
	/*
	 * Open directory
	 */
.ft L
	dirp = opendir(*dirname);
	if (dirp == NULL) {
		res.errno = errno;
		return (&res);
	}
.sp .5
.ft I
	/*
	 * Free previous result
	 */
.ft L
	xdr_free(xdr_readdir_res, &res);
.sp .5
.ft I
	/*
	 * Collect directory entries.
	 * Memory allocated here will be freed by \fIxdr_free\fP
	 * next time \fIreaddir_1\fP is called
	 */
.ft L
	nlp = &res.readdir_res_u.list;
	while (d = readdir(dirp)) {
		nl = *nlp = (namenode *) malloc(sizeof(namenode));
		nl->name = strdup(d->d_name);
		nlp = &nl->next;
	}
	*nlp = NULL;
.sp .5
.ft I
	/*
	 * Return the result
	 */
.ft L
	res.errno = 0;
	closedir(dirp);
	return (&res);
}
.DE
Finally, there is the client side program to call the server:
.DS
.ft I
/*
 * rls.c: Remote directory listing client
 */
.ft L
#include <stdio.h>
#include <rpc/rpc.h>	/* \fIalways need this\fP */
#include "dir.h"		/* \fIwill be generated by rpcgen\fI */
.sp .5
extern int errno;
.sp .5
main(argc, argv)
	int argc;
	char *argv[];
{
	CLIENT *cl;
	char *server;
	char *dir;
	readdir_res *result;
	namelist nl;
.sp .5
	if (argc != 3) {
		fprintf(stderr, "usage: %s host directory\en", 
		  argv[0]);
		exit(1);
	}
.sp .5
.ft I
	/*
	 * Remember what our command line arguments refer to
	 */
.ft L
	server = argv[1];
	dir = argv[2];
.sp .5
.ft I
	/*
	 * Create client "handle" used for calling \fIDIRPROG\fP on the
	 * server designated on the command line.  We tell the RPC package
	 * to use any visible transport when contacting the server.
	 */
.ft L
	cl = clnt_create(server, DIRPROG, DIRVERS, "visible");
	if (cl == NULL) {
.ft I
		/*
		 * Couldn't establish connection with server.
		 * Print error message and die.
		 */
.ft L
		clnt_pcreateerror(server);
		exit(1);
	}
	
.ft I
	/*
	 * Call the remote procedure \fIreaddir\fP on the server
	 */
.ft L
	result = readdir_1(&dir, cl);
	if (result == NULL) {
.ft I
		/*
		 * An error occurred while calling the server.
	 	 * Print error message and die.
		 */
.ft L
		clnt_perror(cl, server);
		exit(1);
	}
.sp .5
.ft I
	/*
	 * Okay, we successfully called the remote procedure.
	 */
.ft L
	if (result->errno != 0) {
.ft I
		/*
		 * A remote system error occurred.
		 * Print error message and die.
		 */
.ft L
		errno = result->errno;
		perror(dir);
		exit(1);
	}
.sp .5
.ft I
	/*
	 * Successfully got a directory listing.
	 * Print it out.
	 */
.ft L
	for (nl = result->readdir_res_u.list; nl != NULL;
	  nl = nl->next) {
		printf("%s\en", nl->name);
	}
	exit(0);
}
.DE
Compile everything, and run.
.DS
sun%  \fBrpcgen dir.x\fP
sun%  \fBcc -c dir_xdr.c\fP
sun%  \fBcc rls.c dir_clnt.c dir_xdr.o -o rls\fP
sun%  \fBcc dir_svc.c dir_proc.c dir_xdr.o -o dir_svc\fP
.sp .5
sun%  \fBdir_svc\fP
.sp .5
moon%  \fBrls sun /usr/pub\fP
\&.
\&..
ascii
eqnchar
greek
kbd
marg8
tabclr
tabs
tabs4
moon%
.DE
.LP
.IX "debugging with rpcgen" "" "debugging with \fIrpcgen\fP"
A final note about
.I rpcgen :
The client program and the server procedure can be tested together
as a single program by simply linking them with each other rather
than with the client and server stubs.  The procedure calls will be
executed as ordinary local procedure calls and the program can be
debugged with a local debugger.  When the program is working, the 
client program can be linked to the client stub produced by
.I rpcgen 
and the server procedures can be linked to the server stub produced
by
.I rpcgen .
.SH
.I
If you do this, you will have to comment out calls to RPC library
routines (e.g. 
.I clnt_create ()),
and have client-side routines call server routines directly.
.LP
.NH 3
\&The C-Preprocessor
.IX rpcgen "C-preprocessor" \fIrpcgen\fP
.LP
The C-preprocessor, 
.I cpp ,
is run on all input files before they are compiled, so all the 
preprocessor directives are legal within a \*Q.x\*U file.  Five symbols 
may have been defined, depending upon which output file is getting 
generated.  They are:
.TS
box tab (&);
lfI lfI
lfL l .
Symbol&Usage
_
RPC_HDR&For header-file output
RPC_XDR&For XDR routine output	
RPC_SVC&For server-skeleton output
RPC_CLNT&For client stub output
RPC_TBL&For index table output
.TE	
.LP
Also,
.I rpcgen 
does a little preprocessing of its own.  Any line that
begins with a percent sign is passed directly into the output file,
without any interpretation of the line.  Here is a simple example that
demonstrates the preprocessing features.
.DS
.ft I
/*
 * time.x: Remote time protocol
 */
.ft L
program TIMEPROG {
		version TIMEVERS {
				unsigned int TIMEGET(void) = 1;
		} = 1;
} = 44;
.sp .5
#ifdef RPC_SVC
%int *
%timeget_1()
%{
%		static int thetime;
%
%		thetime = time(0);
%		return (&thetime);
%}
#endif
.DE
The '%' feature is not generally recommended, as there is no guarantee
that the compiler will stick the output where you intended.
.NH 3
\&\fBrpcgen\fP Programming Notes
.IX rpcgen "other operations" \fIrpcgen\fP
.sp 2
.NH 4
\&Network Types
.IX rpcgen "network types" \fIrpcgen\fP
.LP
.I rpcgen 
takes optional arguments which allow its user to specify a desired
network type or even a specific network identifier.  (For details about 
network selection, see the 
\fI@TitleOf(networkselection)\fR
section of the 
\fI@TitleOf(rpcpg)\fR
chapter of this manual).
.LP
The \fI-s\fP flag creates a server which responds to requests on all
transports of a specified type.  For example, the invocation 
.DS
example#  \fBrpcgen -s datagram_n prot.x\fP
.DE
creates a server which responds to any of the connectionless transports
specified in 
.I NETPATH .
.LP
Similarly, the \fI-n\fP flag creates a server which responds to requests 
only the transport specified by a single network identifier.  Be careful 
with this flag \(em since network identifiers are specific to hosts, the 
server produced may not run on other hosts in the expected way.
.NH 4
\&Timeout Changes
.IX rpcgen "timeout changes" \fIrpcgen\fP
.LP
After sending a request to the server, the client waits for a default
time of 25 seconds to get a reply from it.  This timeout may be changed 
using
.I clnt_control()
Here is a small code fragment to demonstrate the use of
.I clnt_control ():
.DS
struct timeval tv;
CLIENT *cl;
.sp .5
cl = clnt_create("somehost", SOMEPROG, SOMEVERS, "visible");
if (cl == NULL) {
	exit(1);
}
tv.tv_sec = 60;	/* \fIchange timeout to 1 minute\fP */
tv.tv_usec = 0;
clnt_control(cl, CLSET_TIMEOUT, &tv);	
.DE
.NH 4
\&Client Authentication
.IX rpcgen "client authentication" \fIrpcgen\fP
.LP
The client create routines do not, by default, have any facilities 
for client authentication, but the client may sometimes want to 
authenticate itself to the user.  Doing so is trivial, and looks 
about like this:
.DS
CLIENT *cl;
.sp .5
cl = client_create("somehost", SOMEPROG, SOMEVERS, "visible");
if (cl == NULL) {
	/* \fITo set UNIX style authentication\fP */
	cl->cl_auth = authsys_createdefault();
}
.DE
.NH 4
\&User-Provided Define Statements
.IX "define statements" "" "\fIdefine\fP statements"
.LP
.I rpcgen 
also provides a means of defining symbols and assigning values to 
them.  These defined symbols are passed on to the C preprocessor
when it is invoked.  This facility is useful when the user wants 
to, for example, invoke debugging code which is enabled only when
the 
.I DEBUG 
symbol is defined.  For example:
.DS
example%  \fBrpcgen -DDEBUG proto.x\fP
.DE
.NH 4
\&Handling Broadcast on the Server Side
.IX "broadcast RPC"
.IX rpcgen "broadcast RPC" \fIrpcgen\fP
.LP
When a procedure is known to be called via broadcast RPC,
it is usually wise for the server to not reply unless it can provide
some useful information to the client.  This prevents the network
from getting flooded by useless replies.
.LP
To prevent the server from replying, a remote procedure can
return NULL as its result, and the server code generated by
.I rpcgen 
will detect this and not send out a reply.
.LP
Here is an example of a procedure that replies only if it
thinks it is an NFS server:
.DS
void *
reply_if_nfsserver()
{
	char notnull;	/* \fIjust here so we can use its address\fP */
.sp .5
	if (access("/etc/exports", F_OK) < 0) {
		return (NULL);	/* \fIprevent RPC from replying\fP */
	}
.ft I
	/*
	 * return non-null pointer so RPC will send out a reply
	 */
.ft L
	return ((void *)&notnull);
}
.DE
Note that if procedure returns type \*Qvoid *\*U, they must return a non-NULL
pointer if they want RPC to reply for them.
.NH 4
\&Portmonitor Support
.IX "rpcgen portmonitor support" "" "\fIrpcgen\fP portmonitor support"
.LP
Portmonitors such as 
.I inetd 
and
.I listen 
monitor network addresses on behalf of the specified RPC services.
Whenever a request comes in for a particular service, the portmonitor forks
off the server that provides for it.  In many applications, it is useful
for services to hang around after satisfying a servicing request, on the
chance that another will follow.  However, if there is no call within the
specified time, the server will exit and the portmonitor will continue to
monitor requests for its services.  By default, services wait of 120
seconds after servicing a request before exiting.  The user can, however,
change that interval with the \fI-K\fP flag.
.DS
example%  \fBrpcgen -K 20 proto.x\fP
.DE
Here the server will wait only for 20 seconds before exiting.  If you want
the server to exit immediately, \fI-K 0\fP can be used, while if the server
is intended to stay around forever (a normal server) the appropriate
argument is \fI-K -1\fP.
.LP
All servers generated by 
.I rpcgen 
require the following support from portmonitors:
.IP \*(SQ
They should be passed the name of the transport provider through the
environment variable
.I NLSPROVIDER .
.IP \*(SQ
The connection should be passed an open TLI file descriptor 0.
.LP
See the
\fI@TitleOf(upmonitors)\fR
section of
\fI@TitleOf(rpcpg)\fP
for a further discussion of portmonitors.
.NH 4
\&Other Information Passed to Server Procedures
.LP
Server procedures will often want to know more about an RPC call
than just its arguments.  For example, getting authentication information
is important to procedures that want to implement some level of security.
This extra information is actually supplied to the server procedure as a
second argument.  (For details see the structure of 
.I svc_req ,
in the
\fI@TitleOf(auth_ch)\fR
section of 
\fI@TitleOf(rpcpg)\fR\.
Here is an example to demonstrate its use.  What we've
done here is rewrite the previous
.I printmessage_1() 
procedure to only allow root users to print a message to the console.
.DS
int *
printmessage_1(msg, rq)
	char **msg;
	struct svc_req	*rq;
{
	static int result;	/* \fIMust be static\fP */
	FILE *f;
	struct authsys_parms *aup;
.sp .5
	aup = (struct authsys_parms *)rq->rq_clntcred;
	if (aup->aup_uid != 0) {
		result = 0;
		return (&result);
	}
.sp
.ft I
	/*
	 * Same code as before.
	 */
.ft L
}
.DE
.NH 4
\&Dispatch Tables
.IX rpcgen "dispatch tables" \fIrpcgen\fP
.LP
There are a number of cases when dispatch tables are useful.  
For example, the server dispatch routine may need to check
authorization and then invoke the service routine;
or a client library may want to deal with the details of
storage management and XDR data conversion.
.LP
When requested with the 
.I -T 
flag, 
.I rpcgen 
generates RPC dispatch tables for each program defined in the protocol
description file,
.I proto.x ,
in the file 
.I proto_tbl.i .
(There is a sample protocol description file given in the 
\fI@TitleOf(genxdr_ch)\fR
section, above).  The suffix \fI.i\fP stands for \*Qindex\*U .  See 
below for how to use this file when compiling programs.  Each entry 
in the table is a 
.I "struct rpcgen_table"
defined in the header file 
.I proto.h 
as follows:
.DS
struct rpcgen_table {
    char        *(*proc)();
    xdrproc_t   xdr_arg;
    unsigned    len_arg;
    xdrproc_t   xdr_res;
    unsigned    len_res;
};
.DE
where
.IP 
\fIproc\fP is a pointer to the service routine,
.IP 
\fIxdr_arg\fP is a pointer to the input (argument) xdr_routine,
.IP 
\fIlen_arg\fP is the length in bytes of the input argument,
.IP 
\fIxdr_res\fP is a pointer to the output (result) xdr_routine, and
.IP 
\fIlen_res\fP is the length in bytes of the output result.
.LP
The table, named
.I dirprog_1_table ,
is indexed by procedure number.  The variable 
.I dirprog_1_ 
contains the number of entries in the table.
.LP
An example of how to locate an procedure in the dispatch 
tables is demonstrated by the routine 
.I find_proc :
.DS
struct rpcgen_table *
find_proc(proc)
	long	proc;
{
	if (proc >= dirprog_1_nproc) 
		/* \fIerror\fP */
	else
		return (&dirprog_1_table[proc]);
}
.DE
.LP
Each entry in the dispatch table contains a pointer to the corresponding
service routine.  However, that service routine is usually not defined
in the client code.  To avoid generating unresolved external references,
and to require only one source file for the dispatch table, the actual 
service routine initializer is 
.I "RPCGEN_ACTION(proc_ver)" .
.LP
This way, the same dispatch table can be included in both the client and 
the server.  Use the following define when compiling the client:
.DS
#define RPCGEN_ACTION(routine)	0
.DE
and use this define when compiling the server:
.DS
#define RPCGEN_ACTION(routine)	routine
.DE
.NH 3
\&RPC Language
.IX RPCL
.IX rpcgen "RPC Language" \fIrpcgen\fP
.LP
RPC language is an extension of XDR language.  The sole extension is
the addition of the
.I program 
and 
.I version 
types.  For a complete description of the XDR language syntax, see the
\fI@TitleOf(xdr_spec)\fR
chapter.  For a description of the RPC extensions to the XDR language,
see the
\fI@TitleOf(rpc_spec)\fR
chapter.
.LP
However, XDR language is so close to C that if you know C, you know most
of it already.  We describe here the syntax of the RPC language,
showing a few examples along the way.  We also show how the various
RPC and XDR type definitions get compiled into C type definitions in
the output header file.
.NH 4
\&
.IX rpcgen definitions \fIrpcgen\fP
.LP
An RPC language file consists of a series of definitions.
.DS
	definition-list:
		definition ";"
		definition ";" definition-list
.DE
It recognizes five types of definitions.
.DS
	definition:
		enum-definition
		const-definition
		typedef-definition
		struct-definition
		union-definition
		program-definition
.DE
.NH 4
\&
.IX rpcgen enumerations \fIrpcgen\fP
.LP
XDR enumerations have the same syntax as C enumerations.
.DS
	enum-definition:
		"enum" enum-ident "{"
			enum-value-list
		"}"
.sp .5
	enum-value-list:
		enum-value
		enum-value "," enum-value-list
.sp .5
	enum-value:
		enum-value-ident
		enum-value-ident "=" value
.DE
Here is a short example of an XDR enum, and the C enum that it gets
compiled into.
.DS
     enum colortype {      enum colortype {
          RED = 0,              RED = 0,
          GREEN = 1,   -->      GREEN = 1,
          BLUE = 2              BLUE = 2,
     };                    };
                           typedef enum colortype colortype;
.DE
.NH 4
\&
.IX rpcgen constants \fIrpcgen\fP
.LP
XDR constants are symbolic constants that may be used wherever a
integer constant is used, for example, in array size specifications.
.DS
	const-definition:
		"const" const-ident "=" integer
.DE
For example, the following defines a constant
.I DOZEN 
equal to 12.
.DS
	const DOZEN = 12;  -->  #define DOZEN 12
.DE
.NH 4
\&
.IX rpcgen typedef \fIrpcgen\fP
.LP
XDR typedefs have the same syntax as C typedefs.
.DS
	typedef-definition:
		"typedef" declaration
.DE
Here is an example that defines a
.I fname_type 
used for declaring
file name strings that have a maximum length of 255 characters.
.DS
typedef string fname_type<255>; --> typedef char *fname_type;
.DE
.NH 4
\&
.IX rpcgen declarations \fIrpcgen\fP
.LP
In XDR, there are only four kinds of declarations.
.DS
	declaration:
		simple-declaration
		fixed-array-declaration
		variable-array-declaration
		pointer-declaration
.DE
\fB1) Simple declarations\fP are just like simple C declarations.
.DS
	simple-declaration:
		type-ident variable-ident
.DE
Example:
.DS
	colortype color;	--> colortype color;
.DE
\fB2) Fixed-length Array Declarations\fP are just like C array declarations:
.DS
	fixed-array-declaration:
		type-ident variable-ident "[" value "]"
.DE
Example:
.DS
	colortype palette[8];	--> colortype palette[8];
.DE
\fB3) Variable-Length Array Declarations\fP have no explicit syntax
in C, so XDR invents its own using angle-brackets.
.DS
variable-array-declaration:
	type-ident variable-ident "<" value ">"
	type-ident variable-ident "<" ">"
.DE
The maximum size is specified between the angle brackets.  The size may
be omitted, indicating that the array may be of any size.
.DS
	int heights<12>;	/* \fIat most 12 items\fP */
	int widths<>;		/* \fIany number of items\fP */
.DE
Since variable-length arrays have no explicit syntax in C, these
declarations are actually compiled into \*Qstruct\*Us.  For example, the
\*Qheights\*U declaration gets compiled into the following struct:
.DS
	struct {
		u_int heights_len;	/* \fI# of items in array\fP */
		int *heights_val;	/* \fIpointer to array\fP */
	} heights;
.DE
Note that the number of items in the array is stored in the \*Q_len\*U
component and the pointer to the array is stored in the \*Q_val\*U
component.  The first part of each of these component's names is the
same as the name of the declared XDR variable.
.LP
\fB4) Pointer Declarations\fP are made in
XDR exactly as they are in C.  You can't
really send pointers over the network, but you can use XDR pointers
for sending recursive data types such as lists and trees.  The type is
actually called \*Qoptional-data\*U, not \*Qpointer\*U, in XDR language.
.DS
pointer-declaration:
	type-ident "*" variable-ident
.DE
Example:
.DS
	listitem *next;  -->  listitem *next;
.DE
.NH 4
\&
.IX rpcgen structures \fIrpcgen\fP
.LP
An XDR struct is declared almost exactly like its C counterpart.  It
looks like the following:
.DS
	struct-definition:
		"struct" struct-ident "{"
			declaration-list
		"}"
.sp .5
	declaration-list:
		declaration ";"
		declaration ";" declaration-list
.DE
As an example, here is an XDR structure to a two-dimensional
coordinate, and the C structure that it gets compiled into in the
output header file.
.DS
   struct coord {             struct coord {
        int x;       -->           int x;
        int y;                     int y;
   };                         };
                              typedef struct coord coord;
.DE
The output is identical to the input, except for the added
.I typedef 
at the end of the output.  This allows one to use \*Qcoord\*U instead of
\*Qstruct coord\*U when declaring items.
.NH 4
\&
.IX rpcgen unions \fIrpcgen\fP
.LP
XDR unions are discriminated unions, and look quite different from C
unions.  They are more analogous to Pascal variant records than they
are to C unions.
.DS
	union-definition:
		"union" union-ident "switch" "(" simple declaration ")" "{"
			case-list
		"}"
.sp .5
	case-list:
		"case" value ":" declaration ";"
		"case" value ":" declaration ";" case-list
		"default" ":" declaration ";"
.DE
Here is an example of a type that might be returned as the result of a
\*Qread data\*U operation.  If there is no error, return a block of data.
Otherwise, don't return anything.
.DS
	union read_result switch (int errno) {
	case 0:
		opaque data[1024];
	default:
		void;
	};
.DE
It gets compiled into the following:
.DS
	struct read_result {
		int errno;
		union {
			char data[1024];
		} read_result_u;
	};
	typedef struct read_result read_result;
.DE
Notice that the union component of the output struct has the same name as
the type name, except for the trailing \*Q_u\*U.
.NH 4
\&
.IX rpcgen programs \fIrpcgen\fP
.LP
RPC programs are declared using the following syntax:
.DS
program-definition:
	"program" program-ident "{"
		version-list
	"}" "=" value
.sp .5
version-list:
	version ";"
	version ";" version-list
.sp .5
version:
	"version" version-ident "{"
		procedure-list
	"}" "=" value
.sp .5
procedure-list:
	procedure ";"
	procedure ";" procedure-list
.sp .5
procedure:
	type-ident procedure-ident "(" type-ident ")" "=" value
.DE
For example, here is the time protocol, revisited:
.DS
.ft I
/*
 * time.x: Get or set the time.  Time is represented as number of seconds
 * since 0:00, January 1, 1970.
 */
.ft L
program TIMEPROG {
	version TIMEVERS {
		unsigned int TIMEGET(void) = 1;
		void TIMESET(unsigned) = 2;
	} = 1;
} = 44;		
.DE
This file compiles into these #defines in the output header file:
.DS
#define TIMEPROG 44
#define TIMEVERS 1
#define TIMEGET 1
#define TIMESET 2
.DE
.NH 4
\&Special Cases
.IX rpcgen "special cases" \fIrpcgen\fP
.LP
There are a few exceptions to the rules described above.
.LP
.B Booleans:
C has no built-in boolean type.  However, the RPC library does a
boolean type called
.I bool_t 
that is either
.I TRUE 
or
.I FALSE .
Things declared as type
.I bool 
in XDR language are compiled into
.I bool_t 
in the output header file.
.LP
Example:
.DS
	bool married;  -->  bool_t married;
.DE
.B Strings:
C has no built-in string type, but instead uses the null-terminated
\*Qchar *\*U convention.  In XDR language, strings are declared using the
\*Qstring\*U keyword, and compiled into \*Qchar *\*Us in the output header
file.  The maximum size contained in the angle brackets specifies the
maximum number of characters allowed in the strings (not counting the
.I NULL 
character).  The maximum size may be left off, indicating a string
of arbitrary length.
.LP
Examples:
.DS
	string name<32>;  -->  char *name;	
	string longname<>;  -->  char *longname;	
.DE
.B "Opaque Data:"
Opaque data is used in RPC and XDR to describe untyped data, that is,
just sequences of arbitrary bytes.  It may be declared either as a
fixed or variable length array.
.DS
Examples:
	opaque diskblock[512];  -->  char diskblock[512];
.sp .5
	opaque filedata<1024>;  -->  struct {
									u_int filedata_len;
									char *filedata_val;
								} filedata;
.DE
.B Voids:
In a void declaration, the variable is not named.  The declaration is
just \*Qvoid\*U and nothing else.  Void declarations can only occur in two
places: union definitions and program definitions (as the argument or
result of a remote procedure).
