/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user or with the express written consent of
 * Sun Microsystems, Inc.
 *
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
/*      @(#)svc.h 1.55 91/03/11 SMI      */

/*
 * svc.h, Server-side remote procedure call interface.
 *
 * Copyright (C) 1984, Sun Microsystems, Inc.
 */

#ifndef _RPC_SVC_H
#define _RPC_SVC_H

#include <rpc/rpc_com.h>

/*
 * This interface must manage two items concerning remote procedure calling:
 *
 * 1) An arbitrary number of transport connections upon which rpc requests
 * are received. They are created and registered by routines in svc_generic.c,
 * svc_vc.c and svc_dg.c; they in turn call xprt_register and 
 * xprt_unregister.
 *
 * 2) An arbitrary number of locally registered services.  Services are
 * described by the following four data: program number, version number,
 * "service dispatch" function, a transport handle, and a boolean that
 * indicates whether or not the exported program should be registered with a
 * local binder service;  if true the program's number and version and the
 * address from the transport handle are registered with the binder.
 * These data are registered with rpcbind via svc_reg().
 *
 * A service's dispatch function is called whenever an rpc request comes in
 * on a transport.  The request's program and version numbers must match
 * those of the registered service.  The dispatch function is passed two
 * parameters, struct svc_req * and SVCXPRT *, defined below.
 */

enum xprt_stat {
	XPRT_DIED,
	XPRT_MOREREQS,
	XPRT_IDLE
};

/*
 * Server side transport handle
 */
typedef struct {
	int		xp_fd;
#define xp_sock		xp_fd
	u_short		xp_port;	 /* associated port number.
					  * Obsoleted, but still used to
					  * specify whether rendezvouser
					  * or normal connection
					  */
	struct xp_ops {
#if (__cplusplus || c_plusplus)
	    bool_t	(*xp_recv)(...);	 /* receive incoming requests */
	    xprt_stat	(*xp_stat)(...); /* get transport status */
	    bool_t	(*xp_getargs)(...); /* get arguments */
	    bool_t	(*xp_reply)(...);	 /* send reply */
	    bool_t	(*xp_freeargs)(...);/* free mem allocated for args */
	    void	(*xp_destroy)(...); /* destroy this struct */
#elif __STDC__
	    bool_t	(*xp_recv)(...);	 /* receive incoming requests */
	    enum xprt_stat (*xp_stat)(...); /* get transport status */
	    bool_t	(*xp_getargs)(...); /* get arguments */
	    bool_t	(*xp_reply)(...);	 /* send reply */
	    bool_t	(*xp_freeargs)(...);/* free mem allocated for args */
	    void	(*xp_destroy)(...); /* destroy this struct */
#else
	    bool_t	(*xp_recv)();	 /* receive incoming requests */
	    enum xprt_stat (*xp_stat)(); /* get transport status */
	    bool_t	(*xp_getargs)(); /* get arguments */
	    bool_t	(*xp_reply)();	 /* send reply */
	    bool_t	(*xp_freeargs)();/* free mem allocated for args */
	    void	(*xp_destroy)(); /* destroy this struct */
#endif
	} *xp_ops;
	int		xp_addrlen;	 /* length of remote addr. Obsoleted */
	char		*xp_tp;		 /* transport provider device name */
	char		*xp_netid;	 /* network token */
	struct netbuf	xp_ltaddr;	 /* local transport address */
	struct netbuf	xp_rtaddr;	 /* remote transport address */
	char		xp_raddr[16];	 /* remote address. Now obsoleted */
	struct opaque_auth xp_verf;	 /* raw response verifier */
	caddr_t		xp_p1;		 /* private: for use by svc ops */
	caddr_t		xp_p2;		 /* private: for use by svc ops */
	caddr_t		xp_p3;		 /* private: for use by svc lib */
} SVCXPRT;

/*
 *  Approved way of getting address of caller
 */
#define svc_getrpccaller(x) (&(x)->xp_rtaddr)

/*
 * Operations defined on an SVCXPRT handle
 *
 * SVCXPRT		*xprt;
 * struct rpc_msg	*msg;
 * xdrproc_t		 xargs;
 * caddr_t		 argsp;
 */
#define SVC_RECV(xprt, msg)				\
	(*(xprt)->xp_ops->xp_recv)((xprt), (msg))
#define svc_recv(xprt, msg)				\
	(*(xprt)->xp_ops->xp_recv)((xprt), (msg))

#define SVC_STAT(xprt)					\
	(*(xprt)->xp_ops->xp_stat)(xprt)
#define svc_stat(xprt)					\
	(*(xprt)->xp_ops->xp_stat)(xprt)

#define SVC_GETARGS(xprt, xargs, argsp)			\
	(*(xprt)->xp_ops->xp_getargs)((xprt), (xargs), (argsp))
#define svc_getargs(xprt, xargs, argsp)			\
	(*(xprt)->xp_ops->xp_getargs)((xprt), (xargs), (argsp))

#define SVC_REPLY(xprt, msg)				\
	(*(xprt)->xp_ops->xp_reply) ((xprt), (msg))
#define svc_reply(xprt, msg)				\
	(*(xprt)->xp_ops->xp_reply) ((xprt), (msg))

#define SVC_FREEARGS(xprt, xargs, argsp)		\
	(*(xprt)->xp_ops->xp_freeargs)((xprt), (xargs), (argsp))
#define svc_freeargs(xprt, xargs, argsp)		\
	(*(xprt)->xp_ops->xp_freeargs)((xprt), (xargs), (argsp))

#define SVC_DESTROY(xprt)				\
	(*(xprt)->xp_ops->xp_destroy)(xprt)
#define svc_destroy(xprt)				\
	(*(xprt)->xp_ops->xp_destroy)(xprt)


/*
 * Service request
 */
struct svc_req {
	u_long		rq_prog;	/* service program number */
	u_long		rq_vers;	/* service protocol version */
	u_long		rq_proc;	/* the desired procedure */
	struct opaque_auth rq_cred;	/* raw creds from the wire */
	caddr_t		rq_clntcred;	/* read only cooked cred */
	SVCXPRT		*rq_xprt;	/* associated transport */
};


/*
 * Service registration
 *
 * svc_reg(xprt, prog, vers, dispatch, nconf)
 *	SVCXPRT *xprt;
 *	u_long prog;
 *	u_long vers;
 *	void (*dispatch)();
 *	struct netconfig *nconf;
 */
#ifdef __cplusplus
extern "C" bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
extern "C" bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
extern "C" bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
#elif (__STDC__ || c_plusplus)
extern bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
extern bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
extern bool_t	svc_reg (SVCXPRT *, u_long, u_long, void (*)(...), struct netconfig *);
#else
extern bool_t	svc_reg ();
extern bool_t	svc_reg ();
extern bool_t	svc_reg ();
#endif

/*
 * Service un-registration
 *
 * svc_unreg(prog, vers)
 *	u_long prog;
 *	u_long vers;
 */
#ifdef __cplusplus
extern "C" void	svc_unreg (u_long, u_long);
#elif (__STDC__ || c_plusplus)
extern void	svc_unreg (u_long, u_long);
#else
extern void	svc_unreg ();
#endif

/*
 * Transport registration.
 *
 * xprt_register(xprt)
 *	SVCXPRT *xprt;
 */
#ifdef __cplusplus
extern "C" void	xprt_register (SVCXPRT *);
#elif (__STDC__ || c_plusplus)
extern void	xprt_register (SVCXPRT *);
#else
extern void	xprt_register ();
#endif

/*
 * Transport un-register
 *
 * xprt_unregister(xprt)
 *	SVCXPRT *xprt;
 */
#ifdef __cplusplus
extern "C" void	xprt_unregister (SVCXPRT *);
#elif (__STDC__ || c_plusplus)
extern void	xprt_unregister (SVCXPRT *);
#else
extern void	xprt_unregister ();
#endif


/*
 * When the service routine is called, it must first check to see if it
 * knows about the procedure;  if not, it should call svcerr_noproc
 * and return.  If so, it should deserialize its arguments via 
 * SVC_GETARGS (defined above).  If the deserialization does not work,
 * svcerr_decode should be called followed by a return.  Successful
 * decoding of the arguments should be followed the execution of the
 * procedure's code and a call to svc_sendreply.
 *
 * Also, if the service refuses to execute the procedure due to too-
 * weak authentication parameters, svcerr_weakauth should be called.
 * Note: do not confuse access-control failure with weak authentication!
 *
 * NB: In pure implementations of rpc, the caller always waits for a reply
 * msg.  This message is sent when svc_sendreply is called.  
 * Therefore pure service implementations should always call
 * svc_sendreply even if the function logically returns void;  use
 * xdr.h - xdr_void for the xdr routine.  HOWEVER, connectionful rpc allows
 * for the abuse of pure rpc via batched calling or pipelining.  In the
 * case of a batched call, svc_sendreply should NOT be called since
 * this would send a return message, which is what batching tries to avoid.
 * It is the service/protocol writer's responsibility to know which calls are
 * batched and which are not.  Warning: responding to batch calls may
 * deadlock the caller and server processes!
 */

#ifdef __cplusplus
extern "C" bool_t	svc_sendreply (SVCXPRT *, xdrproc_t, caddr_t);
extern "C" void	svcerr_decode (SVCXPRT *);
extern "C" void	svcerr_weakauth (SVCXPRT *);
extern "C" void	svcerr_noproc (SVCXPRT *);
extern "C" void	svcerr_progvers (SVCXPRT *, u_long, u_long);
extern "C" void	svcerr_auth (SVCXPRT *, enum auth_stat);
extern "C" void	svcerr_noprog (SVCXPRT *);
extern "C" void	svcerr_systemerr (SVCXPRT *);
#elif (__STDC__ || c_plusplus)
extern bool_t	svc_sendreply (SVCXPRT *, xdrproc_t, caddr_t);
extern void	svcerr_decode (SVCXPRT *);
extern void	svcerr_weakauth (SVCXPRT *);
extern void	svcerr_noproc (SVCXPRT *);
extern void	svcerr_progvers (SVCXPRT *, u_long, u_long);
extern void	svcerr_auth (SVCXPRT *, enum auth_stat);
extern void	svcerr_noprog (SVCXPRT *);
extern void	svcerr_systemerr (SVCXPRT *);
#else
extern bool_t	svc_sendreply ();
extern void	svcerr_decode ();
extern void	svcerr_weakauth ();
extern void	svcerr_noproc ();
extern void	svcerr_progvers ();
extern void	svcerr_auth ();
extern void	svcerr_noprog ();
extern void	svcerr_systemerr ();
#endif
    
/*
 * Lowest level dispatching -OR- who owns this process anyway.
 * Somebody has to wait for incoming requests and then call the correct
 * service routine.  The routine svc_run does infinite waiting; i.e.,
 * svc_run never returns.
 * Since another (co-existant) package may wish to selectively wait for
 * incoming calls or other events outside of the rpc architecture, the
 * routine svc_getreq is provided.  It must be passed readfds, the
 * "in-place" results of a select call (see select, section XXX).
 */

/*
 * Global keeper of rpc service descriptors in use
 * dynamic; must be inspected before each call to select 
 */
extern fd_set svc_fdset;
#define svc_fds svc_fdset.fds_bits[0]	/* compatibility */

/*
 * a small program implemented by the svc_rpc implementation itself;
 * also see clnt.h for protocol numbers.
 */
#ifdef __cplusplus
extern "C" void rpctest_service (...);
extern "C" void	svc_getreq (int);
extern "C" void	svc_getreqset (fd_set *);	/* takes fdset instead of int */
extern "C" void	svc_run ();
#elif c_plusplus
extern void rpctest_service (...);
extern void	svc_getreq (int);
extern void	svc_getreqset (fd_set *);	/* takes fdset instead of int */
extern void	svc_run ();
#elif __STDC__
extern void rpctest_service (...);
extern void	svc_getreq (int);
extern void	svc_getreqset (fd_set *);	/* takes fdset instead of int */
extern void	svc_run (void);
#else
extern void rpctest_service ();
extern void	svc_getreq ();
extern void	svc_getreqset ();
extern void	svc_run ();
#endif


/*
 * These are the existing service side transport implementations
 */
/*
 * Transport independent svc_create routine.
 */
#ifdef __cplusplus
extern " C" int svc_create (void (*)(...), u_long, u_long, char * /* dispatch, prognum, versnum, nettype*/); /*
	void (*dispatch)();		-- dispatch routine
	u_long prognum;			-- program number
	u_long versnum;			-- version number
	char *nettype;			-- network type
*/
#elif (__STDC__ || c_plusplus)
extern int svc_create (void (*)(...), u_long, u_long, char * /* dispatch, prognum, versnum, nettype*/);
#else
extern int svc_create ();
#endif

/*
 * Generic server creation routine. It takes a netconfig structure
 * instead of a nettype.
 */
#ifdef __cplusplus
extern SVCXPRT	* svc_tp_create (void (*)(...), u_long, u_long, struct netconfig * /* dispatch, prognum, versnum, nconf*/); /*
	void (*dispatch)();		-- dispatch routine
	u_long prognum;			-- program number
	u_long versnum;			-- version number
	struct netconfig *nconf;	-- netconfig structure
*/
#elif (__STDC__ || c_plusplus)
extern SVCXPRT	* svc_tp_create (void (*)(...), u_long, u_long, struct netconfig *);
#else
extern SVCXPRT	* svc_tp_create ();
#endif

/*
 * Generic TLI create routine
 */
#ifdef __cplusplus
extern "C" SVCXPRT * svc_tli_create (int, struct netconfig *, struct t_bind *, u_int, u_int /* fd, nconf, bindaddr, sendsz, recvsz*/); /*
	int fd;			 	-- connection end point
	struct netconfig *nconf;	-- netconfig structure for network
	struct t_bind *bindaddr;	-- local bind address
	u_int sendsz;			-- max sendsize
	u_int recvsz;			-- max recvsize
*/
#elif (__STDC__ || c_plusplus)
extern SVCXPRT * svc_tli_create (int, struct netconfig *, struct t_bind *, u_int, u_int);
#else
extern SVCXPRT * svc_tli_create ();
#endif

/*
 * Connectionless and connectionful create routines
 */
#ifdef __cplusplus
extern "C" SVCXPRT	* svc_vc_create (int, u_int, u_int /* fd, sendsize, recvsize*/); /*
	int fd;				-- open connection end point
	u_int sendsize;			-- max send size
	u_int recvsize;			-- max recv size
*/

extern "C" SVCXPRT	* svc_dg_create (int, u_int, u_int /* fd, sendsize, recvsize*/); /*
	int fd;				-- open connection end point
	u_int sendsize;			-- max send size
	u_int recvsize;			-- max recv size
*/
#elif (__STDC__ || c_plusplus)
extern SVCXPRT	* svc_vc_create (int, u_int, u_int);
extern SVCXPRT	* svc_dg_create (int, u_int, u_int);
#else
extern SVCXPRT	* svc_vc_create ();
extern SVCXPRT	* svc_dg_create ();
#endif

/*
 * the routine takes any *open* TLI file
 * descriptor as its first input and is used for open connections.
 */
#ifdef __cplusplus
extern "C" SVCXPRT * svc_fd_create (int, u_int, u_int /* fd, sendsize, recvsize*/); /*
	int fd;				-- open connection end point
	u_int sendsize;			-- max send size
	u_int recvsize;			-- max recv size
*/
#elif (__STDC__ || c_plusplus)
extern SVCXPRT * svc_fd_create (int, u_int, u_int);
#else
extern SVCXPRT * svc_fd_create ();
#endif

/*
 * Memory based rpc (for speed check and testing)
 */
#ifdef __cplusplus
extern "C" SVCXPRT * svc_raw_create ();
#elif __STDC__
extern SVCXPRT * svc_raw_create (void);
#else
extern SVCXPRT * svc_raw_create ();
#endif

#ifdef PORTMAP
/* For backword compatibility */
#include <rpc/svc_soc.h>
#endif

#endif /* !_RPC_SVC_H */
