/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user or with the express written consent of
 * Sun Microsystems, Inc.
 *
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
#if !defined(lint) && defined(SCCSIDS)
static char sccsid[] = "@(#)clnt_dg.c 1.31 91/03/11 Copyr 1988 Sun Micro";
#endif

/*
 * clnt_dg.c, Copyright (C) 1988, Sun Microsystems, Inc.
 *
 * Implements a connectionless client side RPC.
 */

#include <rpc/rpc.h>
#include <rpc/rac.h>
#include <rpc/rac_private.h>
#ifndef	NDEBUG
#include <stdio.h>
#endif
#include <assert.h>
#include <errno.h>
#ifdef SYSLOG
#include <sys/syslog.h>
#else
#define	LOG_ERR 3
#endif /* SYSLOG */

#define	RPC_MAX_BACKOFF		30 /* seconds */
#define	MCALL_MSG_SIZE		24
extern int errno;
extern int t_errno;

enum clnt_dg_receive_stat { RS_RESEND, RS_ERROR, RS_OK };
enum clnt_dg_xdrin_stat { XS_CALLAGAIN, XS_ERROR, XS_OK };

extern char *malloc();
static struct clnt_ops *clnt_dg_ops();
static enum clnt_stat clnt_dg_marshall();
static enum clnt_stat clnt_dg_send();
static enum clnt_dg_receive_stat clnt_dg_receive();
static enum clnt_dg_xdrin_stat clnt_dg_xdrin();

static void		rac_dg_drop();
static enum clnt_stat	rac_dg_poll();
static enum clnt_stat	rac_dg_recv();
static void		*rac_dg_send();
static bool_t		rachandle_is_valid();
static struct callinfo		*xid_to_callinfo();

/*
 * Private data kept per client handle
 */
struct cu_data {
	/* per-CLIENT information */
	int			cu_fd;		/* connections fd */
	bool_t			cu_closeit;	/* opened by library */
	struct timeval		cu_wait;	/* retransmit interval */
	struct timeval		cu_total;	/* total time for the call */
	struct netbuf		cu_addr;	/* remote address */
	char			cu_mcallproto[MCALL_MSG_SIZE];	/* prototype marshalled callmsg */
	u_int			cu_msize;	/* #bytes in cu_mcallproto */
	u_int			cu_sendsz;	/* per-call send size */
	u_int			cu_recvsz;	/* per-call receive size */
	u_long			cu_xidseed;	/* XID seed */
	struct callinfo		*cu_calls;	/* per-call information chain */
};

struct callinfo {
	u_int			ci_flags;	/* per-call flags */
#define	CI_ASYNC	1
#define	CI_SYNC		2
	struct rpc_err		ci_error;	/* call error information */
	int			ci_nrefreshes;	/* times to refresh cred */
	struct timeval		ci_timewaited;	/* time we've waited so far */
	struct timeval		ci_rexmittime;	/* time until rexmit due */
	struct timeval		ci_calltimeout;	/* total time for the call */
	struct timeval		ci_sendtime;	/* time of rac_dg_send call */
	u_long			ci_xid;		/* transaction id */
	u_long			ci_proc;	/* remote procedure */
	xdrproc_t		ci_xargs;	/* XDR routine for arguments */
	caddr_t			ci_argsp;	/* pointer to args buffer */
	xdrproc_t		ci_xresults;	/* XDR routine for results */
	caddr_t			ci_resultsp;	/* pointer to results buffer */
	char			*ci_outbuf;	/* per-call output buffer */
	XDR			ci_outxdrs;
	u_int			ci_xdrpos;	/* position in ci_outxdrs */
	struct t_unitdata	*ci_trdata;
	struct cu_data		*ci_cu;		/* per-CLIENT information */
	struct callinfo		*ci_next;	/* info on ``next'' call */
};
static struct callinfo	*alloc_callinfo();
static struct callinfo	*find_callinfo();
static void		free_callinfo();
static void		dequeue_callinfo();

/*
 * Connectionless client creation returns with client handle parameters.
 * Default options are set, which the user can change using clnt_control().
 * fd should be open and bound.
 * NB: The cl->cl_auth is initialized to null authentication.
 * 	Caller may wish to set this something more useful.
 *
 * sendsz and recvsz are the maximum allowable packet sizes that can be
 * sent and received. Normally they are the same, but they can be
 * changed to improve the program efficiency and buffer allocation.
 * If they are 0, use the transport default.
 *
 * If svcaddr is NULL, returns NULL.
 */
CLIENT *
clnt_dg_create(fd, svcaddr, program, version, sendsz, recvsz)
	int fd;				/* open file descriptor */
	struct netbuf *svcaddr;		/* servers address */
	u_long program;			/* program number */
	u_long version;			/* version number */
	u_int sendsz;			/* buffer recv size */
	u_int recvsz;			/* buffer send size */
{
	CLIENT *cl = NULL;			/* client handle */
	register struct cu_data *cu = NULL;	/* private data */
	struct t_info tinfo;
	struct rpc_msg call_msg;
	struct timeval now;
	XDR tmpxdrs;

	if (svcaddr == (struct netbuf *)NULL) {
		rpc_createerr.cf_stat = RPC_UNKNOWNADDR;
		return ((CLIENT *)NULL);
	}

	if ((cl = (CLIENT *) mem_alloc(sizeof (CLIENT))) == (CLIENT *)NULL)
		goto err;
	cu = (struct cu_data *) mem_alloc(sizeof (*cu));
	if (cu == (struct cu_data *)NULL)
		goto err;
	cu->cu_addr = *svcaddr;
	if ((cu->cu_addr.buf = mem_alloc(svcaddr->len)) == NULL)
		goto err;
	(void) memcpy(cu->cu_addr.buf, svcaddr->buf, (int)svcaddr->len);
	/* Other values can also be set through clnt_control() */
	cu->cu_wait.tv_sec = 15;	/* heuristically chosen */
	cu->cu_wait.tv_usec = 0;
	cu->cu_total.tv_sec = -1;
	cu->cu_total.tv_usec = -1;

	/* XID is set in clnt_dg_marshall() */
	call_msg.rm_call.cb_prog = program;
	call_msg.rm_call.cb_vers = version;
	xdrmem_create(&tmpxdrs, cu->cu_mcallproto, MCALL_MSG_SIZE, XDR_ENCODE);
	if (! xdr_callhdr(&tmpxdrs, &call_msg)) {
		goto err;
	}
	cu->cu_msize = XDR_GETPOS(&tmpxdrs);
	XDR_DESTROY(&tmpxdrs);

	if (t_getinfo(fd, &tinfo) == -1) {
		if ((sendsz == 0) || (recvsz == 0)) {
			rpc_createerr.cf_stat = RPC_TLIERROR;
			rpc_createerr.cf_error.re_errno = 0;
			rpc_createerr.cf_error.re_terrno = t_errno;
			return ((CLIENT *)NULL);
		}
	} else {
		/*
		 * Find the receive and the send size
		 */
		sendsz = _rpc_get_t_size((int)sendsz, tinfo.tsdu);
		recvsz = _rpc_get_t_size((int)recvsz, tinfo.tsdu);
	}
	/*
	 * Should be multiple of 4 for XDR.
	 */
	cu->cu_sendsz = ((sendsz + 3) / 4) * 4;
	cu->cu_recvsz = ((recvsz + 3) / 4) * 4;

	(void) gettimeofday(&now, (struct timezone *) 0);
	cu->cu_xidseed = getpid() ^ now.tv_sec ^ now.tv_usec;

	/*
	 * By default, closeit is always FALSE. It is users responsibility
	 * to do a t_close on it, else the user may use clnt_control
	 * to let clnt_destroy do it for him/her.
	 */
	cu->cu_closeit = FALSE;
	cu->cu_fd = fd;
	cu->cu_calls = (struct callinfo *) NULL;
	cl->cl_ops = clnt_dg_ops();
	cl->cl_private = (caddr_t)cu;
	cl->cl_auth = authnone_create();
	cl->cl_tp = (char *) NULL;
	cl->cl_netid = (char *) NULL;
	return (cl);
err:
	(void) syslog(LOG_ERR, "clnt_dg_create: out of memory");
	rpc_createerr.cf_stat = RPC_SYSTEMERROR;
	rpc_createerr.cf_error.re_errno = errno;
	rpc_createerr.cf_error.re_terrno = 0;
	if (cl) {
		mem_free((caddr_t)cl, sizeof (CLIENT));
		if (cu)
			mem_free((caddr_t)cu, sizeof (*cu));
	}
	return ((CLIENT *)NULL);
}

static enum clnt_stat
clnt_dg_call(cl, proc, xargs, argsp, xresults, resultsp, utimeout)
	register CLIENT	*cl;		/* client handle */
	u_long		proc;		/* procedure number */
	xdrproc_t	xargs;		/* xdr routine for args */
	caddr_t		argsp;		/* pointer to args */
	xdrproc_t	xresults;	/* xdr routine for results */
	caddr_t		resultsp;	/* pointer to results */
	struct timeval	utimeout;	/* seconds to wait before giving up */
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci;

	if ((ci = find_callinfo(cu, CI_SYNC)) == (struct callinfo *) NULL)
		if ((ci = alloc_callinfo(cu, CI_SYNC)) == (struct callinfo *) 0)
		return (RPC_SYSTEMERROR);
	ci->ci_nrefreshes = 2;	/* number of times to refresh cred */
	ci->ci_proc = proc;
	ci->ci_xargs = xargs;
	ci->ci_argsp = argsp;
	ci->ci_xresults = xresults;
	ci->ci_resultsp = resultsp;
	xdrmem_create(&(ci->ci_outxdrs), ci->ci_outbuf, cu->cu_sendsz, XDR_ENCODE);
	ci->ci_xid = ++cu->cu_xidseed;
	
	if (cu->cu_total.tv_usec == -1) {
		ci->ci_calltimeout = utimeout;	/* use supplied timeout */
	} else {
		ci->ci_calltimeout = cu->cu_total; /* use default timeout */
	}
	ci->ci_trdata = NULL;

	timerclear(&ci->ci_timewaited);
	ci->ci_rexmittime = cu->cu_wait;

call_again:
#ifdef	PRINTFS
	printf("clnt_dg_call:  call_again\n");
#endif
	if (clnt_dg_marshall(cl, ci) != RPC_SUCCESS) {
		dequeue_callinfo(cu, ci);
		free_callinfo(ci);
		return (ci->ci_error.re_status);
	}
#ifdef	PRINTFS
	else
		printf("clnt_dg_call:  clnt_dg_marshall succeeded\n");
#endif

send_again:
	if (clnt_dg_send(ci) != RPC_SUCCESS) {
		dequeue_callinfo(cu, ci);
		free_callinfo(ci);
		return (ci->ci_error.re_status);
	}
#ifdef	PRINTFS
	else
		printf("clnt_dg_call:  clnt_dg_send succeeded\n");
#endif

	/*
	 * Hack to provide rpc-based message passing
	 */
	if (! timerisset(&ci->ci_calltimeout))
		return (ci->ci_error.re_status = RPC_TIMEDOUT);

	switch ((int) clnt_dg_receive(ci)) {
	case (int) RS_RESEND:
#ifdef	PRINTFS
		printf("clnt_dg_receive returned RS_RESEND\n");
#endif
		goto send_again;
		/* NOTREACHED */
		
	case (int) RS_ERROR:
#ifdef	PRINTFS
		printf("clnt_dg_receive returned error %d\n", ci->ci_error.re_status);
#endif
		return (ci->ci_error.re_status);
		/* NOTREACHED */
		
	case (int) RS_OK:
#ifdef	PRINTFS
		printf("clnt_dg_receive returned OK\n");
#endif
		break;
	}
	
	switch (clnt_dg_xdrin(cl, ci)) {
	case (int) XS_CALLAGAIN:
#ifdef	PRINTFS
		printf("clnt_dg_xdrin returned CALL_AGAIN\n");
#endif
		goto call_again;

	case (int) XS_ERROR:
	case (int) XS_OK:
#ifdef	PRINTFS
		printf("clnt_dg_xdrin returned %d\n", ci->ci_error.re_status);
#endif
		return (ci->ci_error.re_status);
	}
	/* NOTREACHED */
}

static enum clnt_stat
clnt_dg_marshall(cl, ci)
	CLIENT		*cl;		/* client handle */
	register struct callinfo *ci;
{
	register XDR *xdrs = &(ci->ci_outxdrs);
	register struct cu_data	*cu = ci->ci_cu;

	assert(cl);
	assert(ci);
	assert(cu);
	assert(ci->ci_outbuf);
	assert(cu->cu_mcallproto);
	assert(cu->cu_msize);
	xdrs->x_op = XDR_ENCODE;
	XDR_SETPOS(xdrs, cu->cu_msize);
	(void) memcpy(ci->ci_outbuf, cu->cu_mcallproto, (int) cu->cu_msize);
	/*
	 * the transaction id is the first thing in the output buffer
	 */
	*(u_long *) ci->ci_outbuf = ++ci->ci_xid;
#ifdef	PRINTFS
	printf("clnt_dg_marashall:  xid %d\n", ci->ci_xid);
#endif
	if ((! XDR_PUTLONG(xdrs, (long *)&ci->ci_proc)) ||
	    (! AUTH_MARSHALL(cl->cl_auth, xdrs)) ||
	    (! (*ci->ci_xargs)(xdrs, ci->ci_argsp)))
		return (ci->ci_error.re_status = RPC_CANTENCODEARGS);
	else
		return (RPC_SUCCESS);
}

static enum clnt_stat
clnt_dg_send(ci)
	register struct callinfo *ci;
{
	register struct cu_data	*cu = ci->ci_cu;
	struct t_unitdata tu_data;

	assert(ci);
	assert(cu);
	assert(cu->cu_fd >= 0);
	assert(ci->ci_outbuf);
	tu_data.addr = cu->cu_addr;
	tu_data.udata.buf = ci->ci_outbuf;
	tu_data.udata.len = (int) XDR_GETPOS(&ci->ci_outxdrs);
	tu_data.opt.len = 0;
	if (t_sndudata(cu->cu_fd, &tu_data) == -1) {
		ci->ci_error.re_terrno = t_errno;
		ci->ci_error.re_errno = errno;
		return (ci->ci_error.re_status = RPC_CANTSEND);
	} else
		return (RPC_SUCCESS);
}


static enum clnt_dg_receive_stat
clnt_dg_receive(ci)
	register struct callinfo *ci;
{
	register struct cu_data	*cu = ci->ci_cu;
	fd_set readfds;
	fd_set mask;
	struct t_unitdata *tmp_trdata = (struct t_unitdata *) NULL;
	u_long pktxid;
	int res;		/* result of operations */
	
	assert(ci);
	assert(cu);
	FD_ZERO(&mask);
	FD_SET(cu->cu_fd, &mask);
	for (;;) {
		extern void (*_svc_getreqset_proc)();
		extern fd_set svc_fdset;
		int fds;

		if (tmp_trdata != (struct t_unitdata *) NULL) {
			t_free((char *) tmp_trdata, T_UNITDATA);
			tmp_trdata = (struct t_unitdata *) NULL;
		}
		readfds = mask;

		/*
		 * This provides for callback support.  When a client
		 * recv's a call from another client on the server fd's,
		 * it calls _svc_getreqset(&readfds) which would return
		 * after serving all the server requests.  Also look under
		 * svc.c
		 */
		if (_svc_getreqset_proc) {
			for (fds = 0; fds < howmany(FD_SETSIZE, NFDBITS); fds++)
				readfds.fds_bits[fds] |=
					svc_fdset.fds_bits[fds];
		}
		switch (select(_rpc_dtbsize(), &readfds, (fd_set *)NULL,
				(fd_set *)NULL, &ci->ci_rexmittime)) {

		case 0:
			ci->ci_timewaited.tv_sec += ci->ci_rexmittime.tv_sec;
			ci->ci_timewaited.tv_usec += ci->ci_rexmittime.tv_usec;
			while (ci->ci_timewaited.tv_usec >= 1000000) {
				ci->ci_timewaited.tv_sec++;
				ci->ci_timewaited.tv_usec -= 1000000;
			}
			/* update the time to next retransmission */
			if (ci->ci_rexmittime.tv_sec < RPC_MAX_BACKOFF) {
				ci->ci_rexmittime.tv_usec *= 2;
				ci->ci_rexmittime.tv_sec *= 2;
				while (ci->ci_rexmittime.tv_usec >= 1000000) {
					ci->ci_rexmittime.tv_sec++;
					ci->ci_rexmittime.tv_usec -= 1000000;
				}
			}

			if (timercmp(&ci->ci_timewaited, &ci->ci_calltimeout, <))
				return (RS_RESEND);
			else {
				ci->ci_error.re_status = RPC_TIMEDOUT;
				return (RS_ERROR);
			}

		/*
		 * buggy in other cases because ci->ci_timewaited is not being
		 * updated.
		 */
		case -1:
			if (errno != EBADF) {
				errno = 0;	/* reset it */
				continue;
			}
			ci->ci_error.re_errno = errno;
			ci->ci_error.re_terrno = 0;
			ci->ci_error.re_status = RPC_CANTRECV;
			return (RS_ERROR);
		}

		if (!FD_ISSET(cu->cu_fd, &readfds)) {
			/* must be for server side of the house; for callback */
			(*_svc_getreqset_proc)(&readfds);
			continue;	/* do select again */
		}

		/* We have some data now */
		tmp_trdata = (struct t_unitdata *)t_alloc(cu->cu_fd,
				T_UNITDATA, T_ALL);
		if (tmp_trdata == (struct t_unitdata *)NULL) {
			ci->ci_error.re_errno = errno;
			ci->ci_error.re_terrno = t_errno;
			ci->ci_error.re_status = RPC_SYSTEMERROR;
			return (RS_ERROR);
		}

		do {
			int moreflag;	/* flag indicating more data */

			moreflag = 0;
 			if (errno == EINTR) {
				/*
				 * Must make sure errno was not already
				 * EINTR in case t_rcvudata() returns -1.
				 * This way will only stay in the loop
				 * if getmsg() sets errno to EINTR.
				 */
				errno = 0;
 			}
			res = t_rcvudata(cu->cu_fd, tmp_trdata, &moreflag);
			if ((moreflag & T_MORE) ||
			    (tmp_trdata->udata.len > cu->cu_recvsz)) {
#ifdef	PRINTFS
				printf("clnt_dg_receive:  moreflag %d, udata.len %d, recvsz %d\n", moreflag, tmp_trdata->udata.len, cu->cu_recvsz);
#endif
				/*
				 * Drop this packet. I ain't got any
				 * more space.
				 */
				res = -1;
				/* I should not really be doing this */
				errno = 0;
				/*
				 * XXX: Not really Buffer overflow in the
				 * sense of TLI.
				 */
				t_errno = TBUFOVFLW;
			}
		} while (res < 0 && errno == EINTR);
		if (res < 0) {
#ifdef sun
			if (errno == EWOULDBLOCK)
#else
			if (errno == EAGAIN)
#endif
				continue;
			if (t_errno == TLOOK) {
				int old;

				old = t_errno;
				if (t_rcvuderr(cu->cu_fd, (struct t_uderr *) NULL) == 0)
					continue;
				else
					ci->ci_error.re_terrno = old;
			} else {
				ci->ci_error.re_terrno = t_errno;
			}
			t_free((char *) tmp_trdata, T_UNITDATA);
			ci->ci_error.re_errno = errno;
			ci->ci_error.re_status = RPC_CANTRECV;
			return (RS_ERROR);
		}
		if (tmp_trdata->udata.len < sizeof (u_long))
			/* tmp_trdata is freed at the top of the for loop */
			continue;
		/*
		 *	If the returned XID happens to match ours, we're in
		 *	luck.  If not, we have to search for an in-progress
		 *	call to which the reply should be attached.
		 */
		pktxid = *(u_long *) (tmp_trdata->udata.buf);
		if (pktxid != ci->ci_xid) {
			register struct callinfo *p;

#ifdef	PRINTFS
			printf("clnt_dg_receive:  pktxid (%x) != ci_xid (%x)\n", pktxid, ci->ci_xid);
#endif
			p = xid_to_callinfo(cu, pktxid);
			/* don't overwrite a previous reply */
			if (p && p->ci_trdata == (struct t_unitdata *) NULL) {
				p->ci_trdata = tmp_trdata;
				/* prevent a t_free on the packet we gave away */
				tmp_trdata = (struct t_unitdata *) NULL;
			}
			/* else, tmp_trdata is freed at the top of the for loop */
			continue;
		}
#ifdef	PRINTFS
		else
			printf("clnt_dg_receive:  xid match\n");
#endif

		/* we now assume we have the proper reply */
		ci->ci_trdata = tmp_trdata;
		break;
	}
	return (RS_OK);
}

static enum clnt_dg_xdrin_stat
clnt_dg_xdrin(cl, ci)
	register CLIENT	*cl;		/* client handle */
	register struct callinfo *ci;
{
	struct rpc_msg reply_msg;
	register XDR *xdrs = &ci->ci_outxdrs;
	XDR reply_xdrs;
	bool_t ok;

	assert(cl);
	assert(ci);
	assert(ci->ci_trdata);
	assert(ci->ci_trdata->udata.buf);
	assert(ci->ci_trdata->udata.len <= ci->ci_cu->cu_sendsz);
	reply_msg.acpted_rply.ar_verf = _null_auth;
	reply_msg.acpted_rply.ar_results.where = ci->ci_resultsp;
	reply_msg.acpted_rply.ar_results.proc = ci->ci_xresults;

	/*
	 * now decode and validate the response
	 */
	xdrmem_create(&reply_xdrs, ci->ci_trdata->udata.buf,
			(u_int)ci->ci_trdata->udata.len, XDR_DECODE);
	ok = xdr_replymsg(&reply_xdrs, &reply_msg);
	t_free((char *)ci->ci_trdata, T_UNITDATA);
	ci->ci_trdata = (struct t_unitdata *) NULL;
	/* XDR_DESTROY(&reply_xdrs);	save a few cycles on noop destroy */
	if (ok) {
		_seterr_reply(&reply_msg, &(ci->ci_error));
		if (ci->ci_error.re_status == RPC_SUCCESS) {
			if (! AUTH_VALIDATE(cl->cl_auth,
				&reply_msg.acpted_rply.ar_verf)) {
				ci->ci_error.re_status = RPC_AUTHERROR;
				ci->ci_error.re_why = AUTH_INVALIDRESP;
			}
			if (reply_msg.acpted_rply.ar_verf.oa_base != NULL) {
				xdrs->x_op = XDR_FREE;
				(void)xdr_opaque_auth(xdrs,
					&(reply_msg.acpted_rply.ar_verf));
			}
		} /* end successful completion */
		else {
			/* maybe our credentials need to be refreshed ... */
			if (ci->ci_nrefreshes > 0 && AUTH_REFRESH(cl->cl_auth)) {
				ci->ci_nrefreshes--;
				return (XS_CALLAGAIN);
			}
		} /* end of unsuccessful completion */
	} /* end of valid reply message */
	else {
		ci->ci_error.re_status = RPC_CANTDECODERES;
		return (XS_ERROR);
	}
	return (XS_OK);
}

/*
 *      The action of this function is not well-defined in the face of
 *      asynchronous calls.  We do the best we can by first trying to
 *      find a synchronous callinfo structure and if none is found,
 *      taking the first call in the chain.  Finally, we assume that
 *	the error must have been from a rac_send() failure and look in
 *	the rac_senderr structure.
 */
static void
clnt_dg_geterr(cl, errp)
	CLIENT *cl;
	struct rpc_err *errp;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci;

	for (ci = cu->cu_calls; ci; ci = ci->ci_next)
		if (ci->ci_flags & CI_SYNC) {
			*errp = ci->ci_error;
			return;
		}
	if (ci == (struct callinfo *) 0 && cu->cu_calls != (struct callinfo *) 0)
		*errp = cu->cu_calls->ci_error;
	else
		/*
		 *	No calls in progress at all - assume this was a
		 *	rac_send failure.
		 */
		*errp = rac_senderr;
}

/*
 *      The action of this function is not well-defined in the face of
 *      asynchronous calls.  We do the best we can by first trying to
 *      find a synchronous callinfo structure and if none is found,
 *      taking the first call in the chain.
 */
static bool_t
clnt_dg_freeres(cl, xdr_res, res_ptr)
	CLIENT *cl;
	xdrproc_t xdr_res;
	caddr_t res_ptr;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci;
	register XDR *xdrs = (XDR *) 0;

	for (ci = cu->cu_calls; ci; ci = ci->ci_next)
		if (ci->ci_flags & CI_SYNC) {
			xdrs = &ci->ci_outxdrs;
			break;
		}
	if (xdrs == (XDR *) 0 && ci == (struct callinfo *) 0 &&
	    cu->cu_calls != (struct callinfo *) 0)
		xdrs = &cu->cu_calls->ci_outxdrs;

	if (xdrs) {
		xdrs->x_op = XDR_FREE;
		return ((*xdr_res)(xdrs, res_ptr));
	} else
		return (FALSE);
}

static void
clnt_dg_abort(/* h */)
	/* CLIENT *h; */
{
}

static bool_t
clnt_dg_control(cl, request, info)
	CLIENT *cl;
	int request;
	char *info;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;

	switch (request) {
	case CLSET_FD_CLOSE:
		cu->cu_closeit = TRUE;
		return (TRUE);
	case CLSET_FD_NCLOSE:
		cu->cu_closeit = FALSE;
		return (TRUE);
	}

	/* for other requests which use info */
	if (info == NULL)
		return (FALSE);
	switch (request) {
	case CLSET_TIMEOUT:
		if (time_not_ok((struct timeval *)info))
			return (FALSE);
  		cu->cu_total = *(struct timeval *)info;
  		break;
	case CLGET_TIMEOUT:
		*(struct timeval *) info = cu->cu_total;
		break;
	case CLGET_SERVER_ADDR:		/* Give him the fd address */
		/* Now obsolete. Only for backword compatibility */
		(void) memcpy(info, cu->cu_addr.buf, (int)cu->cu_addr.len);
		break;
	case CLSET_RETRY_TIMEOUT:
		if (time_not_ok((struct timeval *)info))
			return (FALSE);
		cu->cu_wait = *(struct timeval *) info;
		break;
	case CLGET_RETRY_TIMEOUT:
		*(struct timeval *) info = cu->cu_wait;
		break;
	case CLGET_FD:
		*(int *) info = cu->cu_fd;
		break;
	case CLGET_SVC_ADDR:
		*(struct netbuf *) info = cu->cu_addr;
		break;
	case CLRAC_DROP:
		rac_dg_drop(cl, (struct callinfo *) info);
		break;
	case CLRAC_POLL:
		return ((bool_t) rac_dg_poll(cl, (struct callinfo *) info));
	case CLRAC_RECV:
		return ((bool_t) rac_dg_recv(cl, (struct callinfo *) info));
	case CLRAC_SEND:
		return ((bool_t) rac_dg_send(cl, (struct rac_send_req *) info));
	default:
		return (FALSE);
	}
	return (TRUE);
}

static void
clnt_dg_destroy(cl)
	CLIENT *cl;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci, *nextci;

	if (cu->cu_closeit)
		(void) t_close(cu->cu_fd);
	for (ci = cu->cu_calls; ci; ci = nextci) {
		nextci = ci->ci_next;
		if (ci->ci_trdata)
			t_free((char *)ci->ci_trdata, T_UNITDATA);
		XDR_DESTROY(&(ci->ci_outxdrs));
		/*
		 *	Don't destroy the one allocated synchronous callinfo
		 *	structure.
		 *
		 *	if ((ci->ci_flags & CI_SYNC) == 0)
		 */
		free_callinfo(ci);
	}
	if (cu->cu_addr.buf)
		(void) free(cu->cu_addr.buf);
	(void) mem_free((caddr_t) cu, sizeof (*cu));
	if (cl->cl_netid && cl->cl_netid[0])
		(void) mem_free(cl->cl_netid, strlen(cl->cl_netid) +1);
	if (cl->cl_tp && cl->cl_tp[0])
		(void) mem_free(cl->cl_tp, strlen(cl->cl_tp) +1);
	(void) mem_free((caddr_t)cl, sizeof (CLIENT));
}

static struct clnt_ops *
clnt_dg_ops()
{
	static struct clnt_ops ops;

	if (ops.cl_call == NULL) {
		ops.cl_call = clnt_dg_call;
		ops.cl_abort = clnt_dg_abort;
		ops.cl_geterr = clnt_dg_geterr;
		ops.cl_freeres = clnt_dg_freeres;
		ops.cl_destroy = clnt_dg_destroy;
		ops.cl_control = clnt_dg_control;
	}
	return (&ops);
}

/*
 * Make sure that the time is not garbage.  -1 value is allowed.
 */
static bool_t
time_not_ok(t)
	struct timeval *t;
{
	return (t->tv_sec < -1 || t->tv_sec > 100000000 ||
		t->tv_usec < -1 || t->tv_usec > 1000000);
}

static struct callinfo *
alloc_callinfo(cu, flags)
	struct cu_data *cu;
	u_int flags;
{
	register struct callinfo *ci;

	/*
	 *	Memory is arranged as:
	 *
	 *	------------------------------------
	 *	| callinfo structure | send buffer |
	 *	------------------------------------
	 *
	 *	with the receive buffer allocated via t_alloc()
	 */
	ci = (struct callinfo *) mem_alloc(sizeof (*ci) + cu->cu_sendsz);
	if (ci == (struct callinfo *) NULL)
		return ((struct callinfo *) NULL);
	ci->ci_trdata = (struct t_unitdata *) NULL;
	ci->ci_outbuf = ((char *) ci) + sizeof (*ci);

	ci->ci_flags = flags;
	ci->ci_cu = cu;
	if (cu->cu_calls != (struct callinfo *) 0) {
		ci->ci_next = cu->cu_calls;
		cu->cu_calls = ci;
	} else {
		ci->ci_next = (struct callinfo *) 0;
		cu->cu_calls = ci;
	}

	return (ci);
}

static void
free_callinfo(ci)
struct callinfo	*ci;
{
	(void) mem_free((char *) ci, sizeof (*ci) + ci->ci_cu->cu_sendsz);
}

static struct callinfo	*
find_callinfo(cu, flags)
struct cu_data	*cu;
register u_int	flags;
{
	register struct callinfo	*ci;

	for (ci = cu->cu_calls; ci; ci = ci->ci_next)
		if (ci->ci_flags & flags)
			return (ci);

	return ((struct callinfo *) 0);
}

static void
dequeue_callinfo(cu, targetci)
struct cu_data	*cu;
struct callinfo	*targetci;
{
	register struct callinfo	*ci, *prevci = (struct callinfo *) 0;

	for (ci = cu->cu_calls; ci; ci = ci->ci_next) {
		if (ci == targetci)
			if (cu->cu_calls == ci)
				cu->cu_calls = ci->ci_next;
			else {
				assert(prevci != (struct callinfo *) 0);
				prevci->ci_next = ci->ci_next;
			}
		prevci = ci;
	}
}
static void
rac_dg_drop(cl, h)
	CLIENT		*cl;
	struct callinfo	*h;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci, *prevci;

	for (ci = cu->cu_calls, prevci = (struct callinfo *) NULL; ci; ci = ci->ci_next)
		if (ci == h && (ci->ci_flags & CI_ASYNC)) {
			if (cu->cu_calls == ci)
				cu->cu_calls = ci->ci_next;
			else {
				assert(prevci != (struct callinfo *) NULL);
				prevci->ci_next = ci->ci_next;
			}
			if (ci->ci_trdata)
				t_free((char *)ci->ci_trdata, T_UNITDATA);
			XDR_DESTROY(&(ci->ci_outxdrs));
			free_callinfo(ci);
			return;
		} else
			prevci = ci;
}

static enum clnt_stat
rac_dg_poll(cl, h)
	CLIENT		*cl;
	struct callinfo	*h;
{
	register struct cu_data	*cu;
	fd_set readfds;
	struct timeval now, delta1, delta2;
	struct timeval polltimeout;
	struct t_unitdata	*tmp_trdata = (struct t_unitdata *) NULL;
	struct callinfo *ci;
	u_long pktxid;
	int res;		/* result of operations */

	if (rachandle_is_valid(cl, h))
		cu = h->ci_cu;
	else
		return (RPC_STALERACHANDLE);
	assert(cl);
	assert(h);
	assert(cu);
	assert(cu->cu_fd >= 0);
	
	/*
	 *	If a packet's already been received (possibly by someone else
	 *	doing a poll), return success immediately.
	 */
	if (h->ci_trdata != (struct t_unitdata *) NULL) {
#ifdef	PRINTFS
		printf("rac_dg_poll:  packet waiting for handle %x\n", h);
#endif
		return (RPC_SUCCESS);
	}

	FD_ZERO(&readfds);
	FD_SET(cu->cu_fd, &readfds);
	timerclear(&polltimeout);
	switch (select(cu->cu_fd + 1, &readfds, (fd_set *) NULL,
		       (fd_set *) NULL, &polltimeout)) {
	case 0:
		/*
		 *	Compute the time difference between now and the time
		 *	we last called clnt_dg_send().
		 */
		(void) gettimeofday(&now, (struct timezone *) 0);
		delta1.tv_sec = now.tv_sec - h->ci_sendtime.tv_sec;
		delta1.tv_usec = now.tv_usec - h->ci_sendtime.tv_usec;
		if (delta1.tv_usec < 0) {
			delta1.tv_sec--;	/* need to ``borrow'' */
			delta1.tv_usec += 1000000;
		}
		assert(delta1.tv_sec >= 0);
		assert(delta1.tv_usec >= 0);
		
		/*
		 *	``Delta1'' is the time we've waited since the last call
		 *	to clnt_dg_send();  ``h->ci_timewaited'' contains the
		 *	cumulative wait from the first send to the most recent.
		 *
		 *	``Delta2'' will contain the total waiting time.  If it
		 *	exceeds the total call timeout, give up.
		 */
		delta2.tv_sec = delta1.tv_sec + h->ci_timewaited.tv_sec;
		delta2.tv_usec = delta1.tv_usec + h->ci_timewaited.tv_usec;
		while (delta2.tv_usec >= 1000000) {
			delta2.tv_sec++;
			delta2.tv_usec -= 1000000;
		}
		if (timercmp(&delta2, &h->ci_calltimeout, >)) {
			rac_dg_drop(cl, h);
			return (RPC_TIMEDOUT);
		}

		/*
		 *	Nothing there for us, but the call hasn't timed out.
		 *	If ``delta1'' is greater than the retransmit timeout,
		 *	retransmit the packet for the user and recompute the
		 *	retransmit time.
		 */
		if (timercmp(&delta1, &h->ci_rexmittime, >)) {
			h->ci_sendtime = now;	/* lie by a few microseconds */
			h->ci_timewaited = delta1;	/* remember time waited so far */
#ifdef	PRINTFS
			if (clnt_dg_send(h) != RPC_SUCCESS)
				printf("rac_dg_poll:  clnt_dg_send failed\n");
			else
				printf("rac_dg_poll:  clnt_dg_send succeeded\n");
#else
			(void) clnt_dg_send(h);
#endif
			if (h->ci_rexmittime.tv_sec < RPC_MAX_BACKOFF) {
				h->ci_rexmittime.tv_usec *= 2;
				h->ci_rexmittime.tv_sec *= 2;
				while (h->ci_rexmittime.tv_usec >= 1000000) {
					h->ci_rexmittime.tv_sec++;
					h->ci_rexmittime.tv_usec -= 1000000;
				}
			}
		}
		return (RPC_INPROGRESS);
		
	case -1:
		if (errno == EBADF) {
			h->ci_error.re_errno = EBADF;
			h->ci_error.re_terrno = 0;
			h->ci_error.re_status = RPC_CANTRECV;
			return (h->ci_error.re_status = RPC_CANTRECV);
		} else {
			errno = 0;	/* reset it */
			return (RPC_INPROGRESS);
		}
		/* NOTREACHED */
	}
	assert(FD_ISSET(cu->cu_fd, &readfds));	/* paranoia */
	
	/*
	 *	Select says we have a packet.  Receive it.  If it`s for us,
	 *	great!  If it's for some other customer of this CLIENT handle,
	 *	figure out who and hang the packet off his callinfo structure.
	 */
	
	tmp_trdata = (struct t_unitdata *)t_alloc(cu->cu_fd, T_UNITDATA, T_ALL);
	if (tmp_trdata == (struct t_unitdata *) NULL) {
		h->ci_error.re_errno = errno;
		h->ci_error.re_terrno = t_errno;
		return (h->ci_error.re_status = RPC_SYSTEMERROR);
	}

	do {
		int moreflag;	/* flag indicating more data */
		
		moreflag = 0;
		if (errno == EINTR) {
			/*
			 * Must make sure errno was not already
			 * EINTR in case t_rcvudata() returns -1.
			 * This way will only stay in the loop
			 * if getmsg() sets errno to EINTR.
			 */
			errno = 0;
		}
		res = t_rcvudata(cu->cu_fd, tmp_trdata, &moreflag);
		if ((moreflag & T_MORE) ||
		    (tmp_trdata->udata.len > cu->cu_recvsz)) {
#ifdef	PRINTFS
			printf("rac_dg_poll:  moreflag %d, udata.len %d, recvsz %d\n", moreflag, tmp_trdata->udata.len, cu->cu_recvsz);
#endif
			/*
			 * Drop this packet. I ain't got any
			 * more space.
			 */
			res = -1;
			/* I should not really be doing this */
			errno = 0;
			/*
			 * XXX: Not really Buffer overflow in the
			 * sense of TLI.
			 */
			t_errno = TBUFOVFLW;
		}
	} while (res < 0 && errno == EINTR);
	if (res < 0) {
#ifdef sun
		if (errno == EWOULDBLOCK)
#else
		if (errno == EAGAIN)
#endif
			return (RPC_INPROGRESS);
		if (t_errno == TLOOK) {
			int old;
			
			old = t_errno;
			if (t_rcvuderr(cu->cu_fd, (struct t_uderr *) NULL) == 0)
				return (RPC_INPROGRESS);
			else
				h->ci_error.re_terrno = old;
		} else {
			h->ci_error.re_terrno = t_errno;
		}
		t_free((char *) tmp_trdata, T_UNITDATA);

		h->ci_error.re_errno = errno;
		return (h->ci_error.re_status = RPC_CANTRECV);
	}
	if (tmp_trdata->udata.len < sizeof (u_long)) {
		t_free((char *) tmp_trdata, T_UNITDATA);
		return (RPC_INPROGRESS);
	}
	/*
	 *	If the returned XID happens to match ours, we're in luck.
	 *	If not, we have to search for an in-progress call to which
	 *	the reply should be attached.
	 */
	pktxid = *(u_long *) (tmp_trdata->udata.buf);
	if (pktxid == h->ci_xid) {
		assert(h->ci_trdata == (struct t_unitdata *) NULL);
#ifdef	PRINTFS
		printf("rac_dg_poll:  xid match\n");
#endif
		h->ci_trdata = tmp_trdata;
		return (RPC_SUCCESS);
	} else {
#ifdef	PRINTFS
		printf("rac_dg_poll:  pktxid (%x) != ci_xid (%x)\n", pktxid, h->ci_xid);
#endif
		ci = xid_to_callinfo(cu, pktxid);
		/* don't overwrite a previous reply */
		if (ci && ci->ci_trdata == (struct t_unitdata *) NULL) {
#ifdef	PRINTFS
			printf("rac_dg_poll:  found owner of xid %d:  handle %x\n", pktxid, ci);
#endif
			ci->ci_trdata = tmp_trdata;
		} else
			/* no owner found or reply already present - drop it */
			t_free((char *) tmp_trdata, T_UNITDATA);
		return (RPC_INPROGRESS);
	}
	/* NOTREACHED */
}

static enum clnt_stat
rac_dg_recv(cl, h)
	CLIENT		*cl;
	struct callinfo	*h;
{
	if (!rachandle_is_valid(cl, h))
		return (RPC_STALERACHANDLE);

	/*
	 *	If a packet has been received (indicated by a non-NULL
	 *	ci_trdata field), XDR it.  Otherwise, we act just like
	 *	normal, blocking, RPC.
	 */
	if (h->ci_trdata != (struct t_unitdata *) NULL) {
		switch (clnt_dg_xdrin(cl, h)) {
		case (int) XS_CALLAGAIN:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_xdrin returned CALL_AGAIN\n");
#endif
			rac_dg_drop(cl, h);
			return (RPC_AUTHERROR);
		case (int) XS_ERROR:
		case (int) XS_OK:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_xdrin returned %d\n", h->ci_error.re_status);
#endif
			rac_dg_drop(cl, h);
			return (h->ci_error.re_status);
		}
		/* NOTREACHED */
	} else {
receive_again:
		switch ((int) clnt_dg_receive(h)) {
		case (int) RS_RESEND:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_receive returned RS_RESEND\n");
#endif
			goto send_again;
			/* NOTREACHED */
			
		case (int) RS_ERROR:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_receive returned error %d\n", h->ci_error.re_status);
#endif
			rac_dg_drop(cl, h);
			return (h->ci_error.re_status);
			/* NOTREACHED */
			
		case (int) RS_OK:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_receive returned OK\n");
#endif
			break;
		}
		
		switch (clnt_dg_xdrin(cl, h)) {
		case (int) XS_CALLAGAIN:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_xdrin returned CALL_AGAIN\n");
#endif
			break;
			
		case (int) XS_ERROR:
		case (int) XS_OK:
#ifdef	PRINTFS
			printf("rac_dg_recv:  clnt_dg_xdrin returned %d\n", h->ci_error.re_status);
#endif
			rac_dg_drop(cl, h);
			return (h->ci_error.re_status);
		}

#ifdef	PRINTFS
		printf("rac_dg_recv:  call_again\n");
#endif
		if (clnt_dg_marshall(cl, h) != RPC_SUCCESS) {
			rac_dg_drop(cl, h);
			return (h->ci_error.re_status);
		}
#ifdef	PRINTFS
		else
			printf("rac_dg_recv:  clnt_dg_marshall succeeded\n");
#endif
		
send_again:
		if (clnt_dg_send(h) != RPC_SUCCESS) {
			rac_dg_drop(cl, h);
			return (h->ci_error.re_status);
		}
#ifdef	PRINTFS
		else
			printf("rac_dg_recv:  clnt_dg_send succeeded\n");
#endif
		
		goto receive_again;
	}
}

static void *
rac_dg_send(cl, h)
	CLIENT		*cl;
	struct rac_send_req *h;
{
	register struct cu_data *cu = (struct cu_data *) cl->cl_private;
	register struct callinfo *ci;

	if ((ci = alloc_callinfo(cu, CI_ASYNC)) == (struct callinfo *) NULL) {
		rac_senderr.re_status = RPC_SYSTEMERROR;
		return ((void *) NULL);
	}
	ci->ci_nrefreshes = 2;	/* number of times to refresh cred */
	ci->ci_proc = h->proc;
	ci->ci_xargs = h->xargs;
	ci->ci_argsp = (caddr_t) h->argsp;
	ci->ci_xresults = h->xresults;
	ci->ci_resultsp = (caddr_t) h->resultsp;
	xdrmem_create(&(ci->ci_outxdrs), ci->ci_outbuf, cu->cu_sendsz, XDR_ENCODE);
	
	ci->ci_xid = ++cu->cu_xidseed;
	ci->ci_calltimeout = h->timeout;
	ci->ci_trdata = NULL;

	timerclear(&ci->ci_timewaited);
	ci->ci_rexmittime = cu->cu_wait;

	if (clnt_dg_marshall(cl, ci) != RPC_SUCCESS)
		return ((void *) NULL);

#ifdef	PRINTFS
	printf("rac_dg_call:  calling clnt_dg_send\n");
#endif
	(void) gettimeofday(&ci->ci_sendtime, (struct timezone *) 0);
	if (clnt_dg_send(ci) != RPC_SUCCESS) {
		dequeue_callinfo(cu, ci);
		free_callinfo(ci);
		return ((void *) NULL);
	}
#ifdef	PRINTFS
	else
		printf("rac_dg_call:  clnt_dg_send succeeded\n");
#endif

	return ((void *) ci);
}

static bool_t
rachandle_is_valid(cl, h)
	CLIENT		*cl;
	struct callinfo	*h;
{
	register struct callinfo *ci;

	for (ci = ((struct cu_data *) cl->cl_private)->cu_calls; ci; ci = ci->ci_next) {
		if (ci == h && (ci->ci_flags & CI_ASYNC))
			return (TRUE);
	}
	return (FALSE);
}

static struct callinfo *
xid_to_callinfo(cu, xid)
	struct cu_data	*cu;
	u_long		xid;
{
	register struct callinfo *ci;

	for (ci = cu->cu_calls; ci; ci = ci->ci_next)
		if (xid == ci->ci_xid)
			return (ci);
	
	return ((struct callinfo *) NULL);
}
