/*
 * Sun RPC is a product of Sun Microsystems, Inc. and is provided for
 * unrestricted use provided that this legend is included on all tape
 * media and as a part of the software program in whole or part.  Users
 * may copy or modify Sun RPC without charge, but are not authorized
 * to license or distribute it to anyone else except as part of a product or
 * program developed by the user or with the express written consent of
 * Sun Microsystems, Inc.
 *
 * SUN RPC IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING THE
 * WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun RPC is provided with no support and without any obligation on the
 * part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY SUN RPC
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
#ifndef lint
static        char sccsid[] = "@(#)table.c	1.8 91/03/11 TIRPC 1.0";
#endif

/*
 * This file implements the table routines.
 *
 *	Written 4-27-89 Copyright (c) 1989 by Sun Microsystems, Inc.
 */

#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <ctype.h>
#include "table.h"

extern int errno;

/* Misc macros */
#define T_FIELD(i) (th->fields + (i)) /* The field entry indexed by i */

/*
 * table_buf()
 * 
 * Checks to see if the table buffer is allocated, does so if it is not.
 */
static char *
_table_buf(th)
	TABLE	*th;
{
	if (!(th->s_data))
		th->s_data = (char *) malloc(MAXBUF);
	return (th->s_data);
}

/*
 * table_tokens()
 * 
 * Checks to see if the table token buffer is allocated, does so if it is not.
 */

static char **
_table_tokens(th)
	TABLE	*th;
{
	if (!th->tokens)
		th->tokens = (char **) malloc(MAXTOKEN * sizeof(char *));
	return (th->tokens);
}

/*
 * Determine if the character we are looking at is one of the separator
 * characters.
 */
int
issep(x, sep)
	char	x,
		*sep;
{
	while (*sep) {
		if (*sep == x)
			return 1;
		sep++;
	}
	
	return 0;
}

/*
 * tokenize()
 *
 *	This utility function will convert a string of values separated by
 * the separator, into a series of tokens. It returns a pointer to it't token
 * list, and the number of tokens found.
 */

int
tokenize(str, len, tokens, sep)
	char	*str;	  /* string to be tokenized 		*/
	int	len;	  /* length of the data     		*/
	char	**tokens; /* Place to put the token pointers 	*/
	char	*sep;	  /* The separator(s)	  		*/
{
	int	tok_num,	/* Number of our token */
		j,		/* scratch counter var */
		null_tok;	/* flag for detecting null tokens */

	/* Sanity check */
	if (! tokens) {
		return 0;
	}

	j = 0;
	tok_num = 0;
	null_tok = 0;

	while ((j < len) && (*str != TBL_COMMENT) && (*str != '\0')
			&& (tok_num < MAXTOKEN)) {
		/* Eliminate leading break characters */
		while (issep(*str,sep) && (j < len)) {

			/* 
			 * Kludge to process null tokens, when two non
			 * whitespace tokens are seen in a row they are
			 * assumed to represent a null token, which is
			 * created as a entry in the token array pointing
	 		 * to "\0".
			 */
			if ((null_tok) && !isspace(*str)) {
				*(tokens+tok_num) = str;
				tok_num++;
				*str = '\0';
				null_tok = 0;
			} else if (!isspace(*str))	/* prime the pump */
				null_tok = 1;
			
			str++;	/* move ahead our pointers */
			j++;
		}
		if (*str == TBL_COMMENT || *str == '\0')
			break;

		/* Save a pointer to the front of it. */
		*(tokens+tok_num) = str++;
		j++;
 		tok_num++;

		/* Find the other end of this token in the string */
		while (! issep(*str, sep) && (j < len)) {
			str++;
			j++;
		}
		*str = '\0'; /* terminate it with a NUL */
		str++;
		j++;
	
	}
	if (tok_num < MAXTOKEN)
		*(tokens+tok_num) = NULL;

	return tok_num;

}

/*
 * opentable()
 *
 * This function will open a data table and initialze the handle to contain
 * the relevant information. It returns 1 on error, with handle->error set
 * to the specific error.
 *
 * WARNING! This routine assumes that the TABLE handle that you pass to it
 *  was initialized to zero when created.
 */
int
opentable(th, stayopen)
	TABLE	*th;
	int	stayopen;
{
	int	i,j;			/* temporary counter variables	  */
	char	*buf = _table_buf(th);	/* Our internal buffer space	  */
	char	**tokens = _table_tokens(th); /* A token list		  */

	/* Make sure the buffer is available */
	if (buf == NULL) {
		th->error = TBE_NOMEM;
		return 1;
	}
	if (tokens == NULL) {
		th->error = TBE_NOMEM;
		return 1;
	}
	if (stayopen)
		th->flags |= TBF_STAYOPEN;
	else
		th->flags &= ~TBF_STAYOPEN;

	/* Open the file that contains the table */
	if (th->fd == NULL) {
		th->fd = fopen(th->file, "r");
		if (th->fd == NULL) {
			th->error = errno;
			return 1;
		}
	} else {
	    rewind(th->fd);
	}

	/* Check to see if this is a free form table */
	if ((th->flags & TBF_FREEFORM) == 0)
		return 0;

	/* Initialize the field indexes to bogus values */
	for (i = 0; i < th->n_fields; i++) 
		(th->fields+i)->index = -1; /* Initialize to no match */
	

	/* Find the field header line, (first non comment line) */
	do {
		(void) fgets(buf, MAXBUF, th->fd);
		if (feof(th->fd)) { /* no data */
			th->error = errno;
			fclose(th->fd);
			th->fd = NULL;
			return 1;
		}
	} while (*buf == TBL_COMMENT || *buf == '\0');

	/* parse out the field names */

	/* first tokenize the line */
	th->n_tokens = tokenize(buf, strlen(buf)-1, th->tokens, th->brk);

	if (th->n_tokens < th->n_fields) {
		th->error = TBE_TOKEN_MISMATCH; /* Wrong number returned */
		fclose(th->fd);
		th->fd = NULL;
		return 1;
	}

	/* Note that if the last field is multivalued this _must_ match */
	if ((th->flags & TBF_MULTIVALUE) && (th->n_tokens != th->n_fields)) {
		th->error = TBE_TOKEN_MISMATCH; /* Wrong number returned */
		fclose(th->fd);
		th->fd = NULL;
		return 1;
	}

	for (j = 0; j < th->n_tokens; j++) {

		/* Match up the token with it's field entry */
		for (i = 0; i < th->n_fields; i++) {
			if ((T_FIELD(i)->index == -1) &&
			    strcasecmp(T_FIELD(i)->name, *(tokens+j)) == 0) {
				T_FIELD(i)->index = j;
				break;
			}
		}
		
	}

	/* Now we check that we found all of our fields ... */
	for (j = 0; j < th->n_tokens; j++)
		if (T_FIELD(j)->index == -1) {
			th->error = TBE_TOKEN_MISMATCH;
			fclose(th->fd);
			th->fd = NULL;
			return 1;
		}

	/* If all tokens matched field names, then we are set */
	return 0;
}	



/* 
 * readtable()
 *
 *	This function is analagous to the getXXXent() routine, except that
 * it takes this table handle, and stuffs the values into that. It ignores
 * lines that begin with the comment symbol.
 * 
 * Tokens with the place holder TBL_PLACEHOLDER return as returned but have
 * null strings. 
 */

int
readtable(th)
	TABLE	*th;	/* The table handle */
{
	char	*buf = _table_buf(th);
	char	**tokens = _table_tokens(th);

	/* Make sure the buffer is available */
	if (buf == NULL) {
		th->error = TBE_NOMEM;
		return 1;
	}
	if (tokens == NULL) {
		th->error = TBE_NOMEM;
		return 1;
	}

	/* Check to see if the handle is valid */
	if (th->fd == NULL) {
		/* reopen the file */
		th->fd = fopen(th->file, "r");
		if (th->fd == NULL) {
			th->error = errno;
			return 1;
		}
	}

	/* find next non-comment line */
	do {
		(void) fgets(buf, MAXBUF, th->fd);
	} while ((!feof(th->fd)) && (*buf == TBL_COMMENT || *buf == '\0'));

	/* if read error like EOF */
	if (ferror(th->fd) || feof(th->fd)) {
		th->error = errno;
		return 1;
	}

	if (th->flags & TBF_MULTILINE) {
		/* check for extra lines needed */
	}
	
	/* break it up into chunks */
	th->n_tokens = tokenize(buf, strlen(buf)-1, th->tokens, th->brk);
	
	/* Too few is an error, to many is ok for multivalued entries */
	if (th->n_tokens < th->n_fields) {
		th->error = TBE_NOT_ENOUGH;
		return 1;
	}

	if(interpret(th) == 0) {	/* success */
	    return (0);
	} else {			/* failed */
	    return (1);
	}
}

/*
 * interpret()
 *
 *	This function will take the data, and the table handle and interpret
 * the values. Generally this results in initializing a structure which the
 * wrapper routine will return to the caller.
 *
 * It returns 1 on failure.
 */

int
interpret(th)
	TABLE	*th;	  /* The table pointer */
{
	int	i;	/* A temporary counter variable */
	char	**tokens = th->tokens;

	/* Sanity check on the token list */
	if (! tokens) {
		th->error = TBE_NO_TOKENS;
		return 1;
	}

	for (i = 0; i < th->n_fields; i++) {

		/* Check for a default signal, token == DEFAULT or "" */
		if ((strcmp(*(tokens+T_FIELD(i)->index),TBL_DEFAULT) == 0) ||
		    *(*(tokens+T_FIELD(i)->index)) == '\0') {
			T_FIELD(i)->result = T_FIELD(i)->defrslt;
		} else if (! (*(T_FIELD(i)->parse))( *(tokens + T_FIELD(i)->index), 
						 T_FIELD(i)->defrslt, 
						 T_FIELD(i)->result))
			return 1;
	}

	/* 
	 * Do the multivalue magic after parsing everything else. This takes
	 * the "nth" token pointer and copies it into the last fields result
 	 * pointer (which had better be a char **) and this makes tokens(n)
	 * thru (n_tokens) the extra values.
	 */
	if (th->flags & TBF_MULTIVALUE) 
		*((void **)(T_FIELD(th->n_fields)->result)) = 
				(void *)(tokens+th->n_fields);

	return 0;
}

/*
 * closetable()
 *
 *	This function is analagous to endXXXent(), it "closes" the file.
 * Unless the "stayopen" flag is set, in which case it simply rewinds it.
 */

int
closetable(th)
	TABLE	*th;
{
	if (th->fd == NULL)
		return 0;
    
	if ((th->flags & TBF_STAYOPEN) != 0) {
		rewind(th->fd);
	} else {
		fclose(th->fd);
		th->fd = NULL;
	}
	return 0;
}

/*
 * searchbyfield()
 *
 * 	This routine implements the file version of the getXXXbyYYY() routines.
 * Like the wrapper routines, it can be called without calling open first.
 */

int
searchbyfield(th, index, value, compare)
	TABLE	*th;		/* The table to search 			   */
	int	index;		/* The field of interest 		   */
	void	*value;		/* A pointer to the value of interest 	   */
	int	(*compare)();	/* A function that will check for equality */
{
	char	*buf = _table_buf(th);

	/* First, make sure the table is accessible */
	if (th->fd == NULL) {
		/* Try to open it up */
		if (opentable(th, 0)) {
			return 1;
		}
	} else {
		/* if already open reset it to the beginning of the file */
		rewind(th->fd);
	
		/* skip past the field line, to first data line */
		do {
			(void) fgets(buf, MAXBUF, th->fd);
		} while ((!feof(th->fd)) && (*buf == TBL_COMMENT ||
			*buf == '\0'));
	}

	while (! readtable(th)) {
		if ((*compare)(T_FIELD(index)->result, value) == 0)
			return 0;
	}
	
	/* while will exit, and th->error will contain the reason */
	return 1;
}

void
tbl_perror(err)
	int	err;
{
	switch (err) {
		case TBE_NOMEM :
			fprintf(stderr,"not enough memory.\n");
			break;
		case TBE_UNKNOWN_TOKEN :
			fprintf(stderr,"field name not recognized.\n");
			break;
		case TBE_NOFILE :
			fprintf(stderr,"table file is not open.\n");
			break;
		case TBE_NOT_ENOUGH :
			fprintf(stderr,"insufficient data on input record.\n");
			break;
		case TBE_TOKEN_MISMATCH :
			fprintf(stderr,"token/field mismatch.\n");
			break;
		case TBE_NO_TOKENS :
			fprintf(stderr,"no tokens specified.\n");
			break;
		default :
			fprintf(stderr,"OS_Level error %d\n",err);
			break;
	}
}
