/*
 
 *  inodes.c --
 
 *      Index node implementation for PC NFS file server.  The server
 
 *      builds its view of the filesystem everytime a lookup call is
 
 *      generated for a file, and initially when the file system export
 
 *      list is parsed.  This package handles DOS and UNIX style pathnames.
 
 *
 
 *  Author:
 
 *      See-Mong Tan
 
 */
 

 
#include "common.h"
 

 
/* forward declaration of local procedures */
 
extern void getcomp();
 
extern u_long insertapath();
 
extern u_long newinodeno();
 

 
/* the inode dump file */
 
#define INODEFILE "inode.dmp"
 

 
/* the global inode file pointer */
 
FILE *inodefp;
 

 
  /* last inode number assigned - starting from 2, 1 is root directory of all */
 
static u_long lastinodeno = 2;
 

 
typedef struct DIRNODE {
 
	char name[20];		/* dos file name */
 
	u_long inode;		/* it's inode number */
 
	int drive;		/* drive of file */
 
	struct DIRNODE *next;	/* next file in same directory */
 
	struct DIRNODE *subdir;	/* ptr to subdirectory entries */
 
	struct DIRNODE *parent;	/* ptr to parent directory */
 
	struct DIRNODE *sp;	/* search path, for reconstructing path names */
 
} Dir_t;
 

 
  /* the directory structure begins here */
 
static Dir_t *RootDir;
 

 
  /* the inode table - list of directory node pointers */
 
typedef Dir_t *Inode_t;
 

 
#define MAXINODES 8192
 
static Inode_t InodeTable[MAXINODES];	/* 8K of inodes available */
 

 
/*
 
 *  bool_t inode_init() --
 
 *      Initializes the inode module.  Reads the previous (if any) inodes
 
 *      created and adds them to the directory tree.  Has a very simple
 
 *      minded export file parser.
 
 */
 
bool_t inode_init()
 
{
 
	FILE *expfp;
 
	char line[MAXPATHNAMELEN];
 
	char path[MAXPATHNAMELEN];
 
	extern void showtree();
 
	int r;
 

 
	/* read from export list first */
 
	if ((expfp = fopen(EXPORTS, "r")) == NULL)
 
		return FALSE;
 
	while(fgets(line, MAXPATHNAMELEN -1, expfp) == line) {
 
		int i;
 

 
		/* get the pathname alone */
 
		for(i = 0; line[i] != ' ' && line[i] != '\t' && line[i] != '\0'
 
		    && line[i] != '\n'; i++)
 
			path[i] = line[i];
 
		path[i] = '\0';
 
		if (i < 2) 
 
			break;		/* invalid name */
 
		(void) insertapath(path, NULL, &RootDir);
 
	}
 
	(void) fclose(expfp);
 
	/* rebuild old filesystem tree */
 
	/* open for reading and appending */
 
	if ((inodefp = fopen(INODEFILE, "a+")) == NULL)
 
		return FALSE;
 

 
	while(fscanf(inodefp, "%s", path) != EOF)
 
		(void) insertapath(path, NULL, &RootDir);
 
	(void) fclose(inodefp);
 
#ifdef DEBUG
 
	showtree(RootDir);	/* dump tree so far */
 
#endif
 
	
 
	return TRUE;
 
}
 

 
/*		
 
 *  void getcomp(char *comp, char **path) --
 
 *      Gets component name of *path and puts it in comp, and advances
 
 *      *path to beginning of next component.  Handles UNIX or DOS style
 
 *      pathnames.
 
 */
 
#define VALIDCHARS(c) (c == '.' || c == '_' || c == '!' || c == '@' || \
 
		       c == '#' || c == '~' || c == '%' || c == '&' || \
 
		       c == '$' || c == '-' || c == '(' || c == ')' || \
 
		       c == '*' || c == '}' || c == '{' || c == '^' || \
 
		       c == '\'' || c == '`' || c == '?')
 
void getcomp(comp, path)
 
	char *comp;
 
	char **path;
 
{
 
	if (! (isalnum(**path) || VALIDCHARS(**path)))
 
		(*path)++;
 
	while(isalnum(**path) || VALIDCHARS(**path)) {
 
		*comp++ = **path;
 
		(*path)++;
 
	}
 
	if (**path == ':')
 
		(*path)++;
 
	*comp = '\0';
 
}
 

 
/*
 
 *  Dir_t *makenewdirnode(char *name, Dir_t *parent) --
 
 *      Returns new directory node initialized with name.
 
 */
 
Dir_t *makenewdirnode(name, parent)
 
	char *name;
 
	Dir_t *parent;
 
{
 
	Dir_t *new;
 

 
	new = (Dir_t *) malloc(sizeof(Dir_t));
 
	if (new == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	(void) strcpy(new->name, name);
 
	new->inode = newinodeno();
 
	new->parent = parent;
 
	new->next = new->subdir = new->sp = NULL;
 
	InodeTable[new->inode] = new;
 
	
 
	return new;
 
}
 

 
/*
 
 *  Dir_t *findindirlist(name, Dir_t *parent, Dir_t **dirlist) --
 
 *      Finds the node with name in dirlist and returns a ptr to it, or makes
 
 *      a new node if none is found, and returns a pointer to the new node.
 
 */
 
Dir_t *findindirlist(name, parent, dirlist)
 
	char *name;
 
	Dir_t *parent;
 
	Dir_t **dirlist;
 
{
 
	int lexico;		/* lexicographic comparator */
 
	Dir_t *new;
 
	Dir_t *p, *q;		/* for inserting down the list */
 

 
	if (*dirlist == NULL) {				  /* NULL entry */
 
		*dirlist = makenewdirnode(name, parent);
 

 
		return *dirlist;
 
	}
 
	if ((lexico = strcmp(name, (*dirlist)->name)) < 0) {
 
		Dir_t *tmp;			/* must insert in front */
 

 
		new = makenewdirnode(name, parent);
 
		tmp = *dirlist;
 
		*dirlist = new;
 
		new->next = tmp;
 

 
		return new;
 
	}
 
	else if (lexico == 0) 			/* found the node */
 
		return *dirlist;
 

 
	/* cdr down the list and find node or find place to insert */
 
	p = (*dirlist)->next;  q = *dirlist;
 
	while(p != NULL) {
 
		if ((lexico = strcmp(name, p->name)) == 0)
 
			return p;		/* found it */
 
		else if (lexico < 0)
 
			break;			/* should insert here */
 
		q = p; p = p->next;		/* go on to next elt */
 
	}
 
	/* this is where we insert */
 
	new = makenewdirnode(name, parent);
 
	q->next = new;
 
	new->next = p;
 

 
	return new;
 
}
 

 
/*
 
 *  u_long insertapath(char *path, Dir_t *parent, Dir_t **dirlist) 
 
 *      Inserts path into dirlist and returns final inode number.
 
 */
 
u_long insertapath(path, parent, dirlist)
 
	char *path;
 
	Dir_t *parent;
 
	Dir_t **dirlist;
 
{
 
	char comp[20];		/* 11 is max length of one component */
 
	Dir_t *compnode;	/* the component node */
 

 
	getcomp(comp, &path);				/* get component */
 
	compnode = findindirlist(comp, parent, dirlist);
 
	if (*path == '\0')
 
		return compnode->inode;	/* no more descents needed */
 
	else
 
		return insertapath(path, compnode, &(compnode->subdir));
 
}
 

 
/*
 
 *  u_long addpathtodirtree(char *path) --
 
 *      Adds a path to the directory tree and return it's inode number.
 
 */
 
u_long addpathtodirtree(path)
 
	char *path;
 
{
 
	u_long inode;
 

 
	inodefp = fopen(INODEFILE, "a");
 
	(void) fprintf(inodefp, "%s\n", path);	/* add to inode file */
 
	(void) fclose(inodefp);
 
	return insertapath(path, NULL, &RootDir);
 

 
	return inode;
 
}
 

 
/*
 
 *  void shownode(Dir_t *dirt) --
 
 *      Debugging aid.  Dumps the node.
 
 */
 
void shownode(dirt)
 
	Dir_t *dirt;
 
{
 
	(void) printf("Name: %s;\t Inode #: %ld\n", dirt->name, dirt->inode);
 
}
 

 
/*
 
 *  void showtree(Dir_t *dirt) --
 
 *      Debugging aid.  Dumps the tree.
 
 */
 
void showtree(dirt)
 
	Dir_t *dirt;
 
{
 
	while(dirt != NULL) {
 
		shownode(dirt);
 
		if (dirt->subdir != NULL) {
 
			(void) printf("And in this directory... \n");
 
			showtree(dirt->subdir);
 
		}
 
		dirt = dirt->next;
 
	}
 
}
 

 
/*
 
 *  u_long pntoin(char *path) --
 
 *      Returns inode number corresponding to path.  Path should already
 
 *      exist.  Returns -1 for error.
 
 */
 
#define isdot(path, len) (*((path) + (len) -1) == '.')
 
#define isdotdot(path, len) (isdot(path,len) && (*((path) + (len) - 2)) == '.')
 
u_long pntoin(path)
 
	char *path;
 
{
 
	char comp[20];		/* component of filename */
 
	Dir_t *p;
 
	int len;
 

 
	len = strlen(path);
 
	if (isdotdot(path, len))
 
		*(path + len -3) = '\0';
 
	else if (isdot(path, len))
 
		*(path + len -2) = '\0';
 
	
 
	p = RootDir;				/* start search from root */
 
	while(p != NULL) {			/* search down the tree */
 
		int lex;
 

 
		getcomp(comp, &path);		/* search across the tree */
 
		while(p != NULL) {
 
			if ((lex = strcmp(comp, p->name)) == 0)
 
				break;
 
			else if (lex < 0)	/* overshot - not found */
 
				return -1;
 
			else
 
				p = p->next;	/* across on this level */
 
		}
 
		if (p == NULL)
 
			return -1;
 
		else if (*path == '\0') 
 
			return p->inode;
 
		else
 
			p = p->subdir;		/* down one level */
 
	}
 
	return -1;
 
}
 

 
/*
 
 *  char *intopn(u_long inode) --
 
 *      Converts inode to path name in dos format.  DOS style path name
 
 *	is returned.  A NULL is returned if there is no such inode.
 
 */
 
char *intopn(inode)
 
	u_long inode;
 
{
 
	Dir_t *dirp, *tmp;
 
	char *path;
 

 
	if ((dirp = InodeTable[inode]) == NULL)
 
		return NULL;
 

 
	/* make space for path */
 
	path = (char *) malloc(MAXPATHNAMELEN);
 
	if (path == NULL) {
 
		(void) fprintf(stderr, "nfs: out of memory\n");
 
		abort();
 
	}
 
	/* move upwards and set parents to point to this child */
 
	while(dirp->parent != NULL) {
 
		Dir_t *parent;
 

 
		parent = dirp->parent;
 
		parent->sp = dirp;	/* set search path of parent */
 
		dirp = parent;		/* go to parent */
 
	}
 
	/* first get the drive name set up in path */
 
	(void) sprintf(path, "%c:", *(dirp->name));
 
	tmp = dirp;
 
	dirp = dirp->sp;			/* down the search path */
 
	tmp->sp = NULL;				/* clearing trail behind you */
 
	if (dirp != NULL) {			/* still more after the path */
 
		/* at top now, follow search ptrs down to child */
 
		do {
 
			(void) strcat(path, "\\");
 
			(void) strcat(path, dirp->name);
 
			tmp = dirp;
 
			dirp = dirp->sp;
 
			tmp->sp = NULL;
 
		} while(dirp != NULL);
 
	}
 
	else
 
		(void) strcat(path, "\\");
 

 
	return path;
 
}
 

 
/*
 
 *  char *intoname(u_long inode) --
 
 *      Converts inode to name of file.
 
 */
 
char *intoname(inode)
 
	u_long inode;
 
{
 
	if (inode > MAXINODES)
 
		return NULL;
 

 
	return (InodeTable[inode])->name;
 
}	
 

 
/*
 
 *  u_long parentinode(u_long inode)
 
 *      Returns inode number of parent.
 
 */
 
u_long parentinode(inode)
 
	u_long inode;
 
{
 
	Dir_t *dirp, *parent;
 

 
	if (inode > MAXINODES)
 
		return 0;
 
	if ((dirp = InodeTable[inode]) != NULL && 
 
	    (parent = dirp->parent) != NULL)
 
		return parent->inode;
 
	else
 
		return 0;
 
}
 

 
/*
 
 *  u_long newinodeno() --
 
 *	Returns a new inode number, or aborts if we run out of inodes.
 
 *	A system cleanup must then occur.
 
 */
 
u_long newinodeno()
 
{
 
	if (lastinodeno > MAXINODES) {
 
		(void) fprintf(stderr, "server err: out of inodes\n");
 
		abort();
 
	}
 
	return lastinodeno++;
 
}
 

 
	
 
#ifdef DEBUG
 
main()
 
{
 
	char *s;
 

 
	addpathtodirtree("/c/bin/test");
 
	addpathtodirtree("/d/stan/src/unfsd");
 
	addpathtodirtree("/c/bin/testee");
 
	addpathtodirtree("/d/stan/src/unfsd");
 
	addpathtodirtree("/d/stan/src/unfsd/tester");
 
	showtree(RootDir);
 
	(void) printf("pntoin /d/stan/src is %ld\n", pntoin("/d/stan/src"));
 
	s = intopn(7);
 
	(void) printf("intopn 7 : %s\n", s);
 
	(void) free(s);
 
	s = intopn(7);
 
	(void) printf("intopn 7 : %s\n", s);
 
}
 
#endif
 
